#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <getopt.h>

#include <db.h>
#include <apr.h>
#include <apr_pools.h>

#include <libbttracker.h>

static int run = 0;
static int stopped = 0;

const static struct option opts[] = {
    { "master", 0, 0, 'm' },
    { "help", 0, 0, 'h' },
    { "verbose", 0, 0, 'v' },
    { 0, 0, 0, 0 }
};

static void usage(FILE *fh) {
    fprintf(fh,
        "btt_xml2db mod_bt/" VERSION " - http://www.crackerjack.net/mod_bt/\n"
        "* Imports an XML file containing infohash information into a tracker database.\n\n"
        "Usage: btt_xml2db [--master] [--help] xmlfile homedir\n"
        "\txmlfile\t\t\tXML file (eg; torrents.xml)\n"
        "\thomedir\t\t\tmod_bt data directory (eg; /var/lib/mod_bt)\n\n"
        "\t-m\t--master\tAct as mod_bt master\n"
        "\t-v\t--verbose\tVerbose operation\n"
        "\t\t\t\t(Use when mod_bt is not running)\n"
        "\t-h\t--help\t\tThis help screen\n\n"
    );

    return;
}

static void sigINT(int signum) {
    run = 0;
    stopped = 1;
    return;
}

static void import_infohash(btt_tracker* tracker, DB_TXN *ptxn, xmlNodePtr node, int verbose) {
    btt_txn_iterator iterator[] = {
        { btt_iter_peer_stats, NULL },
        { NULL, NULL }
    };
    
    btt_infohash in_hash = btt_xml2infohash(tracker->p, node);
    btt_infohash* hash;
    DBT hash_key;
    DBT hash_val;
    DBC* hash_cur = NULL;
    DB_TXN* txn = NULL;
    int ret;

    bzero(&hash_key, sizeof(hash_key));
    bzero(&hash_val, sizeof(hash_val));

    hash_key.data = apr_palloc(tracker->p, BT_INFOHASH_LEN);
    hash_key.size = hash_key.ulen = BT_INFOHASH_LEN;
    hash_key.flags = DB_DBT_USERMEM;
 
    hash_val.data = apr_palloc(tracker->p, sizeof(btt_infohash));
    hash_val.ulen = sizeof(btt_infohash);
    /* hash_val.size handled by bzero */
    hash_val.flags = DB_DBT_USERMEM;
    
    memcpy(hash_key.data, in_hash.infohash, BT_INFOHASH_LEN);
    
    if((ret = btt_txn_start(tracker, ptxn, &txn, 0)) != 0) {
        fprintf(stderr, "Failed to start a transaction\n");
        return;
    }
    
    if((ret = btt_txn_load_hashcursor(tracker, tracker->p, txn, &hash_key, &hash_val, &hash_cur, BTT_WRITE_CURSOR(tracker), DB_RMW, 1)) != 0) {
        fprintf(stderr, "Failed to load/create a hash\n");
        goto abort;
    }

    hash = (btt_infohash*)hash_val.data;
    memcpy(hash, &in_hash, sizeof(btt_infohash));
    iterator[0].data = (void*) hash;

    if(verbose) {
        printf("Importing %s (%s)\n", bt_str_infohash(tracker->p, hash->infohash), hash->filename);
    }
    
    hash->peers = hash->seeds = hash->shields = 0;
    if((ret = btt_txn_iterate_peerlist(tracker, tracker->p, hash, txn, 0, DB_RMW, iterator)) != 0) {
        fprintf(stderr, "Failed to refresh the peerlist!\n");
        goto abort;
    }

    if((ret = hash_cur->c_put(hash_cur, &hash_key, &hash_val, DB_CURRENT)) != 0) {
        tracker->db.hashes->err(tracker->db.hashes, ret, "import_infohash(): hash_cur->c_put()");
        goto abort;
    }
    
    if(hash_cur)
        hash_cur->c_close(hash_cur);

    if(txn)
        if((ret = txn->commit(txn, 0)) != 0) {
            tracker->db.env->err(tracker->db.env, ret, "bt_cxn_announce(): txn->commit()");
            goto abort;
        }

    return;
    
    abort:

    if(hash_cur)
        hash_cur->c_close(hash_cur);
 
    if(txn)
        txn->abort(txn);
 
    return;
}

int main(int argc, char** argv) {
    int odone = 0;
    int rv = 0;
    int verbose = 0;
    int oindex = 0;
    int master = 0;
    xmlDocPtr document;
    xmlNodePtr node;
    btt_tracker *tracker;
    
    while(odone != -1) {
        odone = getopt_long(argc, argv, "mhp", opts, &oindex);
        switch(odone) {
            case -1:
                break;
            case 'm':
                master++;
                break;
            case 'v':
                verbose++;
                break;
            case 'h':
                usage(stdout);
                exit(0);
                break;
            case '?':
                usage(stderr);
                exit(1);
                break;
            default:
                fprintf(stderr, "unknown option code 0x%x\n", odone);
                break;
        }
    }

    if(argc - optind != 2) {
        usage(stderr);
        exit(2);
    }
    
    if(!(document = xmlParseFile(argv[optind]))) {
        fprintf(stderr, "Failed to read %s!\n", argv[optind]);
        exit(3);
    }
    
    if(apr_app_initialize(NULL, NULL, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_app_initialize() failed!\n");
        fflush(stderr);
        exit(20);
    }
 
    atexit(apr_terminate);
 
    if(!(tracker = btt_tracker_alloc(NULL, argv[optind+1], master))) {
        fprintf(stderr, "bt_tracker_alloc() failed!\n");
        fflush(stderr);
        exit(5);
    }
 
    if(master)
        tracker->s->start_t = time(NULL);

    signal(SIGINT, sigINT);
    signal(SIGTERM, sigINT);
    signal(SIGPIPE, sigINT);

    if(strcmp((char*)document->children->name, "Infohashes")) {
        fprintf(stderr, "This doesn't look like a mod_bt database dump to me.\n");
        fflush(stderr);
        rv = 21;
        goto stop;
    }
    
    for(node=document->children->children; node; node=node->next) {
        if(!strcmp((char*)node->name, "Infohash")) {
            import_infohash(tracker, NULL, node, verbose);
        }
    }
 
    
    stop:
        
    exit(rv);
}
