/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include <libbtutil.h>
#include <libbtpeer.h>

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>

#include <apr.h>
#include <apr_pools.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

static const char* banner = "\
btp_checkmetafile "VERSION" - check integrity of bittorrent downloads\n\
\n\
";

static apr_status_t progress(btp_torrent* t, btp_peer* p, void* data) {
    printf(
        "Checking ... %.02f%% complete      \r",
        btp_torrent_percent_complete(t)
    );
    fflush(stdout);
    return APR_EAGAIN;
}


/* Exit codes:
 * 0 = All torrents complete
 * 1 = Torrent(s) incomplete
 * 2 = Torrent(s) not yet started
 * 3 = User error
 * 4 = System error
 */

int main(int argc, char **argv) {
    apr_pool_t*         pool = NULL;
    bt_metainfo*        metainfo = NULL;
    btp_file_pool*      file_pool = NULL;
    btp_torrent*        torrent = NULL;
    int i;
    struct stat         sb;
    apr_status_t        ret;
    int rv = 0;
    char buf[255];

    printf(banner);
    
    if(argc < 2) {
        printf(
            "usage: %s file1.torrent file2.torrent file3.torrent ...\n",
            argv[0]
        );
        exit(3);
    }
    
    if(apr_app_initialize(NULL, NULL, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_app_initialize() failed!\n");
        fflush(stderr);
        exit(4);
    }
 
    atexit(apr_terminate);
 
    if(apr_pool_initialize() != APR_SUCCESS) {
        fprintf(stderr, "apr_pool_initialize() failed!\n");
        fflush(stderr);
        exit(4);
    }
 
    if(apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_pool_create() failed!\n");
        fflush(stderr);
        exit(4);
    }

    file_pool = btp_file_pool_create(pool, 2);
    apr_hook_global_pool = pool;
    
    btp_hook_metainfo_sha1(progress, NULL, NULL, APR_HOOK_LAST);
    
    for(i=1;i<argc;i++) {
        if((metainfo = bt_metainfo_parse(pool, argv[i]))) {
            if((torrent = btp_torrent_create(pool, metainfo, file_pool))) {
                BT_STRCPY(torrent->destination, ".");
                if(stat(torrent->info->name, &sb)) {
                    printf("%s: 0.00%% complete (Not yet started)\n", argv[i]);
                    rv = 2;
                } else {
                    double complete;
                    
                    ret = btp_metainfo_check(torrent, NULL);
                    printf("                                            \r");
                    if(ret != APR_SUCCESS) {
                        fprintf(
                            stderr, "Failed to check %s: %s\n",
                            torrent->info->name,
                            apr_strerror(ret, buf, sizeof(buf))
                        );
                        exit(4);
                    }
                    
                    complete = btp_torrent_percent_complete(torrent);
                    
                    printf(
                        "%s: %.02f%% complete\n", torrent->info->name, complete
                    );
                    
                    if(complete < 100 && !rv)
                        rv = 1;
                }
            } else {
                fprintf(stderr, "Failed to initialize torrent\n");
                fflush(stderr);
                exit(4);
            }
        } else {
            fprintf(stderr, "Failed to parse torrent\n");
            fflush(stderr);
            exit(4);
        }
    }
    
    exit(rv);
}
