/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <time.h>
/* other libs */
#include <db.h>
/* local */
#include <libbttracker.h>

btt_infohash* btt_txn_load_hash(
    btt_tracker* tracker, apr_pool_t* p, DB_TXN* txn, DBT* key,
    int cc_flags, int c_flags, int create
) {
 DBT* val = apr_pcalloc(p, sizeof(DBT));
 DBC* cursor = NULL;
 int ret = 0;
 
 val->data = apr_pcalloc(p, sizeof(btt_infohash));
 val->ulen = sizeof(btt_infohash);
 val->flags = DB_DBT_USERMEM;
 
 /* only one possible error location, already logged by load_hashcursor */
 if((ret = btt_txn_load_hashcursor(tracker, p, txn, key, val, &cursor, cc_flags, c_flags, create)) != 0)
  return NULL;

 cursor->c_close(cursor);
 return (btt_infohash*)val->data;
}

int btt_txn_save_hash(btt_tracker* tracker, apr_pool_t* p, DB_TXN* txn, btt_infohash* hash)
{
 DBT key;
 DBT val;
 DBC* cur = NULL;
 btt_infohash val_data;
 int ret = 0;
 
 bzero(&key, sizeof(key));
 bzero(&val, sizeof(val));
 bzero(&val_data, sizeof(val_data));

 key.data = hash->infohash;
 key.size = BT_INFOHASH_LEN;
 key.ulen = BT_INFOHASH_LEN;
 key.flags = DB_DBT_USERMEM;
 
 val.data = &val_data;
 val.ulen = sizeof(val_data);
 val.flags = DB_DBT_USERMEM;
 

 if((ret = btt_txn_load_hashcursor(tracker, p, txn, &key, &val, &cur, BTT_WRITE_CURSOR(tracker), DB_RMW, 1)) != 0)
 {
  tracker->db.hashes->err(tracker->db.hashes, ret, "bt_txn_save_hash(): bt_txn_load_hashcursor()");
  return ret;
 }
 
 val_data = *(hash);
 
 if((ret = cur->c_put(cur, &key, &val, DB_CURRENT)) != 0)
  tracker->db.hashes->err(tracker->db.hashes, ret, "bt_txn_save_hash(): c_put()");

 cur->c_close(cur);
 return ret;
}

int btt_txn_load_hashcursor(btt_tracker* tracker, apr_pool_t* p, DB_TXN* txn, DBT* key, DBT* val, DBC** cursor, int cc_flags, int c_flags, int create)
{
 int ret = 0;
 DBC *rv = *cursor = NULL;

 DBT re_key;
 re_key = *key;
 re_key.data = apr_palloc(p, re_key.ulen);
 memcpy(re_key.data, key->data, key->size);

 if(key->ulen < BT_INFOHASH_LEN)
 {
  tracker->db.hashes->errx(tracker->db.hashes, "bt_txn_load_hashcursor(): key->ulen < %d", BT_INFOHASH_LEN);
  return EINVAL;
 }
  
 if(key->size > BT_INFOHASH_LEN)
 {
  tracker->db.hashes->errx(tracker->db.hashes, "bt_txn_load_hashcursor(): key->size > %d", BT_INFOHASH_LEN);
  return EINVAL;
 }

 if(val->ulen < sizeof(btt_infohash))
 {
  tracker->db.hashes->errx(tracker->db.hashes, "bt_txn_load_hashcursor(): val->ulen < %d", sizeof(btt_infohash));
  return EINVAL;
 }
 
 if(val->size)
 {
  tracker->db.hashes->errx(tracker->db.hashes, "bt_txn_load_hashcursor(): val->size < 0");
  return EINVAL;
 }

 if((ret = tracker->db.hashes->cursor(tracker->db.hashes, txn, &rv, cc_flags)) != 0)
 {
  tracker->db.hashes->err(tracker->db.hashes, ret, "bt_txn_load_hashcursor(%s): cursor()", bt_str_infohash(p, (char*)key->data));
  return ret;
 }
 
 ret = rv->c_get(rv, key, val, c_flags | DB_SET);
 
 if(ret != 0 && (create != 1 || ret != DB_NOTFOUND))
 {
  if(ret != DB_NOTFOUND)
   tracker->db.hashes->err(tracker->db.hashes, ret, "bt_txn_load_hashcursor(%s): c_get", bt_str_infohash(p, (char*)key->data));
  rv->c_close(rv);
  rv = NULL;
 }
 else if(ret == DB_NOTFOUND)
 {
  memcpy(val->data, &new_btt_infohash, sizeof(new_btt_infohash));
  val->size = sizeof(btt_infohash);
  memcpy(((btt_infohash*)val->data)->infohash, re_key.data, BT_INFOHASH_LEN);
  ((btt_infohash*)val->data)->first_t = time(NULL);
  ((btt_infohash*)val->data)->last_t = time(NULL);
  key->size = re_key.size;
  memcpy(key->data, re_key.data, re_key.size);
  
  if((ret = rv->c_put(rv, key, val, DB_KEYFIRST)) != 0)
  {
   tracker->db.hashes->err(tracker->db.hashes, ret, "bt_txn_load_hashcursor(): c_put");
   rv->c_close(rv);
   rv = NULL;
  }
  else if((ret = rv->c_get(rv, key, val, DB_SET | c_flags)) != 0)
  {
   tracker->db.hashes->err(tracker->db.hashes, ret, "bt_txn_load_hashcursor(%s): c_get after c_put", bt_str_infohash(p, (char*)key->data));
   rv->c_close(rv);
   rv = NULL;
  }
  else
  {
   tracker->s->num_hashes++;
  }
 }
 
 *cursor = rv;
 return ret;
}
