/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/* libc */
#include <stdlib.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
/* other libs */ 
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
/* local */
#include <libbttracker.h>

int btt_cxn_announce(btt_tracker* tracker, apr_pool_t* p, DB_TXN* ptxn, const char* args, const char* user_agent, struct sockaddr_in in_address, char** content, int* content_length)
{
 btt_txn_iterator update_iterator[] = 
 {
  { btt_iter_check_peer, NULL },
  { btt_iter_peer_stats, NULL },
  { btt_iter_make_peerlist_check, NULL },
  { btt_iter_make_peerlist_string, NULL },
  { NULL, NULL }
 };
 
 btt_txn_iterator iterator[] =
 {
  { btt_iter_make_peerlist_check, NULL },
  { btt_iter_make_peerlist_string, NULL },
  { NULL, NULL }
 };

 DBT hash_key;
 DBT hash_val;
 DBT peer_key;
 DBT peer_val;
 DBC* hash_cur = NULL;
 DBC* peer_cur = NULL;
 DB_TXN* txn = NULL;
 btt_infohash *hash;
 btt_peer* peer;
 btt_peer in_peer = new_btt_peer;
 btt_iter_content i_content;
 int ret = 0;
 int len = 0;
 int rv = HTTP_OK;
 int np = 0;
 int del = 0;
 time_t interval = 0;
 char* rc = NULL;
 
 bzero(&i_content, sizeof(i_content));
 tracker->s->server_time = time(NULL);

 *content = NULL;
 *content_length = 0;

 tracker->db.env->txn_checkpoint(tracker->db.env, 512, 0, 0);

 if((ret = btt_txn_start(tracker, ptxn, &txn, 0)) != 0)
  return HTTP_SERVER_ERROR;

 tracker->s->announces++;

 if(!btt_peer_req2struct(p, args, &in_peer))
 {
  tracker->s->bad_announces++;
  rv = HTTP_BAD_REQUEST;
  goto err;
 }
 
 bzero(&hash_key, sizeof(hash_key));
 bzero(&hash_val, sizeof(hash_val));
 
 hash_key.data = apr_palloc(p, BT_INFOHASH_LEN);
 hash_key.size = hash_key.ulen = BT_INFOHASH_LEN;
 hash_key.flags = DB_DBT_USERMEM;
 
 hash_val.data = apr_palloc(p, sizeof(btt_infohash));
 hash_val.ulen = sizeof(btt_infohash);
 /* hash_val.size handled by bzero */
 hash_val.flags = DB_DBT_USERMEM;

 memcpy(hash_key.data, in_peer.infohash, BT_INFOHASH_LEN);

 if((ret = btt_txn_load_hashcursor(tracker, p, txn, &hash_key, &hash_val, &hash_cur, BTT_WRITE_CURSOR(tracker), DB_RMW, (tracker->c->flags & BTT_TRACKER_REQUIRE_REG) ? 0 : 1)) != 0)
 {
  tracker->db.hashes->err(tracker->db.hashes, ret, "bt_cxn_announce(): first bt_txn_load_hashcursor()");
  tracker->s->bad_announces++;

  if(ret == DB_NOTFOUND)
  {
   rv = HTTP_NOT_FOUND;
   goto end;
  }
  else
  {
   rv = HTTP_SERVER_ERROR;
   goto err;
  }
 }

 
 hash = (btt_infohash*)(hash_val.data);
 bzero(&peer_key, sizeof(peer_key));
 bzero(&peer_val, sizeof(peer_val));

 peer_key.data = apr_palloc(p, BT_PEERID_LEN + BT_INFOHASH_LEN);
 peer_key.size = peer_key.ulen = BT_PEERID_LEN + BT_INFOHASH_LEN;
 peer_key.flags = DB_DBT_USERMEM;
 
 peer_val.data = apr_pcalloc(p, sizeof(btt_peer));
 peer_val.size = 0;
 peer_val.ulen = sizeof(btt_peer);
 peer_val.flags = DB_DBT_USERMEM;
 
 memcpy(peer_key.data, hash->infohash, BT_INFOHASH_LEN);
 memcpy(peer_key.data + BT_PEERID_LEN, in_peer.peerid, BT_PEERID_LEN);

 if((ret = btt_txn_load_peercursor(tracker, p, txn, &peer_key, &peer_val, &peer_cur, 0, DB_RMW, hash)) != 0)
 {
  tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_announce(): bt_txn_load_peercursor()");
  tracker->s->bad_announces++;
  rv = HTTP_SERVER_ERROR;
  goto err;
 }

 
 peer = (btt_peer*)(peer_val.data);

 if(user_agent && *user_agent)
  strncpy(peer->ua, user_agent, sizeof(peer->ua) - 1);
 
 peer->ua[sizeof(peer->ua)] = 0;
 peer->flags = (peer->flags & BTT_PEER_KEEP_FLAGS) | in_peer.flags;
 strncpy(peer->event, in_peer.event, sizeof(peer->event) - 1);
 peer->event[sizeof(peer->event)-1] = 0;
 
 peer->address = in_peer.address;
 
 if(!in_peer.address.sin_addr.s_addr)
  peer->address.sin_addr = in_address.sin_addr;
 
 peer->real_address = in_address;
 peer->last_t = in_peer.last_t;
 peer->hits ++;
 peer->announce_bytes += in_peer.announce_bytes;
 peer->uploaded = in_peer.uploaded;
 peer->downloaded = in_peer.downloaded;
 peer->left = in_peer.left;
 peer->num_got = 0;
 peer->num_want = in_peer.num_want;

 if(peer->num_want < 0)
  peer->num_want = tracker->c->return_peers;

 if(!peer->first_serve_t)
  peer->first_serve_t = in_peer.last_serve_t;
 
 if((!peer->complete_t) && (in_peer.left == 0))
  peer->complete_t = in_peer.last_t;

 if((ret = peer_cur->c_put(peer_cur, &peer_key, &peer_val, DB_CURRENT)) != 0)
 {
  tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_announce(): first peer_cur->c_put()");
  tracker->s->bad_announces++;
  rv = HTTP_SERVER_ERROR;
  goto err;
 }

 hash->hits ++;

 apr_pool_create(&i_content.pool, p);
 
 if(peer->num_want < hash->peers)
  np = peer->num_want;
 else
  np = hash->peers;
 
 /* somewhat arbitrary */
 i_content.buffer_length = (BT_PEERSTR_LEN * np) + 4096;
 i_content.content = malloc(i_content.buffer_length);
 i_content.content_length = 0;
 i_content.hashandpeer.infohash = hash;
 i_content.hashandpeer.peer = peer;
 i_content.hashandpeer.tracker = tracker;
 i_content.hashandpeer.oldhash = apr_palloc(i_content.pool, sizeof(btt_infohash));
 i_content.hashandpeer.updating = 1;

 BT_MEMCPY(i_content.hashandpeer.oldhash, hash);

 iterator[0].data = &(i_content.hashandpeer);
 iterator[1].data = &i_content;

 update_iterator[0].data = (void*)tracker;
 update_iterator[1].data = (void*)hash;
 update_iterator[2].data = &(i_content.hashandpeer);
 update_iterator[3].data = &i_content;

 if((ret = peer_cur->c_put(peer_cur, &peer_key, &peer_val, DB_CURRENT)) != 0)
 {
  tracker->s->bad_announces++;
  tracker->db.peers->err(tracker->db.peers, ret, "btt_cxn_announce(): first c_put()");
  rv = HTTP_SERVER_ERROR;
  goto err;
 }

 if(i_content.hashandpeer.updating)
  hash->peers = hash->seeds = hash->shields = 0;

 /* technically this modifies and writes but there's some weirdness with locking if i let it ... */
 if((ret = btt_txn_iterate_peerlist(
     tracker, i_content.pool, hash, txn, 0,
     i_content.hashandpeer.updating ? DB_RMW : 0,
     i_content.hashandpeer.updating ? update_iterator : iterator)) != 0
 ) {
  tracker->s->bad_announces++;
  tracker->db.peers->err(tracker->db.peers, ret, "btt_cxn_announce(): bt_txn_iterate_peerlist()");
  rv = HTTP_SERVER_ERROR;
  goto err;
 }

 peer_val.size = 0;

 if(!peer_cur)
 {
  if((ret = btt_txn_load_peercursor(tracker, p, txn, &peer_key, &peer_val, &peer_cur, 0, DB_RMW, hash)) != 0)
  {
   tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_announce(): second bt_txn_load_peercursor()");
   tracker->s->bad_announces++;
   rv = HTTP_SERVER_ERROR;
   goto err;
  }
  else
  {
   peer = peer_val.data;
  }
 }
 else
 {
  if((ret = peer_cur->c_get(peer_cur, &peer_key, &peer_val, DB_SET)) != 0)
  {
   if(ret != DB_NOTFOUND)
   {
    tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_announce(): peer_cur->c_get()");
    tracker->s->bad_announces++;
    rv = HTTP_SERVER_ERROR;
    goto err;
   }
   else
   {
    del = 1;
   }
  }
  else
  {
   peer = peer_val.data;
  }
 }


 rc = apr_palloc(p, i_content.content_length + BT_SHORT_STRING);

 if(hash->peers > tracker->c->return_peer_factor)
  interval = tracker->c->return_interval * ((hash->peers / tracker->c->return_peer_factor) + 1);
 else
  interval = tracker->c->return_interval;

 peer->return_interval = interval;

 if(peer->flags & BTT_PEER_COMPACT)
  len = sprintf(rc, "d8:intervali%lue5:peers%u:", tracker->c->return_interval, i_content.content_length);
 else
  len = sprintf(rc, "d8:intervali%lue5:peersl", tracker->c->return_interval);

 memcpy(&(rc[len]), i_content.content, i_content.content_length);
 len += i_content.content_length;
 rc[len] = 'e';
 len++;

 if(!(peer->flags & BTT_PEER_COMPACT))
 {
  rc[len] = 'e';
  len++;
 }

 rc[len] = '\0';

 free(i_content.content);
 i_content.content = NULL;
 apr_pool_destroy(i_content.pool);
 
 peer->last_t = tracker->s->server_time;
 peer->hits++;
 peer->announce_bytes += i_content.content_length;
 
 if((!strcmp("stopped", peer->event)) && (!del))
 {
  if((ret = peer_cur->c_del(peer_cur, 0)) != 0)
  {
   tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_announce(): peer_cur->c_del()");
   tracker->s->bad_announces++;
   rv = HTTP_SERVER_ERROR;
   goto err;
  }

  hash->peers--;
  if(BTT_PEER_IS_SEED(peer))
   hash->seeds--;
  if(BTT_PEER_IS_SHIELD(peer))
   hash->shields--;
 }
 else if(!del)
 {
  if((ret = peer_cur->c_put(peer_cur, &peer_key, &peer_val, DB_CURRENT)) != 0)
  {
   tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_announce(): second peer_cur->c_put()");
   tracker->s->bad_announces++;
   rv = HTTP_SERVER_ERROR;
   goto err;
  }
 }
 
 peer_cur->c_close(peer_cur);
 peer_cur = NULL;
 
 /* now, finally, we can update the hash stats and finish. */

 hash->last_t = hash->last_peer_t = tracker->s->server_time;
 if(!hash->first_peer_t)
  hash->first_peer_t = hash->last_t;
 
 if(!del)
  if(BTT_PEER_IS_SEED(peer))
  {
   hash->last_seed_t = hash->last_t;
   if(!hash->first_seed_t)
    hash->first_seed_t = hash->last_t;
  }
 
 if(!strcmp("started", peer->event))
  hash->starts++;
 
 if(!strcmp("stopped", peer->event))
  hash->stops++;
 
 if(!strcmp("completed", peer->event))
  hash->completes++;

 if((ret = hash_cur->c_put(hash_cur, &hash_key, &hash_val, DB_CURRENT)) != 0)
 {
  tracker->db.hashes->err(tracker->db.hashes, ret, "bt_cxn_announce(): hash_cur->c_put()");
  tracker->s->bad_announces++;
  rv = HTTP_SERVER_ERROR;
  goto err;
 }

 hash_cur->c_close(hash_cur);
 hash_cur = NULL;

 *content = rc;
 *content_length = len;

 end:

 if(peer_cur)
  peer_cur->c_close(peer_cur);
 
 if(hash_cur)
  hash_cur->c_close(hash_cur);

 if(txn)
  if((ret = txn->commit(txn, 0)) != 0)
  {
   tracker->db.env->err(tracker->db.env, ret, "bt_cxn_announce(): txn->commit()");
   goto err;
  }

 return rv;
 
 err:
 
 if(i_content.content)
  free(i_content.content);
  
 if(peer_cur)
  peer_cur->c_close(peer_cur);
 
 if(hash_cur)
  hash_cur->c_close(hash_cur);
 
 if(txn)
  txn->abort(txn);
 
 return rv;
}
