
#include <apr.h>
#include <apr_errno.h>

#include <string.h>
#include <arpa/inet.h>

#include <libbtpeer/bitfield.h>
#include <libbtpeer/hooks.h>
#include <libbtpeer/peer_hooks.h>
#include <libbtpeer/types/btp_peer.h>
#include <libbtpeer/io.h>
#include <libbtpeer/types/btp_torrent.h>
#include <libbtpeer/types/btp_torrent/pieces.h>

static inline apr_status_t btp_peer_command_state_on(
    btp_peer* p, int state
) {
    p->state = p->state & state;
    return APR_SUCCESS;
}

static inline apr_status_t btp_peer_command_state_off(
    btp_peer* p, int state
) {
    p->state = p->state ^ state;
    return APR_SUCCESS;
}

static apr_status_t choke(
    btp_torrent* t, btp_peer* p, void* data
) {
   return btp_peer_command_state_on(p, BTP_PEER_STATE_THEY_CHOKED);
}

static apr_status_t unchoke(
    btp_torrent* t, btp_peer* p, void* data
) {
   return btp_peer_command_state_off(p, BTP_PEER_STATE_THEY_CHOKED);
}

static apr_status_t interested(
    btp_torrent* t, btp_peer* p, void* data
) {
   return btp_peer_command_state_on(p, BTP_PEER_STATE_THEY_INTERESTED);
}

static apr_status_t uninterested(
    btp_torrent* t, btp_peer* p, void* data
) {
   return btp_peer_command_state_off(p, BTP_PEER_STATE_THEY_INTERESTED);
}

static apr_status_t have(
    btp_torrent*t, btp_peer* p, void* data
) {
    uint32_t piece;
    
    if(p->cmdbuf_len != sizeof(btp_int_t))
        return APR_EINVAL;
    
    piece = ntohl(*((uint32_t*)(p->cmdbuf)));
    btp_bitfield_add(p->pieces, piece);
    
    return APR_SUCCESS;
}

static apr_status_t bitfield(btp_torrent*t, btp_peer* p, void* data) {
    apr_size_t piecebuf_len = (t->info->piece_count / 8);
    if(t->info->piece_count % 8)
        piecebuf_len++;
    
    if(p->cmdbuf_len != piecebuf_len)
        return APR_EINVAL;
    
    p->pieces = apr_palloc(p->p, p->cmdbuf_len);
    memcpy(p->pieces, p->cmdbuf, piecebuf_len);
    
    return APR_SUCCESS;
}

static apr_status_t request(btp_torrent*t, btp_peer* p, void* data) {
    btp_peer_request req;
    btp_int_t* req_data = (btp_int_t*)p->cmdbuf;
    
    if(p->cmdbuf_len != sizeof(btp_int_t) * 3)
        return APR_EINVAL;
    
    req.piece = ntohl(req_data[0]);
    req.begin = ntohl(req_data[1]);
    req.length = ntohl(req_data[2]);
    
    return btp_peer_add_request(p, req);
}

static apr_status_t cancel(btp_torrent*t, btp_peer* p, void* data) {
    btp_peer_request req;
    btp_int_t* req_data = (btp_int_t*)p->cmdbuf;
    
    if(p->cmdbuf_len != sizeof(btp_int_t) * 3)
        return APR_EINVAL;
    
    req.piece = ntohl(req_data[0]);
    req.begin = ntohl(req_data[1]);
    req.length = ntohl(req_data[2]);
    
    return btp_peer_del_request(p, req);
}

static apr_status_t piece(btp_torrent*t, btp_peer* p, void* data) {
    btp_int_t* intbuf = (btp_int_t*) p->cmdbuf;
    uint8_t* databuf = (uint8_t*)p->cmdbuf + (sizeof(btp_int_t) * 2);
    btp_int_t length = p->cmdbuf_len - (sizeof(btp_int_t) * 2);
    btp_int_t piece = intbuf[0];
    btp_int_t offset = intbuf[1];
    btp_int_t left;

    /* do this block-at-a-time */
    for(left=length;left;) {
        int block, boffset, thislen;
        
        block = btp_torrent_piece_to_block(t, piece, offset, &boffset);
        
        if(block > t->block_count)
            return APR_EINVAL;
        
        thislen = t->block_size - boffset;
        
        if(thislen > left)
            thislen = left;
        
        if(!thislen)
            return APR_SUCCESS;

        if(!btp_bitfield_has(t->pieces->block_bits, block)) {
            uint64_t foffset = (t->block_size * block) + boffset;
            bt_off_t actual;
            apr_status_t ret;
            
            if((ret = btp_torrent_io_buffer(
                t, foffset, thislen, databuf, 1, &actual
            )) != APR_SUCCESS)
                return ret;
            
            if(actual != thislen)
                return APR_EOF;
            
            if(thislen == t->block_size && boffset == 0)
                btp_torrent_piece_add_block(t, block);
        }
        
        left -= thislen;
    }
    
    btp_torrent_block_bits_set(t);
    return APR_SUCCESS;
}

void btp_peer_hooks_register() {
    btp_hook_peer_command_choke(choke, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_unchoke(unchoke, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_interested(interested, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_uninterested(
        uninterested, NULL, NULL, APR_HOOK_MIDDLE
    );
    btp_hook_peer_command_have(have, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_bitfield(bitfield, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_request(request, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_cancel(cancel, NULL, NULL, APR_HOOK_MIDDLE);
    btp_hook_peer_command_piece(piece, NULL, NULL, APR_HOOK_MIDDLE);

    return;
}
