/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2001-2003 Bastien Nocera <hadess@hadess.net>
 * Copyright (C) 2006      William Jon McCann <mccann@jhu.edu>
 *
 * moblin-settings-multimedia-keys.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 */

#include <config.h>

#include <string.h>
#include <sys/file.h>
#include <X11/X.h>

/* Gnome headers */
#include <glib/gi18n.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <gconf/gconf-client.h>

#include "eggaccelerators.h"

#include <moblin.h>
#include <volume.h>
#include "msd-media-keys-window.h"
#include "moblin-settings-dbus.h"
#include "moblin-settings-brightness.h"

#define VOLUME_STEP 6           /* percents for one volume button press */

/* we exclude shift, GDK_CONTROL_MASK and GDK_MOD1_MASK since we know what
   these modifiers mean
   these are the mods whose combinations are bound by the keygrabbing code */
#define IGNORED_MODS (0x2000 /*Xkb modifier*/ | GDK_LOCK_MASK  | \
	GDK_MOD2_MASK | GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)
/* these are the ones we actually use for global keys, we always only check
 * for these set */
#define USED_MODS (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK)

typedef struct Acme Acme;

struct Acme {
	GtkWidget *dialog;
	GConfClient *conf_client;

	/* Multihead stuff */
	GdkScreen *current_screen;
	GSList *screens;

	/* MoblinSettingsServer */
	GObject *server;
};

typedef struct {
  guint keysym;
  guint state;
  guint keycode;
} ShortcutKey;

typedef struct {
  int key_type;
  const char *gconf_key;
  ShortcutKey *key;
} ShortcutKeyEntry;

typedef struct RunningApp RunningApp;

struct RunningApp {
	GPid pid;
	gchar *cmd;
};

static ShortcutKeyEntry shortcut_keys_list[HANDLED_SHORTCUT_KEYS];
static GList *runningapps = NULL;

static void
acme_error (char * msg)
{
	GtkWidget *error_dialog;

	error_dialog =
	    gtk_message_dialog_new (NULL,
			    GTK_DIALOG_MODAL,
			    GTK_MESSAGE_ERROR,
			    GTK_BUTTONS_OK,
			    "%s", msg);
	gtk_dialog_set_default_response (GTK_DIALOG (error_dialog),
			GTK_RESPONSE_OK);
	gtk_widget_show (error_dialog);
	g_signal_connect (G_OBJECT (error_dialog),
		"response",
		G_CALLBACK (gtk_widget_destroy), NULL);
}

static gint
find_by_cmd(gconstpointer a, gconstpointer b)
{
	gchar *c = ((RunningApp *)a)->cmd;
	gchar *d = (gchar *)b;

	return strcmp(c, d);
}

static gint
find_by_pid(gconstpointer a, gconstpointer b)
{
	GPid c = ((RunningApp *)a)->pid;
	GPid d = (GPid)b;

	return (c < d)?(-1):(c > d)?(1):(0);
}

static gboolean
this_app_is_already_running (gchar *cmd)
{
	if(runningapps == NULL) return FALSE;
	if(g_list_find_custom (runningapps, (gpointer)cmd, find_by_cmd))
		return TRUE;
	else
		return FALSE;
}

static void
childwatch(GPid pid, gint status, gpointer data)
{
GList *iter;

	if((iter = g_list_find_custom (runningapps, (gpointer)pid, find_by_pid)) != NULL)
	{
		g_free(((RunningApp *)iter->data)->cmd);
		g_free (iter->data);
		runningapps = g_list_delete_link (runningapps, iter);
	}
	g_spawn_close_pid(pid);
}

static void
add_running_app(GPid pid, gchar *cmd)
{
RunningApp *app;

	app = g_new0 (RunningApp, 1);
	app->pid = pid;
	app->cmd = g_strdup(cmd);
	runningapps = g_list_append (runningapps, (gpointer)app);
	g_child_watch_add(pid, childwatch, NULL);
}

static void
execute (gchar *cmd)
{
	gboolean retval = FALSE;
	gchar **argv;
	gint argc;
	GPid child_pid;

	if(this_app_is_already_running(cmd)) return;

	if (g_shell_parse_argv (cmd, &argc, &argv, NULL))
	{
		retval = g_spawn_async (g_get_home_dir (),
			argv, NULL, G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD,
			NULL, NULL, &child_pid, NULL);
		g_strfreev (argv);
	}

	if (retval == FALSE)
	{
		char *msg;

		msg = g_strdup_printf
			(_("Couldn't execute command: %s\n"
			   "Verify that this command exists."),
			 cmd);

		acme_error (msg);
		g_free (msg);
	} else {
		add_running_app(child_pid, cmd);
	}
}

static void
dialog_init (Acme *acme)
{
	if (acme->dialog == NULL) {
		acme->dialog = gsd_media_keys_window_new ();
	}
}

static gboolean
grab_key_real (ShortcutKey *key, GdkWindow *root, gboolean grab, int result)
{
	gdk_error_trap_push ();
	if (grab)
		XGrabKey (GDK_DISPLAY(), key->keycode, (result | key->state),
				GDK_WINDOW_XID (root), True, GrabModeAsync, GrabModeAsync);
	else
		XUngrabKey(GDK_DISPLAY(), key->keycode, (result | key->state),
				GDK_WINDOW_XID (root));
	gdk_flush ();

	gdk_error_trap_pop ();

	return TRUE;
}

/* inspired from all_combinations from gnome-panel/gnome-panel/global-keys.c */
#define N_BITS 32
static void
grab_key (Acme *acme, ShortcutKey *key, gboolean grab)
{
	int indexes[N_BITS];/*indexes of bits we need to flip*/
	int i, bit, bits_set_cnt;
	int uppervalue;
	guint mask_to_traverse = IGNORED_MODS & ~ key->state;

	bit = 0;
	for (i = 0; i < N_BITS; i++) {
		if (mask_to_traverse & (1<<i))
			indexes[bit++]=i;
	}

	bits_set_cnt = bit;

	uppervalue = 1<<bits_set_cnt;
	for (i = 0; i < uppervalue; i++) {
		GSList *l;
		int j, result = 0;

		for (j = 0; j < bits_set_cnt; j++) {
			if (i & (1<<j))
				result |= (1<<indexes[j]);
		}

		for (l = acme->screens; l ; l = l->next) {
			GdkScreen *screen = l->data;
			if (grab_key_real (key, gdk_screen_get_root_window (screen), grab, result) == FALSE)
				return;
		}
	}
}

static gboolean
is_valid_shortcut (const char *string)
{
	if (string == NULL || string[0] == '\0')
		return FALSE;
	if (strcmp (string, "disabled") == 0)
		return FALSE;

	return TRUE;
}

static void
update_kbd_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	Acme *acme = (Acme *) data;
	int i;
	gboolean found = FALSE;

	g_return_if_fail (entry->key != NULL);

	/* Find the key that was modified */
	for (i = 0; i < HANDLED_SHORTCUT_KEYS; i++)
	{
		if (strcmp (entry->key, shortcut_keys_list[i].gconf_key) == 0)
		{
			char *tmp;
			ShortcutKey *key;

			found = TRUE;

			if (shortcut_keys_list[i].key != NULL)
				grab_key (acme, shortcut_keys_list[i].key, FALSE);

			g_free (shortcut_keys_list[i].key);
			shortcut_keys_list[i].key = NULL;

			tmp = gconf_client_get_string (acme->conf_client,
					shortcut_keys_list[i].gconf_key, NULL);

			if (is_valid_shortcut (tmp) == FALSE)
			{
				g_free (tmp);
				break;
			}

			key = g_new0 (ShortcutKey, 1);
			if (egg_accelerator_parse_virtual (tmp, &key->keysym, &key->keycode, &key->state) == FALSE
			    || key->keycode == 0)
			{
				g_free (tmp);
				g_free (key);
				break;
			}

			grab_key (acme, key, TRUE);
			shortcut_keys_list[i].key = key;

			g_free (tmp);

			break;
		}
	}

	if (found != FALSE)
		return;
}

static void
init_screens (Acme *acme)
{
	GdkDisplay *display = gdk_display_get_default ();
	int i;

	if (gdk_display_get_n_screens (display) == 1) {
		GdkScreen *screen = gdk_screen_get_default ();
		acme->screens = g_slist_append (acme->screens, screen);
		acme->current_screen = screen;
		return;
	}

	for (i = 0; i < gdk_display_get_n_screens (display); i++)
	{
		GdkScreen *screen;

		screen = gdk_display_get_screen (display, i);
		if (screen == NULL)
			continue;
		acme->screens = g_slist_append (acme->screens, screen);
	}

	acme->current_screen = (GdkScreen *)acme->screens->data;
}

static void
init_kbd (Acme *acme)
{
	int i;

	for (i = 0; i < HANDLED_SHORTCUT_KEYS; i++)
	{
		char *tmp;
		ShortcutKey *key;

		gconf_client_notify_add (acme->conf_client,
				shortcut_keys_list[i].gconf_key, update_kbd_cb,
				acme, NULL, NULL);

		tmp = gconf_client_get_string (acme->conf_client,
				shortcut_keys_list[i].gconf_key, NULL);
		if (!is_valid_shortcut (tmp)) {
		        g_free (tmp);
			continue;
		}

		key = g_new0 (ShortcutKey, 1);
		if (!egg_accelerator_parse_virtual (tmp, &key->keysym, &key->keycode, &key->state)
		    || key->keycode == 0)
		{
		        g_free (tmp);
			g_free (key);
			continue;
		}
	/*avoid grabbing all the keyboard when KeyCode cannot be retrieved */
		if (key->keycode == AnyKey)
		  {
		    g_warning ("The shortcut key \"%s\" cannot be found on the current system, ignoring the binding", tmp);
		    g_free (tmp);
		    g_free (key);
		    continue;
		  }

		g_free (tmp);

		shortcut_keys_list[i].key = key;

		grab_key (acme, key, TRUE);
	}
}

static void
dialog_show (Acme *acme)
{
	int orig_w, orig_h;
	int screen_w, screen_h;
	int x, y;
	int pointer_x, pointer_y;
	GtkRequisition win_req;
	GdkScreen *pointer_screen;
	GdkRectangle geometry;
	int monitor;

	gtk_window_set_screen (GTK_WINDOW (acme->dialog), acme->current_screen);

	/*
	 * get the window size
	 * if the window hasn't been mapped, it doesn't necessarily
	 * know its true size, yet, so we need to jump through hoops
	 */
	gtk_window_get_default_size (GTK_WINDOW (acme->dialog), &orig_w, &orig_h);
	gtk_widget_size_request (acme->dialog, &win_req);

	if (win_req.width > orig_w)
		orig_w = win_req.width;
	if (win_req.height > orig_h)
		orig_h = win_req.height;

	pointer_screen = NULL;
	gdk_display_get_pointer (gdk_screen_get_display (acme->current_screen),
				 &pointer_screen, &pointer_x,
				 &pointer_y, NULL);
	if (pointer_screen != acme->current_screen) {
		/* The pointer isn't on the current screen, so just
		 * assume the default monitor
		 */
		monitor = 0;
	} else {
		monitor = gdk_screen_get_monitor_at_point (acme->current_screen,
							   pointer_x, pointer_y);
	}

	gdk_screen_get_monitor_geometry (acme->current_screen, monitor,
					 &geometry);

	screen_w = geometry.width;
	screen_h = geometry.height;

	x = ((screen_w - orig_w) / 2) + geometry.x;
	y = geometry.y + (screen_h / 2) + (screen_h / 2 - orig_h) / 2;

	gtk_window_move (GTK_WINDOW (acme->dialog), x, y);

	gtk_widget_show (acme->dialog);

	gdk_display_sync (gdk_screen_get_display (acme->current_screen));
}

static void
do_default_action (Acme *acme, int type)
{
	gchar *string, *command;

	if(BINDCMDKEY(type) == NULL)
		g_assert_not_reached ();

	string = gconf_client_get_string (acme->conf_client,
			BINDCMDKEY(type), NULL);

	if (string == NULL || strcmp (string, "") == 0)
		return;

	command = g_strdup_printf (string, "");

	execute (command);

	g_free (command);
	g_free (string);
}

static void
do_sound_action (Acme *acme, int type)
{
	gboolean muted;
	int vol, sw, pri;
	int vol_step, vol_step_percent, vol_range;
	int has_vol, has_sw;
	static int last_vol = 0;

	vol_step_percent = gconf_client_get_int (acme->conf_client,
					 SNDKEY(VOLUMESHORTCUT_STEP),
					 NULL);

	if (vol_step_percent == 0)
		vol_step_percent = VOLUME_STEP;

	if((pri = pse()) < 0)
		return;

	if(get_support(pri, &has_vol, &has_sw) < 0)
		return;

	if((vol_range=get_volume_range(pri, NULL, NULL)) < 0)
		return;

	vol_step = (vol_range*vol_step_percent)/100;

	switch (type) {
	case MUTE:
		if(has_sw) {
		    get_switch(pri, 0, &sw);
		    set_all_playback_switches_all_channels(!(sw));
		} else {
		    get_volume(pri, 0, &vol);
		    if(vol) {
			last_vol = vol;
			set_all_volume(pri, 0);
		    } else {
			set_all_volume(pri, last_vol);
		    }
		}
		break;
	case VOLUME_DOWN:
		get_volume(pri, 0, &vol);
		set_all_volume(pri, vol - vol_step);
		break;
	case VOLUME_UP:
		get_volume(pri, 0, &vol);
		set_all_volume(pri, vol + vol_step);
		break;
	}

	get_volume(pri, 0, &vol);
	if(has_sw) {
	    get_switch(pri, 0, &sw);
	} else {
	    sw = vol;
	}
	muted = (sw == 0);

	dialog_init (acme);
	gsd_media_keys_window_set_volume_muted (MSD_MEDIA_KEYS_WINDOW (acme->dialog),
						muted);
	gsd_media_keys_window_set_volume_level (MSD_MEDIA_KEYS_WINDOW (acme->dialog),
						vol);
	gsd_media_keys_window_set_action (MSD_MEDIA_KEYS_WINDOW (acme->dialog),
					  MSD_MEDIA_KEYS_WINDOW_ACTION_VOLUME);
	dialog_show (acme);
}

static gboolean
do_multimedia_player_action (Acme *acme, const gchar *key)
{
	return moblin_settings_server_media_player_key_pressed (acme->server, key);
}

static gboolean
do_action (Acme *acme, int type)
{
	switch (type) {
	case MUTE:
	case VOLUME_DOWN:
	case VOLUME_UP:
		do_sound_action (acme, type);
		break;
	case PLAY:
		return do_multimedia_player_action (acme, "Play");
		break;
	case PAUSE:
		return do_multimedia_player_action (acme, "Pause");
		break;
	case STOP:
		return do_multimedia_player_action (acme, "Stop");
		break;
	case PREVIOUS:
		return do_multimedia_player_action (acme, "Previous");
		break;
	case NEXT:
		return do_multimedia_player_action (acme, "Next");
		break;
	case BRIGHTNESS_UP:
	case BRIGHTNESS_DOWN:
	case BRIGHTNESS_MAXMIN:
		do_brightness_action (type);
		break;
	default:
		do_default_action (acme, type);
	}

	return FALSE;
}

static GdkScreen *
acme_get_screen_from_event (Acme *acme, XAnyEvent *xanyev)
{
	GdkWindow *window;
	GdkScreen *screen;
	GSList *l;

	/* Look for which screen we're receiving events */
	for (l = acme->screens; l != NULL; l = l->next)
	{
		screen = (GdkScreen *) l->data;
		window = gdk_screen_get_root_window (screen);

		if (GDK_WINDOW_XID (window) == xanyev->window)
		{
			return screen;
		}
	}

	return NULL;
}

static GdkFilterReturn
acme_filter_events (GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	Acme *acme = (Acme *) data;
	XEvent *xev = (XEvent *) xevent;
	XAnyEvent *xanyev = (XAnyEvent *) xevent;
	guint keycode, state;
	int i;

	/* verify we have a key event */
	if (xev->xany.type != KeyPress &&
			xev->xany.type != KeyRelease)
		return GDK_FILTER_CONTINUE;

	keycode = xev->xkey.keycode;
	state = xev->xkey.state;

	for (i = 0; i < HANDLED_SHORTCUT_KEYS; i++)
	{
		if (shortcut_keys_list[i].key == NULL)
			continue;

		if (shortcut_keys_list[i].key->keycode == keycode &&
				(state & USED_MODS) == shortcut_keys_list[i].key->state)
		{
			switch (shortcut_keys_list[i].key_type) {
			case VOLUME_DOWN:
			case VOLUME_UP:
			case BRIGHTNESS_DOWN:
			case BRIGHTNESS_UP:
				/* auto-repeatable keys */
				if (xev->type != KeyPress)
					return GDK_FILTER_CONTINUE;
				break;
			default:
				if (xev->type != KeyRelease)
					return GDK_FILTER_CONTINUE;
			}

			acme->current_screen = acme_get_screen_from_event
				(acme, xanyev);

			if (do_action (acme, shortcut_keys_list[i].key_type) == FALSE)
				return GDK_FILTER_REMOVE;
			else
				return GDK_FILTER_CONTINUE;
		}
	}

	return GDK_FILTER_CONTINUE;
}

void
moblin_settings_multimedia_keys_init (GConfClient *client)
{
	int i;

	init_alsa_vars();
	for(i = 0; i < HANDLED_SHORTCUT_KEYS; i++)
	{
		shortcut_keys_list[i].key_type = i;
		shortcut_keys_list[i].gconf_key = shortcut_keys[i];
		shortcut_keys_list[i].key = NULL;
	}
}

void
moblin_settings_multimedia_keys_load (GConfClient *client)
{
	GSList *l;
	Acme   *acme;

	acme = g_new0 (Acme, 1);

	acme->conf_client = client;
	gconf_client_add_dir (acme->conf_client,
			      GCONF_BINDING_DIR,
			      GCONF_CLIENT_PRELOAD_ONELEVEL,
			      NULL);

	init_screens (acme);
	init_kbd (acme);

	acme->server = moblin_settings_server_get ();

	/* Start filtering the events */
	for (l = acme->screens; l != NULL; l = l->next)
		gdk_window_add_filter (gdk_screen_get_root_window (l->data),
			acme_filter_events, (gpointer) acme);
}
