/* -*- mode: c; style: linux -*- */

/* moblin-settings-brightness.c
 *
 * Written by Todd Brandt <todd.e.brandt@intel.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "brightness.h"

#define HAL_DBUS_SERVICE                        "org.freedesktop.Hal"
#define HAL_DBUS_PATH_MANAGER                   "/org/freedesktop/Hal/Manager"
#define HAL_DBUS_INTERFACE_MANAGER              "org.freedesktop.Hal.Manager"
#define HAL_DBUS_INTERFACE_DEVICE               "org.freedesktop.Hal.Device"
#define HAL_DBUS_PATH_LAPTOP_PANEL              "/org/freedesktop/Hal/Device/LaptopPanel"
#define HAL_DBUS_INTERFACE_LAPTOP_PANEL         "org.freedesktop.Hal.Device.LaptopPanel"
#define HAL_DBUS_INTERFACE_KBD_BACKLIGHT        "org.freedesktop.Hal.Device.KeyboardBacklight"
#define HAL_DBUS_INTERFACE_LIGHT_SENSOR         "org.freedesktop.Hal.Device.LightSensor"

const int brightval[MAX_BRIGHTNESS_LEVEL-MIN_BRIGHTNESS_LEVEL+1] = {
        10, 30, 40, 50, 60, 70, 80, 100
};

int
brightness_from_level(int level)
{
	int brightness;

	if(level <= MIN_BRIGHTNESS_LEVEL) {
		brightness = brightval[0];
	} else if(level >= MAX_BRIGHTNESS_LEVEL) {
		brightness = brightval[MAX_BRIGHTNESS_LEVEL-MIN_BRIGHTNESS_LEVEL];
	} else {
		brightness = brightval[level-MIN_BRIGHTNESS_LEVEL];
	}

	return brightness;
}

int
level_from_brightness(int brightness)
{
	int i, level;
	int levels = MAX_BRIGHTNESS_LEVEL-MIN_BRIGHTNESS_LEVEL;

	/* this code assumes that the brightval array is */
	/* ordered from lowest to highest */
	if(brightness <= brightval[0]) {
		level = MIN_BRIGHTNESS_LEVEL;
	} else if(brightness > brightval[levels]) {
		level = MAX_BRIGHTNESS_LEVEL;
	} else {
		/* ASSUMPTION:                                    */
		/* brightval[0] > brightness <= brightval[levels] */

		for(i = 1; brightness > brightval[i]; i++);

		/* ASSUMPTION                                     */
		/* brightval[i-1] > brightness <= brightval[i]    */

		if((brightness == brightval[i]) ||
		   ((brightval[i] - brightness) < (brightness - brightval[i-1])))
		{
			level = i;
		} else {
			level = i-1;
		}
	}

	return level;
}

gboolean
get_brightness (BrightnessControl *bc)
{
	GError  *error = NULL;
	gboolean ret;
	int brightness = 0;

	if (bc->proxy == NULL) {
		printf ("WARNING: not connected\n");
		return FALSE;
	}

	ret = dbus_g_proxy_call (bc->proxy, "GetBrightness", &error,
				 G_TYPE_INVALID,
				 G_TYPE_INT, &brightness,
				 G_TYPE_INVALID);
	if (error) {
		printf ("DEBUG: ERROR: %s\n", error->message);
		g_error_free (error);
	}
	if (ret == TRUE) {
		bc->level = level_from_brightness(brightness);
	} else {
		/* abort as the DBUS method failed */
		printf ("WARNING: GetBrightness failed!\n");
	}

	return ret;
}

gboolean
set_brightness (BrightnessControl *bc)
{
	GError  *error = NULL;
	gboolean ret;
	int retval;

	if (bc->proxy == NULL) {
		printf ("WARNING: not connected");
		return FALSE;
	}

	ret = dbus_g_proxy_call (bc->proxy, "SetBrightness", &error,
	           G_TYPE_INT, (int)brightness_from_level(bc->level),
		   G_TYPE_INVALID, G_TYPE_INT, &retval, G_TYPE_INVALID);

	if (error || ret == FALSE) {
		g_warning("SetBrighness Failed: %s\n", 
			  error ? error->message : "unknown error");
		g_error_free (error);
	}

	return ret;
}

gboolean
brightness_dbus_connect (BrightnessControl *bc)
{
	DBusGProxy *proxy = NULL;
	GError *error = NULL;
	gchar **names = NULL;
	int i;

	if(bc->connection == NULL)
	{
	  bc->connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	  if (error) {
	    printf ("WARNING: Could not connect to DBUS daemon: %s", 
	      error->message);
	    g_error_free (error);
	    bc->connection = NULL;
	    return FALSE;
	  }
	  proxy = dbus_g_proxy_new_for_name (bc->connection,
	    HAL_DBUS_SERVICE, HAL_DBUS_PATH_MANAGER, 
	    HAL_DBUS_INTERFACE_MANAGER);
	  dbus_g_proxy_call (proxy, "FindDeviceByCapability", &error,
	    G_TYPE_STRING, "laptop_panel", G_TYPE_INVALID,
	    G_TYPE_STRV, &names, G_TYPE_INVALID);
	  if (error) {
	    g_warning ("%s", error->message);
	    g_error_free (error);
	    bc->connection = NULL;
	    return FALSE;
	  }
	  if (names == NULL || names[0] == NULL) {
	    g_warning ("No devices of capability laptop_panel");
	    if(names) g_free(names);
	    bc->connection = NULL;
	    return FALSE;
	  }
	  bc->udi = g_strdup (names[0]);
	  for (i=0; names[i]; i++) {
	    g_free (names[i]);
	  }
	  g_free (names);
	}

	if (bc->proxy == NULL) {
	  bc->proxy = dbus_g_proxy_new_for_name(bc->connection,
	    HAL_DBUS_SERVICE, bc->udi,
	    HAL_DBUS_INTERFACE_LAPTOP_PANEL);
	  if (bc->proxy == NULL) {
	    g_warning("Cannot connect, maybe the daemon is not running\n");
	    return FALSE;
	  }

	  if(bc->brightness_changed_cb != NULL)
	  {
	    dbus_g_proxy_add_signal (bc->proxy, "BrightnessChanged",
	      G_TYPE_UINT, G_TYPE_INVALID);
	    dbus_g_proxy_connect_signal (bc->proxy, "BrightnessChanged",
	      G_CALLBACK (bc->brightness_changed_cb), bc, NULL);
	  }
	  get_brightness (bc);
	}
	return TRUE;
}

gboolean
brightness_dbus_disconnect (BrightnessControl *bc)
{
	if (bc->proxy != NULL) {
		printf ("DEBUG: removing proxy\n");
		g_object_unref (bc->proxy);
		bc->proxy = NULL;
	}
	return TRUE;
}
