# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Main peaklist panel.

# load libs
import wx
import wx.lib.mixins.listctrl as listmix
import os

# load modules
from nucleus import mwx
from dlg_editpeak import dlgEditPeak
from dlg_treshold import dlgTreshold


class mPeak(wx.ListCtrl, listmix.ListCtrlAutoWidthMixin):
    """ Create and manipulate peaklist. """

    # ----
    def __init__(self, parent, document):

        # set style
        if wx.Platform == '__WXMAC__':
            style=wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.SIMPLE_BORDER
        else:
            style=wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.SUNKEN_BORDER
            
        wx.ListCtrl.__init__(self, parent, -1, style=style)
        listmix.ListCtrlAutoWidthMixin.__init__(self)

        # set smaller font for Mac
        if wx.Platform == '__WXMAC__':
            self.SetFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))

        self.parent = parent
        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document

        # init variables
        self.selectedPeak = -1

        # make peaklist
        self.makePeakList()
    # ----


    # ----
    def makePeakList(self):
        """ Make list of peaks. """

        # row height hack for MSW
        if wx.Platform == '__WXMSW__':
          il = wx.ImageList(1, 16)
          self.SetImageList(il, wx.IMAGE_LIST_SMALL)

        # make header
        self.InsertColumn(0, "#")
        self.InsertColumn(1, "m/z", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(2, "Intens.", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(3, "Annotations", wx.LIST_FORMAT_LEFT)

        # set columns width
        for col in range(4):
            self.SetColumnWidth(col, wx.LIST_AUTOSIZE_USEHEADER)

        # set events
        self.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onPeakSelected)
        self.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.onPeakDeselected)
        self.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onEditPeak)
        self.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onRMU)
        self.Bind(wx.EVT_RIGHT_UP, self.onRMU)
        self.Bind(wx.EVT_COMMAND_RIGHT_CLICK, self.onRMU)
        self.Bind(wx.EVT_CHAR, self.onChar)
        self.Bind(wx.EVT_LEFT_DCLICK, self.onLMDC)
    # ----


    # ----
    def resetModule(self):
        """ Get data from document and update all fields. """
        self.updatePeakList()
    # ----


    # ----
    def updatePeakList(self, visible=None):
        """ Update values in shown peaklist. """

        # set application working
        self.docMonitor('setAppStatus', "Refreshing peaklist...")

        # clear list
        self.DeleteAllItems()

        # get data
        data = self.docData.getPeaks()

        # paste results
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for x in range(self.docData.getPeaklistLength()):

            # format data
            mass = digitsFormat % data[x][0]
            intensity = digitsFormat % data[x][1]
            annotations = data[x][2]

            # show data
            self.InsertStringItem(x, str(x+1))
            self.SetStringItem(x, 1, mass)
            self.SetStringItem(x, 2, intensity)
            self.SetStringItem(x, 3, annotations)

        # set columns width
        if self.GetItemCount():
            autosize = wx.LIST_AUTOSIZE
        else:
            autosize = wx.LIST_AUTOSIZE_USEHEADER
        for col in range(4):
            self.SetColumnWidth(col, autosize)

        # autosize last coll
        lastCol = self.GetColumnCount()
        width = self.GetColumnWidth(lastCol-1)
        self.resizeLastColumn(width)

        # ensure visible last edited peaks
        if visible != None:
            self.EnsureVisible(visible)
            self.SetItemState(visible, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
        else:
            self.selectedPeak = -1

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onPeakSelected(self, evt):
        """ Remember last selected peak. """

        self.selectedPeak = evt.m_itemIndex
        evt.Skip()
    # ----


    # ----
    def onPeakDeselected(self, evt):
        """ Forgot last selected peak. """

        self.selectedPeak = -1
        evt.Skip()
    # ----


    # ----
    def onRMU(self, evt):
        """ Rise popup menu when right-click on peaklist. """

        # make menu items
        menuAddPeakID = wx.NewId()
        menuEditPeakID = wx.NewId()
        menuShowPeakID = wx.NewId()
        menuClearSelectedAnnotsID = wx.NewId()
        menuClearAllAnnotsID = wx.NewId()
        menuDeleteSelectedPeaksID = wx.NewId()
        menuDeleteAllPeaksID = wx.NewId()
        menuSelectAllID = wx.NewId()
        menuSelectAnnotatedID = wx.NewId()
        menuSelectByTresholdID = wx.NewId()
        menuInvertSelectionID = wx.NewId()
        menuCopyToClipboardID = wx.NewId()
        menuRefreshID = wx.NewId()

        # add items to menu
        menu = wx.Menu()
        menu.Append(menuAddPeakID, "Add Peak Manually...")
        menu.Append(menuEditPeakID, "Edit Peak...")
        menu.Append(menuShowPeakID, "Show Peak in Spectrum")
        menu.AppendSeparator()
        menu.Append(menuSelectAllID, "Select All")
        menu.Append(menuSelectAnnotatedID, "Select Annotated")
        menu.Append(menuSelectByTresholdID, "Select by Treshold...")
        menu.Append(menuInvertSelectionID, "Invert Selection")
        menu.AppendSeparator()
        menu.Append(menuClearSelectedAnnotsID, "Clear Selected Annotations")
        menu.Append(menuClearAllAnnotsID, "Clear All Annotations")
        menu.AppendSeparator()
        menu.Append(menuDeleteSelectedPeaksID, "Delete Selected Peaks")
        menu.Append(menuDeleteAllPeaksID, "Delete All Peaks")
        menu.AppendSeparator()
        menu.Append(menuCopyToClipboardID, "Copy to Clipboard")
        menu.AppendSeparator()
        menu.Append(menuRefreshID, "Refresh")

        # bind events to menu
        self.Bind(wx.EVT_MENU, self.onAddPeak, id=menuAddPeakID)
        self.Bind(wx.EVT_MENU, self.onEditPeak, id=menuEditPeakID)
        self.Bind(wx.EVT_MENU, self.onHighlightPeak, id=menuShowPeakID)
        self.Bind(wx.EVT_MENU, self.onSelectAllPeaks, id=menuSelectAllID)
        self.Bind(wx.EVT_MENU, self.onSelectAnnotatedPeaks, id=menuSelectAnnotatedID)
        self.Bind(wx.EVT_MENU, self.onSelectByTreshold, id=menuSelectByTresholdID)
        self.Bind(wx.EVT_MENU, self.onInvertSelection, id=menuInvertSelectionID)
        self.Bind(wx.EVT_MENU, self.onCopyToClipboard, id=menuCopyToClipboardID)
        self.Bind(wx.EVT_MENU, self.onClearSelectedAnnots, id=menuClearSelectedAnnotsID)
        self.Bind(wx.EVT_MENU, self.onClearAllAnnots, id=menuClearAllAnnotsID)
        self.Bind(wx.EVT_MENU, self.onDeleteSelectedPeaks, id=menuDeleteSelectedPeaksID)
        self.Bind(wx.EVT_MENU, self.onDeleteAllPeaks, id=menuDeleteAllPeaksID)
        self.Bind(wx.EVT_MENU, self.onRefresh, id=menuRefreshID)

        # disable some fce if no item selected
        if self.selectedPeak == -1:
            menu.Enable(menuEditPeakID, False)
            menu.Enable(menuShowPeakID, False)
            menu.Enable(menuClearSelectedAnnotsID, False)
            menu.Enable(menuDeleteSelectedPeaksID, False)

        # disable some additional fce if no data
        if not self.docData.getDataStatus('mPeak'):
            menu.Enable(menuSelectAllID, False)
            menu.Enable(menuSelectAnnotatedID, False)
            menu.Enable(menuSelectByTresholdID, False)
            menu.Enable(menuClearAllAnnotsID, False)
            menu.Enable(menuDeleteAllPeaksID, False)
            menu.Enable(menuCopyToClipboardID, False)

        self.PopupMenu(menu)
        menu.Destroy()
        self.SetFocus()
    # ----


    # ----
    def onLMDC(self, evt):
        """ Raise Add dialog when LMDC on free space. """

        # add new peak
        if self.selectedPeak == -1:
            self.onAddPeak()

        # edit current peak
        else:
            self.onEditPeak()
    # ----


    # ----
    def onChar(self, evt):
        """ On key pressed. """

        # get key
        key = evt.GetKeyCode()

        # clear annotations
        if key == wx.WXK_DELETE and evt.m_shiftDown:
            self.onClearSelectedAnnots()

        # delete selected peaks
        elif key == wx.WXK_DELETE:
            self.onDeleteSelectedPeaks()

        # skip other keys
        else:
            evt.Skip()
    # ----


    # ----
    def onAddPeak(self, evt=None):
        """ Rise Add dialog and add new peak. """

        # raise dialog
        peak = [0, 0, '', 0]
        dlg = dlgEditPeak(self, "Add Peak", peak)
        if dlg.ShowModal() == wx.ID_OK:

            # add peak to document peaklist
            self.docData.addPeak(dlg.data)
            dlg.Destroy()

            # update application
            self.docMonitor('onPeaklistChanged', 'add')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onEditPeak(self, evt=None):
        """ Rise Edit dialog and edit peak. """

        # get peak
        peak = self.docData.getPeaks(self.selectedPeak)

        # raise dialog
        dlg = dlgEditPeak(self, "Edit Peak", peak)
        if dlg.ShowModal() == wx.ID_OK:

            # update peak in document peaklist
            self.docData.editPeak(self.selectedPeak, dlg.data)
            dlg.Destroy()

            # update peaklist
            self.updatePeakList(visible=self.selectedPeak)

            # update application
            self.docMonitor('onPeaklistChanged', 'edit')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onHighlightPeak(self, evt=None):
        """ Show selected peak in the spectrum. """

        peak = self.docData.getPeaks(self.selectedPeak)
        self.parent.onSpectrumHighlightPoint(peak[0])
    # ----


    # ----
    def onSelectAllPeaks(self, evt=None):
        """ Select all peaks. """

        # walk through the list
        i = -1
        while True:
            i = self.GetNextItem(i, wx.LIST_NEXT_ALL)
            if i == -1:
                break
            else:
                self.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
                self.selectedPeak = i
    # ----


    # ----
    def onSelectAnnotatedPeaks(self, evt=None):
        """ Select all annotated peaks. """

        # get peaklist data
        data = self.docData.getPeaks()

        # walk through the list
        i = -1
        self.selectedPeak = -1
        for i, peak in enumerate(data):
            if peak[2] != '':
                self.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
                self.selectedPeak = i
            else:
                self.SetItemState(i, 0, wx.LIST_STATE_SELECTED)
    # ----


    # ----
    def onSelectByTreshold(self, evt=None):
        """ Select peaks above intensity treshold. """

        # raise dialog and get params
        treshold = self.config.cfg['mpeak']['treshold']
        direction = self.config.cfg['mpeak']['thdirection']
        dlg = dlgTreshold(self, treshold, direction)
        if dlg.ShowModal() != wx.ID_OK:
            dlg.Destroy()
            return False
        else:
            treshold, direction = dlg.getParams()
            dlg.Destroy()

        # get peaklist data
        data = self.docData.getPeaks()

        # walk through the list
        i = -1
        self.selectedPeak = -1
        for i, peak in enumerate(data):
            if direction == 'above' and peak[1] >= treshold:
                self.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
                self.selectedPeak = i
            elif direction == 'below' and peak[1] <= treshold:
                self.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
                self.selectedPeak = i
            else:
                self.SetItemState(i, 0, wx.LIST_STATE_SELECTED)
    # ----


    # ----
    def onInvertSelection(self, evt=None):
        """ Invert current selection. """

        # walk through the list
        i = -1
        self.selectedPeak = -1
        while True:
            i = self.GetNextItem(i, wx.LIST_NEXT_ALL)
            if i == -1:
                break
            else:
                if self.GetItemState(i, wx.LIST_STATE_SELECTED):
                    self.SetItemState(i, 0, wx.LIST_STATE_SELECTED)
                else:
                    self.SetItemState(i, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
                    self.selectedPeak = i
    # ----


    # ----
    def onClearSelectedAnnots(self, evt=None):
        """ Clear annotations of selected peaks. """

        # get selected peaks
        indexes = mwx.getSelectedListItems(self)
        if not indexes:
            return

        # raise confirm dialog
        dlg = wx.MessageDialog(self, "Delete annotations for selected peaks?", "Clear Annotations", wx.YES_NO|wx.ICON_QUESTION|wx.YES_DEFAULT)
        if dlg.ShowModal() == wx.ID_YES:
            dlg.Destroy()

            # clear selected annotations in document peaklist
            self.docData.deleteAnnotations(indexes)

            # update application
            self.docMonitor('onPeaklistChanged', 'clear')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onClearAllAnnots(self, evt=None):
        """ Clear annotations for all peaks. """

        # raise confirm dialog
        dlg = wx.MessageDialog(self, "Delete all annotations?", "Clear All Annotations", wx.YES_NO|wx.ICON_QUESTION|wx.YES_DEFAULT)
        if dlg.ShowModal() == wx.ID_YES:
            dlg.Destroy()

            # clear all annotations in document peaklist
            self.docData.deleteAnnotations()

            # update application
            self.docMonitor('onPeaklistChanged', 'clear')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onDeleteSelectedPeaks(self, evt=None):
        """ Delete selected peaks. """

        # get selected peaks
        indexes = mwx.getSelectedListItems(self, reverse=True)
        if not indexes:
            return

        # raise confirm dialog
        dlg = wx.MessageDialog(self, "Delete selected peaks?", "Delete Peaks", wx.YES_NO|wx.ICON_QUESTION|wx.YES_DEFAULT)
        if dlg.ShowModal() == wx.ID_YES:
            dlg.Destroy()

            # delete selected peaks in document peaklist
            self.docData.deletePeaks(indexes)

            # update application
            self.docMonitor('onPeaklistChanged', 'delete')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onDeleteAllPeaks(self, evt=None):
        """ Delete all peaks in peaklist. """

        # raise confirm dialog
        dlg = wx.MessageDialog(self, "Delete all peaks in the peaklist?", "Erase Peaklist", wx.YES_NO|wx.ICON_QUESTION|wx.YES_DEFAULT)
        if dlg.ShowModal() == wx.ID_YES:
            dlg.Destroy()

            # erase document peaklist
            self.docData.deletePeaks()

            # update application
            self.docMonitor('onPeaklistChanged', 'erase')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onCopyToClipboard(self, evt=None):
        """ Copy selected peaks data to clipboard. """

        # get selected items
        indexes = mwx.getSelectedListItems(self)

        # get params
        delimiter = self.config.cfg['mpeak']['delimiter']
        copymass = self.config.cfg['mpeak']['copymass']
        copyintens = self.config.cfg['mpeak']['copyintens']
        copyannot = self.config.cfg['mpeak']['copyannot']
        format = '%0.' + `self.config.cfg['common']['digits']` + 'f'

        # check tab delimiter
        if delimiter == 'tab':
            delimiter = '\t'

        # get peaklist data
        peaklist = self.docData.getPeaks()

        # format peaklist
        buff = ''
        count = 0
        for x, peak in enumerate(peaklist):
            if not indexes or x in indexes:
                addDelimiter = False
                count += 1

                # copy values to buffer
                if copymass:
                    buff += format % peak[0]
                    addDelimiter = True
                if copyintens:
                    if addDelimiter:
                        buff += delimiter
                    buff += format % peak[1]
                    addDelimiter = True
                if copyannot:
                    if addDelimiter:
                        buff += delimiter
                    buff += '%s' % peak[2]

                buff += os.linesep

        # make text object for data
        data = wx.TextDataObject()
        data.SetText(buff)

        # paste to clipboard
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(data)
            wx.TheClipboard.Close()

            # raise OK info
            message = 'Peak(s) were copied successfully.\n%d peak(s) copied.' % (count)
            dlg = wx.MessageDialog(self, message, "Copy Peaklist", wx.OK|wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()
        else:
            # raise BAD info
            message = "Clipboard error! Cannot copy data to clipboard."
            dlg = wx.MessageDialog(self, message, "Copy Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def onRefresh(self, evt=None):
        """ Refresh shown peaklist. """
        self.updatePeakList()
    # ----
