/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//*****************************************************************************
//
// File:		MEStatus.cpp
//
// Description:		The main file of the MEStatus application.
//
//*****************************************************************************

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#include <gtkmm.h>
#include <libglademm.h>
#include <glibmm/i18n.h>
#include "MEStatusDlg.h"
#include "IPCAMTStatus.h"
#include "IPCBuffer.h"
#include "AMTStatusStateMachine.h"

#define GETTEXT_PACKAGE "MEStatus"
#define PROGRAMNAME_LOCALEDIR "MEStatusLocale"

/* MEStatusApp: This class is the main class of the MEStatus application. It holds all the
 * actual objects that are used: the MEStatus dialog (GUI), the AMT Status State machine (the logic)
 * and the IPC AMT Status object, which is used to communicate with the UNS daemon.
 * This class is the glue that holds everything together: It supplies AMTStatusStateMachine with a 
 * pointer to the dialog, through the IMEGuiNotifications interface, and it also supplies MEStatusDlg 
 * with a pointer to the state machine, through the IMEGuiCommands interface.
 */
class MEStatusApp
{
public:
  //ctor
  MEStatusApp();
  //dtor
  virtual ~MEStatusApp();

  //run: Runs the MEStatus application with the supplied command line parameters, and returns 
  //an error code for the OS.
  int run(int argc, char *argv[]);

  //APP_ERROR: This enum defines the return values for run()
  enum APP_ERROR
  {
    ERROR_NONE = 0,
    ERROR_NO_GLADE_FILE = 1,
    ERROR_NO_WIDGET = 2,
    ERROR_SM_INIT_FAIL = 3,
    ERROR_GUI_INIT_FAIL = 4,
    ERROR_INIT_LOCK_FAIL = 5
  };

  /*Implementation*/
protected:
  //The timer function that will be called periodically
  virtual bool UpdateStatus();

private:
  MEStatusDlg *m_pDialog; //The GUI
  IPCAMTStatus m_IPCAMTStatus; //The interface to communicate with UNS
  AMTStatusStateMachine m_AMTStatusStateMachine; //The logic
  sigc::connection m_TimerConnection; //The object that maintains the connection to the periodic timer
  int mestatus_lock_fd; //Lock file
  char mestatus_lock_fname[256];

/*Constants*/
  static const Glib::ustring GLADE_FILE;
};

const Glib::ustring MEStatusApp::GLADE_FILE = "res/MEStatus.glade";

//ctor
MEStatusApp::MEStatusApp()
{
  m_pDialog = NULL;
  mestatus_lock_fd = -1;
}

//dtor
MEStatusApp::~MEStatusApp()
{
  //Stop periodic timer
  m_TimerConnection.disconnect();
  //Remove lock
  if (-1 != mestatus_lock_fd) {
    flock(mestatus_lock_fd, LOCK_UN);
    close(mestatus_lock_fd);
    unlink(mestatus_lock_fname);
  }
}

int MEStatusApp::run(int argc, char *argv[])
{
  //Initialize GTK
  Gtk::Main kit(argc, argv);

  //One instance lock
  Glib::RefPtr<Gdk::Display> mydisplay = Gdk::Display::get_default();
  if (mydisplay) {
    int lfd;
    g_snprintf(mestatus_lock_fname, sizeof(mestatus_lock_fname), "/tmp/intel_MEStatus_%s.lock", mydisplay->get_name().c_str());
    umask(0);
    lfd = creat(mestatus_lock_fname, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);
    if (-1 == lfd) {
      Gtk::MessageDialog::MessageDialog(_("Unable to open MEStatus lock file."), false, Gtk::MESSAGE_ERROR).run();
      return ERROR_INIT_LOCK_FAIL;
    }
    if (-1 == flock(lfd, LOCK_EX | LOCK_NB)) {
      Gtk::MessageDialog::MessageDialog(_("Another instance of MEStatus is already started."), false, Gtk::MESSAGE_ERROR).run();
      return ERROR_INIT_LOCK_FAIL;
    }
    mestatus_lock_fd = lfd;
  }

  //Read resources from glade file
  Glib::RefPtr<Gnome::Glade::Xml> refXml;
  try
  {
    refXml = Gnome::Glade::Xml::create(GLADE_FILE);
  }
  catch (Gnome::Glade::XmlError &)
  {
    Gtk::MessageDialog::MessageDialog(_("Couldn't load glade file: ") + GLADE_FILE, false, Gtk::MESSAGE_ERROR).run();
    return ERROR_NO_GLADE_FILE;
  }

  //Create GUI, but don't initialize it yet.
  refXml->get_widget_derived("MEStatusDlg", m_pDialog);
  if (!m_pDialog)
  {
    Gtk::MessageDialog::MessageDialog(_("Couldn't get MEStatusDlg widget"), false, Gtk::MESSAGE_ERROR).run();
    return ERROR_NO_WIDGET;
  }

  //Initialize AMT Status State Machine
  if (!m_AMTStatusStateMachine.Init(m_pDialog))
  {
    Gtk::MessageDialog::MessageDialog(_("Failed to initialize AMT Status State Machine"), false, Gtk::MESSAGE_ERROR).run();
    return ERROR_SM_INIT_FAIL;
  }

  //Initialize IPCAMTStatus object. 
  m_IPCAMTStatus.init(false); //false meanse initialize as client
  //We ignore errors on initialization (at this point).

  //Initialize GUI
  Glib::ustring strError;
  if (!m_pDialog->Init(&m_AMTStatusStateMachine, strError))
  {
    Gtk::MessageDialog::MessageDialog(strError, false, Gtk::MESSAGE_ERROR).run();
    return ERROR_GUI_INIT_FAIL;
  }

  //Install a timer that will call UpdateStatus() every X milliseconds 
  //(according to configuration).
  m_TimerConnection = 
    Glib::signal_timeout().connect(sigc::mem_fun(*this, &MEStatusApp::UpdateStatus), 
				   m_AMTStatusStateMachine.GetCheckAMTStatusTimeout());

  //We update the status once in the beginning, and later it'll be on the timer.
  UpdateStatus();

  //reconnect to know if UNS is still running
  m_IPCAMTStatus.setShmReconnect(true);

  /**** MAIN GUI LOOP ****/
  kit.run();
  /***********************/

  return ERROR_NONE;
}

bool MEStatusApp::UpdateStatus()
{
  if (m_pDialog)
  {
    m_pDialog->SetEnablement(m_IPCAMTStatus.GetEnablement());
  }

  return true; //Tell the framework this is a periodic timer
}

int main(int argc, char *argv[])
{
  //Chdir to application folder
  {
    char exelink[256];
    char exefile[PATH_MAX];
    char *base = NULL;
    snprintf(exelink, sizeof(exelink), "/proc/%lu/exe", getpid());
    if (-1 == readlink(exelink, exefile, PATH_MAX)) {
      strncpy(exefile, argv[0], (PATH_MAX - 1));
      exefile[(PATH_MAX - 1)] = '\0';
    }
    base = strrchr(exefile, '/');
    if (NULL != base) {
      *base = '\0';
      chdir(exefile);
    }
  }

  //To let the program use the translation for the current locale
  bindtextdomain(GETTEXT_PACKAGE, PROGRAMNAME_LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);

  //Create instance of application
  MEStatusApp App;
  
  /* Run application */
  return App.run(argc, argv);
}
 
