/****************************************************************************
* COPYRIGHT (C) 1999 - 2007  EDF R&D, CEA/DEN
* THIS LIBRARY IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
* IT UNDER THE TERMS OF THE GNU LESSER GENERAL PUBLIC LICENSE 
* AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
* EITHER VERSION 2.1 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
*  
* THIS LIBRARY IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
* WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
* MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
* LESSER GENERAL PUBLIC LICENSE FOR MORE DETAILS.
*
* YOU SHOULD HAVE RECEIVED A COPY OF THE GNU LESSER GENERAL PUBLIC LICENSE
* ALONG WITH THIS LIBRARY; IF NOT, WRITE TO THE FREE SOFTWARE FOUNDATION,
* INC., 59 TEMPLE PLACE, SUITE 330, BOSTON, MA 02111-1307 USA
*
******************************************************************************/

/******************************************************************************
 * - Nom du fichier : mdump.c
 *
 * - Description : utilitaire de dump pour fichier MED
 *                 Ce fichier contient les fonctions suivantes
 *                 qui constituent des modeles de programmation
 *                 pour une lecture generique d'un fichier MED :
 *                 - lecture_maillage_non_structure () :
 *                        1. Noeuds.
 *                        2. Mailles.
 *                        3. Faces (connectivite descendante).
 *                        4. Aretes (connectivite descendante).
 *                        5. Familles.
 *                        6. Equivalences.
 *                        7. Joints.
 *                 - lecture_maillage_structure ()     : 
 *                        1. Noeuds.
 *                        2. Mailles.
 *                        3. Familles.
 *                        4. Equivalences.
 *                        5. Joints.
 *                 - lecture_resultats () :
 *                        1. Champs de resultats relatifs  un maillage.
 *                           - Entites :
 *                                - Noeuds
 *                                - Mailles
 *                                - Faces
 *                                - Aretes
 *                           - Gestion des pas de temps et numeros d'ordre :
 *                                  valeurs associees a un ou plusieurs maillages sous 
 *                                  un meme pas de temps.
 *                           - Gestion des profils.
 *                           - Gestion des liens vers des maillages distants
 *                           - Gestion des points de Gauss :
 *                                - localisation des points de Gauss.
 *                 - lecture_parametres_scalaires () : 
 *                           - Valeurs scalaires entieres ou flottantes.
 *                           - Gestion des pas de temps et numeros d'ordre.
 *                 - main() : infos generales + lecture de tous les champs et 
 *                            du fichier MED passe en parametre.
 *
 *****************************************************************************/

#define MESGERR

#ifdef __cplusplus
extern "C" {
#endif

#include <med.h>
#include <med_config.h>
#include <med_utils.h>
#include <med_misc.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifdef __cplusplus
}
#endif

#ifdef PPRO_NT
#define F_OK 0
#else
#include <unistd.h>
#endif

/* indique si on ecrit seulement la structure */
int structure = 0;

/* types geometriques des mailles references dans le modele MED */
med_geometrie_element typmai[MED_NBR_GEOMETRIE_MAILLE] = { MED_POINT1,
							   MED_SEG2,
							   MED_SEG3,
							   MED_TRIA3,
							   MED_TRIA6,
							   MED_QUAD4,
							   MED_QUAD8,
							   MED_TETRA4,
							   MED_TETRA10,
							   MED_HEXA8,
							   MED_HEXA20,
							   MED_PENTA6,
							   MED_PENTA15,
							   MED_PYRA5,
							   MED_PYRA13};
char nommai[MED_NBR_GEOMETRIE_MAILLE] [MED_TAILLE_NOM+1] = {"MED_POINT1",
							    "MED_SEG2",
							    "MED_SEG3",
							    "MED_TRIA3",
							    "MED_TRIA6",
							    "MED_QUAD4",
							    "MED_QUAD8",
							    "MED_TETRA4",
							    "MED_TETRA10",
							    "MED_HEXA8",
							    "MED_HEXA20",
							    "MED_PENTA6",
							    "MED_PENTA15",
							    "MED_PYRA5",
							    "MED_PYRA13"};
med_geometrie_element typfac[MED_NBR_GEOMETRIE_FACE] = {MED_TRIA3,
							MED_TRIA6,
							MED_QUAD4,
							MED_QUAD8};
char nomfac[MED_NBR_GEOMETRIE_FACE][MED_TAILLE_NOM+1] = {"MED_TRIA3",
							 "MED_TRIA6",
							 "MED_QUAD4",
							 "MED_QUAD8"};
med_geometrie_element typare[MED_NBR_GEOMETRIE_ARETE] = {MED_SEG2,
							 MED_SEG3};
char nomare[MED_NBR_GEOMETRIE_ARETE] [MED_TAILLE_NOM+1] = {"MED_SEG2",
							   "MED_SEG3"};

#define USER_MODE MED_COMPACT

med_int lecture_nombre_famille(med_idt fid,char *nommaa)
{
  med_int nfam = MEDnFam(fid,nommaa);
  EXIT_IF(nfam < 0,"lors de la lecture du nombre de familles",NULL);
  fprintf(stdout,"- Nombre de familles : %d \n",nfam);

  return nfam;
}

void lecture_famille_maillage(med_idt fid,char *nommaa,med_int nfam)
{
  med_int i,j;
  med_int natt,ngro;
  char *attdes,*gro;
  med_int *attval,*attide;
  char nomfam[MED_TAILLE_NOM+1];
  med_int numfam;
  char str1[MED_TAILLE_DESC+1];
  char str2[MED_TAILLE_LNOM+1];
  med_err ret = 0;
  int famille_0 = 0;

  fprintf(stdout,"\n(**************************)\n");
  fprintf(stdout,"(* FAMILLES DU MAILLAGE : *)\n");
  fprintf(stdout,"(**************************)\n");

  for (i=0;i<nfam;i++) {

    /* nombre de groupes */
    ngro = MEDnGroupe(fid,nommaa,i+1);
    EXIT_IF(ngro < 0,"lors de la lecture du nombre de groupe d'une famille",
	    NULL);

    /* nombre d'attributs */
    natt = MEDnAttribut(fid,nommaa,i+1);
    EXIT_IF(natt < 0,"lors de la lecture du nombre d'attributs d'une famille",
	    NULL);

    fprintf(stdout,"- Famille %d a %d attributs et %d groupes \n",i+1,natt,
	    ngro);

    /* nom,numero,attributs,groupes */

    /* allocation memoire */
    attide = (med_int*) malloc(sizeof(med_int)*natt);
    EXIT_IF(attide == NULL,NULL,NULL);
    attval = (med_int*) malloc(sizeof(med_int)*natt);
    EXIT_IF(attval == NULL,NULL,NULL);
    attdes = (char *) malloc(MED_TAILLE_DESC*natt+1);
    EXIT_IF(attdes == NULL,NULL,NULL);
    gro = (char*) malloc(MED_TAILLE_LNOM*ngro+1);
    EXIT_IF(gro == NULL,NULL,NULL);

    ret = MEDfamInfo(fid,nommaa,i+1,nomfam,&numfam,attide,attval,
		     attdes,&natt,gro,&ngro);
    EXIT_IF(ret < 0,"lors de la lecture des informations d'une famille",
	    NULL);
	if (numfam == 0)
	  famille_0 = 1;

    if (!structure) {
    /* affichage des resultats */
    fprintf(stdout,"  - Famille de nom %s et de numero %d : \n",nomfam,numfam);
    fprintf(stdout,"  - Attributs : \n");
    for (j=0;j<natt;j++) {
      strncpy(str1,attdes+j*MED_TAILLE_DESC,MED_TAILLE_DESC);
      str1[MED_TAILLE_DESC] = '\0';
      fprintf(stdout,"   ide = %d - val = %d - des = %s\n",*(attide+j),
	      *(attval+j),str1);
    }
    }

    /* on libere la memoire */
    free(attide);
    free(attval);
    free(attdes);

    if (!structure) {
    fprintf(stdout,"  - Groupes :\n");
    for (j=0;j<ngro;j++) {
      strncpy(str2,gro+j*MED_TAILLE_LNOM,MED_TAILLE_LNOM);
      str2[MED_TAILLE_LNOM] = '\0';
      fprintf(stdout,"   gro = %s\n",str2);
    }
    }

    /* on libere la memoire */
    free(gro);
  }
  EXIT_IF(famille_0 != 1,"Pas de famille de numro 0",
	    NULL);

  return;
}

med_int lecture_nombre_equivalence(med_idt fid,char *nommaa)
{
  med_int nequ = MEDnEquiv(fid,nommaa);
  EXIT_IF(nequ < 0,"lors de la lecture du nombre d'equivalences",NULL);
  fprintf(stdout,"- Nombre d'equivalences : %d \n",nequ);

  return nequ;
}

/* nombre de mailles concernees par les equivalences */
#define NBR_MAILLE_EQU  7
void lecture_equivalence_maillage(med_idt fid,char *nommaa,med_int nequ)
{
  med_int i,j,k;
  med_int ncor;
  med_int *cor;
  char equ[MED_TAILLE_NOM+1];
  char des[MED_TAILLE_DESC+1];
  med_int desmai[MED_NBR_GEOMETRIE_MAILLE] = {0,2,3,3,3,4,4,4,4,6,6,5,5,5,5};
  med_int nmailles[MED_NBR_GEOMETRIE_MAILLE];
  med_int desfac[MED_NBR_GEOMETRIE_FACE] = {3,3,4,4};
  med_int nfaces[MED_NBR_GEOMETRIE_FACE];
  med_int desare[MED_NBR_GEOMETRIE_ARETE] = {2,3};
  med_int naretes[MED_NBR_GEOMETRIE_ARETE];
  med_err ret = 0;

  fprintf(stdout,"\n(******************************)\n");
  fprintf(stdout,"(* EQUIVALENCES DU MAILLAGE : *)\n");
  fprintf(stdout,"(******************************)\n");

  if (nequ == 0)
    fprintf(stdout,"- Aucune quivalence \n");

  /* lecture de toutes les equivalences associes a nommaa */
  for (i = 0;i<nequ;i++) {
    fprintf(stdout,"- Equivalence numero : %d ",i+1);

    /* lecture des infos sur l'equivalence */
    ret = MEDequivInfo(fid,nommaa,i+1,equ,des);
    EXIT_IF(ret < 0,"lors de la lecture des informations sur une equivalence",
	    NULL);
    fprintf(stdout,"\n  - Nom de l'equivalence: %s \n",equ);
    fprintf(stdout,"\n  - Description de l'equivalence : %s \n",des);

    /* lecture des correspondances sur les differents types d'entites */

    /* les noeuds */
    ncor = MEDnCorres(fid,nommaa,equ,MED_NOEUD,(med_geometrie_element)0);
    EXIT_IF(ncor < 0,
	    "lors de la lecture du nombre de correspondances d'une equivalence",
	    NULL);
    fprintf(stdout,"\n  - Il y a %d correspondances sur les noeuds \n",ncor);

    if (ncor > 0) {

      /* allocation memoire */
      cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
      EXIT_IF(cor == NULL,NULL,NULL);

      ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_NOEUD,(med_geometrie_element)0);
      EXIT_IF(ret < 0,"lors de la lecture du tableau des correspondances",
	      NULL);
    if (!structure) {
      for (j=0;j<ncor;j++)
	fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",j+1,*(cor+2*j),
		      *(cor+2*j+1));
    }
      free(cor);
    }

    /* sur les mailles : on ne prend pas en compte les mailles 3D */
    for (j=0;j<NBR_MAILLE_EQU;j++) {

      ncor = MEDnCorres(fid,nommaa,equ,MED_MAILLE,typmai[j]);
      EXIT_IF(ncor < 0,
	      "lors de la lecture du nombre de correspondances dans une equivalence",
	      NULL);
      fprintf(stdout,"\n  - Il y a %d correspondances sur les mailles %s \n",ncor,
	      nommai[j]);

      if (ncor > 0) {

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_MAILLE,
			   typmai[j]);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

      if (!structure) {
	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",k+1,
		  *(cor+2*k),*(cor+2*k+1));
      }

	free(cor);
      }
    }

    /* sur les mailles Polygonales */

      ncor = MEDnCorres(fid,nommaa,equ,MED_MAILLE,MED_POLYGONE);
      EXIT_IF(ncor < 0,
	      "lors de la lecture du nombre de correspondances dans une equivalence",
	      NULL);
      fprintf(stdout,"\n  - Il y a %d correspondances sur les mailles polygone \n",ncor);

      if (ncor > 0) {

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_MAILLE,
			   MED_POLYGONE);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

      if (!structure) {
	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",k+1,
		  *(cor+2*k),*(cor+2*k+1));
      }

	free(cor);
      }


    /* sur les faces */
    for (j=0;j<MED_NBR_GEOMETRIE_FACE;j++) {

      ncor = MEDnCorres(fid,nommaa,equ,MED_FACE,typfac[j]);
      EXIT_IF(ncor < 0,
	      "lors de la lecture du nombre de correspondances dans une equivalence",
	      NULL);
      fprintf(stdout,"\n  - Il y a %d correspondances sur les faces %s\n",ncor,
	      nomfac[j]);

      if (ncor > 0) {

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_FACE,
			   typfac[j]);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

      if (!structure) {
	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",k+1,*(cor+2*k),
			  *(cor+2*k+1));
      }

	free(cor);
      }
    }

    /* sur les faces Polygonales */

      ncor = MEDnCorres(fid,nommaa,equ,MED_FACE,MED_POLYGONE);
      EXIT_IF(ncor < 0,
	      "lors de la lecture du nombre de correspondances dans une equivalence",
	      NULL);
      fprintf(stdout,"\n  - Il y a %d correspondances sur les faces polygone \n",ncor);

      if (ncor > 0) {

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_FACE,
			   MED_POLYGONE);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

      if (!structure) {
	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  - Correspondance %d : %d et %d \n",k+1,*(cor+2*k),
			  *(cor+2*k+1));
      }

	free(cor);
      }

    /*  sur les aretes */
    for (j=0;j<MED_NBR_GEOMETRIE_ARETE;j++) {

      ncor = MEDnCorres(fid,nommaa,equ,MED_ARETE,typare[j]);
      EXIT_IF(ncor < 0,"lors de la lecture du nombre de correspondances",
	      NULL);
      fprintf(stdout,"\n  - Il y a %d correspondances sur les aretes %s \n",
	      ncor,nomare[j]);

      if (ncor > 0) {

	/* allocation memoire */
	cor = (med_int*) malloc(sizeof(med_int)*ncor*2);
	EXIT_IF(cor == NULL,NULL,NULL);

	ret = MEDequivLire(fid,nommaa,equ,cor,ncor,MED_ARETE,
			   typare[j]);
	EXIT_IF(ret < 0,"lors de la lecture du tableau des equivalences",
		NULL);

      if (!structure) {
	for (k=0;k<ncor;k++)
	  fprintf(stdout,"\n  Correspondance %d : %d et %d \n",k+1,*(cor+2*k),
			*(cor+2*k+1));
      }

	free(cor);
      }
    }
  }

  return;
}


med_int lecture_nombre_joint(med_idt fid,char *nommaa)
{
  med_int njnt = MEDnJoint(fid,nommaa);
  EXIT_IF(njnt < 0,"lors de la lecture du nombre de joints",NULL);
  fprintf(stdout,"- Nombre de joints : %d \n",njnt);

  return njnt;
}


void lecture_joint_maillage(med_idt fid,char *nommaa,med_int njnt)
{
  med_int i,k;
  char des[MED_TAILLE_DESC+1];
  med_int ndom,nent;
  med_int typ_ent_local,typ_geo_local,typ_ent_distant,typ_geo_distant;
  med_int geo_ent_local,geo_ent_distant;

  char jn[MED_TAILLE_NOM+1];
  char maa_dist[MED_TAILLE_NOM+1];
  char corr[MED_TAILLE_NOM+1];
  med_int *cortab;

  const med_int nb_geo_ent=1+NBR_MAILLE_EQU+1+MED_NBR_GEOMETRIE_ARETE+MED_NBR_GEOMETRIE_FACE+1;
  med_geometrie_element typ_geo_ent[1+NBR_MAILLE_EQU+1+MED_NBR_GEOMETRIE_ARETE+MED_NBR_GEOMETRIE_FACE+1][2]=
    {
      {(med_geometrie_element)MED_NOEUD,(med_geometrie_element)0},
      {(med_geometrie_element)MED_MAILLE,MED_POINT1   },
      {(med_geometrie_element)MED_MAILLE,MED_SEG2     },
      {(med_geometrie_element)MED_MAILLE,MED_SEG3     },
      {(med_geometrie_element)MED_MAILLE,MED_TRIA3    },
      {(med_geometrie_element)MED_MAILLE,MED_TRIA6    },
      {(med_geometrie_element)MED_MAILLE,MED_QUAD4    },
      {(med_geometrie_element)MED_MAILLE,MED_QUAD8    },
      {(med_geometrie_element)MED_MAILLE,MED_POLYGONE },
      {(med_geometrie_element)MED_ARETE ,MED_SEG2     },
      {(med_geometrie_element)MED_ARETE ,MED_SEG3     },
      {(med_geometrie_element)MED_FACE  ,MED_TRIA3    },
      {(med_geometrie_element)MED_FACE  ,MED_TRIA6    },
      {(med_geometrie_element)MED_FACE  ,MED_QUAD4    },
      {(med_geometrie_element)MED_FACE  ,MED_QUAD8    },
      {(med_geometrie_element)MED_FACE  ,MED_POLYGONE }
    };
  char nom_geo_ent[1+NBR_MAILLE_EQU+1+MED_NBR_GEOMETRIE_ARETE+MED_NBR_GEOMETRIE_FACE+1][MED_TAILLE_NOM+1] =
    {
      "MED_NOEUD           ",
      "MED_MAILLE:MED_POINT1   ",
      "MED_MAILLE:MED_SEG2     ",
      "MED_MAILLE:MED_SEG3     ",
      "MED_MAILLE:MED_TRIA3    ",
      "MED_MAILLE:MED_TRIA6    ",
      "MED_MAILLE:MED_QUAD4    ",
      "MED_MAILLE:MED_QUAD8    ",
      "MED_MAILLE:MED_POLYGONE ",
      "MED_ARETE :MED_SEG2     ",
      "MED_ARETE :MED_SEG3     ",
      "MED_FACE  :MED_TRIA3    ",
      "MED_FACE  :MED_TRIA6    ",
      "MED_FACE  :MED_QUAD4    ",
      "MED_FACE  :MED_QUAD8    ",
      "MED_FACE  :MED_POLYGONE "
    };
  med_err ret = 0;

  fprintf(stdout,"\n(******************************)\n");
  fprintf(stdout,"(* JOINTS DU MAILLAGE       : *)\n");
  fprintf(stdout,"(******************************)\n");

  if (njnt == 0)
    fprintf(stdout,"- Aucun joint \n");

  /* lecture de touts les joints associes a nommaa */
  for (i = 0;i<njnt;i++) {
    fprintf(stdout,"- Joint numero : %d ",i+1);

    /* lecture des infos sur le joint */
    ret=MEDjointInfo(fid,nommaa,i+1,jn,des,&ndom,maa_dist);
    EXIT_IF(ret < 0,"lors de la lecture des informations sur un joint",
	    NULL);
    fprintf(stdout,"\n  - Nom du joint: %s \n",jn);
    fprintf(stdout,"\n  - Description du joint      : %s ",des);
    fprintf(stdout,"\n  - Domaine en regard         : %d ",ndom);
    fprintf(stdout,"\n  - Maillage distant          : %s ",maa_dist);


    /* lecture des correspondances sur les differents types d'entites */
    /* recherche du type des entites en regard... passage par toutes les combinaisons */
    for (geo_ent_local=0;geo_ent_local<nb_geo_ent;geo_ent_local++) {
	for (geo_ent_distant=0;geo_ent_distant<nb_geo_ent;geo_ent_distant++) {
	  typ_ent_local = typ_geo_ent[geo_ent_local][0];
	  typ_geo_local = typ_geo_ent[geo_ent_local][1];
	  typ_ent_distant = typ_geo_ent[geo_ent_distant][0];
	  typ_geo_distant = typ_geo_ent[geo_ent_distant][1];

	  if (nent=MEDjointnCorres(fid,nommaa,jn,(med_entite_maillage)typ_ent_local,(med_geometrie_element)typ_geo_local,
				   (med_entite_maillage)typ_ent_distant,(med_geometrie_element)typ_geo_distant)) {
	    if (nent > 0) {
	      fprintf(stdout,"\n\t\t- nb de couples d'entites en regard (local,distant)=(%s,%s) : %d \n",
		      nom_geo_ent[geo_ent_local],nom_geo_ent[geo_ent_distant],
		      nent);
	      fprintf(stdout,"  %d \n",nent);
	      cortab = (med_int*) malloc(sizeof(med_int)*nent*2);
	      if (MEDjointLire(fid,nommaa,jn,cortab,nent*2,
				   (med_entite_maillage)typ_ent_local,(med_geometrie_element)typ_geo_local,
                   (med_entite_maillage)typ_ent_distant,(med_geometrie_element)typ_geo_distant) < 0) {
	        fprintf(stdout,"\n\t\t- Erreur a la lecture des correspondances sur (%d,%d,%d,%d)",
			typ_ent_local,typ_geo_local,typ_ent_distant,typ_geo_distant,0);
	      }
	      else {
            if (!structure) {
		for (k=0;k<nent;k++)
		  fprintf(stdout,"\n\t\t- Correspondance %d : %d et %d ",k+1,
			  *(cortab+2*k),*(cortab+2*k+1));
			}
	      }
	      free(cortab);
	    }

	  }
	}
      }
  }

  return;
}


med_int lecture_nombre_noeuds_maillage_non_structure(med_idt fid,
						     char *nommaa)
{
  med_int nnoe = MEDnEntMaa(fid,nommaa,MED_COOR,MED_NOEUD,(med_geometrie_element)0,(med_connectivite)0);
  EXIT_IF(nnoe < 0,"lors de la lecture du nombre de noeuds",NULL);
  fprintf(stdout,"- Nombre de noeuds : %d \n",nnoe);

  return nnoe;
}


void lecture_noeuds_maillage_non_structure(med_idt fid,
					   char *nommaa,
					   med_int mdim,
					   med_int nnoe,
					   med_mode_switch mode_coo)
{
  med_float *coo;
  char nomcoo[3*MED_TAILLE_PNOM+1];
  char unicoo[3*MED_TAILLE_PNOM+1];
  char *nomnoe;
  med_int *numnoe;
  med_int *nufano;
  med_repere rep;
  med_booleen inonoe,inunoe;
  med_err ret = 0;
  med_int i;
  char str[MED_TAILLE_PNOM+1];

  /* Allocations memoires */
  /* table des coordonnees
     profil : (dimension * nombre de noeuds ) */
  coo = (med_float*) malloc(sizeof(med_float)*nnoe*mdim);
  EXIT_IF(coo == NULL,NULL,NULL);
  /* table  des numeros, des numeros de familles des noeuds
     profil : (nombre de noeuds) */
  numnoe = (med_int*) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(numnoe == NULL,NULL,NULL);
  nufano = (med_int*) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(nufano == NULL,NULL,NULL);
  /* table des noms des noeuds
     profil : (nnoe*MED_TAILLE_PNOM+1) */
  nomnoe = (char*) malloc(MED_TAILLE_PNOM*nnoe+1);
  EXIT_IF(nomnoe == NULL,NULL,NULL);

  /* lecture des noeuds :
     - coordonnees
     - noms (optionnel dans un fichier MED)
     - numeros (optionnel dans un fichier MED)
     - numeros des familles */
  ret = MEDnoeudsLire(fid,nommaa,mdim,coo,mode_coo,&rep,
		      nomcoo,unicoo,nomnoe,&inonoe,numnoe,&inunoe,
		      nufano,nnoe);
  EXIT_IF(ret < 0,"lors de la lecture des noeuds du maillage \n",NULL);

  /* affichage des resultats */
  fprintf(stdout,"\n(************************)\n");
  fprintf(stdout,"(* NOEUDS DU MAILLAGE : *)\n");
  fprintf(stdout,"(************************)\n");
  fprintf(stdout,"- Type de repere des coordonnees : %d \n",rep);
  fprintf(stdout,"- Nom des coordonnees : \n");
  for (i=0;i<mdim;i++) {
    strncpy(str,nomcoo+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
    str[MED_TAILLE_PNOM] = '\0';
    fprintf(stdout," %s ",str);
  }
  fprintf(stdout,"\n- Unites des coordonnees : \n");
  for (i=0;i<mdim;i++) {
    strncpy(str,unicoo+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
    str[MED_TAILLE_PNOM] = '\0';
    fprintf(stdout," %s ",str);
  }
  if (!structure) {
  fprintf(stdout,"\n- Coordonnees des noeuds : \n");
  for (i=0;i<nnoe*mdim;i++) {
    if (mode_coo == MED_FULL_INTERLACE && !(i % mdim))
      fprintf(stdout,"\n [ %5d ] : ", (i/mdim + 1) );
    if (mode_coo == MED_NO_INTERLACE && ! (i % nnoe))
      fprintf(stdout,"\n\n ");
    fprintf(stdout," %-+9.6f ",*(coo+i));
  }

  if (inonoe) {
    fprintf(stdout,"\n- Noms des noeuds : \n");
    for (i=0;i<nnoe;i++) {
      strncpy(str,nomnoe+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }
  }
  if (inunoe) {
    fprintf(stdout,"\n- Numeros des noeuds : \n");
    for (i=0;i<nnoe;i++)
      fprintf(stdout," %d ",*(numnoe+i));
  }
  fprintf(stdout,"\n- Numeros des familles des noeuds : \n");
  for (i=0;i<nnoe;i++)
    fprintf(stdout," %d ",*(nufano+i));
  fprintf(stdout,"\n");
  }


  /* liberation memoire */
  free(coo);
  free(nomnoe);
  free(numnoe);
  free(nufano);

  return;
}


med_int lecture_nombre_mailles_standards(med_idt fid,
					 char *nommaa,
					 med_geometrie_element typ_geo,
					 med_connectivite typ_con,
					 const int indice)
{
    med_int nmailles = MEDnEntMaa(fid,nommaa,MED_CONN,MED_MAILLE,
				  typ_geo,typ_con);
    EXIT_IF(nmailles < 0," lors de la lecture du nombre de mailles",NULL);
    fprintf (stdout,"- Nombre de mailles de type %s : %d \n",nommai[indice],
	     nmailles);

    return nmailles;
}

void lecture_mailles_standards(med_idt fid,
			       char *nommaa,
			       med_int mdim,
			       med_int *nmailles,
			       med_mode_switch mode_coo,
			       med_connectivite typ_con)
{
  med_int taille;
  med_int *connectivite;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_booleen inoele, inuele;
  med_geometrie_element typgeo;
  med_int desmai[MED_NBR_GEOMETRIE_MAILLE] = {0,2,3,3,3,4,4,4,4,6,6,5,5,5,5};
  med_int i,j;
  med_err ret = 0;
  char str[MED_TAILLE_PNOM+1];

  fprintf(stdout,"\n(**************************)\n");
  fprintf(stdout,"(* ELEMENTS DU MAILLAGE : *)\n");
  fprintf(stdout,"(**************************)");

  /* Lecture des connectivites, noms, numeros des mailles */
  for (i=0;i<MED_NBR_GEOMETRIE_MAILLE;i++)
    if (nmailles[i] > 0) {

      switch(typ_con) {
      case MED_NOD :
	taille = typmai[i]%100;
	break;

      case MED_DESC :
	taille = desmai[i];
	break;

      default :
	ret = -1;
      }

      /* allocation memoire */
      connectivite = (med_int*) malloc(sizeof(med_int)*taille*nmailles[i]);
      EXIT_IF(connectivite == NULL,NULL,NULL);
      nomele = (char*) malloc(sizeof(char)*MED_TAILLE_PNOM*nmailles[i]+1);
      EXIT_IF(nomele == NULL,NULL,NULL);
      numele = (med_int*) malloc(sizeof(med_int)*nmailles[i]);
      EXIT_IF(numele == NULL,NULL,NULL);
      nufael = (med_int*) malloc(sizeof(med_int)*nmailles[i]);
      EXIT_IF(nufael == NULL,NULL,NULL);

      /* lecture des donnes */
      ret = MEDelementsLire(fid,nommaa,mdim,connectivite,mode_coo,
			    nomele,&inoele,numele,&inuele,nufael,
			    nmailles[i],MED_MAILLE,typmai[i],typ_con);
      EXIT_IF(ret < 0,"lors de la lecture des mailles",NULL);

      if (!structure) {
      /* affichage des resultats */
      fprintf(stdout,"\n\n- Mailles de type %s : ", nommai[i]);
      fprintf(stdout,"\n  - Connectivit : \n");
      for (j=0;j<nmailles[i]*taille;j++) {
	    if (mode_coo == MED_FULL_INTERLACE && !(j % taille))
	      fprintf(stdout,"\n [ %5d ] : ", (j/taille +1) );
	    if (mode_coo == MED_NO_INTERLACE && !(j % nmailles[i]))
	      fprintf(stdout,"\n");
	    fprintf(stdout," %9d ",*(connectivite+j));
      }

      if (inoele) {
	fprintf(stdout,"\n  - Noms : \n");
	for (j=0;j<nmailles[i];j++) {
	  strncpy(str,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	  str[MED_TAILLE_PNOM] = '\0';
	  fprintf(stdout," %s ",str);
	}
      }
      if (inuele) {
	fprintf(stdout,"\n  - Numeros :\n");
	for (j=0;j<nmailles[i];j++)
	  fprintf(stdout," %d ",*(numele+j));
      }
      fprintf(stdout,"\n  - Numros de familles : \n");
      for (j=0;j<nmailles[i];j++)
	fprintf(stdout," %d ",*(nufael+j));
	  }

      /* liberation memoire */
      free(connectivite);
      free(nomele);
      free(numele);
      free(nufael);
    }

  return;
}


med_int lecture_nombre_mailles_polygones(med_idt fid,
					 char *nommaa,
					 med_connectivite typ_con)
{
  med_int nmpolygones = MEDnEntMaa(fid,nommaa,MED_CONN,MED_MAILLE,
				   MED_POLYGONE,typ_con);
  EXIT_IF(nmpolygones < 0,"lors de la lecture du nombre de mailles polygone\n",
	  NULL);
  fprintf(stdout,"- Nombre de mailles de type MED_POLYGONE : %d \n",
	  nmpolygones);

  return nmpolygones;
}

void lecture_mailles_polygones(med_idt fid,
			       char *nommaa,
			       med_int nmpolygones,
			       med_mode_switch mode_coo,
			       med_connectivite typ_con)
{
  med_int i,j;
  med_err ret = 0;
  med_int taille;
  med_int *connectivite;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_int *indexp;
  int ind1,ind2,np;
  med_int nf;
  char tmp[MED_TAILLE_NOM+1];
  med_err ret1,ret2,ret3;
  med_int nfa;

  /* lecture des mailles de type MED_POLYGONE */

  /* quelle taille pour  le tableau des connectivites ? */
  ret = MEDpolygoneInfo(fid,nommaa,MED_MAILLE,typ_con,&taille);
  EXIT_IF(ret < 0,"lors de la lecture des parametres des mailles polygones",
	  NULL);

  /* allocation memoire */
  indexp = (med_int *) malloc(sizeof(med_int)*(nmpolygones+1));
  EXIT_IF(indexp == NULL,NULL,NULL);
  connectivite = (med_int *) malloc(sizeof(med_int)*taille);
  EXIT_IF(connectivite == NULL,NULL,NULL);
  numele = (med_int *) malloc(sizeof(med_int)*nmpolygones);
  EXIT_IF(numele == NULL,NULL,NULL);
  nufael = (med_int *) malloc(sizeof(med_int)*nmpolygones);
  EXIT_IF(nufael == NULL,NULL,NULL);
  nomele = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*nmpolygones+1);
  EXIT_IF(nomele == NULL,NULL,NULL);

  /* lecture de la connectivite des mailles polygones */
  ret = MEDpolygoneConnLire(fid,nommaa,indexp,nmpolygones+1,connectivite,
			      MED_MAILLE,typ_con);
  EXIT_IF(ret < 0,"lors de la lecture des connectivites des mailles polygones",
	  NULL);

  /* lecture noms */
  ret1 = MEDnomLire(fid,nommaa,nomele,nmpolygones,MED_MAILLE,MED_POLYGONE);

  /* lecture des numeros */
  ret2 = (med_int) MEDnumLire(fid,nommaa,numele,nmpolygones,
			      MED_MAILLE,MED_POLYGONE);

  /* lecture des numeros de familles */
  ret3 = MEDfamLire(fid,nommaa,nufael,nmpolygones,MED_MAILLE,MED_POLYGONE);

  if (!structure) {
  /* affichage des resultats */
  fprintf(stdout,"\n\n- Mailles de type MED_POLYGONE : ");
    for (i=0;i<nmpolygones;i++) {
      fprintf(stdout,"\n >> Maille MED_POLYGONE %d : \n",i+1);
      fprintf(stdout,"\n  - Connectivit : ");
      ind1 = *(indexp+i)-1;
      ind2 = *(indexp+i+1)-1;
      for (j=ind1;j<ind2;j++)
	printf(" %d ",*(connectivite+j));
      if (ret1 == 0) {
	strncpy(tmp,nomele+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	tmp[MED_TAILLE_PNOM] = '\0';
	fprintf(stdout,"\n  - Nom : %s \n",tmp);
      }
      if (ret2 == 0)
	fprintf(stdout,"\n  - Numero : %d \n",*(numele+i));
      fprintf(stdout,"\n  - Numro de famille : %d \n",*(nufael+i));
    }
  }

    /* on libere la memoire */
    free(indexp);
    free(connectivite);
    free(numele);
    free(nufael);
    free(nomele);

    return;
}


med_int lecture_nombre_mailles_polyedres(med_idt fid,
					 char *nommaa,
					 med_connectivite typ_con)
{
  med_int npolyedres = MEDnEntMaa(fid,nommaa,MED_CONN,MED_MAILLE,
				  MED_POLYEDRE,typ_con);
  EXIT_IF(npolyedres < 0,"lors de la lecture du nombre de mailles polyedre \n",
	  NULL);
  fprintf(stdout,"- Nombre de mailles de type MED_POLYEDRE : %d \n",
	  npolyedres);

  return npolyedres;
}


void lecture_mailles_polyedres(med_idt fid,
			       char *nommaa,
			       med_int npolyedres,
			       med_mode_switch mode_coo,
			       med_connectivite typ_con)
{
  med_int i,j,k;
  med_err ret = 0;
  med_int taille;
  med_int *connectivite;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_int *indexp, *indexf;
  int ind1,ind2,np;
  med_int nf;
  char tmp[MED_TAILLE_PNOM+1];
  med_err ret1,ret2,ret3;
  med_int nfa;
  med_int nnoe;


  /* lecture des parametres de base */
  ret = MEDpolyedreInfo(fid,nommaa,typ_con,&nf,&taille);
  EXIT_IF(ret < 0,"lors de la lecture des parametres des mailles polyedres0",
	  NULL);

  /* allocation memoire */
  indexp = (med_int *) malloc(sizeof(med_int)*(npolyedres+1));
  EXIT_IF(indexp == NULL,NULL,NULL);
  indexf = (med_int *) malloc(sizeof(med_int)*nf);
  EXIT_IF(indexf == NULL,NULL,NULL);
  connectivite  = (med_int *) malloc(sizeof(med_int)*taille);
  EXIT_IF(connectivite == NULL,NULL,NULL);
  numele = (med_int *) malloc(sizeof(med_int)*npolyedres);
  EXIT_IF(numele == NULL,NULL,NULL);
  nufael = (med_int *) malloc(sizeof(med_int)*npolyedres);
  EXIT_IF(nufael == NULL,NULL,NULL);
  nomele = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*npolyedres+1);
  EXIT_IF(nomele == NULL,NULL,NULL);

  /* lecture de la connectivite des mailles polyedres */
  ret = MEDpolyedreConnLire(fid,nommaa,indexp,npolyedres+1,indexf,nf,
			    connectivite,typ_con);
  EXIT_IF(ret < 0,
	  "lors de la lecture de la connectivite des mailles polyedres",
	  NULL);

  /* lecture des noms */
  ret1 = MEDnomLire(fid,nommaa,nomele,npolyedres,MED_MAILLE,MED_POLYEDRE);

  /* lecture des numeros */
  ret2 = MEDnumLire(fid,nommaa,numele,npolyedres,MED_MAILLE,MED_POLYEDRE);

  /* lecture des numeros de familles */
  ret3 = MEDfamLire(fid,nommaa,nufael,npolyedres,MED_MAILLE,MED_POLYEDRE);

  if (!structure) {
  /* affichage des resultats */
  fprintf(stdout,"\n\n- Mailles de type MED_POLYEDRE : ");
  for (i=0;i<npolyedres;i++) {
    fprintf(stdout,"\n >> Maille MED_POLYEDRE %d : \n",i+1);
    fprintf(stdout,"\n  - Connectivit : \n");
    nfa  = *(indexp+i+1) - *(indexp+i);
    /* ind1 = indice dans "indexf" pour acceder aux numeros des faces */
    ind1 = *(indexp+i) - 1;
    for (j=0;j<nfa;j++) {
      if (typ_con == MED_NOD) {
	/* ind2 = indice dans "connectivite"
	   pour acceder au premier noeud de la face */
	ind2 = *(indexf+ind1+j) - 1;
	nnoe = *(indexf+ind1+j+1) - *(indexf+ind1+j);
	fprintf(stdout,"   - Face %d : [ ", j+1);
	for (k=0;k<nnoe;k++)
	  printf(" %d ",*(connectivite+ind2+k));
	printf(" ] \n");
      }
      else {
	nfa  = *(indexp+i+1) - *(indexp+i);
	/* ind1 = indice dans "connectivite"
	   pour acceder aux numeros des faces */
	ind1 = *(indexp+i) - 1;
	for (j=0;j<nfa;j++)
	  fprintf(stdout,"   - Face %d de numero : %d et de type %d \n", j+1,
		  *(connectivite+ind1+j),*(indexf+ind1+j));
      }
    }
    if (ret1 == 0) {
      strncpy(tmp,nomele+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      tmp[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout,"\n  - Nom : %s \n",tmp);
    }
    if (ret2 == 0)
      fprintf(stdout,"\n  - Numero : %d \n",*(numele+i));
    fprintf(stdout,"\n  - Numro de famille : %d \n",*(nufael+i));
  }
  }

  /* on libere la memoire */
  free(indexp);
  free(indexf);
  free(connectivite);
  free(numele);
  free(nufael);
  free(nomele);

  return;
}

med_int lecture_nombre_faces_standards(med_idt fid,
				       char *nommaa,
				       med_geometrie_element typ_geo,
				       const med_int indice
				       )
{
  med_int nfaces = MEDnEntMaa(fid,nommaa,MED_CONN,MED_FACE,typ_geo,
			      MED_DESC);
  EXIT_IF(nfaces < 0,"lors de la lecture du nombre de faces",NULL);
  fprintf (stdout,"- Nombre de faces de type %s : %d \n",
	   nomfac[indice],nfaces);

  return nfaces;
}

void lecture_faces_standard(med_idt fid,
			    char *nommaa,
			    med_int mdim,
			    med_int *nfaces,
			    med_mode_switch mode_coo)
{
  med_int taille;
  med_int *connectivite;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_booleen inoele,inuele;
  med_geometrie_element typgeo;
  med_int i,j;
  med_err ret = 0;
  char str[MED_TAILLE_PNOM+1];
  med_int desfac[MED_NBR_GEOMETRIE_FACE] = {3,3,4,4};

  for (i=0;i<MED_NBR_GEOMETRIE_FACE;i++)
    if (nfaces[i] > 0 ) {

      /*  taille de la description : nombre d'aretes */
      taille = desfac[i];

      /* allocation memoire */
      connectivite = (med_int*)malloc(sizeof(med_int)*taille*nfaces[i]);
      EXIT_IF(connectivite == NULL,NULL,NULL);
      nomele = (char*)malloc(sizeof(char)*MED_TAILLE_PNOM*nfaces[i]+1);
      EXIT_IF(nomele == NULL,NULL,NULL);
      numele = (med_int*)malloc(sizeof(med_int)*nfaces[i]);
      EXIT_IF(numele == NULL,NULL,NULL);
      nufael = (med_int*)malloc(sizeof(med_int)*nfaces[i]);
      EXIT_IF(nufael == NULL,NULL,NULL);

      /* lecture des donnes */
      ret = MEDelementsLire(fid,nommaa,mdim,connectivite,mode_coo,
			    nomele,&inoele,numele,&inuele,nufael,
			    nfaces[i],MED_FACE,typfac[i],
			    MED_DESC);
      EXIT_IF(ret < 0,"lors de la lecture des faces",NULL);

      if (!structure) {
      /* affichage des resultats */
      fprintf(stdout,"\n- Faces de type %s : ", nomfac[i]);
      fprintf(stdout,"\n  - Connectivit : \n");
      for (j=0;j<nfaces[i]*taille;j++) {
	    if (mode_coo == MED_FULL_INTERLACE && !(j % taille))
	      fprintf(stdout,"\n [ %5d ] : ", (j/taille+1) );
	    if (mode_coo == MED_NO_INTERLACE && !(j % nfaces[i]))
	      fprintf(stdout,"\n");
	    fprintf(stdout," %9d ",*(connectivite+j));
	  }

      if (inoele) {
	fprintf(stdout,"\n  - Noms : \n");
	for (j=0;j<nfaces[i];j++) {
	  strncpy(str,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	  str[MED_TAILLE_PNOM] = '\0';
	  fprintf(stdout," %s ",str);
	}
      }
      if (inuele) {
	fprintf(stdout,"\n  - Numeros :\n");
	for (j=0;j<nfaces[i];j++)
	  fprintf(stdout," %d ",*(numele+j));
      }
      fprintf(stdout,"\n  - Numros de familles : \n");
      for (j=0;j<nfaces[i];j++)
	fprintf(stdout," %d ",*(nufael+j));
	  }

      /* liberation memoire */
      free(connectivite);
      free(nomele);
      free(numele);
      free(nufael);
    }

  return;
}

med_int lecture_nombre_faces_polygones(med_idt fid,
				       char *nommaa)
{
  med_int nfpolygones = MEDnEntMaa(fid,nommaa,MED_CONN,MED_FACE,MED_POLYGONE,
				   MED_DESC);
  EXIT_IF(nfpolygones < 0,"lors de la lecture du nombre de faces polygone \n",
	  NULL);
  fprintf(stdout,"- Nombre de faces de type MED_POLYGONE : %d \n",
	  nfpolygones);

  return nfpolygones;
}

void lecture_faces_polygones(med_idt fid,
			     char *nommaa,
			     med_int nfpolygones,
			     med_mode_switch mode_coo)
{
  med_int i,j;
  med_err ret = 0;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_int *connectivite;
  med_int taille;
  med_int *indexp;
  int ind1,ind2,np;
  med_int nf;
  char tmp[MED_TAILLE_NOM+1];
  med_err ret1,ret2,ret3;
  med_int nfa;

  /* quelle taille pour  le tableau des connectivites ? */
  ret = MEDpolygoneInfo(fid,nommaa,MED_FACE,MED_DESC,&taille);
  EXIT_IF(ret < 0,"lors de la lecture des parametres des faces polygones",
	    NULL);

  /* allocation memoire */
  indexp = (med_int *) malloc(sizeof(med_int)*(nfpolygones+1));
  EXIT_IF(indexp == NULL,NULL,NULL);
  connectivite = (med_int *) malloc(sizeof(med_int)*taille);
  EXIT_IF(connectivite == NULL,NULL,NULL);
  numele = (med_int *) malloc(sizeof(med_int)*nfpolygones);
  EXIT_IF(numele == NULL,NULL,NULL);
  nufael = (med_int *) malloc(sizeof(med_int)*nfpolygones);
  EXIT_IF(nufael == NULL,NULL,NULL);
  nomele = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*nfpolygones+1);
  EXIT_IF(nomele == NULL,NULL,NULL);

  /* lecture de la connectivite des faces polygones */
  ret = MEDpolygoneConnLire(fid,nommaa,indexp,nfpolygones+1,connectivite,
			    MED_FACE,MED_DESC);
  EXIT_IF(ret < 0,"lors de la lecture des connectivites des faces polygones",
	  NULL);

  /* lecture des noms */
  ret1 = MEDnomLire(fid,nommaa,nomele,nfpolygones,
		    MED_FACE,MED_POLYGONE);

  /* lecture des numeros */
  ret2 = (med_int) MEDnumLire(fid,nommaa,numele,nfpolygones,
			      MED_FACE,MED_POLYGONE);

  /* lecture des numeros de familles */
  ret3 = MEDfamLire(fid,nommaa,nufael,nfpolygones,MED_FACE,MED_POLYGONE);

  if (!structure) {
  /* affichage des resultats */
  fprintf(stdout,"\n\n- Faces de type MED_POLYGONE : ");
  for (i=0;i<nfpolygones;i++) {
    fprintf(stdout,"\n >> Face MED_POLYGONE %d : \n",i+1);
    fprintf(stdout,"\n  - Connectivit : ");
    ind1 = *(indexp+i)-1;
    ind2 = *(indexp+i+1)-1;
    for (j=ind1;j<ind2;j++)
      fprintf(stdout," %d ",*(connectivite+j));
    if (ret1 == 0) {
      strncpy(tmp,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      tmp[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout,"\n  - Nom : %s \n",tmp);
    }
    if (ret2 == 0)
      fprintf(stdout,"\n  - Numero : %d \n",*(numele+j));
    fprintf(stdout,"\n  - Numro de famille : %d \n",*(nufael+j));
  }
  }

  /* on libere la memoire */
  free(indexp);
  free(connectivite);
  free(numele);
  free(nufael);
  free(nomele);

  return;
}


med_int lecture_nombre_aretes_standards(med_idt fid,
					char *nommaa,
					med_geometrie_element typ_geo,
					const med_int indice)
{
  med_int naretes = MEDnEntMaa(fid,nommaa,MED_CONN,MED_ARETE,typ_geo,MED_DESC);
  EXIT_IF(naretes < 0,"lors de la lecture du nombre d'aretes",NULL);
  fprintf (stdout,
	   "- Nombre d'aretes de type %s : %d \n",nomare[indice],naretes);

  return naretes;
}

void lecture_aretes_standards(med_idt fid,
			      char *nommaa,
			      med_int mdim,
			      med_int *naretes,
			      med_mode_switch mode_coo)
{
  med_int taille;
  med_int *connectivite;
  char *nomele;
  med_int *numele;
  med_int *nufael;
  med_booleen inoele,inuele;
  med_geometrie_element typgeo;
  med_int desare[MED_NBR_GEOMETRIE_ARETE] = {2,3};
  med_int i,j;
  med_err ret = 0;
  char str[MED_TAILLE_PNOM+1];

  for (i=0;i<MED_NBR_GEOMETRIE_ARETE;i++)
    if (naretes[i] > 0) {

      taille = desare[i];
      /* dimension de l'arete  */
      /* edim = typare[i] / 100; */

      /* allocation memoire */
      connectivite = (med_int*)malloc(sizeof(med_int)*taille*naretes[i]);
      EXIT_IF(connectivite == NULL,NULL,NULL);
      nomele = (char*)malloc(sizeof(char)*MED_TAILLE_PNOM*naretes[i]+1);
      EXIT_IF(nomele == NULL,NULL,NULL);
      numele = (med_int*)malloc(sizeof(med_int)*naretes[i]);
      EXIT_IF(numele == NULL,NULL,NULL);
      nufael = (med_int*)malloc(sizeof(med_int)*naretes[i]);
      EXIT_IF(nufael == NULL,NULL,NULL);

      /* lecture des donnes */
      ret = MEDelementsLire(fid,nommaa,mdim,connectivite,mode_coo,
			    nomele,&inoele,numele,&inuele,nufael,
			    naretes[i],MED_ARETE,typare[i],
			    MED_DESC);
      EXIT_IF(ret < 0,"lors de la lecture des aretes",
	      NULL);

      if (!structure) {
      /* affichage des resultats */
      fprintf(stdout,"\n- Aretes de type %s : ", nomare[i]);
      fprintf(stdout,"\n  - Connectivit : \n");
      for (j=0;j<naretes[i]*taille;j++) {
  	    if (mode_coo == MED_FULL_INTERLACE && !(j % taille))
	      fprintf(stdout,"\n [ %5d ] : ", (j/taille+1) );
	    if (mode_coo == MED_NO_INTERLACE && !(j % naretes[i]))
	      fprintf(stdout,"\n");
	    fprintf(stdout," %9d ",*(connectivite+j));
	  }

      if (inoele) {
	fprintf(stdout,"\n  - Noms : \n");
	for (j=0;j<naretes[i];j++) {
	  strncpy(str,nomele+j*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	  str[MED_TAILLE_PNOM] = '\0';
	  fprintf(stdout," %s ",str);
	}
      }
      if (inuele) {
	fprintf(stdout,"\n  - Numeros :\n");
	for (j=0;j<naretes[i];j++)
	  fprintf(stdout," %d ",*(numele+j));
      }
      fprintf(stdout,"\n  - Numros de familles : \n");
      for (j=0;j<naretes[i];j++)
	fprintf(stdout," %d ",*(nufael+j));
	  }

      /* liberation memoire */
      free(connectivite);
      free(nomele);
      free(numele);
      free(nufael);
    }

  return;
}


/******************************************************************************
 * - Nom de la fonction : lecture_maillage_non_structure
 * - Description : lecture et affichage d'un maillage MED_NON_STRUCTURE.
 * - Parametres :
 *     - fid                       (IN) : ID du fichier MED.
 *     - nommaa                    (IN) : nom du maillage a lire.
 *     - mdim                      (IN) : dimension du maillage.
 *     - mode_coo                  (IN) : mode de stockage en memoire :
 *                                        MED_FULL_INTERLACE : entrelace |
 *                                        MED_NO_INTERLACE : non entrelace.
 *     - typ_con                   (IN) : mode de connectivite :
 *                                        MED_DESC : descendante |
 *                                        MED_NOD : nodale.
 *     - lecture_en_tete_seulement (IN) : mode de lecture et d'affichage.
 ******************************************************************************/

void lecture_maillage_non_structure(med_idt fid,
				    char *nommaa,
				    med_int mdim,
				    med_mode_switch mode_coo,
				    med_connectivite typ_con,
				    const int lecture_en_tete_seulement)
{
  med_err ret = 0;
  med_int i;
  /* nombre d'objets MED : noeuds, mailles, faces, aretes , ... */
  med_int nnoe;
  med_int nmailles[MED_NBR_GEOMETRIE_MAILLE];
  med_int nfaces[MED_NBR_GEOMETRIE_FACE];
  med_int naretes[MED_NBR_GEOMETRIE_ARETE];
  /* polygones et polyedres */
  med_int nmpolygones, npolyedres, nfpolygones;
  /* familles */
  med_int nfam;
  /* equivalences */
  med_int nequ;
  /* joints */
  med_int njnt;

  /* Combien de noeuds dans le maillage ? */
  nnoe = lecture_nombre_noeuds_maillage_non_structure(fid,nommaa);

  /* Combien de mailles, faces ou aretes pour chaque type geometrique ? */
  for (i=0;i<MED_NBR_GEOMETRIE_MAILLE;i++)
    nmailles[i] = lecture_nombre_mailles_standards(fid,nommaa,typmai[i],
						   typ_con,i);

  /* Combien de mailles polygones quelconques ? */
  nmpolygones = lecture_nombre_mailles_polygones(fid,nommaa,typ_con);

  /* Combien de mailles polyedres quelconques ? */
  npolyedres = lecture_nombre_mailles_polyedres(fid,nommaa,typ_con);

  /* Pour la connectivite descendante */
  if (typ_con == MED_DESC) {

    /* Combien de faces : types geometriques standards ? */
    for (i=0;i<MED_NBR_GEOMETRIE_FACE;i++)
      nfaces[i] = lecture_nombre_faces_standards(fid,nommaa,typfac[i],i);

    /* Combien de faces polygones quelconques ? */
    nfpolygones = lecture_nombre_faces_polygones(fid,nommaa);

    /* Combien d'aretes */
    for (i=0;i<MED_NBR_GEOMETRIE_ARETE;i++) 
      naretes[i] = lecture_nombre_aretes_standards(fid,nommaa,typare[i],i);
  }

  /* combien de familles ? */
  nfam = lecture_nombre_famille(fid,nommaa);

  /* combien d'equivalences ? */
  nequ = lecture_nombre_equivalence(fid,nommaa);

  /* combien de joints ? */
  njnt = lecture_nombre_joint(fid,nommaa);

  /* en fonction du mode de lecture, on continue ou non */
  if (lecture_en_tete_seulement)
    return;

  /****************************************************************************
  *                       LECTURE DES NOEUDS                                  *
  ****************************************************************************/
  lecture_noeuds_maillage_non_structure(fid,nommaa,mdim,nnoe,mode_coo);


  /****************************************************************************
  *                       LECTURE DES ELEMENTS                                *
  * Mailles :                                                                 *
  * - Types geometriques classiques : MED_SEG2, MED_SEG3, MED_TRIA3, ...      *
  * - Polygones quelconques.                                                  *
  * - Polyedres quelconques.                                                  *
  * Faces (connectivite descendante uniquement) :                             *
  * - Types geometriques classiques.                                          *
  * - Polygones quelconques.                                                  *
  ****************************************************************************/

  /* lecture et affichage des mailles */
  lecture_mailles_standards(fid,nommaa,mdim,nmailles,mode_coo,typ_con);

  if (nmpolygones > 0)
    lecture_mailles_polygones(fid,nommaa,nmpolygones,mode_coo,typ_con);

  if (npolyedres > 0)
    lecture_mailles_polyedres(fid,nommaa,npolyedres,mode_coo,typ_con);

  /* lecture et affichage des faces en connectivite descendante uniquement */
  if (typ_con == MED_DESC) {
    lecture_faces_standard(fid,nommaa,mdim,nfaces,mode_coo);
    if (nfpolygones > 0)
      lecture_faces_polygones(fid,nommaa,nfpolygones,mode_coo);
  }

  /* lecture et affichage des aretes en connectivite descendante uniquement */
  if (typ_con == MED_DESC)
    lecture_aretes_standards(fid,nommaa,mdim,naretes,mode_coo);

  /****************************************************************************
  *                      LECTURE DES FAMILLES                                 *
  ****************************************************************************/
  lecture_famille_maillage(fid,nommaa,nfam);


  /****************************************************************************
  *                       LECTURE DES EQUIVALENCES                            *
  ****************************************************************************/
  lecture_equivalence_maillage(fid,nommaa,nequ);


  /****************************************************************************
  *                       LECTURE DES JOINTS                                  *
  ****************************************************************************/
  lecture_joint_maillage(fid,nommaa,njnt);

  return;
}


void lecture_caracteristiques_grille(med_idt fid, 
				     char *nommaa,
				     med_int mdim,
				     med_int *nind,
				     med_int *nnoe,
				     med_int *nmai,
				     med_type_grille *type)
{
  med_err ret = 0;
  med_int axe;
  med_int *structure_grille;
  med_table quoi;
  med_int j;

  /* lecture de la nature du maillage structure : MED_GRILLE_CARTESIENNE ,
     ...*/
  ret = MEDnatureGrilleLire(fid,nommaa,type);
  EXIT_IF(ret < 0,"a la lecture du type d'une grille ",NULL);

  switch(*type) {

  case MED_GRILLE_CARTESIENNE :
  case MED_GRILLE_POLAIRE :
    if (*type == MED_GRILLE_CARTESIENNE)
      fprintf(stdout,"- Type de grille : MED_GRILLE_CARTESIENNE \n");
    else
      fprintf(stdout,"- Type de grille : MED_GRILLE_POLAIRE \n");
    for (axe=1;axe<=mdim;axe++) {
      switch(axe) {

      case 1:
	quoi = MED_COOR_IND1;
	break;

      case 2:
	quoi = MED_COOR_IND2;
	break;

      case 3:
	quoi = MED_COOR_IND3;
	break;
      }
      nind[axe - 1] = MEDnEntMaa(fid,nommaa,quoi,MED_NOEUD,(med_geometrie_element)0,(med_connectivite)0);
      EXIT_IF(nind[axe - 1] < 0,
	      "lors de la lecture de la taille d'un indice d'une grille",
	      NULL);
      *nnoe = nind[axe - 1] * (*nnoe);
      *nmai = (nind[axe - 1] - 1) * (*nmai);
      fprintf(stdout,
	      "- Taille de l'indice de l'axe %d des coordonnees : %d \n",
	      axe,nind[axe - 1]);
    }
    break;

  case MED_GRILLE_STANDARD:
    fprintf(stdout,"- Type de grille : MED_GRILLE_DESTRUCTUREE \n");
    *nnoe = MEDnEntMaa(fid,nommaa,MED_COOR,MED_NOEUD,(med_geometrie_element)0,(med_connectivite)0);
    EXIT_IF(*nnoe < 0,"lors de la lecture du nombre de noeuds du maillage "
	    ,nommaa);

    /* on alloue la memoire */
    structure_grille = (med_int *) malloc(sizeof(med_int)*mdim);
    EXIT_IF(structure_grille == NULL,NULL,NULL);
    /* on lit la structure de la grille
       et on affiche le resultat */
    ret = MEDstructureCoordLire(fid,nommaa,mdim,structure_grille);
    EXIT_IF(ret < 0,"lors de la lecture de la structure de la grille",
	    NULL);
    fprintf(stdout,"- Structure de la grille : [ ");
    for (j=0;j<mdim;j++) {
      *nmai = (*(structure_grille+j) - 1) * (*nmai);
      fprintf(stdout," %d ",*(structure_grille+j));
    }
    fprintf(stdout," ] \n");
    /* on nettoie la memoire */
    free(structure_grille);
    break;
  }

  fprintf(stdout,"- Nombre de noeuds : %d \n",*nnoe);
  fprintf(stdout,"- Nombre de mailles : %d \n",*nmai);

  return;
}

void lecture_noeuds_maillage_structure(med_idt fid, char *nommaa,
				       med_int mdim,med_int *nind,med_int nnoe,
				       med_type_grille type,
				       const med_mode_switch mode_coo)
{
  med_err ret = 0;
  med_int axe,i,j;
  char comp_ind[MED_TAILLE_PNOM+1];
  char unit_ind[MED_TAILLE_PNOM+1];
  char comp[3*MED_TAILLE_PNOM+1];
  char unit[3*MED_TAILLE_PNOM+1];
  char str[MED_TAILLE_PNOM+1];
  med_float *coo = NULL;
  med_float *indices = NULL;
  med_table quoi;
  med_repere repere;
  med_int *nufano = NULL;
  med_int *numnoe = NULL;
  char *nomnoe = NULL;

  fprintf(stdout,"\n(*************************)\n");
  fprintf(stdout,"(* NOEUDS DE LA GRILLE : *)\n");
  fprintf(stdout,"(*************************)\n");

  switch(type) {

  case MED_GRILLE_CARTESIENNE :
  case MED_GRILLE_POLAIRE :
    /* on affiche les coordonnees de chacun des axes */
    for (axe = 1; axe<=mdim; axe++) {
      /* on alloue la memoire */
      indices = (med_float *) malloc(sizeof(med_float)*nind[axe - 1]);
      EXIT_IF(indices == NULL,NULL,NULL);
      /* on lit le tableau des indices de coordonnees
         et on affiche le resultat */
      ret = MEDindicesCoordLire(fid,nommaa,mdim,indices,nind[axe - 1],
				axe,comp_ind,unit_ind);
      EXIT_IF(ret < 0,"lors de la lecture d'un tableau d'indice",
	      NULL);
      fprintf(stdout,"\n - Axe %s [%s] : [ ",comp_ind,unit_ind);
      for (j=0;j<nind[axe - 1];j++)
	fprintf(stdout," %f ",*(indices+j));
      printf(" ] \n");
      /* on nettoie la memoire */
      free(indices);
    }
    break;

  case MED_GRILLE_STANDARD:
    /* on alloue la memoire */
    coo = (med_float *) malloc(sizeof(med_float)*nnoe*mdim);
    EXIT_IF(coo == NULL,NULL,NULL);
    /* on va lire les coordonnees des noeuds */
    ret = MEDcoordLire(fid,nommaa,mdim,coo,MED_FULL_INTERLACE,MED_ALL,NULL,0,
			     &repere,comp,unit);
    EXIT_IF(ret < 0,"lors de la lecture des noeuds du maillage",NULL);
    /* on affiche le resultat */
    fprintf(stdout,"- Nom des coordonnees : \n");
    for (i=0;i<mdim;i++) {
      strncpy(str,comp+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }
    fprintf(stdout,"\n- Unites des coordonnees : \n");
    for (i=0;i<mdim;i++) {
      strncpy(str,unit+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }
    if (!structure) {
    fprintf(stdout,"\n - Coordonnees des noeuds : [ ");
    for (j=0;j<nnoe*mdim;j++)
      fprintf(stdout," %f ",*(coo+j));
    fprintf(stdout," ] \n");
    }

    /* on nettoie la memoire */
    free(coo);
    break;
  }

  /* lecture et affichage des :
     - numeros de familles des noeuds
     - noms des noeuds (optionnel)
     - numeros des noeuds (optionnel) */

  /* on alloue la memoire */
  numnoe = (med_int *) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(numnoe == NULL,NULL,NULL);
  nomnoe = (char*) malloc(MED_TAILLE_PNOM*nnoe+1);
  EXIT_IF(nomnoe == NULL,NULL,NULL);
  nufano = (med_int *) malloc(sizeof(med_int)*nnoe);
  EXIT_IF(nufano == NULL,NULL,NULL);

  /* on va lire les numeros de familles des noeuds */
  ret = MEDfamLire(fid,nommaa,nufano,nnoe,MED_NOEUD,(med_geometrie_element)0);
  EXIT_IF(ret < 0,"lors de la lecture des numeros de familles des noeuds",
	  NULL);
  if (!structure) {
  /* on affiche le resultat */
  fprintf(stdout,"\n- Numeros des familles des noeuds : \n");
  for (i=0;i<nnoe;i++)
    fprintf(stdout," %d ",*(nufano+i));
  fprintf(stdout,"\n");
  }

  /* on va lire et afficher les noms des noeuds */
  if (MEDnomLire(fid,nommaa,nomnoe,nnoe,MED_NOEUD,MED_POINT1) == 0) {
    if (!structure) {
    fprintf(stdout,"\n- Noms des noeuds : \n");
    for (i=0;i<nnoe;i++) {
      strncpy(str,nomnoe+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
      str[MED_TAILLE_PNOM] = '\0';
      fprintf(stdout," %s ",str);
    }
    }
  }

  /* on va lire et afficher les numeros des noeuds */
  if (MEDnumLire(fid,nommaa,numnoe,nnoe,MED_NOEUD,MED_POINT1) == 0) {
    if (!structure) {
    fprintf(stdout,"\n- Numeros des noeuds : \n");
    for (i=0;i<nnoe;i++)
      fprintf(stdout," %d ",*(numnoe+i));
    }
  }

  /* on nettoie la memoire */
  free(nufano);
  free(numnoe);
  free(nomnoe);

  return;
}


void lecture_mailles_maillage_structure(med_idt fid, char *nommaa,
					const med_int mdim,const med_int nmai)
{
  med_err ret = 0;
  med_int i;
  med_int *nufael = NULL;
  char *nomele = NULL;
  med_int *numele = NULL;
  char str[MED_TAILLE_PNOM+1]; 
  /* type geometrique des elements */
  med_geometrie_element typgeo;

  fprintf(stdout,"\n(***************************)\n");
  fprintf(stdout,"(* ELEMENTS DE LA GRILLE : *)\n");
  fprintf(stdout,"(***************************)\n");

  /* type des mailles */
  switch(mdim) {
  case 0 :
    typgeo = MED_POINT1;
    break;
  case 1 :
    typgeo = MED_SEG2;
    break;
  case 2 :
    typgeo = MED_QUAD4;
    break;
  default :
    typgeo = MED_HEXA8;
  }

  /* On va lire et afficher :
   * - Les numeros de familles
   * - Les noms (optionnel)
   * - Les numeros (optionnel)
   */

  /* on alloue la memoire */
  numele = (med_int *) malloc(sizeof(med_int)*nmai);
  EXIT_IF(numele == NULL,NULL,NULL);
  nomele = (char *) malloc(sizeof(char)*MED_TAILLE_PNOM*nmai+1);
  EXIT_IF(nomele == NULL,NULL,NULL);
  nufael = (med_int *) malloc(sizeof(med_int)*nmai);
  EXIT_IF(nufael == NULL,NULL,NULL);

  /* lecture des numeros de famille */
  ret = MEDfamLire(fid,nommaa,nufael,nmai,MED_MAILLE,typgeo);
  if (ret < 0)
    for (i=0;i<nmai;i++)
      *(nufael+i) = 0;

  if (!structure) {
  /* on affiche le resultat */
  fprintf(stdout,"\n- Numeros des familles des mailles : \n");
  for (i=0;i<nmai;i++)
    fprintf(stdout," %d ",*(nufael+i));
  fprintf(stdout,"\n");
  }

  /* on va lire et afficher les noms des mailles */
  if (MEDnomLire(fid,nommaa,nomele,nmai,MED_MAILLE,typgeo) == 0) {
    if (!structure) {
    fprintf(stdout,"\n  - Noms : \n");
    for (i=0;i<nmai;i++) {
	  strncpy(str,nomele+i*MED_TAILLE_PNOM,MED_TAILLE_PNOM);
	  str[MED_TAILLE_PNOM] = '\0';
	  fprintf(stdout," %s ",str);
	}
	}
  }

  /* on va lire et afficher les numeros des mailles */
  if (MEDnumLire(fid,nommaa,numele,nmai,MED_MAILLE,typgeo) == 0) {
    if (!structure) {
    fprintf(stdout,"\n  - Numeros :\n");
    for (i=0;i<nmai;i++)
      fprintf(stdout," %d ",*(numele+i));
    }
  }

  /* on libere la memoire */
  free(nufael);
  free(nomele);
  free(numele);

  return;
}


void lecture_maillage_structure(med_idt fid, char *nommaa,med_int mdim, 
				const med_mode_switch mode_coo,
				const int lecture_en_tete_seulement)
{
  med_err ret = 0;
  /* nombre de valeurs selon les axes du repere */
  med_int nind[3];
  med_int nnoe = 1;
  med_int nmai = 1;
  /* type de la grille */
  med_type_grille type;
  /* nombre de familles */
  med_int nfam;
  /* nombre d'equivalences */
  med_int nequ;
  /* nombre de joints */
  med_int njnt;

  /* lecture selon la nature de la grille des caracteristiques 
     du maillage :
     - nombre de noeuds
     - nombre de mailles
  */
  lecture_caracteristiques_grille(fid,nommaa,mdim,nind,&nnoe,&nmai,&type);

  /* nombre de familles */
  nfam = lecture_nombre_famille(fid,nommaa);

  /* nombre d'equivalences  */
  nequ = lecture_nombre_equivalence(fid,nommaa);

  /* combien de joints */
  njnt = lecture_nombre_joint(fid,nommaa);

  if (lecture_en_tete_seulement)
    return ;

  /****************************************************************************
  *                      LECTURE DES NOEUDS                                   *
  ****************************************************************************/
  lecture_noeuds_maillage_structure(fid,nommaa,mdim,nind,nnoe,type,mode_coo);

  /****************************************************************************
  *                      LECTURE DES ELEMENTS                                 *
  ****************************************************************************/
  lecture_mailles_maillage_structure(fid,nommaa,mdim,nmai);

  /****************************************************************************
  *                      LECTURE DES FAMILLES                                 *
  ****************************************************************************/
  lecture_famille_maillage(fid,nommaa,nfam);

  /****************************************************************************
  *                       LECTURE DES EQUIVALENCES                            *
  ****************************************************************************/
  lecture_equivalence_maillage(fid,nommaa,nequ);

  /****************************************************************************
  *                       LECTURE DES JOINTS                                  *
  ****************************************************************************/
  lecture_joint_maillage(fid,nommaa,njnt);

  return ;
}



med_err getFieldsOn(med_idt fid, char* maillage,
		    char * nomcha, med_type_champ typcha, med_int ncomp,
		    med_entite_maillage entite, med_mode_switch stockage) {

  int j,k,l,m,n,nb_geo;
  med_int nbpdtnor=0,pflsize,*pflval,ngauss=0,ngroup,*vale=NULL,nval;
  med_int numdt=0,numo=0,lnsize,nbrefmaa;
  med_float *valr=NULL,dt=0.0;
  med_err ret=0;
  med_booleen local;
  char pflname [MED_TAILLE_NOM+1]="";
  char locname [MED_TAILLE_NOM+1]="";
  char * lien = NULL;
  char maa_ass [MED_TAILLE_NOM+1]="";
  char dt_unit [MED_TAILLE_PNOM+1]="";
  med_geometrie_element * type_geo;
  med_geometrie_element typ_noeud[1] = { MED_NONE };
  med_geometrie_element typmai[MED_NBR_GEOMETRIE_MAILLE+2] = {MED_POINT1,
							      MED_SEG2,
							      MED_SEG3,
							      MED_TRIA3,
							      MED_QUAD4,
							      MED_TRIA6,
							      MED_QUAD8,
							      MED_TETRA4,
							      MED_PYRA5,
							      MED_PENTA6,
							      MED_HEXA8,
							      MED_TETRA10,
							      MED_PYRA13,
							      MED_PENTA15,
							      MED_HEXA20,
							      MED_POLYGONE,
							      MED_POLYEDRE};
  med_geometrie_element typfac[MED_NBR_GEOMETRIE_FACE+1] = {MED_TRIA3,
							    MED_TRIA6,
							    MED_QUAD4,
							    MED_QUAD8,
							    MED_POLYGONE};
  med_geometrie_element typare[MED_NBR_GEOMETRIE_ARETE] = {MED_SEG2,MED_SEG3};

  char ** AFF; 

  switch (entite) {
  case MED_NOEUD :
    type_geo = typ_noeud;
    nb_geo   = 1;
    AFF      = MED_GEOMETRIE_NOEUD_AFF;
    break;
  case  MED_MAILLE :
  case  MED_NOEUD_MAILLE :
    type_geo = typmai;
    nb_geo   = MED_NBR_GEOMETRIE_MAILLE+2;
    AFF      = MED_GEOMETRIE_MAILLE_AFF;
    break;
  case  MED_FACE :
    type_geo = typfac;
    nb_geo   = MED_NBR_GEOMETRIE_FACE+1;
    AFF      = MED_GEOMETRIE_FACE_AFF;
    break;
  case  MED_ARETE :
    type_geo = typare;
    nb_geo   = MED_NBR_GEOMETRIE_ARETE;
    AFF      = MED_GEOMETRIE_ARETE_AFF;
   break;
  }

  for (k=0;k<nb_geo;k++) {

    /* Combien de (PDT,NOR) a lire */
    nbpdtnor = MEDnPasdetemps(fid,nomcha,entite,type_geo[k]); 
    if (nbpdtnor < 1 ) continue;

    for (j=0;j<nbpdtnor;j++) {

      if ( MEDpasdetempsInfo(fid,nomcha,entite,type_geo[k],
			     j+1, &ngauss, &numdt, &numo, dt_unit,
			     &dt, maa_ass, &local, &nbrefmaa) <0) {
	MESSAGE("Erreur a la demande d'information sur (pdt,nor) : "); 
	ISCRUTE(numdt); ISCRUTE(numo);
	ret = -1; continue;
      };

      printf("\n  +Pas de Temps n.%i (%f) [%s], n. d'ordre %i, avec %i pts de gauss sur le maillage par defaut.\n",numdt,dt,dt_unit,numo,ngauss);

      printf("\tLe maillage par defaut est : |%s|, sur un total de : %i maillages associes\n",
	     maa_ass, nbrefmaa);

      /* Le maillage reference est-il porte par un autre fichier */
      if ( !local ) {

	if ( (lnsize=MEDnValLien(fid,maa_ass) ) < 0 )  {
	  MESSAGE("Erreur a la lecture de la taille du lien : ");
	  SSCRUTE(maa_ass);
	  ret = -1;
	} else {

	  lien = (char *)malloc(lnsize*sizeof(char) + 1);
	  EXIT_IF(lien == NULL,NULL,NULL);

	  if ( MEDlienLire(fid, lien, maa_ass) < 0 )  {
	    MESSAGE("Erreur a la lecture du lien : ");
	    SSCRUTE(maa_ass);SSCRUTE(lien);
	    ret = -1;
	  } else {
        lien[lnsize] = '\0';
	    printf("\tLe maillage |%s| est porte par un fichier distant |%s|\n"
		   ,maa_ass,lien);
	  }
	  free(lien);
	}
      }

      /* Combien de maillages lies aux (nomcha,ent,geo,numdt,numo)  */
      /* Notons que cette information est egalement disponible a partir de MEDpasdetempsInfo */
      if ( (nbrefmaa = MEDnChampRef(fid,nomcha,entite,type_geo[k],numdt,numo) ) < 0 ) {
	MESSAGE("Erreur a la demande du nombre de maillages references par le champ : ");
	SSCRUTE(nomcha); ISCRUTE(numdt); ISCRUTE(numo);
	ret = -1; continue;
      };

      for (l=0;l<nbrefmaa;l++) {

	if ( MEDchampRefInfo(fid,nomcha,entite,type_geo[k],
			     l+1,numdt, numo, maa_ass, &local, &ngauss) <0 ) {
	  MESSAGE("Erreur a la demande d'information sur le maillage utilise par le champ n : "); 
	  ISCRUTE(l+1);
	  ret = -1; continue;
	};

	/* Prend en compte le nbre de pt de gauss automatiquement */
	if ((nval = MEDnVal(fid,nomcha,entite,type_geo[k],numdt,numo,maa_ass,USER_MODE)) <= 0)   {
	  MESSAGE("Erreur a la lecture du nombre de valeurs du champ : "); 
	  SSCRUTE(nomcha);ISCRUTE(entite);ISCRUTE(type_geo[k]);
	  ISCRUTE(numdt);ISCRUTE(numo);SSCRUTE(maa_ass);ISCRUTE(USER_MODE);
	  ret = -1; continue;
	};

	if (! strcmp(maa_ass,maillage)) {
	  printf("\t- Il y a %d valeurs en mode %i. Chaque entite %s\
 de type geometrique %s associes au maillage |%s| a %i pts de gauss \n",
	       nval,USER_MODE,MED_ENTITE_MAILLAGE_AFF[(int)entite],AFF[k],
		 maa_ass,ngauss);

	  /* Le maillage reference est-il porte par un autre fichier */
	  if ( !local ) {

	    if ( (lnsize=MEDnValLien(fid,maa_ass) ) < 0 )  {
	      MESSAGE("Erreur a la lecture de la taille du lien : ");
	      SSCRUTE(maa_ass);
	      ret = -1;
	    } else {
	      lien = (char *)malloc(lnsize*sizeof(char) + 1);
	      EXIT_IF(lien == NULL,NULL,NULL);

	      if ( MEDlienLire(fid, lien, maa_ass) < 0 )  {
		MESSAGE("Erreur a la lecture du lien : ");
		SSCRUTE(maa_ass);SSCRUTE(lien);
		ret = -1;
	      } else {
        lien[lnsize] = '\0';
		printf("\tLe maillage |%s| est porte par un fichier distant |%s|\n",maa_ass,lien);
	      }
	    free(lien);
	    }
	  }

	/*Lecture des valeurs du champ */
	if (typcha == MED_FLOAT64) {

	  valr = (med_float*) calloc(ncomp*nval,sizeof(med_float));
	  EXIT_IF(valr == NULL,NULL,NULL);
	  if ( MEDchampLire(fid,maa_ass,nomcha,(unsigned char*)valr,stockage,
			    MED_ALL,locname,
			    pflname,USER_MODE,entite,
			    type_geo[k],numdt,numo) < 0 ) {
	    MESSAGE("Erreur a la lecture du nombre de valeurs du champ : ");
	    SSCRUTE(nomcha);ISCRUTE(entite);ISCRUTE(type_geo[k]);
	    ISCRUTE(numdt);ISCRUTE(numo);SSCRUTE(maa_ass);
	    ret = -1;
	  };

	} else {

	  vale = (med_int*) calloc(ncomp*nval,sizeof(med_int));
	  EXIT_IF(vale == NULL,NULL,NULL);
	  if ( MEDchampLire(fid,maa_ass,nomcha,(unsigned char*)vale,stockage,MED_ALL,locname,
			    pflname,USER_MODE,entite,type_geo[k],numdt,numo) < 0 ) {
	    MESSAGE("Erreur a la lecture des valeurs du champ : "); 
	    SSCRUTE(nomcha);ISCRUTE(entite);ISCRUTE(type_geo[k]);
	    ISCRUTE(numdt);ISCRUTE(numo);SSCRUTE(maa_ass);
	    ret = -1;
	  };	
	  
	}

    if (entite == MED_NOEUD_MAILLE )
      ngroup = type_geo[k] % 100;
    else
      ngroup = ngauss;

      switch (stockage) {
      
      case MED_FULL_INTERLACE : 
      if (!structure) {
	  printf("\t- Valeurs :\n\t");
	  for (m=0;m<nval/ngroup;m++) {
	    printf("|");
	    for (n=0;n<ngroup*ncomp;n++)
	      if (typcha == MED_FLOAT64)
		printf(" %f ",*(valr+(m*ngroup*ncomp)+n));
	      else
		printf(" %d ",*(vale+(m*ngroup*ncomp)+n));

	  }
	  }
	break;

      case MED_NO_INTERLACE :
      if (!structure) {
	  printf("\t- Valeurs :\n\t");
	  for (m=0;m<ncomp;m++) {
	    printf("|");
	    for (n=0;n<nval;n++)
	      if (typcha == MED_FLOAT64)
		printf(" %f ",*(valr+(m*nval)+n));
	      else
		printf(" %d ",*(vale+(m*nval)+n));
	  }
	  }
	break;
      }
      
    if (!structure) {
	printf("|\n");
	}
      if (typcha == MED_FLOAT64) {
	if ( valr ) {free(valr);valr = NULL;}}
      else
	if (vale) { free(vale);vale = NULL; }
      
      /*Lecture du profil associe */
	if (strcmp(pflname,MED_NOPFL) == 0 )
	  printf("\t- Profil : MED_NOPFL\n");
	else {
	
	  if ( (pflsize = MEDnValProfil(fid,pflname)) <0 )  {
	    MESSAGE("Erreur a la lecture du nombre de valeurs du profil : "); 
	    SSCRUTE(pflname);
	    ret = -1; continue;
	  }
		  
	  printf("\t- Profil : |%s| de taille %i\n",pflname,pflsize);
	  
	  pflval = (med_int*) malloc(sizeof(med_int)*pflsize);
	  EXIT_IF(pflval == NULL,NULL,NULL);
	  if ( MEDprofilLire(fid,pflval,pflname) <0) {
	    MESSAGE("Erreur a la lecture des valeurs du profil : ");
	    SSCRUTE(pflname);
	    ret = -1;
	  }
      if (!structure) {
	  printf("\t");
	  for (m=0;m<pflsize;m++) printf(" %i ",*(pflval+m));
	  printf("\n");
	  }
	  free(pflval);

	}
	}

      }
    }
  } /* fin for sur les mailles*/

  return ret;
}

/******************************************************************************
 * 
 * - Nom de la fonction : lecture_resultats
 * - Description : lecture et affichage des champs de resultats
 *                 associe a un  maillage MED.
 * - Parametres :
 *     - fid                       (IN) : ID du fichier MED.
 *     - maillage                  (IN) : nom du maillage maillage.
 *     - mode_coo                  (IN) : mode de stockage en memoire :
 *                                        MED_FULL_INTERLACE |
 *                                        MED_NO_INTERLACE.
 *     - lecture_en_tete_seulement (IN) : mode de lecture.
 ******************************************************************************/

void lecture_resultats(med_idt fid, 
		       char * maillage,
		       med_mode_switch mode_coo,
		       int lecture_en_tete_seulement)
{
  med_err ret,lret;
  char pflname[MED_TAILLE_NOM+1]="",nomlien[MED_TAILLE_NOM+1]="";
  char * lien = NULL;
  char *comp, *unit;
  char nomcha  [MED_TAILLE_NOM+1]="";
  char locname[MED_TAILLE_NOM+1]="";
  med_int mdim,ncomp,ncha,npro,nln,pflsize,*pflval,nval,nloc,ngauss;
  med_type_champ typcha;
  med_maillage type;
  int t1,t2,t3;
  med_geometrie_element type_geo;
  med_float *refcoo, *gscoo, *wg;
  int i,j;
  
  if (! lecture_en_tete_seulement) {
    fprintf(stdout,"\n(************************)\n");
    fprintf(stdout,"(* CHAMPS DU MAILLAGE : *)\n");
    fprintf(stdout,"(************************)\n");
  }

  /* combien de champs dans le fichier */
  ncha = MEDnChamp(fid,0);
  EXIT_IF(ncha < 0,"lors de la lecture du nombre de champs",NULL);
  fprintf(stdout,"- Nombre de champs : %d \n",ncha);

  if (lecture_en_tete_seulement)
    return;

  /****************************************************************************
  *                       LECTURE DES CHAMPS                                  *
  ****************************************************************************/
  ret = 0;

  /* lecture de tous les champs  pour le maillage selectionne */
  for (i =0;i<ncha;i++) {
    lret = 0;
    printf("\nChamp numero : |%d| \n",i+1);

    /* Lecture du nombre de composantes */
    if ((ncomp = MEDnChamp(fid,i+1)) < 0) {
      MESSAGE("Erreur  la lecture du nombre de composantes : ");
      ISCRUTE(ncomp);
      ret = -1; continue;
    }

    /* Lecture du type du champ, des noms des composantes et du nom de
       l'unit*/
    comp = (char*) malloc(ncomp*MED_TAILLE_PNOM+1);
    EXIT_IF(comp == NULL,NULL,NULL);
    unit = (char*) malloc(ncomp*MED_TAILLE_PNOM+1);
    EXIT_IF(unit == NULL,NULL,NULL);

    if ( MEDchampInfo(fid,i+1,nomcha,&typcha,comp,unit,ncomp) < 0 ) {
      MESSAGE("Erreur  la demande d'information sur les champs : ");
      ret = -1; continue;
    }
      
    printf("Nom du champ : |%s| de type |%d|\n",nomcha,typcha);
    printf("Nom des composantes : |%s|\n",comp);
    printf("Unites des composantes : |%s| \n",unit);
    
    free(comp);
    free(unit);
    
    /* champs aux noeuds */
    lret = getFieldsOn(fid, maillage, nomcha, typcha, ncomp, MED_NOEUD,mode_coo);
    
    /* champs sur les elements et aux points de Gauss */
    if (lret == 0) lret = getFieldsOn(fid, maillage, nomcha, typcha, ncomp, MED_MAILLE,mode_coo);
    else { MESSAGE("Erreur  la lecture des champs aux noeuds "); ret = -1; continue;}
   
    if (lret == 0) lret = getFieldsOn(fid, maillage, nomcha, typcha, ncomp, MED_FACE,mode_coo);
    else { MESSAGE("Erreur  la lecture des champs aux mailles "); ret = -1; continue;}
   
    if (lret == 0) lret = getFieldsOn(fid, maillage, nomcha, typcha, ncomp, MED_ARETE,mode_coo);
    else {MESSAGE("Erreur  la lecture des champs aux faces "); ret = -1; continue;}
    
    if (lret == 0) lret = getFieldsOn(fid, maillage, nomcha, typcha, ncomp, MED_NOEUD_MAILLE,mode_coo);
    else {MESSAGE("Erreur a la lecture des champs aux aretes "); ret = -1; continue;}
    
    if  (lret != 0) {MESSAGE("Erreur a la lecture des champs aux noeuds des mailles "); ret = -1;};
  }


  /* Interrogation des profils */
  npro = MEDnProfil(fid);

  printf("\nNombre de profils stockes : %i\n\n",npro);
  for (i=1 ; i <= npro ; i++ ) {
    if ( MEDprofilInfo(fid, i, pflname, &nval) < 0)  {
      MESSAGE("Erreur a la demande d'information sur le profil n : "); ISCRUTE(i);
      ret = -1;continue;
    }
    printf("\t- Profil n%i de nom |%s| et de taille %i\n",i,pflname,nval);
    pflval = (med_int*) malloc(sizeof(med_int)*nval);
    if ( MEDprofilLire(fid, pflval, pflname) < 0) {
      MESSAGE("Erreur a la lecture des valeurs du profil : "); 
      SSCRUTE(pflname);
      ret = -1;
    } else {
      if (!structure) {
      printf("\t");
      for (j=0;j<nval;j++) printf(" %i ",*(pflval+j));
      printf("\n\n");
      }
    }
    free(pflval);
  }
  
  /* Interrogation des liens */
  nln = MEDnLien(fid);
  
  printf("\nNombre de liens stockes : %i\n\n",nln);
  for (i=1 ; i <= nln ; i++ ) {
    if ( MEDlienInfo(fid, i, nomlien, &nval) < 0)  {
      MESSAGE("Erreur a la demande d'information sur le lien n : "); ISCRUTE(i);
      ret = -1;continue;
    }
    printf("\t- Lien n%i de nom |%s| et de taille %i\n",i,nomlien,nval);

    lien = (char *)malloc(nval*sizeof(char) + 1);
    EXIT_IF(lien == NULL,NULL,NULL);

    if ( MEDlienLire(fid, lien, nomlien) < 0 )  {
      MESSAGE("Erreur a la lecture du lien : "); 
      SSCRUTE(nomlien);SSCRUTE(lien);
      ret = -1; 
    } else {
      lien[nval] = '\0';
      printf("\t\t|%s|\n\n",lien);
    }
    free(lien);
  }
  
  /* Interrogation des localisations des points de GAUSS */
  nloc = MEDnGauss(fid);
  
  printf("\nNombre de localisations stockees : %i\n\n",nloc);
  for (i=1 ; i <= nloc ; i++ ) {
    if ( MEDgaussInfo(fid, i, locname, &type_geo, &ngauss) < 0)  {
      MESSAGE("Erreur a la demande d'information sur la localisation n : "); ISCRUTE(i);
      ret = -1;continue;
    }
    printf("\t- Loc. n%i de nom |%s| et nbr. de pts de GAUSS %i\n",i,locname,ngauss);
    t1 = (type_geo%100)*(type_geo/100);
    t2 = ngauss*(type_geo/100);
    t3 = ngauss;
    refcoo = (med_float *) malloc(sizeof(med_float)*t1 );
    gscoo  = (med_float *) malloc(sizeof(med_float)*t2 );
    wg     = (med_float *) malloc(sizeof(med_float)*t3 );
    
    if ( MEDgaussLire(fid, refcoo, gscoo, wg, mode_coo, locname ) < 0) {
      MESSAGE("Erreur a la lecture des valeurs de la localisation : "); 
      SSCRUTE(locname);
      ret = -1;
    } else {
      if (!structure) {
      printf("\t  Coordonnees de l'element de reference de type %i :\n\t\t",type_geo);
      for (j=0;j<t1;j++) printf(" %f ",*(refcoo+j));
      printf("\n");
      printf("\t  Localisation des points de GAUSS : \n\t\t");
      for (j=0;j<t2;j++) printf(" %f ",*(gscoo+j)); 
      printf("\n");
      printf("\t  Poids associes aux points de GAUSS :\n\t\t");
      for (j=0;j<t3;j++) printf(" %f ",*(wg+j));    
      printf("\n\n");
      }
    }
    free(refcoo);
    free(gscoo);
    free(wg);
  }

  return;
}

/******************************************************************************
 *
 * - Nom de la fonction : lecture_parametres_scalaires
 * - Description : lecture des parametres scalaires definis
 *                 hors champs et maillages.
 * - Parametres :
 *     - fid                    (IN) : ID du fichier MED.
 *     - lecture_en_tete_seule  (IN) : mode de lecture.
 *
 ******************************************************************************/

void lecture_parametres_scalaires(med_idt fid,
				  int lecture_en_tete_seulement)
{
  med_err ret = 0;
  char nom_scalaire[MED_TAILLE_NOM+1];
  char description[MED_TAILLE_DESC+1];
  med_int vali;
  med_float valr;
  med_int i,n,npdt,j;
  med_type_champ type;
  med_int numdt,numo;
  med_float dt;
  char dt_unit[MED_TAILLE_PNOM+1];

  fprintf(stdout,"\n(*******************************)\n");
  fprintf(stdout,"(* VARIABLES SCALAIRES       : *)\n");
  fprintf(stdout,"(*******************************)\n");

  /* Combien de variables scalaire ? */
  n = MEDnScalaire(fid);
  EXIT_IF(n < 0,"lors de la lecture du nombre de scalaires",NULL);
  fprintf(stdout,"- Nombre de variables scalaires : %d\n",n);

  if (lecture_en_tete_seulement)
    return ;

  for (i=1;i<=n;i++) {

    /* Lecture des infos (type,description) */
    ret = MEDscalaireInfo(fid,i,nom_scalaire,&type,description);
    EXIT_IF(ret < 0,"lors de la lecture des parametres d'un scalaire",NULL);
    fprintf(stdout,"- Scalaire n%d de nom %s \n",i,nom_scalaire);
    if (type == MED_FLOAT64)
      fprintf(stdout,"  Type flottant. \n");
    else
      fprintf(stdout,"  Type entier. \n");
    printf("  Description associee : [%s] \n",description);

    /* Pour chaque scalaire on regarde les valeurs associees
       eventuellement a des pas de temps et des numeros d'ordre */
    npdt = MEDnScalairePasdetemps(fid,nom_scalaire);
    EXIT_IF(npdt < 0,
	    "lors de la lecture du nombre de pas de temps d'un scalaire"
	    ,NULL);
    fprintf(stdout,"   Nombre de valeurs stockees : %d \n",npdt);

    for (j=1;j<=npdt;j++) {
      ret = MEDscalairePasdetempsInfo(fid,nom_scalaire,j,
				      &numdt,dt_unit,&dt,&numo);
      EXIT_IF(ret < 0,
	      "lors de la lecture des parametres d'un pas de temps d'un scalaire",
	      NULL);

      if (numdt == MED_NOPDT)
	fprintf(stdout,"   - Aucun de pas de temps \n");
      else
	fprintf(stdout,
		"   - Pas de de temps de numero %d de valeur %f [%s] \n",numdt,
	       dt,dt_unit);
      if (numo == MED_NONOR)
	fprintf(stdout,"   - Aucun numero d'ordre \n");
      else
	fprintf(stdout,"   - Numero d'ordre : %d \n",numo);

      if (type == MED_FLOAT64) {
	ret = MEDscalaireFlottantLire(fid,nom_scalaire,&valr,numdt,numo);
	fprintf(stdout,"   - Valeur : %f \n",valr);
      }
      else {
	ret = MEDscalaireEntierLire(fid,nom_scalaire,&vali,numdt,numo);
	fprintf(stdout,"   - Valeur : %d \n",vali);
      }
      EXIT_IF(ret < 0,"lors de la lecture de la valeur d'un scalaire",NULL);

    }
  }

  return ;
}


med_idt ouverture_fichier_MED(char *fichier)
{
  med_idt fid;
  med_err ret = 0;
  med_int majeur,mineur,release;

  /* on regarde si le fichier existe */
  ret = (int) access(fichier,F_OK);
  EXIT_IF(ret < 0,"Le fichier n'est pas accessible ou n'existe pas ",
	  fichier);

  /* on regarde s'il s'agit d'un fichier au format HDF 5 */
  ret = MEDformatConforme(fichier);
  EXIT_IF(ret < 0,"Le fichier n'est pas au format HDF V5 ",
	  fichier);

  /* Quelle version de MED est utilise par mdump ? */
  MEDversionDonner(&majeur,&mineur,&release);
  fprintf(stdout,
	  "- Lecture du fichier  l'aide de la bibliotheque MED V%d.%d.%d \n",
	  majeur,mineur,release);

  /* Ouverture du fichier MED en lecture seule */
  fid = MEDouvrir(fichier,MED_LECTURE);
  EXIT_IF( fid < 0,"Ouverture du du fichier ",fichier);

  MEDversionLire(fid, &majeur, &mineur, &release);
  EXIT_IF((majeur < 2 || majeur == 2 && mineur < 2), "Le fichier est antrieur  la version 2.2", NULL);

  return fid;
}

void lecture_en_tete(med_idt fid,char* fichier)
{
  med_int long_fichier_en_tete;
  char *fichier_en_tete;
  med_err ret = 0;

  /* lecture de l'en-tete du fichier (optionnel) */
  long_fichier_en_tete = MEDlFichDes(fid);
  if (long_fichier_en_tete > 0) {
    fichier_en_tete = (char *) malloc(sizeof(char)*(long_fichier_en_tete+1));
    EXIT_IF(fichier_en_tete == NULL,NULL,NULL);

    /* on va lire dans le fichier */
    ret = MEDfichEntete(fid,MED_FICH_DES,fichier_en_tete);
    EXIT_IF(ret < 0,"lecture de l'en-tete du fichier",fichier);

    /* on affiche */
    fprintf(stdout,"- En-tete du fichier : %s \n",fichier_en_tete);

    /* on libere la memoire */
    free(fichier_en_tete);
  }

  return;
}

void parametrage(med_mode_switch *mode_coo,
		 med_connectivite *typ_con)
{
  int reponse;

  fprintf(stdout,"(*****************)\n");
  fprintf(stdout,"(* PARAMETRAGE : *)\n");
  fprintf(stdout,"(*****************)\n");
  fprintf(stdout,"- Mode d'affichage des coordonnes des noeuds ? \n");
  fprintf(stdout,"  1. Mode entrelac : taper 1 \n");
  fprintf(stdout,"  2. Mode non entrelac : taper 2 \n");
  reponse = 0;
  do {
    fprintf(stdout,"  Reponse : ");
    scanf("%d",&reponse);
  } while (reponse != 1 && reponse != 2);
  if (reponse == 1)
    *mode_coo = MED_FULL_INTERLACE;
  else
    *mode_coo = MED_NO_INTERLACE;

  fprintf(stdout,"- Connectivit des lments ? \n");
  fprintf(stdout,"  1. Nodale : taper 1 \n");
  fprintf(stdout,"  2. Descendante : taper 2 \n");
  reponse = 0;
  do {
    fprintf(stdout,"  Reponse : ");
    scanf("%d",&reponse);
  } while (reponse != 1 && reponse != 2);
  if (reponse == 1)
    *typ_con = MED_NOD;
  else
    *typ_con = MED_DESC;

  return;
}


void lecture_information_maillage(med_idt fid,
				  int numero,
				  char *nommaa,
				  med_int *mdim,
				  med_maillage *type_maillage,
				  char *maillage_description)
{
  med_int edim;
  char nom_universel[MED_TAILLE_LNOM+1];
  med_err ret = 0;

  fprintf(stdout,"\n(********************************************)\n");
  fprintf(stdout,"(* INFORMATIONS GENERALES SUR LE MAILLAGE : *)\n");
  fprintf(stdout,"(********************************************)\n");

  /* lecture du nom et de la dimension du maillage */
  ret = MEDmaaInfo(fid,numero,nommaa,mdim,type_maillage,maillage_description);
  EXIT_IF(ret < 0,"Lecture des informations sur le maillage",NULL);

  /* lecture de la dimension de l'espace de calcul */
  edim = MEDdimEspaceLire(fid,nommaa);

  /* affichage des donnees lues */
  fprintf(stdout,"- Nom du maillage : <<%s>>\n",nommaa);
  fprintf(stdout,"- Dimension du maillage : %d\n",*mdim);
  if (edim > *mdim) 
    fprintf(stdout,"- La dimension de l'espace est %d \n",edim);
  if (*type_maillage == MED_NON_STRUCTURE)
    fprintf(stdout,"- Type du maillage : MED_NON_STRUCTURE \n");
  else
    fprintf(stdout,"- Type du maillage : MED_STRUCTURE \n");
  fprintf(stdout,"- Description associee au maillage : %s\n",
	  maillage_description);

 /* lecture du nom universel (presence optionnelle) */
 ret = MEDunvLire(fid,nommaa,nom_universel);
 if (ret == 0)
   fprintf(stdout,"- Nom universel du maillage : %s \n",nom_universel);

 return;
}

/******************************************************************************
 *
 * - Nom de la fonction : main
 * - Description : fonction "main" de l'outil de DUMP d'un fichier MED.
 * - Parametres :
 *     - argc  (IN) : nombre d'arguments sur la ligne de commande.
 *     - argv  (IN) : liste des arguments.
 *
 ******************************************************************************/

int main (int argc, char **argv)
{
  med_err ret = 0;
  med_idt fid;
  int i;
  int numero;
  med_mode_switch mode_coo;
  med_connectivite typ_con;
  int lecture_en_tete_seulement = 0;
  char version_hdf[10];
  char version_med[10];
  med_int mdim,nmaa;
  /* nom du maillage */
  char nommaa[MED_TAILLE_NOM+1];
  char maillage_description[MED_TAILLE_DESC+1];
  med_maillage type_maillage;
  med_int edim;
  med_int majeur_lu,mineur_lu,release_lu;
  med_int nprofils;
  int decalage;

  /****************************************************************************
  *                  TEST DU NOMBRE D'ARGUMENTS                               *
  *                  argument 1 = nom du fichier MED                          *
  ****************************************************************************/
  
  structure = 0;
  decalage = 0;
  if (argc > 2 && strcmp(argv[1], "--structure") == 0) {
    argc--;
    decalage = 1;
    structure = 1;
  }

  EXIT_IF(argc != 2 && argc != 5,"nombre de parametres incorrects\n",NULL);


  /****************************************************************************
  *                      OUVERTURE DU FICHIER EN LECTURE                      *
  ****************************************************************************/
  fid = ouverture_fichier_MED(argv[1 + decalage]);


  /****************************************************************************
   *                     QUESTIONS PRELIMINAIRES                              *
   *    1. Mode d'affichage des coordonnees (entrelace ou non) ?              *
   *    2. Connectivite des elements (nodale ou descendante) ?                *
   ***************************************************************************/
  fprintf(stdout,"\n >>>>>> DUMP DU FICHIER %s >>>>>>\n",argv[1 + decalage]);

  /* lecture et affichage de l'en-tete du fichier */
  lecture_en_tete(fid,argv[1 + decalage]);

  if (argc == 2) 
    parametrage(&mode_coo,&typ_con);
  else {
    if (! strcmp(argv[3 + decalage],"NODALE"))
      typ_con = MED_NOD;
    if (! strcmp(argv[3 + decalage],"DESCENDANTE"))
      typ_con = MED_DESC;

    if (!strcmp(argv[4 + decalage],"NO_INTERLACE"))
      mode_coo = MED_NO_INTERLACE;
    if (!strcmp(argv[4 + decalage],"FULL_INTERLACE"))
      mode_coo = MED_FULL_INTERLACE;
    if (! strcmp(argv[4 + decalage],"LECTURE_EN_TETE_SEULEMENT"))
      lecture_en_tete_seulement = 1;
  }


  /****************************************************************************
   *                      QUEL MAILLAGE LIRE ?                                *
   ***************************************************************************/
  nmaa = MEDnMaa(fid);
  EXIT_IF(nmaa < 0,"lors de la lecture du nombre de maillages",NULL);

  /* Quel maillage lire ? */
  if (argc == 2) {
    fprintf(stdout,"- Il y a %d maillages dans ce fichier \n",nmaa);
    fprintf(stdout,"  Lequel voulez-vous lire (1|2|3|...|n) ?\n");
    do {
      fprintf(stdout,"  Reponse : ");
      scanf("%d",&numero);
    } while (numero > nmaa || numero <= 0);
  }
  else {
    numero = atoi(argv[2 + decalage]);
    EXIT_IF(numero > nmaa || numero <= 0,"ce numero de maillage n'existe pas",
	    NULL);
  }

  /****************************************************************************
   *                       PARAMETRES SCALAIRES                               *
   ****************************************************************************/

  /* on va lire l'ensemble des parametres scalaire */
  lecture_parametres_scalaires(fid,lecture_en_tete_seulement);

  /****************************************************************************
   *                       INFOS GENERALES SUR LE MAILLAGE                    *
   ****************************************************************************/
  lecture_information_maillage(fid,numero,nommaa,&mdim,&type_maillage,
			       maillage_description);


 /****************************************************************************
  *                      LECTURE DU MAILLAGE ET DES RESULTATS ASSOCIES        *
  ****************************************************************************/
 if (type_maillage == MED_NON_STRUCTURE)
   lecture_maillage_non_structure(fid,nommaa,mdim,mode_coo,typ_con,
				  lecture_en_tete_seulement);
 else
   lecture_maillage_structure(fid,nommaa,mdim,mode_coo,
			      lecture_en_tete_seulement);

 /* on lit ensuite les resultats associes au maillage selectionne */
 lecture_resultats(fid,nommaa,mode_coo,lecture_en_tete_seulement);

 /****************************************************************************
 *                      FERMETURE DU FICHIER                                 *
 ****************************************************************************/
 ret = MEDfermer(fid);
 EXIT_IF(ret < 0,"lors de la fermeture du fichier",argv[1 + decalage]);

 fprintf(stdout,"\n >>>>>> FIN DU DUMP DU FICHIER %s >>>>>>\n",argv[1 + decalage]);

 return EXIT_SUCCESS;
}
