""" A wizard page for creating a new project. """


# Major package imports.
import wx

# Enthought library imports.
from enthought.naming.api import Binding, Context
from enthought.pyface.api import HeadingText, ImageResource
from enthought.pyface.wizard.api import WizardPage
from enthought.traits.api import Any, Instance, Str, HasTraits
from enthought.naming.ui.api import NamingTree, NamingTreeModel

class ProjectDetails(HasTraits):
    """ The details of the new project. """

    project_name = Str


class NewProjectPage(WizardPage):
    """ A wizard page for creating a new project. """

    # The image used when displaying error messages.
    ERROR_IMAGE = ImageResource('error')

    #### 'NewProjectPage' interface ###########################################

    # The object that the wizard is configuring.
    obj = Any

    # The root of the template hierarchy to be displayed in the tree.
    root = Instance(Context)

    # The text shown in the 'pretty' title bar.
    text = Str('Choose a project template')

    ###########################################################################
    # 'WizardPage' interface.
    ###########################################################################

    def create_page(self, parent):
        """ Creates the wizard page. """

        panel = wx.Panel(parent, -1, style=wx.CLIP_CHILDREN)
        sizer = wx.BoxSizer(wx.VERTICAL)
        panel.SetSizer(sizer)
        panel.SetAutoLayout(True)

        # The 'pretty' title bar ;^)
        title = HeadingText(panel, text=self.text)
        sizer.Add(title.control, 0, wx.EXPAND | wx.BOTTOM, 5)

        # The tree used to select the project template.
        tree = self._create_tree(panel)
        tree.SetSizeHints(-1, 150)    # fixme - move this into Tree (or Widget)

        sizer.Add(tree, 1, wx.EXPAND)

        # The project name etc.
        details = self._create_details(panel)
        sizer.Add(details, 0, wx.EXPAND)

        # A panel used to display any error messages etc.
        self._error_panel = error_panel = self._create_error_panel(panel)
        error_panel.Show(False)
        sizer.Add(error_panel, 0, wx.EXPAND | wx.TOP | wx.LEFT, 5)

        # Resize the panel to match the sizer's minimum size.
        sizer.Fit(panel)

        # This catches the case when the page is complete by default.
        self._validate()

        return panel

    ###########################################################################
    # Protected 'NewResourcePage' interface.
    ###########################################################################

    def _create_details(self, parent):
        """ Creates the resource details panel. """

        # The root of the workspace.
        from enthought.envisage import get_application
        workspace = get_application().service_registry.get_service(
                  'enthought.envisage.project.IWorkspace'
            )

        default_project_name = self.obj.name = self._get_default_name(
            'New Project', workspace
        )

        details = ProjectDetails(project_name=default_project_name)
        details.on_trait_change(self._on_project_name_changed, 'project_name')

        ui = details.edit_traits(parent=parent, kind='subpanel')
        ui.control.SetFocus()

        return ui.control

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_tree(self, parent):
        """ Creates the template tree. """

        model = NamingTreeModel(root=Binding(name='root', obj=self.root))

        tree = NamingTree(parent, model=model, show_root=False)
        tree.expand_all()
        tree.on_trait_change(self._on_selection_changed, 'selection')

        return tree.control

    def _create_error_panel(self, parent):
        """ Creates the error panel. """

        panel = wx.Panel(parent, -1, style=wx.CLIP_CHILDREN)
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        panel.SetSizer(sizer)
        panel.SetAutoLayout(True)

        # The error bitmap.
        bmp = self.ERROR_IMAGE.create_image().ConvertToBitmap()
        error_bitmap = wx.StaticBitmap(panel, -1, bmp)
        sizer.Add(error_bitmap, 0, wx.EXPAND)

        # The error text.
        self._error_text = wx.StaticText(panel, -1, '')
        sizer.Add(self._error_text, 1, wx.EXPAND | wx.LEFT, 5)

        # Resize the panel to match the sizer's minimum size.
        sizer.Fit(panel)

        return panel

    def _validate(self):
        """ Validate the project name etc. """

        project_name = self.obj.name

        # Validate the project name.
        project_name_valid = self._validate_project_name(project_name)

        # Hide/Show the error panel.
        self._error_panel.Show(not project_name_valid and len(project_name)>0)

        # The page is complete when a template is selected and the project name
        # is valid.
        self.complete = project_name_valid and self.obj.template is not None

        return

    def _validate_project_name(self, project_name):
        """ Returns True if the project name is valid. """

        # The root of the workspace.
        from enthought.envisage import application
        workspace = application.service_registry.get_service(
                  'enthought.envisage.project.IWorkspace'
            )

        if len(project_name) > 0:
            if project_name in workspace.list_names(''):
                self._error_text.SetLabel(
                    'A project named "%s" already exists in the workspace' \
                    % project_name
                )
                valid = False

            else:
                valid = True

        else:
            valid = False

        return valid

    def _get_default_name(self, prefix, context):
        """ Returns a name that is not already bound in a context. """

        names = context.list_names('')
        name  = prefix

        id = 2
        while name in names:
            name = '%s (%d)' % (prefix, id)
            id += 1

        return name

    #### Trait event handlers #################################################

    def _on_selection_changed(self, selection):
        """ Called when a node in the tree is selected. """

        # The tree is in single selection mode (note that the selection is a
        # list of 'Binding's).
        if len(selection) > 0:
            # Templates are grouped into categories (ie. 'Context's).  The user
            # cannot proceed if a category is selected, only when an actual
            # template is selected.
            if isinstance(selection[0].obj, Context):
                self.obj.template = None

            else:
                self.obj.template = selection[0].obj

        self._validate()

        return

    def _on_project_name_changed(self, old, new):
        """ Called when the name is changed. """

        self.obj.name = new.strip()
        self._validate()

        return

#### EOF ######################################################################
