""" An editor whose content is provided by a traits UI. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.traits.api import Any, Instance, Str
from enthought.traits.ui.api import UI

# Local imports.
from editor import Editor


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class TraitsUIEditor(Editor):
    """ An editor whose content is provided by a traits UI. """

    #### 'Editor' interface ###################################################

    # The resource that the editor is creating/editing.
    resource = Any

    #### 'TraitsUIEditor' interface ###########################################

    # The traits ui that represents the editor.
    #
    # The framework sets this to the value returned by 'create_ui'.
    ui = Instance(UI)

    # The name of the traits UI view used to create the ui.
    view = Str

    ###########################################################################
    # 'Window' interface.
    ###########################################################################

    def close(self):
        """ Closes the window. """

        # Give the traits UI a chance to clean itself up.
        if self.ui is not None:
            logger.debug('disposing traits UI for editor [%s]', self)
            self.ui.dispose()

        # Now, actually close the window.
        super(TraitsUIEditor, self).close()

        return

    ###########################################################################
    # Protected 'Window' interface.
    ###########################################################################

    def _create_contents(self, parent):
        """ Creates the window contents.

        'parent' is the toolkit-specific control that is the editor's parent.

        Overridden to call 'create_ui' to get the traits UI.

        """

        self.ui = self.create_ui(parent)

        return self.ui.control

    ###########################################################################
    # 'TraitsUIEditor' interface.
    ###########################################################################

    def create_ui(self, parent):
        """ Creates the traits UI that represents the editor.

        By default it calls 'edit_traits' on the editor's 'resource'. If you
        want more control over the creation of the traits UI then override!

        """

        ui = self.resource.edit_traits(
            kind   = 'panel',
            parent = parent,
            view   = self.view
        )

        return ui

#### EOF ######################################################################
