""" A plugin that offers a UI for user preferences. """


# Enthought library imports.
from enthought.envisage import Plugin
from enthought.traits.api import Instance

# Plugin imports.
from preference_plugin_definition import PreferencePages

# Local imports.
from workbench_preference_node import WorkbenchPreferenceNode


class PreferencePlugin(Plugin):
    """ A plugin that offers a UI for user preferences. """

    # The shared plugin instance.
    instance = None

    #### 'PreferencePlugin' interface #########################################

    # The root of the preference node hierarchy.
    root = Instance(WorkbenchPreferenceNode)
    
    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, **kw):
        """ Creates a new plugin. """

        # Base-class constructor.
        Plugin.__init__(self, **kw)

        # Set the shared instance.
        PreferencePlugin.instance = self
        
        return

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plugin.

        Called exactly once when the plugin is first required.

        """

        # Get all contributions to the preference pages extension point.
        extensions = self.get_extensions(PreferencePages)

        pages = []
        for extension in extensions:
            pages.extend(extension.pages)
            
        # Sort the pages by the length of their category path.  This makes
        # it easy for us to create the preference hierarchy as we know that
        # all of a node's parents will have already been created.
        def sort(a, b):
            ca = a.category.split('/')
            cb = b.category.split('/')

            return cmp(len(ca), len(cb))

        pages.sort(sort)

        # Create a corresponding preference node hierarchy (the root of the
        # hierachy is NOT displayed in the preference dialog).
        self.root = root = WorkbenchPreferenceNode(name='Root')

        for page in pages:
            # Get the page's parent preference node.
            parent = self._get_parent(root, page)

            # Add a child node representing the page.
            parent.append(
                WorkbenchPreferenceNode(
                    id         = page.id,
                    name       = page.name,
                    class_name = page.class_name,
                    help_id    = page.help_id
                )
            )

        return
    
    def stop(self, application):
        """ Stops the plugin.

        Called exactly once when either the plugin is manually stopped or the
        application exits.

        """

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _get_parent(self, root, page):
        """ Returns the page's parent preference node. """

        parent = root
        
        if len(page.category) > 0:
            components = page.category.split('/')
            for component in components:
                parent = parent.lookup(component)

        return parent
        
#### EOF ######################################################################
