#-----------------------------------------------------------------------------
#
# Copyright (c) 2006 by Enthought, Inc.
# All rights reserved.
#
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
#
# Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

"""
A resource type to recognize resources that represent actions.

"""

# Enthought library imports
from enthought.envisage.resource import ObjectResourceType
from enthought.envisage.resource.action_resource import ActionResource
from enthought.envisage.resource.action_resource_node_type import \
    ActionResourceNodeType
from enthought.envisage.resource.action_resource_serializer import \
    ActionResourceSerializer
from enthought.traits.api import Instance

# Local imports


class ActionResourceResourceType(ObjectResourceType):
    """
    A resource type to recognize resources that represent actions.

    These resources are typically used in controls, such as tree controls, in
    order to lead a user to the appropriate actions to take within that
    control.  They are meant to be inserted programmatically within a context
    when the action could be performed.

    Note that, because these actions are meant to be programmatically inserted,
    they are not serialized as part of the context.  We accomplish the
    non-serialization by specifying a custom serializer for this resource type.

    This implementation defers all 'look and feel' behavior to the recognized
    action resource.

    """

    ##########################################################################
    # Attributes
    ##########################################################################

    #### public 'ResourceType' interface #####################################

    # A description of the kind of domain object this resource type represents.
    type = Instance(ActionResource)


    ##########################################################################
    # 'ResourceType' interface.
    ##########################################################################

    #### public interface ####################################################

    def get_name(self, obj):
        """
        Return the name of the specified object of this resource type.

        Overridden to use the name of the recognized resource.

        """

        return obj.name


    def set_name(self, obj, name):
        """
        Sets the name of an object of this resource type.

        Overridden to make name changing a no-op since we don't want users
        changing the name of our actions.

        """

        return


    #### protected interface #################################################

    def _node_type_default(self):
        """
        Initialize the node type for this class.

        Overridden to return an ActionNodeType.

        """

        return ActionResourceNodeType(resource_type=self)


    def _serializer_default(self):
        """
        Initialize the serializer for this class.

        Overridden to ensure our resources don't get serialized.

        """

        return ActionResourceSerializer()


#### EOF #####################################################################

