""" Builds menus from action, group and menu extensions. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage.ui.workbench_action_proxy \
     import WorkbenchActionProxy, WorkbenchActionProxyFactory
from enthought.pyface.action.api import Group, MenuManager
from enthought.traits.api import Any, HasPrivateTraits, Str


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class MenuBuilder(HasPrivateTraits):
    """ Builds menus from action, group and menu extensions. """

    action_path = Str('path')
    window = Any
    ###########################################################################
    # 'MenuBuilder' interface.
    ###########################################################################

    def build_menu(self, application, menu_manager, menus, groups, actions):
        """ Builds a menu from a list of extensions.

        The extensions contain actions, groups and menus.

        """

        self._add_menus(application, menu_manager, menus)
        self._add_actions(application, menu_manager, actions)

        return

##     def build_menu(self, menu_manager, menus, groups, actions):
##         """ Builds a menu from a list of extensions.

##         The extensions contain actions, groups and menus.

##         """

##         menus_and_groups = menus + groups

##         def sort(a, b):
##             return cmp(len(a.path.split('/')), len(b.path.split('/')))

##         menus_and_groups.sort(sort)

##         for item, defined_in in menus_and_groups:
##             # fixme: Determine whether we are adding a menu or a group in a
##             # nicer way!
##             if hasattr(item, 'groups'):
##                 # A menu
##                 self._add_menus(menu_manager, [(item, defined_in)])

##             else:
##                 # A group.
##                 self._add_groups(menu_manager, [(item, defined_in)])

##         self._add_actions(menu_manager, actions)

##         return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _add_menus(self, application, menu_manager, menus):
        """ Creates a menu manager. """

        for menu, extension_id in menus:
            # Parse the menu path.
            #
            # All but the last component must be the Id of an existing menu
            # manager.
            components = menu.path.split('/')

            # If there is only one component then the menu is added to the top
            # level menu manager.
            if len(components) == 1:
                manager = menu_manager

            # Otherwise, we follow the path until we find the menu to add to.
            else:
                manager = menu_manager.find_item('/'.join(components[:-1]))

            # Make sure that the manager has an 'additions' group.
            additions = manager.find_group('additions')
            if additions is None:
                manager.append(Group(name='additions'))

            # The last component is the name of the group to add the menu to.
            # If the group name is the empty string, the menu is added to the
            # 'additions' group.
            group_name = components[-1]
            if len(group_name) == 0:
                group_name = 'additions'

            # If the specified group doesn't exist, but the requested group
            # name indicates we want to add additional groups to the root menu,
            # then prepare references to the root menu.
            group = manager.find_group(group_name)
            if (group is None) and (group_name == "[root]"):
                sub_menu_manager = manager

            # Otherwise, it is an error if the group doesn't exist
            elif group is None:
                logger.error("Menu group not found %s" % menu.path)
                continue

            # If the group exists, we want to add the new menu to it, but not
            # if a menu already exists with that id.  In that case, we do allow
            # additional groups to be contributed to the existing menu.
            else:
                sub_menu_manager = group.find(menu.id)
                if sub_menu_manager is None:
                    if menu.class_name == '':
                        sub_menu_manager = MenuManager(id=menu.id, name=menu.name)
                    else:
                        menu_class = application.import_symbol(menu.class_name)
                        sub_menu_manager = menu_class(id=menu.id, name=menu.name)
                    group.append(sub_menu_manager)

            # Create all of the menu's groups.
            for new_group in menu.groups:
                # Don't add the group if it already exists.
                if sub_menu_manager.find_group(new_group.id) is None:
                    # Default to -1 to make sure that the 'additions' group is
                    # always the last one.
                    index = -1

                    # Atttempt to find the requested insertion point for the
                    # new group.
                    for i in range(len(sub_menu_manager.groups)):
                        if new_group.before == sub_menu_manager.groups[i].id:
                            index = i
                            break
                        elif new_group.after == sub_menu_manager.groups[i].id:
                            index = i + 1
                            break;

                    # Add the group at the request location
                    sub_menu_manager.insert(index, Group(id=new_group.id))

        return

    def _add_groups(self, menu_manager, groups):
        """ Adds groups to a menu manager. """

        for group, extension_id in groups:
            # Parse the menu path.
            #
            # All but the last component must be the Id of an existing menu
            # manager.
            components = group.path.split('/')

            # If there is only one component then the menu is added to the top
            # level menu manager.
            if len(components) == 0:
                manager = menu_manager

            # Otherwise, we follow the path until we find the menu to add to.
            else:
                #manager = menu_manager.find_item('/'.join(components[:-1]))
                manager = menu_manager.find_item('/'.join(components))

            # If a menu already exists with this Id then we don't create a new
            # menu (obviously!), but we do allow additional groups to be
            # contributed to it.
            if manager.find_group(group.id) is None:
                manager.append(Group(id=group.id))

        return

    def _add_actions(self, application, menu_manager, actions):
        """ 'Hang' actions on the menu bar. """

        factory = WorkbenchActionProxyFactory()

        for action, extension_id in actions:
            path = getattr(action, self.action_path)

            # If the path is empty then the action is not displayed on a menu.
            if len(path) > 0:
                # All but the last path component must be the Id of an
                # existing menu manager.
                components = path.split('/')
                if len(components) == 1:
                    manager = menu_manager
                else:
                    # Follow the path until we find the menu to add the action
                    # to.
                    manager = menu_manager.find_item('/'.join(components[:-1]))
                    if not manager:
                        logger.error("Menu manager not found %s" % components[:-1])
                        continue


                # The last component is the name of the group to add the action
                # to.  If the group name is the empty string, the action is
                # added to the 'additions' group.
                group_name = components[-1]
                if len(group_name) == 0:
                    group_name = 'additions'

                # Make sure that the group exists!
                group = manager.find_group(group_name)
                if group is None:
                    logger.error("Menu group not found %s" % group_name)
                    continue
                    #raise ValueError("Menu group not found %s" % path)

                if not isinstance(action, WorkbenchActionProxy):
                    action = factory.create_instance(
                        action, self.window, extension_id
                    )

                group.append(action)

        return

#### EOF ######################################################################
