""" The plugin activator starts and stops plugin implementations. """


# Enthought library imports.
from enthought.traits.api import Dict, Event, HasTraits, Instance, List
from enthought.traits.api import Property, Str
from enthought.util.graph import topological_sort

# Local imports.
from plugin import Plugin
from plugin_factory import PluginFactory


# Setup a logger for this module.
import logging; logger=logging.getLogger(__name__)


class PluginActivator(HasTraits):
    """ Starts and stops plugin implementations. """

    #### 'PluginActivator' interface ##########################################

    # The application that the activator is part of.
    application = Instance('enthought.envisage.core.application.Application')

    # All of the plugin implementations that have been created (plugins are
    # created lazily just before they are started).
    plugins = Property(List(Instance(Plugin)))

    # The start order. This is a list containing the ID of *every* plugin
    # definition in the definition registry, ordered by satisfying all of
    # their ''requires'' attributes (this list says nothing about whether or not
    # each plugin has been started).
    start_order = List(Str)

    ####  Events ####

    # Fired when a plugin is starting.
    plugin_starting = Event(Instance(Plugin))

    # Fired when a plugin has started.
    plugin_started = Event(Instance(Plugin))

    # Fired when a plugin is stopping.
    plugin_stopping = Event(Instance(Plugin))

    # Fired when a plugin has stopped.
    plugin_stopped = Event(Instance(Plugin))

    #### Private interface ####################################################

    # All of the plugin implementations that have been created.
    _plugins = Dict(Str, Instance(Plugin))

    # A plugin factory (creates plugin implementations from definitions).
    _plugin_factory = Instance(PluginFactory)

    # The ID of every plugin that has been started or is in the process of
    # being started.
    _started = Dict(Str, Instance(Plugin))

    ###########################################################################
    # 'PluginActivator' interface.
    ###########################################################################

    #### Initializers #########################################################

    def _start_order_default(self):
        """ Trait initializer for '''start_order'''. """

        registry = self.application.plugin_definition_registry

        # Build a dependency graph.
        graph = {}
        for definition in registry.definitions:
            graph.setdefault(definition.id, []).extend(definition.requires)

        start_order = topological_sort(graph)
        start_order.reverse()

        return start_order

    #### Properties ###########################################################

    def _get_plugins(self):
        """ Returns a list of all active plugin implementations. """

        return self._plugins.values()

    #### Methods ##############################################################

    def get_plugin(self, id):
        """ Return a plugin for a given id if present, else return None
        """

        if self._plugins.has_key(id):
            return self._plugins[id]
        else:
            return None

    def start_plugin(self, id):
        """ Starts the plugin with the specified *id*.

        Notes
        -----
        * The plugin definition ''must'' have already been loaded.
        * All plugins required by the specified plugin will also be started.
        * If the plugin has already been started, then does nothing.

        """

        # Make sure we don't try to start a plugin that has already been
        # started or is in the process of being started.
        if id not in self._started:
            registry = self.application.plugin_definition_registry

            definition = registry.get_definition(id)
            if definition is None:
                raise ValueError('No such plugin definition %s' % id)

            if definition.enabled:
                # Create a plugin implementation from the definition.
                plugin = self._plugin_factory.create_plugin(definition)

                # This makes sure that we don't try to start the plugin again
                # if one of the plugins it requires tries to start it (via a
                # call to 'load_extensions' etc).
                self._started[id] = plugin

                # We also keep track of all plugin implementations (whether or
                # not they have been stopped).
                self._plugins[id] = plugin

                # Make sure that all plugins that are required by the current
                # plugin have been started.
                for required in definition.requires:
                    self.start_plugin(required)

                # Event notification.
                logger.info('plugin %s starting', id)
                self.plugin_starting = plugin

                # Start it up!
                plugin.start(self.application)

                # Event notification.
                logger.info('plugin %s started', id)
                self.plugin_started = plugin

            else:
                logger.debug('plugin %s disabled - not starting', id)

        return

    def stop_plugin(self, id):
        """ Stops the plugin with the specified *id*.

        If the plugin has not been started, then this does nothing.

        """

        # If the plugin has not been started then do nothing,.
        plugin = self._started[id]
        if plugin is not None:
            logger.info('plugin %s stopping', id)

            # Event notification.
            self.plugin_stopping = plugin

            # Stop it!
            plugin.stop(self.application)

            # Unregister all services offered by the plugin.
            plugin.unregister_all_services()

            # This makes sure that we can re-start the plugin (not that this is
            # usually necessary!).
            del self._started[id]

            # Event notification.
            self.plugin_stopped = plugin

            logger.info('plugin %s stopped', id)

        else:
            logger.debug('plugin %s not started' % id)

        return

    def start_all(self):
        """ Starts all plugins in the appropriate order.

        This method starts all plugins whose ''autostart'' attribute is
        '''True''' and all plugins that those plugins require.

        """

        registry = self.application.plugin_definition_registry

        logger.debug('plugin start order %s', self.start_order)
        for id in self.start_order:
            definition = registry.get_definition(id)
            try:
                if definition.autostart:
                    self.start_plugin(definition.id)

            except Exception:
                logger.exception('error starting plugin %s', id)

        return

    def stop_all(self):
        """ Stops all plugins.

        Plugins are stopped in the reverse order that they were started.

        """

        stop_order = self.start_order[:]
        stop_order.reverse()

        for id in stop_order:
            if self._started.has_key(id):
                try:
                    self.stop_plugin(id)

                except:
                    logger.exception('error stopping plugin %s', id)

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    #### Initializers #########################################################

    def __plugin_factory_default(self):
        """ Trait initializer for '''_plugin_factory'''. """

        return PluginFactory(application=self.application)

#### EOF ######################################################################
