#-----------------------------------------------------------------------------
#
#  Copyright (c) 2006 by Enthought, Inc.
#  All rights reserved.
#
#  Author: Martin Chilvers <martin@enthought.com>
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

""" A factory that allows for lazy-loading of menu actions.
"""

# Standard library imports.
import logging

# Enthought library imports.
from enthought.pyface.api import ImageResource
from enthought.traits.api import HasTraits, Instance

# Local imports
from action_proxy import ActionProxy


# Setup a logger for this module.
logger = logging.getLogger(__name__)


class ActionFactory(HasTraits):
    """ A factory that allows for lazy-loading of menu actions.
    """

    ##########################################################################
    # Traits
    ##########################################################################

    #### public 'ActionFactory' interface ####################################

    # The application that the actions are for.
    application = Instance('enthought.envisage.Application')


    ##########################################################################
    # 'ContextActionFactory' interface.
    ##########################################################################

    #### public interface ####################################################

    def create_action(self, extension):
        """ Creates an action implementation from an action extension.

            Unless the action extension specifies *lazy_load* as **False**,
            a proxy action is created that loads the actual action
            implementation on demand.

            Wherever possible, set *lazy_load* to **True**.
        """
        if extension.lazy_load:
            action = self._create_proxy(extension)
        else:
            action = self._create_action(extension)

        return action


    #### protected interface #################################################

    def _create_action(self, extension):
        """ Creates an action implementation.
        """
        logger.debug('Creating action instance for extension [%s]',
            extension.id)

        # We are about to actually import the action class, so make sure that
        # the plugin that contributed it has been started before doing the
        # import.
        plugin_definition = extension._action_set_._definition_
        self.application.start_plugin(plugin_definition.id)
        klass = self.application.import_symbol(extension.class_name)

        # Create an instance of it.
        action = klass(
            accelerator = extension.accelerator,
            description = extension.description,
            id = extension.id,
            name = extension.name,
            style = extension.style,
            tooltip = extension.tooltip,
            window = None
            )

        # We don't set these in the constructor in case the action uses a
        # static trait handler to listen for changes to the 'checked' trait
        # (which is a common idiom). If we did set it in the constructor then
        # it might fire before the name, style and window traits have been
        # initialized (since we can't guarantee the order of traits
        # initialization via a **kw argument).
        action.checked = extension.checked
        action.enabled = extension.enabled

        # Create the action's image resource relative to the directory that
        # the plugin definition lives in.
        action.image = ImageResource(
            name=extension.image, search_path=[plugin_definition.location]
            )

        return action


    def _create_proxy(self, extension):
        """ Creates an action proxy.
        """
        logger.debug('Creating action proxy for extension [%s]',
            extension.id)

        # The plugin definition that the action was defined in.
        plugin_definition = extension._action_set_._definition_

        proxy = ActionProxy(
            bundle = extension,

            application = self.application,
            accelerator = extension.accelerator,
            checked = extension.checked,
            class_name = extension.class_name,
            defined_in = plugin_definition.id,
            description = extension.description,
            enabled = extension.enabled,
            id = extension.id,
            image_path = extension.image,
            lazy_load = extension.lazy_load,
            name = extension.name,
            style = extension.style,
            tooltip = extension.tooltip,
            window = None,

            # fixme: Hacks!
            enabled_when  = extension.enabled_when,
            disabled_when = extension.disabled_when
            )

        return proxy


#### EOF #####################################################################
