/***************************************************************************
 *   Copyright (C) 2005 by Robert Maxe                                     *
 *   mutex@spray.se                                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
//-----------------------------------------------------------------------------------------------------------------------
#include <kmessagebox.h>

#include <klocale.h>
#include <kpopupmenu.h>
#include <qdatetime.h>
#include <qtextbrowser.h>
#include <qprogressbar.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qptrlist.h>
#include <qcursor.h>
#include <qlabel.h>
#include "programview.h"
#include "programitem.h"
#include "channelitem.h"
#include "mainwidget.h"
#include "program.h"
#include "favouritemaster.h"
//-----------------------------------------------------------------------------------------------------------------------
BaseView::BaseView(MainWidget *parent, ProgramMaster *programMaster, KListView *lv,
	QPtrList<Program> *programs, QWidget *frame)
{
	m_programMaster = programMaster;
	m_parent = parent;
	m_programs = programs;
	m_lv = lv;
	m_frame = frame;

	if(m_lv) {
		connect(m_lv, SIGNAL(selectionChanged()), this, SLOT(updateDesc()));
		connect(m_lv, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)), this, SLOT(showPopupMenu(KListView*, QListViewItem*, const QPoint&)));
		connect(m_lv, SIGNAL(doubleClicked(QListViewItem*)), m_parent, SLOT(toggleDetails()));
	}
	connect(m_parent->ws_main, SIGNAL(aboutToShow(QWidget*)), this, SLOT(aboutToShowCheck(QWidget*)));
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::loadSettings(KConfig *cfg)
{
	cfg->setGroup(m_frame->name());
	m_showColWidth = cfg->readNumEntry("showColWidth", 300);
	m_favsOnly = cfg->readBoolEntry("favsOnly", false);
	m_hidePast = cfg->readBoolEntry("hidePast", true);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::saveSettings(KConfig *cfg)
{
	cfg->setGroup(m_frame->name());
	if(m_lv)
		cfg->writeEntry("showColWidth", m_lv->columnWidth(1));
	cfg->writeEntry("favsOnly", m_favsOnly);
	cfg->writeEntry("hidePast", m_hidePast);
	cfg->sync();
}
//-----------------------------------------------------------------------------------------------------------------------
bool BaseView::refresh()
{
	if(!m_lv) return(false);
	QListViewItemIterator it(m_lv);

	if(!it.current()) return(false);

	while(it.current()) {
		it.current()->repaint();
		it.current()->setVisible(true);
		filter((ProgramItem*)it.current());
		++it;
	}
	m_lv->sort();
	updateDesc();
	return(true);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::filter(ProgramItem *pi)
{
	hidePast(pi);
	if(pi->isVisible())
		hideChannel(pi);
	if(pi->isVisible())
		hideCategories(pi);
	if(pi->isVisible())
		hideBlacklist(pi);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::hidePast(ProgramItem *item)
{
	if(!m_hidePast) return;

	if(ProgramMaster::hasEnded(item->program()))
		item->setVisible(false);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::hideChannel(ProgramItem *item)
{
	QListViewItemIterator channels(m_parent->lv_channel, QListViewItemIterator::Selected);
	Program *program = item->program();
	Channel *channel;
	
	if(!channels.current()) return;
	while(channels.current()) {
		channel = ((ChannelItem*)channels.current())->channel();
		if(program->channelId() == channel->id())
			return;
		++channels;
	}
	item->setVisible(false);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::hideBlacklist(ProgramItem *item)
{
	if(m_parent->blistMaster()->isBlacklisted(item->program()->title()))
		item->setVisible(false);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::hideCategories(ProgramItem *pi)
{
	QListViewItemIterator it(m_parent->lv_category, QListViewItemIterator::Selected);
	if(!it.current()) return;

	while(it.current()) {
		if(pi->program()->category().contains(it.current()->text(0), false) > 0)
			return;
		++it;
	}
	pi->setVisible(false);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::hideNonFavs(ProgramItem *item)
{
	if(m_favsOnly && !m_parent->favouriteMaster()->isFavourite(item->program()->title()))
		item->setVisible(false);
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::updateDesc()
{
	if(!m_lv) return;

	QListViewItem *lvi = m_lv->selectedItem();
	QDateTime now(QDateTime::currentDateTime());
	QString descr, temp, search = m_parent->le_search->text(), info = i18n("Info"), actors = i18n("Actors");
	QTime left;
	bool air = false;
	bool hlDesc = (m_parent->cb_searchdesc->isChecked() && m_lv == m_parent->lv_search? true : false);
	bool hlActr = (m_parent->cb_searchactors->isChecked() && m_lv == m_parent->lv_search? true : false);

	if(lvi && lvi->isVisible()) {
		Program *program = ((ProgramItem*)lvi)->program();
		m_parent->tl_showname->setText(program->title() + "  (" + program->channel()->customName() + ")");
		m_parent->tl_timeinfo->setText(program->startDT().toString("hh:mm") + " - " + program->stopDT().toString("hh:mm") + "  (" + program->duration().toString("h:mm") + ")");
		left = left.addSecs((ProgramMaster::elapsed(program)).secsTo(program->duration()));
		m_parent->tl_elapsed->setText(ProgramMaster::elapsed(program).toString("h:mm") + " + " + left.toString("h:mm"));
		m_parent->pb_elapsed->setProgress(ProgramMaster::elapsed(program).hour()*60+ProgramMaster::elapsed(program).minute(), program->duration().hour()*60+program->duration().minute());

		if(!program->subtitle().isEmpty()) {
			descr += "<b>" + i18n("Title") + ":</b> " + program->subtitle();
			air = true;
		}		
		if(!program->desc().isEmpty()) {
			if(!descr.isEmpty()) descr += "<br>";
			if(hlDesc) {
				temp = program->desc();
				temp = temp.replace(search, ("<font color=\"blue\">" + search.upper() + "</font>"), false);
				descr += "<b>" + info + ":</b> " + temp;
			} else {
				descr += "<b>" + info + ":</b> " + program->desc();
			}
		} else
			air = true;
		if(m_parent->prefs()->cb_directortag->isChecked() && !program->director().isEmpty()) {
			if(!descr.isEmpty()) descr += "<br>";
			descr += "<br><b>" + i18n("Director") + ":</b> " + program->director();
			air = true;
		}
		if(m_parent->prefs()->cb_actorstag->isChecked() && !program->actors().isEmpty()) {
			if(!air) descr += "<br>";
			if(!descr.isEmpty()) descr += "<br>";
			if(hlActr) {
				temp = program->actors();
				temp = temp.replace(search, ("<font color=\"blue\">" + search.upper() + "</font>"), false);
				descr += "<b>" + actors + ":</b> " + temp;
			} else {
				descr += "<b>" + actors + ":</b> " + program->actors();
			}
			air = true;
		}
		if(m_parent->prefs()->cb_episodetag->isChecked() && !program->episode().isEmpty()) {
			if(!air) descr += "<br>";
			if(!descr.isEmpty()) descr += "<br>";
			descr += "<b>" + i18n("Episode") + ":</b> " + program->episode();
			air = true;
		}
		if(m_parent->prefs()->cb_categorytag->isChecked() && !program->category().isEmpty()) {
			if(!air) descr += "<br>";
			if(!descr.isEmpty()) descr += "<br>";
			descr += "<b>" + i18n("Category") + ":</b> " + program->category();
			air = true;
		}
		if(m_parent->prefs()->cb_datetag->isChecked() && !program->date().isEmpty()) {
			if(!air) descr += "<br>";
			if(!descr.isEmpty()) descr += "<br>";
			descr += "<b>" + i18n("Date") + ":</b> " + program->date();
			air = true;
		}
		if(m_parent->prefs()->cb_aspecttag->isChecked() && !program->aspect().isEmpty()) {
			if(!air) descr += "<br>";
			if(!descr.isEmpty()) descr += "<br>";
			descr += "<b>" + i18n("Aspect") + ":</b> " + program->aspect();
			air = true;
		}
		m_parent->tb_desc->setText(descr);

		m_parent->fr_details1->show();
		m_parent->fr_details2->show();
	} else {
		m_parent->fr_details1->hide();
		m_parent->fr_details2->hide();
		m_parent->tb_desc->clear();
	}
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::saveSelection()
{
	if(m_lv && m_lv->currentItem()) {
		Program *program = ((ProgramItem*)m_lv->currentItem())->program();
		m_selection = program->start() + program->channelId();
	} else
		m_selection = "";
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::restoreSelection()
{
	if(!m_lv) return;
	QListViewItemIterator it(m_lv, QListViewItemIterator::Visible);

	if(m_selection.isEmpty())
		return;
	while(it.current()) {
		Program *program = ((ProgramItem*)it.current())->program();
		if(m_selection == (program->start() + program->channelId())) {
			m_lv->setCurrentItem(it.current());
			break;
		}
		++it;
	}
}
//-----------------------------------------------------------------------------------------------------------------------
void BaseView::aboutToShowCheck(QWidget *wid)
{
	if(wid == m_frame)
		aboutToShow();
}
//-----------------------------------------------------------------------------------------------------------------------
