/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QtGui>
#include <QtGlobal>


/////////////////////// Local includes
#include "mzLabWnd.hpp"
#include "mainWindow.hpp"
#include "application.hpp"
#include "mzLabInputOligomerTableViewDlg.hpp"
#include "mzLabOutputOligomerTableViewDlg.hpp"
#include "oligomerPair.hpp"
#include "fragmentOligomer.hpp"


namespace massXpert
{

  MzLabWnd::MzLabWnd(QString &filePath)
  {
    m_forciblyClose = false;
    m_noDelistWnd = false;

    if (filePath.isEmpty())
      {
	QMessageBox::warning(0, 
			      tr("massXpert - Calculator"),
			      tr("Polymer chemistry definition "
				  "filepath empty."),
			      QMessageBox::Ok);
      
	return;
      }
  
    m_polChemDef.setFilePath(filePath);

    if (!initialize())
      {
	QMessageBox::warning(0, 
			      tr("massXpert - Calculator"),
			      tr("Failed to initialize the mz lab window."),
			      QMessageBox::Ok);
      }

    setWindowTitle(tr("%1 %2[*]")
		    .arg(tr("massXpert: mz Lab - "))
		    .arg(m_polChemDef.name()));
    
    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);
    
    settings.beginGroup("mz_lab_wnd");

    restoreGeometry(settings.value("geometry").toByteArray());

    m_ui.leftSplitter->
      restoreState(settings.value("leftSplitter").toByteArray());
    m_ui.rightSplitter->
      restoreState(settings.value("rightSplitter").toByteArray());
    
    settings.endGroup();

    show();
  }
    

  MzLabWnd::~MzLabWnd()
  {
    while(m_dlgList.size())
      delete m_dlgList.takeFirst();
  }
  

  void 
  MzLabWnd::closeEvent(QCloseEvent *event)
  {
    Application *application = static_cast<Application *>(qApp);

    // We are asked to close the window even if it has unsaved data.
    if (m_forciblyClose)
      {
	m_forciblyClose = false;
      
	// We have to delist the window.
	if(!m_noDelistWnd)
	  {
	    m_noDelistWnd = false;
	    int index = application->mzLabWndList()->indexOf(this);
	  
	    application->mzLabWndList()->takeAt(index);
	  }
      
	writeSettings();
	
	event->accept();
      
	return;
      }

    if (maybeSave())
      {
	// We are asked not to remove this window from the list of all
	// the mz lab windows. This occurs when all the windows are
	// closed in a raw in the application object.

	if(m_noDelistWnd)
	  {
	    m_noDelistWnd = false;
	  
	    writeSettings();

	    event->accept();
	  
	    return;
	  }
      
	// We must remove this window from the application's list of
	// such windows:

	int index = application->mzLabWndList()->indexOf(this);
      
	application->mzLabWndList()->takeAt(index);
  
	writeSettings();
      
	event->accept();
      }
    else 
      {
	event->ignore();
      }
  }
  
  
  bool
  MzLabWnd::initialize()
  {
    m_ui.setupUi(this);
    
    QPixmap pixmap(":/images/massxpert-icon-32.png");
    QIcon icon(pixmap);
    setWindowIcon(icon);
  
    setAttribute(Qt::WA_DeleteOnClose);
    statusBar()->setSizeGripEnabled(true);

    m_ui.ionizationChargeSpinBox->setRange(1, 1000000000);
    m_ui.ionizationLevelSpinBox->setRange(0, 1000000000);

    m_ui.incrementChargeSpinBox->setRange(-1000000000, 1000000000);

    // The tolerance when filtering mono/avg masses...
    QStringList stringList;
  
    stringList << tr("AMU") << tr("PCT") << tr("PPM");
  
    m_ui.toleranceComboBox->insertItems(0, stringList);
  
    m_ui.toleranceComboBox->setCurrentIndex(0);
    
    m_ui.toleranceComboBox->setToolTip(tr("AMU: atom mass unit \n"
					    "PCT: percent \n"
					    "PPM: part per million"));

    m_ui.toleranceLineEdit->setText("1");
    

    ////// Connection of the SIGNALS and SLOTS //////

    connect(m_ui.applyFormulaPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(applyFormulaPushButton()));
  
    connect(m_ui.applyMassPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(applyMassPushButton()));
  
    connect(m_ui.applyThresholdPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(applyThresholdPushButton()));
  
    connect(m_ui.incrementChargePushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(incrementChargePushButton()));
  
    connect(m_ui.reionizePushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(reionizePushButton()));
  
    connect(m_ui.performMatchWorkPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(performMatchWorkPushButton()));

    connect(m_ui.newInputListPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(newInputList()));

    connect(m_ui.catalogue1ListWidget,
	     SIGNAL(itemClicked(QListWidgetItem *)),
	     this,
	     SLOT(inputListWidgetItemClicked(QListWidgetItem *)));
    
    connect(m_ui.catalogue2ListWidget,
	     SIGNAL(itemClicked(QListWidgetItem *)),
	     this,
	     SLOT(inputListWidgetItemClicked(QListWidgetItem *)));

    connect(m_ui.deleteInputListPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(deleteInputListItem()));

    connect(m_ui.helpPushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(help()));


    if (!m_polChemDef.renderXmlPolChemDefFile())
      return false;

    m_ionizeRule = m_polChemDef.ionizeRule();
    
    m_ui.ionizationFormulaLineEdit->setText(m_ionizeRule.formula());
    m_ui.ionizationChargeSpinBox->setValue(m_ionizeRule.charge());
    m_ui.ionizationLevelSpinBox->setValue(m_ionizeRule.level());
    
    m_ui.reIonizationFormulaLineEdit->setText(m_ionizeRule.formula());
    m_ui.reIonizationChargeSpinBox->setValue(m_ionizeRule.charge());
    m_ui.reIonizationLevelSpinBox->setValue(m_ionizeRule.level());
    
    return true;
  }
 

  PolChemDef *
  MzLabWnd::polChemDef()
  {
    return &m_polChemDef;
  }
  
  
  const IonizeRule &
  MzLabWnd::ionizeRule() const
  {
    return m_ionizeRule;
  }
  
  
  void 
  MzLabWnd::writeSettings()
  {
    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);
    
    settings.beginGroup("mz_lab_wnd");
    
    settings.setValue("geometry", saveGeometry());

    settings.setValue("leftSplitter", m_ui.leftSplitter->saveState());
    settings.setValue("rightSplitter", m_ui.rightSplitter->saveState());
    
    settings.endGroup();
  }
  

  void
  MzLabWnd::updateWindowTitle()
  {
    setWindowTitle(tr("%1 %2[*]")
		    .arg(tr("massXpert: mz Lab - "))
		    .arg(m_polChemDef.name()));
  }
  

  double
  MzLabWnd::calculateTolerance(double value)
  {
    int index = m_ui.toleranceComboBox->currentIndex();
    QString text = m_ui.toleranceLineEdit->text();
    
    bool ok = false;
    double nominal = qAbs(text.toDouble(&ok));
    
    if (!nominal && !ok)
      return -1;
  
    if (index == 0)
      {
	// MXT_MASS_TOLERANCE_AMU
  
	return nominal;
      }
    else if (index == 1)
      {
	// MXT_MASS_TOLERANCE_PCT

	return(nominal *(value / 100));
      }
    else if (index == 2)
      {
	// MXT_MASS_TOLERANCE_PPM

	return(nominal *(value / 1000000));
      }
    else
      Q_ASSERT(0);
    
    return -1;
  }


  void 
  MzLabWnd::fillMolecularMassList(const OligomerList &oligomerList,
				   QList<double> *listM, MassType &massType)
  {
    Q_ASSERT(listM);

    double mass = 0;
    
    for (int iter = 0; iter < oligomerList.size(); ++iter)
      {
	Oligomer *oligomer = oligomerList.at(iter);
	
	FragmentOligomer *fragOligomer =
	  dynamic_cast<FragmentOligomer *>(oligomer);
	
	if(fragOligomer)
	  {
	    // Depending on the massType, select the proper molecular
	    // mass. Note that it is not possible that the mass be < 0.
	    mass = fragOligomer->molecularMass(massType);
	  }
	else
	  mass = oligomer->molecularMass(massType);
	
	if(mass < 0)
	  qFatal("Fatal error at %s@%d. Program aborted. Mass was:%.5f",
		  __FILE__, __LINE__, mass);
	
	listM->append(mass);
      }
  }
  


  void
  MzLabWnd::applyFormulaPushButton()
  {
    MzLabInputOligomerTableViewDlg *dlg = 0;
    
    if (!inputListDlg(&dlg))
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, select one input list first.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }

    QString text = m_ui.formulaLineEdit->text();
    if (text.isEmpty())
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, enter a valid formula.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }
    
    Formula formula(text);
    
    if (!formula.validate(m_polChemDef.atomList()))
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "The formula '%3' is not valid.")
			      .arg(__FILE__)
			      .arg(__LINE__)
			      .arg(formula.text()));
	return;
      }

    // At this point we can ask the list modification.

    bool inPlace = m_ui.inPlaceCalculationCheckBox->isChecked();
    
    dlg->applyFormula(formula, inPlace);
  }
  


  void
  MzLabWnd::applyMassPushButton()
  {
    MzLabInputOligomerTableViewDlg *dlg = 0;
    
    if (!inputListDlg(&dlg))
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, select one input list first.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }
    
    QString text = m_ui.massLineEdit->text();

    if (text.isEmpty())
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, enter a valid mass.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }
    
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();
    
    bool ok = false;
    
    double mass = locale.toDouble(text, &ok);
    
    if (!mass && !ok)
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Failed to convert %3 to double")
			      .arg(__FILE__)
			      .arg(__LINE__)
			      .arg(text));
	return;
      }
    
    // At this point we can ask the list modification.

    bool inPlace = m_ui.inPlaceCalculationCheckBox->isChecked();
    
    dlg->applyMass(mass, inPlace);
  }


  void
  MzLabWnd::applyThresholdPushButton()
  {
    MzLabInputOligomerTableViewDlg *dlg = 0;
    
    if (!inputListDlg(&dlg))    
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, select one input list first.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }
    
   QString text = m_ui.thresholdLineEdit->text();

    if (text.isEmpty())
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, enter a valid threshold.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }


    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();
    
    bool ok = false;
    
    double threshold = locale.toDouble(text, &ok);
    
    if (!threshold && !ok)
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Failed to convert %3 to double")
			      .arg(__FILE__)
			      .arg(__LINE__)
			      .arg(text));
	return;
      }
    
    // At this point we can ask the list modification.

    bool inPlace = m_ui.inPlaceCalculationCheckBox->isChecked();
    bool onMz = m_ui.mzThresholdRadioButton->isChecked();

    dlg->applyThreshold(threshold, inPlace, onMz);
  }


  void 
  MzLabWnd::incrementChargePushButton()
  {
    MzLabInputOligomerTableViewDlg *dlg = 0;
    
    if (!inputListDlg(&dlg))    
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, select one input list first.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }
    
    int increment = m_ui.incrementChargeSpinBox->value();

    if (!increment)
      {
	QMessageBox::warning 
	 (this, 
	   tr("massXpert: mz Lab"),
	   tr("%1@%2\n"
	       "Please, enter a valid increment.")
	   .arg(__FILE__)
	   .arg(__LINE__),
	   QMessageBox::Ok);

	return;
      }
    
    // At this point we can ask the list modification.
    
    bool inPlace = m_ui.inPlaceCalculationCheckBox->isChecked();
    
    dlg->applyChargeIncrement(increment, inPlace);
  }


  void
  MzLabWnd::reionizePushButton()
  {
    MzLabInputOligomerTableViewDlg *dlg = 0;
    
    if (!inputListDlg(&dlg))    
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Please, select one input list first.")
			      .arg(__FILE__)
			      .arg(__LINE__));
	return;
      }
    
    IonizeRule ionizeRule;
    
    ionizeRule.setFormula(m_ui.reIonizationFormulaLineEdit->text());
    ionizeRule.setCharge(m_ui.reIonizationChargeSpinBox->value());
    ionizeRule.setLevel(m_ui.reIonizationLevelSpinBox->value());

    if (!ionizeRule.validate(m_polChemDef.atomList()))
      {
	QMessageBox::warning 
	 (this, 
	   tr("massXpert: mz Lab"),
	   tr("%1@%2\n"
	       "Failed to validate ionization rule.")
	   .arg(__FILE__)
	   .arg(__LINE__),
	   QMessageBox::Ok);

	return;
      }
    
    // At this point we can ask the list modification.
    
    bool inPlace = m_ui.inPlaceCalculationCheckBox->isChecked();

    dlg->applyIonizeRule(ionizeRule, inPlace);
  }
  
  
  void
  MzLabWnd::performMatchWorkPushButton()

  {
    // We have to get the name of two lists. The first list's item
    // should be matched to the second list's items. The user might ask
    // that the second list's items be modified to match the first
    // list's ones.

    MzLabInputOligomerTableViewDlg *dlg1 = 0;
    MzLabInputOligomerTableViewDlg *dlg2 = 0;

    if (!inputListsDlg(&dlg1, &dlg2))    
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("This feature performs matches between "
				  "two input lists.\n"
				  "Please, select two input lists first."),
			      QMessageBox::Ok);
	return;
      }


    // The general idea is to first create for each m/z--z list a
    // corresponding list of M values(that is to first deionize the
    // m/z into M). Next, the comparison will be performed on the M
    // lists. This will ensure that all peaks are matched even if
    // simulation peaks are for z=1 and z=2, for example, and the
    // measured m/z has peaks with z=3. 

    // The m/z--z lists and the M lists will retain a complete
    // correlation, that is the order of their components will not be
    // modified, and the their number neither. Which means that, for
    // example, the item in m/z list 1 [index 0] will correspond to
    // the item in M list [index 0].

    // The comparison of M list 1 with M list 2 will possibly yield
    // matches which will be reported in an output list of which the
    // items will retain links to the initial m/z lists 1 and 2. This
    // way, it will be possible to trace back the matched m/z items
    // for a given match.

    // The dlg1 holds the list that has the measured data(that is
    // factual m/z values that cannot be changed in any way). The dlg2
    // holds the theoretical m/z values that might undergo chemical
    // modifications in order to seek matches in the dlg1 list.

    // The two lists might not have the same item count.

    MassType massType1 = dlg1->massType();
    MassType massType2 = dlg2->massType();

    if (massType1 != massType2)
      {
	int ret = 
	  QMessageBox::question(this, tr("massXpert - mz Lab"),
				tr("The lists to match are not of the same "
				   "mass type.\n\nContinue?"),
				QMessageBox::Yes | QMessageBox::No);
       
	if(ret == QMessageBox::No)
	  return;
      }


    // For clearness...
    MassType massType = massType1;

    //    qDebug() << __FILE__ << __LINE__
    //             << "massType:" << massType;

    // Allocate first the oligomer shell that we'll use later.
    OligomerPair *oligomerPair = 0;

    // What kind of match are we willing?
    bool trueMatches = m_ui.trueMatchesCheckBox->isChecked();
    // qDebug() << __FILE__ << __LINE__ 
    //          << "trueMatches:" << trueMatches;
    
    bool falseMatches = m_ui.falseMatchesCheckBox->isChecked();
    // qDebug() << __FILE__ << __LINE__ 
    //          << "falseMatches:" << falseMatches;

    // Create the two lists that will hold the double values.

    // First list.

    const OligomerList &listMz1 = dlg1->oligomerList();
    QList<double> listM1;
    fillMolecularMassList(listMz1, &listM1, massType);
    Q_ASSERT(listMz1.size() == listM1.size());

    // Second list.

    const OligomerList &listMz2 = dlg2->oligomerList();
    QList<double> listM2;
    fillMolecularMassList(listMz2, &listM2, massType);
    Q_ASSERT(listMz2.size() == listM2.size());

    // Allocate the new list of OligomerPair instances into which
    // we'll append OligomerPairs instances corresponding to
    // successful matches between oligomers.

    QList <OligomerPair *> *oligomerPairList = 
      new QList <OligomerPair *>();

    // At this point we have the two lists with molecular masses. We
    // can start doing the matches.
        
    for (int iter = 0; iter < listM1.size(); ++iter)
      {
	double mass1 =  listM1.at(iter);

        double tolerance = calculateTolerance(mass1);
        
        double lowerBorder = mass1 - tolerance;
        double upperBorder = mass1 + tolerance;
        
        bool foundMatch = false;
        
        for(int jter = 0; jter < listM2.size(); ++jter)
	  {
	    double mass2 =  listM2.at(jter);
            
            if (mass2 >= lowerBorder &&
		mass2 <= upperBorder)
	      {
                // qDebug() << __FILE__ << __LINE__
                //          << "M1 at index:" << iter 
                //          << "matches"
                //          << "M2 at index:" << jter; 
                
		// At this point we can create a new OligomerPair.
                if(trueMatches)
                  {
                    oligomerPair =
                      new OligomerPair(listMz1.at(iter), listMz2.at(jter),
                                       massType,
                                       qAbs(mass2 - mass1), 
                                       true /* isMatching */,
                                       "NOT_SET");

                    oligomerPairList->append(oligomerPair);
                  }
                
                // Whatever asked by the user (trueMatches or not), we
                // must tell the following code that a match was
                // found.
                foundMatch = true;
              }
          }
        
        // Now that we have finished dealing with mass1, let's check
        // if a match was found. If not and the non-matches are
        // requested, store a locally allocated oligomerPair.

        if(!foundMatch && falseMatches)
          {
            // Allocate a duplicate oligomer identical to the oligomer
            // that had no match in the listMz2.
            Oligomer *oligomer = new Oligomer(*listMz1.at(iter));

            oligomerPair = 
              new OligomerPair(listMz1.at(iter), oligomer,
                               massType,
                               0, 
                               false /* isMatching */,
                               "NOT_SET");
            
            oligomerPairList->append(oligomerPair);
          }
      }
    // End of
    // for (int iter = 0; iter < listM1.size(); ++iter)
    
    
    // qDebug() << __FILE__ << __LINE__
    //          << "Appended oligomer pair:"
    //          << oligomerPair->oligomer1()->name()
    //          << oligomerPair->oligomer2()->name()
    //          << "checksum:" << oligomerPair->oligomer1()->
    //   polymer()->checksum();
    
    //    qDebug() << __FILE__ << __LINE__
    //             << "Found pairs:" << oligomerPairList->size();
    
    // Allocate the new dialog, we pass 
  
    QString dialogName = tr("%1 vs %2")
                .arg(dlg1->name())
      .arg(dlg2->name());
    
    // Ownership of oligomerPairList is going to be transferred to the
    // dialog being created (note that to the dialog and not the table
    // view model in the dialog's table view).

    MzLabOutputOligomerTableViewDlg *dlg =
      new MzLabOutputOligomerTableViewDlg(this, oligomerPairList, 
                                          massType1, dlg1, dlg2, 
                                          dlg1->name(), dlg2->name());
    dlg->show();
  }
   

  bool
  MzLabWnd::inputListDlg(MzLabInputOligomerTableViewDlg **dlg)
  {
    QListWidgetItem *item = 0;
    
    // Which list of lists is selected ?
    
    if (m_ui.catalogue1RadioButton->isChecked())
      item = m_ui.catalogue1ListWidget->currentItem();
    else
      item = m_ui.catalogue2ListWidget->currentItem();
    
    if (!item)
      return false;
    
    // Get the name of the dialog window represented by the item and
    // find the dialog window itself.
    
    QString text = item->text();
    
    MzLabInputOligomerTableViewDlg *dialog = findDlg(text);
    Q_ASSERT(dialog);
    
    *dlg = dialog;
    
    return true;
  }
  
  
 bool
 MzLabWnd::inputListsDlg(MzLabInputOligomerTableViewDlg **dlg1,
			  MzLabInputOligomerTableViewDlg **dlg2)
  {
    QListWidgetItem *item1 = 0;
    QListWidgetItem *item2 = 0;

    // Which lists of lists is selected ?
    
    item1 = m_ui.catalogue1ListWidget->currentItem();
    item2 = m_ui.catalogue2ListWidget->currentItem();
    
    if (!item1 || !item2)
      return false;
        
    // Get the name of the dialog window represented by the two items
    // and find the corresponding dialog windows.
    
    QString text = item1->text();
    
    MzLabInputOligomerTableViewDlg *dialog = findDlg(text);
    Q_ASSERT(dialog);
    
    *dlg1 = dialog;

    text = item2->text();
    
    dialog = 0;
    
    dialog = findDlg(text);
    Q_ASSERT(dialog);
    
    *dlg2 = dialog;
    
    return true;
  }
  
  
  MzLabInputOligomerTableViewDlg *
  MzLabWnd::newInputList()
  {
    return newInputList(QString(), MXT_MASS_NONE, false);
  }
  
  MzLabInputOligomerTableViewDlg *
  MzLabWnd::newInputList(QString name, MassType massType, bool isFragment)
  {
    // Ask that the user give a name to the list.
    
    bool ok;
    
    QString localName = name;
    
    if(localName.isEmpty())
      {
        // Each dialog must have a name. So ask for one.
        localName = 
          QInputDialog::getText(this, 
                                tr("Give a name to the new input list"),
                                tr("List name:"), 
                                QLineEdit::Normal,
                                tr("New input list name"), &ok);
        
        if (localName.isEmpty() || !ok)
          return 0;
      }
    else
      {
        // The name is already set, only ask the user to confirm it.
        
        localName = 
          QInputDialog::getText(this, 
                                tr("Confirm the name of the new input list"),
                                tr("List name:"), 
                                QLineEdit::Normal,
                                localName, &ok);
        
        if (localName.isEmpty() || !ok)
          return 0;
      }
    


    // Make sure the name is not already taken.
    if (findDlg(localName))
      {
	QMessageBox::warning(this,
                             tr("massXpert: mz Lab"),
                             tr("%1@%2\n"
                                "The '%3' name is already taken.")
                             .arg(__FILE__)
                             .arg(__LINE__)
                             .arg(localName),
                             QMessageBox::Ok);
	return 0;
      }
    
    MzLabInputOligomerTableViewDlg *dlg = 
      new MzLabInputOligomerTableViewDlg(this, localName, massType, isFragment);
    
    dlg->show();
    
    // Now we can add the item to the lists of input...

    m_ui.catalogue1ListWidget->addItem(localName);
    m_ui.catalogue2ListWidget->addItem(localName);

    // Finally we can add the list to the list of dialog windows.
    
    m_dlgList.append(dlg);

    setWindowModified(true);
    
    return dlg;
  }
  

  void
  MzLabWnd::help()
  {
    // Explain to the user what's the point of having matches
    // performed.

    QString msg(tr("To perform a match operation one must have\n"
                   "two lists of (m/z,z) pairs already available.\n\n"
                   "One list must be selected in the Catalogue 1\n"
                   "and the other list must be selected in the Catalogue 2.\n\n"
                   "There are two kinds of \"match work\" operations : \n"
                   "One that aims at finding matches between (m/z,z) pairs\n"
                   "in each of the two lists and one that aims at finding\n"
                   "(m/z,z) pairs that have no match in the other list.\n\n"
                   "How are match (or non-match) works performed ?\n"
                   "One match work is performed by searching in\n"
                   "the first list (Catalogue 1) all the (m/z,z) pairs \n"
                   "that are also in the second list (Catalogue 2).\n\n"
                   "One non-match work is performed by searching in the\n"
                   "first list (Catalogue 1) all the (m/z,z) pairs that are\n"
                   "not found in the second list (Catalogue 2).\n"));
    
    QMessageBox::warning(this,
                         tr("massXpert: mz Lab"),
                         msg,
                         QMessageBox::Ok);
  }
  

  MzLabInputOligomerTableViewDlg *
  MzLabWnd::findDlg(const QString &name)
  {
    for (int iter = 0; iter < m_dlgList.size(); ++iter)
      {
	MzLabInputOligomerTableViewDlg *dlg = m_dlgList.at(iter);
	
	if(dlg->name() == name)
	  return dlg;
      }
    
    return 0;
  }
  

  void
  MzLabWnd::inputListWidgetItemClicked(QListWidgetItem *item)
  {
    QString text = item->text();
    
    MzLabInputOligomerTableViewDlg *dlg = findDlg(text);
    Q_ASSERT(dlg);
    
    dlg->show();
  }


  void
  MzLabWnd::deleteInputListItem()
  {
    // Do this for input 1 list only.

    QListWidgetItem *item = m_ui.catalogue1ListWidget->currentItem();
    if (!item)
      return ;

    // Get the name of the dialog window represented by the item and
    // find the dialog window itself.

    QString text = item->text();
    
    // Destroy the dialog window and update the 2 lists contents.
    destroyDlg(text);

    // If the last dialog window is destroyed, then the lab is not
    // modified anymore.
    if (!m_dlgList.size())
      setWindowModified(false);
  }
  


  void
  MzLabWnd::destroyDlg(const QString &name)
  {
    MzLabInputOligomerTableViewDlg *dlg = findDlg(name);
    Q_ASSERT(dlg);

    // Remove the dialog window from the list of dialog windows and
    // delete it.
#ifdef DEBUG
    int result = m_dlgList.removeAll(dlg);
    Q_ASSERT(result);
#else
    m_dlgList.removeAll(dlg);
#endif
    delete(dlg);

    // Update the listWidgets(1 and 2) corresponding items.
    
    QList<QListWidgetItem *> list = 
      m_ui.catalogue1ListWidget->findItems(name,
					Qt::MatchExactly);
    Q_ASSERT(list.size() == 1);
    
    QListWidgetItem *item = list.first();
    
    m_ui.catalogue1ListWidget->takeItem(m_ui.catalogue1ListWidget->row(item));
    
    if (item)
      delete(item);

    list = m_ui.catalogue2ListWidget->findItems(name,
					     Qt::MatchExactly);
    Q_ASSERT(list.size() == 1);
    
    item = list.first();
    
    m_ui.catalogue2ListWidget->takeItem(m_ui.catalogue2ListWidget->row(item));
    
    if (item)
      delete(item);
  }
  
  
  bool 
  MzLabWnd::maybeSave()
  {
    // Returns true if we can continue(either saved ok or discard). If
    // save failed or cancel we return false to indicate to the caller
    // that something is wrong.

    if (isWindowModified())
      {
	QMessageBox::StandardButton ret;
	ret = QMessageBox::warning 
	 (this, tr("massXpert - mz Lab"),
	   tr("The mz Lab has been modified.\n"
	       "OK to exit?"),
	   QMessageBox::Yes | QMessageBox::No);
      
	if(ret == QMessageBox::Yes)
	  return true;
	else
	  return false;
      }
    
    return true;
  }


} // namespace massXpert
