/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QDebug>
#include <QMouseEvent>
#include <QHeaderView>
#include <QMessageBox>

/////////////////////// Local includes
#include "mzLabInputOligomerTableView.hpp"
#include "mzLabInputOligomerTableViewModel.hpp"

#include "oligomer.hpp"

#include "mzLabInputOligomerTableViewDlg.hpp"

#include "cleaveOligomerTableViewMimeData.hpp"
#include "fragmentOligomerTableViewMimeData.hpp"
#include "massSearchOligomerTableViewMimeData.hpp"

#include "application.hpp"

namespace massXpert
{

  MzLabInputOligomerTableView::MzLabInputOligomerTableView(QWidget *parent)
    : QTableView(parent)
  {

    setAlternatingRowColors(true);

    setSortingEnabled(true);

    setAcceptDrops(true);
    setDropIndicatorShown(true);

    QHeaderView *headerView = horizontalHeader();
    headerView->setClickable(true);
    headerView->setMovable(true);

    connect(this,
	     SIGNAL(activated(const QModelIndex &)),
	     this,
	     SLOT(itemActivated(const QModelIndex &)));  
  }


  MzLabInputOligomerTableView::~MzLabInputOligomerTableView()
  {

  }

  void 
  MzLabInputOligomerTableView::setOligomerList(OligomerList *oligomerList)
  {
    mp_oligomerList = oligomerList;
  }
  
  
  const OligomerList *
  MzLabInputOligomerTableView::oligomerList()
  {
    return mp_oligomerList;
  }
  

  void 
  MzLabInputOligomerTableView::setParentDlg(MzLabInputOligomerTableViewDlg *dlg)
  {
    Q_ASSERT(dlg);
    
    mp_parentDlg = dlg;
  }
  
  
  MzLabInputOligomerTableViewDlg *
  MzLabInputOligomerTableView::parentDlg()
  {
    return mp_parentDlg;
  }
  
  void 
  MzLabInputOligomerTableView::setMzLabWnd(MzLabWnd *wnd)
  {
    Q_ASSERT(wnd);
    
    mp_mzLabWnd = wnd;
  }
  

  MzLabWnd *
  MzLabInputOligomerTableView::mzLabWnd()
  {
    return mp_mzLabWnd;
  }
  

  void 
  MzLabInputOligomerTableView::setSourceModel
  (MzLabInputOligomerTableViewModel *model)
  {
    mp_sourceModel = model;
  }
  

  MzLabInputOligomerTableViewModel *
  MzLabInputOligomerTableView::sourceModel()
  {
    return mp_sourceModel;
  }
  

  void 
  MzLabInputOligomerTableView::mousePressEvent(QMouseEvent *mouseEvent)
  {
    if (mouseEvent->buttons() & Qt::RightButton)
      {
	//contextMenu->popup(mouseEvent->globalPos());
	return;
      }
  
    QTableView::mousePressEvent(mouseEvent);
  }

  
  void  
  MzLabInputOligomerTableView::dragEnterEvent(QDragEnterEvent *event)
  {
    const CleaveOligomerTableViewMimeData *tableViewData =
      qobject_cast<const CleaveOligomerTableViewMimeData *>(event->mimeData());
    
    if (tableViewData)
      {
	event->acceptProposedAction();
	event->accept();
	return ;
      }
    else if (event->mimeData()->hasFormat("text/plain")) 
      {
	event->acceptProposedAction();
	event->accept();
	return ;
      }
    
    event->ignore();
    
    return;
  }
  
  
  void  
  MzLabInputOligomerTableView::dragMoveEvent(QDragMoveEvent *event)
  {
    event->setDropAction(Qt::CopyAction);
    event->accept();
  }
  
  void  
  MzLabInputOligomerTableView::dropEvent(QDropEvent *event)
  {
    // Each time a new drop happens, we have to perform some checks. 

    // It is compulsory that the user has selected either mono or avg
    // as the mass type for the parent dialog window. Ask what's being
    // currently checked (mono and avg are two checkboxes).

    MassType previousMassType = mp_parentDlg->previousMassType();
    MassType massType = mp_parentDlg->massType();

    if(massType == MXT_MASS_NONE)
      {
        // Ask the user to clearly indicate what is the mass type
        // required).

        massType = mp_parentDlg->massTypeQuery();
      }
    
    // By now we have a current massType.

    // Check if one drop already occurred in the past and if it was
    // with the same mass type as now. Otherwise alert the user.

    if (previousMassType != MXT_MASS_NONE && previousMassType != massType)
      {
	int ret = 
	  QMessageBox::question(this, tr("massXpert: mzLab"),
				 tr("You are switching mass types." 
				     "Ok to continue ?"),
				 QMessageBox::Yes | QMessageBox::No,
				 QMessageBox::No);
	
	if(ret == QMessageBox::No)
	  {
	    event->ignore();
	    return;
	  }
      }
    
    OligomerList *oligomerList = new OligomerList("NOT_SET");
    
    // We may have data from different locations. We test them in
    // sequence. In the called functions, we get the data from the
    // dropped material into the temporary oligomerList allocated
    // right above.

    // A cleavage (CleaveOligomerTableViewMimeData).
    if(testCleaveOligomerDataDropped(event, oligomerList) == -1)
      {
        // A fragmentation (FragmentOligomerTableViewMimeData).
        if(testFragmentOligomerDataDropped(event, oligomerList) == -1)
          {
            // A mass search (MassSearchOligomerTableViewMimeData).
            if(testMassSearchOligomerDataDropped(event, oligomerList) == -1)
              {
                // An arbitrary text.
                testSimpleTextDataDropped(event, oligomerList);
              }
            
          }
      }

    // At this point we should have an oligomerList with oligomers in
    // it.

    if (!oligomerList->size())
      {
	event->ignore();

	delete oligomerList;
	
	return;
      }
    

    // We use a sort proxy model, but when adding oligomers to the
    // model we *have to* use the source model, that is not the proxy
    // model. Thus we have written a function in *this* TableView that
    // returns the source model.
    
    MzLabInputOligomerTableViewModel *tableViewModel = sourceModel();
	
    while(oligomerList->size())
      {
	// At this point we want to add the oligomers to the model
	// data. we simultaneously remove the oligomer from the
	// list as the ownership passes to the model.
	tableViewModel->addOligomer(oligomerList->takeFirst());
      }
	
    // We want the action to be a copy action and not a move
    // action because the data come from "static" data residing in
    // the tableView of the cleavage dialog window.
    event->setDropAction(Qt::CopyAction);
    event->accept();

    delete oligomerList;

    // Let the dialog know that we have set masses of type
    // massType. This way, we'll be able to check that later drops
    // are for the same mass type.
    mp_parentDlg->setPreviousMassType(massType);
    
    return;
  }
  

  int 
  MzLabInputOligomerTableView::testCleaveOligomerDataDropped
  (QDropEvent *event, OligomerList *oligomerList)
  {
    PolChemDef *polChemDef = mp_mzLabWnd->polChemDef();

    const CleaveOligomerTableViewMimeData *tableViewMimeData =
      qobject_cast<const CleaveOligomerTableViewMimeData *> 
      (event->mimeData());
  
    if(!tableViewMimeData)
      return -1;
    
    const CleaveOligomerTableView *tableView = 
      tableViewMimeData->tableView();
	
    // Perform the copying of the data... The list below will
    // receive a copy of each oligomer in the drag start tableView.
	
    int count = tableView->selectedOligomers(oligomerList, 
                                             oligomerList->size());
	
    if(!count)
      {
        return 0;
      }
        
    // We absolutely have to change the mp_polChemDef of the oligomer,
    // as it points to the polymer chemistry definition tied to the
    // sequence editor window's polymer, but we cannot make sure that
    // polymer chemistry definition will remain there. Thus we change
    // that polymer chemistry definition pointer to point to ours.

    for (int iter = 0; iter < oligomerList->size(); ++iter)
      {
        Oligomer *oligomer = oligomerList->at(iter);
		
        oligomer->setPolChemDef(polChemDef);
      }

    return count;
  }

  
  int 
  MzLabInputOligomerTableView::testFragmentOligomerDataDropped
  (QDropEvent *event, OligomerList *oligomerList)
  {
    PolChemDef *polChemDef = mp_mzLabWnd->polChemDef();

    const FragmentOligomerTableViewMimeData *tableViewMimeData =
      qobject_cast<const FragmentOligomerTableViewMimeData *> 
      (event->mimeData());
    
    if(!tableViewMimeData)
      return -1;
    
    // The data are from a fragmentation simulated in
    // massXpert, thus we can set the data format to fragment:
    mp_parentDlg->setFragment(true);
    
    const FragmentOligomerTableView *tableView  = 
      tableViewMimeData->tableView();
    
    // Perform the copying of the data... The list below will
    // receive a copy of each oligomer in the drag start tableView.
    
    int count = tableView->selectedOligomers(oligomerList, 
                                             oligomerList->size());
    
    if (!count)
      {
        return 0;
      }
    
    // We absolutely have to change the mp_polChemDef of the oligomer,
    // as it points to the polymer chemistry definition tied to the
    // sequence editor window's polymer, but we cannot make sure that
    // polymer chemistry definition will remain there. Thus we change
    // that polymer chemistry definition pointer to point to ours.
    
    for(int iter = 0; iter < oligomerList->size(); ++iter)
      {
        FragmentOligomer *oligomer = 
          static_cast<FragmentOligomer *>(oligomerList->at(iter));
	
        oligomer->setPolChemDef(polChemDef);
	
        // qDebug() << __FILE__ << __LINE__
        //          << "charge:" << oligomer->charge();
        // oligomer->ionizeRule().debugPutStdErr();
      }
    
    return count;
  }
  

  int 
  MzLabInputOligomerTableView::testMassSearchOligomerDataDropped
  (QDropEvent *event, OligomerList *oligomerList)
  {
    PolChemDef *polChemDef = mp_mzLabWnd->polChemDef();

    const MassSearchOligomerTableViewMimeData *tableViewMimeData =
      qobject_cast<const MassSearchOligomerTableViewMimeData *> 
      (event->mimeData());
	
    if (!tableViewMimeData)
      return -1;
    
    const MassSearchOligomerTableView *tableView  = 
      tableViewMimeData->tableView();
    
    // Perform the copying of the data... The list below will
    // receive a copy of each oligomer in the drag start tableView.
    
    int count = tableView->selectedOligomers(oligomerList, 
                                             oligomerList->size());
    
    if(!count)
      {
        return 0;
      }

    // We absolutely have to change the mp_polChemDef of the oligomer,
    // as it points to the polymer chemistry definition tied to the
    // sequence editor window's polymer, but we cannot make sure that
    // polymer chemistry definition will remain there. Thus we change
    // that polymer chemistry definition pointer to point to ours.
    
    for (int iter = 0; iter < oligomerList->size(); ++iter)
      {
        Oligomer *oligomer = oligomerList->at(iter);
	
        oligomer->setPolChemDef(polChemDef);
      }

    return count;
  }
  
  
  int 
  MzLabInputOligomerTableView::testSimpleTextDataDropped
  (QDropEvent *event, OligomerList *oligomerList)
  {
    PolChemDef *polChemDef = mp_mzLabWnd->polChemDef();
    const QList<Atom*> &atomList = polChemDef->atomList();

    IonizeRule ionizeRule = mp_mzLabWnd->ionizeRule();
    int totalCharge = ionizeRule.charge() * ionizeRule.level();
    
    bool isFragment = mp_parentDlg->isFragment();
    
    int count = 0;
    
    // Note that if the fragments checkbox is checked, then oligomers
    // that are created here on the basis of the data should be
    // FragmentOligomer instances specially, for their ionization
    // rule, to take into account the fact that a FragmentOligomer
    // instance of charge one has actually no ionizeRule formula, as
    // it was created ionized from the beginning.

    if(!event->mimeData()->hasText())
      {
        return -1;
      }
    
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();
		
    QString text = event->mimeData()->text();
		
    // The text that we get might have a single item
    // per line, in which case this is the mass 'm',
    // or two items per line separated with a blank,
    // then these are the mass 'm' and the charge 'z'.
		
    QStringList stringList = text.split("\n",
                                        QString::SkipEmptyParts,
                                        Qt::CaseSensitive);
	
    // qDebug() << stringList;

    // At this point we either have strings like
    // "195.12392 3"(charge is present) or
    // "195.12392"(charge is absent).

    // Remember that the data should be read using the
    // locale of the system.

    for (int iter = 0; iter < stringList.size(); ++iter)
      {
        QString text = stringList.at(iter);
	    
        QStringList stringList = text.split(QRegExp("\\s+"),
                                            QString::SkipEmptyParts); 
	    
        if(stringList.size() == 1)
          {
            // Only a mass, and no charge, apparently.
            bool ok = false;
		
            double mass = locale.toDouble(stringList.first(), &ok);
		
            if (!mass && !ok)
              {
                QMessageBox::warning(this,
                                     tr("massXpert - Drag and drop"),
                                     tr("%1@%2\n"
                                        "Failed to convert %3 to double")
                                     .arg(__FILE__)
                                     .arg(__LINE__)
                                     .arg(stringList.first()));
		    
                event->ignore();
			    
                return -1;
              }
		
            Oligomer *oligomer = 0;
			    
            if(!isFragment)
              {
                // Note that if the ionizeRule has a total charge
                //(charge() * level()), then we have to set this
                // oligomer to charged state(totalCharge ? true : false).
                oligomer = 
                  new Oligomer(polChemDef, "NOT_SET",
                               Ponderable(mass, mass),
                               ionizeRule,
                               CalcOptions(),
                               (totalCharge ? true : false),
                               -1, -1);
              }
            else
              {
                // We are dealing with an oligomer
                // obtained via fragmentation, its
                // treatement is thus somewhat more
                // subtle.
				
                if(totalCharge == 1)
                  {
                    // The fragment oligomer is
                    // charged only once, which means
                    // we are creating an oligomer
                    // with no ionization rule, as, by
                    // defintion a fragmentation
                    // oligomer is created charged one
                    // with no proactive ionization.

                    IonizeRule tempIonizeRule;
				    
                    oligomer = 
                      new Oligomer(polChemDef, "NOT_SET",
                                   Ponderable(mass, mass),
                                   tempIonizeRule,
                                   CalcOptions(),
                                   false, 
                                   -1, -1);
                  }
                else
                  {
                    // The oligomer is ionized more
                    // than once. The ionization rule
                    // that we'll use is ionizeRule.
				    
                    IonizeRule tempIonizeRule;

                    tempIonizeRule.setFormula(ionizeRule.formula());

                    int incrementCharge = totalCharge - 1;
                    int requiredLevel = incrementCharge / ionizeRule.charge();

                    tempIonizeRule.setCharge(ionizeRule.charge());
                    tempIonizeRule.setLevel(requiredLevel);

                    if (!tempIonizeRule.validate(atomList))
                      {
                        QMessageBox::warning(this,
                                             tr("massXpert - Drag and drop"),
                                             tr("%1@%2\n"
                                                "Failed to validate the "
                                                "ionization rule")
                                             .arg(__FILE__)
                                             .arg(__LINE__));
					
                        event->ignore();
					
                        return -1;
                      }
				    
                    // Create an oligomer that IS
                    // ionized.
                    oligomer = 
                      new Oligomer(polChemDef, "NOT_SET",
                                   Ponderable(mass, mass),
                                   tempIonizeRule,
                                   CalcOptions(),
                                   true, 
                                   -1, -1);
                  }
              }
			    

            // qDebug() << __FILE__ << __LINE__
            //          << "charge:" << oligomer->charge();
	    // oligomer->ionizeRule().debugPutStdErr();
			
            oligomerList->append(oligomer);

            ++count;
          }
        else if (stringList.size() == 2)
          {
            // One mass and one charge, apparently.
            bool ok = false;
		
            double mass = locale.toDouble(stringList.first(), &ok);
		
            if (!mass && !ok)
              {
                QMessageBox::warning(this,
                                     tr("massXpert - Drag and drop"),
                                     tr("%1@%2\n"
                                        "Failed to convert %3 to double")
                                     .arg(__FILE__)
                                     .arg(__LINE__)
                                     .arg(stringList.first()));
		    
                event->ignore();

                return -1;
              }

            // Work with the charge
            int charge = locale.toInt(stringList.at(1), &ok);
		
            if (!charge && !ok)
              {
                QMessageBox::warning(this,
                                     tr("massXpert - Drag and drop"),
                                     tr("%1@%2\n"
                                        "Failed to convert %3 to int")
                                     .arg(__FILE__)
                                     .arg(__LINE__)
                                     .arg(stringList.at(1)));
		    
                event->ignore();
		   
                return -1;
              }
		
            if(!isFragment)
              {
                ionizeRule.setLevel(charge);

                // Note that if the ionizeRule has a total charge
                //(charge() * level()), then we have to set this
                // oligomer to charged state(totalCharge ? true : false).
                Oligomer *oligomer = 
                  new Oligomer(polChemDef, "NOT_SET",
                               Ponderable(mass, mass),
                               ionizeRule,
                               CalcOptions(),
                               (totalCharge ? true : false),
                               -1, -1);

                // qDebug() << __FILE__ << __LINE__
                //          << "charge:" << oligomer->charge();
                // oligomer->ionizeRule().debugPutStdErr();

                oligomerList->append(oligomer);

                ++count;
              }
            else // if (isFragment)
              {
                // We are dealing with an oligomer
                // obtained via fragmentation, its
                // treatement is thus somewhat more
                // subtle.
				
                if(charge == 1)
                  {
                    // The fragment oligomer is
                    // charged only once, which means
                    // we are creating an oligomer
                    // with no ionization rule, as, by
                    // defintion a fragmentation
                    // oligomer is created charged one
                    // with no proactive ionization.

                    IonizeRule tempIonizeRule;
				    
                    FragmentOligomer *oligomer = 
                      new FragmentOligomer(polChemDef, "NOT_SET",
                                           Ponderable(mass, mass),
                                           tempIonizeRule,
                                           false, -1, -1);
				    
                    // qDebug() << __FILE__ << __LINE__
                    //          << "charge:" 
                    //          << static_cast<FragmentOligomer *>(oligomer)->
                    //   charge();
                    // oligomer->ionizeRule().debugPutStdErr();
				    
                    oligomerList->append(oligomer);

                    ++count;
                  }
                else if (charge > 1)
                  {
                    // The oligomer is ionized more
                    // than once. The ionization rule
                    // that we'll use is ionizeRule.
				    
                    IonizeRule tempIonizeRule;

                    tempIonizeRule.setFormula(ionizeRule.formula());

                    int incrementCharge = charge - 1;
                    int requiredLevel = incrementCharge / ionizeRule.charge();

                    tempIonizeRule.setCharge(ionizeRule.charge());
                    tempIonizeRule.setLevel(requiredLevel);

                    if (!tempIonizeRule.validate(atomList))
                      {
                        QMessageBox::warning(this,
                                             tr("massXpert - Drag and drop"),
                                             tr("%1@%2\n"
                                                "Failed to validate the "
                                                "ionization rule")
                                             .arg(__FILE__)
                                             .arg(__LINE__));
					
                        event->ignore();
					
                        return -1;
                      }
				    
                    // Create an oligomer that IS
                    // ionized.
                    FragmentOligomer *oligomer = 
                      new FragmentOligomer(polChemDef, "NOT_SET",
                                           Ponderable(mass, mass),
                                           tempIonizeRule,
                                           true, -1, -1, CalcOptions());
				    
                    // qDebug() << __FILE__ << __LINE__
                    //          << "charge:" 
                    //          << static_cast<FragmentOligomer *>(oligomer)->
                    //   charge();
                    // oligomer->ionizeRule().debugPutStdErr();
				    
                    oligomerList->append(oligomer);

                    ++count;
                  }
                else if (!charge)
                  {
                    QMessageBox::warning(this,
                                         tr("massXpert - Drag and drop"),
                                         tr("%1@%2\n"
                                            "Charge of a fragmentation oligomer "
                                            "cannot be 0")
                                         .arg(__FILE__)
                                         .arg(__LINE__));
				    
                    event->ignore();
				    
                    return -1;
                  }
              }

          }
        // End of else if (stringList.size() == 2)
        else
          {
            QMessageBox::warning(this,
                                 tr("massXpert - Drag and drop"),
                                 tr("%1@%2\n"
                                    "Failed to read line %3")
                                 .arg(__FILE__)
                                 .arg(__LINE__)
                                 .arg(stringList.first()));
		
            event->ignore();
		
            return -1;
          }
      }

    return count;
  }

  
  QString * 
  MzLabInputOligomerTableView::selectedDataAsPlainText()
  {
    // Provide a most simple list of (m/z,z) pairs like 1021.21 2 for
    // all the selected items in the table view.

    QString *resultString = new QString();
    
    // We first have to get the selection model for the proxy model.

    QItemSelectionModel *selModel = selectionModel();

    MzLabInputOligomerTableViewModel* modelSource = sourceModel();
    
    // Now get the selection ranges.
    
    QItemSelection proxyItemSelection = selModel->selection();

    QSortFilterProxyModel *sortModel = 
      static_cast<QSortFilterProxyModel *>(model());
    
    QItemSelection sourceItemSelection = 
      sortModel->mapSelectionToSource(proxyItemSelection);

    QModelIndexList modelIndexList = sourceItemSelection.indexes();

    int modelIndexListSize = modelIndexList.size();
    
    // Attention, if we select one single row, our modelIndexList will
    // be of size 2, because in one single row there are two cells:
    // each cell for each column, and there are 2 columns. Thus, when
    // we iterate in the modelIndexList, we'll have to take care of
    // this and make sure we are not putting each selected row's data
    // two times. For this, we make sure we are not handling the same
    // row twice or more, by storing the processed rows in a list of
    // integers and by checking for existence of that row each time a
    // new index is processed.
    
    QList<int> processedRowList;
    
    for (int iter = 0; iter < modelIndexListSize; ++iter)
      {
	QModelIndex index = modelIndexList.at(iter);
	
	Q_ASSERT(index.isValid());

        // Get to know what's the row of the index.

        int row = index.row();
        
        if(processedRowList.contains(row))
          continue;
        else
          processedRowList.append(row);
        
        // Get the m/z and the z values through the use of the model's
        // data function, which expects an index, which we create
        // using the model's index() function below.

        // First the m/z.
        QModelIndex itemModelIndex = 
          modelSource->index(row, MZ_LAB_INPUT_OLIGO_MASS_COLUMN);
        
        Q_ASSERT(itemModelIndex.isValid());
        
        QString massString =
          modelSource->data(itemModelIndex, Qt::DisplayRole).toString();
        
        // Second the charge.
        itemModelIndex = 
          modelSource->index(row, MZ_LAB_INPUT_OLIGO_CHARGE_COLUMN);
        
        Q_ASSERT(itemModelIndex.isValid());
        
        QString chargeString =
          modelSource->data(itemModelIndex, Qt::DisplayRole).toString();
        
        
        resultString->append(QString("%1 %2\n")
                             .arg(massString)
                             .arg(chargeString));
      }
    
    return resultString;
  }
  

  void 
  MzLabInputOligomerTableView::itemActivated(const QModelIndex &index)
  {
    if (!index.isValid())
      return;

    MzLabInputOligomerTableViewSortProxyModel *sortModel = 
      static_cast<MzLabInputOligomerTableViewSortProxyModel *>(model());
    
    QModelIndex sourceIndex = sortModel->mapToSource(index);

    int row = sourceIndex.row();

    Oligomer *oligomer = mp_oligomerList->at(row);

    // qDebug() << __FILE__ << __LINE__
    //          << oligomer->name();
     
     // From the oligomer, check that its mp_polymer member is still
     // non-0.
    
    const Polymer *polymer = oligomer->polymer();
    
    if (!polymer)
      {
	// qDebug() << __FILE__ << __LINE__;
	
	return;
      }
    
    // Get the sequence editor window pointer out of the oligomer, if
    // it has one in the form of a NoDeletePointerProp.

    Prop *prop = const_cast<Oligomer *>(oligomer)->prop("SEQUENCE_EDITOR_WND");
    
    if (!prop)
      {
        // qDebug() << __FILE__ << __LINE__;
	
	return;
      }
    
    SequenceEditorWnd *editorWnd = 
      static_cast<SequenceEditorWnd *>(prop->data());

    if (!editorWnd)
      return;
    
    CoordinateList *coordinateList = 
      static_cast<CoordinateList *>(const_cast<Oligomer *>(oligomer));
    
    // Remove the previous selection, so that we can start fresh.
    editorWnd->mpa_editorGraphicsView->resetSelection();
  
    for (int iter = 0; iter < coordinateList->size(); ++iter)
      {
	Coordinates *coordinates = coordinateList->at(iter);
	
	int start = coordinates->start();
	int end = coordinates->end();
	
        // qDebug() << __FILE__ << __LINE__
        //          << "oligomer:" << oligomer
        //          << "polymer:" << polymer
        //          << "start:" << start << "end:" << end;
        
	if(start >= polymer->size() ||
	    end >= polymer->size())
	  {
	    QMessageBox::warning(this,
				  tr("massXpert - Cleavage"),
				  tr("%1@%2\n"
				      "The monomer indices do not correspond "
				      "to a valid polymer sequence range.\n"
				      "Avoid modifying the sequence while "
				      "working with oligomers.")
				  .arg(__FILE__)
				  .arg(__LINE__),
				  QMessageBox::Ok);
	    
	    return;
	  }
	
	editorWnd->mpa_editorGraphicsView->setSelection(*coordinates, 
							 true, false);
      }
    
    editorWnd->updateSelectedSequenceMasses();  
  }

} // namespace massXpert
