/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QDebug>


/////////////////////// Local includes
#include "coordinates.hpp"


namespace massXpert
{

  ///////////////////////// Coordinates /////////////////////////
  ///////////////////////// Coordinates /////////////////////////
  ///////////////////////// Coordinates /////////////////////////
  ///////////////////////// Coordinates /////////////////////////
  ///////////////////////// Coordinates /////////////////////////



  Coordinates::Coordinates(int start, int end)
  {
    if (start > end)
      {
	m_start = end;
	m_end = start;
      }
    else
      {
	m_start = start;
	m_end = end;
      }
  }


  Coordinates::Coordinates(const Coordinates &other)
    : m_start(other.m_start), m_end(other.m_end)
  {
    int temp;
    
    if (m_start > m_end)
      {
	temp = m_end;
	m_end = m_start;
	m_start = temp;
      }
  }


  Coordinates::~Coordinates()
  {
  }


  void
  Coordinates::setStart(int value)
  {
    if (value > m_end)
      {
	m_start = m_end;
	m_end = value;
      }
    else
      {
	m_start = value;
      }
  }
  

  int
  Coordinates::start() const
  {
    return m_start;
  }


  void 
  Coordinates::incrementEnd()
  {
    ++m_end;
  }


  void 
  Coordinates::setEnd(int value)
  {
    m_end = value;
  }


  int
  Coordinates::end() const
  {
    return m_end;
  }


  int
  Coordinates::length() const
  {
    return(m_end - m_start + 1);
  }


  QString
  Coordinates::indicesAsText() const
  {
    QString text = QString("[%1--%2]")
      .arg(m_start)
      .arg(m_end);
  
    return text;
  }


 QString
  Coordinates::positionsAsText() const
  {
    QString text = QString("[%1--%2]")
      .arg(m_start + 1)
      .arg(m_end + 1);
  
    return text;
  }


  void
  Coordinates::reset()
  {
    m_start = 0;
    m_end = 0;
  }



  ///////////////////////// CoordinateList /////////////////////////
  ///////////////////////// CoordinateList /////////////////////////
  ///////////////////////// CoordinateList /////////////////////////
  ///////////////////////// CoordinateList /////////////////////////
  ///////////////////////// CoordinateList /////////////////////////





  CoordinateList::CoordinateList(QString comment, 
				  QList<Coordinates *> *list)
    : m_comment(comment)
  {
    if (list)
      {
	for(int iter = 0 ; iter < list->size(); ++iter)
	  {
	    Coordinates *iterCoordinates = list->at(iter);
	  
	    Coordinates *coordinates = new Coordinates(*iterCoordinates);
	  
	    append(coordinates);
	  }
      }
  }


  CoordinateList::CoordinateList(const CoordinateList &other)
    : m_comment(other.m_comment)
  {
    for (int iter = 0 ; iter < other.size(); ++iter)
      {
	Coordinates *iterCoordinates = other.at(iter);
      
	Coordinates *coordinates = new Coordinates(*iterCoordinates);

	append(coordinates);
      }
  }


  CoordinateList::~CoordinateList()
  {
    // The members of the list were allocated with new()...  
    qDeleteAll(begin(), end());
    clear();
  }


  void 
  CoordinateList::setCoordinates(const Coordinates &coordinates)
  {
    empty();
    
    Coordinates *newCoordinates = new Coordinates(coordinates);
    append(newCoordinates);
  }
    

  void 
  CoordinateList::setCoordinates(const CoordinateList &list)
  {
    empty();
    
    for (int iter = 0; iter < list.size(); ++iter)
      {
	Coordinates *coordinates = new Coordinates(*list.at(iter));
	append(coordinates);
      }
  }
    

  void 
  CoordinateList::appendCoordinates(const Coordinates &coordinates)
  {
    Coordinates *newCoordinates = new Coordinates(coordinates);
    
    append(newCoordinates);
  }
    

  void 
  CoordinateList::setComment(QString text)
  {
    m_comment = text;
  }


  QString 
  CoordinateList::comment() const
  {
    return m_comment;
  }


  int
  CoordinateList::leftMostCoordinates(QList<int> &indexList) const
  {
    if (isEmpty())
      return 0;
    
    while(!indexList.isEmpty())
      indexList.removeFirst();
    
    int leftMostValue = first()->start();
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	int start = coordinates->start();
	
	if(start < leftMostValue)
	  leftMostValue = start;
      }
    
    // At this point we now what's the leftmost index. We can use that
    // index to now search for all the items that are also leftmost.
    
    for (int iter = 0; iter < size(); ++iter)
      {
	if(at(iter)->start() == leftMostValue)
	  indexList.append(iter);
      }
    
    return indexList.size();
  }
  

  bool
  CoordinateList::isLeftMostCoordinates(Coordinates *coordinates) const
  {
    Q_ASSERT(coordinates);
    
    // Are the coordinates the leftmost coordinates of *this
    // CoordinateList ?
    
    int value = coordinates->start();
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	if(value > coordinates->start())
	  return false;
      }
    
    return true;
  }
  

  int
  CoordinateList::rightMostCoordinates(QList<int> &indexList) const
  {
    if (isEmpty())
      return 0;
    
    while(!indexList.isEmpty())
      indexList.removeFirst();
    
    int rightMostValue = first()->end();
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	int end = coordinates->end();
	
	if(end > rightMostValue)
	  rightMostValue = end;
      }
    
    // At this point we now what's the rightmost index. We can use
    // that index to now search for all the items that are also
    // rightmost.
    
    for (int iter = 0; iter < size(); ++iter)
      {
	if(at(iter)->end() == rightMostValue)
	  indexList.append(iter);
      }
    
    return indexList.size();
  }
  

  bool
  CoordinateList::isRightMostCoordinates(Coordinates *coordinates) const
  {
    Q_ASSERT(coordinates);
    
    // Are the coordinates the rightmost coordinates of *this
    // CoordinateList ?
    
    int value = coordinates->end();
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	if(value < coordinates->end())
	  return false;
      }
    
    return true;
  }
  

  bool 
  CoordinateList::encompassIndex(int index) const
  {
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	if(index >= coordinates->start() && 
	    index <= coordinates->end())
	  return true;
      }
    
    return false;
  }
 

  QString 
  CoordinateList::indicesAsText() const
  {
    QString text;
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	text += QString("[%1-%2]")
	  .arg(coordinates->start())
	  .arg(coordinates->end());
      }
    
    return text;
  }
  
  
  QString 
  CoordinateList::positionsAsText() const
  {
    QString text;
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	text += QString("[%1-%2]")
	  .arg(coordinates->start() + 1)
	  .arg(coordinates->end() + 1);
      }
    
    return text;
  }

  
  void 
  CoordinateList::empty()
  {
    while(!isEmpty())
      delete(takeFirst());
  }
  
  
  void 
  CoordinateList::debugPutStdErr()
  {
    qDebug() << __FILE__ << __LINE__
	      << "CoordinateList:";
    
    QString text;
    
    for (int iter = 0; iter < size(); ++iter)
      {
	Coordinates *coordinates = at(iter);
	
	text += coordinates->indicesAsText();
      }
    
    qDebug() << __FILE__ << __LINE__
	      << text;
  }
  
} // namespace massXpert
