/* This file is part of the KDE libraries
   Copyright (C) 2006 KovoKs <info@kovoks.nl>

   This file is bases on a file
        Copyright (C) 2005 Waldo Bastian <bastian@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "tooltip.h"
#include "addresslabel.h"

#include <qstylesheet.h>
#include <qsimplerichtext.h>

#include <kcursor.h>
#include <kglobalsettings.h>
#include <kdebug.h>

namespace Mailody
{

static QString qrichtextify( const QString& text )
{
  if ( text.isEmpty() || text[0] == '<' )
    return text;

  QStringList lines = QStringList::split('\n', text);
  for(QStringList::Iterator it = lines.begin(); it != lines.end(); ++it)
  {
    *it = QStyleSheet::convertFromPlainText( *it,
                                              QStyleSheetItem::WhiteSpaceNormal );
  }

  return lines.join(QString::null);
}

AddressLabel::AddressLabel( QWidget *parent, const char *name )
 : QLabel ( parent, name )
{
    setAlignment( Qt::WordBreak );
    setMouseTracking( true );
    m_tt = new ToolTip(this);
    connect(m_tt,SIGNAL(leftClicked(const QString&)),
            SIGNAL(leftClicked(const QString&)));
    connect(m_tt,SIGNAL(rightClicked(const QString&)),
            SIGNAL(rightClicked(const QString&)));

}

void AddressLabel::setDefaultWidth(int defaultWidth)
{
  m_defaultWidth = defaultWidth;
  updateGeometry();
}

QSizePolicy AddressLabel::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Minimum, false);
}

QSize AddressLabel::minimumSizeHint() const
{
  QString qt_text = qrichtextify( text() );
  int pref_width = 0;
  int pref_height = 0;
  QSimpleRichText rt(qt_text, font());
  pref_width = m_defaultWidth;
  rt.setWidth(pref_width);
  int used_width = rt.widthUsed();
  if (used_width <= pref_width)
  {
    while(true)
    {
      int new_width = (used_width * 9) / 10;
      rt.setWidth(new_width);
      int new_height = rt.height();
      if (new_height > pref_height)
        break;
      used_width = rt.widthUsed();
      if (used_width > new_width)
        break;
    }
    pref_width = used_width;
  }
  else
  {
    if (used_width > (pref_width *2))
      pref_width = pref_width *2;
    else
      pref_width = used_width;
  }

  return QSize(pref_width, rt.height());
}

QSize AddressLabel::sizeHint() const
{
  return minimumSizeHint();
}

void AddressLabel::mouseReleaseEvent ( QMouseEvent * e )
{
    QSimpleRichText rt(text(), font());
    rt.setWidth(width());
    QString link = rt.anchorAt( e->pos() );
    if (link)
        leftClicked(link);
    else
        m_tt->setText(QCursor::pos(), text());
}

void AddressLabel::contextMenuEvent ( QContextMenuEvent * e )
{
    QSimpleRichText rt(text(), font());
    rt.setWidth(width());
    QString link = rt.anchorAt( e->pos() );
    if (link)
        rightClicked(link);
}

void AddressLabel::mouseMoveEvent( QMouseEvent * e )
{
    QSimpleRichText rt(text(), font());
    rt.setWidth(width());
    if ( rt.anchorAt( e->pos() ) )
        setCursor(KCursor::handCursor());
    else
        setCursor(KCursor::arrowCursor());
}

void AddressLabel::setText( const QString &text )
{
  QLabel::setText(text);
  adjustSize();
}

void AddressLabel::virtual_hook( int, void* )
{ /*BASE::virtual_hook( id, data );*/ }

}
#include "addresslabel.moc"
