/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file ArrowPlotting.h
    \brief Definition of the Template class ArrowPlotting.
    
    Magics Team - ECMWF 2005
    
    Started: Thu 17-Mar-2005
    
    Changes:
    
*/

#ifndef ArrowPlotting_H
#define ArrowPlotting_H

#include "magics.h"

#include "WindPlotting.h"
#include "ArrowPlottingAttributes.h"
#include "Arrow.h"

namespace magics {

template <class P>
class ArrowPlotting: public WindPlotting<P>, public ArrowPlottingAttributes{

public:
	ArrowPlotting(): southArrows_(0), northArrows_(0) {}
	virtual ~ArrowPlotting() {}

	virtual void set(const map<string, string>& map) 
		{ ArrowPlottingAttributes::set(map); }

	virtual void set(const XmlNode& node) 
		{ ArrowPlottingAttributes::set(node); }
		
	bool accept(const string& node) { return ArrowPlottingAttributes::accept(node);; }

	virtual WindPlotting<P>* clone() {
		ArrowPlotting<P>* object = new ArrowPlotting<P>();
		object->copy(*this);
		return object;
	}

	virtual void operator()(const PaperPoint& point, double x, double y) 
	{ 
		double speed = sqrt((x*x)+(y*y));
		if (speed < this->min_speed_ || speed > this->max_speed_) return;

		if ((*this->calm_)(point, x, y)) return; 

		if ( point.y() >= 0 ) 
			northArrows_->push_back(ArrowPoint(x, y, point));
		else southArrows_->push_back(ArrowPoint(x, y, point));

		// When WIND_ARROW_UNIT_VELOCITY is set to 0.0, MAGICS will calculate the maximum 
		// wind speed for the geographical area requested and will use this maximum for the unit velocity. 
		// This means that the unit velocity will vary depending on the area being plotted. 

		if ( this->unit_velocity_ == 0 )
		{
			if ( speed > this->maxVelocity_ ) this->maxVelocity_ = speed;
			northArrows_->setScale(this->maxVelocity_);
			southArrows_->setScale(this->maxVelocity_);
		}
	}

	virtual void prepare(BasicGraphicsObjectContainer& task)
	{ 
		// Calm Indicator 
		(*this->calm_).colour(*this->colour_);
		(*this->calm_).height(this->calm_indicator_size_);
		(*this->calm_).below(this->calm_below_);
		(*this->calm_).prepare(task);

		// Setting for north Hemisphere wind	
		northArrows_ = new Arrow();
		northArrows_->setColour(*this->colour_);
		northArrows_->setThickness(this->thickness_);
		northArrows_->setStyle(this->style_);
		northArrows_->setScale(this->unit_velocity_);
		northArrows_->setHemisphere(NORTH);
		northArrows_->setArrowPosition(this->origin_position_);
		northArrows_->setHeadIndex(this->head_);
		northArrows_->setHeadRatio(this->ratio_);

		 // Setting for south Hemisphere wind	
		southArrows_ = new Arrow();
		southArrows_->setColour(*this->colour_);
		southArrows_->setThickness(this->thickness_);
		southArrows_->setStyle(this->style_);
		southArrows_->setScale(this->unit_velocity_);
		southArrows_->setHemisphere(SOUTH);
		southArrows_->setArrowPosition(this->origin_position_);
		southArrows_->setHeadIndex(this->head_);
		southArrows_->setHeadRatio(this->ratio_);
		
		this->maxVelocity_ = int_MIN;
		
		task.push_back(southArrows_);
		task.push_back(northArrows_);		
	}

	void visit(LegendVisitor& legend)
	{
		if ( !this->legend_ ) return;
		double speed = (this->unit_velocity_) ? this->unit_velocity_ : this->maxVelocity_;
		Arrow* arrow = new Arrow();
		(*arrow).setColour(*this->colour_);
		arrow->setThickness(this->thickness_);
		arrow->setScale(speed);
		arrow->setArrowPosition(this->origin_position_);
		arrow->setHeadIndex(this->head_);
		arrow->setHeadRatio(this->ratio_);

		ostringstream text;
		text <<  speed << " m/s";
		LegendEntry * entry = new ArrowEntry(text.str(), arrow);

		legend.add(entry);
	}

protected:
	//! Method to print string about this class on to a stream of type ostream (virtual).
	virtual void print(ostream& out) const
	 	{ out << "ArrowPlotting<P>"; } 
	Arrow* southArrows_;
	Arrow* northArrows_;
	double  maxVelocity_;

private:
	//! Copy constructor - No copy allowed
	ArrowPlotting(const ArrowPlotting&);
	//! Overloaded << operator to copy - No copy allowed
	ArrowPlotting& operator=(const ArrowPlotting&);
    
// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const ArrowPlotting<P>& p)
		{ p.print(s); return s; }
};

} // namespace magics


#endif
