/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \mainpage Magics++

 \section intro What is Magics++?

 Magics++ is the second generation of a meteorological graphics 
 library developed at the <a href="http://www.ecmwf.int"><i>European 
 Centre for Medium Range Forecasts (ECMWF)</i></a>. This library is 
 developed in C++ and offers output in various formats such as PostScript, 
 PDF, PNG, SVG, KML and OpenGL. 

 \section install How-to install

 Before installation you have to compile Magics++. To do so, simply 
 unpack the tarball in an appropiate directory and run <i>./configure</i> 
 followed by <i>make</i>. You can type <i>./configure --help</i> to get all
 options of configuring Magics++. 
 
 To install type <i>make install</i>. Depending on the 
 choosen installation directory you need root permission.

 For more information you should read the <a href="Magics++InstallationGuide.pdf">installation guide</a>.

 \section interfaces Magics++ interfaces

 Magics++ offers different interfaces to access its graphics 
 functionality: C++ (for higher-level applicatiosn such as Metview), C, Fortran 77
 Python and MagML. The Fortran interface is intended to be backwards compatible with older 
 versions (mainly the the 6.x series) of Magics.

 \section modules More information

 - \ref interpolation "Interpolation and contouring"
 - \ref hilo "Computations of High &amp; Lows"
 - \ref titles "Automatic titles"
 - \ref projections "Geographical projections"
 - \ref drivers "Output drivers"
 - \ref coastlines "Coastlines"
 - \ref colours "Colours"
 
 
 \section links Links
 
 - <a href="http://www.ecmwf.int/publications/manuals/magics/magplus/index.html">Magics++ homepage</a>
 - <a href="http://www.ecmwf.int/services/computing/training/material/com_mag.html">Training course</a>
 - <a href="file:///scratch/graphics/cgi/mpptestsuite/html/test/fortran/timings.html">Magics++ test suite [ECMWF internal]</a>
 
 \section copyright License
 
  Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

  Licensed under the Apache License, Version 2.0 (the "License"); 
  you may not use this file except in compliance with the License. 
  You may obtain a copy of the License at 

	http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software 
  distributed under the License is distributed on an "AS IS" BASIS, 
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
  See the License for the specific language governing permissions and 
  limitations under the License.

*/

/*!
  \defgroup coastlines Coastlines
  
  \section overviewCoast Overview
  
  Coastlines are stored in NetCDF files and are stored in the <i>share/</i> directory.
  They are generated from GHSSH data files stored in the <i>gshhs/</i> directory.
  
  \section makeCoast Regeneration of the Coastlines
  
  Go into the main Magics++ directory and run <i>make clean-coast</i> followed 
  by <i>make coast</i>.
*/

/*! \file magics.h

  \brief This file contains all global definition for the MagPlus library.
  \author Graphics Section, ECMWF
  \license Apache License 2.0

  Started: January 2004

  Changes:
  
*/
#ifndef magicsplusplus_H
#define magicsplusplus_H

#include <magics_config.h>
#include <unistd.h>           // for AIX
#include <climits>

// do the export restriction only if NOT for Metview
#ifndef MAGICS_METVIEW

#ifdef WIN32
  #define MAGICS_IMPORT __declspec(dllimport)
  #define MAGICS_EXPORT __declspec(dllexport)
  #define MAGICS_DLLLOCAL
  #define MAGICS_DLLPUBLIC
#endif

#if (__GNUC__ - 0 > 3) || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 2)
  #define MAGICS_GCC 4
  #define MAGICS_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define MAGICS_EXPORT    __attribute__ ((visibility("default")))
  #define MAGICS_IMPORT    __attribute__ ((visibility("default")))
  #define MAGICS_DLLLOCAL  __attribute__ ((visibility("hidden")))
  #define MAGICS_DLLPUBLIC __attribute__ ((visibility("default")))
#else
  #define MAGICS_GCC 3
  #define MAGICS_NO_EXPORT
  #define MAGICS_IMPORT
  #define MAGICS_EXPORT
  #define MAGICS_DLLLOCAL
  #define MAGICS_DLLPUBLIC
#endif

#else
  #define MAGICS_GCC 3
  #define MAGICS_NO_EXPORT
  #define MAGICS_IMPORT
  #define MAGICS_EXPORT
  #define MAGICS_DLLLOCAL
  #define MAGICS_DLLPUBLIC
#endif

//#if __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 >= 2)
//#define MAGICS_DEPRECATED __attribute__ ((deprecated))
//#else
//#define MAGICS_DEPRECATED
//#endif

#include <iostream>
#include <cstring>
#include <string>
#include <sstream>
#include <fstream>

using std::string;
using std::ostream;
using std::endl;
using std::ends;
using std::cout;
using std::cerr;
using std::ostringstream;
using std::ofstream;
using std::ifstream;

#include <list>
#include <vector>
#include <stack>
#include <memory>
#include <map>
#include <set>
#include <iterator>
#include <numeric>
#include <cfloat>
#include <algorithm>

using std::allocator;
using std::vector;
using std::pair;
using std::list;
using std::map;
using std::set;
using std::auto_ptr;

#include <exception>
using std::exception;

#include <cassert>
#include <cmath>

#if defined(MAGICS_AIX_XLC) || defined(MAGICS_WINDOWS_CYGWIN)
inline MAGICS_NO_EXPORT double exp10(const double r) { return exp(r*2.302585093);} // exp(r*ln(10))
#endif

/*! \namespace magics

 The <I>magics</I> namespace ancapsulate all elements of Magics++.
 The namespace prevents confusions with other libraries or 
 older versions of Magics.
 
 You can use this namespace by simple prefix all Magics elements
 by <I>magics::</I> or by typing <I>using namespace magics;</I>
 before any calls in your program.

*/
namespace magics {



const int int_MAX = INT_MAX;
const int int_MIN = INT_MIN;


const double double_MIN = DBL_MIN;

template <class P>
class magvector : public std::vector<P>
{
public:
    magvector() {}
    magvector(const std::vector<P>& list) :  std::vector<P>(list) {}
    virtual ~magvector() {} 
    virtual MAGICS_NO_EXPORT void print(ostream& out) const
    { 
        if ( this->size() < 10 ) {
            out << "Vector[";
            string sep = "";
	    const unsigned int si = this->size();
            for (unsigned int i = 0; i < si; i++) {
                out << sep << (*this)[i];
                sep = ", ";
            }
            out << "]";
        }
        else {
            unsigned int nb = this->size();
            out << "Vector[" << (*this)[0] << ", " << (*this)[1] << ", " << (*this)[2];
            out << "...." << (*this)[nb-3] << ", " << (*this)[nb-2]  << ", " << (*this)[nb-1];
            out << "(" << nb << " elements)]";
        }
    }
// -- Friends
	friend MAGICS_NO_EXPORT ostream& operator<<(ostream& s,const magvector<P>& p)
		{ p.print(s); return s; }
};


typedef magvector<string>	stringarray;
typedef magvector<string>	stringarray;
typedef magvector<int>		intarray;
typedef magvector<int>		intarray;
typedef magvector<double>	doublearray;
typedef magvector<double>	floatarray;

enum LineStyle { M_SOLID , M_DASH , M_DOT , M_CHAIN_DASH , M_CHAIN_DOT };
enum Hemisphere { NORTH , SOUTH };
enum Justification { MLEFT, MCENTRE, MRIGHT };
enum VerticalAlign { MNORMAL, MTOP, MCAP, MHALF, MBASE, MBOTTOM };
//enum VerticalAlign { MBASE, MTOP, MHALF, MBOTTOM };   // if change you need to change also share/magics/PostScriptMacros2.dat
enum Shading { M_SH_NONE, M_SH_SOLID, M_SH_HATCH, M_SH_DOT };
enum ArrowPosition { M_TAIL, M_CENTRE };
enum DisplayType { ABSOLUTE, INLINE, BLOCK, NONE, HIDDEN };
enum ListPolicy { M_LASTONE, M_CYCLE };
enum GraphicsFormat {PS, EPS, PDF, SVG, KML, PNG, X, CPS, CSVG, GIF, AGIF, JPG, OPENGL};
 
inline MAGICS_NO_EXPORT string getOutputString()
{
	const string outputs = MAGICS_OUTPUTS;
	return outputs;
}

const double epsilon = 1.25e-7;

template<class T>
inline MAGICS_NO_EXPORT T abs(const T a) { return (a < 0) ? -a : a; }

#ifndef PI
const double PI = atan(1.)*4.;
#endif

inline MAGICS_NO_EXPORT double RAD(const double r) { return r*PI/180.;}
inline MAGICS_NO_EXPORT double DEG(const double d) { return d*180./PI;}
inline MAGICS_NO_EXPORT bool zero(const double v) { return abs(v) < epsilon; }
inline MAGICS_NO_EXPORT bool same(const double a, const double b) { return zero(a-b); }

//! Global function to read env variables
inline MAGICS_NO_EXPORT string getEnvVariable(const string var)
{
	const char* ww = getenv(var.c_str());
	if(ww) return string(ww);
	return "";
}

//! Global function to return the Magics++ version
/*! comes from config.h !!! */
inline string getMagicsVersion()
{
	const string version = MAGICS_VERSION;
	if(sizeof(long)==8) return version+string(" (64 bit)");
	return version;
}

//! Global function to return the Magics++ version for ID line
/*! comes from config.h !!! */
inline string getMagicsVersionString()
{
	const string magics  = MAGICS_NAME;
	const string version = getMagicsVersion();
	return magics+string(" ")+version;
}

//! Global function to return Magics release information
inline string getMagicsRelease()
{
	std::string f;
	const string name    = MAGICS_NAME;
	const string version = getMagicsVersion();
	const string date    = MAGICS_RELEASE_DATE;
	f = name + " " + version + " released on " + date; // comes from config.h !!!
	return f;
}

//! Global function to get information how Magics was compiled
inline string getMagicsConfigureInfo()
{
	std::string f("configure ");
	return f;
}

// inline MAGICS_NO_EXPORT int upper_case(const int c) { return toupper(c);}
inline MAGICS_NO_EXPORT int lower_case(const int c) { return tolower(c);}

//! Make an lowercase copy of s:
inline MAGICS_NO_EXPORT string lowerCase(const string& s)
{
	string out;
	for ( string::const_iterator l = s.begin(); l != s.end(); ++l)
		out.push_back(tolower(*l));
	return out;
}

/*!
  \brief compares two strings
*/
inline MAGICS_NO_EXPORT bool magCompare(const string &s1, const string &s2)
{
	if(s1.size() != s2.size()) return false;
#ifndef MAGICS_WINDOWS_CYGWIN
	return !( strcasecmp(s1.c_str(),s2.c_str()) );
#endif
}

/*!
  \brief returns the biggest integer inside a double
*/
inline double maground(double x)
{
	return floor(x + 0.5);
}

inline double tonumber(const string& str) 
{
	double r;
	std::stringstream ss(str);
	ss >> r;
	return r;
}

template <class T>
inline string tostring(const T& in) 
{
	std::ostringstream out;
	out << in;
	return out.str();
}


/*!
  \brief checks if file readable or not
*/
inline MAGICS_NO_EXPORT bool fileReadable(const string &strFile)
{
    FILE *fp = fopen (strFile.c_str(), "r");
    if (fp != NULL)
    {
        fclose (fp);
        return true;
    }
    else
    {
        return false;
    }
}

/*!
  \brief checks if file readable or not

  \note Design to work at ECMWF only! Graphics Section members are excluded 
  from the list.
  \warning The log file can grow very quickly and needs constantly checking.
*/
inline MAGICS_NO_EXPORT void writeLog(const string &interface)
{
  if(magCompare(MAGICS_SITE,"ecmwf"))
  {
   string logpath = getEnvVariable("MAG_LOGGING_FILE");
   if(logpath.empty()) logpath = "/vol/netlog/magics/usage_log";
   FILE* filePtr_ = fopen(logpath.c_str(),"a");

   if(filePtr_)
   {
	string id = getEnvVariable("USER"); //user-id

	if(! (id=="cgm" || id=="cgs" || id=="cgi" || id=="cgk" || id=="cgr" ) )
	{
		char            host[256];
		char            mytime[24];
		time_t          when;

		time(&when);
		strncpy(mytime,ctime(&when),24); 
		string smytime(mytime,24);      // date time

		string arch = getEnvVariable("ARCH");
		if(arch.empty()) arch = "Unknown architecture";
		
		if(gethostname(host, sizeof(host)) == -1) strcpy(host, "unknown-host!?!?");

		string inter;
		if(magCompare(interface,"magml")) inter = "x"; 
		else if(magCompare(interface,"fortran")) inter = "*"; 
		else inter = "?";

		string logline = inter + id + "/" +  smytime + "/"+ getMagicsVersion() + "D/" + host + "/" + arch;
		fprintf(filePtr_,"%s\n",logline.c_str());

		fflush(filePtr_);
		fclose(filePtr_);
	}//endif NOT cgm or cgs
   }
  }
}

#define MAGPLUS_PATH_TO_SHARE_ "/share/magics/"
#define MAGPLUS_LINK_  "http://www.ecmwf.int/publications/manuals/magics"
#define MAGPLUS_PATH_TO_PS_FONTS_ POSTSCRIPT_FONT_PATH;
}
#endif
