/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file OpenGLDriverObserver.h
    \brief Definition of the OpenGLDriverObserver.
    \author Graphics Section, ECMWF
    
    Started: Jan 2005    
*/

#ifndef OpenGLDriverObserver_H
#define OpenGLDriverObserver_H

#include <magics.h>
#include <Log.h>
#include <PaperPoint.h>
#include <GeoPoint.h>
#include <OpenGLCommon.h>

class MtInputEvent;

namespace magics {

class Layout;

class SelectionObject : public vector<PaperPoint> 
{
public:
	SelectionObject(const Layout& layout): layout_ (layout), zoomLevel_(-1) {};
	~SelectionObject() {};

	const Layout& layout() { return layout_; }
	int zoomLevel() {return zoomLevel_;}
	void zoomLevel(int z) {zoomLevel_=z;} 

protected:
	const Layout& layout_;
	int zoomLevel_;
	
};

class MagicsEvent;

class OpenGLDriverObserver {

public:
	OpenGLDriverObserver() {};
	virtual ~OpenGLDriverObserver() {};
    
	//! Method called when the Widget is ready to receive and display
	// Graphical Object 
	virtual void ready() {}

	virtual void update() {}

	//! Method called when the Widget needs to receive the graphical tree again. 
	virtual void redraw() {}

	//! Method called in ToolTipMode :  x and y are in magics coordinates
	virtual void tooltip(SelectionObject*)  {}
	//! Method called in PointSelectionMode  :  x and y are in magics coordinates
	virtual void pointSelection(SelectionObject*)  {}
	//! Method called in AreaSelectionMode  :  x and y are in magics coordinates
	virtual void areaSelection(SelectionObject*)  {}
	//! Method called in LineSelectionMode  :  x and y are in magics coordinates
	virtual void lineSelection(SelectionObject*)  {}
	//! Method called in polygoneSelectionMode  :  x's and y's are in magics coordinates
	virtual void polygonSelection(SelectionObject*)  {}
	//! Method called in polylineSelectionMode  :  x's and y's are in magics coordinates
	virtual void polylineSelection(SelectionObject*)  {}
	virtual void selection(SelectionObject*)  {}
	virtual void zoomLevelSelection(SelectionObject*)  {}

	//! Method called after input x and y are in user coordinates 
#ifdef later
	virtual void input(const string& area, float x, float y. const )
	{
		Log::info() <<"NEEDS IMPLEMENTING: OpenGLDriverObserver::input("<<x<<","<<y<<") in " << area << " called."<< "\n";
	}
#endif

	//! method called after a resize
	virtual void resize( float width, float height)
	{
		Log::info() <<"NEEDS IMPLEMENTING: OpenGLDriverObserver::resize("<<width<<","<<height<<") called."<< "\n";
	}

	//! Method to dispatch the zoomstack status change from the OpenGL driver to the OpenGL driver manager
	virtual void zoomControlStatusForOpenGLGui(bool) {}

	//! Method to dispatch the magnifier status change from the OpenGL driver to the OpenGL driver manager
	virtual void  magnifierControlStatusForOpenGLGui(bool) {}

	//! Method to dispatch the input events from the OpenGL driver to the OpenGL driver manager
	virtual void  inputEventDispatchFromDriver(MtInputEvent*) {};	

	//! Method called to start and stop the layers???	
	virtual void  startLayerMenu() {};			
	virtual void  stopLayerMenu() {};
	
	//! Method to dispatch the framechange event from the OpenGL driver to the OpenGL driver manager
	virtual void  currentFrameChanged() {};	
	
	virtual void  setInteractiveMode(InteractiveMode) {};			

	typedef void (OpenGLDriverObserver::*InteractiveModeFunction)(InteractiveMode);	
	typedef void (OpenGLDriverObserver::*SelectionFunction)(SelectionObject*);
	typedef void (OpenGLDriverObserver::*Function)();
	typedef void (OpenGLDriverObserver::*BoolFunction)(bool);
	typedef void (OpenGLDriverObserver::*IntFunction)(int);
	typedef void (OpenGLDriverObserver::*StringFunction)(const string&);
	typedef void (OpenGLDriverObserver::*InputEventFunction)(MtInputEvent*);
	
	virtual void notify(MagicsEvent&) {}

protected:
	//! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream& out) const { out << " OpenGLDriverObserver " << "\n"; }

private:
	//! Copy constructor - No copy allowed
	OpenGLDriverObserver(const OpenGLDriverObserver&);
	//! Overloaded << operator to copy - No copy allowed
	OpenGLDriverObserver& operator=(const OpenGLDriverObserver&);

// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const OpenGLDriverObserver& p)
		{ p.print(s); return s; }
};

typedef  OpenGLDriverObserver* OpenGLDriverObserverPtr;

} // namespace magics
#endif
