
/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 	http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLZoomStackGui.cc
    \brief Implementation of the OpenGLZoomStackGui class.
    \author Graphics Section, ECMWF

    Started: March 2008
*/

#include "Label.h"
#include "PaperPoint.h"
#include "Text.h"
#include "OpenGLZoomStackGui.h"
#include <OpenGLLayoutNode.h>
#include <OpenGLDriver.h>
#include <OpenGLNode.h>
#include <OpenGLPushButtonWidget.h>
#include <OpenGLCellBarWidget.h>
#include <OpenGLTextureItem.h>
#include <OpenGLFboItem.h>
#include <OpenGLCore.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include <math.h>

#include <MtInputEvent.h>

OpenGLZoomStackGui::OpenGLZoomStackGui(OpenGLDriver *driver,string name) : OpenGLGui(driver,name)
{
	//Widget width is fixed
	width_=65;

	//Cell size
	cellWidth_=20.;
	cellHeight_=20.;

	borderWidth_=5;
	borderHeight_=5;

	//Push button size
	pbWidth_=20;
	pbHeight_=20;

	//Gap between push buttons and cell-bar
	pbGapBar_=5.;

	//Gap between push buttons and gui border
	pbGapBorder_ =5;

	//Initially we do not have levels
	levelNum_=0;
	actLevel_=-1;

	//Preview
	levelPreviewWidth_=180;
	levelPreviewBorder_=5;
	levelPreviewBorderColour_=Colour(0.8,0.8,0.8);

	focusedLevel_=-1;

	bgTex_= new OpenGLTextureItem;
	bgTex_->transparent(false);

	visible_=false;

	x_=-1;
	y_=-1;

	//Cellbar
	actCellColour_=Colour(1/255.,251./255.,243./255.);
	cellColour_=Colour(152/255.,165/255.,215/255.);

	//
	readSettings();

	//Build the gui (create its children)
	build();

	//Get the current level_num and set geometry accordingly
	//and render gui
	update();
}


OpenGLZoomStackGui::~OpenGLZoomStackGui()
{
	delete bgTex_;
	clearPreviewTex();
}

void OpenGLZoomStackGui::readSettings()
{		
	string fconf=getEnvVariable("MAGPLUS_HOME") + MAGPLUS_PATH_TO_SHARE_ + "ui/zoomstack_gui.conf";
	ifstream in(fconf.c_str(), ifstream::in);

        if(in)
        {   
  		char c[256];        
                while(in.getline(c,256))
                {
			string stmp=c; 
 
   			if(stmp.find("#") == string::npos)
			{
				string stmp=c;
				stringstream ss(stmp);

				string buf;
        			vector<string> sv;
				while (ss >> buf)
                        	{
					sv.push_back(buf);
				}
				
				if(sv.size() == 4)
				{
					Colour *col=0;

					float rgb[3];
					for(int i=0; i < 3; i++)
					{
						istringstream rgbstr(sv[i+1]);
						rgbstr >> rgb[i];
					}	

					if(sv[0].find("cell") != string::npos)
					{
						cellColour_.setColour(rgb[0]/255.,rgb[1]/255.,rgb[2]/255.);
					}
					else if(sv[0].find("actCell") != string::npos)
					{
						actCellColour_.setColour(rgb[0]/255.,rgb[1]/255.,rgb[2]/255.);
					}
					else if(sv[0].find("previewBorder") != string::npos)
					{
						levelPreviewBorderColour_.setColour(rgb[0]/255.,rgb[1]/255.,rgb[2]/255.);
					}					
				}
			}
		}
                  
	}

}

//-----------------------------------------------------------------------
// Build the gui by creating the children. At this point we do not set the
// geometry
//-----------------------------------------------------------------------

void OpenGLZoomStackGui::build()
{
	if(children_.size() != 0) return;
	if(built_ == true) return;

	OpenGLBaseWidget* instance=(OpenGLBaseWidget*) this;

	//Up button
	pbUp_ = new OpenGLPushButtonWidget(driver_,instance,"pbUp");
	pbUp_->setPixmap(getSharedPngFileName("up","20x20"),"png");
	//pbUp_->setPixmapWidth(pbWidth_-4);
	//pbUp_->setPixmapHeight(pbHeight_-4);
	pbUp_->setWidth(pbWidth_);
	pbUp_->setHeight(pbHeight_);
	//pbUp_->setColour("background",Colour(0.9,0.9,0.9));
	pbUp_->addCallback(OpenGLZoomStackGui::stepUp_cb,this,OpenGLWidget::ActivateCallback);
	addChild(pbUp_);

	//Down button
	pbDown_ = new OpenGLPushButtonWidget(driver_,instance,"pbDown");
	pbDown_->setPixmap(getSharedPngFileName("down","20x20"),"png");
	//pbDown_->setPixmapWidth(pbWidth_-4);
	//pbDown_->setPixmapHeight(pbHeight_-4);
	pbDown_->setWidth(pbWidth_);
	pbDown_->setHeight(pbHeight_);
	//pbDown_->setColour("background",Colour(0.9,0.9,0.9));
	pbDown_->addCallback(OpenGLZoomStackGui::stepDown_cb,this,OpenGLWidget::ActivateCallback);
	addChild(pbDown_);

	//Cell-bar
	bar_ = new OpenGLCellBarWidget(driver_,"bar");
	bar_->setWidth(cellWidth_);
	bar_->setCellWidth(cellWidth_);
	bar_->setCellHeight(cellHeight_);
	bar_->setColour("cellAct",actCellColour_);
	bar_->setColour("cell",cellColour_);
	//bar_->setSliderPixmap(getSharedPngFileName("round_button","16x16"),"png");
	bar_->addCallback(OpenGLZoomStackGui::stepTo_cb,this,OpenGLWidget::ValueChangedCallback);
	bar_->addCallback(OpenGLZoomStackGui::focusedLevelChanged_cb,this,OpenGLWidget::FocusChangedCallback);
	addChild(bar_);

	OpenGLGui::build();

	built_=true;
}

void OpenGLZoomStackGui::updateGeometry()
{
	int height_ori=height_;

	setHeight(2*(pbGapBorder_+pbGapBar_+pbHeight_)+levelNum_*cellHeight_+titleBarHeight_+borderHeight_);

	if(x_< 0 && y_ < 0)
	{
		setX(driver_->dimensionX()-width_-40);
		setY(driver_->dimensionY()-height_-120);
	}
	else
	{
		setY(y_+height_ori-height_);
	}

	int centreX=width_/2.;

	pbDown_->setX(x_+centreX-pbWidth_/2.);
	pbDown_->setY(y_+pbGapBorder_);

	pbUp_->setX(x_+centreX-pbWidth_/2.);
	pbUp_->setY(y_+height_-pbGapBorder_-pbHeight_-titleBarHeight_-borderHeight_);

	bar_->createCells(levelNum_,actLevel_);

	bar_->setX(x_+centreX-cellWidth_/2.);
	bar_->setY(y_+pbGapBorder_+pbHeight_+pbGapBar_);
}

//---------------------------------------
// Update gui state and geometry
//---------------------------------------

void OpenGLZoomStackGui::update()
{
	//Get the new number of levels
	int levelNumOri=levelNum_;
	int actLevelOri=actLevel_;

	//Find the first zoomable layout and
	//get the current level values
	list<OpenGLNode*> gll_list;
	driver_->glTree()->findLayout(gll_list);

	currentLayout_=0;
	for(list<OpenGLNode*>::iterator it=gll_list.begin(); it != gll_list.end(); it++)
	{
		OpenGLLayoutNode* gll = (OpenGLLayoutNode*) (*it);
		if(gll->layout().isZoomable() == true)
		{
			currentLayout_=gll;
			levelNum_=gll->layout().zoomLevels();
			actLevel_=gll->layout().zoomCurrentLevel();

			Log::dev() << "Zoomstack --> " << levelNum_ << "  " << actLevel_ << endl;
			break;
		}
	}

	//We cannot handle too many levels!!
	if(levelNum_ > 30)
	{
		levelNum_=0;
		actLevel_=0;
	}

	//Clear the bg texture
	bgTex_->clear();

	// By default levelNum = 0 and actLevel = -1
	// and there are no zoom levels.
	// We save he preview map into a texture
	if(levelNum_<=0)
	{
		//The gui is not enabled yet
		enabled_=false;

		//Clear the preview textures
		clearPreviewTex();

		//Save the preview map into a texture
		if(levelNum_==0 && actLevel_==-1 && currentLayout_ != 0)
		{
			OpenGLTextureItem *atex= new OpenGLTextureItem;
			atex->transparent(false);
			generateLevelPreviewTex(atex);
			levelTex_.push_back(atex);
		}

		return;
	}
	else
	{
		enabled_=true;
	}

	//If the levelNum or the current level has changed.
	//Here we cannot treat correctly the situation when simply
	//the current level has changed.
	if(levelNumOri != levelNum_  || actLevelOri != actLevel_)
	{
		focusedLevel_=-1;

		//If there are less levels
		if(levelNumOri > levelNum_)
		{
			for(int i=levelNumOri-1; i> levelNum_; i--)
			{
				delete levelTex_.back();
				levelTex_.pop_back();
			}

			generateLevelPreviewTex(levelTex_.back());
		}

		//If there are more levels
		else if(levelNumOri < levelNum_)
		{
			OpenGLTextureItem *atex= new OpenGLTextureItem;
			atex->transparent(false);
			generateLevelPreviewTex(atex);
			levelTex_.push_back(atex);
		}

		//If the level num is the same. Here we should not regenerate
		//the texture all the time.
		else if(levelNumOri == levelNum_)
		{
			generateLevelPreviewTex(levelTex_[actLevel_]);
		}
	}

	// frame buffer ????
	//driver_->mapBgImageToFb(0,x_,y_,x_+width_,y_+height_+12);

	//Update widget geometry
	updateGeometry();

	//Generate a new texture for the gui
	if(levelNum_ != levelNumOri)
	{
		clearTexture();
		//tex_->generate();
	}

	updateButtonState();

	rendered_=false;
}

//-------------------------------------------------
// Callbacks - static wrappers
//-------------------------------------------------

void OpenGLZoomStackGui::stepUp_cb(void *object,void *data)
{
	OpenGLZoomStackGui *instance = (OpenGLZoomStackGui*) object;
	instance->stepUp();
}

void OpenGLZoomStackGui::stepDown_cb(void *object,void *data)
{
	OpenGLZoomStackGui *instance = (OpenGLZoomStackGui*) object;
	instance->stepDown();
}

void OpenGLZoomStackGui::stepTo_cb(void *object,void *data)
{
	OpenGLZoomStackGui *instance = (OpenGLZoomStackGui*) object;
	instance->stepTo((int*) data);
}

void OpenGLZoomStackGui::focusedLevelChanged_cb(void *object,void *data)
{
	OpenGLZoomStackGui *instance = (OpenGLZoomStackGui*) object;
	instance->focusedLevelChanged((int*) data);
}

//-------------------------------------------------
// Callbacks - concrete implementations
//-------------------------------------------------

void OpenGLZoomStackGui::stepUp()
{
	if(actLevel_ < levelNum_-1)
	{
		setActLevel(actLevel_+1);
	}
}

void OpenGLZoomStackGui::stepDown()
{
	if(actLevel_>0)
	{
		setActLevel(actLevel_-1);
	}
}

void OpenGLZoomStackGui::stepTo(int *level)
{
	if(actLevel_ != *level)
	{
		setActLevel(*level);
	}
}

void OpenGLZoomStackGui::setActLevel(int level)
{
	if(level >=0 && level < levelNum_)
	{
		//Update will set it later!!!!!!!
		//actLevel_=level;

		bgTex_->clear();

		SelectionObject so(currentLayout_->layout());
		so.zoomLevel(level);
		driver_->notifyObservers(&OpenGLDriverObserver::zoomLevelSelection,&so);
		driver_->notifyObservers(&OpenGLDriverObserver::update);
	}
}

void OpenGLZoomStackGui::focusedLevelChanged(int *level)
{
	renderLevelPreview(*level);
	focusedLevel_=*level;
	changed_=true;

	//if(*level != focusedLevel_)
	//{
	//	renderLevelPreview(*level);
	//	focusedLevel_=*level;
	//}
}

void OpenGLZoomStackGui::renderLevelPreview(int level)
{
	int w, h, xp1, xp2, yp1, yp2;
	int bw=levelPreviewBorder_;

	//If the texture
	if(bgTex_->isFilled())
	{
		xp1=bgTex_->x();
		xp2=xp1+bgTex_->width();
		yp1=bgTex_->y();
		yp2=yp1+bgTex_->height();

		//bgTex_->mapTexture(xp1,yp1,xp2,yp2);

		/*for(int i=0; i < 100; i++)
		{
			bgTex_->generateFromFb(xp1-400,yp1-300,xp2-xp1,yp2-yp1,GL_RGB);
			bgTex_->mapTexture(xp1-400,yp1-300,xp2-400,yp2-300);
			bgTex_->clear();
			
		}*/
		//Log::dev()<< "bg text map: " << xp1 << " " << yp1 << " "  << xp2 << " " << yp2 << endl;
	}

	if(level != -1)
	{
		w=levelTex_[level]->width();
		h=levelTex_[level]->height();

		if(x_-20-w > 50)
		{
			xp1=x_-20-w;
			xp2=x_-20;
			yp1=y_+height_-h;
			yp2=y_+height_;
		}
		else
		{
			xp1=x_+width_+20;
			xp2=xp1+w;
			yp1=y_+height_-h;
			yp2=y_+height_;
		}
		
		int xpb=xp1-bw;
		int ypb=yp1-bw;
		int wb=w+2*bw;
		int hb=h+2*bw;
		
		/*float xpb=floor(xp1-bw);
		float ypb=floor(yp1-bw);
		float wb=ceil(w+2*bw);
		float hb=ceil(h+2*bw);*/

		//bgTex_->clear();
		//bgTex_->generateFromFb(xpb,ypb,wb,hb,GL_RGB);

		//Log::dev()<< "bg text gen: " << floor(xp1-bw) << " " << floor(yp1-bw) << " "  << //floor(xp1-bw) + ceil(w+2*bw) << " " << floor(yp1-bw)+ceil(h+2*bw) << endl;

		//Log::dev()<< "bg text gen: " << xpb << " " << ypb << " "  << xpb+wb << " " << ypb+hb << endl;


		glPushAttrib(GL_POLYGON_BIT);
		glFrontFace(GL_CCW);
		glEnable(GL_CULL_FACE);
		glCullFace(GL_BACK);

		/*glPolygonMode(GL_FRONT,GL_LINE);

		Colour col(0.,0.,0.);
		glColor3f(col.red(),col.blue(),col.green());
		glRectf(xp1,yp1,xp2,yp2);*/

		glPolygonMode(GL_FRONT,GL_FILL);
		Colour col=levelPreviewBorderColour_;
		glColor3f(col.red(),col.green(),col.blue());
		glRectf(xp1-bw,yp1-bw,xp2+bw,yp2+bw);

		glPopAttrib();

		levelTex_[level]->mapTexture(xp1,yp1,xp2,yp2);

		//Log::dev()<< "zoom text map: " << xp1 << " " << yp1 << " "  << xp2 << " " << yp2 << endl;
	}
	else
	{
		bgTex_->clear();
	}

	//Swap buffers
	//driver_->swapFb();
}

void OpenGLZoomStackGui::generateLevelPreviewTex(OpenGLTextureItem *tex)
{
	//Find the preview layout in the specified layout
	list<OpenGLNode*> gln;
	driver_->glTree()->findPreviewLayout(gln);

	OpenGLPreviewLayoutNode* glm=0;

	for(list<OpenGLNode*>::iterator it=gln.begin();it != gln.end(); it++)
	{
		glm=(OpenGLPreviewLayoutNode*) *it;
		break;
	}

	if(glm == 0) return;

    	//Get the preview layout area
	int x=glm->winX();
	int y=glm->winY();
	int w=glm->winWidth();
	int h=glm->winHeight();

	//Define the area where the preview plot will be rendered
	float tw=levelPreviewWidth_;
	if(tw > driver_->dimensionX()-20-width_-50)
	{
		tw = driver_->dimensionX()-20-width_-50;
	}

	float r=tw/static_cast<float>(w);
	float th=static_cast<float>(h)*r;

	if(th > levelPreviewWidth_)
	{
		th=levelPreviewWidth_;
		r=th/static_cast<float>(h);
		tw=static_cast<float>(w)*r;
	}
	
	if(th > driver_->dimensionY()-20-20)
	{
		th=driver_->dimensionY()-20-20;
		r=th/static_cast<float>(h);
		tw=static_cast<float>(w)*r;
	}
	
	tex->generateFromImage(0,tw,th,GL_RGB);

	OpenGLFboItem *fboTmp;
	if(fbo_)
	{
		fboTmp=new OpenGLFboItem;
		fboTmp->bind();
		fboTmp->attachTexture(tex->id());
	}

	//Save the fb content into a tmp texture.
        //We suppose that the rendered preview size <= than the layout size!!!
	//OpenGLTextureItem *tex_tmp= new OpenGLTextureItem;
	//tex_tmp->transparent(false);
	//tex_tmp->generateFromFb(x,y,w,h,GL_RGB);

	//Render the preview plot
	driver_->renderPreview(glm,x,y,tw,th);

	//Save the preview plot into a texture
	//tex->generateFromFb(x,y,static_cast<int>(tw),static_cast<int>(th),GL_RGB);

	//Reload the saved fb content
	//tex_tmp->mapTexture(x,y,w,h);

	//Detelet the tmp texture
	//delete tex_tmp;
	if(fbo_)
	{
		fboTmp->unBind();
		delete fboTmp;
	}
}

void OpenGLZoomStackGui::clearPreviewTex()
{
	for(int i =0; i < levelTex_.size(); i++)
	{
		delete levelTex_[i];
	}
	levelTex_.clear();
}


void OpenGLZoomStackGui::show()
{
	if(focusedLevel_ != -1)
	{
		if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
		{
			OpenGLCore::instance()->fboFb()->bind();
			
		}

		renderLevelPreview(focusedLevel_);

		if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
		{
			OpenGLCore::instance()->fboFb()->unBind();
		}

		renderChild(bar_);
	}

	OpenGLGui::show();
	changed_=false;
}



void OpenGLZoomStackGui::event(MtInputEvent *event)
{
	if(event->type()== Mt::KeyPressEvent)
	{
		MtKeyEvent *kev = static_cast<MtKeyEvent*>(event);

		if(kev->key() == Mt::Key_Up)
		{
			stepUp();
		}
		else if(kev->key() == Mt::Key_Down)
		{
			stepDown();
		}
		return;

	}

	if(!visible_) return;
	OpenGLGui::event(event);
}

void OpenGLZoomStackGui::notifyObserverOfVisibilityChange(bool b)
{
	//driver_->notifyObservers(&OpenGLDriverObserver::zoomControlStatus,b);
}

void OpenGLZoomStackGui::updateButtonState()
{
	int c=bar_->getActCell();
	if(c==0)
	{
		pbDown_->setEnabled(false);
	}
	else
	{
		pbDown_->setEnabled(true);
	}	

	if(c == bar_->getCellNum()-1)
	{		
		pbUp_->setEnabled(false);
	}
	else
	{	
		pbUp_->setEnabled(true);
	}
}