/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLDriverManagerManager.cc
    \brief Definition of OpenGLDriverManager
    \author Graphics Section, ECMWF

    Started: May 2008
*/

#include "OpenGLGui.h"
#include "OpenGLDriverManager.h"
#include "OpenGLZoomStackGui.h"
#include "OpenGLMagnifier.h"
#include "OpenGLPickObjectCreator.h"
#include <OpenGLDriver.h>
#include <OpenGLCore.h>

#include "MagicsEvent.h"

#include <GL/gl.h>
#include <GL/glu.h>

#include "MtInputEvent.h"

OpenGLDriverManager::OpenGLDriverManager(OpenGLDriver *d) : driver_(d)  
{
	interactiveMode_=IAM_NONE;
	
	gui_["zoom"] = new OpenGLZoomStackGui(driver_,"zoomGui");
	gui_["magnifier"] =  new OpenGLMagnifier(driver_,"magnifierGui");

	tool_[IAM_ZOOM] = new OpenGLPickObjectCreator(d);
	//tool_[IAM_PICK] =new OpenGLPickObjectManager(d);
	
	focusedGui_=0;
	activeTool_=0;		
} 

void
OpenGLDriverManager::setInteractiveMode(InteractiveMode new_mode)
{
	if(interactiveMode_ == new_mode)
	{		
		leaveMode(interactiveMode_);
		interactiveMode_=IAM_NONE;
		//mapAllGui(true,true);			
	}
	else
	{			
		leaveMode(interactiveMode_);

		interactiveMode_=IAM_NONE;
		//mapAllGui(true,false);

		interactiveMode_=new_mode;
		enterMode(interactiveMode_);

	}				
}

void OpenGLDriverManager::leaveMode(InteractiveMode mode)
{
	if(tool_.find(mode) != tool_.end())
	{
		if(mode == IAM_ZOOM)
		{
			MagicsCursorEvent event;
			driver_->notifyObservers(event);
		}

		tool_[mode]->leaveMode();
	}
}

void OpenGLDriverManager::enterMode(InteractiveMode mode)
{	
	if(tool_.find(mode) != tool_.end())
	{		
		tool_[mode]->enterMode(mode);

		if(mode == IAM_ZOOM)
		{
			MagicsCursorEvent event("Cross");
			driver_->notifyObservers(event);
			
		}
	}
}

void  OpenGLDriverManager::magnifierControlStatusForOpenGLGui(bool s)
{
	setGuiVisibility(gui_["magnifier"],s); 
}

void  OpenGLDriverManager::zoomControlStatusForOpenGLGui(bool s)
{
	setGuiVisibility(gui_["zoom"],s);
}

void OpenGLDriverManager::setGuiVisibility(OpenGLGui *ag,bool s)
{
	ag->setVisible(s);
	
	if(ag->enabled())
	{	
		if(ag->visible() == false)
		{
			ag->clear();
			if(ag == focusedGui_)
			{
				focusedGui_=0;
			}
		}
		plotToFb();
		show();
		swapFb();
	}
}


void OpenGLDriverManager::currentFrameChanged()
{		
	for(mglGuiMap::iterator it=gui_.begin(); it != gui_.end(); it++)					
	{						
		OpenGLGui *ag=it->second;             
		ag->currentFrameChanged();
	}

	show();
	swapFb();
}


void
OpenGLDriverManager::inputEventDispatchFromDriver(MtInputEvent *event)
{		
	//First check the active gui		
	if(focusedGui_ != 0)
	{
		focusedGui_->event(event);

		if(focusedGui_->active())
		{
			plotToFb();
			showTheRest(focusedGui_);
			//focusedGui_->event(event);
			focusedGui_->show();
			swapFb();
		}
		else if(focusedGui_->focused() == false)
		{
			plotToFb();
			show();
			swapFb();		
			focusedGui_=0;		
		}		
		return;
	}	
		
	if(interactiveMode_==IAM_ZOOM)
	{
		MagicsCursorEvent cursor("Cross");
		driver_->notifyObservers(cursor);
	}
	
	//Then check the active tool		
	if(activeTool_ != 0)
	{
		activeTool_->event(event);				
		if(activeTool_->active())
		{
			return;
		}
		activeTool_ = 0;			
	}
			
	//If there is neither active gui nor active tool then:
		
	//Go through all the guis			
	for(mglGuiMap::iterator it=gui_.begin(); it != gui_.end(); it++)					
	{						
		OpenGLGui *ag=it->second;
                if(ag->enabled() ==true && ag->visible() == true)
		{
			ag->event(event);
				
			//new active gui
			if(ag->focused())
			{									
				if(interactiveMode_==IAM_ZOOM)
				{
					MagicsCursorEvent event;
					driver_->notifyObservers(event);
				}
				focusedGui_=ag;

				plotToFb();
				show();
				swapFb();
				return;
			}		
		}
	}
	
	//Check the available tool (only one tool can work at a time!)
	if(tool_.find(interactiveMode_) != tool_.end())
	{				
		tool_[interactiveMode_]->event(event);
		if(tool_[interactiveMode_]->active())
		{									
			activeTool_ = tool_[interactiveMode_];
			return;
		}							
	}
}	


void OpenGLDriverManager::update()
{	
	Log::dev() << "update -->" << endl;

	for(mglGuiMap::iterator it=gui_.begin(); it != gui_.end(); it++)					
	{						
		OpenGLGui *ag=it->second;             
		ag->update();
	}

	show();
	swapFb();
}

void OpenGLDriverManager::show()
{
	//if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	//{
		//OpenGLCore::instance()->bgToFb();
		for(mglGuiMap::iterator it=gui_.begin(); it != gui_.end(); it++)					
		{						
			OpenGLGui *ag=it->second;
                	if(ag->visible() && ag->enabled())
			{
				ag->show();
			}
		}
	//}
}

void OpenGLDriverManager::showTheRest(OpenGLGui *notThis)
{
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		for(mglGuiMap::iterator it=gui_.begin(); it != gui_.end(); it++)					
		{						
			OpenGLGui *ag=it->second;
                	if(ag != notThis && ag->visible() && ag->enabled())
			{
				ag->show();
			}
		}
	}
}

void OpenGLDriverManager::plotToFb()
{	
	OpenGLCore::instance()->bgToFb();
}

void OpenGLDriverManager::swapFb()
{
	driver_->swapFb();
}
