# -*- coding: utf-8 -*-

# Author: Natalia Bidart <nataliabidart@gmail.com>
#
# Copyright 2010-2011 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Magicicada GTK UI."""

import logging
import os
import time

from gettext import gettext as _

import gtk

# optional Launchpad integration
# this shouldn't crash if not found as it is simply used for bug reporting
try:
    import LaunchpadIntegration
    LAUNCHPAD_AVAILABLE = True
except ImportError:
    LAUNCHPAD_AVAILABLE = False

from magicicada import syncdaemon
from magicicada.dbusiface import NOT_SYNCHED_PATH
from magicicada.helpers import (
    humanize_bytes,
    log,
    NO_OP,
)
from magicicada.gui.gtk.helpers import Buildable, get_data_file
from magicicada.gui.gtk.operations import Operations
from magicicada.gui.gtk.listings import (
    FoldersButton,
    PublicFilesButton,
    SharesToMeButton,
    SharesToOthersButton,
)


UBUNTU_ONE_ROOT = os.path.expanduser('~/Ubuntu One')

logger = logging.getLogger('magicicada.gui.gtk')

# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Unused argument, we need them for GTK callbacks
# pylint: disable=W0613


class MetadataDialog(Buildable):
    """The metadata dialog."""

    filename = 'metadata.ui'

    changed_message = {
        syncdaemon.CHANGED_NONE: (u'Synchronized', gtk.STOCK_APPLY),
        syncdaemon.CHANGED_SERVER: (u'With server changes, downloading',
                                    gtk.STOCK_GO_DOWN),
        syncdaemon.CHANGED_LOCAL: (u'With local changes, uploading',
                                   gtk.STOCK_GO_UP),
    }

    def run(self):
        """Run the dialog."""
        self.dialog.show()

    def got_metadata(self, path, data):
        """Activate the information elements and hide the spinner."""
        # stop and hide the spinner
        self.spinner.stop()
        self.spinner.hide()

        # title with crude path
        self.dialog.set_title("Metadata for %s" % (path,))

        # the icon is taken from crude path, no matter if synched or not
        if os.path.isdir(path):
            icon = gtk.STOCK_DIRECTORY
        else:
            icon = gtk.STOCK_FILE
        self.filetype_image.set_from_stock(icon, gtk.ICON_SIZE_MENU)

        # set the data in the elements
        if data == NOT_SYNCHED_PATH:
            # metadata path doesn't exist for syncdaemon, show crude path,
            # error message, and quit
            self.path_label.set_text(path)
            self.filepath_hbox.show()
            self.basic_info_label.set_text(NOT_SYNCHED_PATH)
            self.basic_info_label.show()
            return

        # show the nice path
        self.path_label.set_text(data['path'])
        self.filepath_hbox.show()

        # prepare simple text
        simple = []
        stat = data['stat']
        if stat is not None:
            size = stat['st_size']
            try:
                size = humanize_bytes(size)
            except (ValueError, TypeError):
                logger.exception('Error while humanizing bytes')
            simple.append("Size: %s" % (size,))
            tstamp = stat['st_mtime']
            simple.append("Modified on %s" % (time.ctime(tstamp),))

        # set state message and image
        state, stock = self.changed_message.get(data['changed'],
                                                ('Unknown state', None))
        simple.append(state)
        simple_text = "\n".join(simple)
        if stock is None:
            self.state_image.hide()
        else:
            self.state_image.set_from_stock(stock, gtk.ICON_SIZE_LARGE_TOOLBAR)
        self.state_hbox.show()

        # prepare detailed text
        raw = data['raw_result']
        detailed_text = '\n'.join('%s: %s' % i for i in raw.iteritems())

        # fill and show
        self.basic_info_label.set_text(simple_text)
        self.basic_info_label.show()
        self.detailed_info_textview.get_buffer().set_text(detailed_text)
        self.details_expander.show()

    def on_dialog_close(self, widget, data=None):
        """Close the dialog."""
        self.dialog.hide()


class MagicicadaUI(Buildable):
    """Magicicada GUI main class."""

    CURRENT_ROW = '<b><span foreground="#000099">%s</span></b>'
    STATUS_JOINER = ' - '
    STATUS = {
        'initial': _('Service is not started, click Start to continue.'),
    }

    _u1_root = UBUNTU_ONE_ROOT
    filename = 'main.ui'

    def __init__(self, on_destroy=NO_OP):
        """Init."""
        super(MagicicadaUI, self).__init__()
        self.sd = syncdaemon.SyncDaemon()

        if LAUNCHPAD_AVAILABLE:
            # for more information about LaunchpadIntegration:
            # wiki.ubuntu.com/UbuntuDevelopment/Internationalisation/Coding
            helpmenu = self.builder.get_object('helpMenu')
            if helpmenu:
                LaunchpadIntegration.set_sourcepackagename('magicicada')
                LaunchpadIntegration.add_items(helpmenu, 0, False, True)

        self._on_destroy = on_destroy

        animation_filename = get_data_file('media', 'loader-ball.gif')
        self.loading_animation = gtk.gdk.PixbufAnimation(animation_filename)
        active_filename = get_data_file('media', 'active-016.png')
        self.active_indicator = gtk.gdk.pixbuf_new_from_file(active_filename)

        self.metadata_dialogs = {}

        folders = FoldersButton(syncdaemon_instance=self.sd)
        shares_to_me = SharesToMeButton(syncdaemon_instance=self.sd)
        shares_to_others = SharesToOthersButton(syncdaemon_instance=self.sd)
        self.public_files = PublicFilesButton(syncdaemon_instance=self.sd)
        for button in (folders, shares_to_me, shares_to_others,
                       self.public_files):
            self.toolbar.insert(button, -1)
        self.toolbar.set_sensitive(False)

        self._icons = {}
        for size in (16, 32, 48, 64, 128):
            icon_filename = get_data_file('media', 'logo-%.3i.png' % size)
            self._icons[size] = gtk.gdk.pixbuf_new_from_file(icon_filename)
        self.main_window.set_icon_list(*self._icons.values())
        gtk.window_set_default_icon_list(*self._icons.values())

        self._status_icons = {}
        for style in ('idle', 'working', 'alert'):
            fname = get_data_file('media', 'icon-%s-16.png' % style)
            self._status_icons[style] = gtk.gdk.pixbuf_new_from_file(fname)
        self.status_icon = self.builder.get_object('status_icon')
        self.status_icon.set_from_pixbuf(self._status_icons['idle'])

        about_fname = get_data_file('media', 'logo-128.png')
        self.about_dialog.set_logo(gtk.gdk.pixbuf_new_from_file(about_fname))

        self.operations = Operations(syncdaemon_instance=self.sd)
        self.main_box.pack_start(self.operations, expand=True)

        self.sd.on_started_callback = self.on_started
        self.sd.on_stopped_callback = self.on_stopped
        self.sd.on_connected_callback = self.on_connected
        self.sd.on_disconnected_callback = self.on_disconnected
        self.sd.on_online_callback = self.on_online
        self.sd.on_offline_callback = self.on_offline
        self.sd.status_changed_callback = self.on_status_changed
        self.sd.on_metadata_ready_callback = self.on_metadata_ready
        self.sd.on_initial_data_ready_callback = self.on_initial_data_ready
        self.sd.on_initial_online_data_ready_callback = \
            self.on_initial_online_data_ready

        self.widget_is_visible = lambda w: w.get_visible()
        self.widget_enabled = lambda w: self.widget_is_visible(w) and \
                                        w.is_sensitive()
        self.update()

    # GTK callbacks

    def on_destroy(self, widget=None, data=None):
        """Called when this widget is destroyed."""
        self.sd.shutdown()
        self._on_destroy()

    on_main_window_destroy = on_destroy

    def on_quit_activate(self, widget, data=None):
        """Signal handler for closing the program."""
        self.on_main_window_destroy(self.main_window)

    def on_about_activate(self, widget, data=None):
        """Display the about box."""
        self.about_dialog.run()
        self.about_dialog.hide()

    def on_start_clicked(self, widget, data=None):
        """Start syncdaemon."""
        self.sd.start()
        self.start.set_sensitive(False)
        self._start_loading(self.is_started)

    def on_stop_clicked(self, widget, data=None):
        """Stop syncdaemon."""
        self.toolbar.set_sensitive(False)

        if self.widget_enabled(self.disconnect):
            self.on_disconnect_clicked(self.disconnect)
        self.connect.set_sensitive(False)

        self.stop.set_sensitive(False)
        self.sd.quit()

    def on_connect_clicked(self, widget, data=None):
        """Connect syncdaemon."""
        self.sd.connect()
        self.connect.set_sensitive(False)
        self._start_loading(self.is_connected)

    def on_disconnect_clicked(self, widget, data=None):
        """Disconnect syncdaemon."""
        self.disconnect.set_sensitive(False)
        self.sd.disconnect()

    def on_file_chooser_open_clicked(self, widget, data=None):
        """Close the file_chooser dialog."""
        self.file_chooser.response(gtk.FILE_CHOOSER_ACTION_OPEN)

    def on_file_chooser_show(self, widget, data=None):
        """Close the file_chooser dialog."""
        self.file_chooser.set_current_folder(self._u1_root)

    def on_metadata_clicked(self, widget, data=None):
        """Show metadata for a path choosen by the user."""
        res = self.file_chooser.run()
        self.file_chooser.hide()
        if res != gtk.FILE_CHOOSER_ACTION_OPEN:
            return

        path = self.file_chooser.get_filename()
        assert path is not None
        dialog = MetadataDialog()
        self.metadata_dialogs[path] = dialog
        self.sd.get_metadata(path)
        dialog.run()

    def on_status_icon_activate(self, widget, data=None):
        """Systray icon was clicked."""
        if self.widget_is_visible(self.main_window):
            self.main_window.hide()
        else:
            self.main_window.show()

    # SyncDaemon callbacks

    @log(logger)
    def on_started(self, *args, **kwargs):
        """Callback'ed when syncadaemon is started."""
        self.start.hide()
        self.stop.show()
        self.stop.set_sensitive(True)
        self._activate_indicator(self.is_started)
        self.connect.set_sensitive(True)

        self._update_status_label(self.sd.current_state)

    @log(logger)
    def on_stopped(self, *args, **kwargs):
        """Callback'ed when syncadaemon is stopped."""
        if self.widget_enabled(self.stop):
            self.on_stop_clicked(self.stop)

        self.stop.hide()
        self.start.show()
        self.start.set_sensitive(True)
        self.connect.set_sensitive(False)

        self._activate_indicator(self.is_started, sensitive=False)
        self._activate_indicator(self.is_connected, sensitive=False)
        self._activate_indicator(self.is_online, sensitive=False)

        self._update_status_label(self.sd.current_state)

    @log(logger)
    def on_connected(self, *args, **kwargs):
        """Callback'ed when syncadaemon is connected."""
        self.connect.hide()
        self.disconnect.show()
        self.disconnect.set_sensitive(True)
        self._activate_indicator(self.is_connected)
        self._start_loading(self.is_online)
        self._start_loading(self.is_online)

        self._update_status_label(self.sd.current_state)

    @log(logger)
    def on_disconnected(self, *args, **kwargs):
        """Callback'ed when syncadaemon is disconnected."""
        self.disconnect.hide()
        self.connect.show()
        self.connect.set_sensitive(True)

        self._activate_indicator(self.is_connected, sensitive=False)
        self._activate_indicator(self.is_online, sensitive=False)

        self._update_status_label(self.sd.current_state)

    @log(logger)
    def on_online(self, *args, **kwargs):
        """Callback'ed when syncadaemon is online."""
        self.is_online.set_sensitive(True)
        self._activate_indicator(self.is_online)
        self._update_status_label(self.sd.current_state)

    @log(logger)
    def on_offline(self, *args, **kwargs):
        """Callback'ed when syncadaemon is offline."""
        self._activate_indicator(self.is_online, sensitive=False)
        self._update_status_label(self.sd.current_state)

    @log(logger)
    def on_status_changed(self, *args, **kwargs):
        """Callback'ed when the SD status changed."""
        self.update(*args, **kwargs)

    @log(logger)
    def on_metadata_ready(self, path, metadata):
        """Lower layer has the requested metadata for 'path'."""
        if path not in self.metadata_dialogs:
            logger.info("on_metadata_ready: path %r not in stored paths!",
                        path)
            return

        dialog = self.metadata_dialogs[path]
        dialog.got_metadata(path, metadata)

    @log(logger, level=logging.INFO)
    def on_initial_data_ready(self):
        """Initial data is now available in syncdaemon."""
        self.toolbar.set_sensitive(True)
        self.public_files.set_sensitive(False)
        self.operations.load()

    @log(logger, level=logging.INFO)
    def on_initial_online_data_ready(self):
        """Online initial data is now available in syncdaemon."""
        self.public_files.set_sensitive(True)

    # custom

    def _start_loading(self, what):
        """Set a loader animation on 'what'."""
        what.set_sensitive(True)
        what.set_from_animation(self.loading_animation)

    def _activate_indicator(self, what, sensitive=True):
        """Set ready pixbuf on 'what' and make it 'sensitive'."""
        what.set_sensitive(sensitive)
        what.set_from_pixbuf(self.active_indicator)

    def _update_status_label(self, state):
        """Update the status label based on SD state."""
        values = (v for v in (state.name, state.description,
                              state.queues, state.connection) if v)
        text = self.STATUS_JOINER.join(values)
        if not (text or state.is_started):
            text = self.STATUS['initial']
        logger.debug('setting status label to %r', text)
        self.status_label.set_text(text)

    def update(self, *args, **kwargs):
        """Update UI based on SD current state."""
        current_state = self.sd.current_state
        logger.debug('updating UI with state %r', current_state)

        self._activate_indicator(self.is_online,
                                 sensitive=current_state.is_online)

        if current_state.is_started:
            self.on_started()
            if current_state.is_connected:
                self.on_connected()
                if current_state.is_online:
                    self.on_online()
            else:
                self.on_disconnected()
        else:
            self.on_disconnected()
            self.on_stopped()

        self._update_status_label(current_state)

        # change status icon
        state = kwargs.get('state')
        if state == syncdaemon.STATE_IDLE:
            self.status_icon.set_from_pixbuf(self._status_icons['idle'])
        elif state == syncdaemon.STATE_WORKING:
            self.status_icon.set_from_pixbuf(self._status_icons['working'])
        elif state == syncdaemon.STATE_ALERT:
            self.status_icon.set_from_pixbuf(self._status_icons['alert'])
