/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009-2011 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#include "utils.h"

static void get_value(char* __restrict__ pos, uint64_t* __restrict__ value) SA_NONNULL;

extern int kernel_version;
__thread FILE* file_meminfo;
__thread char m_line_buffer[512];

int sa_open_memory() {
	file_meminfo = NULL;

	if (kernel_version < 11)
		return ENOTSUP;

	file_meminfo = fopen("/proc/meminfo", "r");
	if (file_meminfo == NULL)
		return EIO;

	return 0;
}

int sa_close_memory() {
	if (file_meminfo != NULL)
		fclose(file_meminfo);
	return 0;
}

int sa_get_memory(struct sa_memory* dst) {
	if (dst == NULL)
		return EINVAL;

	rewind(file_meminfo);
	if (fflush(file_meminfo) != 0)
		return EIO;

	int aux = 0; // for better performance
	while (fgets(m_line_buffer, sizeof(m_line_buffer), file_meminfo) != NULL) {
		if (aux < 1 && strncmp(m_line_buffer, "MemTotal:", sizeof("MemTotal:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("MemTotal:") - 1], &dst->total);
		else if (aux < 2 && strncmp(m_line_buffer, "MemFree:", sizeof("MemFree:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("MemFree:") - 1], &dst->free);
		else if (aux < 3 && strncmp(m_line_buffer, "Buffers:", sizeof("Buffers:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("Buffers:") - 1], &dst->buffers);
		else if (aux < 4 && strncmp(m_line_buffer, "Cached:", sizeof("Cached:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("Cached:") - 1], &dst->cached);
		else if (aux < 5 && strncmp(m_line_buffer, "SwapCached:", sizeof("SwapCached:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("SwapCached:") - 1], &dst->swap_cached);
		else if (aux < 6 && strncmp(m_line_buffer, "Active:", sizeof("Active:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("Active:") - 1], &dst->active);
		else if (aux < 7 && strncmp(m_line_buffer, "Inactive:", sizeof("Inactive:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("Inactive:") - 1], &dst->inactive);
		else if (aux < 8 && strncmp(m_line_buffer, "SwapTotal:", sizeof("SwapTotal:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("SwapTotal:") - 1], &dst->swap_total);
		else if (aux < 9 && strncmp(m_line_buffer, "SwapFree:", sizeof("SwapFree:") - 1) == 0)
			get_value(&m_line_buffer[sizeof("SwapFree:") - 1], &dst->swap_free);
		else if (aux < 10 && strncmp(m_line_buffer, "Dirty:", sizeof("Dirty:") - 1) == 0) {
			get_value(&m_line_buffer[sizeof("Dirty:") - 1], &dst->dirty);
			break;
		} else
			continue;
		++aux;
	}

	return 0;
}

static void get_value(char* __restrict__ pos, uint64_t* __restrict__ value) {
	while (isblank(*pos))
		++pos;

	*value = strtoull(pos, NULL, 10);
	*value *= 1024;
}
