/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>

#include <libsysactivity/libsysactivity.h>

void print_process_info(struct sa_process* process) {
#ifdef SA_PROCESS_PID
	printf("pid %d, ", (int) process->pid);
#endif
#ifdef SA_PROCESS_UID
	printf("uid %d, ", process->uid);
#endif
#ifdef SA_PROCESS_GID
	printf("gid %d, ", process->gid);
#endif
#ifdef SA_PROCESS_FILENAME
	printf("filename %s, ", process->filename);
#endif
#ifdef SA_PROCESS_CMDLINE
	printf("cmdline %s, ", process->cmdline);
#endif
#ifdef SA_PROCESS_STATE
	printf("state %d, ", process->state);
#endif
#ifdef SA_PROCESS_PARENT_PID
	printf("parent pid %d, ", (int) process->parent_pid);
#endif
#ifdef SA_PROCESS_PGRP
	printf("pgrp %d, ", (int) process->pgrp);
#endif
#ifdef SA_PROCESS_SID
	printf("sid %d, ", (int) process->sid);
#endif
#ifdef SA_PROCESS_TTY
	printf("tty %d, ", (int) process->tty);
#endif
#ifdef SA_PROCESS_USER_TIME
	printf("user_time %"PRIu64", ", process->user_time);
#endif
#ifdef SA_PROCESS_SYS_TIME
	printf("sys_time %"PRIu64", ", process->sys_time);
#endif
#ifdef SA_PROCESS_NICE
	printf("nice %d, ", process->nice);
#endif
#ifdef SA_PROCESS_THREADS
	printf("threads %d, ", process->threads);
#endif
#ifdef SA_PROCESS_START_TIME
	printf("start_time %"PRIu64", ", process->start_time);
#endif
#ifdef SA_PROCESS_VM_SIZE
	printf("vm_size %d, ", process->vm_size);
#endif
#ifdef SA_PROCESS_RSS
	printf("rss %d ", process->rss);
#endif
	printf("\n");
}

void test_process_info(struct sa_process* process) {
#ifdef SA_PROCESS_FILENAME
	size_t len = strnlen(process->filename, SA_PROCESS_FILENAME);
	if (len == 0) {
		printf("\nERROR: The length of the filename of the process %d is zero\n", process->pid);
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_PROCESS_PARENT_PID
	if (process->parent_pid > 1) {
		struct sa_process parent;
		int ret = sa_get_process(process->parent_pid, &parent);
		if (ret != 0) {
			printf("\nERROR: Can't retrieve the parent (%d) of the process %d \n", parent.pid, process->pid);
			exit(EXIT_FAILURE);
		}
	}
#endif
#ifdef SA_PROCESS_THREADS
	if (process->threads == 0) {
		printf("\nERROR: The number of threads of the process %d is zero.", process->threads);
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_PROCESS_NICE
	if (process->nice < -20 || process->nice > 19) {
		printf("\nERROR: Nice value of the process %d is out of range (%d).", process->pid, process->nice);
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_PROCESS_VM_SIZE
	if (process->vm_size == 0) {
		printf("\nERROR: vm size of the process %d is zero (%d).", process->pid);
		exit(EXIT_FAILURE);
	}
#endif
}

void* get_process_info(void* arg) {
	int ret;
#ifdef SA_OPEN_PROCESS
	ret = sa_open_process();
	if (ret != 0) {
		printf("sa_open_process(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

	pid_t pid, written;
	ret = sa_get_processes_ids(&pid, 1, &written);
	if (ret != ENOMEM || written != 1) { // The value returned has to be ENOMEM because i asked for just one pid
		printf("sa_get_processes_ids(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}

	ret = sa_count_processes(&pid);
	if (ret != 0) {
		printf("sa_count_processes(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	pid_t* pids = malloc(pid * sizeof(pid_t));
	ret = sa_get_processes_ids(pids, pid, &written);
	if (ret != 0) {
		printf("sa_get_processes_ids(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	pid_t i;
	printf("pids:");
	for (i = 0; i < written; i++) {
		printf(" %d", (int) pids[i]);
	}
	printf("\n");
	free(pids);

	struct sa_process pro;
	ret = sa_get_process(1, &pro);
	if (ret != 0) {
		printf("sa_get_process(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	print_process_info(&pro);
	printf("\n");

	struct sa_process* processes = NULL;
	for (i = 0; i < 6; i++) {
		ret = sa_count_processes(&pid);
		if (ret != 0 || pid < 2) {
			printf("sa_count_processes(): %s\n", strerror(ret));
			exit(EXIT_FAILURE);
		}
		printf("there are %d processes\n", (int) pid);

		uint16_t j;
		processes = (struct sa_process*) realloc(processes, pid * sizeof(struct sa_process));
		written = 0;
		ret = sa_get_processes(processes, pid, &written);
		if (ret != 0 || written < 2) {
			printf("sa_get_processes(): %s\n", strerror(ret));
			exit(EXIT_FAILURE);
		}
		for (j = 0; j < written; j++) {
			print_process_info(&processes[j]);
		}

		printf("\n");
		sleep(1);
	}
	if (processes != NULL)
		free(processes);
#ifdef SA_CLOSE_PROCESS
	ret = sa_close_process();
	if (ret != 0) {
		printf("sa_close_process(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

#ifdef SA_OPEN_PROCESS
	ret = sa_open_process();
	if (ret != 0) {
		printf("sa_open_process(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_CLOSE_PROCESS
	ret = sa_close_process();
	if (ret != 0) {
		printf("sa_close_process(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
	return NULL;
}

int main() {
	pthread_t thread1;
	struct timespec delay;
	delay.tv_sec = 0;
	delay.tv_nsec = 500000000;

	pthread_create(&thread1, NULL, get_process_info, NULL);
	nanosleep(&delay, NULL);
	get_process_info(NULL);

	return EXIT_SUCCESS;
}
