/* ButtonAreaLayout.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 *
 * Copyright (C) 2005-2007 Universiteit Gent
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 *
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */
package be.ugent.caagt.swirl.dialogs;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;

/**
 * Layout manager for button panels. Behaves in a similar manner to
 * <code>FlowLayout</code>. It lays out all components from left to right,
 * flushed right. The widths of all components will be set to the largest
 * preferred size width.
 */
public class ButtonAreaLayout implements LayoutManager {
    
    //
    private static final int HORIZONTAL_GAP = 5;
    
    
    // implements LayoutManager
    public void addLayoutComponent(String string, Component comp) {
        // empty
    }
    
    // implements LayoutManager
    public void removeLayoutComponent(Component c) {
        // empty
    }
    
    // implements LayoutManager
    public void layoutContainer(Container container) {
        Component[] components = container.getComponents();
        int nr = components.length;
        if (nr == 0)
            return ;
        
        // determine width of each component
        int width = 0;
        for (Component component : components)
            width  = Math.max(width, component.getPreferredSize().width);
        
        // height of inside border
        Insets insets = container.getInsets();
        int height = container.getHeight() - insets.top - insets.bottom;
        
        
        // layout components from left to right
        int xoffs = container.getWidth() + HORIZONTAL_GAP - insets.right - nr*(width + HORIZONTAL_GAP);
        for (Component component : components) {
            int h = component.getPreferredSize().height;
            int y = (height - h) / 2 + insets.top;
            component.setBounds(xoffs, y, width, h);
            xoffs += width + HORIZONTAL_GAP;
        }
    }
    
    //
    public Dimension minimumLayoutSize(Container container) {
        int height = 0;
        int width = 0;
        for (Component component : container.getComponents()) {
            Dimension dimension = component.getPreferredSize();
            height = Math.max(height, dimension.height);
            width  = Math.max(width, dimension.width);
        }
        Insets insets = container.getInsets();
        int nr = container.getComponentCount();
        return new Dimension(
                insets.left + insets.right + nr * (width + HORIZONTAL_GAP) - HORIZONTAL_GAP,
                insets.top + insets.bottom + height
                );
    }
    
    /**
     * The preferred layout size is the same as the minimum layout size.
     */
    public Dimension preferredLayoutSize(Container c) {
        return minimumLayoutSize(c);
    }
}
