/* ButtonFactory.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */
package be.ugent.caagt.swirl.actions;

import java.util.ResourceBundle;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JRadioButton;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JToggleButton;

/**
 * Provides various methods to create buttons from internationalized
 * descriptions.<p>
 * All class methods given here take a resource bundle <code>bundle</code>
 * and a key string <code>key</code>
 * as arguments. In each case, the corresponding resource 
 * string <code>bundle.getString(key)</code> is used as a description 
 * string for the button, as defined in {@link Description}.
 */
public final class ButtonFactory {

    /**
     * Make sure clients do not instantiate this class.
     */
    private ButtonFactory () {}

    /**
     * Create a new {@link JMenu} with a description taken from the given
     * resource bundle.
     */
    public static JMenu createJMenu (ResourceBundle bundle, String key) {
        JMenu result = new JMenu();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JMenuItem} with a description taken from the given
     * resource bundle. Users are encouraged to build menu items from actions
     * instead.
     */
    public static JMenuItem createJMenuItem (ResourceBundle bundle, String key) {
        JMenuItem result = new JMenuItem();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JRadioButtonMenuItem} with a description taken from
     * the given resource bundle. Users are encouraged to build menu items
     * from actions instead.
     */
    public static JRadioButtonMenuItem createJRadioButtonMenuItem (
        ResourceBundle bundle, String key) {
        JRadioButtonMenuItem result = new JRadioButtonMenuItem();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JCheckBoxMenuItem} with a description taken from the
     * given resource bundle. Users are encouraged to build menu items from
     * actions instead.
     */
    public static JCheckBoxMenuItem createJCheckBoxMenuItem (
        ResourceBundle bundle, String key) {
        JCheckBoxMenuItem result = new JCheckBoxMenuItem();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JButton} with a description taken from the given
     * resource bundle. Users are encouraged to build menu items from actions
     * instead.
     */
    public static JButton createJButton (ResourceBundle bundle, String key) {
        JButton result = new JButton();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JRadioButton} with a description taken from the
     * given resource bundle. Users are encouraged to build menu items from
     * actions instead.
     */
    public static JRadioButton createJRadioButton (ResourceBundle bundle,
        String key) {
        JRadioButton result = new JRadioButton();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JToggleButton} with a description taken from the
     * given resource bundle. Users are encouraged to build menu items from
     * actions instead.
     */
    public static JToggleButton createJToggleButton (ResourceBundle bundle,
        String key) {
        JToggleButton result = new JToggleButton();
        new Description(bundle.getString (key)).init (result);
        return result;
    }

    /**
     * Create a new {@link JCheckBox} with a description taken from the given
     * resource bundle. Users are encouraged to build menu items from actions
     * instead.
     */
    public static JCheckBox createJCheckBox (ResourceBundle bundle, String key) {
        JCheckBox result = new JCheckBox();
        new Description(bundle.getString (key)).init (result);
        return result;
    }
}
