/* SelectOnFocusGained.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl;

import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import javax.swing.text.JTextComponent;

/**
 * Focus listener which automatically selects the contents of a text component when it
 * gains focus. This is a singleton class without a public constructor. 
 * The shared singleton instance of this class should be used through the
 * class method {@link #register}.<p>
 * As an example, the following code creates a text field 
 * {@code textField} and ensures that the contents of the field
 * will be selected automatically whenever we enter the field (e.g., by tabbing
 * to it or by moving the focus to it programmatically).
 * <pre>
 *    JTextField textField = new JTextField (10);
 *    SelectOnFocusGained.register (textField);
 * </pre>
 */
public class SelectOnFocusGained extends  FocusAdapter {
    
    // cannot be instantiated by client
    private SelectOnFocusGained () {
    }
    
    //
    private static final SelectOnFocusGained SHARED = new SelectOnFocusGained ();
    
    /**
     * Registers a shared focus listener of this type with the given component.
     * The effect is that the the contents of the component will be selected
     * automatically whenever the field gains the focus.
     */
    public static void register (JTextComponent component) {
        component.addFocusListener (SHARED);
    }
    
    /**
     * Unregisters the shared focus listener with the given component.
     */
    public static void unregister (JTextComponent component) {
        component.removeFocusListener (SHARED);
    }
    
    @Override public void focusGained (FocusEvent e) {
        if (! e.isTemporary ()) {
            JTextComponent field = (JTextComponent)e.getSource ();
            field.setCaretPosition (field.getText ().length ());
            field.moveCaretPosition (0);
        }
    }
}

