/* debug.hh - debugging macros
 * Copyright 2003 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_DEBUG_HH
#define SHEVEK_DEBUG_HH

#include <iostream>
#include <iomanip>
#include <string>
#include <ctype.h>

#ifdef NDEBUG

#define startfunc
#define dbg(x)

#else // !defined (NDEBUG)

namespace shevek
{
  extern bool _debug_dbg, _debug_startfunc;
  /// Make a hex-dump of data and send it to target.
  /** On the left, a hexadecimal representation of the bytes in data is printed.
   *  On the right data itself, with non-printable characters replaced by def.
   */
  void dump (std::string const &data, std::ostream &target, char def = '.');
}

/// Print a message about the function that is being executed (if enabled at run-time).
/** If SHEVEK_DEBUG contains %startfunc, this will print a message when
 *  the function is entered.  For this to happen, it must be written
 *  at the start of a function, followed by a semicolon.
 */
#define startfunc \
do { if (shevek::_debug_startfunc) std::cerr << "Debug: entering " \
  << __PRETTY_FUNCTION__ << '\n'; } while (0)

/// Print a debugging message (if enabled at run-time).
/** If SHEVEK_DEBUG contains %dbg, this will print the file and line this
 *  occurs in, followed by the message x.
 */
#define dbg(x) do { if (shevek::_debug_dbg) std::cerr << __FILE__ << ':' \
	<< __LINE__ << '(' << __FUNCTION__ << "): " << x << '\n'; } while (0)

#endif // !defined (NDEBUG)

#endif // defined (SHEVEK_DEBUG_HH)
