/*
 *  Copyright (c) 2004-2012 Gert Wollny <gw.fossdev@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  As an exception to this license, "NEC C&C Research Labs" may use
 *  this software under the terms of the GNU Lesser General Public License as
 *  published by the Free Software Foundation.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <mialm/vector3d.h>

#include <limits.h>
#include <string.h>
#include <math.h>

#ifndef NDEBUG
static int n_vectors = 0; 
void print_vectors_left(void)
{
	if (n_vectors)
		g_warning("left %d vectors behind", n_vectors);
}
#endif

#define MAXFLOAT FLT_MAX

static GObject *__mia_vector3d_constructor (GType type,
					    guint n_construct_properties,
					    GObjectConstructParam *
					    construct_properties);
static void
__mia_vector3d_class_init (gpointer g_class, gpointer g_class_data);

static void
__mia_vector3d_instance_init (GTypeInstance * instance, gpointer g_class);


GType
mia_vector3d_get_type (void)
{
	static GType type = 0;
	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MiaVector3dClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			__mia_vector3d_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (MiaVector3d),
			0,	/* n_preallocs */
			__mia_vector3d_instance_init,/* instance_init */
			NULL
		};
		type = g_type_register_static (G_TYPE_OBJECT,
					       "MiaVector3dType", &info, 0);
	}
	return type;
}


/**
 * mia_vector3d_new:
 * @x: component of new vector 
 * @y: component of new vector 
 * @z: component of new vector 
 * @returns: a newly created MiaVector3d initialized with the given components
 * 
 */
MiaVector3d *
mia_vector3d_new (gfloat x, gfloat y, gfloat z)
{
	MiaVector3d *obj =
		(MiaVector3d *) g_object_new (MIA_TYPE_VECTOR3D, NULL);
	/* add your code here */
	obj->x = x;
	obj->y = y;
	obj->z = z;
	return obj;
}

/**
 * mia_vector3d_dup:
 * @orig: original Mia3DVector 
 * @returns: a new MiaVector3d instance resembling @orig
 * 
 */

MiaVector3d *
mia_vector3d_dup (const MiaVector3d * orig)
{
	MiaVector3d *obj =
		(MiaVector3d *) g_object_new (MIA_TYPE_VECTOR3D, NULL);
	memcpy (&obj->x, &orig->x, 3 * sizeof (gfloat));
	return obj;
}

/**
 * mia_vector3d_dot:
 * @a: 
 * @b: 
 * @returns: the dot product of @a and @b
 * 
 */

gfloat
mia_vector3d_dot (const MiaVector3d * a, const MiaVector3d * b)
{
	g_assert (MIA_IS_VECTOR3D (a));
	g_assert (MIA_IS_VECTOR3D (b));
	return a->x * b->x + a->y * b->y + a->z * b->z;

}

static inline gfloat
sqr (gfloat x)
{
	return x * x;
}

/**
 * mia_vector3d_dot:
 * @self: 
 * @returns: the norm of @self
 * 
 */

gfloat
mia_vector3d_get_norm (MiaVector3d * self)
{
	g_assert (MIA_IS_VECTOR3D (self));
	return sqrt (sqr (self->x) + sqr (self->y) + sqr (self->z));
}

/**
 * mia_vector3d_scale:
 * @self: MiaVector3d to be scaled
 * @f: 
 * @returns: (scaled) @self for convinience
 * Scales @self by factor @f. 
 */


MiaVector3d *
mia_vector3d_scale (MiaVector3d * self, gfloat f)
{
	g_assert (MIA_IS_VECTOR3D (self));
	self->x *= f;
	self->y *= f;
	self->z *= f;
	return self;
}

/**
 * mia_vector3d_normalize:
 * @self: MiaVector3d to be normalized
 * @returns: @self (for convinience)
 */
MiaVector3d *
mia_vector3d_normalize (MiaVector3d * self)
{
	gfloat norm = mia_vector3d_get_norm (self);
	if (norm > 0)
		mia_vector3d_scale (self, 1.0f / norm);
	return self;
}

/**
 * mia_vector3d_copy:
 * @dest: Existing MiaVector3d instance 
 * @src: Existing MiaVector3d instance 
 * @returns: @dest (for convinience)
 * Copies the contens of @src to @dest 
 */
MiaVector3d *
mia_vector3d_copy (MiaVector3d * dest, const MiaVector3d * src)
{
	g_assert (MIA_IS_VECTOR3D (dest));
	g_assert (MIA_IS_VECTOR3D (src));
	memcpy (&dest->x, &src->x, 3 * sizeof (gfloat));
	return dest;
}

/**
 * mia_vector3d_addup:
 * @self: This Mia3DVector will be changed
 * @other: 
 * @returns: @self (for convinience)
 * Adds @other to @self componentwise
 */
MiaVector3d *
mia_vector3d_addup (MiaVector3d * self, const MiaVector3d * other)
{
	g_assert (MIA_IS_VECTOR3D (self));
	g_assert (MIA_IS_VECTOR3D (other));
	self->x += other->x;
	self->y += other->y;
	self->z += other->z;
	return self;
}

/**
 * mia_vector3d_set:
 * @self: This Mia3DVector will be changed
 * @x: 
 * @y: 
 * @z:   
 * @returns: @self (for convinience)
 * Sets the components of @self to the given values
 */
MiaVector3d *
mia_vector3d_set (MiaVector3d * self, gfloat x, gfloat y, gfloat z)
{
	MIA_IS_VECTOR3D (self);
	self->x = x;
	self->y = y;
	self->z = z;
	return self;
}


/**
 * mia_vector3d_add:
 * @self:   input Mia3DVector 
 * @other:  input Mia3DVector  
 * @result: output Mia3DVector 
 * @returns: if @result != NULL then @result, else a newly created Mia3DVector
 * Adds @self and @other component wise and stores the result in @result. 
 * if result = NULL, then a new Mia3DVector instance is created and returned. 
 */

MiaVector3d *
mia_vector3d_add (const MiaVector3d * self, const MiaVector3d * other,
		  MiaVector3d * result)
{
	g_return_val_if_fail (MIA_IS_VECTOR3D (self), NULL);
	g_return_val_if_fail (MIA_IS_VECTOR3D (other), NULL);

	if (!result)
		result = mia_vector3d_dup (self);
	else
	{
		g_assert (MIA_IS_VECTOR3D (result));
		mia_vector3d_copy (result, self);
	}

	mia_vector3d_addup (result, other);
	return result;
}

/**
 * mia_vector3d_equal:
 * @a:  input Mia3DVector 
 * @b:  input Mia3DVector  
 * @returns: %TRUE if @a and @b are equal (componentwise) else %FALSE
 */

gboolean
mia_vector3d_equal (const MiaVector3d * a, const MiaVector3d * b)
{
	g_return_val_if_fail (MIA_IS_VECTOR3D (a), FALSE);
	g_return_val_if_fail (MIA_IS_VECTOR3D (b), FALSE);
	return a->x == b->x && a->y == b->y && a->z == b->z;
}

typedef struct __MiaVector3dPropertyInit _MiaVector3dPropertyInit;
struct __MiaVector3dPropertyInit
{
	gchar *name;
	gchar *nick;
	gchar *descr;
	gfloat min, max;
	gfloat def;
	GParamFlags flags;
	gint id;
};


static _MiaVector3dPropertyInit  __vector3d_properties[mia_vector3d_prop_counter] = {
	{"x", "x", "value_x", -MAXFLOAT, MAXFLOAT, 0.0f,
	 G_PARAM_READWRITE, mia_vector3d_prop_x},
	{"y", "y", "value y", -MAXFLOAT, MAXFLOAT, 0.0f,
	 G_PARAM_READWRITE, mia_vector3d_prop_y},
	{"z", "z", "value z", -MAXFLOAT, MAXFLOAT, 0.0f,
	 G_PARAM_READWRITE, mia_vector3d_prop_z}
};


/**
 * mia_vector3d_xmlio_read:
 * @state: holst the current state of the XML SAX parser
 * @property:  the name of the property, where the MiaVector3d should be stored
 */

void
mia_vector3d_xmlio_read(ParserState * state, const gchar *property)
{
	gfloat x,y,z;
	MiaVector3d *v;
	GValue value = G_VALUE_INIT;
	GObject *obj = G_OBJECT (state->pss->parent->data);
	
	g_assert (obj);
	
	g_value_init (&value, G_TYPE_POINTER);

	sscanf (state->pss->ch->str, "%g %g %g", &x, &y, &z);

	v = mia_vector3d_new (x, y, z);
	g_value_set_pointer (&value, (gpointer) v);
	g_object_set_property(obj, property, &value);
}

/**
 * mia_vector3d_xmlio_write:
 * @root: 
 * @ns: xml name space
 * @tag: 
 * @v: 
 * @returns: %TRUE if vector was NULL or sucessfully added to the xml node %FALSE otherwise
 * The function adds @v as tag @tag to the given xml @root node
 */

gboolean
mia_vector3d_xmlio_write (xmlNodePtr root, xmlNsPtr ns, const gchar * tag,
		      const MiaVector3d * v)
{
	gchar text[1024];
	if (!v)
		return TRUE;

	snprintf (text, 1023, "%g %g %g", v->x, v->y, v->z);
	return (xmlNewTextChild (root, ns, (const xmlChar *)tag, (const xmlChar *)text) != NULL);
}


static void
__mia_vector3d_set_property (GObject * object,
			     guint property_id,
			     const GValue * value, GParamSpec * pspec)
{
	MiaVector3d *self = (MiaVector3d *) object;

	switch (property_id)
	{

	case mia_vector3d_prop_x:
		self->x = g_value_get_float (value);
		break;
	case mia_vector3d_prop_y:
		self->y = g_value_get_float (value);
		break;
	case mia_vector3d_prop_z:
		self->z = g_value_get_float (value);
		break;

	default:
		/* We don't have any other property... */
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id,
						   pspec);
	}
}

static void
__mia_vector3d_get_property (GObject * object,
			     guint property_id,
			     GValue * value, GParamSpec * pspec)
{
	MiaVector3d *self = (MiaVector3d *) object;

	switch (property_id)
	{
	case mia_vector3d_prop_x:
		g_value_set_float (value, self->x);
		break;
	case mia_vector3d_prop_y:
		g_value_set_float (value, self->y);
		break;
	case mia_vector3d_prop_z:
		g_value_set_float (value, self->z);
		break;
	default:
		/* We don't have any other property... */
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id,
						   pspec);
	}
}

static void
__mia_vector3d_instance_init (GTypeInstance * UNUSED(instance), gpointer UNUSED(g_class))
{
	//      MiaVector3d *self = (MiaVector3d *)instance;
#ifndef NDEBUG	
	n_vectors++;
#endif	
}

static GObject *
__mia_vector3d_constructor (GType type,
			    guint n_construct_properties,
			    GObjectConstructParam * construct_properties)
{
	GObject *obj;
	{
		/* Invoke parent constructor. */
		MiaVector3dClass *klass;
		GObjectClass *parent_class;
		klass = MIA_VECTOR3D_CLASS (g_type_class_peek
					    (MIA_TYPE_VECTOR3D));
		parent_class =
			G_OBJECT_CLASS (g_type_class_peek_parent (klass));
		obj = parent_class->constructor (type, n_construct_properties,
						 construct_properties);
	}
	/* add your code here */

	return obj;
}

static void
__mia_vector3d_dispose (GObject * UNUSED(obj))
{
	/* add your code herenothing to dispose */

}

static void
__mia_vector3d_finalize (GObject * UNUSED(obj))
{
	/* add your destruction code here */
#ifndef NDEBUG	
	n_vectors--;
#endif	
}



static void
__mia_vector3d_class_init (gpointer g_class, gpointer UNUSED(g_class_data))
{
	int i;
	_MiaVector3dPropertyInit *lp;
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	/*MiaLandmarkClass *klass = MIA_LANDMARK_CLASS (g_class); */

	gobject_class->set_property = __mia_vector3d_set_property;
	gobject_class->get_property = __mia_vector3d_get_property;
	gobject_class->dispose = __mia_vector3d_dispose;
	gobject_class->finalize = __mia_vector3d_finalize;
	gobject_class->constructor = __mia_vector3d_constructor;

	lp = __vector3d_properties;
	/* add your code here (e.g. define properties) */
	for (i = 0; i < mia_vector3d_prop_counter - 1; i++, ++lp)
	{
		GParamSpec *pspec = g_param_spec_float (lp->name, lp->nick,
							lp->descr,
							lp->min, lp->max,
							lp->def,
							lp->flags);
		g_object_class_install_property (gobject_class, lp->id,
						 pspec);
	}
}
