/*
 * Copyright (C) 2012  Christian Mollekopf <mollekopf@kolabsys.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "commonconversion.h"
#include <kolabformat/errorhandler.h>

#include <KDE/KSystemTimeZones>
#include <KDE/KDebug>

namespace Kolab {
    namespace Conversion {
        
KDateTime toDate(const Kolab::cDateTime &dt)
{
    KDateTime date;
    if (!dt.isValid()) { //We rely on this codepath, so it's not an error
        //         qDebug() << "invalid datetime converted";
        return KDateTime();
    }
    if (dt.isDateOnly()) { //Date only
        date.setDateOnly(true);
        date.setDate(QDate(dt.year(), dt.month(), dt.day()));
        date.setTimeSpec(KDateTime::Spec(KDateTime::ClockTime));
    } else {
        date.setDate(QDate(dt.year(), dt.month(), dt.day()));
        date.setTime(QTime(dt.hour(), dt.minute(), dt.second()));
        if (dt.isUTC()) { //UTC
            date.setTimeSpec(KDateTime::Spec(KDateTime::UTC));
        } else if (!dt.timezone().empty()) { //Timezone
            KTimeZone tz = KSystemTimeZones::zone(QString::fromStdString(dt.timezone())); //Needs ktimezoned (timezone daemon running) http://api.kde.org/4.x-api/kdelibs-apidocs/kdecore/html/classKSystemTimeZones.html
            if (!tz.isValid()) {
                Error() << "timezone not found" << QString::fromStdString(dt.timezone());
                if (!KSystemTimeZones::isTimeZoneDaemonAvailable()) {
                    Error() << "ktimezoned is not available and required for timezone interpretation";
                }
                tz = KTimeZone::utc(); //Don't crash
            }
            date.setTimeSpec(KDateTime::Spec(tz));
        } else { //Floating
            date.setTimeSpec(KDateTime::Spec(KDateTime::ClockTime));
        }
    }
    Q_ASSERT(date.timeSpec().isValid());
    Q_ASSERT(date.isValid());
    return date;
}

cDateTime fromDate(const KDateTime &dt)
{
    if (!dt.isValid()) {
        //         qDebug() << "invalid datetime converted";
        return cDateTime();
    }
    cDateTime date;
    if (dt.isDateOnly()) { //Date only
        const QDate &d = dt.date();
        date.setDate(d.year(), d.month(), d.day());
    } else {
        const QDate &d = dt.date();
        date.setDate(d.year(), d.month(), d.day());
        const QTime &t = dt.time();
        date.setTime(t.hour(), t.minute(), t.second());
        if (dt.timeType() == KDateTime::UTC) { //UTC
            date.setUTC(true);
        } else if (dt.timeType() == KDateTime::TimeZone) { //Timezone
            //TODO handle local timezone?
            date.setTimezone(dt.timeZone().name().toStdString()); //FIXME use system independent name according to spec
        } else if (dt.timeType() != KDateTime::ClockTime) {
            Error() << "invalid timespec, assuming floating time" << dt.timeType();
            return date;
        }
    }
    Q_ASSERT(date.isValid());
    return date;
}

QStringList toStringList(const std::vector<std::string> &l)
{
    QStringList list;
    foreach(const std::string &s, l) {
        list.append(QString::fromStdString(s));
    }
    return list;
}

std::vector<std::string> fromStringList(const QStringList &l)
{
    std::vector<std::string> list;
    foreach(const QString &s, l) {
        list.push_back(s.toStdString());
    }
    return list;
}

    }
}