/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDXMLFILE_H
#define ISDXMLFILE_H

/**
 * This class represents the XML file in the instrudeo file format.
 * The entire file is kept in memory (using the DOM structure)
 * until it is asked to write itself out.
 *
 * @author Bram Biesbrouck <b@beligum.org>
*/

#include <list>
#include <set>

#include <glibmm/ustring.h>
#include <libxml++/libxml++.h>

#include <libinstrudeo/isdrecordingmetainfo.h>
#include <libinstrudeo/isdobject.h>

using namespace std;

#undef LOG_HEADER
#define LOG_HEADER "Error while accessing XML meta file: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----XML-CONSTANTS-----
#define ROOT_NAME "instrudeometa"
#define ROOT_VERSION_NAME "version"
#define TITLE_NAME "title"
#define QUESTION_NAME "question"
#define DESCRIPTION_NAME "description"
#define PLATFORM_NAME "platform"
#define PLATFORM_ID_NAME "id"
#define PLATFORM_TITLE_NAME "title"
#define CATEGORY_NAME "category"
#define CATEGORY_ID_NAME "id"
#define CATEGORY_TITLE_NAME "title"
#define COMMENTLIST_NAME "commentlist"
#define COMMENTLIST_LANGUAGE_NAME "language"
#define COMMENT_NAME "comment"
#define COMMENT_ID_NAME "id"
#define COMMENT_TYPE_NAME "type"
#define COMMENT_POSITION_X_NAME "x"
#define COMMENT_POSITION_Y_NAME "y"
#define COMMENT_WIDTH_NAME "width"
#define COMMENT_HEIGHT_NAME "height"
#define COMMENT_MIRRORMODE_NAME "mirrormode"
#define COMMENT_STARTTIME_NAME "starttime"
#define COMMENT_DURATION_NAME "duration"
#define COMMENT_COLOR_R_NAME "color_r"
#define COMMENT_COLOR_G_NAME "color_g"
#define COMMENT_COLOR_B_NAME "color_b"
#define COMMENT_COLOR_A_NAME "color_a"
#define COMMENT_TEXTCOLOR_R_NAME "textcolor_r"
#define COMMENT_TEXTCOLOR_G_NAME "textcolor_g"
#define COMMENT_TEXTCOLOR_B_NAME "textcolor_b"
#define COMMENT_TEXTCOLOR_A_NAME "textcolor_a"

class ISDWSPlatform;
class ISDWSCategory;
class ISDCommentbox;
class ISDVideoCanvas;

class ISDXmlFile : public ISDObject
{

 public:
    //-----CONSTANTS-----
    
    //-----CONSTRUCTORS-----
    /**
     * Standard constructor that creates an empty XML-structure.
     */
    ISDXmlFile();

    /**
     * Constructor that opens and parses the specified XML file.
     * The videoCanvas is needed as parent for the (pre-)created commentBoxes.
     *
     * @param videoCanvas The videoCanvas.
     * @param fileName The XML-fileName.
     */
    ISDXmlFile(string fileName, ISDVideoCanvas* videoCanvas);
    
    ~ISDXmlFile();

    //-----STATIC METHODS-----
    static Glib::ustring getVersionNumberString();

    //-----METHODS-----
    /**
     * Writes the current data in this class to the file.
     * OutputFileName will be filled with the tempFile name used if createNew was True.
     *
     * @param outputFileName The name of the file to write the data to.
     *                       Will be filled with the path of the newly created (temporary) file if createNew was true.
     * @param createNew Indicates if a new file must be created (true) or the existing one must be used (false)
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode writeToFile(string& outputFileName, bool createNew=false);

    //-----GETTER/SETTER METHODS-----
    /**
     * Returns the meta-info-object of this recording that will be written
     * to disk in the writeToFile() method.
     *
     * @return The object or NULL if an error occurred.
     */
    ISDRecordingMetaInfo* getMetaInfo();

    /**
     * Adds a comment to the recording.
     * This merely adds the recording to the list of comments
     * that are written to disk with the writeToFile() method.
     *
     * @param commentbox The new commentbox.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode addComment(ISDCommentbox* commentbox);

    /**
     * Removes the comment from the recording, deletes the object,
     * and sets the pointer to NULL. The changes are written to disk
     * in the writeToFile() method.
     *
     * @param commentbox The comment to remove.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode removeComment(ISDCommentbox*& commentbox);

    /**
     * Searches through the xml file for all comments that must be displayed at
     * the specified position and creates them with the specified parent.
     * If the specified language wasn't found, nothing happens.
     * Note: the comments are allocated in this method, so if you don't need them
     *       anymore, you are responsible for the deletion.
     * Note2: The comments are merely created with the specified parent; this implies you
     *        are responsible for handling duplicates. The naive way is to delete all the
     *        child-commentboxes of the videoCanvas before calling this method.
     * Note3: If NULL is specified for the language, catchall-comments will be disabled too,
     *        if you want to only enable catchall's, pass ISD_COMMENTBOX_LANGUAGE_CATCHALL as the lang.
     *        Passing ISD_COMMENTBOX_LANGUAGE_CATCHALL __will_not_enable_all_comments, but just catchalls.
     *
     * @param videoCanvas A reference to the videoCanvas.
     * @param position The position in the videoStream.
     * @param lang The language(-list) to display, ISD_COMMENTBOX_LANGUAGE_CATCHALL or NULL to disable comments.
     * @param noChange Is set to true if nothing changed.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode updateCommentsForPosition(ISDVideoCanvas* videoCanvas, int position, Glib::ustring* lang, bool& noChange);

    /**
     * Returns all comment-objects in this recording.
     *
     * @return The list of all comments.
     */
    list<ISDCommentbox*>& getAllCommentObjects();

    /**
     * Searches through all comments and returns all distinct languages found.
     *
     * Note: this method doesn't report the ISD_COMMENTBOX_LANGUAGE_CATCHALL language.
     *
     * @param list The list of all language codes.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode getCommentLanguages(set<Glib::ustring>& langList);

    /**
     * Generates a comment-ID that is unique across the currently created comment-ID's
     *
     * @return The unique ID or -1 if an error occurred.
     */
    int getUniqueCommentId();

    /**
     * The following two methods can be used to keep track of the commentbox-changes
     * during updates.
     *
     * @return A list of all commentboxes that were added/removed in the last update (empty if none).
     */
    list<ISDCommentbox*>& getAddedInLastUpdate();
    list<ISDCommentbox*>& getRemovedInLastUpdate();
    
 protected:
    
    //-----METHODS-----
    /**
     * Inits a newly created XML DOM Document with the necessary elements and data
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode initNewDocument();

    /**
     * Searches the comment-set for a comment with the specified ID.
     * If none was found, NULL is returned.
     *
     * @param id The id to search.
     * @return The found commentbox or NULL of none was found.
     */
    ISDCommentbox* getComment(int id);

    //-----XML ACCESS METHODS-----
    /**
     * Creates a new language-list element with the specified language.
     * If the list is already present, nothing is done.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param language The language for the new list.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode addNewLanguageList(xmlpp::Document* xmlDocument, Glib::ustring language);

    /**
     * The value of the element with elementName is returned or an empty string 
     * if no such element exists in the current document.
     * Only direct children of the root node are searched.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param elementName The name of the element to search for.
     * @return The value of the element or an empty string if nothing was found.
     */
    Glib::ustring getStringValue(xmlpp::Document* xmlDocument, const string elementName);

    /**
     * Fills the list of the metaInfo-object with the platforms, found in the XML file.
     * If none were found, no platforms are set.
     * If an error occurred, the error flag is set.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param metaInfo The metaInfo object.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode readPlatformList(xmlpp::Document* xmlDocument, ISDRecordingMetaInfo& metaInfo);

    /**
     * Fills the list of the metaInfo-object with the categories, found in the XML file.
     * If none were found, no categories are set.
     * If an error occurred, the error flag is set.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param metaInfo The metaInfo object.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode readCategoryList(xmlpp::Document* xmlDocument, ISDRecordingMetaInfo& metaInfo);

    /**
     * Allocates a new platform and fills the fields with the fields of the platform-element.
     * If something went wrong, the value of commentbox is undefined.
     *
     * @param elem The element to substract the data from.
     * @param cat The returned platform.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode fillPlatformFromElement(xmlpp::Element* elem, ISDWSPlatform*& cat);

    /**
     * Allocates a new category and fills the fields with the fields of the category-element.
     * If something went wrong, the value of commentbox is undefined.
     *
     * @param elem The element to substract the data from.
     * @param cat The returned category.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode fillCategoryFromElement(xmlpp::Element* elem, ISDWSCategory*& cat);

    /**
     * The value of the element with name elementName is changed to value or
     * a new element is inserted if none existed by that name.
     * Only direct children of the root node are searched and the root will be the
     * parent if an element gets inserted.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param elementName The name of the element to search for.
     * @param value The value to insert.
     * @return A pointer to the element where the value was inserted or NULL on error.
     */
    xmlpp::Element* setStringValue(xmlpp::Document* xmlDocument, const string elementName, Glib::ustring value);

    /**
     * Adds an empty element as a parent child with the name elmentName
     *
     * @param xmlDocument The XML DOM Document object.
     * @param elementNam The name of the new element.
     * @return A pointer to the new element or NULL on error.
     */
    xmlpp::Element* addEmptyElement(xmlpp::Document* xmlDocument, const string elementName);

    /**
     * Finds the language list with the specified language.
     * Returns NULL if no such list was found.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param lang The language to search for.
     * @return The languageList-element that was found or NULL if none was found by this lang.
     */
    xmlpp::Element* searchLanguageList(xmlpp::Document* xmlDocument, Glib::ustring lang);

    /**
     * Finds the comment with the specified id and language.
     * Returns NULL if no such comment was found.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param id The id to search for.
     * @param lang The language of the comment.
     * @return The commentbox-element that was found or NULL if none was found by this id.
     */
    xmlpp::Element* searchComment(xmlpp::Document* xmlDocument, int id, Glib::ustring lang);

    /**
     * Saves the values of the commentbox to the XML document.
     * If elem is NULL, a new comment is inserted, otherwise, the values of that element
     * are updated. If the comment holds a language that isn't present as a language-list
     * in the XML file, a new language-list is created.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param elem The element to update or NULL to insert a new element.
     * @param commentbox The commentbox containing the new values.
     * @returns The element that was updated (==elem) or the new element or NULL if an error occurred.
     */
    xmlpp::Element* saveComment(xmlpp::Document* xmlDocument, xmlpp::Element* elem, ISDCommentbox* commentbox);

    /**
     * Fills the elem with the values of commentbox, no matter what values are currently present in elem;
     * thus, overwriting them all.
     *
     * @param elem The element to update;
     * @param commentbox The commentbox containing the new values.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode fillElementFromComment(xmlpp::Element* elem, ISDCommentbox* commentbox);

    /**
     * Allocates a new commentbox and fills the fields with the fields of the commentbox-element.
     * If something went wrong, the value of commentbox is undefined.
     *
     * @param parent A reference to the videoCanvas, needed for the constructor of the commentBox.
     *               I didn't check it, but I assume this can be NULL, use with caution.
     * @param elem The element to substract the data from.
     * @param commentbox The returned commentbox.
     * @param lang The language of the comment.
     * @return Returns a code that indicates success or failure.
     */
    ISDObject::ISDErrorCode fillCommentFromElement(ISDVideoCanvas* parent, xmlpp::Element* elem, ISDCommentbox*& commentbox, Glib::ustring& lang);

    /**
     * List all the comments in the xmlFile, no matter what language
     *
     * @param xmlDocument The XML DOM Document object.
     * @return The set with the comments, may be empty.
     */
    xmlpp::NodeSet getAllComments(xmlpp::Document* xmlDocument);

    /**
     * Searches for a comment with the specified ID, no matter the language.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param id The ID to search for.
     * @return true if a comment with that ID exists, false otherwise.
     */
    bool commentExists(xmlpp::Document* xmlDocument, int id);

    /**
     * Looks up the language of the specified commentbox-element.
     *
     * @param xmlDocument The XML DOM Document object.
     * @param elem The commentbox-element to search the language for.
     * @return The languagelist element of the specified commentbox-element or NULL if something went wrong.
     */
    xmlpp::Element* getCommentLanguage(xmlpp::Document* xmlDocument, xmlpp::Element* elem);
	
    //-----VARIABLES-----
    /**
     * This variable is used to speed up the updateCommentsForPosition() method
     * significantly. It must be kept synchonized with the xml file through the
     * add and delete methods.
     */
    list<ISDCommentbox*> allComments;
    list<ISDCommentbox*> addedInLastUpdate;
    list<ISDCommentbox*> removedInLastUpdate;
    ISDRecordingMetaInfo metaInfo;
};

#endif
