/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/


#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdrectangle.h>
#include <libinstrudeo/isdseekbackcalculator.h>

//-----CONSTRUCTORS-----
ISDSeekBackCalculator::ISDSeekBackCalculator()
 : ISDObject()
{
}

//-----DESTRUCTOR-----
ISDSeekBackCalculator::~ISDSeekBackCalculator()
{
}

//-----PUBLIC METHODS-----
ISDObject::ISDErrorCode ISDSeekBackCalculator::calcSeekBack(ISDRectangle* rect)
{
    ISDObject::ISDErrorCode retVal;
    currentRect = rect;
    freeRects.clear();
 
    //fill the list with a copy of the source rect, default copy-constructor is good
    ISDRectangle startRect = *rect;
    freeRects.push_back(startRect);

    if ((retVal = doCalcSeekBack()) != ISD_SUCCESS) {
	LOG_WARNING(("Error occurred while calculating seekback rect at time." + rect->frameTime));
	RETURN_ERROR(retVal);
    }

    /*
     * now, currentRect points at the seekback rect,
     */
    rect->setSeekBackRectPos(currentRect->filePos);
    //cout << "rect at time " << rect->frameTime << " has seekback rect at filePos " << currentRect->filePos << endl;

    RETURN_SUCCESS;
}

//-----PRIVATE METHODS-----
ISDObject::ISDErrorCode ISDSeekBackCalculator::doCalcSeekBack()
{
    std::list<ISDRectangle>::iterator iter;

    while (!freeRects.empty()) {
	//check if we reached the first rect
	if (currentRect->getPrevious()==NULL) {
	    /*
	     * The first rect is always the same size as the framebuffer,
	     * thus, overwriting everything and has no seekback rect.
	     */
	    break;
	}
	else {
	    currentRect = currentRect->getPrevious();
	}
	for (iter=freeRects.begin(); iter!=freeRects.end(); iter++){
	    //check for full overlap
	    if (cutOutRect(&(*iter), currentRect)==true) {
		/*
		 * the erase-method returns the rect after the erased one, so we need to
		 * go back one (because of iter++).
		 */
		iter = freeRects.erase(iter);
		iter--;
	    }
	}
    }

    RETURN_SUCCESS;
}
bool ISDSeekBackCalculator::shadowsRect(ISDRectangle* sourceRect, ISDRectangle* targetRect)
{
    return
	sourceRect->x <= targetRect->x &&
	sourceRect->x+sourceRect->width >= targetRect->x+targetRect->width &&
	sourceRect->y <= targetRect->y &&
	sourceRect->y+sourceRect->height >= targetRect->y+targetRect->height;
	
}
bool ISDSeekBackCalculator::cutOutRect(ISDRectangle* rect, ISDRectangle* cutRect)
{
    //some help-variables for convenience
    int rectBottom = rect->y + rect->height;
    int rectRight = rect->x + rect->width;
    int cutRectBottom = cutRect->y + cutRect->height;
    int cutRectRight = cutRect->x + cutRect->width;
    
    /*
     * check for overlap; if the cutout rect doesn't have overlap with the target-rect,
     * there can never be any cutout.
     * Two rects DON'T overlap if any of these is true:
     *
     * 1. The right side of cutRect is to the left (or the same) of the left side of rect
     * 2. The left side of cutRect is to the right (or the same) of the right side of rect
     * 3. The upper side of cutRect is below (or the same) the lower side of rect
     * 4. The lower side of cutRect is above (or the same) the upper side of rect
     *
     * In all other cases, they overlap.
     */
    if (
	!(
	    (cutRectRight <= rect->x) ||
	    (cutRect->x >= rectRight) ||
	    (cutRect->y >= rectBottom) ||
	    (cutRectBottom <= rect->y)
	  )
	)
    {
	/*
	 * Calculate the section (overlap-rect) of the cutRect and the rect.
	 */
	int overTop = cutRect->y < rect->y ? rect->y : cutRect->y;
	int overLeft = cutRect->x < rect->x ? rect->x : cutRect->x;
	int overBottom = cutRectBottom < rectBottom ? cutRectBottom : rectBottom;
	int overRight = cutRectRight < rectRight ? cutRectRight : rectRight;
	
	/*
	 * Calculate the 8 possible rectangles that can be generated by cutting the overlap-rect
	 * out of rect.
	 * Note: the overlap-rect never exceeds the rect's dimension.
	 * If the width or height of the generated rect equals zero, discard it.
	 * Note2: rectLeft == rect->x
	 *        rectTop == rect->y
	 */
	int x, y, w, h;
	
	/*
	 * the first 3 rects have the same height, so if one of them is zero, don't proceed,
	 * but skip to rect number 4
	 */
	h = overTop - rect->y;
	if (h > 0) {
	    y = rect->y;
	    
	    //rect 1 (top left)
	    w = overLeft - rect->x;
	    if (w > 0) {
		x = rect->x;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }
	    
	    //rect 2 (top center)
	    w = overRight - overLeft;
	    if (w > 0) {
		x = overLeft;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }

	    //rect 3 (top right)
	    w = rectRight - overRight;
	    if (w > 0) {
		x = overRight;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }
	}

	/*
	 * The middle two rects
	 */
	h = overBottom - overTop;
	if (h > 0) {
	    y = overTop;
	    
	    //rect 4 (middle left)
	    w = overLeft - rect->x;
	    if (w > 0) {
		x = rect->x;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }
	    
	    //rect 5 (middle right)
	    w = rectRight - overRight;
	    if (w > 0) {
		x = overRight;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }
	}

	/*
	 * the bottom 3 rects
	 */
	h = rectBottom - overBottom;
	if (h > 0) {
	    y = overBottom;
	    
	    //rect 6 (bottom left)
	    w = overLeft - rect->x;
	    if (w > 0) {
		x = rect->x;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }
	    
	    //rect 7 (bottom center)
	    w = overRight - overLeft;
	    if (w > 0) {
		x = overLeft;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }

	    //rect 8 (bottom right)
	    w = rectRight - overRight;
	    if (w > 0) {
		x = overRight;
		freeRects.push_back(ISDRectangle(0, x, y, w, h, 0, 0, 0, NULL, NULL, 0));
	    }
	}
	

	return true;
    }
    return false;
}
