/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <list>

#include <libinstrudeo/isdprogresscallback.h>
#include <libinstrudeo/isdffmpegexporter.h>
#include <libinstrudeo/isdvideocanvas.h>
#include <libinstrudeo/isdvideoproperties.h>
#include <libinstrudeo/isdseekbackcalculator.h>
#include <libinstrudeo/isdrectangle.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdvnclogimporter.h>
#include <libinstrudeo/isdxmlfile.h>
#include <libinstrudeo/isddatafile.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdcommentbox.h>
#include <libinstrudeo/isdrecording.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while accessing Instrudeo recording: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----CONSTRUCTORS-----
ISDRecording::ISDRecording(string fileName)
    : ISDObject(), dataFile(NULL), xmlFile(NULL), videoCanvas(NULL), videoProperties(NULL)
{
    //this initialises the dataFile, xmlFile, videoCanvas and videoProperties variables too
    if (unpackRecordingFile(fileName)!=ISD_SUCCESS) {
	lastError = ISD_FILE_ERROR;
	return;
    }

    //preload the first rectangle
    bool noChange;
    if (dataFile->updateCanvas(videoCanvas, 0, noChange)!=ISD_SUCCESS) {
	LOG_WARNING("Error while preloading the first rectangle.");
	lastError = ISD_INIT_ERROR;
	return;
    }
}
ISDRecording::ISDRecording(ISDImporter& importer, ISDProgressCallback* callbackClass)
    : ISDObject(), dataFile(NULL), xmlFile(NULL), videoCanvas(NULL), videoProperties(NULL)
{
    //create a temp unpack directory
    if (ISDUtils::getInstance()->createTempDir(unpackDirectory)!=ISD_SUCCESS) {
	LOG_WARNING("Failed to create a temp directory while importing a record.");
	return;
    }

    string dataFileName = unpackDirectory + RELATIVE_DATA_FILE_NAME;

    //tell the importer to use the data file in the temp dir for the new binary file
    if (importer.convertTo(dataFileName, callbackClass, false)!=ISD_SUCCESS) {
	lastError = ISD_FILE_ERROR;
	return;
    }
    
    //create the data file
    dataFile = new ISDDataFile(dataFileName, videoProperties);
    if (dataFile==NULL || dataFile->error() ||
	videoProperties==NULL || videoProperties->error()) {
	LOG_WARNING("Error while constructing/parsing the data file.");
	lastError = ISD_FILE_ERROR;
	return;
    }

    //create a new, empty xml file
    xmlFile = new ISDXmlFile();
    if (xmlFile==NULL || xmlFile->error()) {
	LOG_WARNING("Error while constructing/parsing the XMLfile object.");
	lastError = ISD_FILE_ERROR;
	return;
    }

    //create the videoCanvas
    videoCanvas = new ISDVideoCanvas(videoProperties);
    if (videoCanvas==NULL || videoCanvas->error()) {
	LOG_WARNING("Error while constructing the VideoCanvas object.");
	lastError = ISD_INIT_ERROR;
	return;
    }

    //preload the first rectangle
    bool noChange;
    if (dataFile->updateCanvas(videoCanvas, 0, noChange)!=ISD_SUCCESS) {
	LOG_WARNING("Error while preloading the first rectangle.");
	lastError = ISD_INIT_ERROR;
	return;
    }
}

//-----DESTRUCTOR-----
ISDRecording::~ISDRecording()
{
    if (xmlFile!=NULL) {
	delete xmlFile;
	xmlFile = NULL;
    }
    if (dataFile!=NULL) {
	delete dataFile;
	dataFile = NULL;
    }
    if (videoCanvas!=NULL) {
	delete videoCanvas;
	videoCanvas = NULL;
    }
    if (videoProperties!=NULL) {
	delete videoProperties;
	videoProperties = NULL;
    }
    //don't worry if the dir doesn't exist
    deleteUnpackDirectory();
}

//-----PUBLIC METHODS-----
ISDObject::ISDErrorCode ISDRecording::packRecording(string fileName)
{
    /*
     * There are two possibilities:
     * 
     * 1. A foreign file was imported
     *    The up-to-date dataFile is in the (temp) unpack directory and has the right filename
     *    The up-to-date xmlFile is in memory only.
     *
     * 2. An instrudeo file was opened
     *    The up-to-date dataFile is in the unpack directory and has the right filename
     *    The up-to-date xmlFile is in memory, the original one in the unpack directory.
     *
     * In both cases, we can use the XML file in memory and the data file on disk to pack the recording.
     */
    
    if (xmlFile!=NULL && dataFile!=NULL) {
	//create the XML file in the unpackDir
	//note: it doesn't matter the original unpacked XML file gets overwritten,
	// we keep that info in memory anyway.
	string xmlFileName = unpackDirectory + RELATIVE_META_FILE_NAME;
 
	//delete any existing file, ignoring errors
	REMOVE_FILE(xmlFileName);
	if (xmlFile->writeToFile(xmlFileName, false)!=ISD_SUCCESS) {
	    LOG_WARNING("Couldn't create temp XML file for packing.");
	    RETURN_ERROR(ISD_PACK_ERROR);
	}

	//delete any existing file, ignoring errors
	REMOVE_FILE(fileName);
	if (PACK(fileName, xmlFileName, dataFile->getFileName())) {
	    LOG_WARNING("Error while packing the recording files.");
	    RETURN_ERROR(ISD_PACK_ERROR);
	}
	
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_PACK_ERROR);
    }
}
ISDObject::ISDErrorCode ISDRecording::exportRecording(string fileName, Glib::ustring* lang,
						      ISDProgressCallback* callbackClass,
						      char* pixelBuffer,
						      ISDExporter::ISDExportFormat format)
{
    ISDObject::ISDErrorCode retVal;

    string ext = ISDUtils::getInstance()->getExtension(fileName);

    //remove the file, no matter what
    REMOVE_FILE(fileName);

    /*
     * The FFmpeg encoder is used by default.
     * If we are dealing with FLV, we prefer libflv
     * NOT ANYMORE; got rid of libflv
     */ 
    ISDExporter* exporter;
    //if (ext=="flv" || ext=="FLV") {
    //exporter = new ISDFLVExporter(fileName, this, lang, callbackClass);
    //}
    //else {
    exporter = new ISDFFmpegExporter(fileName, this, lang, callbackClass);
    //}

    if (exporter->error() != ISD_SUCCESS) {
	delete exporter;
	RETURN_ERROR(exporter->error());
    }
    
    if ((retVal = exporter->exportRecording(pixelBuffer)) != ISD_SUCCESS) {
	delete exporter;
	RETURN_ERROR(retVal);
    }

    delete exporter;
    
    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDRecording::update(int pos, Glib::ustring* lang, bool& noVideoChange, bool& noCommentChange)
{
    //check for bad datafile
    if (dataFile==NULL || dataFile->error() ||
	xmlFile==NULL || xmlFile->error()) {
	LOG_WARNING("Tried to access the videostream without initialising/clean datafile or xmlfile");
	return ISD_INIT_ERROR;
    }

    ISDObject::ISDErrorCode retVal;

    /*
     * Merge the comments of the xmlfile with the videoCanvas.
     * Note: ignore errors, the xmlFile-object will print its own error messages
     *       and we can still continue with the pixel-data if some error happened
     *       in the commentbox-creation (and assume/hope the user will notice something
     *       went wrong).
     */
    xmlFile->updateCommentsForPosition(videoCanvas, pos, lang, noCommentChange);
    retVal = dataFile->updateCanvas(videoCanvas, pos, noVideoChange);

    return retVal;
}
ISDXmlFile* ISDRecording::getXmlFile()
{
    if (xmlFile!=NULL) {
	return xmlFile;
    }
    else {
	return NULL;
    }
}
ISDVideoCanvas* ISDRecording::getVideoCanvas()
{
    if (dataFile!=NULL) {
	return videoCanvas;
    }
    else {
	return NULL;
    }
}
ISDVideoProperties* ISDRecording::getVideoProperties()
{
    if (dataFile!=NULL) {
	return videoProperties;
    }
    else {
	return NULL;
    }
}
ISDDataFile* ISDRecording::getDataFile()
{
    if (dataFile!=NULL) {
	return dataFile;
    }
    else {
	return NULL;
    }
}

//-----PRIVATE/PROTECTED METHODS-----
ISDObject::ISDErrorCode ISDRecording::unpackRecordingFile(string fileName)
{
    string cmd;
    
    //construct the unpack directory
    string unpackFolder = ISDUtils::getInstance()->getFileDirectory(fileName) +
	+ RELATIVE_UNPACK_FOLDER + DIR_DELIM;
    if (CREATE_DIR(unpackFolder)) {
	LOG_CRITICAL("Could not create temp directory for unpacking recording-files: "+unpackFolder);
	RETURN_ERROR(ISD_UNPACK_ERROR);
    }

    //unpack the recording
    if (UNPACK(fileName, unpackFolder)) {
	LOG_CRITICAL("Error while unpacking recording-data.");
	RETURN_ERROR(ISD_UNPACK_ERROR);
    }

    //check if correct extracted files exist (read or write permissions)
    string metaFileName = unpackFolder + RELATIVE_META_FILE_NAME;
    if (!VALID_FILE(metaFileName)) {
	LOG_CRITICAL("Couldn't access the extracted meta file, aborting.");
	//cleanup
	deleteUnpackDirectory();
	RETURN_ERROR(ISD_UNPACK_ERROR);
    }
    string dataFileName = unpackFolder + RELATIVE_DATA_FILE_NAME;
    if (!VALID_FILE(dataFileName)) {
	LOG_CRITICAL("Couldn't access the extracted data file, aborting.");
	//cleanup
	deleteUnpackDirectory();
	RETURN_ERROR(ISD_UNPACK_ERROR);
    }

    //init variables
    unpackDirectory = unpackFolder;

    dataFile = new ISDDataFile(dataFileName, videoProperties);
    if (dataFile==NULL || dataFile->error() ||
	videoProperties==NULL || videoProperties->error()) {
	LOG_WARNING("Error while constructing/parsing the data file.");
	RETURN_ERROR(ISD_FILE_ERROR);
    }

    videoCanvas = new ISDVideoCanvas(videoProperties);
    if (videoCanvas==NULL || videoCanvas->error()) {
	LOG_WARNING("Error while constructing the VideoCanvas object.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }

    xmlFile = new ISDXmlFile(metaFileName, videoCanvas);
    if (xmlFile==NULL || xmlFile->error()) {
	LOG_WARNING("Error while constructing/parsing the XML file.");
	RETURN_ERROR(ISD_FILE_ERROR);
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDRecording::deleteUnpackDirectory()
{
    if (unpackDirectory.length()>0){
	int error = REMOVE_DIR(unpackDirectory);
	if (error) {
	    LOG_WARNING("Error occurred while deleting the temp unpack directory.");
	    RETURN_ERROR(ISD_CLEANUP_ERROR);
	}
    }
}

