/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/

/**
 * Separate program to export a .isd file to any video format supported by ffmpeg.
 * Use this program instead of ISDRecording::exportRecording, unless you know
 * what you are doing (OpenGL context switch issues).
 *
 * The format of the program-call is:
 *
 * isdexport infilepath outfilepath languagecode [-p]
 *
 * Optionally, a fourth parameter can be supplied to indicate
 * we want the program to print progress values that can be read
 * by another program.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <glibmm/ustring.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <GL/gl.h>
#include <GL/glx.h>

#include <libinstrudeo/isdobject.h>
#include <libinstrudeo/isdrecording.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdvideoproperties.h>
#include <libinstrudeo/isdprogresscallback.h>

#include "isdexporterrors.h"

using namespace std;

//----CHILD CLASS-----
/**
 * This class is used to monitor the progress of the export and import routines.
 * It subclasses ISDProgressCallback, and implements
 * its procentDone() method, that is called regularly to report the procent
 * of exported/imported data.
 */
class ISDExportProgressCallback : public ISDProgressCallback
{
  public:
    ISDExportProgressCallback() {}
    /*
     * Just pipe the value into stdout, the calling program should process this.
     */
    virtual void procentDone(float procent) {
	cout << procent << endl;
    }
};

int main(int argc, char* argv[])
{
    string infile, outfile, langCode;
    int width, height, depth;
    bool showProgress;

    if (argc != 4 && argc != 5) {
	cerr << "Please specify a source, target and language-code." << endl;
	return ISD_EXPORT_FORMAT_ERROR;
    }

    infile = argv[1];
    outfile = argv[2];
    langCode = argv[3];
    showProgress = (argc==5 && !strcmp(argv[4], "-p"));

    if (!VALID_FILE(infile)) {
	cerr << "Source recording isn't valid." << endl;
	return ISD_EXPORT_ARGS_ERROR;
    }

    /*
     * This is a workaround to know the video dimensions before actually
     * exporting the file. If we open the recording before we initialise the OpenGL
     * context, the commentboxes will not render correctly, so we open it twice.
     * Here is the first time, to know the dimension.
     *
     * I know this is quite ugly, be we only need to do this once...
     */
    ISDRecording* recording1 = new ISDRecording(infile);
    if (recording1->error() != ISDObject::ISD_SUCCESS) {
	cerr << "Error while opening source recording." << endl;
	return ISD_EXPORT_ISD_ERROR;
    }
    ISDVideoProperties* videoProps = recording1->getVideoProperties();
    width = videoProps->getWidth();
    height = videoProps->getHeight();
    depth = videoProps->getDepth();
    delete(recording1);

    // Connect to the X server
    Display* dpy = XOpenDisplay(NULL);
    XSynchronize(dpy, 0);
    if (dpy == NULL){
	cerr << "Error while connecting to X server." << endl;
	return ISD_EXPORT_DISPLAY_ERROR;
    }
    int screen = DefaultScreen(dpy);

    //Check if GLX extension is available
    int dummy;
    if(!glXQueryExtension(dpy, &dummy, &dummy)){
	cerr << "Couldn't query GLX extension." << endl;
	return ISD_EXPORT_GLX_ERROR;
    }

    // Get the X visual
    int attribList[] = {GLX_RGBA, None};
    XVisualInfo* vis = glXChooseVisual(dpy, screen, attribList);
    if (vis == NULL) {
	cerr << "Error while fetching X visuals." << endl;
	return ISD_EXPORT_X_ERROR;
    }

    // Create the offscreen rendering buffer
    Pixmap pixmap = XCreatePixmap(dpy, RootWindow(dpy, screen),
				  width, height, depth);
    GLXPixmap glPixmap = glXCreateGLXPixmap(dpy, vis, pixmap);

    if (glPixmap == None) {
	cerr << "Error while creating offscreen rendering buffer." << endl;
	return ISD_EXPORT_GLX_ERROR;
    }

    // Create the new GL context
    GLXContext cx = glXCreateContext(dpy, vis, NULL, GL_FALSE);
    if (cx == NULL) {
	cerr << "Error while creating OpenGL context." << endl;
	return ISD_EXPORT_GLX_CONTEXT_ERROR;
    }

    // Switch contexts
    if (glXMakeCurrent(dpy, glPixmap, cx) == False) {
	cerr << "Error while switching to OpenGL context." << endl;
	return ISD_EXPORT_GLX_CONTEXT_ERROR;
    }

    /*
     * Now, open the recording for the second time.
     * On the heap, so we can cleanup before OpenGL.
     */
    ISDRecording* recording = new ISDRecording(infile);
    if (recording->error() != ISDObject::ISD_SUCCESS) {
	cerr << "Error while opening source recording." << endl;
	return ISD_EXPORT_ISD_ERROR;
    }

    //init the OpenGL viewport
    glViewport(0, 0, (GLsizei)width, (GLsizei)height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, (GLdouble)width, 0.0, (GLdouble)height, -1000, 1000);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    Glib::ustring langPtr = langCode;

    ISDObject::ISDErrorCode retVal;

    if (showProgress) {
	//create the callback class, so we are able to cout the percentages.
	ISDExportProgressCallback callback;
	retVal = recording->exportRecording(outfile, &langPtr, &callback);
    }
    else {
	retVal = recording->exportRecording(outfile, &langPtr, NULL);
    }

    if (retVal != ISDObject::ISD_SUCCESS) {
	cerr << "Error while exporting the recording." << endl;
	return ISD_EXPORT_ISD_ERROR;
    }

    //TODO: segfaults if this is enabled???
    //delete(recording);

    return EXIT_SUCCESS;
}
