/******************************************************************************************************************************************
 ciconview.c
******************************************************************************************************************************************/

#include "ciconview.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CIconViewItemFieldValueAPIListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// field value api listener
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconViewItemFieldValueAPIListener::OnStateChange (CObject *inItemField)
{
	// retreive the item field value instance
	CItemFieldValue *inItemFieldValue = static_cast <CItemFieldValue *> (inItemField);

	// retreive the generic icon view item instance
	CIconViewItem *inIconViewItem = static_cast <CIconViewItem *> (inItemFieldValue -> GetOwner());

	// set the icon view item its new item fields values and states
	if (inIconViewItem != NULL) inIconViewItem -> SetItemFieldValues (CItemFieldValues (inIconViewItem -> GetItemFieldValues()));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CIconViewItem);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewItem::CIconViewItem	 (CIconView *inOwner, const CItemFieldValues &inFieldValues, const CObjectListener *inListener) THROWABLE
	      :CControl	    	 (inOwner, inListener),
	       m_ItemFieldValues (inFieldValues),
	       m_GtkTreeIter	 ()
{
	// specified field values type check in
	if (m_ItemFieldValues.GetLength() != 2)
	{
		// first field value is specified ?
		if (m_ItemFieldValues.GetLength() == 1) 
		{
			// check the first instance type
			if (!(*m_ItemFieldValues[0]) -> ClassIs (__metaclass(CItemFieldValuePixbuf)))
			throw new CException ("CIconViewItem : the 0 index field value should be a pixbuf item specification. Aborting...");

			// instanciate the second index instance
			m_ItemFieldValues += new CItemFieldValueString ();
		}
		// instanciate the default instances
		else
		{
			m_ItemFieldValues += new CItemFieldValuePixbuf (new CPixbuf("null"));
			m_ItemFieldValues += new CItemFieldValueString ();
		}
	}
	// we got the instances, check their type
	else
	{
		// the first index should be a pixbuf representation
		if (!(*m_ItemFieldValues[0]) -> ClassIs (__metaclass(CItemFieldValuePixbuf)))
			throw new CException ("CIconViewItem : the 0 index field value should be a pixbuf item specification. Aborting...");

		// the second index should be a string representation
		if (!(*m_ItemFieldValues[1]) -> ClassIs (__metaclass(CItemFieldValueString)))
			throw new CException ("CIconViewItem : the 1 index field value should be a string item specification. Aborting...");
	}

	// set the icon view item values and create the associated gtk tree iter of the icon view
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewItem::CIconViewItem	 (CIconView *inOwner, const CItemFieldValuePixbuf *inPixbuf, const CItemFieldValueString *inString,
			     	  const CObjectListener *inListener)
	      :CControl	     	 (inOwner, inListener),
	       m_ItemFieldValues (2, const_cast <CItemFieldValuePixbuf *> (inPixbuf), const_cast <CItemFieldValueString *> (inString)),
	       m_GtkTreeIter	 ()
{
	// set the icon view item values and create the associated gtk tree iter of the icon view
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewItem::CIconViewItem	 (CIconView *inOwner, const CPixbuf *inPixbuf, const CString &inString, const CObjectListener *inListener)
	      :CControl	 	 (inOwner, inListener),
	       m_ItemFieldValues (2, new CItemFieldValuePixbuf (inPixbuf), new CItemFieldValueString (inString)),
	       m_GtkTreeIter	 ()
{
	// set the icon view item values and create the associated gtk tree iter of the icon view
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewItem::~CIconViewItem ()
{
	// get the gtkol icon view owner of this instance
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));

	// remove the item value from the gtk widget icon view
	if (inIconView != NULL && inIconView -> GetGtkWidget() != NULL)
		::gtk_list_store_remove (GTK_LIST_STORE(::gtk_icon_view_get_model (GTK_ICON_VIEW(inIconView -> GetGtkWidget()))),
					 &m_GtkTreeIter);

	// delete the handled icon item instances values and their associated api listener
	for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) delete *m_ItemFieldValues[i-1];
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk tree iter associated struct
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkTreeIter * CIconViewItem::GetGtkTreeIter () const
{
	// ok, cast and return
	return const_cast <GtkTreeIter *> (&m_GtkTreeIter);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// icon view item search
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewItem * CIconViewItem::GetIconViewItem (const GtkTreeIter *inGtkTreeIter)
{
	// pointer check
	if (inGtkTreeIter == NULL) return NULL;

	// get the whole application icon view instances
	CComponents inComponents (CComponent::GetComponents (__metaclass(CIconViewItem)));

	// go through
	for (size_t i=inComponents.GetLength(), j=0; i>0; i--, j++)
		if (static_cast <CIconViewItem *> (*inComponents[j]) -> m_GtkTreeIter.user_data == inGtkTreeIter->user_data)
			return static_cast <CIconViewItem *> (*inComponents[j]);

	// not found
	return NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CIconViewItem::OwnerMustBe () const
{
	return __metaclasses(CIconView);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// owner affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CIconViewItem::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// owner type check !
	if (!CheckSetOwner (inOwner)) return false;

	// get potential previous owner
	CComponent *inOldOwner = GetOwner ();

	// get the previous icon view owner
	CIconView *inOldIconView = inOldOwner != NULL ? inOldOwner -> ClassIs (__metaclass(CIconView)) ? 
					static_cast <CIconView *> (inOldOwner) : 
				 	static_cast <CIconView *> (inOldOwner -> GetOwner (__metaclass(CIconView))) : NULL;

	// get the new icon view owner
	CIconView *inNewIconView = inOwner != NULL ? inOwner -> ClassIs (__metaclass(CIconView)) ? 
				 	static_cast <CIconView *> (inOwner) : 
				 	static_cast <CIconView *> (inOwner -> GetOwner (__metaclass(CIconView))) : NULL;

	// skelton call
	if (!CControl::SetOwner (inOwner, inIndex)) return false;

	// if there were an old icon view widget for this item instance
	if (inOldIconView != NULL && m_GtkTreeIter.user_data != NULL)
	
		// remove the item from its old owner
		::gtk_list_store_remove (GTK_LIST_STORE(::gtk_icon_view_get_model (GTK_ICON_VIEW(inOldIconView->GetGtkWidget()))),
					 &m_GtkTreeIter);

	// if the new owner is specified
	if (inOwner != NULL)
	{
		// append requested
		if (inIndex < 0)

			// append this item at the end of the store
			::gtk_list_store_append (GTK_LIST_STORE(::gtk_icon_view_get_model (GTK_ICON_VIEW(inNewIconView->GetGtkWidget()))), 
						 &m_GtkTreeIter);

		// insert requested (index range checked by gtk itself)
		else
			
			// insert this item into the store
			::gtk_list_store_insert (GTK_LIST_STORE(::gtk_icon_view_get_model (GTK_ICON_VIEW(inNewIconView->GetGtkWidget()))), 
						 &m_GtkTreeIter, inIndex);
	}

	// item field values affectation
	return SetItemFieldValues (CItemFieldValues (m_ItemFieldValues));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item bounds
//-----------------------------------------------------------------------------------------------------------------------------------------
TBounds CIconViewItem::GetBounds () const
{
	// the outBounds, no ccontrol attribute handling as we won't calculate it until requested due to performance reason...
	TBounds outBounds;

	// get our icon view handler
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));

	// pointer check
	if (inIconView == NULL) return outBounds;

	// get this item in owner index, be sure to count only on icon view item instances
	SInt16 inIndex = CComponent::GetInOwnerIndex (inIconView, this, __metaclass(CIconViewItem));

	// check index
	if (inIndex < 0) return outBounds;

	// get the icon view margin i.e. extra place added at the top, left, right and bottom of the view
	UInt16 inMargin = inIconView -> GetMargin ();

	// as gtk does not provide an api to retreive such an information, try our best by requesting at least the icon total size
	// while getting its pixbuf representation ! - FIXME : I did'nt found any other way to get an icon view item bounds -
	CPixbuf *inPixbuf = GetControlPixbuf ();

	// pointer check
	if (inPixbuf == NULL) return outBounds;

	// set the outBounds dimension
	outBounds = inPixbuf -> GetSize();

	// free the pixbuf
	delete inPixbuf;

	// set the outBounds minimal x and y point
	outBounds.x = outBounds.y = inMargin;

	// index check
	if (inIndex == 0) return outBounds;

	// get the icon view maximum columns number if any
	SInt16 inColNumber = inIconView -> GetColumns ();

	// get the icon view row and col spacing
	UInt16 inRowSpacing = inIconView -> GetRowSpacing ();
	UInt16 inColSpacing = inIconView -> GetColSpacing ();

	// no columns number restriction
	if (inColNumber < 0)
	{
		// get the icon total view size
		TSize inSize (static_cast <TSize> (inIconView -> GetBounds()));

		// get the line maximum items number within the given view size
		SInt32 nw = (inSize.w - 2*inMargin) / (outBounds.w + inColSpacing);
		
		// bounds x and y point calculation
		outBounds.x += (inIndex % nw) * (outBounds.w + inColSpacing);
		outBounds.y += (inIndex / nw) * (outBounds.h + inRowSpacing);
	}
	// columns number restriction
	else
	{
		// bounds x and y point calculation
		outBounds.x += (inIndex % inColNumber) * (outBounds.w + inColSpacing);
		outBounds.y += (inIndex / inColNumber) * (outBounds.h + inRowSpacing);
	}

	// ok, we got it !
	return outBounds;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// pixbuf representation
//-----------------------------------------------------------------------------------------------------------------------------------------
CPixbuf * CIconViewItem::GetControlPixbuf () const
{
	// get the tree view owner
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));

	// the gtk tree view
	GtkIconView *inGtkIconView = NULL;

	// pointer check
	if (inIconView == NULL || (inGtkIconView = GTK_ICON_VIEW(inIconView->GetGtkWidget())) == NULL) return NULL;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_icon_view_get_model(inGtkIconView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// create the associated pixmap
	GdkPixmap *inGdkPixmap = ::gtk_icon_view_create_drag_icon (inGtkIconView, inGtkTreePath);

	// get the pixmap size
	gint inWidth, inHeight; ::gdk_drawable_get_size (inGdkPixmap, &inWidth, &inHeight);

	// create the pixbuf instance
	CPixbuf *outPixbuf = new CPixbuf (inGdkPixmap, PIXELFORMAT_32, TBounds(0,0,inWidth,inHeight));

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);

	// release the pixmap pointer
	::gdk_pixmap_unref (inGdkPixmap);

	// be paranoïd...
	if (outPixbuf -> GetPixelFormat() != PIXELFORMAT_32)
	{
		// delete it !
		delete outPixbuf;

		// stop
		return NULL;
	}

	// keep dark pixels (that is, darker than gray)
	UInt8 *Line = outPixbuf -> GetBaseAddress();
	for (size_t i=inHeight; i>0; i--)
	{
		UInt32 *Pixel = reinterpret_cast <UInt32 *> (Line);
		for (size_t j=inWidth; j>0; j--, Pixel++)
			if ((*Pixel&0x00FFFFFF) <= 0x007F7F7F) *Pixel = 0xFF7F7F7F; else *Pixel = 0L;
		Line += outPixbuf -> GetRowBytes();
	}

	// set the default border representation of the dragged control (opaque gray borders), direct draw on
	// the pixbuf's buffer (see static declaration of CPixbuf::DrawLine and CPixbuf::DrawRectangle function for details in cgraphics.h)
	CPixbuf::DrawRectangle (TBounds (0L, 0L, inWidth-1L, inHeight-1L), outPixbuf->GetBaseAddress(), PIXELFORMAT_32, 
				outPixbuf->GetRowBytes(), 0xFF7F7F7F, 1.0f);

	// ok
	return outPixbuf;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// field values affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CIconViewItem::SetItemFieldValues (const CItemFieldValues &inItemFieldValues)
{
	// specification check
	if (inItemFieldValues.GetLength() != 2 || !(*inItemFieldValues[0]) -> ClassIs (__metaclass(CItemFieldValuePixbuf)) ||
	    !(*inItemFieldValues[1]) -> ClassIs (__metaclass(CItemFieldValueString))) return false;

	// delete the old values if any
	if (&inItemFieldValues != &m_ItemFieldValues)
		for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) 
			if (*inItemFieldValues[i-1] != *m_ItemFieldValues[i-1]) 
				delete *m_ItemFieldValues[i-1];

	// pointers local copy
	m_ItemFieldValues = inItemFieldValues;

	// go through the field values instances and set the friend protected attributes we are supposed to handle
	for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--)
	{ 
		// our icon view item handles the field value instance
		(*m_ItemFieldValues[i-1]) -> m_Owner = this;

		// set the item field value api listener so that when the programmer performs modifications on that field, the gtkol engine 
		// can handle gui modifications accordinaly
		if ((*m_ItemFieldValues[i-1]) -> m_Listener == NULL) 
			(*m_ItemFieldValues[i-1]) -> AssignListener (new CIconViewItemFieldValueAPIListener());
	}

	// get the icon view widget this item is under
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));
	
	// pointer check, memorization done anyway
	if (inIconView == NULL) return true;

	// set the field values
	::gtk_list_store_set (GTK_LIST_STORE(::gtk_icon_view_get_model (GTK_ICON_VIEW(inIconView->GetGtkWidget()))), &m_GtkTreeIter, 
		0, static_cast <CItemFieldValuePixbuf *> (*m_ItemFieldValues[0]) -> m_FieldValue -> GetPixbuf(), 
		1, static_cast <CItemFieldValueString *> (*m_ItemFieldValues[1]) -> m_FieldValue.Get(), -1);

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// field values access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValues CIconViewItem::GetItemFieldValues () const
{
	return m_ItemFieldValues;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// field value indexed access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValue & CIconViewItem::operator [] (const size_t inIndex) const
{
	return **m_ItemFieldValues[inIndex];
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// first field value access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValuePixbuf & CIconViewItem::GetItemFieldValuePixbuf () const
{
	return static_cast <CItemFieldValuePixbuf &> (**m_ItemFieldValues[0]);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// second field value access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValueString & CIconViewItem::GetItemFieldValueString () const
{
	return static_cast <CItemFieldValueString &> (**m_ItemFieldValues[1]);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// path selection
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconViewItem::Select (const bool inSelect)
{
	// get the tree view owner
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));

	// the gtk tree view
	GtkIconView *inGtkIconView = NULL;

	// pointer check
	if (inIconView == NULL || (inGtkIconView = GTK_ICON_VIEW(inIconView->GetGtkWidget())) == NULL) return;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_icon_view_get_model(inGtkIconView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// pointer check
	if (inGtkTreePath == NULL) return;

	// property check
	if (inSelect) 
		::gtk_icon_view_select_path (inGtkIconView, inGtkTreePath); 
	else 
		::gtk_icon_view_unselect_path (inGtkIconView, inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// is selected ?
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CIconViewItem::IsSelected () const
{
	// get the tree view owner
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));

	// the gtk tree view
	GtkIconView *inGtkIconView = NULL;

	// pointer check
	if (inIconView == NULL || (inGtkIconView = GTK_ICON_VIEW(inIconView->GetGtkWidget())) == NULL) return false;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_icon_view_get_model(inGtkIconView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// pointer check
	if (inGtkTreePath == NULL) return false;

	// check
	bool outResult = ::gtk_icon_view_path_is_selected (inGtkIconView, inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);

	// ok
	return outResult;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// activate the icon view item
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconViewItem::Activate ()
{
	// get the tree view owner
	CIconView *inIconView = static_cast <CIconView *> (GetOwner (__metaclass(CIconView)));

	// the gtk tree view
	GtkIconView *inGtkIconView = NULL;

	// pointer check
	if (inIconView == NULL || (inGtkIconView = GTK_ICON_VIEW(inIconView->GetGtkWidget())) == NULL) return;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_icon_view_get_model(inGtkIconView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// pointer check
	if (inGtkTreePath == NULL) return;

	// ok
	::gtk_icon_view_item_activated (inGtkIconView, inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// icon view item serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconViewItem::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CControl::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_ICONVIEWITEM_ELEMENT);

			// modify the io xml node so that the overwritten definitions would continue under this new xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// instanciate the fields xml element under the current xml node
			newXMLElement = new CXMLElement (ioXMLElementNode, XML_ICONVIEWITEM_FIELDS_ELEMENT);

			// foreach of the fields, delegate the serialization process
			for (size_t i=m_ItemFieldValues.GetLength(), j=0; i>0; i--, j++)
			{
				// each of the field serializes under the current node
				CXMLElementNode *inXMLNode = newXMLElement -> GetXMLElementNode ();

				// serialize the field
				(*m_ItemFieldValues[j]) -> Serialize (inXMLNode, XML_WRITE);
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the node this instance should read from
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_ICONVIEWITEM_ELEMENT);

			// check we got it
                        if (inXMLNode == NULL)
                                throw new CException (CString("CIconViewItem::Serialize, specified xml node is not a \"") +
                                                              XML_ICONVIEWITEM_ELEMENT + CString("\" element one."), 
						      	      __exception(XMLPARSE));

			// modify the io xml node so that the overwritten definitions would continue under this new xml node
			ioXMLElementNode = inXMLNode;

			// get the fields xml child element node
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_ICONVIEWITEM_FIELDS_ELEMENT);

			// check we got it
			if (inXMLNode != NULL)
			{
				// delete the handled icon view item instances values if any
				if (m_ItemFieldValues.GetLength() > 0) 
				{
					for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) delete *m_ItemFieldValues[i-1];
					m_ItemFieldValues.Delete (0, m_ItemFieldValues.GetLength());
				}

				// get the "fields" xml element children nodes
				CXMLElementNodes inXMLNodes (::xml_node_get_children (inXMLNode));

				// foreach child of the "children" element, launch the global serialization process
				for (size_t i=inXMLNodes.GetLength(), j=0; i>0; i--, j++)
				{
					// launch the instanciation process
					CSerialized *inSerialized = CSerialized::Instanciate (*inXMLNodes[j]) THROWABLE;

					// pointer check
					if (inSerialized == NULL) continue;

					// check the pointer type
					if (!inSerialized -> ClassIs (__metaclass(CItemFieldValue))) { delete inSerialized; continue; }

					// add the field value to the local buffer
					m_ItemFieldValues += static_cast <CItemFieldValue *> (inSerialized);
				}
			}

			// set the item field values
			SetItemFieldValues (CItemFieldValues (m_ItemFieldValues));
		}
		break;
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CIconViewListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewListener::CIconViewListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewListener::~CIconViewListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CIconView);

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemActivated
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::OnItemActivated (GtkIconView *, GtkTreePath *inGtkTreePath, gpointer inData)
{
	// retreive the gtkol icon view instance
	CIconView *inIconView = reinterpret_cast <CIconView *> (inData);

	// pointer check
	if (inIconView == NULL) return;

	// get the associated gtk tree iter
	GtkTreeIter inGtkTreeIter;
	gtk_tree_model_get_iter (::gtk_icon_view_get_model (GTK_ICON_VIEW(inIconView->GetGtkWidget())), &inGtkTreeIter, inGtkTreePath);

	// retreive the gtkol icon view item instance
	CIconViewItem *inItem = CIconViewItem::GetIconViewItem (&inGtkTreeIter);

	// pointer check
	if (inItem == NULL) return;

	// send the notification
	if (inIconView -> GetListener() != NULL)
		static_cast <CIconViewListener *> (inIconView -> GetListener()) -> OnItemActivated (inIconView, inItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemSelectionChanged
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::OnItemSelectionChanged (GtkIconView *, gpointer inData)
{
	// retreive our gtkol icon view instance
	CIconView *inIconView = reinterpret_cast <CIconView *> (inData);

	// send the notification to the listener if any
	if (inIconView != NULL && inIconView -> GetListener() != NULL)
		static_cast <CIconViewListener *> (inIconView -> GetListener()) -> OnItemSelectionChanged (inIconView);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemEdited
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::OnItemEdited (GtkCellRendererText *, const gchar *inPath, const gchar *inNewValue, gpointer inData)
{
	// retreive the gtkol icon view instance
	CIconView *inIconView = reinterpret_cast <CIconView *> (inData);

	// pointer check
	if (inIconView == NULL) return;

	// default to an update
	bool doChange = true;

	// get the path string associated gtk tree path
	GtkTreePath *inGtkTreePath = gtk_tree_path_new_from_string (inPath);

	// pointer check
	if (inGtkTreePath == NULL) return;

	// get the associated gtk tree iter
	GtkTreeIter inGtkTreeIter;
	gtk_tree_model_get_iter (::gtk_icon_view_get_model (GTK_ICON_VIEW(inIconView->GetGtkWidget())), &inGtkTreeIter, inGtkTreePath);

	// release the path
	::gtk_tree_path_free (inGtkTreePath);

	// retreive the gtkol icon view item instance
	CIconViewItem *inItem = CIconViewItem::GetIconViewItem (&inGtkTreeIter);

	// pointer check
	if (inItem == NULL) return;

	// the considered string
	CString ioString (inNewValue);

	// send the notification
	if (inIconView -> GetListener() != NULL)
		static_cast <CIconViewListener *> (inIconView -> GetListener()) -> OnItemEdited (inIconView, inItem, ioString, doChange);

	// do we have to change the field value ?
	if (doChange)
	{
		// keep the value into the item field values
		static_cast <CItemFieldValueString *> (*inItem->m_ItemFieldValues[1]) -> m_FieldValue = ioString;

		// set the value on the gui
		::gtk_list_store_set (GTK_LIST_STORE(::gtk_icon_view_get_model (GTK_ICON_VIEW(inIconView->GetGtkWidget()))),
				      &inGtkTreeIter, 1, ioString.Get(), -1);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconView::CIconView		 (CContainer *inOwner, const CIconViewListener *inListener)
	  :CLayout   		 (inOwner, inListener),
	   m_Editable		 (false),
	   m_GtkCellRendererText (NULL)
{
	// instanciation process launch
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconView::~CIconView ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CIconView::PerformWidgetInstanciate ()
{
	// create the icon view model
	GtkListStore *inGtkListStore = ::gtk_list_store_new (2, GDK_TYPE_PIXBUF, G_TYPE_STRING);

	// ok, instanciate the icon view on the given model
	GtkWidget *outGtkWidget = ::gtk_icon_view_new_with_model (GTK_TREE_MODEL(inGtkListStore));

	// unref the store
	::g_object_unref (inGtkListStore);

	// delete potential old cell renderers
	::gtk_cell_layout_clear (GTK_CELL_LAYOUT(outGtkWidget));

	// instanciate the pixbuf cell renderer and the text cell renderer
	GtkCellRenderer *inGtkCellRendererPixbuf = ::gtk_cell_renderer_pixbuf_new ();
			 m_GtkCellRendererText	 = ::gtk_cell_renderer_text_new   ();

	// pack the renderers into the icon view
	::gtk_cell_layout_pack_start (GTK_CELL_LAYOUT(outGtkWidget), inGtkCellRendererPixbuf, true);
	::gtk_cell_layout_pack_start (GTK_CELL_LAYOUT(outGtkWidget), m_GtkCellRendererText,   true);

	// map the cell layouts attributes on the assosiated model
	::gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT(outGtkWidget), inGtkCellRendererPixbuf, "pixbuf", 0, NULL);
	::gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT(outGtkWidget), m_GtkCellRendererText,   "text",   1, NULL);

	// edition signal connection
	::g_signal_connect (m_GtkCellRendererText, "edited", G_CALLBACK(CIconView::OnItemEdited), this);

	// ok
	return outGtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::PerformWidgetInitialize ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// signal connections
	::g_signal_connect (G_OBJECT(inGtkWidget), "item-activated", 	G_CALLBACK(CIconView::OnItemActivated), 	this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "selection-changed", G_CALLBACK(CIconView::OnItemSelectionChanged), 	this);

	// show the job
	Show ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected listener type
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CIconView::ListenerMustBe () const
{
	return __metaclass(CIconViewListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CIconView::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children types
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CIconView::ChildMustBe () const
{
	return __metaclasses(CIconViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// control at pos
//-----------------------------------------------------------------------------------------------------------------------------------------
CControl * CIconView::GetControlAtPoint (const TPoint &inRelativePoint) const
{
	// retreive our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return const_cast <CIconView *> (this);

	// get the item path at the given position
	GtkTreePath *inGtkTreePath = NULL; 
	::gtk_icon_view_get_item_at_pos (GTK_ICON_VIEW(inGtkWidget), inRelativePoint.x, inRelativePoint.y, &inGtkTreePath, NULL);
	
	// check the returned path
	if (inGtkTreePath == NULL) return const_cast <CIconView *> (this);

	// get the icon view model
	GtkTreeModel *inModel = (GtkTreeModel *) ::gtk_icon_view_get_model (GTK_ICON_VIEW(inGtkWidget));

	// get the associated tree iter
	GtkTreeIter inGtkTreeIter; ::gtk_tree_model_get_iter (GTK_TREE_MODEL(inModel), &inGtkTreeIter, inGtkTreePath);

	// release the path
	::gtk_tree_path_free (inGtkTreePath);

	// get the associated CIconViewItem
	CIconViewItem *inIconViewItem = CIconViewItem::GetIconViewItem (&inGtkTreeIter);

	// return the control
	return inIconViewItem != NULL ? static_cast <CControl *> (inIconViewItem) : const_cast <CIconView *> (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// drop index at point
//-----------------------------------------------------------------------------------------------------------------------------------------
SInt16 CIconView::GetDropIndexAtPoint (const TPoint &inRelativePoint, const CControl *) const
{
	// get the control at the specified point
	CControl *inReference = GetControlAtPoint (inRelativePoint);

	// pointer check
	if (inReference == NULL) return -1;

	// get the reference control in owner index, be sure to count only on icon view item instances
	SInt16 inIndex = CComponent::GetInOwnerIndex (this, inReference, __metaclass(CIconViewItem));

	// get the reference control bounds
	TBounds inBounds (inReference -> GetBounds());

	// ok, check the x point to see if the drop would occur before or after the reference control
	return (inRelativePoint.x < (inBounds.x + (inBounds.w/2))) ? inIndex > 0 ? inIndex-1 : 0 : inIndex;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// editable
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetEditable (const bool inEditable)
{
	// set the cell renderer property
	if (m_GtkCellRendererText != NULL) ::g_object_set (m_GtkCellRendererText, "editable", inEditable, NULL);

	// keep property
	m_Editable = inEditable;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// editable
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CIconView::GetEditable () const
{
	// ok
	return m_Editable;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selection mode 
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetSelectionMode (const GtkSelectionMode inSelectionMode)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check and property affectation
	if (inGtkWidget != NULL) ::gtk_icon_view_set_selection_mode (GTK_ICON_VIEW(inGtkWidget), inSelectionMode);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selection mode
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkSelectionMode CIconView::GetSelectionMode () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// property request
	return inGtkWidget != NULL ? ::gtk_icon_view_get_selection_mode (GTK_ICON_VIEW(inGtkWidget)) : GTK_SELECTION_NONE;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// current selection
//-----------------------------------------------------------------------------------------------------------------------------------------
CIconViewItems CIconView::GetSelection () const
{
	// out buffer list
	CIconViewItems outItems;

	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return outItems;

	// retreive the gtk model
	GtkTreeModel *inGtkTreeModel = reinterpret_cast <GtkTreeModel *> (::gtk_icon_view_get_model (GTK_ICON_VIEW(inGtkWidget)));

	// retreive the selected items
	GList *inGList = ::gtk_icon_view_get_selected_items (GTK_ICON_VIEW(inGtkWidget)); GList *inGListItem = inGList;

	// go through the given glist
	while (inGListItem != NULL)
	{
		// the requested gtk tree iter element
		GtkTreeIter inGtkTreeIter; 
		::gtk_tree_model_get_iter (inGtkTreeModel, &inGtkTreeIter, reinterpret_cast <GtkTreePath *> (inGListItem -> data));

		// get the associated gtkol icon view item
		CIconViewItem *inIconViewItem = CIconViewItem::GetIconViewItem (&inGtkTreeIter);

		// output buffer addon
		if (inIconViewItem != NULL) outItems += inIconViewItem;

		// next item
		inGListItem = inGListItem -> next;
	}

	// free the list
	::g_list_foreach (inGList, reinterpret_cast <void (*) (void *, void *)> (::gtk_tree_path_free), NULL);
	::g_list_free 	 (inGList);

	// ok
	return outItems;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set orientation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetOrientation (const GtkOrientation inGtkOrientation)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// property affectation
	if (inGtkWidget != NULL) ::gtk_icon_view_set_orientation (GTK_ICON_VIEW(inGtkWidget), inGtkOrientation);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get orientation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkOrientation CIconView::GetOrientation () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_orientation (GTK_ICON_VIEW(inGtkWidget)) : GTK_ORIENTATION_HORIZONTAL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// columns number
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetColumns (const SInt16 inNumber)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// property affectation
	if (inGtkWidget != NULL) ::gtk_icon_view_set_columns (GTK_ICON_VIEW(inGtkWidget), inNumber);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// columns number
//-----------------------------------------------------------------------------------------------------------------------------------------
SInt16 CIconView::GetColumns () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_columns (GTK_ICON_VIEW(inGtkWidget)) : -1;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item width
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetItemWidth (const SInt16 inItemWidth)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_icon_view_set_item_width (GTK_ICON_VIEW(inGtkWidget), inItemWidth);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item width
//-----------------------------------------------------------------------------------------------------------------------------------------
SInt16 CIconView::GetItemWidth () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_item_width (GTK_ICON_VIEW(inGtkWidget)) : 0;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// spacing
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetSpacing (const UInt16 inSpacing)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_icon_view_set_spacing (GTK_ICON_VIEW(inGtkWidget), inSpacing);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// spacing
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CIconView::GetSpacing () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_spacing (GTK_ICON_VIEW(inGtkWidget)) : 0;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// row spacing
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetRowSpacing (const UInt16 inRowSpacing)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_icon_view_set_row_spacing (GTK_ICON_VIEW(inGtkWidget), inRowSpacing);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// row spacing
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CIconView::GetRowSpacing () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_row_spacing (GTK_ICON_VIEW(inGtkWidget)) : 0;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// col spacing
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetColSpacing (const UInt16 inColSpacing)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_icon_view_set_column_spacing (GTK_ICON_VIEW(inGtkWidget), inColSpacing);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// col spacing
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CIconView::GetColSpacing () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_column_spacing (GTK_ICON_VIEW(inGtkWidget)) : 0;
}
		
//-----------------------------------------------------------------------------------------------------------------------------------------
// margin
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::SetMargin (const UInt16 inMargin)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_icon_view_set_margin (GTK_ICON_VIEW(inGtkWidget), inMargin);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// margin
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CIconView::GetMargin () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_icon_view_get_margin (GTK_ICON_VIEW(inGtkWidget)) : 0;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// icon view serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CIconView::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CLayout::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// create a new xml element node for this instance serialization
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_ICONVIEW_ELEMENT);

			// modify the io xml element node so that the potential overloaded definitions will continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// get the selection mode
			CString inSelectionMode; switch (GetSelectionMode())
			{
				case GTK_SELECTION_NONE		: inSelectionMode = CString("none");	 break;
				case GTK_SELECTION_SINGLE	: inSelectionMode = CString("single");	 break;
				case GTK_SELECTION_BROWSE	: inSelectionMode = CString("browse"); 	 break;
				case GTK_SELECTION_MULTIPLE 	: inSelectionMode = CString("multiple"); break;
			}

			// add the selection mode attribute
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_SELECTION_MODE, inSelectionMode);

			// get the selected items if any
			CString inSelection; CIconViewItems inCurrentSelection (GetSelection());
			for (size_t i=inCurrentSelection.GetLength(), j=0; i>0; i--, j++)
			{
				SInt16 index = CComponent::GetInOwnerOccurence (this, *inCurrentSelection[j], __metaclass(CIconViewItem));
				if (index >= 0)
				{
					inSelection += CString ((UInt32)index);
					if (i > 1) inSelection += CString(",");
				}
			}

			// add the selection attribute
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_SELECTED, inSelection);

			// get the orientation
			if (GetOrientation() == GTK_ORIENTATION_VERTICAL)
				newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_ORIENTATION, CString("vertical"));
			else
				newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_ORIENTATION, CString("horizontal"));

			// add the last numeric attributes
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_EDITABLE,	CString(GetEditable()?"true":"false"));
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_COLUMNS, 	CString((SInt32)GetColumns()));
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_ITEM_WIDTH, 	CString((SInt32)GetItemWidth()));
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_SPACING, 	CString((UInt32)GetSpacing()));
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_ROW_SPACING, 	CString((UInt32)GetRowSpacing()));
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_COL_SPACING, 	CString((UInt32)GetColSpacing()));
			newXMLElement -> AddAttribute (XML_ICONVIEW_ATTR_MARGIN, 	CString((UInt32)GetMargin()));
		}
		break;

		// xml load
		case XML_READ :
		{
			// get our interested xml node
			CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_ICONVIEW_ELEMENT);

			// check we got an expected ctreeview node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CIconView::Serialize, specified xml node is not a \"") +
                                                              XML_ICONVIEW_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// get the selection mode property and the selected items
			CString inSelectionMode (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_SELECTION_MODE).GetValue());
			CString inSelection	(::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_SELECTED).	GetValue());

			// set the selection mode of the tree view
			if (inSelectionMode == CString("single"  )) SetSelectionMode (GTK_SELECTION_SINGLE  ); else
			if (inSelectionMode == CString("browse"  )) SetSelectionMode (GTK_SELECTION_BROWSE  ); else
			if (inSelectionMode == CString("multiple")) SetSelectionMode (GTK_SELECTION_MULTIPLE); else
			SetSelectionMode (GTK_SELECTION_NONE);

			// analyse the specified items selection i.e. separate the icon view items id
			CStrings inSelections (inSelection.Cut (CString(","), true));

			// get the icon view items
			CComponents inIconViewItems (GetSubComponents (__metaclass(CIconViewItem)));

			// foreach one, select it !
			if (inSelection.GetLength() > 0) for (size_t i=inSelections.GetLength(), j=0; i>0; i--, j++)
			{
				// get the component of specified index
				CComponent *inComponent = inSelections[j]->ToULong() < inIconViewItems.GetLength() ? 
							  *inIconViewItems[inSelections[j]->ToULong()] : NULL;

				// check we got it
				if (inComponent != NULL)

					// select it...
					static_cast <CIconViewItem *> (inComponent) -> Select (true);

				// not found or not an icon view item
				else

					// throw an exception, we can't afford that !
					throw new CException ("CIconView::Serialize, specified ciconview item index \"" +
							      (*inSelections[j]) + "\" on \"" +
                                                              XML_ICONVIEW_ELEMENT + 
							      CString("\" element cannot be handled, wrong \"" + 
							      XML_ICONVIEW_ATTR_SELECTED + "\" attribute specification."), 
							      __exception(XMLPARSE));
			}

			// retreive the orientation property
			CString inOrientation (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_ORIENTATION).GetValue());

			// set it
			if (inOrientation == CString("vertical")) 
				SetOrientation (GTK_ORIENTATION_VERTICAL);
			else
				SetOrientation (GTK_ORIENTATION_HORIZONTAL);

			// set the last numeric properties
			SetEditable   (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_EDITABLE)	  .GetValue().ToBool ());
			SetColumns    (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_COLUMNS)	  .GetValue().ToLong ());
			SetItemWidth  (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_ITEM_WIDTH) .GetValue().ToLong ());
			SetSpacing    (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_SPACING)	  .GetValue().ToULong());
			SetRowSpacing (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_ROW_SPACING).GetValue().ToULong());
			SetColSpacing (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_COL_SPACING).GetValue().ToULong());
			SetMargin     (::xml_node_get_attribute (inXMLNode, XML_ICONVIEW_ATTR_MARGIN)	  .GetValue().ToULong());
		}
		break;
	}
}




