/*
 *  @(#)ITFTestData.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1.autodoc.v1;


import net.sourceforge.groboutils.autodoc.v1.testserver.TestDataFactory;
import net.sourceforge.groboutils.autodoc.v1.testserver.DefaultTestData;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestInfo;

import junit.framework.AssertionFailedError;

import java.util.Vector;

/**
 * An interface used to briefly describe a test and the gathered data associated
 * for the test in a particular framework.  These should be created for
 * a framework through the {@link TestDataFactory} class.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:51:56 $
 * @since      March 30, 2002
 */
public class ITFTestData extends DefaultTestData
{
    private Vector issueIDs = new Vector();
    private Vector errors = new Vector();
    private Vector failures = new Vector();
    private int testCount = 0;
    private long startTime = -1;
    private long endTime = -1;
    
    
    public ITFTestData( TestInfo info )
    {
        super( info );
    }
    
    
    /**
     * This may be called multiple times.  Each time will add a new issue for
     * the corresponding test info.  If <tt>id</tt> is <tt>null</tt>, then it
     * will be ignored.  Likewise, any duplicate IDs passed in will be ignored.
     *
     * @param id the issue id to associate with this test.
     */
    public void addIssueID( String id )
    {
        if (id != null && !this.issueIDs.contains( id ))
        {
            this.issueIDs.addElement( id );
        }
    }
    
    
    /**
     * 
     */
    public void addError( Throwable error )
    {
        if (error != null)
        {
            this.errors.addElement( error );
        }
    }
    
    
    /**
     * 
     */
    public void addFailure( AssertionFailedError failure )
    {
        if (failure != null)
        {
            this.failures.addElement( failure );
        }
    }
    
    
    /**
     * 
     */
    public void addTest()
    {
        ++this.testCount;
    }
    
    
    /**
     * 
     */
    public void setStartTime( long startTime )
    {
        this.startTime = startTime;
    }
    
    
    /**
     * 
     */
    public void setEndTime( long endTime )
    {
        this.endTime = endTime;
    }
    
    
    /**
     * Retrieves all issue IDs for this TestInfo object.  This will never return
     * <tt>null</tt>, but may return an empty array.
     */
    public String[] getIssues()
    {
        String[] issues = new String[ this.issueIDs.size() ];
        this.issueIDs.copyInto( issues );
        return issues;
    }
    
    
    /**
     * 
     */
    public Throwable[] getErrors()
    {
        Throwable t[] = new Throwable[ this.errors.size() ];
        this.errors.copyInto( t );
        return t;
    }
    
    
    /**
     * 
     */
    public AssertionFailedError[] getFailures()
    {
        AssertionFailedError s[] = new AssertionFailedError[
            this.failures.size() ];
        this.failures.copyInto( s );
        return s;
    }
    
    
    /**
     * 
     */
    public int getSuccessCount()
    {
        int c = this.testCount - this.errors.size() - this.failures.size();
        if (c < 0)
        {
            c = 0;
        }
        return c;
    }
    
    
    /**
     * 
     */
    public int getTestCount()
    {
        return this.testCount;
    }
    
    
    /**
     * 
     */
    public long getRunTime()
    {
        if (this.startTime < 0 || this.endTime < 0)
        {
            return -1L;
        }
        return this.endTime - this.startTime;
    }
}

