/*
Copyright 2010 Vincent Carmona
vinc4mai@gmail.com

This file is part of rghk.

    rghk is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    rghk is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ZiK; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <ruby.h>
#include <gdk/gdkx.h>
#include <gdk/gdkevents.h>
#include <X11/Xlib.h>
//xev to see keycode
//take a look at http://www.gnu.org/prep/standards/standards.html

//Modifier Key
enum Modifier
{
	RGHK_SHIFT_MASK=1<<0,	
	RGHK_LOCK_MASK=1<<1,	
	RGHK_CONTROL_MASK=1<<2,	
	RGHK_MOD1_MASK=1<<3,	
	RGHK_MOD2_MASK=1<<4,	
	RGHK_MOD3_MASK=1<<5,	
	RGHK_MOD4_MASK=1<<6,	
	RGHK_MOD5_MASK=1<<7,
	RGHK_NUM_LOCK_MASK=RGHK_MOD2_MASK//Is it true for all keyboards?
	//RGHK_SCROLL_LOCK=??
};

VALUE mGlobalHotKeys;
VALUE mModifier;
VALUE cKeyBinder;

/*
 * call-seq: bind(key, modifier) {block}
 * 
 * key: a key value. Gdk::Keyval constants can be used.
 * 
 * modifier: a GlobalHotKeys::Modifier.
 * 
 * Set a global hotkey.
 * block will be called when the key and the modifier are hitted.
*/
VALUE
kb_bind(VALUE self, VALUE key, VALUE modifier)
{
	GdkWindow *rootwin;
	Display *xdisplay;
	uint keycode;
	/*	uint ignored[]={0, RGHK_LOCK_MASK, RGHK_NUM_LOCK_MASK, RGHK_SCROLL_LOCK,
		RGHK_LOCK_MASK|RGHK_NUM_LOCK_MASK, RGHK_LOCK_MASK|RGHK_SCROLL_LOCK, RGHK_NUM_LOCK_MASK|RGHK_SCROLL_LOCK,
		RGHK_LOCK_MASK|RGHK_NUM_LOCK_MASK|RGHK_SCROLL_LOCK};*/
	uint ignored[]={0, RGHK_LOCK_MASK, RGHK_NUM_LOCK_MASK, RGHK_LOCK_MASK|RGHK_NUM_LOCK_MASK};
	uint mod;

	if (rb_funcall(rb_iv_get(cKeyBinder, "stock"), rb_intern("include?"), 1, self)==Qtrue)
		rb_raise(rb_eException, "KeyBinder allready binded.");

	rb_iv_set(self, "@key", key);
	rb_iv_set(self, "@mod", modifier);
	rb_iv_set(self, "block", rb_block_proc());
	rb_ary_push(rb_iv_get(cKeyBinder, "stock"), self);

	rootwin=gdk_get_default_root_window();//stock this window in variable once for all?
	xdisplay=GDK_WINDOW_XDISPLAY(rootwin);

	keycode=XKeysymToKeycode(xdisplay, FIX2UINT(key));
	if (keycode==0)
		rb_raise(rb_eException, "Invalid key value.");

	if NIL_P(modifier)
		mod=0;
	else
		mod=FIX2UINT(modifier);

	int i;
	for (i=0; i<4 ; i++)
	{
		XGrabKey(xdisplay, keycode, mod|ignored[i],  GDK_WINDOW_XWINDOW(rootwin), False,
			GrabModeAsync, GrabModeAsync);
	}
//XGrabKey(xdisplay, keycode, AnyModifier, GDK_WINDOW_XWINDOW(rootwin), False, GrabModeAsync, GrabModeAsync);
	return Qtrue;
}

/*
 * Unset a global hotkey.
*/
VALUE
kb_unbind(VALUE self)
{
	VALUE del, ret;
	GdkWindow *rootwin;
	Display *xdisplay;
	uint keycode, mod;
	uint ignored[]={0, RGHK_LOCK_MASK, RGHK_NUM_LOCK_MASK, RGHK_LOCK_MASK|RGHK_NUM_LOCK_MASK};

	del=rb_funcall(rb_iv_get(cKeyBinder, "stock"), rb_intern("delete"), 1, self);
	if NIL_P(del)
		return Qfalse;
	
	rootwin=gdk_get_default_root_window();
	xdisplay=GDK_WINDOW_XDISPLAY(rootwin);
	keycode=XKeysymToKeycode(xdisplay, FIX2UINT(rb_iv_get(self, "@key")));
	mod=FIX2UINT(rb_iv_get(self, "@mod"));

	int i;
	for (i=0; i<4 ; i++)
		XUngrabKey(xdisplay, keycode, mod|ignored[i], GDK_WINDOW_XWINDOW (rootwin));
	return Qtrue;
}

/*
 * Unset a all global hotkeys.
*/
VALUE
kb_unbind_all(VALUE self)
{
	rb_iterate(rb_each, rb_iv_get(cKeyBinder, "stock"), kb_unbind, Qnil);
	return Qnil;
}

VALUE
process(VALUE kb_obj, VALUE xkey)
{
//rb_funcall(rb_stdout, rb_intern("puts"), 1, RARRAY(xkey)->ptr[1]);
	uint keycode=XKeysymToKeycode(GDK_WINDOW_XDISPLAY(gdk_get_default_root_window()), FIX2UINT(rb_iv_get(kb_obj, "@key")));

	if (keycode==FIX2UINT(RARRAY(xkey)->ptr[0]))
	{
		uint ignored=RGHK_LOCK_MASK|RGHK_NUM_LOCK_MASK;
		uint mod=FIX2UINT(RARRAY(xkey)->ptr[1])&~ignored;
		uint keymod;

		if NIL_P(rb_iv_get(kb_obj, "@mod"))
			keymod=0;
		else
			keymod=FIX2UINT(rb_iv_get(kb_obj, "@mod"));

		if (keymod==mod)
			rb_funcall(rb_iv_get(kb_obj, "block"), rb_intern("call"), 1, kb_obj);
	}
	return Qnil;
}

GdkFilterReturn
filter_func(GdkXEvent *gdk_xevent, GdkEvent *event, gpointer data)
{
	GdkFilterReturn ret;
	XEvent *xevent=(XEvent*)gdk_xevent;

	if (xevent->type==KeyPress)
	{
		VALUE xev=rb_ary_new3(2, INT2FIX(xevent->xkey.keycode), INT2FIX(xevent->xkey.state));

		rb_iterate(rb_each, rb_iv_get(cKeyBinder, "stock"), process, xev);
	}

	return GDK_FILTER_CONTINUE;
}

// :nodoc: It is called by rghk when it is required.
VALUE
mInit(VALUE self)
{
	gdk_window_add_filter(gdk_get_default_root_window(), filter_func, NULL);
	return Qtrue;
}

void
Init_rghk()
{
/*
 * rghk provides a simple way to set global hotkeys.
 * 
 * Example:
 *   kb=GlobalHotKeys::KeyBinder.new
 *   kb.bind(Gdk::Keyval::GDK_a,  GlobalHotKeys::Modifier::CONTROL_MASK){puts 'Binded'}
 *   kb.unbind
 * 
 * It probably a good idea to require 'gtk2' before 'rghk'.
 * The librairy rghk  was not tested without ruby/gtk2.
*/
	mGlobalHotKeys=rb_define_module("GlobalHotKeys");
	rb_define_module_function(mGlobalHotKeys, "init", mInit, 0);

	mModifier=rb_define_module_under(mGlobalHotKeys, "Modifier");
	rb_define_const(mModifier, "SHIFT_MASK", INT2FIX(RGHK_SHIFT_MASK));
	rb_define_const(mModifier, "CONTROL_MASK", INT2FIX(RGHK_CONTROL_MASK));
	rb_define_const(mModifier, "MOD1_MASK", INT2FIX(RGHK_MOD1_MASK));//Alt
	//rb_define_const(mModifier, "MOD2_MASK", INT2FIX(RGHK_MOD2_MASK));//NUM_LOCK
	rb_define_const(mModifier, "MOD3_MASK", INT2FIX(RGHK_MOD3_MASK));
	rb_define_const(mModifier, "MOD4_MASK", INT2FIX(RGHK_MOD4_MASK));
	rb_define_const(mModifier, "MOD5_MASK", INT2FIX(RGHK_MOD5_MASK));//AltGr?

	cKeyBinder=rb_define_class_under(mGlobalHotKeys, "KeyBinder", rb_cObject);
	rb_iv_set(cKeyBinder, "stock", rb_ary_new());
	rb_define_method(cKeyBinder, "bind", kb_bind, 2);
	rb_define_method(cKeyBinder, "unbind", kb_unbind, 0);
	rb_define_module_function(cKeyBinder, "unbind_all", kb_unbind_all,0);
}
