/*--------------------------------------------------------------------------+
$Id: XMLReader.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.xml;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import edu.tum.cs.commons.assertion.CCSMPre;

/**
 * Utility class for reading XML documents. Please consult test case
 * {@link XMLReaderTest} to see how this class is intended to be used.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 069D289898A054917CFA09EFB65AAE80
 */
public abstract class XMLReader<E extends Enum<E>, A extends Enum<A>, X extends Exception> {

	/** The current DOM element. */
	private Element currentDOMElement;

	/** The schema URL to use or <code>null</code> if no schema is used. */
	private final URL schemaURL;

	/** Resolver used by the writer. */
	private final IXMLResolver<E, A> xmlResolver;

	/** File to parse. */
	private final File file;

	/** XML encoding of the file. */
	private final String encoding;

	/**
	 * Create new reader.
	 * 
	 * @param file
	 *            the file to be read
	 * @param xmlResolver
	 *            resolvers used by this reader
	 */
	public XMLReader(File file, IXMLResolver<E, A> xmlResolver) {
		this(file, null, null, xmlResolver);
	}

	/**
	 * Create reader.
	 * 
	 * @param file
	 *            the file to be read
	 * @param encoding
	 *            XML encoding of the file. No encoding is set if
	 *            <code>null</code>.
	 * @param xmlResolver
	 *            resolvers used by this reader
	 */
	public XMLReader(File file, String encoding, IXMLResolver<E, A> xmlResolver) {
		this(file, encoding, null, xmlResolver);
	}

	/**
	 * Create reader.
	 * 
	 * @param file
	 *            the file to be read
	 * @param schemaURL
	 *            the URL pointing to the schema that is used for validation. No
	 *            validation will be performed if <code>null</code>.
	 * @param xmlResolver
	 *            resolvers used by this reader
	 */
	public XMLReader(File file, URL schemaURL, IXMLResolver<E, A> xmlResolver) {
		this(file, null, schemaURL, xmlResolver);
	}

	/**
	 * Create reader.
	 * 
	 * @param file
	 *            the file to be read
	 * @param encoding
	 *            XML encoding of the file. No encoding is set if
	 *            <code>null</code>.
	 * @param schemaURL
	 *            the URL pointing to the schema that is used for validation. No
	 *            validation will be performed if <code>null</code>.
	 * @param xmlResolver
	 *            resolvers used by this reader
	 */
	public XMLReader(File file, String encoding, URL schemaURL,
			IXMLResolver<E, A> xmlResolver) {
		CCSMPre.isFalse(file == null, "File may not be null.");
		CCSMPre.isFalse(xmlResolver == null, "XML resolver may not be null.");
		this.file = file;
		this.encoding = encoding;
		this.schemaURL = schemaURL;
		this.xmlResolver = xmlResolver;
	}

	/**
	 * Get <code>boolean</code> value of an attribute.
	 * 
	 * @return the boolean value, semantics for non-translatable or
	 *         <code>null</code> values is defined by
	 *         {@link Boolean#valueOf(String)}.
	 */
	protected boolean getBooleanAttribute(A attribute) {
		String value = getStringAttribute(attribute);
		return Boolean.valueOf(value);
	}

	/**
	 * Get the text content of a child element of the current element.
	 * 
	 * @param childElement
	 *            the child element
	 * @return the text or <code>null</code> if the current element doesn't have
	 *         the requested child element
	 */
	protected String getChildText(E childElement) {

		Element domElement = getChildElement(childElement);
		if (domElement == null) {
			return null;
		}

		return domElement.getTextContent();
	}

	/**
	 * Translate attribute value to an enumeration element.
	 * 
	 * @param attribute
	 *            the attribute
	 * @param enumClass
	 *            the enumeration class
	 * 
	 * @return the enum value, semantics for non-translatable or
	 *         <code>null</code> values is defined by
	 *         {@link Enum#valueOf(Class, String)}.
	 */
	protected <T extends Enum<T>> T getEnumAttribute(A attribute,
			Class<T> enumClass) {
		String value = getStringAttribute(attribute);
		return Enum.valueOf(enumClass, value);
	}

	/**
	 * Get <code>int</code> value of an attribute.
	 * 
	 * @return the int value, semantics for non-translatable or
	 *         <code>null</code> values is defined by
	 *         {@link Integer#valueOf(String)}.
	 */
	protected int getIntAttribute(A attribute) {
		String value = getStringAttribute(attribute);
		return Integer.valueOf(value);
	}

	/**
	 * Get <code>long</code> value of an attribute.
	 * 
	 * @return the int value, semantics for non-translatable or
	 *         <code>null</code> values is defined by
	 *         {@link Integer#valueOf(String)}.
	 */
	protected long getLongAttribute(A attribute) {
		String value = getStringAttribute(attribute);
		return Long.valueOf(value);
	}

	/**
	 * Get attribute value.
	 * 
	 * 
	 * @return the attribute value or <code>null</code> if attribute is
	 *         undefined.
	 */
	protected String getStringAttribute(A attribute) {
		return currentDOMElement.getAttribute(xmlResolver
				.resolveAttributeName(attribute));
	}

	/**
	 * Get text content of current node.
	 */
	protected String getText() {
		return currentDOMElement.getTextContent();
	}

	/**
	 * Parse file. This sets the current element focus to the document root
	 * element. If schema URL was set the document is validated against the
	 * schema.
	 * <p>
	 * Sub classes should typically wrap this method with a proper error
	 * handling mechanism.
	 * 
	 * @throws SAXException
	 *             if a parsing exceptions occurs
	 * @throws IOException
	 *             if an IO exception occurs.
	 */
	protected void parseFile() throws SAXException, IOException {

		FileInputStream stream = new FileInputStream(file);

		try {

			InputSource input = new InputSource(stream);
			if (encoding != null) {
				input.setEncoding(encoding);
			}

			Document document;
			if (schemaURL == null) {
				document = XMLUtils.parse(input);
			} else {
				document = XMLUtils.parse(input, schemaURL);
			}
			currentDOMElement = document.getDocumentElement();
		} finally {
			stream.close();
		}
	}

	/**
	 * Process the child elements of the current element with a given processor.
	 * Target elements are specified by
	 * {@link IXMLElementProcessor#getTargetElement()}.
	 * 
	 * @param processor
	 *            the processor used to process the elements
	 * @throws X
	 *             if the processor throws an exception
	 */
	protected void processChildElements(IXMLElementProcessor<E, X> processor)
			throws X {
		String targetElementName = xmlResolver.resolveElementName(processor
				.getTargetElement());
		processElementList(processor, getChildElements(targetElementName));
	}

	/**
	 * Process all descendant elements of the current element with a given
	 * processor. In contrast to
	 * {@link #processChildElements(IXMLElementProcessor)}, not only direct
	 * child elements are processed. Descendant elements are processed in the
	 * sequence they are found during a top-down, left-right traversal of the
	 * XML document.
	 * <p>
	 * Target elements are specified by
	 * {@link IXMLElementProcessor#getTargetElement()}.
	 * 
	 * @param processor
	 *            the processor used to process the elements
	 * @throws X
	 *             if the processor throws an exception
	 */
	protected void processDecendantElements(IXMLElementProcessor<E, X> processor)
			throws X {
		String targetElementName = xmlResolver.resolveElementName(processor
				.getTargetElement());

		NodeList descendantNodes = currentDOMElement
				.getElementsByTagName(targetElementName);

		processElementList(processor, XMLUtils.elementNodes(descendantNodes));
	}

	/**
	 * Processes the elements in the list with the given processor
	 * 
	 * @param processor
	 *            the processor used to process the elements
	 * @param elements
	 *            list of elements that get processed
	 * @throws X
	 *             if the processor throws an exception
	 */
	private void processElementList(IXMLElementProcessor<E, X> processor,
			List<Element> elements) throws X {
		Element oldElement = currentDOMElement;

		for (Element child : elements) {
			currentDOMElement = child;
			processor.process();
		}

		currentDOMElement = oldElement;
	}

	/**
	 * Get the first child element of the the specified type of the current DOM
	 * element.
	 * 
	 * @param elementType
	 *            the desired element type
	 * @return the child element or <code>null</code> if not present.
	 */
	private Element getChildElement(E elementType) {

		NodeList nodeList = currentDOMElement.getChildNodes();
		String elementName = xmlResolver.resolveElementName(elementType);

		for (int i = 0; i < nodeList.getLength(); i++) {
			Node child = nodeList.item(i);
			if (isTargetElement(child, elementName)) {
				return (Element) child;
			}
		}

		return null;
	}

	/**
	 * Get all child elements of the current element with the given name.
	 * 
	 * @param targetElementName
	 *            the name of the target element
	 * @return list of elements
	 */
	private List<Element> getChildElements(String targetElementName) {
		NodeList nodeList = currentDOMElement.getChildNodes();
		ArrayList<Element> list = new ArrayList<Element>();

		for (int i = 0; i < nodeList.getLength(); i++) {
			Node child = nodeList.item(i);

			if (isTargetElement(child, targetElementName)) {
				list.add((Element) nodeList.item(i));
			}
		}

		return list;
	}

	/** Checks if a node is a target elements */
	private boolean isTargetElement(Node child, String targetElementName) {
		return child.getNodeType() == Node.ELEMENT_NODE
				&& child.getLocalName().equals(targetElementName);
	}
}