/*
* libcafix - communicates with CASIO graphing calculators.
* Copyright (C) 2001  Gran Weinholt <weinholt@linux.nu>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* $Id: header.c,v 1.3 2001/09/07 17:57:27 weinholt Exp $
*/
#include <cafix.h>
#include <string.h>
#include <stdlib.h>

static char rcsid[] = "$Id: header.c,v 1.3 2001/09/07 17:57:27 weinholt Exp $\n";

void cafix_hdr_generate(cafixdata * data)
{
    /*
     * Generate CRC and raw header.
     */
    int i;
    char crc;

    /* FIXME: maybe implement a feature that just skips the generation
       of the header if the crc is correct and the data is correct or
       the datatype/format is unknown. This way we should be able
       to send data we didn't even knew existed. */

    /* Clear the header */
    memset(data->rawheader, 255, sizeof(data->rawheader));
    data->hlength = 49;	/* default length of header */

    /* Generate the header based on type */
    switch (data->format) {
    case FMT_PROGRAM:
        memcpy(data->rawheader, "TXT\0PG\0\0", 8);
	data->rawheader[8] = data->length / 256;
	data->rawheader[9] = (data->length % 256) + 2;
        strncpy(data->rawheader + 10, data->name,
	    strlen(data->name) > 8 ? 8 : strlen(data->name));
	data->rawheader[34] = 'N';
	data->rawheader[35] = 'L';
	break;
    case FMT_BACKUP:
	memcpy(data->rawheader, data->data + 2048, 48);
	memcpy(data->rawheader + 33, "\0\x10\0\0\0\0", 6);
	break;
    case FMT_END:
	memcpy(data->rawheader, "END", 3);
	break;
    case FMT_VARIABLE:
	memcpy(data->rawheader, "VAL\0", 4);
	memcpy(data->rawheader + 18, "VariableR\x0a", 10);
	memcpy(data->rawheader + 10, data->name,
	   strlen(data->name) > 8 ? 8 : strlen(data->name));
	data->status = STAT_NOT_IMPLEMENTED;
	/* XXX: implement more datatypes here */
	if (data->datatype == DATA_VM) {
	    memcpy(data->rawheader + 4, "VM", 2);
	    memcpy(data->rawheader + 6, "\0\1\0\1", 4);
	}
	break;
    default:
	data->status = STAT_NOT_IMPLEMENTED;
	break;
    } /* XXX: implement more headers */

    /* Generate the CRC for the header */
    for (crc = i = 0; i < data->hlength - 1; i++)
	crc = (crc + data->rawheader[i]) % 256;
    data->rawheader[data->hlength - 1] = (0 - crc) % 256;

    /*printf("We generated a header:\n");
    for (i=0;i<data->hlength;i++)
	printf("%d%c ",(signed char) data->rawheader[i], data->rawheader[i]);
    printf("\n");*/
}

/* Try to find out what this is using only the 7 first chars...
   Called from cafic_hdr_receive() in comm.c */
void cafix_hdr_get_format(cafixdata *data)
{
    /* Read cafix.h for a description of these. */
    data->hlength = 49;
    if (memcmp(&data->rawheader, "MEM", 3) == 0)
	data->format = FMT_BACKUP;
    if (memcmp(&data->rawheader, "DD@", 3) == 0) {
	data->format = FMT_BWDUMP;
	data->hlength = 39;
    }
    if (memcmp(&data->rawheader, "DC@", 3) == 0) {
	data->format = FMT_CDUMP;
	data->hlength = 39;
    }
    if (memcmp(&data->rawheader, "FNC", 3) == 0)
	data->format = FMT_FUNCTION;
    if (memcmp(&data->rawheader, "IMG", 3) == 0)
	data->format = FMT_PICTURE;
    if (memcmp(&data->rawheader, "TXT", 3) == 0)
	data->format = FMT_PROGRAM;
    if (memcmp(&data->rawheader, "VAL", 3) == 0)
	data->format = FMT_VARIABLE;
    if (memcmp(&data->rawheader, "REQ", 3) == 0)
	data->format = FMT_REQUEST;
    if (memcmp(&data->rawheader, "END", 3) == 0)
	data->format = FMT_END;

    /* Now test for datatype */    
    if (memcmp(&data->rawheader[4], "BU", 2) == 0)
	data->datatype = DATA_BU;
    if (memcmp(&data->rawheader[4], "FT", 2) == 0)
	data->datatype = DATA_FT;
    if (memcmp(&data->rawheader[4], "GF", 2) == 0)
	data->datatype = DATA_GF;
    if (memcmp(&data->rawheader[4], "GM", 2) == 0)
	data->datatype = DATA_GM;;
    if (memcmp(&data->rawheader[4], "LT", 2) == 0)
	data->datatype = DATA_LT;
    if (memcmp(&data->rawheader[4], "MT", 2) == 0)
	data->datatype = DATA_MT;
    if (memcmp(&data->rawheader[4], "PC", 2) == 0)
	data->datatype = DATA_PC;
    if (memcmp(&data->rawheader[4], "PG", 2) == 0)
	data->datatype = DATA_PG;
    if (memcmp(&data->rawheader[4], "RF", 2) == 0)
	data->datatype = DATA_RF;
    if (memcmp(&data->rawheader[4], "RR", 2) == 0)
	data->datatype = DATA_RR;
    if (memcmp(&data->rawheader[4], "SE", 2) == 0)
	data->datatype = DATA_SE;
    if (memcmp(&data->rawheader[4], "TR", 2) == 0)
	data->datatype = DATA_TR;
    if (memcmp(&data->rawheader[4], "VM", 2) == 0)
	data->datatype = DATA_VM;
    if (memcmp(&data->rawheader[4], "WD", 2) == 0)
	data->datatype = DATA_WD;

    /* I borrowed a CFX-9800G from a friend... CASIOs have changed */
    if (memcmp(&data->rawheader, "BUTYPEA02", 9) == 0) {
	/* A backup from a CFX-9800G, I think :) */
	data->hlength = 39;
	data->format = FMT_BACKUP;
	data->datatype = DATA_9800_TYPEA02;
    }
    if (memcmp(&data->rawheader, "EN", 2) == 0) {
	/* A program but not a program? 9800 magic. */
	data->hlength = 39;
	data->format = FMT_EDITOR;
    }
    if (data->rawheader[0] == 'P' && data->rawheader[2] == '\0' && data->rawheader[3] == '\0') {
	/* A program that lives in P0,P1,... I think. */
	data->hlength = 39;
	data->format = FMT_PROGRAM;
	data->datatype = DATA_9800_PZ;
    }
}

void cafix_hdr_translate(cafixdata *data)
{
    /* 
     * Extract data to header, then calculate CRC.
     */
    int i;

/*    printf("We have a header:\n");
    for (i = 0; i < data->hlength; i++)
	printf("%d%c ", (signed char) data->rawheader[i], (signed char) data->rawheader[i]);
    printf("\n");*/

    /* XXX: i wonder if there isn't an easier (more general) way to get
       the length */
    switch (data->format) {
    case FMT_REQUEST:
	data->length = 0;
	data->unstackable = 1;
	break;
    case FMT_BACKUP:
	if (data->datatype == DATA_9800_TYPEA02)
	    data->length = 32768;
	else
	    data->length = (unsigned char)data->rawheader[8] * 256 + (unsigned char)data->rawheader[9] - 2;
	data->unstackable = 1;
	break;
    case FMT_END:
	data->length = 0;
	data->unstackable = 1;
	break;
    case FMT_VARIABLE:
        data->length = data->rawheader[8] * 256 + (unsigned char)data->rawheader[9] - 2;
	if (data->datatype == DATA_VM)
	    data->length = 14;	// XXX: sometimes it's 48?
	/* FIXME: search for DATA_MT and fix all matrix code */
	if (data->datatype == DATA_MT)
	    data->length = 2*(data->rawheader[7]+data->rawheader[9])/*-24*/;
	break;
    case FMT_PROGRAM:
	if (data->datatype == DATA_9800_PZ) {
	    if (data->rawheader[1] == 'Z')
		data->length = 242;	/* Probably ALL wrong, look in comm.c */
	    else
		data->length = data->rawheader[4]-2;	/* FIXME: obviously wrong */
	} else
	    data->length = data->rawheader[8] * 256 + (unsigned char)data->rawheader[9] - 2;
	break;
    case FMT_CDUMP:
	data->length = 3075;
	data->unstackable = 1;
	break;
    case FMT_BWDUMP:
	data->length = 10000;
	data->unstackable = 1;
	break;
    case FMT_UNKNOWN:
    default:
	// FIXME: this isn't universal
        data->length = data->rawheader[8] * 256 + (unsigned char)data->rawheader[9] - 2;
	break;
    }

    /* Extract name from the raw header. */
    if ((data->name = calloc(1, 10)) == NULL) {
	data->status = STAT_ERROR_MEMORY;
	return;
    }
    
    /* Get the name, this is pretty standard on the 9x50. */
    for (i = 0; i < 8; i++)
	if ((unsigned char)data->rawheader[10+i] != 0xff)
	    strncat(data->name, data->rawheader + 10 + i, 1);

    if (data->datatype == DATA_9800_PZ)
	strncat(data->name, data->rawheader, 2);

    /* We don't want to allocate negative memory later on :) */
    if (data->length < 0)
	data->length = 0;	/* XXX: is this really needed? */
}
