// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Collections.Generic;

namespace Anculus.Core
{
	public static class Translation
	{
		private static ITranslationBackend _backend;
		private static bool _hasBackend;
		
		public static ITranslationBackend Backend
		{
			get { return _backend; }
			set {
				if (value == null)
					throw new ArgumentNullException ("Backend");
			
				_backend = value;
				_hasBackend = true;
			}
		}

		public static string GetString (string key)
		{
			if (key == null)
				throw new ArgumentNullException ("key");

			if (_hasBackend)
				return _backend.GetString (key);
			return key;
		}

		public static string GetString (string key, params object[] args)
		{
			if (key == null)
				throw new ArgumentNullException ("key");

			if (_hasBackend)
				return _backend.GetString (key, args);
			return String.Format (key, args);
		}

		public static string GetPluralString (string key, string plural, int n)
		{
			if (key == null)
				throw new ArgumentNullException ("key");
			if (plural == null)
				throw new ArgumentNullException ("plural");
			if (n < 0)
				throw new ArgumentException ("n");

			if (_hasBackend) {
				return _backend.GetPluralString (key, plural, n);
			} else {
				if (n == 1)
					return key;
				else
					return plural;
			}
		}

		public static string GetPluralString (string key, string plural, int n, params object[] args)
		{
			if (key == null)
				throw new ArgumentNullException ("key");
			if (plural == null)
				throw new ArgumentNullException ("plural");
			if (n < 0)
				throw new ArgumentException ("n");

			if (_hasBackend) {
				return _backend.GetPluralString (key, plural, n, args);
			} else {
				if (n == 1)
					return String.Format (key, args);
				else
					return String.Format (plural, args);
			}
		}
	}
}