/*
 Copyright 2004, Revolution Linux Inc., Nicolas Dufresne, Yves Trudeau
 Copyright 2006, CRIM, Stephane Bond

 This file is part of the MILLE-XTERM distribution.
 See the MILLE-XTERM (english) and/or the MILLE (french) project web site

 http://www.revolutionlinux.com/mille-xterm/
 http://www.mille.ca/

 The MILLE-XTERM framework is covered by the GNU General Public License. See
 the COPYING file in the top-level MILLE-XTERM directory. Software packages
 that are included in the MILLE-XTERM distribution have their own licenses.

 -------------------------------------------------------------------------

 This script creates the tables needed for the Mille-XTERM configurator.

 Revisions history:
	- 13 jul 2004: Creation
  - 10 feb 2006: Major update, nodes structure optimisation, support for permissions and inventory

 Last changes :
 Release 1.0
 - Add hwgroups column to status
 - Add tables for inventory and hwgroups : computers, computershw, pciids, hwgroupsrule
 - Add permissions table
 - Add attributesDefDesc for localized strings
 - Add attributeClass to the attributesDef table
 - Add nested set algorithme (leftval, rightval and triggers) to improve queries on nodes tree
 - Support for different node type : change isnode to nodetype and add reference column to the nodes table
*/

/* The nodes table contains the tree of nodes, an entry
   can be a node or a leaf depending of the value of nodeType.
   If the entry is a node, the fields MAC and reference are not used.
   id_parent is a circular reference to id.

 - id_parent is the ancestor in the tree. For the root node, id_parent = id
 - leftval and rightval, used for the nested set algorithm
 - name is the name of the node
 - nodetype take the following values :
   0 = node : a node in the tree, can contain childs nodes ans leafs
   1 = computer : leaf with mac address
   2 = hwgroup : hardware group defined with a set of rules
   3 = user : user present in an external system (reference field contains username)
   4 = group : group present in an external system (reference field contains groupname)
 - MAC is the mac address of the node when nodetype = 1
 - reference is the key of the entity in the external system
*/

CREATE TABLE nodes (
	id SERIAL PRIMARY KEY,
	id_parent INTEGER NOT NULL REFERENCES nodes (id),
	leftval INTEGER NOT NULL DEFAULT 0,
	rightval INTEGER NOT NULL DEFAULT 0,
	name VARCHAR(50) DEFAULT '',
	nodetype INTEGER DEFAULT 0,
	mac VARCHAR(17) DEFAULT '00:00:00:00:00:00',
	reference VARCHAR(254)
	CONSTRAINT invalidName CHECK (nodetype <> 0 OR LENGTH(name) > 0)
  CONSTRAINT invalidMac CHECK (nodetype <> 1 OR mac::text > '00:00:00:00:00:00'::text)
  CONSTRAINT invalidReference CHECK ((nodetype <> 2 AND nodetype <> 3) OR LENGTH(reference) > 0)
  CONSTRAINT invalidType CHECK (nodetype >= 0 AND nodetype <= 4)
);
CREATE INDEX nodes_name_IDX ON nodes (name);
CREATE INDEX nodes_mac_IDX ON nodes (mac);
CREATE INDEX nodes_reference_IDX ON nodes (reference);
CREATE INDEX nodes_nodetype_IDX ON nodes (nodetype);
CREATE INDEX nodes_leftrightval_IDX ON nodes (leftval, rightval);
CREATE INDEX nodes_id_parent ON nodes (id_parent, id);
CREATE UNIQUE INDEX nodes_reference_type_unique_IDX ON nodes (nodetype,reference);

-- nodes needs a root node_group with id = 0
INSERT INTO nodes (id,id_parent,name) VALUES (0,0,'RootNode');
INSERT INTO nodes (id,id_parent,name) VALUES (-1,0,'Trash');



/* The attributesDef table contains the definition of the attributes

 - name is the name of the attribute
 - attributeClass is the category of attribute (i.e. "ltsp", "profile", "inventory", ...)
 - attributeType is the type of attribute:
    0 = String with mask (default),
    1 = list,
    2 = selection
   When attributeType = 1 (list), the possible values are found in attributesDefDict
   List of facultatives values can be present in attributesDefDict when attributeType = 0
 - mask is the mask to apply to the value in regular expression
 - ediatble : can be edited online
*/
CREATE TABLE attributesDef (
  id SERIAL PRIMARY KEY,
  name VARCHAR(50) UNIQUE,
  attributeClass VARCHAR(50) DEFAULT 'ltsp',
  attributeType INTEGER DEFAULT 0 CHECK ( attributeType >=0 AND attributeType <= 2),
  mask VARCHAR(254) DEFAULT E'^[^\']{0,254}$',
  editable INTEGER DEFAULT 0 CHECK (editable = 1 OR editable = 0));
CREATE INDEX attributesDef_name_IDX ON attributesDef (name);



/* The attributesDefDesc table contains localized descriptions of attributes

 - attributesdef_id reference attributesDef
 - locale is the language code of the string (i.e. "en_US", "fr_CA")
 - locname is the localizen name of the attribute
 - desc is a long description of the attribute. Can be used for contextual help
*/
CREATE TABLE attributesDefDesc (
  attributesdef_id INTEGER NOT NULL REFERENCES attributesDef (id),
  locale CHAR(5) NOT NULL,
  locname VARCHAR(50),
  description VARCHAR(254),
  PRIMARY KEY (attributesdef_id, locale));



/* The attributes table contains the attributes of a group or a node

 - nodes_id is the nodes owner of this attributes
 - attributesDef_id is the definition of the attributes
 - value is the actual value of the attributes
 - cumulative, in multiselect mode, cumulate values of parents instead of replacing it
*/
CREATE TABLE attributes (
	nodes_id INTEGER NOT NULL REFERENCES nodes (id),
	attributesDef_id INTEGER NOT NULL REFERENCES attributesDef (id),
	value VARCHAR(254),
	cumulative INTEGER DEFAULT 0 CHECK (cumulative = 1 OR cumulative = 0),
	PRIMARY KEY (nodes_id, attributesDef_id));
CREATE INDEX attributes_value_IDX ON attributes (value,attributesDef_id);



/* The attributesDefDict contains the possible values of an attributes when
   its attributeType = 1 (list)

 - attributesDef_id is the attribute definition of type list owner of the value
 - sortval is a hint for sorting the list
*/
CREATE TABLE attributesDefDict (
	attributesDef_id INTEGER NOT NULL REFERENCES attributesDef (id),
	value VARCHAR(254),
	sortval INTEGER,
	PRIMARY KEY (attributesDef_id,value));
CREATE INDEX attributesDefDict_attributes_id_IDX ON attributesDefDict (attributesDef_id);



/* The attributesSelect contains results from a type 2 attributesType (muliselect list)

	- nodes_id references an node attributes
	- attributesDef_id references attributes definition
	- attributesDefDict_id references an element from attributes definition dictionary

*/
CREATE TABLE attributesSelect (
	nodes_id INTEGER,
	attributesDef_id INTEGER,
	value VARCHAR(254),
	FOREIGN KEY (nodes_id,attributesDef_id) REFERENCES attributes (nodes_id,attributesDef_id),
	FOREIGN KEY (attributesDef_id,value) REFERENCES attributesDefDict (attributesDef_id,value) ON DELETE cascade ON UPDATE cascade,
	PRIMARY KEY (nodes_id,attributesDef_id,value));



/* The permissions table store administration privileges on the structure

 - granteetype
   1 = username
   2 = groupname
 - granteekey is the user or group name who have permissions
 - attributes specify an attribute name or pattern affected by permissions. Example :
   "" = permissions on tree modifications (edit hierarchy, add child, ...)
   "*" = permissions on all attributes
   "ltsp.*" = permission on all ltsp attributes
   "SECRREN_01" = permission only on attribute "SECRREN_01"
   "SECRREN_*" = all attributes begining with "SECRREN_"
 - scope of the permission
   0 = the current node only
   1 = the current node and his subtree
 - canadd new attribute / child node
 - candelete existing attribute  / child node
 - canread attribute / child node
 - canwrite allow attributes modifications

If they are permissions set at different level of hierarchy, the closest permission will be apply
If a user obtain two permission on the same node from two different groups, the most privilegious permission will be selected.
*/
CREATE TABLE permissions (
  nodes_id INTEGER NOT NULL REFERENCES nodes (id),
  granteetype INTEGER NOT NULL CHECK (granteetype = 1 OR granteetype = 2),
  granteekey VARCHAR(254) NOT NULL,
  attributes VARCHAR(50) NOT NULL,
  scope INTEGER CHECK (scope = 1 OR scope = 0),
  canadd INTEGER CHECK (canadd = 1 OR canadd = 0),
  candelete INTEGER CHECK (candelete = 1 OR candelete = 0),
  canread INTEGER CHECK (canread = 1 OR canread = 0),
  canwrite INTEGER CHECK (canwrite = 1 OR canwrite = 0),
  PRIMARY KEY (nodes_id, granteetype, granteekey, attributes));
CREATE INDEX permissions_granteekey_IDX ON permissions (granteekey);



/* The status table store actual status of leaves.

 - mac, the mac address of the node accessing the configurator (mandatory)
 - nodeIpAdd, the ip address of the node accessing the configurator if available
 - nfsIp, the ip address of the nfs server
 - xServIp, the ip address of the X server (needed for code = 2 and 3)
 - display, the X display number of the leave accessing the configurator
   if available
 - username, the name of the user on the node accessing the configurator
   if available
 - code, the request asked to the configurator. Currently supported values
   are:
	  1 = Node asks for config (giving ip and nfsIp)
	  2 = Starting X session for display manager
	  3 = Starting X session for user session
	  4 = Logoff
 - hwgroups, list of currently used hardware groups separated by coma (field computed on the code 1 request)
 - publickey, public key of active session
 - termAccess, Access level on terminals. Currently supported values are:
      0 = None
      1 = Can access terminals of same node (FireWall rule)
      2 = Can open session on terminals of same node (ssh key)
      3 = Can open session on all terminals (for remote control assistance)
 - lastBootServIp, timestamp of last nfsIp modification
 - lastAppServIp, timestamp of last nfsIp modification
*/
CREATE TABLE status (
	id INTEGER NOT NULL PRIMARY KEY REFERENCES nodes (id),
	mac VARCHAR(17) NOT NULL CONSTRAINT macNotUnique UNIQUE,
	ip VARCHAR(15) NOT NULL CONSTRAINT ipNotUnique UNIQUE,
	bootServIp VARCHAR(15) NOT NULL,
	appServIp VARCHAR(15) DEFAULT '0.0.0.0',
	display VARCHAR(50) DEFAULT '-1',
	username VARCHAR(50) DEFAULT '',
	code INTEGER NOT NULL CHECK (code > 0 AND code <= 4),
	bootServRetry INTEGER DEFAULT 0,
	appServRetry INTEGER DEFAULT 0,
	hwgroups VARCHAR(254) DEFAULT '',
	termAccess INTEGER DEFAULT 0,
	publickey TEXT DEFAULT '',
	ts_lastBootServIp TIMESTAMP DEFAULT 'now',
	ts_lastAppServIp TIMESTAMP DEFAULT '-infinity'
	CONSTRAINT invalidTermAccess CHECK (termAccess = 0 OR (code = 3 AND termAccess >=0 AND termAccess <=3))
	CONSTRAINT invalidAppServIp CHECK (code = 1 OR (code > 1 AND appServIP > '0.0.0.0'))
	CONSTRAINT invalidDisplay CHECK (code = 1 OR code= 2 OR (code > 2 AND LENGTH(display) > 0 AND display != '-1'))
	CONSTRAINT invalidUsername CHECK (code < 3 OR (code >= 3 AND LENGTH(username) > 0))
	CONSTRAINT notALeave CHECK (mac > '00:00:00:00:00:00'));
CREATE INDEX status_id_IDX ON status (id);
CREATE INDEX status_ip_IDX ON status (ip);
CREATE INDEX status_appServIp_IDX ON status (appServIp);
CREATE INDEX status_bootServIp_IDX ON status (bootServIp);
CREATE INDEX status_username_IDX ON status (username);
CREATE INDEX status_code_IDX ON status (code);



/* The log table store the log of access to the configurator.

 - mac, the mac address of the node accessing the configurator (mandatory)
 - ip, the ip address of the node accessing the configurator if available
 - bootServIp, the ip address of the nfs server
 - appServIp, the ip address of the X server (needed for code = 2 and 3)
 - display, the X display number of the leave accessing the configurator
   if available
 - username, the name of the user on the node accessing the configurator
   if available
 - hwgroups, list of currently used hardware groups separated by coma (field computed on the code 1 request)
 - code, the request asked to the configurator. Currently supported values
   are:
	  1 = Node asks for config (giving ip, bootServIp)
	  2 = Starting X session for display manager
	  3 = Starting X session for user session
	  4 = Logout X session
*/
CREATE TABLE log (
	id_key SERIAL PRIMARY KEY,
	id INTEGER NOT NULL,
	mac VARCHAR(17) NOT NULL,
	ip VARCHAR(15) NOT NULL,
	bootServIp VARCHAR(15) NOT NULL,
	appServIp VARCHAR(15) DEFAULT '0.0.0.0',
	display INTEGER DEFAULT -1,
	username VARCHAR(50) DEFAULT '',
	code INTEGER NOT NULL CHECK (code > 0 AND code <= 4),
	bootServRetry INTEGER DEFAULT 0,
	appServRetry INTEGER DEFAULT 0,
	hwgroups VARCHAR(254) DEFAULT '',
	ts TIMESTAMP DEFAULT 'now'
	CONSTRAINT invalidAppServIp CHECK (code = 1 OR (code > 1 AND appServIP > '0.0.0.0'))
	CONSTRAINT invalidDisplay CHECK (code = 1 OR code= 2 OR (code > 2 AND display != '-1'))
	CONSTRAINT invalidUsername CHECK (code < 3 OR (code >= 3 AND LENGTH(username) > 0))
	CONSTRAINT notALeave CHECK (mac > '00:00:00:00:00:00'));
CREATE INDEX log_mac_IDX ON log (mac);
CREATE INDEX log_ip_IDX ON log (ip);
CREATE INDEX log_appServIp_IDX ON log (appServIp);
CREATE INDEX log_bootServIp_IDX ON log (bootServIp);
CREATE INDEX log_username_IDX ON log (username);
CREATE INDEX log_TS_IDX ON log (ts);



/* Table computershw contains list of hardware specification

 - nodes_id is a reference to a node of type 1 (computer)
 - hwkey is the identifier of the hardware attribute (i.e. "cpufreq", "pcidevice", "memsize", "biosvendor", "cdrom", ...)
 - hwvalue is the value of the hardware attribute
*/
CREATE TABLE computershw (
  nodes_id INTEGER REFERENCES nodes (id) ON DELETE cascade,
  hwkey VARCHAR(50),
  hwvalue VARCHAR(254),
  PRIMARY KEY (nodes_id, hwkey, hwvalue));



/* Table hwgroupsrule contains a rule

 - nodes_id is a reference to a node of type 4 (hwgroup)
 - hwkey is the key used for the rule
 - hwvalue is the value to compare at the rule processing
    "*" = any value
 - operator, support theses values :
    "=" = a computer must have the exact key/value to be member
    ">" = a computer must have the attribute with a value ">" then
    "<" = a computer must have the attribute with a value "<" then
    "~" = a computer must not have the attribute with the specified value

A hwgroup node can habe multiple rules. A computer must respect all rules to be a member of the hwgroup.
*/
CREATE TABLE hwgroupsrule (
  nodes_id INTEGER REFERENCES nodes (id) ON DELETE cascade,
  hwkey VARCHAR(50),
  hwvalue VARCHAR(254),
  operator CHAR(2),
  PRIMARY KEY (nodes_id, hwkey, hwvalue));



/* Table hwkeysdesc contains localized description of hwkeys

 - hwkey, the reference to the hwkey
 - locale is the language code of the string (i.e. "en_US", "fr_CA")
 - locname is the localized name of the hwkey

Presence of hwkeysdesc is facultative. New hwkey can be added by the inventory system without insertin in this table.
*/
CREATE TABLE hwkeysdesc (
  hwkey VARCHAR(50),
  locale CHAR(5),
  locname VARCHAR(50),
  PRIMARY KEY (hwkey, locale));



/* Table pciids contains the name of pci device

 - pciid, the numerical id of a device
 - name, the corresponding name of the device
*/
CREATE TABLE pciids (
  pciid VARCHAR(50) PRIMARY KEY,
  name VARCHAR(254));


