/*
 * configuration.c: logapp configuration handling
 *
 *
 * Copyright (C) 2007 Michael Brunner <mibru@gmx.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#ifndef TIOCGWINSIZE
#include <sys/ioctl.h>
#endif

#include "configuration.h"
#include "logfile.h"


int show_usage = 0;
int show_config = 0;

/* general configuration */
config_t config = {
	.argprefix 		= "logapp_",
	.strip_prefix		= "log",
	.executable		= EXECUTABLE,
	.dumbterm		= 0,
	.detectescape		= 0,
	.usepty			= 1,
	.appendlog		= 0,
	.printsummary		= 0,
	.printlogname		= 0,
	.disable		= 0,
	.disable_keywords	= NULL,
	.logname		= "./logapp.log",
	.logtime		= 0,
	.logreltime		= 0,
	.circularlog		= 0,
	.maxlogsize		= 0,
	.locklogfile		= 1,
	.warnlogfilelock	= 1,
	.maxlogfiles		= 10,
	.logrename		= 1,
	.alignlog               = 1,
	.jointimeout		= 0,
	.configfile		= NULL,
	.configsection		= NULL,
	.custconfigfile		= NULL,
};

/* default search path list for configuration file */
char* configsearchpath[]		= {
	"~/.logapprc",
	"/etc/logapp.conf",
	"/etc/logapp/logapp.conf",
};
#define CONFIGSEARCHPATH_SIZE (sizeof(configsearchpath)/sizeof(char*))

/* configuration for stdout */
pipe_t pstdout = {
	.fh		= 0,
	.cfhno		= STDOUT_FILENO,
	.name		= "stdout",
	.buf		= NULL,
	.charbased	= 0,
	.blen		= 2048,
	.fgcol		= -1,
	.bgcol		= -1,
	.bold		= 0,
	.clip		= -2,
	.eclip  	= 2,
	.escreset	= "\033[0m",
	.esccolor	= NULL,
	.linecount	= 0,
	.lineprefix     = NULL,
	.regexp         = NULL,
	.regbgcol	= 4,
	.bgesccolor    = NULL,
};

/* configuration for stderr */
pipe_t pstderr = {
	.fh		= 0,
	.cfhno		= STDERR_FILENO,
	.name		= "stderr",
	.buf		= NULL,
	.charbased	= 0,
	.blen		= 2048,
	.fgcol		= 1,
	.bgcol		= -1,
	.bold		= 1,
	.clip		= -1,
	.eclip		= 0,
	.escreset	= "\033[0m",
	.esccolor	= NULL,
	.linecount	= 0,
	.lineprefix	= "STDERR: ",
	.regexp         = NULL,
	.regbgcol	= 4,
	.bgesccolor    = NULL,
};

/* logfile configuration */
logfile_t logfile = {
	.fh	= 0,
	.name	= NULL,
	.sizelimit = 0,
	.oldext = ".old",
	.appendnr = 0,
	.head	= ">> logapp:",
	.split	= "== logapp split:",
	.foot	= "<< logapp end:",
	.addnewline = 0,
};

struct winsize ptysize;
struct termios ptytermios, ptytermios_bak;

/* application specific properties */
app_t app = {
	.exe		= NULL,
	.argv		= NULL,
	.argc		= 0,
	.pid		= 0,
	.pstdout	= &pstdout,
	.pstderr	= &pstderr,
	.ptysize        = &ptysize,
	.ptytermios	= &ptytermios,
	.ptytermios_bak = &ptytermios_bak,
	.active		= 0,
	.doexit		= 0,
	.exit_state	= 0,
	.starttime	= 0,
	.logfile	= &logfile,
};

const stringvalue_t boolvalues[] = {
	{ "true",	1 },
	{ "false",	0 },
	{ "on",		1 },
	{ "off",	0 },
	{ NULL,		0 }
};

const stringvalue_t colorvalues[] = {
	{ "disable",	-1 },
	{ "default",	-1 },
	{ "black",	0 },
	{ "red",	1 },
	{ "green",	2 },
	{ "brown",	3 },
	{ "blue",	4 },
	{ "magenta",	5 },
	{ "cyan",	6 },
	{ "white",	7 },
	{ NULL,		0 }
};

const stringvalue_t clipvalues[] = {
	{ "disable",	-1 },
	{ "auto",	-2 },
	{ NULL,		0 }
};

/* all possible configuration parameters with help message */
arglist_t arglist[] = {
	{ '?',	"help",		NULL,		TNONE,	&show_usage,
		"show this help", 0 },
	{ '\0', "configfile",   "FILE",		TSTRING,
		&config.custconfigfile,	"configuration file", 0 },
	{ '\0',	"showconfig",	NULL,		TNONE,	&show_config,
		"print current configuration on the screen and exit", 0 },
	{ '\0', "configsection",   "NAME",	TSTRING,
		&config.configsection, "select configuration section", 0 },
	{ '\0', "disable",	NULL,		TNONE,  &config.disable,
		"disable output handling", 0 },
	{ '\0', "disable_keywords", "STRING",	TSTRING,
		&config.disable_keywords, "comma separated keywords to "
		"disable output handling when found in argument list", 0 },
	{ '\0', "detectescape",	"BOOL",		TBOOL,	&config.detectescape,
		"switch to charbased mode if escape sequence is detected", 0},
	{ '\0', "stdout_blen",	"SIZE",		TUINT,	&pstdout.blen,
		"stdout buffer size", 0 },
	{ '\0', "stderr_blen",	"SIZE",		TUINT,	&pstderr.blen,
		"stderr buffer size", 0 },
	{ '\0', "dumbterm",	"BOOL",		TBOOL,	&config.dumbterm,
		"disable colors for use with dumb terminals", 0 },
#if CONFIG_SUPPORT_PTY
	{ '\0', "usepty",	"BOOL",		TBOOL,	&config.usepty,
		"use PTY for stream redirection", 0 },
#else
	{ '\0', "usepty",	"BOOL",		TBOOL,	&config.usepty,
		"<not supported> use PTY for stream redirection", 0 },
#endif
	{ 'l',  "logfile",	"NAME",		TSTRING, &config.logname,
		"application logfile", 0 },
	{ 'a',  "appendlog",	"BOOL",		TBOOL, &config.appendlog,
		"append to existing logfile", 0 },
	{ 't',  "logtime",	"BOOL",	TBOOL, &config.logtime,
		"add timestamp to each logged line", 0 },
	{ '\0',  "logreltime",	"BOOL",	TBOOL, &config.logreltime,
		"log relative time with --logtime", 0 },
	{ '\0', "alignlog",	"BOOL",	TBOOL, &config.alignlog,
		"use one line for every write in charbased mode", 0},
	{ '\0', "jointimeout",	"TIME",	TUINT, &config.jointimeout,
		"join timeout for packets with alignlog active", 0},
	{ '\0', "locklogfile",	"BOOL",	TBOOL, &config.locklogfile,
		"lock logfile when opening it", 0},
	{ '\0', "warnlogfilelock", "BOOL",	TBOOL, &config.warnlogfilelock,
		"print a warning if current logfile is locked", 0},
	{ '\0', "maxaltlogfiles", "COUNT",	TUINT, &config.maxlogfiles,
		"max # of alternate logfiles on lock", 0},
	{ '\0', "maxlogsize",	"SIZE",	TUINT, &config.maxlogsize,
		"max. logfile size in KiB (0=no limit, 10-4000000) the file "
		"will be truncated if logrename isn't set", 0 },
	{ '\0', "logrename",	"BOOL",	TBOOL, &config.logrename,
		"rename logfile before replacing it", 0},
	{ '\0', "circularlog",	"BOOL",	TBOOL, &config.circularlog,
		"write log in a circular way, keeping the max. size", 0},
	{ '\0', "oldlogext",	"STRING",	TSTRING, &logfile.oldext,
		"extension for old logfile", 0},
	{ 's',  "print_summary", "BOOL",	TBOOL, &config.printsummary,
		"print execution summary", 0 },
	{ 'n',  "print_logname", "BOOL",	TBOOL, &config.printlogname,
		"print logfile name after execution", 0 },
	{ 'f',  "stdout_fgcol",	"COLOR#",	TCOLOR, &pstdout.fgcol,
		"stdout console foreground color (disable=-1, 0-7)", 0 },
	{ 'F',  "stderr_fgcol",	"COLOR#",	TCOLOR,   &pstderr.fgcol,
		"stderr console foreground color (disable=-1, 0-7)", 0 },
	{ 'b',  "stdout_bold",	"BOOL",		TBOOL,	&pstdout.bold,
		"bold stdout console font", 0 },
	{ 'B',  "stderr_bold",	"BOOL",		TBOOL,	&pstderr.bold,
		"bold stderr console font", 0 },
	{ '\0', "stdout_regexp_bgcol",	"COLOR#", TCOLOR, &pstdout.regbgcol,
		"stdout console background color on regexp match (disable=-1, "
		"0-7)", 0 },
	{ '\0', "stderr_regexp_bgcol",	"COLOR#", TCOLOR, &pstderr.regbgcol,
		"stderr console background color on regexp match (disable=-1, "
		"0-7)", 0 },
	{ 'c',  "stdout_clip",	"LENGTH",	TCLIP,	&pstdout.clip,
		"clip stdout console at column LENGTH (disable=-1, auto=-2)", 0 },
	{ 'C',  "stderr_clip",	"LENGTH",	TCLIP,	&pstderr.clip,
		"clip stderr console at column LENGTH (disable=-1, auto=-2)", 0 },
	{ 'p',  "stdout_lineprefix",   "STRING",	TSTRING,
		&pstdout.lineprefix, "logfile line prefix for stdout", 0 },
	{ 'P',  "stderr_lineprefix",   "STRING",	TSTRING,
		&pstderr.lineprefix, "logfile line prefix for stderr", 0 },
	{ 'r',  "stdout_regexp", "STRING", TSTRING, &pstdout.regexp,
		"regular expression to change stdout background color", 0 },
	{ 'R',	"stderr_regexp",   "STRING",	TSTRING, &pstderr.regexp,
		"regular expression to change stderr background color", 0 },
	{ '\0', "stdout_charbased",	"BOOL",	TBOOL,  &pstdout.charbased,
		"handle stdout char- instead of line-based", 0 },
	{ '\0', "stderr_charbased",	"BOOL",	TBOOL,  &pstderr.charbased,
		"handle stderr char- instead of line-based", 0 },
};

const int arglistsize = ((int)(sizeof(arglist)/sizeof(arglist_t)));

void show_configuration(void)
{
	int i;

	message("%s %s\n\n", EXECUTABLE, VERSION);
	message("current configuration:\n");
	message("  argument prefix         %s\n", config.argprefix);
	message("  executable prefix       %s\n", config.strip_prefix);
	message("  active config file      %s\n",
		config.configfile?config.configfile:"<none>");
	message("  custom configfile path  %s\n",
		config.custconfigfile?config.custconfigfile:"<none>");
	for (i=0; i<CONFIGSEARCHPATH_SIZE; i++) {
		message("  config search path (%d)  %s\n", i,
			configsearchpath[i]);
	};
	message("  activate section        %s\n", config.configsection);
	message("  application logfile     %s\n",
		config.logname?config.logname:"<none>");
	message("  append to logfile       %i\n", config.appendlog);
	message("  log timestamps          %i\n", config.logtime);
	message("  log relative time       %i\n", config.logreltime);
	message("  align log writes left   %i\n", config.alignlog);
	message("  join timeout            %i\n", config.jointimeout);
	message("  lock logfiles           %i\n", config.locklogfile);
	message("  warning on logfile lock %i\n", config.warnlogfilelock);
	message("  max. alternate logfiles %u\n", config.maxlogfiles - 1);
	message("  maximum logsize         %u\n", logfile.sizelimit);
	message("  rename logfiles         %i\n", config.logrename);
	message("  circular logfile        %i\n", config.circularlog);
	message("  extension for old logs  %s\n", logfile.oldext);
	
	message("\n  application executable  %s\n",
		app.exe?app.exe:"<none>");
	message("  application argc        %d\n", app.argc);
	for (i=0; i<app.argc; i++) {
		message("  application argv[%d]     %s\n", i,
			app.argv[i]);
	}

	message("\n");
	if (CONFIG_USE_THREADS) {
		message("  use threads             %i\n", 1);
	} else {
		message("  use threads             <not supported>\n");
	}
	if (CONFIG_SUPPORT_PTY) {
		message("  use PTY                 %i\n", config.usepty);
	} else {
		message("  use PTY                 <not supported>\n");
	}

	message("\n  dumb terminal mode      %i\n", config.dumbterm);
	message("  print_summary           %i\n", config.printsummary);
	message("  print_logname           %i\n", config.printlogname);
	message("  disable output handling %i\n", config.disable);
	message("  disable keywords        %s\n",
		config.disable_keywords?config.disable_keywords:"<none>");
	message("  esc-sequence detection  %i\n", config.detectescape);

	message("\n  stdout buffer length    %i\n", app.pstdout->blen);
	message("  stdout foreground color %i\n", app.pstdout->fgcol);
	message("  stdout background color %i\n", (app.pstdout->bgcol == 9)?
		-1:app.pstdout->bgcol);
	message("  stdout regexp bg color  %i\n", (app.pstdout->regbgcol == 9)?
		-1:app.pstdout->regbgcol);
	message("  stdout bold font        %i\n", app.pstdout->bold);
	message("  stdout clip at column   ");
	if (app.pstdout->eclip==1) {
		message("%d\n", app.pstdout->clip);
	} else {
		message("%s\n", app.pstdout->eclip?"auto":"disable");
	}
	message("  stdout line prefix      %s\n", app.pstdout->lineprefix?
		app.pstdout->lineprefix:"<none>");
	message("  stdout regexp           %s\n", app.pstdout->regexp?
		app.pstdout->regexp:"<disabled>");
	message("  stdout charbased        %i\n", app.pstdout->charbased);

	message("\n  stderr buffer length    %i\n", app.pstderr->blen);
	message("  stderr foreground color %i\n", app.pstderr->fgcol);
	message("  stderr background color %i\n", (app.pstderr->bgcol == 9)?
		-1:app.pstderr->bgcol);
	message("  stderr regexp bg color  %i\n", (app.pstderr->regbgcol == 9)?
		-1:app.pstderr->regbgcol);
	message("  stderr bold font        %i\n", app.pstderr->bold);
	message("  stderr clip at column   ");
	if (app.pstderr->eclip==1) {
		message("%d\n", app.pstderr->clip);
	} else {
		message("%s\n", app.pstderr->eclip?"auto":"disable");
	}
	message("  stderr line prefix      %s\n", app.pstderr->lineprefix?
		app.pstderr->lineprefix:"<none>");
	message("  stderr regexp           %s\n", app.pstderr->regexp?
		app.pstderr->regexp:"<disabled>");
	message("  stderr charbased        %i\n", app.pstderr->charbased);
}

char* get_longpath(const char* filename)
{
	char* home;
	char* longpath;

	if (!strncmp(filename, "~/", 2)) {
		home = getenv("HOME");
		if (home == NULL) {
			home = "./";
		};

		longpath = (char*) malloc(strlen(home) + strlen(filename) + 1);
		if (longpath != NULL) {
			strcpy(longpath, home);
			strcat(longpath, "/");
			strcat(longpath, filename + 2);
		}
	} else {
		longpath = strdup(filename);
	}

	if (longpath == NULL) {
		error("out of memory\n");
		return NULL;
	}

	return longpath;
}

static int fixup_pipe(pipe_t *pipe)
{
	if ((pipe->fgcol < -1) || (pipe->fgcol > 7)) {
		error("%s foreground color out of range\n", pipe->name);
		return -1;
	}

	if ((pipe->bgcol < -1) || (pipe->bgcol > 7)) {
		error("%s background color out of range\n", pipe->name);
		return -1;
	}

	if (pipe->regbgcol == -1)
		pipe->regexp = NULL;

	if ((pipe->regbgcol < -1) || (pipe->regbgcol > 7)) {
		error("%s regexp background color out of range\n", pipe->name);
		return -1;
	} else if (pipe->bgcol != pipe->regbgcol) {
		if (pipe->bgcol == -1)
			pipe->bgcol = 9;

		if (pipe->regbgcol == -1)
			pipe->regbgcol = 9;
	}

	switch (pipe->clip) {
		case -1:	pipe->eclip = 0;
				pipe->clip = 80;
				break;
		case -2:	pipe->eclip = 2;
				pipe->clip = 80;
				break;
		default:	if (pipe->clip < 0)
					pipe->eclip = 0;
				else
					pipe->eclip = 1;
	}

	if (pipe->lineprefix != NULL) {
		if (strlen(pipe->lineprefix) == 0) {
			pipe->lineprefix = NULL;
		}
	}

	/* build console escape string for this pipe */
	int len = strlen(pipe->escreset) + 1;

	if (pipe->fgcol >= 0)
		len += 5;
	if (pipe->regbgcol)
		len += 5;
	if (pipe->bold)
		len += 4;

	pipe->esccolor = malloc(len);
	if (!pipe->esccolor) {
		error("out of memory\n");
		return -1;
	}

	sprintf(pipe->esccolor, pipe->escreset);

	if (pipe->fgcol >= 0)
		sprintf(pipe->esccolor+strlen(pipe->esccolor), "\033[3%dm",
			pipe->fgcol);

	if (pipe->regbgcol >= 0) {
		pipe->bgesccolor = pipe->esccolor+strlen(pipe->esccolor) + 3;
		sprintf(pipe->esccolor+strlen(pipe->esccolor), "\033[4%dm",
			pipe->bgcol); 
	} else {
		pipe->bgesccolor = NULL;
	}
	
	if (pipe->bold)
		sprintf(pipe->esccolor+strlen(pipe->esccolor), "\033[1m");

	if (pipe->regexp) {
		if (strlen(pipe->regexp)) {
			if (regcomp(&pipe->preg, pipe->regexp, REG_NOSUB)) {
				error("unable to process regular expression "
				      "for %s\n", pipe->name);
				return -1;
			}
		} else {
			pipe->regexp = NULL;
		}
	}

	return 0;
}

int fixup_config(void)
{
	if (strlen(config.logname) == 0)
		config.logname = NULL;

	/* We have to add the original logfile to maxlogfiles */
	config.maxlogfiles++;

	if (config.maxlogsize == 0) {
		logfile.sizelimit = 0;
	} else if ((config.maxlogsize < 10) || (config.maxlogsize > 4000000)) {
		error("maximum logsize out of range (0=no limit, 10-4000000 "
		      "KiB)\n");
		return -1;
	} else {
		logfile.sizelimit = config.maxlogsize * 1024;
	}

	if (fixup_pipe(app.pstdout))
		return -1;

	if (fixup_pipe(app.pstderr))
		return -1;

	return 0;
}

int get_display_parameters(void)
{
	/* As the terminal settings are not available if the stream is
	 * redirected we try all three standard streams and choose the first
	 * one that works. If none works we continue without the settings... */

	/* Get current terminal window size */
	if (ioctl(STDOUT_FILENO, TIOCGWINSZ, (char*)app.ptysize) < 0) { 
		if (ioctl(STDERR_FILENO, TIOCGWINSZ, (char*)app.ptysize) < 0) {
			if (ioctl(STDIN_FILENO, TIOCGWINSZ,
				  (char*)app.ptysize) < 0) {
				app.ptysize = NULL;
				app.ptytermios = NULL;
				return -1;
			}
		}
	}

	/* Get terminal attributes */
	if (tcgetattr(STDOUT_FILENO, app.ptytermios) < 0) {
		if (tcgetattr(STDERR_FILENO, app.ptytermios) < 0) {
			if (tcgetattr(STDIN_FILENO, app.ptytermios) < 0) {
				error("unable to get window termios\n");
				app.ptytermios = NULL;
				return -1;
			}
		}
	}

	return 0;
}

int adjust_clipping(void)
{
	if (get_display_parameters()) {
		/* We don't seem to have a terminal window and therefore can't
		 * get a size for automatic line clipping - disable it! */
		if (pstdout.eclip == 2) {
			pstdout.eclip = 0;
		}
		if (pstderr.eclip == 2) {
			pstderr.eclip = 0;
		}

		return -1;
	}

	if (pstdout.eclip == 2)
		pstdout.clip = app.ptysize->ws_col;

	if (pstderr.eclip == 2)
		pstderr.clip = app.ptysize->ws_col;

	/* inform the application that window size changed */
	if (app.pid != 0) {
		if (ioctl(app.pstdout->fh, TIOCSWINSZ,
			  (char*)app.ptysize) < 0) {
			error("Unable to set window size");
			return -1;
		}
	}

	return 0;
}

int get_argid(char* arg, int isshort)
{
	int i;

	if (!arg)
		return -1;

	for (i=0; i<arglistsize; i++) {
		if (isshort) {
			if (!arglist[i].shrt)
				continue;
			if (arglist[i].shrt == *arg)
				return(i);
		} else {
			if (!arglist[i].lng)
				continue;
			if (!strcmp(arglist[i].lng, arg))
				return(i);
		}
	}

	return -1;
}

int string2numvalue(char* string, int *value, const stringvalue_t *list )
{
	char*	p;

	if (list) {
		while (strcmp(string, list->string)) {
			list++;
			if (list->string == NULL)
				break;
		}
		if (list->string != NULL) {
			*value = list->value;

			return 0;
		}
	}

	errno = 0;
	*value = (int) strtol(string, &p, 0);
	if (*p || (p==string)) {
		error("invalid parameter\n");
		return -1;
	}
	if (errno==ERANGE) {
		error("value out of range\n");
		return -1;
	}

	return 0;
}

int get_argvalue(int priority, int argid, int* argp, int argc, char* argv[],
		 char* value)
{
	int	tmpint;

	if (!arglist[argid].var) {
		return -1;
	}

	if (value && (arglist[argid].type == TNONE)) {
		error("no value expected\n");
		return -1;
	} else if (!value && (arglist[argid].type != TNONE))  {
		if (*argp < argc -1) {
			(*argp)++;
			if (argv[*argp][0]!='-')
				value = argv[*argp];
		}
		if (!value) {
			error("parameter needs value\n");
			return -1;
		}
	}

	switch (arglist[argid].type) {
		case TNONE:	if (arglist[argid].set > priority)
					break;
				*((int*)arglist[argid].var) = 1;
			   	break;
		case TCLIP:
		case TCOLOR:
		case TINT:
		case TUINT:	switch (arglist[argid].type) {
					case TCLIP:
						string2numvalue(value, &tmpint,
								clipvalues);
						break;
					case TCOLOR:
						string2numvalue(value, &tmpint,
								colorvalues);
						break;
					default:
						string2numvalue(value, &tmpint,
								NULL);
				}
				if ((arglist[argid].type==TUINT)&&(tmpint<0)) {
					error("negative value not allowed for "
					      "this parameter\n");
					return -1;
				}
				if (arglist[argid].set > priority)
					break;
				*((int*)arglist[argid].var) = tmpint;
				break;
		case TSTRING:	if (arglist[argid].set > priority)
					break;
				*((char**)arglist[argid].var) = strdup(value);
				if (*((char**)arglist[argid].var) == NULL) {
					error("out of memory\n");
					return -1;
				}
				break;
		case TBOOL:	if (string2numvalue(value, &tmpint,
						    boolvalues)) {
					error("invalid boolean value\n");
					return -1;
				}
				if (arglist[argid].set > priority)
					break;
				*((int*)arglist[argid].var) = tmpint;
				break;
		default:	error("unknown argument type\n");
				return -1;
	}

	if (arglist[argid].set < priority) 
		arglist[argid].set = priority;

	return 0;

}

int parse_args(int argc, char* argv[])
{
	int i;
	char* arg;
	char* value;
	int argid;
	int argcount = 1;

	if (argc < 1) {
		error("invalid argument count\n");
		return -1;
	}

	/* prepare argc and argv for our main application */
	/* the new argc is still unknown but the old value should be save */
	app.argv = malloc((argc+1)*sizeof(argv[0]));
	if (!app.argv) {
		error("out of memory\n");
		return -1;
	}
	app.argc = 1;

	/* split arguments for applog and our main application */
	for (i=1; i<argc; i++)
	{
		value = NULL;
		if (argv[i][0]=='-') {
			if (argv[i][1]=='-') {
				arg = &argv[i][2];
				/* strip argument prefix */
				if (strstr(arg, config.argprefix)==arg) {
					arg += strlen(config.argprefix);
				} else {
					if (app.exe) {
						app.argv[app.argc]=argv[i];
						app.argc++;
						continue;
					}
				}

				if ((value = strchr(arg, '='))) {
					*value = '\0';
					if (*(value+1)) {
						value++;
					} 
				}

				argid = get_argid(arg, 0);
			} else {
				if (app.exe) {
					app.argv[app.argc]=argv[i];
					app.argc++;
					continue;
				}
				arg = &argv[i][1];
				if (argv[i][2])
					return argcount;

				argid = get_argid(arg, 1);
			}

			if (argid<0)
				return(argcount);

			if (get_argvalue(2, argid, &i, argc, argv, value)) {
				return argcount;
			}

			argcount++;
		} else {
			if (app.exe) {
				app.argv[app.argc]=argv[i];
				app.argc++;
				continue;
			}
			arg = &argv[i][0];
			app.exe = arg;
		}

	}

	/* add executable name to new argv list and terminate new argv */
	app.argv[0] = app.exe;
	app.argv[app.argc] = NULL;

	if (app.exe == NULL)
		app.argc = 0;

	return 0;
}

#define LINEBUFSIZE	251

int parse_configline(char* line)
{
	static int lineno;
	static char* appconfig;
	char* name;
	char* value;
	char* tmp;
	int argid;

	lineno++;

	/* remove comments and spaces */
	name = line + strspn(line, " \t");
	if (strlen(name) == 0)
		return 0;
	value = name + strcspn(name, " \t=");
	*(value) = '\0';
	tmp = value;

	tmp = value + strcspn(value, "\t ");
	value = value + 1 + strspn(value + 1, " \t=");
	if (*value=='\"') {
		value++;
		tmp = strchr(value, '\"');
	} else if (*value=='\'') {
		value++;
		tmp = strchr(value, '\'');
	} else {
		tmp = value + strcspn(value, " \t");
	}
	if (tmp == NULL) {
		warning("unterminated string in config file (line %d)\n",
			lineno);
		return -1;
	}

	*tmp = '\0';
	
	/* check if we change to a tagged section */
	if (!strlen(value) && (name[0] == '[')
		   && (name[strlen(name)-1] == ']') ) {
		name = name + 1;
		name[strlen(name) - 1] = '\0';
		free(appconfig);
		appconfig = strdup(name);
		if (!appconfig) {
			error("out of memory\n");
			return -1;
		}
		return 0;
	}
	
	/* check if the section should be ignored */
	if (appconfig != NULL) {
		if (config.configsection != NULL) {
			char* tmp;
			if (strchr(appconfig, '/') == NULL) {
				tmp = strrchr(config.configsection, '/');
				if (tmp == NULL)
					tmp = config.configsection;
				else
					tmp = tmp + 1;
			} else {
				tmp = config.configsection;
			}
			if (strcmp(appconfig, tmp)) {
				return 0;
			}
		} else {
			return 0;
		}
	}

	/* get the configuration */
	if ((argid = get_argid(name, 0)) >= 0) {
		int argc = 1;
		char* argv[2];
		if ((strlen(value) == 0) && (arglist[argid].type == TNONE)) {
			value = NULL;
		}
		get_argvalue(1, argid, &argc, argc, argv, value);
	} else {
		warning("unknown parameter in config file (line %d)\n",
			lineno);
		return -1;
	}

	return 0;
}

int get_config(void)
{
	FILE* conffile = NULL;
	int i;
	char *filename;
	char linebuf[LINEBUFSIZE];

	/* open config file */
	if (config.custconfigfile == NULL) {
		for (i=0; i<CONFIGSEARCHPATH_SIZE; i++) {
			filename = get_longpath(configsearchpath[i]);

			if (filename == NULL)
				return -1;

			conffile = fopen(filename, "r");
			if (conffile != NULL)
				break;
		}
	} else {
		filename = get_longpath(config.custconfigfile);

		if (filename != NULL)
			conffile = fopen(filename, "r");

		if (conffile == NULL) {
			error("unable to open provided configuration file\n");
			return -1;
		}
	}
	
	if (conffile == NULL)
		return 0;

	/* Application executable name is active config section unless
	 * something else is provided later */
	if (config.configsection==NULL)
		config.configsection = app.exe;

	/* parse each line */
	config.configfile = filename;
	while (fgets(linebuf, LINEBUFSIZE, conffile) != NULL) {
		linebuf[strcspn(linebuf, "#\n")]='\0';
		if (parse_configline(linebuf))
			return -1;
	}

	return 0;
}

#define MAX_KEYWORDS	50
int check_for_disable_keywords(void)
{
	int i,j;
	int key;
	char* toksrc;
	char* tok[MAX_KEYWORDS];

	if (!config.disable_keywords)
		return 0;

	toksrc = strdup(config.disable_keywords);

	key = 0;
	tok[key] = strtok(toksrc, ", ");
	while (tok[key] && (key + 1 < MAX_KEYWORDS)){
		tok[++key] = strtok(NULL, ", ");
	}

	for (i=1; (i<app.argc) && (config.disable==0); i++) {
		for (j=0; (j<key) && (config.disable==0); j++) {
			if (strstr(app.argv[i], tok[j])) {
				config.disable = 1;
			}
		}

	}

	free(toksrc);

	return 0;
}
