/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  - Neither the name of Intel Corp. nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef _LINUX
#include <errno.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <netdb.h>
#include <syslog.h>

#define closesocket close
#else

#include <winsock2.h>
#include <iphlpapi.h>
#include <Ws2tcpip.h>

#endif	// _LINUX

#include <algorithm>
#include "Protocol.h"
#include "LMS_if.h"

#if DEBUGLOG
Protocol::Protocol() : _lme(true)
#else
Protocol::Protocol() : _lme(false)
#endif //DEBUGLOG
{
	_s_tcp = INVALID_SOCKET;
	_s_tls = INVALID_SOCKET;
	_sockets_active = false;
#ifdef _REMOTE_SUPPORT
	_remoteAccessEnabled = false;
#endif
	_AMTIPType = LMS_IP_ADDRESS_SHARED;
	memset(_AMTIPAddress, 0, sizeof(_AMTIPAddress));
	memset(_AMTFQDN, 0, sizeof(_AMTFQDN));
}

Protocol::~Protocol()
{
	Deinit();
	DestroySockets();
}

bool Protocol::Init(EventLogCallback cb, void *param)
{
	_eventLog = cb;
	_eventLogParam = param;

	Deinit();

	if (!_lme.Init(_HECICallback, this)) {
		return false;
	}

#ifdef _REMOTE_SUPPORT
	if (_lme.GetHECI().GetProtocolVersion() == LMS_PROCOL_VERSION) {
		if (!AdapterListInfo::Init(_AdapterCallback,this)) {
			_lme.Deinit();
			return false;
		}
	}
#endif

	_lme.RequestIPFQDN();

	// TODO: add event, and wait here until IP/FQDN info received

#ifdef _REMOTE_SUPPORT
	if (_lme.GetHECI().GetProtocolVersion() == LMS_PROCOL_VERSION) {
		_lme.RequestEntDNSSuffixList();
	}
#endif

	return true;
}

void Protocol::Deinit()
{
	_mapLock.acquire();
	ConnMap::iterator it = _openConnections.begin();

	for (; it != _openConnections.end(); it++) {
		_lme.CloseConnection(it->first, LMS_CLOSE_STATUS_SHUTDOWN);
		closesocket(it->second.s);
	}

	_openConnections.clear();

	_mapLock.release();

	_lme.Deinit();

#ifdef _REMOTE_SUPPORT
	AdapterListInfo::Deinit();
#endif

}

bool Protocol::CreateSockets(SOCKET_STATUS &TCPstatus, SOCKET_STATUS &TLSstatus)
{
	_s_tcp = _createServerSocket(TCP_PORT, TCPstatus);
	if (_s_tcp == INVALID_SOCKET) {
		return false;
	}

	TCPstatus = ACTIVE;

	_s_tls = _createServerSocket(TLS_PORT, TLSstatus);
	if (_s_tls == INVALID_SOCKET) {
		closesocket(_s_tcp);
		return false;
	}

	TLSstatus = ACTIVE;

	_sockets_active = true;
	return true;
}

void Protocol::DestroySockets()
{
	_sockets_active = false;

	if (_s_tcp != INVALID_SOCKET) {
		closesocket(_s_tcp);
		_s_tcp = INVALID_SOCKET;
	}

	if (_s_tls != INVALID_SOCKET) {
		closesocket(_s_tls);
		_s_tls = INVALID_SOCKET;
	}
}

SOCKET Protocol::_createSocket(unsigned long addr, unsigned short port, int type, SOCKET_STATUS &status)
{
	if (type != SOCK_DGRAM) {
		type = SOCK_STREAM;
	}

	SOCKET s = socket(AF_INET, type, 0);
	if (s == INVALID_SOCKET) {
		status = NOT_CREATED;
		return INVALID_SOCKET;
	}

#ifndef _LINUX
	int optval = 1;
	if (setsockopt(s, SOL_SOCKET, SO_EXCLUSIVEADDRUSE,
					(char *)&optval, sizeof(optval)) == SOCKET_ERROR) {
		PRINT("Error: Can't bind socket using exclusive address\n");
		closesocket(s);
		status = NOT_EXCLUSIVE_ADDRESS;
		return INVALID_SOCKET;
	}
#endif

	if (type != SOCK_DGRAM) {
		linger l;
		l.l_onoff = 0;
		l.l_linger = 0;
		if (setsockopt(s, SOL_SOCKET, SO_LINGER,
			(char *)&l, sizeof(l)) == SOCKET_ERROR) {
				closesocket(s);
				status = LINGER_ERROR;
				return INVALID_SOCKET;
			}
	}

	struct sockaddr_in sock_addr;
	memset(&sock_addr, 0, sizeof(sock_addr));

	sock_addr.sin_family = AF_INET;
	sock_addr.sin_addr.s_addr = addr;
	sock_addr.sin_port = htons(port);

	if (bind(s, (struct sockaddr *)&sock_addr, sizeof(sock_addr)) == SOCKET_ERROR) {
		closesocket(s);
		status = NOT_BINDED;
		return INVALID_SOCKET;
	}

	return s;
}

SOCKET Protocol::_createServerSocket(unsigned short port, SOCKET_STATUS &status)
{
	SOCKET s = _createSocket(INADDR_ANY, port, SOCK_STREAM, status);
	if (s == INVALID_SOCKET) {
		return INVALID_SOCKET;
	}

	if (listen(s, 5) == SOCKET_ERROR) {
		closesocket(s);
		status = NOT_LISTENED;
		return INVALID_SOCKET;
	}

	return s;
}

bool Protocol::_setNonBlocking(SOCKET s, bool enable)
{
#ifdef _LINUX
	fcntl(s, F_SETFL, O_NONBLOCK);
#else
	unsigned long param = (enable) ? 1 : 0;

	if (ioctlsocket(s, FIONBIO, &param) == SOCKET_ERROR) {
		return false;
	}
#endif	// _LINUX
	return true;

}

SOCKET Protocol::_connect(unsigned long addr, unsigned short port, int type)
{
	SOCKET_STATUS status;
	SOCKET s = _createSocket(INADDR_ANY, 0, type, status);
	if (s == INVALID_SOCKET) {
		return INVALID_SOCKET;
	}

	struct sockaddr_in sock_addr;
	memset(&sock_addr, 0, sizeof(sock_addr));

	sock_addr.sin_family = AF_INET;
	sock_addr.sin_addr.s_addr = addr;
	sock_addr.sin_port = port;

	if (connect(s, (struct sockaddr *)&sock_addr, sizeof(sock_addr)) == SOCKET_ERROR) {
		closesocket(s);
		return INVALID_SOCKET;
	}

	return s;
}

bool Protocol::_acceptConnection(SOCKET s, unsigned short port)
{
	struct sockaddr_in addr;
	int addrLen = sizeof(addr);
	SOCKET s_new = accept(s, (struct sockaddr *)&addr, (socklen_t *)&addrLen);
	if (s_new == INVALID_SOCKET) {
#if DEBUGLOG
#ifdef _LINUX
		int err = errno;
#else
		int err = GetLastError();
#endif	// _LINUX
		char *msg = _getErrMsg(err);
		PRINT("Error accepting new connection (%d): %s\n", err, msg);
#endif //DEBUGLOG
		return false;
	}

	int local = _isLocal(&addr);
#ifdef _REMOTE_SUPPORT
	int remote = 0;
#endif

	if (local != 1) {	// Not local
#ifdef _REMOTE_SUPPORT	// LMS supports remote
		if (_lme.GetHECI().GetProtocolVersion() == LMS_PROCOL_VERSION) { //Heci supports remote
			_remoteAccessLock.acquire();
			if (_remoteAccessEnabled) {	//AMT allows remote connections
				struct sockaddr_in localAddr;
				int localAddrLen = sizeof(localAddr);
				if (getsockname(s_new, (struct sockaddr *)&localAddr, &localAddrLen) == 0) {
					remote = _isRemoteFromEnterprise(&localAddr);;
				}
				if (remote != 1) {	// Not remote
					PRINT("Error: new connection is not local nor remote from enterprise network:\n");
					PRINT("Local addr %s, Remote addr %s\n", inet_ntoa(localAddr.sin_addr),
						inet_ntoa(addr.sin_addr));
					closesocket(s_new);
					_remoteAccessLock.release();
					return false;
				}
			}
			else {
				PRINT("AMT doesn't allow remote connections. Either AMT is connected directly"
					" to enterprise network or remote access is disabled.\n");
				_remoteAccessLock.release();
				return false;
			}
			_remoteAccessLock.release();
		}
		else {	//Heci doesn't support remote
#endif	// LMS doesn't support remote
			PRINT("Error: new connection is not local (addr %s)\n", inet_ntoa(addr.sin_addr));
			closesocket(s_new);
			return false;
#ifdef _REMOTE_SUPPORT
		}
#endif
	}

	_setNonBlocking(s_new, true);

	// Send an Open Connection Request to the LME
	int connID;
#ifdef _REMOTE_SUPPORT
	if (!_lme.OpenConnection(addr, port, &connID, (remote == 1))) {
#else
	if (!_lme.OpenConnection(addr, port, &connID)) {
#endif
		PRINT("Error: failed to open a new HECI connection\n");
		closesocket(s_new);
		return false;
	}

	PRINT("Opened HECI connection %d for socket %d\n", connID, (int)s_new);
	// add s_new to the list of open connections
	Connection conn;
	conn.s = s_new;

	_mapLock.acquire();
	_openConnections[connID] = conn;
	_mapLock.release();

	return true;
}

int Protocol::Select()
{
	struct timeval tv;
	tv.tv_sec = 1;
	tv.tv_usec = 0;

	int res;
	int fdCount = 0;
	fd_set rset;

	FD_ZERO(&rset);

	FD_SET(_s_tcp, &rset);
	if ((int)_s_tcp > fdCount) {
		fdCount = (int)_s_tcp;
	}

	FD_SET(_s_tls, &rset);
	if ((int)_s_tls > fdCount) {
		fdCount = (int)_s_tls;
	}

	_mapLock.acquire();
	ConnMap::iterator it = _openConnections.begin();

	for (; it != _openConnections.end(); it++) {
		FD_SET(it->second.s, &rset);
		if ((int)it->second.s > fdCount) {
			fdCount = (int)it->second.s;
		}
	}
	_mapLock.release();

	fdCount++;
	res = select(fdCount, &rset, NULL, NULL, &tv);
	if (res == -1) {
#if DEBUGLOG
#ifdef _LINUX
		int err = errno;
#else
		int err = GetLastError();
#endif	// _LINUX
		char *msg = _getErrMsg(err);

		PRINT("Select error (%d): %s\n", err, msg);
#endif //DEBUGLOG
	}

	if (res == 0) {
		return 0;
	}

	if (FD_ISSET(_s_tcp, &rset)) {
		// connection request
		PRINT("TCP connection request\n");
		_acceptConnection(_s_tcp, TCP_PORT);
		FD_CLR(_s_tcp, &rset);
		res--;
	}

	if (FD_ISSET(_s_tls, &rset)) {
		// connection request
		PRINT("TLS connection request\n");
		_acceptConnection(_s_tls, TLS_PORT);
		FD_CLR(_s_tls, &rset);
		res--;
	}

	int i;
	for (i = 0; (res > 0) && (i < fdCount); i++) {
		if (FD_ISSET(i, &rset)) {
			_rxFromSocket(i);
			res--;
		}
	}

	return 1;
}


void Protocol::_closeSocket(SOCKET s, bool lock)
{
	ConnMap::iterator it;

	closesocket(s);
	if (lock) {
		_mapLock.acquire();
	}
	for (it = _openConnections.begin(); it != _openConnections.end(); it++) {
		if (it->second.s == s) {
			break;
		}
	}
	if (it != _openConnections.end()) {
		_openConnections.erase(it);
	}
	if(lock) {
		_mapLock.release();
	}
}

int Protocol::_rxFromSocket(SOCKET s)
{
	int connectionId = -1;
	ConnMap::iterator it;

	_mapLock.acquire();
	for (it = _openConnections.begin(); it != _openConnections.end(); it++) {
		if (it->second.s == s) {
			connectionId = it->first;
			break;
		}
	}
	if (it == _openConnections.end()) {
		// Data received from a socket that is not in the map.
		// Since we only select on our sockets, this means it was
		// in the map, but was removed, probably because we received
		// an End Connection message from the HECI.
		_mapLock.release();
		return 0;
	}
	else {
		_mapLock.release();
	}

	PRINT("received data from socket %d (connection %d)\n", (int)s, connectionId);
	char buffer[1024];
	int res = 0;

	res = recv(s, buffer, sizeof(buffer), 0);
	if (res > 0) {
		// send data to LME
		_lme.SendMessage(connectionId, (UCHAR *)buffer, res);
		buffer[res+1]= '\0';
		PRINT("Received %d bytes from socket %d\n", res, (int)s);
		return 0;
	} else if (res == 0) {
		_closeSocket(s, true);
		PRINT("Received 0 bytes from socket %d. Closing HECI connection\n", (int)s);
		// connection closed
		_lme.CloseConnection(connectionId, LMS_CLOSE_STATUS_CLIENT);
	} else {
#ifdef DEBUGLOG
#ifdef _LINUX
		int err = errno;
#else
		int err = GetLastError();
#endif	// _LINUX
#endif //DEBUGLOG
		_closeSocket(s, true);

#ifdef DEBUGLOG
		char *msg = _getErrMsg(err);
		PRINT("Receive error on socket %d (%d): %s\n", (int)s, err, msg);
#endif //DEBUGLOG
		// error
		_lme.CloseConnection(connectionId, LMS_CLOSE_STATUS_SOCKET);
	}

	return -1;
}

void Protocol::_HECICallback(void *param, UCHAR *buffer, int len, int *status)
{
	Protocol *prot = (Protocol *)param;

	prot->_HECIReceive(buffer, len, status);
}

void Protocol::_HECIReceive(UCHAR *buffer, int len, int *status)
{
	PRINT("HECI receive %d bytes (msg type 0x%02x)\n", len, buffer[0]);
	*status = 0;

	switch (buffer[0]) {
		case LMS_MESSAGE_TYPE_OPEN_CONNECTION_EX:
			{
				LMS_OPEN_CONNECTION_EX_MESSAGE *msg;
				msg = (LMS_OPEN_CONNECTION_EX_MESSAGE *)buffer;

				unsigned long hostAddr = 0;

				if (((msg->Flags) & (HOSTNAME_BIT)) != 0) {
					PRINT("Got client connection request %d for host %s, port %d\n",
						msg->ConnectionId, msg->Host, ntohs(msg->HostPort));

					struct addrinfo *info = NULL;
					struct addrinfo hint;

					memset(&hint, 0, sizeof(hint));
					hint.ai_family = PF_INET;
					//hint.ai_socktype = SOCK_STREAM;
					//hint.ai_protocol = IPPROTO_TCP;

					if (getaddrinfo((char *)msg->Host, NULL, &hint, &info) != 0) {
						*status = 1;
						break;
					}

					if (info == NULL) {
						*status = 1;
						break;
					}

					hostAddr = ((struct sockaddr_in *)info->ai_addr)->sin_addr.s_addr;

					freeaddrinfo(info);

				}
				else {
					PRINT("Got client connection request %d for IP %s, port %d\n",
						msg->ConnectionId, inet_ntoa(*((struct in_addr*)msg->Host)), ntohs(msg->HostPort));

					hostAddr = *((unsigned long *)msg->Host);
				}


				int type;
				switch (msg->Protocol) {
					case LMS_PROTOCOL_TYPE_UDP_IPV4:
						type = SOCK_DGRAM;
						break;
					case LMS_PROTOCOL_TYPE_TCP_IPV4:
					default:
						type = SOCK_STREAM;
						break;
				}

				SOCKET s_new = _connect(hostAddr, msg->HostPort, type);
				if (s_new == INVALID_SOCKET) {
					*status = 1;
					break;
				}

				Connection conn;
				conn.s = s_new;

				_mapLock.acquire();
				_openConnections[msg->ConnectionId] = conn;
				_mapLock.release();
			}
			break;

		case LMS_MESSAGE_TYPE_CLOSE_CONNECTION:
			{
				LMS_CLOSE_CONNECTION_MESSAGE *msg;
				msg = (LMS_CLOSE_CONNECTION_MESSAGE *)buffer;

				PRINT("received close connection msg from HECI for connection %d\n", msg->ConnectionId);
				_mapLock.acquire();

				ConnMap::iterator it = _openConnections.find(msg->ConnectionId);
				if (it != _openConnections.end()) {
					closesocket(it->second.s);
					_openConnections.erase(it);
				}

				_mapLock.release();
			}
			break;

		case LMS_MESSAGE_TYPE_SEND_DATA:
			{
				LMS_SEND_DATA_MESSAGE *msg;
				msg = (LMS_SEND_DATA_MESSAGE *)buffer;

				_mapLock.acquire();

				ConnMap::iterator it = _openConnections.find(msg->ConnectionId);
				if (it != _openConnections.end()) {
					PRINT("sending %d bytes from HECI connection %d to socket %d\n", ntohs(msg->DataLength), msg->ConnectionId, it->second.s);
#ifdef _LINUX
					if(-1 == send(it->second.s, (char *)msg->Data, ntohs(msg->DataLength), MSG_NOSIGNAL)) {
						int senderr = errno;
						if (EPIPE == senderr) {
							_closeSocket(it->second.s, false);
							*status = 1;
						}
					}
#else
					send(it->second.s, (char *)msg->Data, ntohs(msg->DataLength), 0);
#endif
					//msg->Data[ntohs(msg->DataLength) + 1]= '\0';
					//printf("Data = %s\n",(char *)msg->Data);
				}

				_mapLock.release();
			}
			break;

		case LMS_MESSAGE_TYPE_IP_FQDN:
			{
				if (_updateIPFQDN((LMS_IP_FQDN_MESSAGE *)buffer) != 0) {
					ERROR("Error: failed to update IP/FQDN info\n");
				}
			}
			break;

#ifdef _REMOTE_SUPPORT
		case LMS_MESSAGE_TYPE_ENTERPRISE_DNS_SUFFIX_LIST:
			{
				if (_updateDNSSuffixList((LMS_ENTERPRISE_DNS_SUFFIX_LIST_MESSAGE *)buffer) != 0) {
					ERROR("Error: failed to update Enterprise DNS suffix list info\n");
				}
				else {
					_updateEnterpriseAccessStatus(AdapterListInfo::GetLocalDNSSuffixList());
				}
			}
			break;

		case LMS_MESSAGE_TYPE_REMOTE_ACCESS_STATUS:
			{
				LMS_REMOTE_ACCESS_STATUS_MESSAGE * msg = (LMS_REMOTE_ACCESS_STATUS_MESSAGE *)buffer;
				_remoteAccessLock.acquire();
				_remoteAccessEnabled = (msg->RemoteAccessStatus == LMS_REMOTE_ACCESS_STATUS_OK);
				switch (msg->RemoteAccessStatus) {
					case LMS_REMOTE_ACCESS_STATUS_OK:
						PRINT("Remote access is allowed.\n");
						break;
					case LMS_REMOTE_ACCESS_STATUS_DENIED:
						PRINT("Remote access is denied because AMT is directly connected "
							"to enterprise network.\n");
						break;
					case LMS_REMOTE_ACCESS_STATUS_DISABLED:
						PRINT("Remote access is disabled.\n");
						break;
					default:
						PRINT("Remote access is disabled.\n");
						break;
				}
				_remoteAccessLock.release();
			}
			break;
#endif

		default:
			*status = 1;
			break;
	}
}

#ifdef _REMOTE_SUPPORT
void Protocol::_AdapterCallback(void *param, SuffixMap localDNSSuffixes)
{
	Protocol *prot = (Protocol *)param;

	prot->_updateEnterpriseAccessStatus(localDNSSuffixes);
}

void Protocol::_updateEnterpriseAccessStatus(const SuffixMap localDNSSuffixes)
{
	_AMTDNSLock.acquire();

	vector<string>::iterator remIt;
	vector<string>::iterator startIt = _AMTDNSSuffixes.begin();
	vector<string>::iterator endIt = _AMTDNSSuffixes.end();
	SuffixMap::const_iterator locIt;

	bool access = false;
	unsigned long localIp = 0;

	for (locIt = localDNSSuffixes.begin(); locIt != localDNSSuffixes.end(); locIt++) {
		remIt = find(startIt, endIt, locIt->second);
		if (remIt != _AMTDNSSuffixes.end()) {
			access = true;
			localIp = locIt->first;
			break;
		}
	}

	_lme.SendEnterpriseAccessStatus(access, localIp);

	_AMTDNSLock.release();
}
#endif

int Protocol::_isLocal(sockaddr_in *addr)
{
#ifdef _LINUX
	// find target addresses
	hostent* targetAddrs = gethostbyname(inet_ntoa(addr->sin_addr));
    if (!targetAddrs)
    {
        return -1;
    }

    if (targetAddrs->h_addr_list[0] == NULL)
    {
        return -1;
    }

    vector<uint32_t> targAddresses;
    for (int i = 0 ; targetAddrs->h_addr_list[i] != NULL ; i++)
    {
        targAddresses.push_back(*((uint32_t *)targetAddrs->h_addr_list[i]));
    }

    //
    // Find local host address list
    //

    int sd;
    struct ifreq ifr, *ifrp;
    struct ifconf ifc;
    char buf[1024];
    int   len, i;
    unsigned char *ip_data;

    sd = socket(AF_INET, SOCK_DGRAM, 0);
    if (sd < 0)
    {
        return -1;
    }


    memset(buf, 0, sizeof(buf));
    ifc.ifc_len = sizeof(buf);
    ifc.ifc_buf = buf;

    // get the system network interfaces
    if (ioctl (sd, SIOCGIFCONF, (char *)&ifc) < 0)
    {
        close(sd);
        return -1;
    }
    len = ifc.ifc_len;

    vector<uint32_t> localAddresses;
    for (i = 0; i < len; i += sizeof(struct ifreq))
    {
        ifrp = (struct ifreq *)((char *) ifc.ifc_buf + i);
        strncpy(ifr.ifr_name, ifrp->ifr_name, IFNAMSIZ);

        // get the IP addresses
        if (ioctl(sd, SIOCGIFADDR, &ifr) < 0)
        {
            continue;
        }

        ip_data = (unsigned char *) &ifr.ifr_addr.sa_data;
        uint32_t addr = ((sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr;

        if (addr == 0)
        {
            continue;
        }

        localAddresses.push_back(addr);
    }
    close(sd);

    std::sort(localAddresses.begin(),localAddresses.end());
    std::sort(targAddresses.begin(),targAddresses.end());

    if (std::includes(localAddresses.begin(),localAddresses.end(),targAddresses.begin(),targAddresses.end()))
		return 1;
	else
		return -1;
#else
	MIB_IPADDRTABLE *addrTable = NULL;
	DWORD size = sizeof(MIB_IPADDRTABLE);
	DWORD ret;
	int result = -1;

	unsigned long localAddr = inet_addr("127.0.0.1");
	if (addr->sin_addr.S_un.S_addr == localAddr) {
		result = 1;
		goto out;
	}

	addrTable = (MIB_IPADDRTABLE *)malloc(size);
	if (addrTable == NULL) {
		goto out;
	}

	// Call GetIpAddrTable once to get needed size
	ret = GetIpAddrTable(addrTable, &size, 0);

	if (ret != NO_ERROR) {
		free(addrTable);
		addrTable = NULL;

		if (ret == ERROR_INSUFFICIENT_BUFFER) {
			addrTable = (MIB_IPADDRTABLE *)malloc(size);
			if (addrTable == NULL) {
				goto out;
			}

			// Call GetIpAddrTable a second time with the correct size
			ret = GetIpAddrTable(addrTable, &size, 0);
			if (ret != NO_ERROR) {
				goto out;
			}
		} else {
			goto out;
		}
	}


	for (DWORD i = 0; i < addrTable->dwNumEntries; i++) {
		MIB_IPADDRROW *addrRow = &addrTable->table[i];

		if ((addrRow->dwAddr != 0) &&
			!(addrRow->wType & MIB_IPADDR_DISCONNECTED) &&
			!(addrRow->wType & MIB_IPADDR_DELETED)) {

			PRINT("local addr: %s\n", inet_ntoa(*((struct in_addr*)&(addrRow->dwAddr))));
			if (addrRow->dwAddr == addr->sin_addr.S_un.S_addr) {
				result = 1; // found
				goto out;
			}
		}
	}

	// not found
	result = 0;

out:
	if (addrTable) {
		free(addrTable);
	}

	return result;
#endif	// _LINUX
}

#ifdef _REMOTE_SUPPORT
int Protocol::_isRemoteFromEnterprise(sockaddr_in *addr)
{
	int result = 0;

	string dnsSuffix = AdapterListInfo::GetDNSSuffixFromLocalIP(addr->sin_addr.s_addr);
	if (dnsSuffix.empty()) {
		return 0;
	}

	_AMTDNSLock.acquire();

	vector<string>::iterator it;
	for (it = _AMTDNSSuffixes.begin(); it != _AMTDNSSuffixes.end(); it++) {
		if (it->compare(dnsSuffix) == 0) {
			result = 1;
			break;
		}
	}

	_AMTDNSLock.release();

	return result;
}
#endif

int Protocol::_handleFQDNChange(const char *fqdn)
{
	char *hostFile = "hosts";
	char *tmpFile = "hosts-lms.tmp";
	bool hasFqdn = false;
#define LMS_MAX_FILENAME_LEN 1024
	char inFileName[LMS_MAX_FILENAME_LEN] = "";
	char oldFqdn[FQDN_MAX_SIZE + 1];
	char outFileName[LMS_MAX_FILENAME_LEN] = "";
	char host[FQDN_MAX_SIZE + 1];
#define LMS_MAX_LINE_LEN 1023
	char line[LMS_MAX_LINE_LEN + 1];
#define LMS_LINE_SIG_FIRST_WORDS "# LMS GENERATED "
#define LMS_LINE_SIG_LAST_WORD "LINE"
#define LMS_LINE_SIG_LAST_WORD_LEN 4
#define LMS_LINE_SIG LMS_LINE_SIG_FIRST_WORDS LMS_LINE_SIG_LAST_WORD
#define lmsstr(s) lmsname(s)
#define lmsname(s) #s
#define LMS_LINE_FORMAT "127.0.0.1       %s %s " LMS_LINE_SIG
#define LMS_LINE_SCAN_FORMAT "127.0.0.1 %" lmsstr(FQDN_MAX_SIZE) "s %" lmsstr(FQDN_MAX_SIZE) "s " LMS_LINE_SIG_FIRST_WORDS "%" lmsstr(LMS_LINE_SIG_LAST_WORD_LEN) "c"
	char tmpsige[LMS_LINE_SIG_LAST_WORD_LEN];

#ifdef _LINUX

	char *dir = "/etc/";

#else

	char *sysDrive;
	char *dir = "\\system32\\drivers\\etc\\";

	sysDrive = getenv("SystemRoot");
	if (NULL == sysDrive) {
		return -1;
	}

	// sanity check before string copying
	if (LMS_MAX_FILENAME_LEN < (strnlen(sysDrive, LMS_MAX_FILENAME_LEN)
	                            + strnlen(dir, LMS_MAX_FILENAME_LEN)
	                            + strnlen(hostFile, LMS_MAX_FILENAME_LEN) + 1)) {
		return -1;
	}
	// sanity check before string copying
	if (LMS_MAX_FILENAME_LEN < (strnlen(sysDrive, LMS_MAX_FILENAME_LEN)
	                            + strnlen(dir, LMS_MAX_FILENAME_LEN)
	                            + strnlen(tmpFile, LMS_MAX_FILENAME_LEN) + 1)) {
		return -1;
	}

	strncpy(inFileName, sysDrive, LMS_MAX_FILENAME_LEN - 1);
	strncpy(outFileName, sysDrive, LMS_MAX_FILENAME_LEN - 1);

#endif	// _LINUX

	strncat(inFileName, dir, LMS_MAX_FILENAME_LEN - 1);
	strncat(outFileName, dir, LMS_MAX_FILENAME_LEN - 1);
	strncat(inFileName, hostFile, LMS_MAX_FILENAME_LEN - 1);
	strncat(outFileName, tmpFile, LMS_MAX_FILENAME_LEN - 1);

	FILE *ifp = fopen(inFileName, "r");
	if (NULL == ifp) {
		_eventLog(_eventLogParam, TEXT("Error: Can't open hosts file"), EVENTLOG_ERROR_TYPE);
		return -1;
	}

	FILE *ofp = fopen(outFileName, "w");
	if (NULL == ofp) {
		_eventLog(_eventLogParam, TEXT("Error: Can't create temporary hosts file"), EVENTLOG_ERROR_TYPE);
		fclose(ifp);
		return -1;
	}

	// First create a copy of the hosts file, without lines that were
	// previously added by the LMS.
	// Go over each line and copy it to the tmp file.
	while (fgets(line, sizeof(line), ifp)) {
		// don't copy the line if it was generated by the LMS
		memset(oldFqdn, 0, sizeof(oldFqdn));
		memset(tmpsige, 0, sizeof(tmpsige));
		if (0 == (
		    (3 == sscanf(line, LMS_LINE_SCAN_FORMAT, oldFqdn, host, tmpsige))
		    ? strncmp(tmpsige, LMS_LINE_SIG_LAST_WORD, LMS_LINE_SIG_LAST_WORD_LEN)
		    : (-2))
		) {
			if (0 == strncmp((char *)fqdn, oldFqdn, FQDN_MAX_SIZE)) {
				// copy the old LMS line too, since it's up to date
				fprintf(ofp, "%s", line);
				hasFqdn = true;
			}
			continue;
		}

		fprintf(ofp, "%s", line);

		while ((LMS_MAX_LINE_LEN == strnlen(line, LMS_MAX_LINE_LEN))
		    && ('\n' != line[LMS_MAX_LINE_LEN - 1])
		    && (fgets(line, sizeof(line), ifp))) {
			fprintf(ofp, "%s", line);
		}
	}

	if (hasFqdn) {
		fclose(ofp);
		fclose(ifp);
		unlink(outFileName);
		return 0;
	}

	// If the original hosts file does not end with a new line character,
	// add a new line at the end of the new file before adding our line.
	fseek(ifp, -1, SEEK_END);
	char lastChar = fgetc(ifp);
	if ('\n' != lastChar) {
		fprintf(ofp, "\n");
	}

	memset(host, 0, FQDN_MAX_SIZE + 1);
	strncpy(host, fqdn, FQDN_MAX_SIZE);
	char *lmsdot = strchr(host, '.');
	if (NULL != lmsdot) {
		lmsdot[0] = '\0';
	}
	// Add the specified FQDN to the end of the tmp file
	fprintf(ofp, LMS_LINE_FORMAT "\n", fqdn, host);

	fclose(ofp);
	fclose(ifp);

	// Copy the tmp file in place of the original hosts file
#ifdef _LINUX
	sprintf(line, "cp -f %s %s", outFileName, inFileName);
#else
	sprintf(line, "copy /y %s %s", outFileName, inFileName);
#endif	// _LINUX
	if (system(line) != 0) {
		_eventLog(_eventLogParam, TEXT("Error: Can't update hosts file"), EVENTLOG_ERROR_TYPE);
		return -1;
	}
	_eventLog(_eventLogParam, TEXT("hosts file updated"), EVENTLOG_INFORMATION_TYPE);

	//sprintf(line, "del /f %s", outFileName);
	//system(line);

	return 0;
}

int Protocol::_updateIPFQDN(LMS_IP_FQDN_MESSAGE *msg)
{
	if (strcmp((char *)msg->FQDN, (char *)_AMTFQDN) != 0) {
		if (_handleFQDNChange((char *)(msg->FQDN)) < 0) {
			ERROR("Error: failed to update FQDN info\n");
			return -1;
		}
	}

	_AMTIPType = msg->IPType;
	memcpy(_AMTIPAddress, msg->AMTIPAddress, sizeof(_AMTIPAddress));
	memcpy(_AMTFQDN, msg->FQDN, sizeof(_AMTFQDN));

	PRINT("Got IP: %s, FQDN: %s\n", inet_ntoa(*((struct in_addr*)_AMTIPAddress)), _AMTFQDN);

	return 0;
}

#ifdef _REMOTE_SUPPORT
int Protocol::_updateDNSSuffixList(LMS_ENTERPRISE_DNS_SUFFIX_LIST_MESSAGE *msg)
{
	_AMTDNSLock.acquire();

	_AMTDNSSuffixes.clear();

	char *current = (char *)msg->Data;
	while (current < (char *)msg->Data + ntohs(msg->DataLength)) {
		string dnsSuffix = current;
		if (!dnsSuffix.empty()) {
			_AMTDNSSuffixes.push_back(dnsSuffix);
			PRINT("Got AMT DNS suffix: %s\n", dnsSuffix.c_str());
		}
		current += dnsSuffix.length() + 1;
	}
	_AMTDNSLock.release();

	return 0;
}
#endif

char *Protocol::_getErrMsg(DWORD err)
{
	static char buffer[1024];
#ifdef _LINUX
	strerror_r(err, buffer, sizeof(buffer) - 1);
#else
	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
					NULL,
					err,
					0,
					buffer,
					sizeof(buffer) - 1,
					0);
#endif	// _LINUX
	return buffer;
}

LMEConnection& Protocol::GetLMEConnection()
{
	return _lme;
}
