/* OggEnc
 *
 * This program is distributed under the GNU General Public License, version 2.
 * A copy of this license is included with this source.
 *
 * Copyright 2000-2002, Michael Smith <msmith@xiph.org>
 *
 * Portions from Vorbize, (c) Kenneth Arnold <kcarnold@yahoo.com>
 * and libvorbis examples, (c) Monty <monty@xiph.org>
 */
#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <locale.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

struct option
{
# if (defined __STDC__ && __STDC__) || defined __cplusplus
  const char *name;
# else
  char *name;
# endif
  /* has_arg can't be an enum because some compilers complain about
     type mismatches in all the code that assumes it is an int.  */
  int has_arg;
  int *flag;
  int val;
};

# define no_argument            0
# define required_argument      1
# define optional_argument      2

extern int getopt (int ___argc, char *const *___argv, const char *__shortopts);
extern int getopt_long (int ___argc, char *const *___argv,
                        const char *__shortopts,
                        const struct option *__longopts, int *__longind);
extern int getopt_long_only (int ___argc, char *const *___argv,
                             const char *__shortopts,
                             const struct option *__longopts, int *__longind);

extern int _getopt_internal (int ___argc, char *const *___argv,
                             const char *__shortopts,
                             const struct option *__longopts, int *__longind,
                             int __long_only);


#define setbinmode(x) {}
#define DEFAULT_NAMEFMT_REMOVE "/"
#define DEFAULT_NAMEFMT_REPLACE NULL

#define __ENCODE_H

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2001             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *

 ********************************************************************

 function: libvorbis codec headers

 ********************************************************************/

#define _vorbis_codec_h_

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: toplevel libogg include

 ********************************************************************/
#define _OGG_H

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: #ifdef jail to whip a few platforms into the UNIX ideal.

 ********************************************************************/
#define _OS_TYPES_H

/* make it easy on the folks that want to compile the libs with a
   different malloc than stdlib */
#define _ogg_malloc  malloc
#define _ogg_calloc  calloc
#define _ogg_realloc realloc
#define _ogg_free    free

#define __CONFIG_TYPES_H__

/* these are filled in by configure */
typedef int16_t ogg_int16_t;
typedef u_int16_t ogg_uint16_t;
typedef int32_t ogg_int32_t;
typedef u_int32_t ogg_uint32_t;
typedef int64_t ogg_int64_t;

typedef struct {
  long endbyte;
  int  endbit;

  unsigned char *buffer;
  unsigned char *ptr;
  long storage;
} oggpack_buffer;

/* ogg_page is used to encapsulate the data in one Ogg bitstream page *****/

typedef struct {
  unsigned char *header;
  long header_len;
  unsigned char *body;
  long body_len;
} ogg_page;

/* ogg_stream_state contains the current encode/decode state of a logical
   Ogg bitstream **********************************************************/

typedef struct {
  unsigned char   *body_data;    /* bytes from packet bodies */
  long    body_storage;          /* storage elements allocated */
  long    body_fill;             /* elements stored; fill mark */
  long    body_returned;         /* elements of fill returned */


  int     *lacing_vals;      /* The values that will go to the segment table */
  ogg_int64_t *granule_vals; /* granulepos values for headers. Not compact
				this way, but it is simple coupled to the
				lacing fifo */
  long    lacing_storage;
  long    lacing_fill;
  long    lacing_packet;
  long    lacing_returned;

  unsigned char    header[282];      /* working space for header encode */
  int              header_fill;

  int     e_o_s;          /* set when we have buffered the last packet in the
                             logical bitstream */
  int     b_o_s;          /* set after we've written the initial page
                             of a logical bitstream */
  long    serialno;
  long    pageno;
  ogg_int64_t  packetno;      /* sequence number for decode; the framing
                             knows where there's a hole in the data,
                             but we need coupling so that the codec
                             (which is in a seperate abstraction
                             layer) also knows about the gap */
  ogg_int64_t   granulepos;

} ogg_stream_state;

/* ogg_packet is used to encapsulate the data and metadata belonging
   to a single raw Ogg/Vorbis packet *************************************/

typedef struct {
  unsigned char *packet;
  long  bytes;
  long  b_o_s;
  long  e_o_s;

  ogg_int64_t  granulepos;
  
  ogg_int64_t  packetno;     /* sequence number for decode; the framing
				knows where there's a hole in the data,
				but we need coupling so that the codec
				(which is in a seperate abstraction
				layer) also knows about the gap */
} ogg_packet;

typedef struct {
  unsigned char *data;
  int storage;
  int fill;
  int returned;

  int unsynced;
  int headerbytes;
  int bodybytes;
} ogg_sync_state;

/* Ogg BITSTREAM PRIMITIVES: bitstream ************************/

extern void  oggpack_writeinit(oggpack_buffer *b);
extern void  oggpack_writetrunc(oggpack_buffer *b,long bits);
extern void  oggpack_writealign(oggpack_buffer *b);
extern void  oggpack_writecopy(oggpack_buffer *b,void *source,long bits);
extern void  oggpack_reset(oggpack_buffer *b);
extern void  oggpack_writeclear(oggpack_buffer *b);
extern void  oggpack_readinit(oggpack_buffer *b,unsigned char *buf,int bytes);
extern void  oggpack_write(oggpack_buffer *b,unsigned long value,int bits);
extern long  oggpack_look(oggpack_buffer *b,int bits);
extern long  oggpack_look1(oggpack_buffer *b);
extern void  oggpack_adv(oggpack_buffer *b,int bits);
extern void  oggpack_adv1(oggpack_buffer *b);
extern long  oggpack_read(oggpack_buffer *b,int bits);
extern long  oggpack_read1(oggpack_buffer *b);
extern long  oggpack_bytes(oggpack_buffer *b);
extern long  oggpack_bits(oggpack_buffer *b);
extern unsigned char *oggpack_get_buffer(oggpack_buffer *b);

extern void  oggpackB_writeinit(oggpack_buffer *b);
extern void  oggpackB_writetrunc(oggpack_buffer *b,long bits);
extern void  oggpackB_writealign(oggpack_buffer *b);
extern void  oggpackB_writecopy(oggpack_buffer *b,void *source,long bits);
extern void  oggpackB_reset(oggpack_buffer *b);
extern void  oggpackB_writeclear(oggpack_buffer *b);
extern void  oggpackB_readinit(oggpack_buffer *b,unsigned char *buf,int bytes);
extern void  oggpackB_write(oggpack_buffer *b,unsigned long value,int bits);
extern long  oggpackB_look(oggpack_buffer *b,int bits);
extern long  oggpackB_look1(oggpack_buffer *b);
extern void  oggpackB_adv(oggpack_buffer *b,int bits);
extern void  oggpackB_adv1(oggpack_buffer *b);
extern long  oggpackB_read(oggpack_buffer *b,int bits);
extern long  oggpackB_read1(oggpack_buffer *b);
extern long  oggpackB_bytes(oggpack_buffer *b);
extern long  oggpackB_bits(oggpack_buffer *b);
extern unsigned char *oggpackB_get_buffer(oggpack_buffer *b);

/* Ogg BITSTREAM PRIMITIVES: encoding **************************/

extern int      ogg_stream_packetin(ogg_stream_state *os, ogg_packet *op);
extern int      ogg_stream_pageout(ogg_stream_state *os, ogg_page *og);
extern int      ogg_stream_flush(ogg_stream_state *os, ogg_page *og);

/* Ogg BITSTREAM PRIMITIVES: decoding **************************/

extern int      ogg_sync_init(ogg_sync_state *oy);
extern int      ogg_sync_clear(ogg_sync_state *oy);
extern int      ogg_sync_reset(ogg_sync_state *oy);
extern int	ogg_sync_destroy(ogg_sync_state *oy);

extern char    *ogg_sync_buffer(ogg_sync_state *oy, long size);
extern int      ogg_sync_wrote(ogg_sync_state *oy, long bytes);
extern long     ogg_sync_pageseek(ogg_sync_state *oy,ogg_page *og);
extern int      ogg_sync_pageout(ogg_sync_state *oy, ogg_page *og);
extern int      ogg_stream_pagein(ogg_stream_state *os, ogg_page *og);
extern int      ogg_stream_packetout(ogg_stream_state *os,ogg_packet *op);
extern int      ogg_stream_packetpeek(ogg_stream_state *os,ogg_packet *op);

/* Ogg BITSTREAM PRIMITIVES: general ***************************/

extern int      ogg_stream_init(ogg_stream_state *os,int serialno);
extern int      ogg_stream_clear(ogg_stream_state *os);
extern int      ogg_stream_reset(ogg_stream_state *os);
extern int      ogg_stream_reset_serialno(ogg_stream_state *os,int serialno);
extern int      ogg_stream_destroy(ogg_stream_state *os);
extern int      ogg_stream_eos(ogg_stream_state *os);

extern void     ogg_page_checksum_set(ogg_page *og);

extern int      ogg_page_version(ogg_page *og);
extern int      ogg_page_continued(ogg_page *og);
extern int      ogg_page_bos(ogg_page *og);
extern int      ogg_page_eos(ogg_page *og);
extern ogg_int64_t  ogg_page_granulepos(ogg_page *og);
extern int      ogg_page_serialno(ogg_page *og);
extern long     ogg_page_pageno(ogg_page *og);
extern int      ogg_page_packets(ogg_page *og);

extern void     ogg_packet_clear(ogg_packet *op);


typedef struct vorbis_info{
  int version;
  int channels;
  long rate;

  /* The below bitrate declarations are *hints*.
     Combinations of the three values carry the following implications:
     
     all three set to the same value: 
       implies a fixed rate bitstream
     only nominal set: 
       implies a VBR stream that averages the nominal bitrate.  No hard 
       upper/lower limit
     upper and or lower set: 
       implies a VBR bitstream that obeys the bitrate limits. nominal 
       may also be set to give a nominal rate.
     none set:
       the coder does not care to speculate.
  */

  long bitrate_upper;
  long bitrate_nominal;
  long bitrate_lower;
  long bitrate_window;

  void *codec_setup;
} vorbis_info;

/* vorbis_dsp_state buffers the current vorbis audio
   analysis/synthesis state.  The DSP state belongs to a specific
   logical bitstream ****************************************************/
typedef struct vorbis_dsp_state{
  int analysisp;
  vorbis_info *vi;

  float **pcm;
  float **pcmret;
  int      pcm_storage;
  int      pcm_current;
  int      pcm_returned;

  int  preextrapolate;
  int  eofflag;

  long lW;
  long W;
  long nW;
  long centerW;

  ogg_int64_t granulepos;
  ogg_int64_t sequence;

  ogg_int64_t glue_bits;
  ogg_int64_t time_bits;
  ogg_int64_t floor_bits;
  ogg_int64_t res_bits;

  void       *backend_state;
} vorbis_dsp_state;

typedef struct vorbis_block{
  /* necessary stream state for linking to the framing abstraction */
  float  **pcm;       /* this is a pointer into local storage */ 
  oggpack_buffer opb;
  
  long  lW;
  long  W;
  long  nW;
  int   pcmend;
  int   mode;

  int         eofflag;
  ogg_int64_t granulepos;
  ogg_int64_t sequence;
  vorbis_dsp_state *vd; /* For read-only access of configuration */

  /* local storage to avoid remallocing; it's up to the mapping to
     structure it */
  void               *localstore;
  long                localtop;
  long                localalloc;
  long                totaluse;
  struct alloc_chain *reap;

  /* bitmetrics for the frame */
  long glue_bits;
  long time_bits;
  long floor_bits;
  long res_bits;

  void *internal;

} vorbis_block;

/* vorbis_block is a single block of data to be processed as part of
the analysis/synthesis stream; it belongs to a specific logical
bitstream, but is independant from other vorbis_blocks belonging to
that logical bitstream. *************************************************/

struct alloc_chain{
  void *ptr;
  struct alloc_chain *next;
};

/* vorbis_info contains all the setup information specific to the
   specific compression/decompression mode in progress (eg,
   psychoacoustic settings, channel setup, options, codebook
   etc). vorbis_info and substructures are in backends.h.
*********************************************************************/

/* the comments are not part of vorbis_info so that vorbis_info can be
   static storage */
typedef struct vorbis_comment{
  /* unlimited user comment fields.  libvorbis writes 'libvorbis'
     whatever vendor is set to in encode */
  char **user_comments;
  int   *comment_lengths;
  int    comments;
  char  *vendor;

} vorbis_comment;


/* libvorbis encodes in two abstraction layers; first we perform DSP
   and produce a packet (see docs/analysis.txt).  The packet is then
   coded into a framed OggSquish bitstream by the second layer (see
   docs/framing.txt).  Decode is the reverse process; we sync/frame
   the bitstream and extract individual packets, then decode the
   packet back into PCM audio.

   The extra framing/packetizing is used in streaming formats, such as
   files.  Over the net (such as with UDP), the framing and
   packetization aren't necessary as they're provided by the transport
   and the streaming layer is not used */

/* Vorbis PRIMITIVES: general ***************************************/

extern void     vorbis_info_init(vorbis_info *vi);
extern void     vorbis_info_clear(vorbis_info *vi);
extern int      vorbis_info_blocksize(vorbis_info *vi,int zo);
extern void     vorbis_comment_init(vorbis_comment *vc);
extern void     vorbis_comment_add(vorbis_comment *vc, char *comment); 
extern void     vorbis_comment_add_tag(vorbis_comment *vc, 
				       char *tag, char *contents);
extern char    *vorbis_comment_query(vorbis_comment *vc, char *tag, int count);
extern int      vorbis_comment_query_count(vorbis_comment *vc, char *tag);
extern void     vorbis_comment_clear(vorbis_comment *vc);

extern int      vorbis_block_init(vorbis_dsp_state *v, vorbis_block *vb);
extern int      vorbis_block_clear(vorbis_block *vb);
extern void     vorbis_dsp_clear(vorbis_dsp_state *v);
extern double   vorbis_granule_time(vorbis_dsp_state *v,
				    ogg_int64_t granulepos);

/* Vorbis PRIMITIVES: analysis/DSP layer ****************************/

extern int      vorbis_analysis_init(vorbis_dsp_state *v,vorbis_info *vi);
extern int      vorbis_commentheader_out(vorbis_comment *vc, ogg_packet *op);
extern int      vorbis_analysis_headerout(vorbis_dsp_state *v,
					  vorbis_comment *vc,
					  ogg_packet *op,
					  ogg_packet *op_comm,
					  ogg_packet *op_code);
extern float  **vorbis_analysis_buffer(vorbis_dsp_state *v,int vals);
extern int      vorbis_analysis_wrote(vorbis_dsp_state *v,int vals);
extern int      vorbis_analysis_blockout(vorbis_dsp_state *v,vorbis_block *vb);
extern int      vorbis_analysis(vorbis_block *vb,ogg_packet *op);

extern int      vorbis_bitrate_addblock(vorbis_block *vb);
extern int      vorbis_bitrate_flushpacket(vorbis_dsp_state *vd,
					   ogg_packet *op);

/* Vorbis PRIMITIVES: synthesis layer *******************************/
extern int      vorbis_synthesis_headerin(vorbis_info *vi,vorbis_comment *vc,
					  ogg_packet *op);

extern int      vorbis_synthesis_init(vorbis_dsp_state *v,vorbis_info *vi);
extern int      vorbis_synthesis_restart(vorbis_dsp_state *v);
extern int      vorbis_synthesis(vorbis_block *vb,ogg_packet *op);
extern int      vorbis_synthesis_trackonly(vorbis_block *vb,ogg_packet *op);
extern int      vorbis_synthesis_blockin(vorbis_dsp_state *v,vorbis_block *vb);
extern int      vorbis_synthesis_pcmout(vorbis_dsp_state *v,float ***pcm);
extern int      vorbis_synthesis_lapout(vorbis_dsp_state *v,float ***pcm);
extern int      vorbis_synthesis_read(vorbis_dsp_state *v,int samples);
extern long     vorbis_packet_blocksize(vorbis_info *vi,ogg_packet *op);

extern int      vorbis_synthesis_halfrate(vorbis_info *v,int flag);
extern int      vorbis_synthesis_halfrate_p(vorbis_info *v);

/* Vorbis ERRORS and return codes ***********************************/

#define OV_FALSE      -1  
#define OV_EOF        -2
#define OV_HOLE       -3

#define OV_EREAD      -128
#define OV_EFAULT     -129
#define OV_EIMPL      -130
#define OV_EINVAL     -131
#define OV_ENOTVORBIS -132
#define OV_EBADHEADER -133
#define OV_EVERSION   -134
#define OV_ENOTAUDIO  -135
#define OV_EBADPACKET -136
#define OV_EBADLINK   -137
#define OV_ENOSEEK    -138

typedef void TIMER;
typedef long (*audio_read_func)(void *src, float **buffer, int samples);
typedef void (*progress_func)(char *fn, long totalsamples, 
		long samples, double time);
typedef void (*enc_end_func)(char *fn, double time, int rate, 
		long samples, long bytes);
typedef void (*enc_start_func)(char *fn, char *outfn, int bitrate, 
        float quality, int qset, int managed, int min_br, int max_br);
typedef void (*error_func)(char *errormessage);


void *timer_start(void);
double timer_time(void *);
void timer_clear(void *);
int create_directories(char *);

void update_statistics_full(char *fn, long total, long done, double time);
void update_statistics_notime(char *fn, long total, long done, double time);
void update_statistics_null(char *fn, long total, long done, double time);
void start_encode_full(char *fn, char *outfn, int bitrate, float quality, int qset,
        int managed, int min, int max);
void start_encode_null(char *fn, char *outfn, int bitrate, float quality, int qset,
        int managed, int min, int max);
void final_statistics(char *fn, double time, int rate, long total_samples,
		long bytes);
void final_statistics_null(char *fn, double time, int rate, long total_samples,
		long bytes);
void encode_error(char *errmsg);

typedef struct {
    char *arg;
    char *val;
} adv_opt;

typedef struct
{
	char **title;
	int title_count;
	char **artist;
	int artist_count;
	char **album;
	int album_count;
	char **comments;
	int comment_count;
	char **tracknum;
	int track_count;
	char **dates;
	int date_count;
	char **genre;
	int genre_count;
    adv_opt *advopt;
    int advopt_count;
	int copy_comments;

	int quiet;

	int rawmode;
	int raw_samplesize;
	int raw_samplerate;
	int raw_channels;
    int raw_endianness;

	char *namefmt;
    char *namefmt_remove;
    char *namefmt_replace;
	char *outfile;

	/* All 3 in kbps */
    int managed;
	int min_bitrate;
	int nominal_bitrate;
	int max_bitrate;

	/* Float from 0 to 1 (low->high) */
	float quality;
    int quality_set;

    int resamplefreq;
    int downmix;
    float scale;

	unsigned int serial;
} oe_options;

typedef struct
{
	vorbis_comment *comments;
	unsigned int serialno;

	audio_read_func read_samples;
	progress_func progress_update;
	enc_end_func end_encode;
	enc_start_func start_encode;
	error_func error;
	
	void *readdata;

	long total_samples_per_channel;
	int channels;
	long rate;
	int samplesize;
    int endianness;
    int resamplefreq;
	int copy_comments;

	/* Various bitrate/quality options */
    int managed;
	int bitrate;
	int min_bitrate;
	int max_bitrate;
	float quality;
    int quality_set;
    adv_opt *advopt;
    int advopt_count;

	FILE *out;
	char *filename;
	char *infilename;
} oe_enc_opt;


int oe_encode(oe_enc_opt *opt);

#define __AUDIO_H

int setup_resample(oe_enc_opt *opt);
void clear_resample(oe_enc_opt *opt);
void setup_downmix(oe_enc_opt *opt);
void clear_downmix(oe_enc_opt *opt);
void setup_scaler(oe_enc_opt *opt, float scale);
void clear_scaler(oe_enc_opt *opt);

typedef struct
{
	int (*id_func)(unsigned char *buf, int len); /* Returns true if can load file */
	int id_data_len; /* Amount of data needed to id whether this can load the file */
	int (*open_func)(FILE *in, oe_enc_opt *opt, unsigned char *buf, int buflen);
	void (*close_func)(void *);
	char *format;
	char *description;
} input_format;


typedef struct {
	short format;
	short channels;
	int samplerate;
	int bytespersec;
	short align;
	short samplesize;
} wav_fmt;

typedef struct {
	short channels;
	short samplesize;
	long totalsamples;
	long samplesread;
	FILE *f;
	short bigendian;
} wavfile;

typedef struct {
	short channels;
	int totalframes;
	short samplesize;
	int rate;
	int offset;
	int blocksize;
} aiff_fmt;

typedef wavfile aifffile; /* They're the same */

input_format *open_audio_file(FILE *in, oe_enc_opt *opt);

int raw_open(FILE *in, oe_enc_opt *opt);
int wav_open(FILE *in, oe_enc_opt *opt, unsigned char *buf, int buflen);
int aiff_open(FILE *in, oe_enc_opt *opt, unsigned char *buf, int buflen);
int wav_id(unsigned char *buf, int len);
int aiff_id(unsigned char *buf, int len);
void wav_close(void *);
void raw_close(void *);

long wav_read(void *, float **buffer, int samples);
long wav_ieee_read(void *, float **buffer, int samples);
long raw_read_stereo(void *, float **buffer, int samples);

#define _(X) (X)
#define textdomain(X)
#define bindtextdomain(X, Y)
#define N_(X) (X)


#define VERSION_STRING "OggEnc v1.0.1 (libvorbis 1.0.1)\n"
#define COPYRIGHT "(c) 2000-2003 Michael Smith <msmith@xiph.org>\n"

#define CHUNK 4096 /* We do reads, etc. in multiples of this */

struct option long_options[] = {
	{"quiet",0,0,'Q'},
	{"help",0,0,'h'},
	{"comment",1,0,'c'},
	{"artist",1,0,'a'},
	{"album",1,0,'l'},
	{"title",1,0,'t'},
    {"genre",1,0,'G'},
	{"names",1,0,'n'},
    {"name-remove",1,0,'X'},
    {"name-replace",1,0,'P'},
	{"output",1,0,'o'},
	{"version",0,0,'v'},
	{"raw",0,0,'r'},
	{"raw-bits",1,0,'B'},
	{"raw-chan",1,0,'C'},
	{"raw-rate",1,0,'R'},
    {"raw-endianness",1,0, 0},
	{"bitrate",1,0,'b'},
	{"min-bitrate",1,0,'m'},
	{"max-bitrate",1,0,'M'},
	{"quality",1,0,'q'},
	{"date",1,0,'d'},
	{"tracknum",1,0,'N'},
	{"serial",1,0,'s'},
    {"managed", 0, 0, 0},
    {"resample",1,0,0},
    {"downmix", 0,0,0},
    {"scale", 1, 0, 0}, 
    {"advanced-encode-option", 1, 0, 0},
	{"discard-comments", 0, 0, 0},
	{NULL,0,0,0}
};
	
static char *generate_name_string(char *format, char *remove_list, 
        char *replace_list, char *artist, char *title, char *album, 
        char *track, char *date, char *genre);
static void parse_options(int argc, char **argv, oe_options *opt);
static void build_comments(vorbis_comment *vc, oe_options *opt, int filenum, 
		char **artist,char **album, char **title, char **tracknum, char **date,
        char **genre);
static void usage(void);

int main(int argc, char **argv)
{
	/* Default values */
	oe_options opt = {NULL, 0, NULL, 0, NULL, 0, NULL, 0, NULL, 
			  0, NULL, 0, NULL, 0, NULL, 0, 1, 0, 0,16,44100,2, 0, NULL,
			  DEFAULT_NAMEFMT_REMOVE, DEFAULT_NAMEFMT_REPLACE, 
			  NULL, 0, -1,-1,-1,.3,-1,0, 0,0.f, 0}; 

	int i;

	char **infiles;
	int numfiles;
	int errors=0;

	parse_options(argc, argv, &opt);

	if(optind >= argc)
	{
		fprintf(stderr, _("%s%s\nERROR: No input files specified. Use -h for help.\n"), VERSION_STRING, COPYRIGHT);
		return 1;
	}
	else
	{
		infiles = argv + optind;
		numfiles = argc - optind;
	}

	/* Now, do some checking for illegal argument combinations */

	for(i = 0; i < numfiles; i++)
	{
		if(!strcmp(infiles[i], "-") && numfiles > 1)
		{
			fprintf(stderr, _("ERROR: Multiple files specified when using stdin\n"));
			exit(1);
		}
	}

	if(numfiles > 1 && opt.outfile)
	{
		fprintf(stderr, _("ERROR: Multiple input files with specified output filename: suggest using -n\n"));
		exit(1);
	}

	if(opt.serial == 0)
	{
		/* We randomly pick a serial number. This is then incremented for each file */
		srand(time(NULL));
		opt.serial = rand();
	}

	for(i = 0; i < numfiles; i++)
	{
		/* Once through the loop for each file */

		oe_enc_opt      enc_opts;
		vorbis_comment  vc;
		char *out_fn = NULL;
		FILE *in, *out = NULL;
		int foundformat = 0;
		int closeout = 0, closein = 0;
		char *artist=NULL, *album=NULL, *title=NULL, *track=NULL;
        char *date=NULL, *genre=NULL;
		input_format *format;

		/* Set various encoding defaults */

		enc_opts.serialno = opt.serial++;
		enc_opts.progress_update = update_statistics_full;
        enc_opts.start_encode = start_encode_full;
		enc_opts.end_encode = final_statistics;
		enc_opts.error = encode_error;
		enc_opts.comments = &vc;
		enc_opts.copy_comments = opt.copy_comments;

		/* OK, let's build the vorbis_comments structure */
		build_comments(&vc, &opt, i, &artist, &album, &title, &track, 
                &date, &genre);

		if(!strcmp(infiles[i], "-"))
		{
			setbinmode(stdin);
			in = stdin;
            infiles[i] = NULL;
			if(!opt.outfile)
			{
				setbinmode(stdout);
				out = stdout;
			}
		}
		else
		{
			in = fopen(infiles[i], "rb");

			if(in == NULL)
			{
				fprintf(stderr, _("ERROR: Cannot open input file \"%s\": %s\n"), infiles[i], strerror(errno));
				free(out_fn);
				errors++;
				continue;
			}

			closein = 1;
		}

		/* Now, we need to select an input audio format - we do this before opening
		   the output file so that we don't end up with a 0-byte file if the input
		   file can't be read */

		if(opt.rawmode)
		{
			enc_opts.rate=opt.raw_samplerate;
			enc_opts.channels=opt.raw_channels;
			enc_opts.samplesize=opt.raw_samplesize;
            enc_opts.endianness=opt.raw_endianness;
			raw_open(in, &enc_opts);
			foundformat=1;
		}
		else
		{
			format = open_audio_file(in, &enc_opts);
			if(format)
			{
                if(!opt.quiet)
				    fprintf(stderr, _("Opening with %s module: %s\n"), 
					    	format->format, format->description);
				foundformat=1;
			}

		}

		if(!foundformat)
		{
			fprintf(stderr, _("ERROR: Input file \"%s\" is not a supported format\n"), infiles[i]?infiles[i]:"(stdin)");
    		if(closein)
				fclose(in);
			errors++;
			continue;
		}

		/* Ok. We can read the file - so now open the output file */

		if(opt.outfile && !strcmp(opt.outfile, "-"))
		{
			setbinmode(stdout);
			out = stdout;
		}
		else if(out == NULL)
		{
			if(opt.outfile)
			{
				out_fn = strdup(opt.outfile);
			}
			else if(opt.namefmt)
			{
				out_fn = generate_name_string(opt.namefmt, opt.namefmt_remove, 
                        opt.namefmt_replace, artist, title, album, track,date,
                        genre);
			}
            /* This bit was widely derided in mid-2002, so it's been removed */
            /*
			else if(opt.title)
			{
				out_fn = malloc(strlen(title) + 5);
				strcpy(out_fn, title);
				strcat(out_fn, ".ogg");
			}
            */
			else if(infiles[i])
			{
				/* Create a filename from existing filename, replacing extension with .ogg */
				char *start, *end;

				start = infiles[i];
				end = strrchr(infiles[i], '.');
				end = end?end:(start + strlen(infiles[i])+1);
			
				out_fn = malloc(end - start + 5);
				strncpy(out_fn, start, end-start);
				out_fn[end-start] = 0;
				strcat(out_fn, ".ogg");
			}
            else {
                fprintf(stderr, _("WARNING: No filename, defaulting to \"default.ogg\"\n"));
                out_fn = strdup("default.ogg");
            }

            /* Create any missing subdirectories, if possible */
            if(create_directories(out_fn)) {
                if(closein)
                    fclose(in);
				fprintf(stderr, _("ERROR: Could not create required subdirectories for output filename \"%s\"\n"), out_fn);
				errors++;
				free(out_fn);
				continue;
            }

			out = fopen(out_fn, "wb");
			if(out == NULL)
			{
				if(closein)
					fclose(in);
				fprintf(stderr, _("ERROR: Cannot open output file \"%s\": %s\n"), out_fn, strerror(errno));
				errors++;
				free(out_fn);
				continue;
			}	
			closeout = 1;
		}

		/* Now, set the rest of the options */
		enc_opts.out = out;
		enc_opts.comments = &vc;
		enc_opts.filename = out_fn;
		enc_opts.infilename = infiles[i];
        enc_opts.managed = opt.managed;
		enc_opts.bitrate = opt.nominal_bitrate; 
		enc_opts.min_bitrate = opt.min_bitrate;
		enc_opts.max_bitrate = opt.max_bitrate;
		enc_opts.quality = opt.quality;
		enc_opts.quality_set = opt.quality_set;
        enc_opts.advopt = opt.advopt;
        enc_opts.advopt_count = opt.advopt_count;

        if(opt.resamplefreq && opt.resamplefreq != enc_opts.rate) {
            int fromrate = enc_opts.rate;
            enc_opts.resamplefreq = opt.resamplefreq;
            if(setup_resample(&enc_opts)) {
                errors++;
                goto clear_all;
            }
            else if(!opt.quiet)
                fprintf(stderr, _("Resampling input from %d Hz to %d Hz\n"), fromrate, opt.resamplefreq);
        }

        if(opt.downmix) {
            if(enc_opts.channels == 2) {
                setup_downmix(&enc_opts);
                if(!opt.quiet)
                    fprintf(stderr, _("Downmixing stereo to mono\n"));
            }
            else {
                fprintf(stderr, _("ERROR: Can't downmix except from stereo to mono\n"));
                errors++;
                if(opt.resamplefreq && opt.resamplefreq != enc_opts.rate)
                    clear_resample(&enc_opts);
                goto clear_all;
            }
        }

        if(opt.scale > 0.f) {
            setup_scaler(&enc_opts, opt.scale);
            if(!opt.quiet)
                fprintf(stderr, _("Scaling input to %f\n"), opt.scale);
        }


		if(!enc_opts.total_samples_per_channel)
			enc_opts.progress_update = update_statistics_notime;

		if(opt.quiet)
		{
            enc_opts.start_encode = start_encode_null;
			enc_opts.progress_update = update_statistics_null;
			enc_opts.end_encode = final_statistics_null;
		}

		if(oe_encode(&enc_opts))
			errors++;

        if(opt.scale > 0)
            clear_scaler(&enc_opts);
        if(opt.downmix)
            clear_downmix(&enc_opts);
        if(opt.resamplefreq && opt.resamplefreq != enc_opts.rate)
            clear_resample(&enc_opts);
clear_all:

		if(out_fn) free(out_fn);
        if(opt.outfile) free(opt.outfile);
		vorbis_comment_clear(&vc);
		if(!opt.rawmode) 
			format->close_func(enc_opts.readdata);

		if(closein)
			fclose(in);
		if(closeout)
			fclose(out);
	}/* Finished this file, loop around to next... */

	return errors?1:0;

}

static void usage(void)
{
	fprintf(stdout, 
		_("%s%s\n"
		"Usage: oggenc [options] input.wav [...]\n"
		"\n"
		"OPTIONS:\n"
		" General:\n"
		" -Q, --quiet          Produce no output to stderr\n"
		" -h, --help           Print this help text\n"
		" -r, --raw            Raw mode. Input files are read directly as PCM data\n"
		" -B, --raw-bits=n     Set bits/sample for raw input. Default is 16\n"
		" -C, --raw-chan=n     Set number of channels for raw input. Default is 2\n"
		" -R, --raw-rate=n     Set samples/sec for raw input. Default is 44100\n"
        " --raw-endianness     1 for bigendian, 0 for little (defaults to 0)\n"
		" -b, --bitrate        Choose a nominal bitrate to encode at. Attempt\n"
		"                      to encode at a bitrate averaging this. Takes an\n"
		"                      argument in kbps. This uses the bitrate management\n"
        "                      engine, and is not recommended for most users.\n"
        "                      See -q, --quality for a better alternative.\n"
		" -m, --min-bitrate    Specify a minimum bitrate (in kbps). Useful for\n"
		"                      encoding for a fixed-size channel.\n"
		" -M, --max-bitrate    Specify a maximum bitrate in kbps. Useful for\n"
		"                      streaming applications.\n"
		" -q, --quality        Specify quality between 0 (low) and 10 (high),\n"
		"                      instead of specifying a particular bitrate.\n"
		"                      This is the normal mode of operation.\n"
        "                      Fractional qualities (e.g. 2.75) are permitted\n"
        "                      Quality -1 is also possible, but may not be of\n"
        "                      acceptable quality.\n"
        " --resample n         Resample input data to sampling rate n (Hz)\n"
        " --downmix            Downmix stereo to mono. Only allowed on stereo\n"
        "                      input.\n"
		" -s, --serial         Specify a serial number for the stream. If encoding\n"
		"                      multiple files, this will be incremented for each\n"
		"                      stream after the first.\n"
		" --discard-comments   Prevents comments in FLAC and Ogg FLAC files from\n"
		"                      being copied to the output Ogg Vorbis file.\n"
		"\n"
		" Naming:\n"
		" -o, --output=fn      Write file to fn (only valid in single-file mode)\n"
		" -n, --names=string   Produce filenames as this string, with %%a, %%t, %%l,\n"
		"                      %%n, %%d replaced by artist, title, album, track number,\n"
		"                      and date, respectively (see below for specifying these).\n"
		"                      %%%% gives a literal %%.\n"
        " -X, --name-remove=s  Remove the specified characters from parameters to the\n"
        "                      -n format string. Useful to ensure legal filenames.\n"
        " -P, --name-replace=s Replace characters removed by --name-remove with the\n"
        "                      characters specified. If this string is shorter than the\n"
        "                      --name-remove list or is not specified, the extra\n"
        "                      characters are just removed.\n"
        "                      Default settings for the above two arguments are platform\n"
        "                      specific.\n"
		" -c, --comment=c      Add the given string as an extra comment. This may be\n"
		"                      used multiple times. The argument should be in the\n"
        "                      format \"tag=value\".\n"
		" -d, --date           Date for track (usually date of performance)\n"
		" -N, --tracknum       Track number for this track\n"
		" -t, --title          Title for this track\n"
		" -l, --album          Name of album\n"
		" -a, --artist         Name of artist\n"
        " -G, --genre          Genre of track\n"
		"                      If multiple input files are given, then multiple\n"
		"                      instances of the previous five arguments will be used,\n"
		"                      in the order they are given. If fewer titles are\n"
		"                      specified than files, OggEnc will print a warning, and\n"
		"                      reuse the final one for the remaining files. If fewer\n"
		"                      track numbers are given, the remaining files will be\n"
		"                      unnumbered. For the others, the final tag will be reused\n"
		"                      for all others without warning (so you can specify a date\n"
		"                      once, for example, and have it used for all the files)\n"
		"\n"
		"INPUT FILES:\n"
		" OggEnc input files must currently be 24, 16, or 8 bit PCM WAV, AIFF, or AIFF/C\n"
		" files, 32 bit IEEE floating point WAV, and optionally FLAC or Ogg FLAC. Files\n"
                "  may be mono or stereo (or more channels) and any sample rate.\n"
		" Alternatively, the --raw option may be used to use a raw PCM data file, which\n"
		" must be 16 bit stereo little-endian PCM ('headerless wav'), unless additional\n"
		" parameters for raw mode are specified.\n"
		" You can specify taking the file from stdin by using - as the input filename.\n"
		" In this mode, output is to stdout unless an output filename is specified\n"
		" with -o\n"
		"\n"), VERSION_STRING, COPYRIGHT);
}

static int strncpy_filtered(char *dst, char *src, int len, char *remove_list, 
        char *replace_list)
{
    char *hit, *drop_margin;
    int used=0;

    if(remove_list == NULL || *remove_list == 0)
    {
        strncpy(dst, src, len-1);
        dst[len-1] = 0;
        return strlen(dst);
    }

    drop_margin = remove_list + (replace_list == NULL?0:strlen(replace_list));

    while(*src && used < len-1)
    {
        if((hit = strchr(remove_list, *src)) != NULL)
        {
            if(hit < drop_margin)
            {
                *dst++ = replace_list[hit - remove_list];
                used++;
            }
        }
        else
        {
            *dst++ = *src;
            used++;
        }
        src++;
    }
    *dst = 0;

    return used;
}

static char *generate_name_string(char *format, char *remove_list,
        char *replace_list, char *artist, char *title, char *album, 
        char *track, char *date, char *genre)
{
	char *buffer;
	char next;
	char *string;
	int used=0;
	int buflen;

	buffer = calloc(CHUNK+1,1);
	buflen = CHUNK;

	while(*format && used < buflen)
	{
		next = *format++;

		if(next == '%')
		{
			switch(*format++)
			{
				case '%':
					*(buffer+(used++)) = '%';
					break;
				case 'a':
					string = artist?artist:_("(none)");
					used += strncpy_filtered(buffer+used, string, buflen-used, 
                            remove_list, replace_list);
					break;
				case 'd':
					string = date?date:_("(none)");
					used += strncpy_filtered(buffer+used, string, buflen-used,
                            remove_list, replace_list);
					break;
                case 'g':
                    string = genre?genre:_("(none)");
                    used += strncpy_filtered(buffer+used, string, buflen-used,
                            remove_list, replace_list);
                    break;
				case 't':
					string = title?title:_("(none)");
					used += strncpy_filtered(buffer+used, string, buflen-used,
                            remove_list, replace_list);
					break;
				case 'l':
					string = album?album:_("(none)");
					used += strncpy_filtered(buffer+used, string, buflen-used,
                            remove_list, replace_list);
					break;
				case 'n':
					string = track?track:_("(none)");
					used += strncpy_filtered(buffer+used, string, buflen-used,
                            remove_list, replace_list);
					break;
				default:
					fprintf(stderr, _("WARNING: Ignoring illegal escape character '%c' in name format\n"), *(format - 1));
					break;
			}
		}
		else
			*(buffer + (used++)) = next;
	}

	return buffer;
}

static void parse_options(int argc, char **argv, oe_options *opt)
{
	int ret;
	int option_index = 1;

	while((ret = getopt_long(argc, argv, "A:a:b:B:c:C:d:G:hl:m:M:n:N:o:P:q:QrR:s:t:vX:", 
					long_options, &option_index)) != -1)
	{
		switch(ret)
		{
			case 0:
                if(!strcmp(long_options[option_index].name, "managed")) {
		            if(!opt->managed){
                        if(!opt->quiet)
            		        fprintf(stderr, 
                                    _("Enabling bitrate management engine\n"));
                        opt->managed = 1;
        		    }
                }
                else if(!strcmp(long_options[option_index].name, 
                            "raw-endianness")) {
				    if (opt->rawmode != 1)
    				{
	    				opt->rawmode = 1;
		    			fprintf(stderr, _("WARNING: Raw endianness specified for non-raw data. Assuming input is raw.\n"));
			    	}
				    if(sscanf(optarg, "%d", &opt->raw_endianness) != 1) {
                        fprintf(stderr, _("WARNING: Couldn't read endianness argument \"%s\"\n"), optarg);
    					opt->raw_endianness = 0;
                    }
                }
                else if(!strcmp(long_options[option_index].name,
                            "resample")) {
				    if(sscanf(optarg, "%d", &opt->resamplefreq) != 1) {
                        fprintf(stderr, _("WARNING: Couldn't read resampling frequency \"%s\"\n"), optarg);
    					opt->resamplefreq = 0;
                    }
                    if(opt->resamplefreq < 100) /* User probably specified it
                                                   in kHz accidently */
                        fprintf(stderr, 
                                _("Warning: Resample rate specified as %d Hz. Did you mean %d Hz?\n"), 
                                opt->resamplefreq, opt->resamplefreq*1000);
                }
                else if(!strcmp(long_options[option_index].name, "downmix")) {
                    opt->downmix = 1;
                }
                else if(!strcmp(long_options[option_index].name, "scale")) {
                    opt->scale = atof(optarg);
				    if(sscanf(optarg, "%f", &opt->scale) != 1) {
                        opt->scale = 0;
                        fprintf(stderr, _("Warning: Couldn't parse scaling factor \"%s\"\n"), 
                                optarg);
                    }
                }
                else if(!strcmp(long_options[option_index].name, "advanced-encode-option")) {
                    char *arg = strdup(optarg);
                    char *val;

                    val = strchr(arg, '=');
                    if(val == NULL) {
                        fprintf(stderr, _("No value for advanced encoder option found\n"));
                        continue;
                    }
                    else
                        *val++=0;

                    opt->advopt = realloc(opt->advopt, (++opt->advopt_count)*sizeof(adv_opt));
                    opt->advopt[opt->advopt_count - 1].arg = arg;
                    opt->advopt[opt->advopt_count - 1].val = val;
                }
                else if(!strcmp(long_options[option_index].name, "discard-comments")) {
		  opt->copy_comments = 0;
		}

                else {
				    fprintf(stderr, _("Internal error parsing command line options\n"));
				    exit(1);
                }

				break;
			case 'a':
				opt->artist = realloc(opt->artist, (++opt->artist_count)*sizeof(char *));
				opt->artist[opt->artist_count - 1] = strdup(optarg);
				break;
			case 'c':
                if(strchr(optarg, '=') == NULL) {
                    fprintf(stderr, _("Warning: Illegal comment used (\"%s\"), ignoring.\n"), optarg);
                    break;
                }
				opt->comments = realloc(opt->comments, (++opt->comment_count)*sizeof(char *));
				opt->comments[opt->comment_count - 1] = strdup(optarg);
				break;
			case 'd':
				opt->dates = realloc(opt->dates, (++opt->date_count)*sizeof(char *));
				opt->dates[opt->date_count - 1] = strdup(optarg);
				break;
            case 'G':
                opt->genre = realloc(opt->genre, (++opt->genre_count)*sizeof(char *));
                opt->genre[opt->genre_count - 1] = strdup(optarg);
                break;
			case 'h':
				usage();
				exit(0);
				break;
			case 'l':
				opt->album = realloc(opt->album, (++opt->album_count)*sizeof(char *));
				opt->album[opt->album_count - 1] = strdup(optarg);
				break;
			case 's':
				/* Would just use atoi(), but that doesn't deal with unsigned
				 * ints. Damn */
				if(sscanf(optarg, "%u", &opt->serial) != 1)
					opt->serial = 0; /* Failed, so just set to zero */
				break;
			case 't':
				opt->title = realloc(opt->title, (++opt->title_count)*sizeof(char *));
				opt->title[opt->title_count - 1] = strdup(optarg);
				break;
			case 'b':
   				if(sscanf(optarg, "%d", &opt->nominal_bitrate)
    					!= 1) {
	    			fprintf(stderr, _("Warning: nominal bitrate \"%s\" not recognised\n"), optarg);
		    		opt->nominal_bitrate = -1;
				}

				break;
			case 'm':
				if(sscanf(optarg, "%d", &opt->min_bitrate)
						!= 1) {
					fprintf(stderr, _("Warning: minimum bitrate \"%s\" not recognised\n"), optarg);
					opt->min_bitrate = -1;
				}
				if(!opt->managed){
				  if(!opt->quiet)
				    fprintf(stderr, 
					    _("Enabling bitrate management engine\n"));
				  opt->managed = 1;
				}
				break;
			case 'M':
				if(sscanf(optarg, "%d", &opt->max_bitrate)
						!= 1) {
					fprintf(stderr, _("Warning: maximum bitrate \"%s\" not recognised\n"), optarg);
					opt->max_bitrate = -1;
				}
				if(!opt->managed){
				  if(!opt->quiet)
				    fprintf(stderr, 
					    _("Enabling bitrate management engine\n"));
				  opt->managed = 1;
				}
				break;
			case 'q':
				if(sscanf(optarg, "%f", &opt->quality) != 1) {
					fprintf(stderr, _("Quality option \"%s\" not recognised, ignoring\n"), optarg);
					break;
				}
				opt->quality_set=1;
				opt->quality *= 0.1;
				if(opt->quality > 1.0f)
				{
					opt->quality = 1.0f;
					fprintf(stderr, _("WARNING: quality setting too high, setting to maximum quality.\n"));
				}
				break;
			case 'n':
				if(opt->namefmt)
				{
					fprintf(stderr, _("WARNING: Multiple name formats specified, using final\n"));
					free(opt->namefmt);
				}
				opt->namefmt = strdup(optarg);
				break;
            case 'X':
				if(opt->namefmt_remove && opt->namefmt_remove != 
                        DEFAULT_NAMEFMT_REMOVE)
				{
					fprintf(stderr, _("WARNING: Multiple name format filters specified, using final\n"));
					free(opt->namefmt_remove);
				}
				opt->namefmt_remove = strdup(optarg);
				break;
            case 'P':
				if(opt->namefmt_replace && opt->namefmt_replace != 
                        DEFAULT_NAMEFMT_REPLACE)
                {
					fprintf(stderr, _("WARNING: Multiple name format filter replacements specified, using final\n"));
					free(opt->namefmt_replace);
				}
				opt->namefmt_replace = strdup(optarg);
				break;
			case 'o':
				if(opt->outfile)
				{
					fprintf(stderr, _("WARNING: Multiple output files specified, suggest using -n\n"));
					free(opt->outfile);
				}
				opt->outfile = strdup(optarg);
				break;
			case 'Q':
				opt->quiet = 1;
				break;
			case 'r':
				opt->rawmode = 1;
				break;
			case 'v':
				fprintf(stdout, VERSION_STRING);
				exit(0);
				break;
			case 'B':
				if (opt->rawmode != 1)
				{
					opt->rawmode = 1;
					fprintf(stderr, _("WARNING: Raw bits/sample specified for non-raw data. Assuming input is raw.\n"));
				}
				if(sscanf(optarg, "%u", &opt->raw_samplesize) != 1)
				{
					opt->raw_samplesize = 16; /* Failed, so just set to 16 */
					fprintf(stderr, _("WARNING: Invalid bits/sample specified, assuming 16.\n"));
				}
				if((opt->raw_samplesize != 8) && (opt->raw_samplesize != 16))
				{
					fprintf(stderr, _("WARNING: Invalid bits/sample specified, assuming 16.\n"));
				}
				break;
			case 'C':
				if (opt->rawmode != 1)
				{
					opt->rawmode = 1;
					fprintf(stderr, _("WARNING: Raw channel count specified for non-raw data. Assuming input is raw.\n"));
				}
				if(sscanf(optarg, "%u", &opt->raw_channels) != 1)
				{
					opt->raw_channels = 2; /* Failed, so just set to 2 */
					fprintf(stderr, _("WARNING: Invalid channel count specified, assuming 2.\n"));
				}
				break;
			case 'N':
				opt->tracknum = realloc(opt->tracknum, (++opt->track_count)*sizeof(char *));
				opt->tracknum[opt->track_count - 1] = strdup(optarg);
				break;
			case 'R':
				if (opt->rawmode != 1)
				{
					opt->rawmode = 1;
					fprintf(stderr, _("WARNING: Raw sample rate specified for non-raw data. Assuming input is raw.\n"));
				}
				if(sscanf(optarg, "%u", &opt->raw_samplerate) != 1)
				{
					opt->raw_samplerate = 44100; /* Failed, so just set to 44100 */
					fprintf(stderr, _("WARNING: Invalid sample rate specified, assuming 44100.\n"));
				}
				break;
			case '?':
				fprintf(stderr, _("WARNING: Unknown option specified, ignoring->\n"));
				break;
			default:
				usage();
				exit(0);
		}
	}

}

static void add_tag(vorbis_comment *vc, oe_options *opt,char *name, char *value)
{
  if(name == NULL)
    vorbis_comment_add(vc, value);
  else
    vorbis_comment_add_tag(vc, name, value);
}

static void build_comments(vorbis_comment *vc, oe_options *opt, int filenum, 
		char **artist, char **album, char **title, char **tracknum, 
        char **date, char **genre)
{
	int i;

	vorbis_comment_init(vc);

	for(i = 0; i < opt->comment_count; i++)
		add_tag(vc, opt, NULL, opt->comments[i]);

	if(opt->title_count)
	{
		if(filenum >= opt->title_count)
		{
			if(!opt->quiet)
				fprintf(stderr, _("WARNING: Insufficient titles specified, defaulting to final title.\n"));
			i = opt->title_count-1;
		}
		else
			i = filenum;

		*title = opt->title[i];
		add_tag(vc, opt, "title", opt->title[i]);
	}

	if(opt->artist_count)
	{
		if(filenum >= opt->artist_count)
			i = opt->artist_count-1;
		else
			i = filenum;
	
		*artist = opt->artist[i];
		add_tag(vc, opt, "artist", opt->artist[i]);
	}

    if(opt->genre_count)
    {
        if(filenum >= opt->genre_count)
            i = opt->genre_count-1;
        else
            i = filenum;

        *genre = opt->genre[i];
        add_tag(vc, opt, "genre", opt->genre[i]);
    }

	if(opt->date_count)
	{
		if(filenum >= opt->date_count)
			i = opt->date_count-1;
		else
			i = filenum;
	
		*date = opt->dates[i];
		add_tag(vc, opt, "date", opt->dates[i]);
	}
	
	if(opt->album_count)
	{
		if(filenum >= opt->album_count)
		{
			i = opt->album_count-1;
		}
		else
			i = filenum;

		*album = opt->album[i];	
		add_tag(vc, opt, "album", opt->album[i]);
	}

	if(filenum < opt->track_count)
	{
		i = filenum;
		*tracknum = opt->tracknum[i];
		add_tag(vc, opt, "tracknumber", opt->tracknum[i]);
	}
}

/* OggEnc
 **
 ** This program is distributed under the GNU General Public License, version 2.
 ** A copy of this license is included with this source.
 **
 ** Copyright 2000-2002, Michael Smith <msmith@xiph.org>
 **
 ** AIFF/AIFC support from OggSquish, (c) 1994-1996 Monty <xiphmont@xiph.org>
 **/


/* This program is licensed under the GNU Library General Public License,
 * version 2, a copy of which is included with this program (LICENCE.LGPL).
 *   
 * (c) 2002 Simon Hosie <gumboot@clear.net.nz>
 *
 *
 * A resampler
 *
 * reference:
 * 	'Digital Filters', third edition, by R. W. Hamming  ISBN 0-486-65088-X
 *
 * history:
 *	2002-05-31	ready for the world (or some small section thereof)
 *
 *
 * TOOD:
 * 	zero-crossing clipping in coefficient table
 */

#define _RESAMPLE_H_INCLUDED

typedef float SAMPLE;

typedef struct
{
	unsigned int channels, infreq, outfreq, taps;
	float *table;
	SAMPLE *pool;

	/* dynamic bits */
	int poolfill;
	int offset;
} res_state;

typedef enum
{
	RES_END,
	RES_GAIN,	/* (double)1.0 */
	RES_CUTOFF,	/* (double)0.80 */ 
	RES_TAPS,	/* (int)45 */
	RES_BETA	/* (double)16.0 */
} res_parameter;

int res_init(res_state *state, int channels, int outfreq, int infreq, res_parameter op1, ...);
/*
 * Configure *state to manage a data stream with the specified parameters.  The
 * string 'params' is currently unspecified, but will configure the parameters
 * of the filter.
 *
 * This function allocates memory, and requires that res_clear() be called when
 * the buffer is no longer needed.
 *
 *
 * All counts/lengths used in the following functions consider only the data in
 * a single channel, and in numbers of samples rather than bytes, even though
 * functionality will be mirrored across as many channels as specified here.
 */


int res_push_max_input(res_state const *state, size_t maxoutput);
/*
 *  Returns the maximum number of input elements that may be provided without
 *  risk of flooding an output buffer of size maxoutput.  maxoutput is
 *  specified in counts of elements, NOT in bytes.
 */


int res_push_check(res_state const *state, size_t srclen);
/*
 * Returns the number of elements that will be returned if the given srclen
 * is used in the next call to res_push().
 */


int res_push(res_state *state, SAMPLE **dstlist, SAMPLE const **srclist, size_t srclen);
int res_push_interleaved(res_state *state, SAMPLE *dest, SAMPLE const *source, size_t srclen);
/*
 * Pushes srclen samples into the front end of the filter, and returns the
 * number of resulting samples.
 *
 * res_push(): srclist and dstlist point to lists of pointers, each of which
 * indicates the beginning of a list of samples.
 *
 * res_push_interleaved(): source and dest point to the beginning of a list of
 * interleaved samples.
 */


int res_drain(res_state *state, SAMPLE **dstlist);
int res_drain_interleaved(res_state *state, SAMPLE *dest);
/*
 * Recover the remaining elements by flushing the internal pool with 0 values,
 * and storing the resulting samples.
 *
 * After either of these functions are called, *state should only re-used in a
 * final call to res_clear().
 */


void res_clear(res_state *state);
/*
 * Free allocated buffers, etc.
 */

#define WAV_HEADER_SIZE 44

/* Macros to read header data */
#define READ_U32_LE(buf) \
	(((buf)[3]<<24)|((buf)[2]<<16)|((buf)[1]<<8)|((buf)[0]&0xff))

#define READ_U16_LE(buf) \
	(((buf)[1]<<8)|((buf)[0]&0xff))

#define READ_U32_BE(buf) \
	(((buf)[0]<<24)|((buf)[1]<<16)|((buf)[2]<<8)|((buf)[3]&0xff))

#define READ_U16_BE(buf) \
	(((buf)[0]<<8)|((buf)[1]&0xff))

/* Define the supported formats here */
input_format formats[] = {
	{wav_id, 12, wav_open, wav_close, "wav", N_("WAV file reader")},
	{aiff_id, 12, aiff_open, wav_close, "aiff", N_("AIFF/AIFC file reader")},
#ifdef HAVE_LIBFLAC
	{flac_id,     4, flac_open, flac_close, "flac", N_("FLAC file reader")},
	{oggflac_id, 32, flac_open, flac_close, "ogg", N_("Ogg FLAC file reader")},
#endif
	{NULL, 0, NULL, NULL, NULL, NULL}
};

input_format *open_audio_file(FILE *in, oe_enc_opt *opt)
{
	int j=0;
	unsigned char *buf=NULL;
	int buf_size=0, buf_filled=0;
	int size,ret;

	while(formats[j].id_func)
	{
		size = formats[j].id_data_len;
		if(size >= buf_size)
		{
			buf = realloc(buf, size);
			buf_size = size;
		}

		if(size > buf_filled)
		{
			ret = fread(buf+buf_filled, 1, buf_size-buf_filled, in);
			buf_filled += ret;

			if(buf_filled < size)
			{ /* File truncated */
				j++;
				continue;
			}
		}

		if(formats[j].id_func(buf, buf_filled))
		{
			/* ok, we now have something that can handle the file */
			if(formats[j].open_func(in, opt, buf, buf_filled)) {
                free(buf);
				return &formats[j];
            }
		}
		j++;
	}

    free(buf);

	return NULL;
}

static int seek_forward(FILE *in, int length)
{
	if(fseek(in, length, SEEK_CUR))
	{
		/* Failed. Do it the hard way. */
		unsigned char buf[1024];
		int seek_needed = length, seeked;
		while(seek_needed > 0)
		{
			seeked = fread(buf, 1, seek_needed>1024?1024:seek_needed, in);
			if(!seeked)
				return 0; /* Couldn't read more, can't read file */
			else
				seek_needed -= seeked;
		}
	}
	return 1;
}


static int find_wav_chunk(FILE *in, char *type, unsigned int *len)
{
	unsigned char buf[8];

	while(1)
	{
		if(fread(buf,1,8,in) < 8) /* Suck down a chunk specifier */
		{
			fprintf(stderr, _("Warning: Unexpected EOF in reading WAV header\n"));
			return 0; /* EOF before reaching the appropriate chunk */
		}

		if(memcmp(buf, type, 4))
		{
			*len = READ_U32_LE(buf+4);
			if(!seek_forward(in, *len))
				return 0;

			buf[4] = 0;
			fprintf(stderr, _("Skipping chunk of type \"%s\", length %d\n"), buf, *len);
		}
		else
		{
			*len = READ_U32_LE(buf+4);
			return 1;
		}
	}
}

static int find_aiff_chunk(FILE *in, char *type, unsigned int *len)
{
	unsigned char buf[8];

	while(1)
	{
		if(fread(buf,1,8,in) <8)
		{
			fprintf(stderr, _("Warning: Unexpected EOF in AIFF chunk\n"));
			return 0;
		}

		*len = READ_U32_BE(buf+4);

		if(memcmp(buf,type,4))
		{
			if((*len) & 0x1)
				(*len)++;

			if(!seek_forward(in, *len))
				return 0;
		}
		else
			return 1;
	}
}



double read_IEEE80(unsigned char *buf)
{
	int s=buf[0]&0xff;
	int e=((buf[0]&0x7f)<<8)|(buf[1]&0xff);
	double f=((unsigned long)(buf[2]&0xff)<<24)|
		((buf[3]&0xff)<<16)|
		((buf[4]&0xff)<<8) |
		 (buf[5]&0xff);

	if(e==32767)
	{
		if(buf[2]&0x80)
			return HUGE_VAL; /* Really NaN, but this won't happen in reality */
		else
		{
			if(s)
				return -HUGE_VAL;
			else
				return HUGE_VAL;
		}
	}

	f=ldexp(f,32);
	f+= ((buf[6]&0xff)<<24)|
		((buf[7]&0xff)<<16)|
		((buf[8]&0xff)<<8) |
		 (buf[9]&0xff);

	return ldexp(f, e-16446);
}

/* AIFF/AIFC support adapted from the old OggSQUISH application */
int aiff_id(unsigned char *buf, int len)
{
	if(len<12) return 0; /* Truncated file, probably */

	if(memcmp(buf, "FORM", 4))
		return 0;

	if(memcmp(buf+8, "AIF",3))
		return 0;

	if(buf[11]!='C' && buf[11]!='F')
		return 0;

	return 1;
}

int aiff_open(FILE *in, oe_enc_opt *opt, unsigned char *buf, int buflen)
{
	int aifc; /* AIFC or AIFF? */
	unsigned int len;
	unsigned char *buffer;
	unsigned char buf2[8];
	aiff_fmt format;
	aifffile *aiff = malloc(sizeof(aifffile));

	if(buf[11]=='C')
		aifc=1;
	else
		aifc=0;

	if(!find_aiff_chunk(in, "COMM", &len))
	{
		fprintf(stderr, _("Warning: No common chunk found in AIFF file\n"));
		return 0; /* EOF before COMM chunk */
	}

	if(len < 18) 
	{
		fprintf(stderr, _("Warning: Truncated common chunk in AIFF header\n"));
		return 0; /* Weird common chunk */
	}

	buffer = alloca(len);

	if(fread(buffer,1,len,in) < len)
	{
		fprintf(stderr, _("Warning: Unexpected EOF in reading AIFF header\n"));
		return 0;
	}

	format.channels = READ_U16_BE(buffer);
	format.totalframes = READ_U32_BE(buffer+2);
	format.samplesize = READ_U16_BE(buffer+6);
	format.rate = (int)read_IEEE80(buffer+8);

    aiff->bigendian = 1;

	if(aifc)
	{
		if(len < 22)
		{
			fprintf(stderr, _("Warning: AIFF-C header truncated.\n"));
			return 0;
		}

		if(!memcmp(buffer+18, "NONE", 4)) 
		{
			aiff->bigendian = 1;
		}
		else if(!memcmp(buffer+18, "sowt", 4)) 
		{
			aiff->bigendian = 0;
		}
		else
		{
			fprintf(stderr, _("Warning: Can't handle compressed AIFF-C (%c%c%c%c)\n"), *(buffer+18), *(buffer+19), *(buffer+20), *(buffer+21));
			return 0; /* Compressed. Can't handle */
		}
	}

	if(!find_aiff_chunk(in, "SSND", &len))
	{
		fprintf(stderr, _("Warning: No SSND chunk found in AIFF file\n"));
		return 0; /* No SSND chunk -> no actual audio */
	}

	if(len < 8) 
	{
		fprintf(stderr, _("Warning: Corrupted SSND chunk in AIFF header\n"));
		return 0; 
	}

	if(fread(buf2,1,8, in) < 8)
	{
		fprintf(stderr, _("Warning: Unexpected EOF reading AIFF header\n"));
		return 0;
	}

	format.offset = READ_U32_BE(buf2);
	format.blocksize = READ_U32_BE(buf2+4);

	if( format.blocksize == 0 &&
		(format.samplesize == 16 || format.samplesize == 8))
	{
		/* From here on, this is very similar to the wav code. Oh well. */
		
		opt->rate = format.rate;
		opt->channels = format.channels;
		opt->read_samples = wav_read; /* Similar enough, so we use the same */
		opt->total_samples_per_channel = format.totalframes;

		aiff->f = in;
		aiff->samplesread = 0;
		aiff->channels = format.channels;
		aiff->samplesize = format.samplesize;
		aiff->totalsamples = format.totalframes;

		opt->readdata = (void *)aiff;

		seek_forward(in, format.offset); /* Swallow some data */
		return 1;
	}
	else
	{
		fprintf(stderr, 
				_("Warning: OggEnc does not support this type of AIFF/AIFC file\n"
				" Must be 8, 16, or 24 bit PCM.\n"));
		return 0;
	}
}


int wav_id(unsigned char *buf, int len)
{
	unsigned int flen;
	
	if(len<12) return 0; /* Something screwed up */

	if(memcmp(buf, "RIFF", 4))
		return 0; /* Not wave */

	flen = READ_U32_LE(buf+4); /* We don't use this */

	if(memcmp(buf+8, "WAVE",4))
		return 0; /* RIFF, but not wave */

	return 1;
}

int wav_open(FILE *in, oe_enc_opt *opt, unsigned char *oldbuf, int buflen)
{
	unsigned char buf[16];
	unsigned int len;
	int samplesize;
	wav_fmt format;
	wavfile *wav = malloc(sizeof(wavfile));

	/* Ok. At this point, we know we have a WAV file. Now we have to detect
	 * whether we support the subtype, and we have to find the actual data
	 * We don't (for the wav reader) need to use the buffer we used to id this
	 * as a wav file (oldbuf)
	 */

	if(!find_wav_chunk(in, "fmt ", &len))
		return 0; /* EOF */

	if(len < 16) 
	{
		fprintf(stderr, _("Warning: Unrecognised format chunk in WAV header\n"));
		return 0; /* Weird format chunk */
	}

	/* A common error is to have a format chunk that is not 16 or 18 bytes
	 * in size.  This is incorrect, but not fatal, so we only warn about 
	 * it instead of refusing to work with the file.  Please, if you
	 * have a program that's creating format chunks of sizes other than
	 * 16 or 18 bytes in size, report a bug to the author.
	 */
	if(len!=16 && len!=18)
		fprintf(stderr, 
				_("Warning: INVALID format chunk in wav header.\n"
				" Trying to read anyway (may not work)...\n"));

	if(fread(buf,1,16,in) < 16)
	{
		fprintf(stderr, _("Warning: Unexpected EOF in reading WAV header\n"));
		return 0;
	}

	/* Deal with stupid broken apps. Don't use these programs.
	 */
	if(len - 16 > 0 && !seek_forward(in, len-16))
	    return 0;

	format.format =      READ_U16_LE(buf); 
	format.channels =    READ_U16_LE(buf+2); 
	format.samplerate =  READ_U32_LE(buf+4);
	format.bytespersec = READ_U32_LE(buf+8);
	format.align =       READ_U16_LE(buf+12);
	format.samplesize =  READ_U16_LE(buf+14);

	if(!find_wav_chunk(in, "data", &len))
		return 0; /* EOF */

	if(format.format == 1)
	{
		samplesize = format.samplesize/8;
		opt->read_samples = wav_read;
	}
	else if(format.format == 3)
	{
		samplesize = 4;
		opt->read_samples = wav_ieee_read;
	}
	else
	{
		fprintf(stderr, 
				_("ERROR: Wav file is unsupported type (must be standard PCM\n"
				" or type 3 floating point PCM\n"));
		return 0;
	}



	if( format.align == format.channels*samplesize &&
			format.samplesize == samplesize*8 && 
    		(format.samplesize == 24 || format.samplesize == 16 || 
             format.samplesize == 8 ||
	     (format.samplesize == 32 && format.format == 3)))
	{
		/* OK, good - we have the one supported format,
		   now we want to find the size of the file */
		opt->rate = format.samplerate;
		opt->channels = format.channels;

		wav->f = in;
		wav->samplesread = 0;
		wav->bigendian = 0;
		wav->channels = format.channels; /* This is in several places. The price
											of trying to abstract stuff. */
		wav->samplesize = format.samplesize;

		if(len)
        {
			opt->total_samples_per_channel = len/(format.channels*samplesize);
		}
		else
		{
			long pos;
			pos = ftell(in);
			if(fseek(in, 0, SEEK_END) == -1)
			{
				opt->total_samples_per_channel = 0; /* Give up */
			}
			else
			{
				opt->total_samples_per_channel = (ftell(in) - pos)/
                    (format.channels*samplesize);
				fseek(in,pos, SEEK_SET);
			}
		}
		wav->totalsamples = opt->total_samples_per_channel;

		opt->readdata = (void *)wav;
		return 1;
	}
	else
	{
		fprintf(stderr, 
				_("ERROR: Wav file is unsupported subformat (must be 8,16, or 24 bit PCM\n"
				"or floating point PCM\n"));
		return 0;
	}
}

long wav_read(void *in, float **buffer, int samples)
{
	wavfile *f = (wavfile *)in;
	int sampbyte = f->samplesize / 8;
	signed char *buf = alloca(samples*sampbyte*f->channels);
	long bytes_read = fread(buf, 1, samples*sampbyte*f->channels, f->f);
	int i,j;
	long realsamples;

	if(f->totalsamples && f->samplesread + 
			bytes_read/(sampbyte*f->channels) > f->totalsamples) {
		bytes_read = sampbyte*f->channels*(f->totalsamples - f->samplesread);
    }

	realsamples = bytes_read/(sampbyte*f->channels);
	f->samplesread += realsamples;
		
	if(f->samplesize==8)
	{
		unsigned char *bufu = (unsigned char *)buf;
		for(i = 0; i < realsamples; i++)
		{
			for(j=0; j < f->channels; j++)
			{
				buffer[j][i]=((int)(bufu[i*f->channels + j])-128)/128.0f;
			}
		}
	}
	else if(f->samplesize==16)
	{
		if(!f->bigendian)
		{
			for(i = 0; i < realsamples; i++)
			{
				for(j=0; j < f->channels; j++)
				{
					buffer[j][i] = ((buf[i*2*f->channels + 2*j + 1]<<8) |
							        (buf[i*2*f->channels + 2*j] & 0xff))/32768.0f;
				}
			}
		}
		else
		{
			for(i = 0; i < realsamples; i++)
			{
				for(j=0; j < f->channels; j++)
				{
					buffer[j][i]=((buf[i*2*f->channels + 2*j]<<8) |
							      (buf[i*2*f->channels + 2*j + 1] & 0xff))/32768.0f;
				}
			}
		}
	}
    else if(f->samplesize==24) 
    {
        if(!f->bigendian) {
            for(i = 0; i < realsamples; i++)
            {
                for(j=0; j < f->channels; j++) 
                {
                    buffer[j][i] = ((buf[i*3*f->channels + 3*j + 2] << 16) |
                      (((unsigned char *)buf)[i*3*f->channels + 3*j + 1] << 8) |
                      (((unsigned char *)buf)[i*3*f->channels + 3*j] & 0xff)) 
                        / 8388608.0f;

                }
            }
        }
        else {
            fprintf(stderr, _("Big endian 24 bit PCM data is not currently "
                              "supported, aborting.\n"));
            return 0;
        }
    }
    else {
        fprintf(stderr, _("Internal error: attempt to read unsupported "
                          "bitdepth %d\n"), f->samplesize);
        return 0;
    }

	return realsamples;
}

long wav_ieee_read(void *in, float **buffer, int samples)
{
	wavfile *f = (wavfile *)in;
	float *buf = alloca(samples*4*f->channels); /* de-interleave buffer */
	long bytes_read = fread(buf,1,samples*4*f->channels, f->f);
	int i,j;
	long realsamples;


	if(f->totalsamples && f->samplesread +
			bytes_read/(4*f->channels) > f->totalsamples)
		bytes_read = 4*f->channels*(f->totalsamples - f->samplesread);
	realsamples = bytes_read/(4*f->channels);
	f->samplesread += realsamples;

	for(i=0; i < realsamples; i++)
		for(j=0; j < f->channels; j++)
			buffer[j][i] = buf[i*f->channels + j];

	return realsamples;
}


void wav_close(void *info)
{
	wavfile *f = (wavfile *)info;

	free(f);
}

int raw_open(FILE *in, oe_enc_opt *opt)
{
	wav_fmt format; /* fake wave header ;) */
	wavfile *wav = malloc(sizeof(wavfile));

	/* construct fake wav header ;) */
	format.format =      2; 
	format.channels =    opt->channels;
	format.samplerate =  opt->rate;
	format.samplesize =  opt->samplesize;
	format.bytespersec = opt->channels * opt->rate * opt->samplesize / 8;
	format.align =       format.bytespersec;
	wav->f =             in;
	wav->samplesread =   0;
	wav->bigendian =     opt->endianness;
	wav->channels =      format.channels;
	wav->samplesize =    opt->samplesize;
    wav->totalsamples =  0;

	opt->read_samples = wav_read;
	opt->readdata = (void *)wav;
	opt->total_samples_per_channel = 0; /* raw mode, don't bother */
	return 1;
}

typedef struct {
    res_state resampler;
    audio_read_func real_reader;
    void *real_readdata;
    float **bufs;
    int channels;
    int bufsize;
    int done;
} resampler;

static long read_resampled(void *d, float **buffer, int samples)
{
    resampler *rs = d;
    long in_samples;
    int out_samples;

    in_samples = res_push_max_input(&rs->resampler, samples);
    if(in_samples > rs->bufsize)
        in_samples = rs->bufsize;

    in_samples = rs->real_reader(rs->real_readdata, rs->bufs, in_samples);

    if(in_samples <= 0) {
        if(!rs->done) {
            rs->done = 1;
            out_samples = res_drain(&rs->resampler, buffer);
            return out_samples;
        }
        return 0;
    }

    out_samples = res_push(&rs->resampler, buffer, (float const **)rs->bufs, in_samples);

    if(out_samples <= 0) {
        fprintf(stderr, _("BUG: Got zero samples from resampler: your file will be truncated. Please report this.\n"));
    }

    return out_samples;
}

int setup_resample(oe_enc_opt *opt) {
    resampler *rs = calloc(1, sizeof(resampler));
    int c;

    rs->bufsize = 4096; /* Shrug */
    rs->real_reader = opt->read_samples;
    rs->real_readdata = opt->readdata;
    rs->bufs = malloc(sizeof(float *) * opt->channels);
    rs->channels = opt->channels;
    rs->done = 0;
    if(res_init(&rs->resampler, rs->channels, opt->resamplefreq, opt->rate, RES_END))
    {
        fprintf(stderr, _("Couldn't initialise resampler\n"));
        return -1;
    }

    for(c=0; c < opt->channels; c++)
        rs->bufs[c] = malloc(sizeof(float) * rs->bufsize);

    opt->read_samples = read_resampled;
    opt->readdata = rs;
    if(opt->total_samples_per_channel)
        opt->total_samples_per_channel = (int)((float)opt->total_samples_per_channel * 
            ((float)opt->resamplefreq/(float)opt->rate));
    opt->rate = opt->resamplefreq;

    return 0;
}

void clear_resample(oe_enc_opt *opt) {
    resampler *rs = opt->readdata;
    int i;

    opt->read_samples = rs->real_reader;
    opt->readdata = rs->real_readdata;
    res_clear(&rs->resampler);

    for(i = 0; i < rs->channels; i++)
        free(rs->bufs[i]);

    free(rs->bufs);

    free(rs);
}

typedef struct {
    audio_read_func real_reader;
    void *real_readdata;
    int channels;
    float scale_factor;
} scaler;

static long read_scaler(void *data, float **buffer, int samples) {
    scaler *d = data;
    long in_samples = d->real_reader(d->real_readdata, buffer, samples);
    int i,j;

    for(i=0; i < d->channels; i++) {
        for(j=0; j < in_samples; j++) {
            buffer[i][j] *= d->scale_factor;
        }
    }

    return in_samples;
}


void setup_scaler(oe_enc_opt *opt, float scale) {
    scaler *d = calloc(1, sizeof(scaler));

    d->real_reader = opt->read_samples;
    d->real_readdata = opt->readdata;

    opt->read_samples = read_scaler;
    opt->readdata = d;
    d->channels = opt->channels;
    d->scale_factor = scale;
}

void clear_scaler(oe_enc_opt *opt) {
    scaler *d = opt->readdata;

    opt->read_samples = d->real_reader;
    opt->readdata = d->real_readdata;

    free(d);
}

typedef struct {
    audio_read_func real_reader;
    void *real_readdata;
    float **bufs;
} downmix;

static long read_downmix(void *data, float **buffer, int samples)
{
    downmix *d = data;
    long in_samples = d->real_reader(d->real_readdata, d->bufs, samples);
    int i;

    for(i=0; i < in_samples; i++) {
        buffer[0][i] = (d->bufs[0][i] + d->bufs[1][i])*0.5f;
    }

    return in_samples;
}

void setup_downmix(oe_enc_opt *opt) {
    downmix *d = calloc(1, sizeof(downmix));

    if(opt->channels != 2) {
        fprintf(stderr, "Internal error! Please report this bug.\n");
        return;
    }
    
    d->bufs = malloc(2 * sizeof(float *));
    d->bufs[0] = malloc(4096 * sizeof(float));
    d->bufs[1] = malloc(4096 * sizeof(float));
    d->real_reader = opt->read_samples;

    d->real_readdata = opt->readdata;

    opt->read_samples = read_downmix;
    opt->readdata = d;

    opt->channels = 1;
}
void clear_downmix(oe_enc_opt *opt) {
    downmix *d = opt->readdata;

    opt->read_samples = d->real_reader;
    opt->readdata = d->real_readdata;
    opt->channels = 2; /* other things in cleanup rely on this */

    free(d->bufs[0]);
    free(d->bufs[1]);
    free(d->bufs);
    free(d);
}

/* OggEnc
 **
 ** This program is distributed under the GNU General Public License, version 2.
 ** A copy of this license is included with this source.
 **
 ** Copyright 2000-2002, Michael Smith <msmith@xiph.org>
 **
 ** Portions from Vorbize, (c) Kenneth Arnold <kcarnold@yahoo.com>
 ** and libvorbis examples, (c) Monty <monty@xiph.org>
 **/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2001             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: vorbis encode-engine setup

 ********************************************************************/

#define _OV_ENC_H_

extern int vorbis_encode_init(vorbis_info *vi,
			      long channels,
			      long rate,
			      
			      long max_bitrate,
			      long nominal_bitrate,
			      long min_bitrate);

extern int vorbis_encode_setup_managed(vorbis_info *vi,
				       long channels,
				       long rate,
				       
				       long max_bitrate,
				       long nominal_bitrate,
				       long min_bitrate);
  
extern int vorbis_encode_setup_vbr(vorbis_info *vi,
				  long channels,
				  long rate,
				  
				  float /* quality level from 0. (lo) to 1. (hi) */
				  );

extern int vorbis_encode_init_vbr(vorbis_info *vi,
				  long channels,
				  long rate,
				  
				  float base_quality /* quality level from 0. (lo) to 1. (hi) */
				  );

extern int vorbis_encode_setup_init(vorbis_info *vi);

extern int vorbis_encode_ctl(vorbis_info *vi,int number,void *arg);

#define OV_ECTL_RATEMANAGE_GET       0x10

#define OV_ECTL_RATEMANAGE_SET       0x11
#define OV_ECTL_RATEMANAGE_AVG       0x12
#define OV_ECTL_RATEMANAGE_HARD      0x13

#define OV_ECTL_LOWPASS_GET          0x20
#define OV_ECTL_LOWPASS_SET          0x21

#define OV_ECTL_IBLOCK_GET           0x30
#define OV_ECTL_IBLOCK_SET           0x31

struct ovectl_ratemanage_arg {
  int    management_active;

  long   bitrate_hard_min;
  long   bitrate_hard_max;
  double bitrate_hard_window;

  long   bitrate_av_lo;
  long   bitrate_av_hi;
  double bitrate_av_window;
  double bitrate_av_window_center;
};

#define READSIZE 1024


int oe_write_page(ogg_page *page, FILE *fp);

#define SETD(toset) \
    do {\
        if(sscanf(opts[i].val, "%lf", &dval) != 1)\
            fprintf(stderr, "For option %s, couldn't read value %s as double\n",\
                    opts[i].arg, opts[i].val);\
        else\
            toset = dval;\
    } while(0)

#define SETL(toset) \
    do {\
        if(sscanf(opts[i].val, "%ld", &lval) != 1)\
            fprintf(stderr, "For option %s, couldn't read value %s as integer\n",\
                    opts[i].arg, opts[i].val);\
        else\
            toset = lval;\
    } while(0)

static void set_advanced_encoder_options(adv_opt *opts, int count, 
        vorbis_info *vi)
{
    int hard = 0;
    int avg = 0;
    struct ovectl_ratemanage_arg ai;
    int i;
    double dval;
    long lval;

    vorbis_encode_ctl(vi, OV_ECTL_RATEMANAGE_GET, &ai);

    for(i=0; i < count; i++) {
        fprintf(stderr, _("Setting advanced encoder option \"%s\" to %s\n"),
                opts[i].arg, opts[i].val);

        if(!strcmp(opts[i].arg, "bitrate_average_window")) {
            SETD(ai.bitrate_av_window);
            avg = 1;
        }
        else if(!strcmp(opts[i].arg, "bitrate_average_window_center")) {
            SETD(ai.bitrate_av_window_center);
            avg = 1;
        }
        else if(!strcmp(opts[i].arg, "bitrate_average_low")) {
            SETL(ai.bitrate_av_lo);
            avg = 1;
        }
        else if(!strcmp(opts[i].arg, "bitrate_average_high")) {
            SETL(ai.bitrate_av_hi);
            avg = 1;
        }
        else if(!strcmp(opts[i].arg, "bitrate_hard_min")) {
            SETL(ai.bitrate_hard_min);
            hard = 1;
        }
        else if(!strcmp(opts[i].arg, "bitrate_hard_max")) {
            SETL(ai.bitrate_hard_max);
            hard = 1;
        }
        else if(!strcmp(opts[i].arg, "bitrate_hard_window")) {
            SETD(ai.bitrate_hard_window);
            hard = 1;
        }
        else if(!strcmp(opts[i].arg, "impulse_noisetune")) {
            double val;
            SETD(val);
            vorbis_encode_ctl(vi, OV_ECTL_IBLOCK_SET, &val);
        }
        else if(!strcmp(opts[i].arg, "lowpass_frequency")) {
            double prev, new;
            SETD(new);
            vorbis_encode_ctl(vi, OV_ECTL_LOWPASS_GET, &prev);
            vorbis_encode_ctl(vi, OV_ECTL_LOWPASS_SET, &new);
            fprintf(stderr, _("Changed lowpass frequency from %f kHz to %f kHz\n"), prev, new);
        }
        else {
            fprintf(stderr, _("Unrecognised advanced option \"%s\"\n"), 
                    opts[i].arg);
        }
    }

    if(hard)
        vorbis_encode_ctl(vi, OV_ECTL_RATEMANAGE_HARD, &ai);
    if(avg)
        vorbis_encode_ctl(vi, OV_ECTL_RATEMANAGE_AVG, &ai);
}

int oe_encode(oe_enc_opt *opt)
{

	ogg_stream_state os;
	ogg_page 		 og;
	ogg_packet 		 op;

	vorbis_dsp_state vd;
	vorbis_block     vb;
	vorbis_info      vi;

	long samplesdone=0;
    int eos;
	long bytes_written = 0, packetsdone=0;
	double time_elapsed;
	int ret=0;
	TIMER *timer;

    if(opt->channels > 255) {
        fprintf(stderr, _("255 channels should be enough for anyone. (Sorry, vorbis doesn't support more)\n"));
        return 1;
    }

    /* get start time. */
    timer = timer_start();

    if(!opt->managed && (opt->min_bitrate>=0 || opt->max_bitrate>=0)){
      fprintf(stderr, _("Requesting a minimum or maximum bitrate requires --managed\n"));
      return 1;
    }

    /* if we had no quality or bitrate spec at all from the user, use
       the default quality with no management --Monty 20020711 */
    if(opt->bitrate < 0 && opt->min_bitrate < 0 && opt->max_bitrate < 0){
      opt->quality_set=1;
    }

    opt->start_encode(opt->infilename, opt->filename, opt->bitrate, opt->quality, 
		      opt->quality_set, opt->managed, opt->min_bitrate, opt->max_bitrate);
    
    /* Have vorbisenc choose a mode for us */
    vorbis_info_init(&vi);
    
    if(opt->quality_set > 0){
        if(vorbis_encode_setup_vbr(&vi, opt->channels, opt->rate, opt->quality)){
	        fprintf(stderr, _("Mode initialisation failed: invalid parameters for quality\n"));
	        vorbis_info_clear(&vi);
	        return 1;
        }

        /* do we have optional hard quality restrictions? */
        if(opt->max_bitrate > 0 || opt->min_bitrate > 0){
            struct ovectl_ratemanage_arg ai;
	        vorbis_encode_ctl(&vi, OV_ECTL_RATEMANAGE_GET, &ai);
	
	        ai.bitrate_hard_min=opt->min_bitrate;
	        ai.bitrate_hard_max=opt->max_bitrate;
	        ai.management_active=1;

	        vorbis_encode_ctl(&vi, OV_ECTL_RATEMANAGE_SET, &ai);

        }


    }else {
        if(vorbis_encode_setup_managed(&vi, opt->channels, opt->rate, 
				     opt->max_bitrate>0?opt->max_bitrate*1000:-1,
				     opt->bitrate*1000, 
				     opt->min_bitrate>0?opt->min_bitrate*1000:-1)){
	        fprintf(stderr, _("Mode initialisation failed: invalid parameters for bitrate\n"));
	        vorbis_info_clear(&vi);
	        return 1;
        }
    }
    
    if(opt->managed && opt->bitrate < 0)
    {
        vorbis_encode_ctl(&vi, OV_ECTL_RATEMANAGE_AVG, NULL);
    }
    else if(!opt->managed)
    {
        /* Turn off management entirely (if it was turned on). */
        vorbis_encode_ctl(&vi, OV_ECTL_RATEMANAGE_SET, NULL);
    }
    
    set_advanced_encoder_options(opt->advopt, opt->advopt_count, &vi);
    
    vorbis_encode_setup_init(&vi);


	/* Now, set up the analysis engine, stream encoder, and other
	   preparation before the encoding begins.
	 */

	vorbis_analysis_init(&vd,&vi);
	vorbis_block_init(&vd,&vb);

	ogg_stream_init(&os, opt->serialno);

	/* Now, build the three header packets and send through to the stream 
	   output stage (but defer actual file output until the main encode loop) */

	{
		ogg_packet header_main;
		ogg_packet header_comments;
		ogg_packet header_codebooks;
		int result;

		/* Build the packets */
		vorbis_analysis_headerout(&vd,opt->comments,
				&header_main,&header_comments,&header_codebooks);

		/* And stream them out */
		ogg_stream_packetin(&os,&header_main);
		ogg_stream_packetin(&os,&header_comments);
		ogg_stream_packetin(&os,&header_codebooks);

		while((result = ogg_stream_flush(&os, &og)))
		{
			if(!result) break;
			ret = oe_write_page(&og, opt->out);
			if(ret != og.header_len + og.body_len)
			{
				opt->error(_("Failed writing header to output stream\n"));
				ret = 1;
				goto cleanup; /* Bail and try to clean up stuff */
			}
		}
	}

	eos = 0;

	/* Main encode loop - continue until end of file */
	while(!eos)
	{
		float **buffer = vorbis_analysis_buffer(&vd, READSIZE);
		long samples_read = opt->read_samples(opt->readdata, 
				buffer, READSIZE);

		if(samples_read ==0)
			/* Tell the library that we wrote 0 bytes - signalling the end */
			vorbis_analysis_wrote(&vd,0);
		else
		{
			samplesdone += samples_read;

			/* Call progress update every 40 pages */
			if(packetsdone>=40)
			{
				double time;

				packetsdone = 0;
				time = timer_time(timer);

				opt->progress_update(opt->filename, opt->total_samples_per_channel, 
						samplesdone, time);
			}

			/* Tell the library how many samples (per channel) we wrote 
			   into the supplied buffer */
			vorbis_analysis_wrote(&vd, samples_read);
		}

		/* While we can get enough data from the library to analyse, one
		   block at a time... */
		while(vorbis_analysis_blockout(&vd,&vb)==1)
		{

			/* Do the main analysis, creating a packet */
			vorbis_analysis(&vb, NULL);
			vorbis_bitrate_addblock(&vb);

			while(vorbis_bitrate_flushpacket(&vd, &op)) 
			{
				/* Add packet to bitstream */
				ogg_stream_packetin(&os,&op);
				packetsdone++;

				/* If we've gone over a page boundary, we can do actual output,
				   so do so (for however many pages are available) */

				while(!eos)
				{
					int result = ogg_stream_pageout(&os,&og);
					if(!result) break;

					ret = oe_write_page(&og, opt->out);
					if(ret != og.header_len + og.body_len)
					{
						opt->error(_("Failed writing data to output stream\n"));
						ret = 1;
						goto cleanup; /* Bail */
					}
					else
						bytes_written += ret; 
	
					if(ogg_page_eos(&og))
						eos = 1;
				}
			}
		}
	}

	ret = 0; /* Success, set return value to 0 since other things reuse it
			  * for nefarious purposes. */

	/* Cleanup time */
cleanup:

	ogg_stream_clear(&os);

	vorbis_block_clear(&vb);
	vorbis_dsp_clear(&vd);
	vorbis_info_clear(&vi);

	time_elapsed = timer_time(timer);
	opt->end_encode(opt->filename, time_elapsed, opt->rate, samplesdone, bytes_written);

	timer_clear(timer);

	return ret;
}

void update_statistics_full(char *fn, long total, long done, double time)
{
	static char *spinner="|/-\\";
	static int spinpoint = 0;
	double remain_time;
	int minutes=0,seconds=0;
	
	remain_time = time/((double)done/(double)total) - time;
	minutes = ((int)remain_time)/60;
	seconds = (int)(remain_time - (double)((int)remain_time/60)*60);

	fprintf(stderr, "\r");
	fprintf(stderr, _("\t[%5.1f%%] [%2dm%.2ds remaining] %c"), 
			done*100.0/total, minutes, seconds, spinner[spinpoint++%4]);
}

void update_statistics_notime(char *fn, long total, long done, double time)
{
	static char *spinner="|/-\\";
	static int spinpoint =0;
	
	fprintf(stderr, "\r");
	fprintf(stderr, _("\tEncoding [%2dm%.2ds so far] %c"), 
            ((int)time)/60, (int)(time - (double)((int)time/60)*60),
			spinner[spinpoint++%4]);
}

int oe_write_page(ogg_page *page, FILE *fp)
{
	int written;
	written = fwrite(page->header,1,page->header_len, fp);
	written += fwrite(page->body,1,page->body_len, fp);

	return written;
}

void final_statistics(char *fn, double time, int rate, long samples, long bytes)
{
	double speed_ratio;
	if(fn)
		fprintf(stderr, _("\n\nDone encoding file \"%s\"\n"), fn);
	else
		fprintf(stderr, _("\n\nDone encoding.\n"));

	speed_ratio = (double)samples / (double)rate / time;
	
	fprintf(stderr, _("\n\tFile length:  %dm %04.1fs\n"),
			(int)(samples/rate/60),
			samples/rate - 
			floor(samples/rate/60)*60);
	fprintf(stderr, _("\tElapsed time: %dm %04.1fs\n"),
			(int)(time/60),
			time - floor(time/60)*60);
	fprintf(stderr, _("\tRate:         %.4f\n"), speed_ratio);
	fprintf(stderr, _("\tAverage bitrate: %.1f kb/s\n\n"), 
		8./1000.*((double)bytes/((double)samples/(double)rate)));
}

void final_statistics_null(char *fn, double time, int rate, long samples, 
		long bytes)
{
	/* Don't do anything, this is just a placeholder function for quiet mode */
}

void update_statistics_null(char *fn, long total, long done, double time)
{
	/* So is this */
}

void encode_error(char *errmsg)
{
	fprintf(stderr, "\n%s\n", errmsg);
}

static void print_brconstraints(int min, int max)
{
    if(min > 0 && max > 0)
        fprintf(stderr, "(min %d kbps, max %d kbps)", min,max);
    else if(min > 0)
        fprintf(stderr, "(min %d kbps, no max)", min);
    else if(max > 0)
        fprintf(stderr, "(no min, max %d kbps)", max);
    else
        fprintf(stderr, "(no min or max)");
}

void start_encode_full(char *fn, char *outfn, int bitrate, float quality, int qset,
        int managed, int min, int max)
{
  if(bitrate>0){
    if(managed>0){
      fprintf(stderr, _("Encoding %s%s%s to \n         "
			"%s%s%s \nat average bitrate %d kbps "),
	      fn?"\"":"", fn?fn:_("standard input"), fn?"\"":"",
	      outfn?"\"":"", outfn?outfn:_("standard output"), outfn?"\"":"",
	      bitrate);
      print_brconstraints(min,max);
      fprintf(stderr, ", \nusing full bitrate management engine\n");
    } else {
      fprintf(stderr, _("Encoding %s%s%s to \n         %s%s%s \nat approximate bitrate %d kbps (VBR encoding enabled)\n"),
	      fn?"\"":"", fn?fn:_("standard input"), fn?"\"":"",
	    outfn?"\"":"", outfn?outfn:_("standard output"), outfn?"\"":"",
	      bitrate);
    }
  }else{
    if(qset>0){
      if(managed>0){
	fprintf(stderr, _("Encoding %s%s%s to \n         %s%s%s \nat quality level %2.2f using constrained VBR "),
		fn?"\"":"", fn?fn:_("standard input"), fn?"\"":"",
		outfn?"\"":"", outfn?outfn:_("standard output"), outfn?"\"":"",
		quality * 10);
	print_brconstraints(min,max);
	fprintf(stderr, "\n");
      }else{
        fprintf(stderr, _("Encoding %s%s%s to \n         %s%s%s \nat quality %2.2f\n"),
			    fn?"\"":"", fn?fn:_("standard input"), fn?"\"":"",
                outfn?"\"":"", outfn?outfn:_("standard output"), outfn?"\"":"",
                quality * 10);
      }
    }else{
      fprintf(stderr, _("Encoding %s%s%s to \n         %s%s%s \nusing bitrate management "),
	      fn?"\"":"", fn?fn:_("standard input"), fn?"\"":"",
	      outfn?"\"":"", outfn?outfn:_("standard output"), outfn?"\"":"");
      print_brconstraints(min,max);
      fprintf(stderr, "\n");
    }
  }
}

void start_encode_null(char *fn, char *outfn, int bitrate, float quality, int qset,
        int managed, int min, int max)
{
}


/* OggEnc
 **
 ** This program is distributed under the GNU General Public License, version 2.
 ** A copy of this license is included with this source.
 **
 ** Copyright 2000, Michael Smith <msmith@xiph.org>
 **
 ** Portions from Vorbize, (c) Kenneth Arnold <kcarnold@yahoo.com>
 ** and libvorbis examples, (c) Monty <monty@xiph.org>
 **/

/* Platform support routines  - win32, OS/2, unix */


#if defined(_WIN32) && defined(_MSC_VER)

void setbinmode(FILE *f)
{
	_setmode( _fileno(f), _O_BINARY );
}
#endif /* win32 */

#ifdef __EMX__
void setbinmode(FILE *f) 
{
	        _fsetmode( f, "b");
}
#endif

#if defined(__WATCOMC__) || defined(__BORLANDC__)
void setbinmode(FILE *f)
{
	setmode(fileno(f), O_BINARY);
}
#endif


#if defined(_WIN32) || defined(__EMX__) || defined(__WATCOMC__)
void *timer_start(void)
{
	time_t *start = malloc(sizeof(time_t));
	time(start);
	return (void *)start;
}

double timer_time(void *timer)
{
	time_t now = time(NULL);
	time_t start = *((time_t *)timer);

    if(now-start)
    	return (double)(now-start);
    else
        return 1; /* To avoid division by zero later, for very short inputs */
}


void timer_clear(void *timer)
{
	free((time_t *)timer);
}

#else /* unix. Or at least win32 */

void *timer_start(void)
{
	struct timeval *start = malloc(sizeof(struct timeval));
	gettimeofday(start, NULL);
	return (void *)start;
}

double timer_time(void *timer)
{
	struct timeval now;
	struct timeval start = *((struct timeval *)timer);

	gettimeofday(&now, NULL);

	return (double)now.tv_sec - (double)start.tv_sec + 
		((double)now.tv_usec - (double)start.tv_usec)/1000000.0;
}

void timer_clear(void *timer)
{
	free((time_t *)timer);
}

#endif

#ifdef _WIN32

#define PATH_SEPS "/\\"
#define mkdir(x,y) _mkdir((x))

/* MSVC does this, borland doesn't? */
#ifndef __BORLANDC__
#define stat _stat
#endif

#else

#define PATH_SEPS "/"

#endif

int create_directories(char *fn)
{
    char *end, *start;
    struct stat statbuf;
    char *segment = malloc(strlen(fn)+1);

    start = fn;
#ifdef _WIN32
    if(strlen(fn) >= 3 && isalpha(fn[0]) && fn[1]==':')
        start = start+2;
#endif

    while((end = strpbrk(start+1, PATH_SEPS)) != NULL)
    {
        memcpy(segment, fn, end-fn);
        segment[end-fn] = 0;

        if(stat(segment,&statbuf)) {
            if(errno == ENOENT) {
                if(mkdir(segment, 0777)) {
                    fprintf(stderr, _("Couldn't create directory \"%s\": %s\n"),
                            segment, strerror(errno));
                    free(segment);
                    return -1;
                }
            }
            else {
                fprintf(stderr, _("Error checking for existence of directory %s: %s\n"), 
                            segment, strerror(errno));
                free(segment);
                return -1;
            }
        }
#if defined(_WIN32) && !defined(__BORLANDC__)
        else if(!(_S_IFDIR & statbuf.st_mode)) {
#elif defined(__BORLANDC__)
        else if(!(S_IFDIR & statbuf.st_mode)) {
#else
        else if(!S_ISDIR(statbuf.st_mode)) {
#endif
            fprintf(stderr, _("Error: path segment \"%s\" is not a directory\n"),
                    segment);
            free(segment);
            return -1;
        }

        start = end+1;
    }

    free(segment);
    return 0;

}



/* resample.c: see resample.h for interesting stuff */


/* Some systems don't define this */
#ifndef M_PI
#define M_PI       3.14159265358979323846 
#endif

static int hcf(int arg1, int arg2)
{
	int mult = 1;

	while (~(arg1 | arg2) & 1)
		arg1 >>= 1, arg2 >>= 1, mult <<= 1;

	while (arg1 > 0)
	{
		if (~(arg1 & arg2) & 1)
		{
			arg1 >>= (~arg1 & 1);
			arg2 >>= (~arg2 & 1);
		}
		else if (arg1 < arg2)
			arg2 = (arg2 - arg1) >> 1;
		else
			arg1 = (arg1 - arg2) >> 1;
	}

	return arg2 * mult;
}


static void filt_sinc(float *dest, int N, int step, double fc, double gain, int width)
{
	double s = fc / step;
	int mid, x;
	float *endpoint = dest + N,
		*base = dest,
		*origdest = dest;
	
	assert(width <= N);

	if ((N & 1) == 0)
	{
		*dest = 0.0;
		dest += width;
		if (dest >= endpoint)
			dest = ++base;
		N--;
	}

	mid = N / 2;
	x = -mid;

	while (N--)
	{
		*dest = (x ? sin(x * M_PI * s) / (x * M_PI) * step : fc) * gain;
		x++;
		dest += width;
		if (dest >= endpoint)
			dest = ++base;
	}
	assert(dest == origdest + width);
}


static double I_zero(double x)
{
	int n = 0;
	double u = 1.0,
		s = 1.0,
		t;

	do
	{
		n += 2;
		t = x / n;
		u *= t * t;
		s += u;
	} while (u > 1e-21 * s);

	return s;
}


static void win_kaiser(float *dest, int N, double alpha, int width)
{
	double I_alpha, midsq;
	int x;
	float *endpoint = dest + N,
		*base = dest,
		*origdest = dest;

	assert(width <= N);

	if ((N & 1) == 0)
	{
		*dest = 0.0;
		dest += width;
		if (dest >= endpoint)
			dest = ++base;
		N--;
	}

	x = -(N / 2);
	midsq = (double)(x - 1) * (double)(x - 1);
	I_alpha = I_zero(alpha);

	while (N--)
	{
		*dest *= I_zero(alpha * sqrt(1.0 - ((double)x * (double)x) / midsq)) / I_alpha;
		x++;
		dest += width;
		if (dest >= endpoint)
			dest = ++base;
	}
	assert(dest == origdest + width);
}


int res_init(res_state *state, int channels, int outfreq, int infreq, res_parameter op1, ...)
{
	double beta = 16.0,
		cutoff = 0.80,
		gain = 1.0;
	int taps = 45;

	int factor;

	assert(state);
	assert(channels > 0);
	assert(outfreq > 0);
	assert(infreq > 0);
	assert(taps > 0);

	if (state == NULL || channels <= 0 || outfreq <= 0 || infreq <= 0 || taps <= 0)
		return -1;

	if (op1 != RES_END)
	{
		va_list argp;
		va_start(argp, op1);
		do
		{
			switch (op1)
			{
			case RES_GAIN:
				gain = va_arg(argp, double);
				break;

			case RES_CUTOFF:
				cutoff = va_arg(argp, double);
				assert(cutoff > 0.01 && cutoff <= 1.0);
				break;

			case RES_TAPS:
				taps = va_arg(argp, int);
				assert(taps > 2 && taps < 1000);
				break;
				
			case RES_BETA:
				beta = va_arg(argp, double);
				assert(beta > 2.0);
				break;
			default:
				assert("arglist" == "valid");
				return -1;
			}
			op1 = va_arg(argp, res_parameter);
		} while (op1 != RES_END);
		va_end(argp);
	}

	factor = hcf(infreq, outfreq);
	outfreq /= factor;
	infreq /= factor;

	/* adjust to rational values for downsampling */
	if (outfreq < infreq)
	{
		/* push the cutoff frequency down to the output frequency */
		cutoff = cutoff * outfreq / infreq; 

        /* compensate for the sharper roll-off requirement
         * by using a bigger hammer */
        taps = taps * infreq/outfreq;
	}

	assert(taps >= (infreq + outfreq - 1) / outfreq);

	if ((state->table = calloc(outfreq * taps, sizeof(float))) == NULL)
		return -1;
	if ((state->pool = calloc(channels * taps, sizeof(SAMPLE))) == NULL)
	{
		free(state->table);
		state->table = NULL;
		return -1;
	}

	state->poolfill = taps / 2 + 1;
	state->channels = channels;
	state->outfreq = outfreq;
	state->infreq = infreq;
	state->taps = taps;
	state->offset = 0;

	filt_sinc(state->table, outfreq * taps, outfreq, cutoff, gain, taps);
	win_kaiser(state->table, outfreq * taps, beta, taps);

	return 0;
}


static SAMPLE sum(float const *scale, int count, SAMPLE const *source, SAMPLE const *trigger, SAMPLE const *reset, int srcstep)
{
	float total = 0.0;

	while (count--)
	{
		total += *source * *scale;

		if (source == trigger)
			source = reset, srcstep = 1;
		source -= srcstep;
		scale++;
	}

	return total;
}


static int push(res_state const * const state, SAMPLE *pool, int * const poolfill, int * const offset, SAMPLE *dest, int dststep, SAMPLE const *source, int srcstep, size_t srclen)
{
	SAMPLE	* const destbase = dest,
		*poolhead = pool + *poolfill,
		*poolend = pool + state->taps,
		*newpool = pool;
	SAMPLE const *refill, *base, *endpoint;
	int	lencheck;


	assert(state);
	assert(pool);
	assert(poolfill);
	assert(dest);
	assert(source);

	assert(state->poolfill != -1);
	
	lencheck = res_push_check(state, srclen);

	/* fill the pool before diving in */
	while (poolhead < poolend && srclen > 0)
	{
		*poolhead++ = *source;
		source += srcstep;
		srclen--;
	}

	if (srclen <= 0)
		return 0;

	base = source;
	endpoint = source + srclen * srcstep;

	while (source < endpoint)
	{
		*dest = sum(state->table + *offset * state->taps, state->taps, source, base, poolend, srcstep);
		dest += dststep;
		*offset += state->infreq;
		while (*offset >= state->outfreq)
		{
			*offset -= state->outfreq;
			source += srcstep;
		}
	}

	assert(dest == destbase + lencheck * dststep);

	/* pretend that source has that underrun data we're not going to get */
	srclen += (source - endpoint) / srcstep;

	/* if we didn't get enough to completely replace the pool, then shift things about a bit */
	if (srclen < state->taps)
	{
		refill = pool + srclen;
		while (refill < poolend)
			*newpool++ = *refill++;

		refill = source - srclen * srcstep;
	}
	else
		refill = source - state->taps * srcstep;

	/* pull in fresh pool data */
	while (refill < endpoint)
	{
		*newpool++ = *refill;
		refill += srcstep;
	}

	assert(newpool > pool);
	assert(newpool <= poolend);

	*poolfill = newpool - pool;

	return (dest - destbase) / dststep;
}


int res_push_max_input(res_state const * const state, size_t maxoutput)
{
	return maxoutput * state->infreq / state->outfreq;
}


int res_push_check(res_state const * const state, size_t srclen)
{
	if (state->poolfill < state->taps)
		srclen -= state->taps - state->poolfill;

	return (srclen * state->outfreq - state->offset + state->infreq - 1) / state->infreq;
}


int res_push(res_state *state, SAMPLE **dstlist, SAMPLE const **srclist, size_t srclen)
{
	int result = -1, poolfill = -1, offset = -1, i;

	assert(state);
	assert(dstlist);
	assert(srclist);
	assert(state->poolfill >= 0);

	for (i = 0; i < state->channels; i++)
	{
		poolfill = state->poolfill;
		offset = state->offset;
		result = push(state, state->pool + i * state->taps, &poolfill, &offset, dstlist[i], 1, srclist[i], 1, srclen);
	}
	state->poolfill = poolfill;
	state->offset = offset;

	return result;
}


int res_push_interleaved(res_state *state, SAMPLE *dest, SAMPLE const *source, size_t srclen)
{
	int result = -1, poolfill = -1, offset = -1, i;
	
	assert(state);
	assert(dest);
	assert(source);
	assert(state->poolfill >= 0);

	for (i = 0; i < state->channels; i++)
	{
		poolfill = state->poolfill;
		offset = state->offset;
		result = push(state, state->pool + i * state->taps, &poolfill, &offset, dest + i, state->channels, source + i, state->channels, srclen);
	}
	state->poolfill = poolfill;
	state->offset = offset;

	return result;
}


int res_drain(res_state *state, SAMPLE **dstlist)
{
	SAMPLE *tail;
	int result = -1, poolfill = -1, offset = -1, i;

	assert(state);
	assert(dstlist);
	assert(state->poolfill >= 0);

	if ((tail = calloc(state->taps, sizeof(SAMPLE))) == NULL)
		return -1;

	for (i = 0; i < state->channels; i++)
	{
		poolfill = state->poolfill;
		offset = state->offset;
		result = push(state, state->pool + i * state->taps, &poolfill, &offset, dstlist[i], 1, tail, 1, state->taps / 2 - 1);
	}
		
	free(tail);

	state->poolfill = -1;

	return result;
}


int res_drain_interleaved(res_state *state, SAMPLE *dest)
{
	SAMPLE *tail;
	int result = -1, poolfill = -1, offset = -1, i;

	assert(state);
	assert(dest);
	assert(state->poolfill >= 0);

	if ((tail = calloc(state->taps, sizeof(SAMPLE))) == NULL)
		return -1;

	for (i = 0; i < state->channels; i++)
	{
		poolfill = state->poolfill;
		offset = state->offset;
		result = push(state, state->pool + i * state->taps, &poolfill, &offset, dest + i, state->channels, tail, 1, state->taps / 2 - 1);
	}
		
	free(tail);

	state->poolfill = -1;

	return result;
}


void res_clear(res_state *state)
{
	assert(state);
	assert(state->table);
	assert(state->pool);

	free(state->table);
	free(state->pool);
	memset(state, 0, sizeof(*state));
}

/* Getopt for GNU.
   NOTE: getopt is now part of the C library, so if you don't know what
   "Keep this file name-space clean" means, talk to drepper@gnu.org
   before changing it!

   Copyright (C) 1987, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99
   	Free Software Foundation, Inc.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


/* Comment out all this code if we are using the GNU C Library, and are not
   actually compiling the library itself.  This code is part of the GNU C
   Library, but also included in many other GNU distributions.  Compiling
   and linking in this code is a waste when using the GNU C library
   (especially if it is a shared library).  Rather than having every GNU
   program understand `configure --with-gnu-libc' and omit the object files,
   it is simpler to just do this in the source for each such file.  */

/* This version of `getopt' appears to the caller like standard Unix `getopt'
   but it behaves differently for the user, since it allows the user
   to intersperse the options with the other arguments.

   As `getopt' works, it permutes the elements of ARGV so that,
   when it is done, all the options precede everything else.  Thus
   all application programs are extended to handle flexible argument order.

   Setting the environment variable POSIXLY_CORRECT disables permutation.
   Then the behavior is completely standard.

   GNU application programs can use a third alternative mode in which
   they can distinguish the relative order of options and other arguments.  */

/* Names for the values of the `has_arg' field of `struct option'.  */

/* For communication from `getopt' to the caller.
   When `getopt' finds an option that takes an argument,
   the argument value is returned here.
   Also, when `ordering' is RETURN_IN_ORDER,
   each non-option ARGV-element is returned here.  */

char *optarg;

/* Index in ARGV of the next element to be scanned.
   This is used for communication to and from the caller
   and for communication between successive calls to `getopt'.

   On entry to `getopt', zero means this is the first call; initialize.

   When `getopt' returns -1, this is the index of the first of the
   non-option elements that the caller should itself scan.

   Otherwise, `optind' communicates from one call to the next
   how much of ARGV has been scanned so far.  */

/* 1003.2 says this must be 1 before any call.  */
int optind = 1;

/* Formerly, initialization of getopt depended on optind==0, which
   causes problems with re-calling getopt as programs generally don't
   know that. */

int __getopt_initialized;

/* The next char to be scanned in the option-element
   in which the last option character we returned was found.
   This allows us to pick up the scan where we left off.

   If this is zero, or a null string, it means resume the scan
   by advancing to the next ARGV-element.  */

static char *nextchar;

/* Callers store zero here to inhibit the error message
   for unrecognized options.  */

int opterr = 1;

/* Set to an option character which was unrecognized.
   This must be initialized on some systems to avoid linking in the
   system's own getopt implementation.  */

int optopt = '?';

/* Describe how to deal with options that follow non-option ARGV-elements.

   If the caller did not specify anything,
   the default is REQUIRE_ORDER if the environment variable
   POSIXLY_CORRECT is defined, PERMUTE otherwise.

   REQUIRE_ORDER means don't recognize them as options;
   stop option processing when the first non-option is seen.
   This is what Unix does.
   This mode of operation is selected by either setting the environment
   variable POSIXLY_CORRECT, or using `+' as the first character
   of the list of option characters.

   PERMUTE is the default.  We permute the contents of ARGV as we scan,
   so that eventually all the non-options are at the end.  This allows options
   to be given in any order, even with programs that were not written to
   expect this.

   RETURN_IN_ORDER is an option available to programs that were written
   to expect options and other ARGV-elements in any order and that care about
   the ordering of the two.  We describe each non-option ARGV-element
   as if it were the argument of an option with character code 1.
   Using `-' as the first character of the list of option characters
   selects this mode of operation.

   The special argument `--' forces an end of option-scanning regardless
   of the value of `ordering'.  In the case of RETURN_IN_ORDER, only
   `--' can cause `getopt' to return -1 with `optind' != ARGC.  */

static enum
{
  REQUIRE_ORDER, PERMUTE, RETURN_IN_ORDER
} ordering;

/* Value of POSIXLY_CORRECT environment variable.  */
static char *posixly_correct;

#ifdef	__GNU_LIBRARY__
/* We want to avoid inclusion of string.h with non-GNU libraries
   because there are many ways it can cause trouble.
   On some systems, it contains special magic macros that don't work
   in GCC.  */
# define my_index	strchr
#else

/* Avoid depending on library functions or files
   whose names are inconsistent.  */

#ifndef getenv
extern char *getenv ();
#endif

static char *
my_index (str, chr)
     const char *str;
     int chr;
{
  while (*str)
    {
      if (*str == chr)
	return (char *) str;
      str++;
    }
  return 0;
}

/* If using GCC, we can safely declare strlen this way.
   If not using GCC, it is ok not to declare it.  */
#ifdef __GNUC__
/* Note that Motorola Delta 68k R3V7 comes with GCC but not stddef.h.
   That was relevant to code that was here before.  */
# if (!defined __STDC__ || !__STDC__) && !defined strlen
/* gcc with -traditional declares the built-in strlen to return int,
   and has done so at least since version 2.4.5. -- rms.  */
extern int strlen (const char *);
# endif /* not __STDC__ */
#endif /* __GNUC__ */

#endif /* not __GNU_LIBRARY__ */

/* Handle permutation of arguments.  */

/* Describe the part of ARGV that contains non-options that have
   been skipped.  `first_nonopt' is the index in ARGV of the first of them;
   `last_nonopt' is the index after the last of them.  */

static int first_nonopt;
static int last_nonopt;

#ifdef _LIBC
/* Bash 2.0 gives us an environment variable containing flags
   indicating ARGV elements that should not be considered arguments.  */

/* Defined in getopt_init.c  */
extern char *__getopt_nonoption_flags;

static int nonoption_flags_max_len;
static int nonoption_flags_len;

static int original_argc;
static char *const *original_argv;

/* Make sure the environment variable bash 2.0 puts in the environment
   is valid for the getopt call we must make sure that the ARGV passed
   to getopt is that one passed to the process.  */
static void
__attribute__ ((unused))
store_args_and_env (int argc, char *const *argv)
{
  /* XXX This is no good solution.  We should rather copy the args so
     that we can compare them later.  But we must not use malloc(3).  */
  original_argc = argc;
  original_argv = argv;
}
# ifdef text_set_element
text_set_element (__libc_subinit, store_args_and_env);
# endif /* text_set_element */

# define SWAP_FLAGS(ch1, ch2) \
  if (nonoption_flags_len > 0)						      \
    {									      \
      char __tmp = __getopt_nonoption_flags[ch1];			      \
      __getopt_nonoption_flags[ch1] = __getopt_nonoption_flags[ch2];	      \
      __getopt_nonoption_flags[ch2] = __tmp;				      \
    }
#else	/* !_LIBC */
# define SWAP_FLAGS(ch1, ch2)
#endif	/* _LIBC */

/* Exchange two adjacent subsequences of ARGV.
   One subsequence is elements [first_nonopt,last_nonopt)
   which contains all the non-options that have been skipped so far.
   The other is elements [last_nonopt,optind), which contains all
   the options processed since those non-options were skipped.

   `first_nonopt' and `last_nonopt' are relocated so that they describe
   the new indices of the non-options in ARGV after they are moved.  */

#if defined __STDC__ && __STDC__
static void exchange (char **);
#endif

static void
exchange (argv)
     char **argv;
{
  int bottom = first_nonopt;
  int middle = last_nonopt;
  int top = optind;
  char *tem;

  /* Exchange the shorter segment with the far end of the longer segment.
     That puts the shorter segment into the right place.
     It leaves the longer segment in the right place overall,
     but it consists of two parts that need to be swapped next.  */

#ifdef _LIBC
  /* First make sure the handling of the `__getopt_nonoption_flags'
     string can work normally.  Our top argument must be in the range
     of the string.  */
  if (nonoption_flags_len > 0 && top >= nonoption_flags_max_len)
    {
      /* We must extend the array.  The user plays games with us and
	 presents new arguments.  */
      char *new_str = malloc (top + 1);
      if (new_str == NULL)
	nonoption_flags_len = nonoption_flags_max_len = 0;
      else
	{
	  memset (__mempcpy (new_str, __getopt_nonoption_flags,
			     nonoption_flags_max_len),
		  '\0', top + 1 - nonoption_flags_max_len);
	  nonoption_flags_max_len = top + 1;
	  __getopt_nonoption_flags = new_str;
	}
    }
#endif

  while (top > middle && middle > bottom)
    {
      if (top - middle > middle - bottom)
	{
	  /* Bottom segment is the short one.  */
	  int len = middle - bottom;
	  register int i;

	  /* Swap it with the top part of the top segment.  */
	  for (i = 0; i < len; i++)
	    {
	      tem = argv[bottom + i];
	      argv[bottom + i] = argv[top - (middle - bottom) + i];
	      argv[top - (middle - bottom) + i] = tem;
	      SWAP_FLAGS (bottom + i, top - (middle - bottom) + i);
	    }
	  /* Exclude the moved bottom segment from further swapping.  */
	  top -= len;
	}
      else
	{
	  /* Top segment is the short one.  */
	  int len = top - middle;
	  register int i;

	  /* Swap it with the bottom part of the bottom segment.  */
	  for (i = 0; i < len; i++)
	    {
	      tem = argv[bottom + i];
	      argv[bottom + i] = argv[middle + i];
	      argv[middle + i] = tem;
	      SWAP_FLAGS (bottom + i, middle + i);
	    }
	  /* Exclude the moved top segment from further swapping.  */
	  bottom += len;
	}
    }

  /* Update records for the slots the non-options now occupy.  */

  first_nonopt += (optind - last_nonopt);
  last_nonopt = optind;
}

/* Initialize the internal data when the first call is made.  */

#if defined __STDC__ && __STDC__
static const char *_getopt_initialize (int, char *const *, const char *);
#endif
static const char *
_getopt_initialize (argc, argv, optstring)
     int argc;
     char *const *argv;
     const char *optstring;
{
  /* Start processing options with ARGV-element 1 (since ARGV-element 0
     is the program name); the sequence of previously skipped
     non-option ARGV-elements is empty.  */

  first_nonopt = last_nonopt = optind;

  nextchar = NULL;

  posixly_correct = getenv ("POSIXLY_CORRECT");

  /* Determine how to handle the ordering of options and nonoptions.  */

  if (optstring[0] == '-')
    {
      ordering = RETURN_IN_ORDER;
      ++optstring;
    }
  else if (optstring[0] == '+')
    {
      ordering = REQUIRE_ORDER;
      ++optstring;
    }
  else if (posixly_correct != NULL)
    ordering = REQUIRE_ORDER;
  else
    ordering = PERMUTE;

#ifdef _LIBC
  if (posixly_correct == NULL
      && argc == original_argc && argv == original_argv)
    {
      if (nonoption_flags_max_len == 0)
	{
	  if (__getopt_nonoption_flags == NULL
	      || __getopt_nonoption_flags[0] == '\0')
	    nonoption_flags_max_len = -1;
	  else
	    {
	      const char *orig_str = __getopt_nonoption_flags;
	      int len = nonoption_flags_max_len = strlen (orig_str);
	      if (nonoption_flags_max_len < argc)
		nonoption_flags_max_len = argc;
	      __getopt_nonoption_flags =
		(char *) malloc (nonoption_flags_max_len);
	      if (__getopt_nonoption_flags == NULL)
		nonoption_flags_max_len = -1;
	      else
		memset (__mempcpy (__getopt_nonoption_flags, orig_str, len),
			'\0', nonoption_flags_max_len - len);
	    }
	}
      nonoption_flags_len = nonoption_flags_max_len;
    }
  else
    nonoption_flags_len = 0;
#endif

  return optstring;
}

/* Scan elements of ARGV (whose length is ARGC) for option characters
   given in OPTSTRING.

   If an element of ARGV starts with '-', and is not exactly "-" or "--",
   then it is an option element.  The characters of this element
   (aside from the initial '-') are option characters.  If `getopt'
   is called repeatedly, it returns successively each of the option characters
   from each of the option elements.

   If `getopt' finds another option character, it returns that character,
   updating `optind' and `nextchar' so that the next call to `getopt' can
   resume the scan with the following option character or ARGV-element.

   If there are no more option characters, `getopt' returns -1.
   Then `optind' is the index in ARGV of the first ARGV-element
   that is not an option.  (The ARGV-elements have been permuted
   so that those that are not options now come last.)

   OPTSTRING is a string containing the legitimate option characters.
   If an option character is seen that is not listed in OPTSTRING,
   return '?' after printing an error message.  If you set `opterr' to
   zero, the error message is suppressed but we still return '?'.

   If a char in OPTSTRING is followed by a colon, that means it wants an arg,
   so the following text in the same ARGV-element, or the text of the following
   ARGV-element, is returned in `optarg'.  Two colons mean an option that
   wants an optional arg; if there is text in the current ARGV-element,
   it is returned in `optarg', otherwise `optarg' is set to zero.

   If OPTSTRING starts with `-' or `+', it requests different methods of
   handling the non-option ARGV-elements.
   See the comments about RETURN_IN_ORDER and REQUIRE_ORDER, above.

   Long-named options begin with `--' instead of `-'.
   Their names may be abbreviated as long as the abbreviation is unique
   or is an exact match for some defined option.  If they have an
   argument, it follows the option name in the same ARGV-element, separated
   from the option name by a `=', or else the in next ARGV-element.
   When `getopt' finds a long-named option, it returns 0 if that option's
   `flag' field is nonzero, the value of the option's `val' field
   if the `flag' field is zero.

   The elements of ARGV aren't really const, because we permute them.
   But we pretend they're const in the prototype to be compatible
   with other systems.

   LONGOPTS is a vector of `struct option' terminated by an
   element containing a name which is zero.

   LONGIND returns the index in LONGOPT of the long-named option found.
   It is only valid when a long-named option has been found by the most
   recent call.

   If LONG_ONLY is nonzero, '-' as well as '--' can introduce
   long-named options.  */

int
_getopt_internal (argc, argv, optstring, longopts, longind, long_only)
     int argc;
     char *const *argv;
     const char *optstring;
     const struct option *longopts;
     int *longind;
     int long_only;
{
  optarg = NULL;

  if (optind == 0 || !__getopt_initialized)
    {
      if (optind == 0)
	optind = 1;	/* Don't scan ARGV[0], the program name.  */
      optstring = _getopt_initialize (argc, argv, optstring);
      __getopt_initialized = 1;
    }

  /* Test whether ARGV[optind] points to a non-option argument.
     Either it does not have option syntax, or there is an environment flag
     from the shell indicating it is not an option.  The later information
     is only used when the used in the GNU libc.  */
#ifdef _LIBC
# define NONOPTION_P (argv[optind][0] != '-' || argv[optind][1] == '\0'	      \
		      || (optind < nonoption_flags_len			      \
			  && __getopt_nonoption_flags[optind] == '1'))
#else
# define NONOPTION_P (argv[optind][0] != '-' || argv[optind][1] == '\0')
#endif

  if (nextchar == NULL || *nextchar == '\0')
    {
      /* Advance to the next ARGV-element.  */

      /* Give FIRST_NONOPT & LAST_NONOPT rational values if OPTIND has been
	 moved back by the user (who may also have changed the arguments).  */
      if (last_nonopt > optind)
	last_nonopt = optind;
      if (first_nonopt > optind)
	first_nonopt = optind;

      if (ordering == PERMUTE)
	{
	  /* If we have just processed some options following some non-options,
	     exchange them so that the options come first.  */

	  if (first_nonopt != last_nonopt && last_nonopt != optind)
	    exchange ((char **) argv);
	  else if (last_nonopt != optind)
	    first_nonopt = optind;

	  /* Skip any additional non-options
	     and extend the range of non-options previously skipped.  */

	  while (optind < argc && NONOPTION_P)
	    optind++;
	  last_nonopt = optind;
	}

      /* The special ARGV-element `--' means premature end of options.
	 Skip it like a null option,
	 then exchange with previous non-options as if it were an option,
	 then skip everything else like a non-option.  */

      if (optind != argc && !strcmp (argv[optind], "--"))
	{
	  optind++;

	  if (first_nonopt != last_nonopt && last_nonopt != optind)
	    exchange ((char **) argv);
	  else if (first_nonopt == last_nonopt)
	    first_nonopt = optind;
	  last_nonopt = argc;

	  optind = argc;
	}

      /* If we have done all the ARGV-elements, stop the scan
	 and back over any non-options that we skipped and permuted.  */

      if (optind == argc)
	{
	  /* Set the next-arg-index to point at the non-options
	     that we previously skipped, so the caller will digest them.  */
	  if (first_nonopt != last_nonopt)
	    optind = first_nonopt;
	  return -1;
	}

      /* If we have come to a non-option and did not permute it,
	 either stop the scan or describe it to the caller and pass it by.  */

      if (NONOPTION_P)
	{
	  if (ordering == REQUIRE_ORDER)
	    return -1;
	  optarg = argv[optind++];
	  return 1;
	}

      /* We have found another option-ARGV-element.
	 Skip the initial punctuation.  */

      nextchar = (argv[optind] + 1
		  + (longopts != NULL && argv[optind][1] == '-'));
    }

  /* Decode the current option-ARGV-element.  */

  /* Check whether the ARGV-element is a long option.

     If long_only and the ARGV-element has the form "-f", where f is
     a valid short option, don't consider it an abbreviated form of
     a long option that starts with f.  Otherwise there would be no
     way to give the -f short option.

     On the other hand, if there's a long option "fubar" and
     the ARGV-element is "-fu", do consider that an abbreviation of
     the long option, just like "--fu", and not "-f" with arg "u".

     This distinction seems to be the most useful approach.  */

  if (longopts != NULL
      && (argv[optind][1] == '-'
	  || (long_only && (argv[optind][2] || !my_index (optstring, argv[optind][1])))))
    {
      char *nameend;
      const struct option *p;
      const struct option *pfound = NULL;
      int exact = 0;
      int ambig = 0;
      int indfound = -1;
      int option_index;

      for (nameend = nextchar; *nameend && *nameend != '='; nameend++)
	/* Do nothing.  */ ;

      /* Test all long options for either exact match
	 or abbreviated matches.  */
      for (p = longopts, option_index = 0; p->name; p++, option_index++)
	if (!strncmp (p->name, nextchar, nameend - nextchar))
	  {
	    if ((unsigned int) (nameend - nextchar)
		== (unsigned int) strlen (p->name))
	      {
		/* Exact match found.  */
		pfound = p;
		indfound = option_index;
		exact = 1;
		break;
	      }
	    else if (pfound == NULL)
	      {
		/* First nonexact match found.  */
		pfound = p;
		indfound = option_index;
	      }
	    else
	      /* Second or later nonexact match found.  */
	      ambig = 1;
	  }

      if (ambig && !exact)
	{
	  if (opterr)
	    fprintf (stderr, _("%s: option `%s' is ambiguous\n"),
		     argv[0], argv[optind]);
	  nextchar += strlen (nextchar);
	  optind++;
	  optopt = 0;
	  return '?';
	}

      if (pfound != NULL)
	{
	  option_index = indfound;
	  optind++;
	  if (*nameend)
	    {
	      /* Don't test has_arg with >, because some C compilers don't
		 allow it to be used on enums.  */
	      if (pfound->has_arg)
		optarg = nameend + 1;
	      else
		{
		  if (opterr)
		    {
		      if (argv[optind - 1][1] == '-')
			/* --option */
			fprintf (stderr,
				 _("%s: option `--%s' doesn't allow an argument\n"),
				 argv[0], pfound->name);
		      else
			/* +option or -option */
			fprintf (stderr,
				 _("%s: option `%c%s' doesn't allow an argument\n"),
				 argv[0], argv[optind - 1][0], pfound->name);
		    }

		  nextchar += strlen (nextchar);

		  optopt = pfound->val;
		  return '?';
		}
	    }
	  else if (pfound->has_arg == 1)
	    {
	      if (optind < argc)
		optarg = argv[optind++];
	      else
		{
		  if (opterr)
		    fprintf (stderr,
			   _("%s: option `%s' requires an argument\n"),
			   argv[0], argv[optind - 1]);
		  nextchar += strlen (nextchar);
		  optopt = pfound->val;
		  return optstring[0] == ':' ? ':' : '?';
		}
	    }
	  nextchar += strlen (nextchar);
	  if (longind != NULL)
	    *longind = option_index;
	  if (pfound->flag)
	    {
	      *(pfound->flag) = pfound->val;
	      return 0;
	    }
	  return pfound->val;
	}

      /* Can't find it as a long option.  If this is not getopt_long_only,
	 or the option starts with '--' or is not a valid short
	 option, then it's an error.
	 Otherwise interpret it as a short option.  */
      if (!long_only || argv[optind][1] == '-'
	  || my_index (optstring, *nextchar) == NULL)
	{
	  if (opterr)
	    {
	      if (argv[optind][1] == '-')
		/* --option */
		fprintf (stderr, _("%s: unrecognized option `--%s'\n"),
			 argv[0], nextchar);
	      else
		/* +option or -option */
		fprintf (stderr, _("%s: unrecognized option `%c%s'\n"),
			 argv[0], argv[optind][0], nextchar);
	    }
	  nextchar = (char *) "";
	  optind++;
	  optopt = 0;
	  return '?';
	}
    }

  /* Look at and handle the next short option-character.  */

  {
    char c = *nextchar++;
    char *temp = my_index (optstring, c);

    /* Increment `optind' when we start to process its last character.  */
    if (*nextchar == '\0')
      ++optind;

    if (temp == NULL || c == ':')
      {
	if (opterr)
	  {
	    if (posixly_correct)
	      /* 1003.2 specifies the format of this message.  */
	      fprintf (stderr, _("%s: illegal option -- %c\n"),
		       argv[0], c);
	    else
	      fprintf (stderr, _("%s: invalid option -- %c\n"),
		       argv[0], c);
	  }
	optopt = c;
	return '?';
      }
    /* Convenience. Treat POSIX -W foo same as long option --foo */
    if (temp[0] == 'W' && temp[1] == ';')
      {
	char *nameend;
	const struct option *p;
	const struct option *pfound = NULL;
	int exact = 0;
	int ambig = 0;
	int indfound = 0;
	int option_index;

	/* This is an option that requires an argument.  */
	if (*nextchar != '\0')
	  {
	    optarg = nextchar;
	    /* If we end this ARGV-element by taking the rest as an arg,
	       we must advance to the next element now.  */
	    optind++;
	  }
	else if (optind == argc)
	  {
	    if (opterr)
	      {
		/* 1003.2 specifies the format of this message.  */
		fprintf (stderr, _("%s: option requires an argument -- %c\n"),
			 argv[0], c);
	      }
	    optopt = c;
	    if (optstring[0] == ':')
	      c = ':';
	    else
	      c = '?';
	    return c;
	  }
	else
	  /* We already incremented `optind' once;
	     increment it again when taking next ARGV-elt as argument.  */
	  optarg = argv[optind++];

	/* optarg is now the argument, see if it's in the
	   table of longopts.  */

	for (nextchar = nameend = optarg; *nameend && *nameend != '='; nameend++)
	  /* Do nothing.  */ ;

	/* Test all long options for either exact match
	   or abbreviated matches.  */
	for (p = longopts, option_index = 0; p->name; p++, option_index++)
	  if (!strncmp (p->name, nextchar, nameend - nextchar))
	    {
	      if ((unsigned int) (nameend - nextchar) == strlen (p->name))
		{
		  /* Exact match found.  */
		  pfound = p;
		  indfound = option_index;
		  exact = 1;
		  break;
		}
	      else if (pfound == NULL)
		{
		  /* First nonexact match found.  */
		  pfound = p;
		  indfound = option_index;
		}
	      else
		/* Second or later nonexact match found.  */
		ambig = 1;
	    }
	if (ambig && !exact)
	  {
	    if (opterr)
	      fprintf (stderr, _("%s: option `-W %s' is ambiguous\n"),
		       argv[0], argv[optind]);
	    nextchar += strlen (nextchar);
	    optind++;
	    return '?';
	  }
	if (pfound != NULL)
	  {
	    option_index = indfound;
	    if (*nameend)
	      {
		/* Don't test has_arg with >, because some C compilers don't
		   allow it to be used on enums.  */
		if (pfound->has_arg)
		  optarg = nameend + 1;
		else
		  {
		    if (opterr)
		      fprintf (stderr, _("\
%s: option `-W %s' doesn't allow an argument\n"),
			       argv[0], pfound->name);

		    nextchar += strlen (nextchar);
		    return '?';
		  }
	      }
	    else if (pfound->has_arg == 1)
	      {
		if (optind < argc)
		  optarg = argv[optind++];
		else
		  {
		    if (opterr)
		      fprintf (stderr,
			       _("%s: option `%s' requires an argument\n"),
			       argv[0], argv[optind - 1]);
		    nextchar += strlen (nextchar);
		    return optstring[0] == ':' ? ':' : '?';
		  }
	      }
	    nextchar += strlen (nextchar);
	    if (longind != NULL)
	      *longind = option_index;
	    if (pfound->flag)
	      {
		*(pfound->flag) = pfound->val;
		return 0;
	      }
	    return pfound->val;
	  }
	  nextchar = NULL;
	  return 'W';	/* Let the application handle it.   */
      }
    if (temp[1] == ':')
      {
	if (temp[2] == ':')
	  {
	    /* This is an option that accepts an argument optionally.  */
	    if (*nextchar != '\0')
	      {
		optarg = nextchar;
		optind++;
	      }
	    else
	      optarg = NULL;
	    nextchar = NULL;
	  }
	else
	  {
	    /* This is an option that requires an argument.  */
	    if (*nextchar != '\0')
	      {
		optarg = nextchar;
		/* If we end this ARGV-element by taking the rest as an arg,
		   we must advance to the next element now.  */
		optind++;
	      }
	    else if (optind == argc)
	      {
		if (opterr)
		  {
		    /* 1003.2 specifies the format of this message.  */
		    fprintf (stderr,
			   _("%s: option requires an argument -- %c\n"),
			   argv[0], c);
		  }
		optopt = c;
		if (optstring[0] == ':')
		  c = ':';
		else
		  c = '?';
	      }
	    else
	      /* We already incremented `optind' once;
		 increment it again when taking next ARGV-elt as argument.  */
	      optarg = argv[optind++];
	    nextchar = NULL;
	  }
      }
    return c;
  }
}

int
getopt (argc, argv, optstring)
     int argc;
     char *const *argv;
     const char *optstring;
{
  return _getopt_internal (argc, argv, optstring,
			   (const struct option *) 0,
			   (int *) 0,
			   0);
}

int
getopt_long (argc, argv, options, long_options, opt_index)
     int argc;
     char *const *argv;
     const char *options;
     const struct option *long_options;
     int *opt_index;
{
  return _getopt_internal (argc, argv, options, long_options, opt_index, 0);
}

int
getopt_long_only (argc, argv, options, long_options, opt_index)
     int argc;
     char *const *argv;
     const char *options;
     const struct option *long_options;
     int *opt_index;
{
  return _getopt_internal (argc, argv, options, long_options, opt_index, 1);
}


/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: simple programmatic interface for encoder mode setup

 ********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: libvorbis codec headers

 ********************************************************************/

#define _V_CODECI_H_

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: PCM data envelope analysis and manipulation

 ********************************************************************/

#define _V_ENVELOPE_

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: modified discrete cosine transform prototypes

 ********************************************************************/

#define _OGG_mdct_H_

/*#define MDCT_INTEGERIZED  <- be warned there could be some hurt left here*/
#ifdef MDCT_INTEGERIZED

#define DATA_TYPE int
#define REG_TYPE  register int
#define TRIGBITS 14
#define cPI3_8 6270
#define cPI2_8 11585
#define cPI1_8 15137

#define FLOAT_CONV(x) ((int)((x)*(1<<TRIGBITS)+.5))
#define MULT_NORM(x) ((x)>>TRIGBITS)
#define HALVE(x) ((x)>>1)

#else

#define DATA_TYPE float
#define REG_TYPE  float
#define cPI3_8 .38268343236508977175F
#define cPI2_8 .70710678118654752441F
#define cPI1_8 .92387953251128675613F

#define FLOAT_CONV(x) (x)
#define MULT_NORM(x) (x)
#define HALVE(x) ((x)*.5f)

#endif


typedef struct {
  int n;
  int log2n;
  
  DATA_TYPE *trig;
  int       *bitrev;

  DATA_TYPE scale;
} mdct_lookup;

extern void mdct_init(mdct_lookup *lookup,int n);
extern void mdct_clear(mdct_lookup *l);
extern void mdct_forward(mdct_lookup *init, DATA_TYPE *in, DATA_TYPE *out);
extern void mdct_backward(mdct_lookup *init, DATA_TYPE *in, DATA_TYPE *out);


#define VE_PRE    16
#define VE_WIN    4
#define VE_POST   2
#define VE_AMP    (VE_PRE+VE_POST-1)

#define VE_BANDS  7
#define VE_NEARDC 15

#define VE_MINSTRETCH 2   /* a bit less than short block */
#define VE_MAXSTRETCH 12  /* one-third full block */

typedef struct {
  float ampbuf[VE_AMP];
  int   ampptr;

  float nearDC[VE_NEARDC];
  float nearDC_acc;
  float nearDC_partialacc;
  int   nearptr;

} envelope_filter_state;

typedef struct {
  int begin;
  int end;
  float *window;
  float total;
} envelope_band;

typedef struct {
  int ch;
  int winlength;
  int searchstep;
  float minenergy;

  mdct_lookup  mdct;
  float       *mdct_win;

  envelope_band          band[VE_BANDS];
  envelope_filter_state *filter;
  int   stretch;

  int                   *mark;

  long storage;
  long current;
  long curmark;
  long cursor;
} envelope_lookup;

extern void _ve_envelope_init(envelope_lookup *e,vorbis_info *vi);
extern void _ve_envelope_clear(envelope_lookup *e);
extern long _ve_envelope_search(vorbis_dsp_state *v);
extern void _ve_envelope_shift(envelope_lookup *e,long shift);
extern int  _ve_envelope_mark(vorbis_dsp_state *v);

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: basic shared codebook operations

 ********************************************************************/

#define _V_CODEBOOK_H_

/* This structure encapsulates huffman and VQ style encoding books; it
   doesn't do anything specific to either.

   valuelist/quantlist are nonNULL (and q_* significant) only if
   there's entry->value mapping to be done.

   If encode-side mapping must be done (and thus the entry needs to be
   hunted), the auxiliary encode pointer will point to a decision
   tree.  This is true of both VQ and huffman, but is mostly useful
   with VQ.

*/

typedef struct static_codebook{
  long   dim;            /* codebook dimensions (elements per vector) */
  long   entries;        /* codebook entries */
  long  *lengthlist;     /* codeword lengths in bits */

  /* mapping ***************************************************************/
  int    maptype;        /* 0=none
			    1=implicitly populated values from map column 
			    2=listed arbitrary values */

  /* The below does a linear, single monotonic sequence mapping. */
  long     q_min;       /* packed 32 bit float; quant value 0 maps to minval */
  long     q_delta;     /* packed 32 bit float; val 1 - val 0 == delta */
  int      q_quant;     /* bits: 0 < quant <= 16 */
  int      q_sequencep; /* bitflag */

  long     *quantlist;  /* map == 1: (int)(entries^(1/dim)) element column map
			   map == 2: list of dim*entries quantized entry vals
			*/

  /* encode helpers ********************************************************/
  struct encode_aux_nearestmatch *nearest_tree;
  struct encode_aux_threshmatch  *thresh_tree;
  struct encode_aux_pigeonhole  *pigeon_tree;

  int allocedp;
} static_codebook;

/* this structures an arbitrary trained book to quickly find the
   nearest cell match */
typedef struct encode_aux_nearestmatch{
  /* pre-calculated partitioning tree */
  long   *ptr0;
  long   *ptr1;

  long   *p;         /* decision points (each is an entry) */
  long   *q;         /* decision points (each is an entry) */
  long   aux;        /* number of tree entries */
  long   alloc;       
} encode_aux_nearestmatch;

/* assumes a maptype of 1; encode side only, so that's OK */
typedef struct encode_aux_threshmatch{
  float *quantthresh;
  long   *quantmap;
  int     quantvals; 
  int     threshvals; 
} encode_aux_threshmatch;

typedef struct encode_aux_pigeonhole{
  float min;
  float del;

  int  mapentries;
  int  quantvals;
  long *pigeonmap;

  long fittotal;
  long *fitlist;
  long *fitmap;
  long *fitlength;
} encode_aux_pigeonhole;

typedef struct codebook{
  long dim;           /* codebook dimensions (elements per vector) */
  long entries;       /* codebook entries */
  long used_entries;  /* populated codebook entries */
  const static_codebook *c;

  /* for encode, the below are entry-ordered, fully populated */
  /* for decode, the below are ordered by bitreversed codeword and only
     used entries are populated */
  float        *valuelist;  /* list of dim*entries actual entry values */  
  ogg_uint32_t *codelist;   /* list of bitstream codewords for each entry */

  int          *dec_index;  /* only used if sparseness collapsed */
  char         *dec_codelengths;
  ogg_uint32_t *dec_firsttable;
  int           dec_firsttablen;
  int           dec_maxlength;

} codebook;

extern void vorbis_staticbook_clear(static_codebook *b);
extern void vorbis_staticbook_destroy(static_codebook *b);
extern int vorbis_book_init_encode(codebook *dest,const static_codebook *source);
extern int vorbis_book_init_decode(codebook *dest,const static_codebook *source);
extern void vorbis_book_clear(codebook *b);

extern float *_book_unquantize(const static_codebook *b,int n,int *map);
extern float *_book_logdist(const static_codebook *b,float *vals);
extern float _float32_unpack(long val);
extern long   _float32_pack(float val);
extern int  _best(codebook *book, float *a, int step);
extern int _ilog(unsigned int v);
extern long _book_maptype1_quantvals(const static_codebook *b);

extern int vorbis_book_besterror(codebook *book,float *a,int step,int addmul);
extern long vorbis_book_codeword(codebook *book,int entry);
extern long vorbis_book_codelen(codebook *book,int entry);



extern int vorbis_staticbook_pack(const static_codebook *c,oggpack_buffer *b);
extern int vorbis_staticbook_unpack(oggpack_buffer *b,static_codebook *c);

extern int vorbis_book_encode(codebook *book, int a, oggpack_buffer *b);
extern int vorbis_book_errorv(codebook *book, float *a);
extern int vorbis_book_encodev(codebook *book, int best,float *a, 
			       oggpack_buffer *b);

extern long vorbis_book_decode(codebook *book, oggpack_buffer *b);
extern long vorbis_book_decodevs_add(codebook *book, float *a, 
				     oggpack_buffer *b,int n);
extern long vorbis_book_decodev_set(codebook *book, float *a, 
				    oggpack_buffer *b,int n);
extern long vorbis_book_decodev_add(codebook *book, float *a, 
				    oggpack_buffer *b,int n);
extern long vorbis_book_decodevv_add(codebook *book, float **a,
				     long off,int ch, 
				    oggpack_buffer *b,int n);

#define BLOCKTYPE_IMPULSE    0
#define BLOCKTYPE_PADDING    1
#define BLOCKTYPE_TRANSITION 0 
#define BLOCKTYPE_LONG       1

#define PACKETBLOBS 15

typedef struct vorbis_block_internal{
  float  **pcmdelay;  /* this is a pointer into local storage */ 
  float  ampmax;
  int    blocktype;

  ogg_uint32_t   packetblob_markers[PACKETBLOBS];
} vorbis_block_internal;

typedef void vorbis_look_floor;
typedef void vorbis_look_residue;
typedef void vorbis_look_transform;

/* mode ************************************************************/
typedef struct {
  int blockflag;
  int windowtype;
  int transformtype;
  int mapping;
} vorbis_info_mode;

typedef void vorbis_info_floor;
typedef void vorbis_info_residue;
typedef void vorbis_info_mapping;

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: random psychoacoustics (not including preecho)

 ********************************************************************/

#define _V_PSY_H_

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: fft transform

 ********************************************************************/

#define _V_SMFT_H_


typedef struct {
  int n;
  float *trigcache;
  int *splitcache;
} drft_lookup;

extern void drft_forward(drft_lookup *l,float *data);
extern void drft_backward(drft_lookup *l,float *data);
extern void drft_init(drft_lookup *l,int n);
extern void drft_clear(drft_lookup *l);

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: libvorbis backend and mapping structures; needed for 
           static mode headers

 ********************************************************************/

/* this is exposed up here because we need it for static modes.
   Lookups for each backend aren't exposed because there's no reason
   to do so */

#define _vorbis_backend_h_

/* this would all be simpler/shorter with templates, but.... */
/* Floor backend generic *****************************************/
typedef struct{
  void                   (*pack)  (vorbis_info_floor *,oggpack_buffer *);
  vorbis_info_floor     *(*unpack)(vorbis_info *,oggpack_buffer *);
  vorbis_look_floor     *(*look)  (vorbis_dsp_state *,vorbis_info_floor *);
  void (*free_info) (vorbis_info_floor *);
  void (*free_look) (vorbis_look_floor *);
  void *(*inverse1)  (struct vorbis_block *,vorbis_look_floor *);
  int   (*inverse2)  (struct vorbis_block *,vorbis_look_floor *,
		     void *buffer,float *);
} vorbis_func_floor;

typedef struct{
  int   order;
  long  rate;
  long  barkmap;

  int   ampbits;
  int   ampdB;

  int   numbooks; /* <= 16 */
  int   books[16];

  float lessthan;     /* encode-only config setting hacks for libvorbis */
  float greaterthan;  /* encode-only config setting hacks for libvorbis */

} vorbis_info_floor0;


#define VIF_POSIT 63
#define VIF_CLASS 16
#define VIF_PARTS 31
typedef struct{
  int   partitions;                /* 0 to 31 */
  int   partitionclass[VIF_PARTS]; /* 0 to 15 */

  int   class_dim[VIF_CLASS];        /* 1 to 8 */
  int   class_subs[VIF_CLASS];       /* 0,1,2,3 (bits: 1<<n poss) */
  int   class_book[VIF_CLASS];       /* subs ^ dim entries */
  int   class_subbook[VIF_CLASS][8]; /* [VIF_CLASS][subs] */


  int   mult;                      /* 1 2 3 or 4 */ 
  int   postlist[VIF_POSIT+2];    /* first two implicit */ 


  /* encode side analysis parameters */
  float maxover;     
  float maxunder;  
  float maxerr;    

  float twofitweight;  
  float twofitatten;

  int   n;

} vorbis_info_floor1;

/* Residue backend generic *****************************************/
typedef struct{
  void                 (*pack)  (vorbis_info_residue *,oggpack_buffer *);
  vorbis_info_residue *(*unpack)(vorbis_info *,oggpack_buffer *);
  vorbis_look_residue *(*look)  (vorbis_dsp_state *,
				 vorbis_info_residue *);
  void (*free_info)    (vorbis_info_residue *);
  void (*free_look)    (vorbis_look_residue *);
  long **(*class)      (struct vorbis_block *,vorbis_look_residue *,
			float **,int *,int);
  int  (*forward)      (struct vorbis_block *,vorbis_look_residue *,
			float **,float **,int *,int,long **);
  int  (*inverse)      (struct vorbis_block *,vorbis_look_residue *,
			float **,int *,int);
} vorbis_func_residue;

typedef struct vorbis_info_residue0{
/* block-partitioned VQ coded straight residue */
  long  begin;
  long  end;

  /* first stage (lossless partitioning) */
  int    grouping;         /* group n vectors per partition */
  int    partitions;       /* possible codebooks for a partition */
  int    groupbook;        /* huffbook for partitioning */
  int    secondstages[64]; /* expanded out to pointers in lookup */
  int    booklist[256];    /* list of second stage books */

  float  classmetric1[64];  
  float  classmetric2[64];  

} vorbis_info_residue0;

/* Mapping backend generic *****************************************/
typedef struct{
  void                 (*pack)  (vorbis_info *,vorbis_info_mapping *,
				 oggpack_buffer *);
  vorbis_info_mapping *(*unpack)(vorbis_info *,oggpack_buffer *);
  void (*free_info)    (vorbis_info_mapping *);
  int  (*forward)      (struct vorbis_block *vb);
  int  (*inverse)      (struct vorbis_block *vb,vorbis_info_mapping *);
} vorbis_func_mapping;

typedef struct vorbis_info_mapping0{
  int   submaps;  /* <= 16 */
  int   chmuxlist[256];   /* up to 256 channels in a Vorbis stream */
  
  int   floorsubmap[16];   /* [mux] submap to floors */
  int   residuesubmap[16]; /* [mux] submap to residue */

  int   coupling_steps;
  int   coupling_mag[256];
  int   coupling_ang[256];

} vorbis_info_mapping0;

#ifndef EHMER_MAX
#define EHMER_MAX 56
#endif

/* psychoacoustic setup ********************************************/
#define P_BANDS 17      /* 62Hz to 16kHz */
#define P_LEVELS 8      /* 30dB to 100dB */
#define P_LEVEL_0 30.    /* 30 dB */
#define P_NOISECURVES 3

#define NOISE_COMPAND_LEVELS 40
typedef struct vorbis_info_psy{
  int   blockflag;

  float ath_adjatt;
  float ath_maxatt;

  float tone_masteratt[P_NOISECURVES];
  float tone_centerboost;
  float tone_decay;
  float tone_abs_limit;
  float toneatt[P_BANDS];

  int noisemaskp;
  float noisemaxsupp;
  float noisewindowlo;
  float noisewindowhi;
  int   noisewindowlomin;
  int   noisewindowhimin;
  int   noisewindowfixed;
  float noiseoff[P_NOISECURVES][P_BANDS];
  float noisecompand[NOISE_COMPAND_LEVELS];

  float max_curve_dB;

  int normal_channel_p;
  int normal_point_p;
  int normal_start;
  int normal_partition;
  double normal_thresh;
} vorbis_info_psy;

typedef struct{
  int   eighth_octave_lines;

  /* for block long/short tuning; encode only */
  float preecho_thresh[VE_BANDS];
  float postecho_thresh[VE_BANDS];
  float stretch_penalty;
  float preecho_minenergy;

  float ampmax_att_per_sec;

  /* channel coupling config */
  int   coupling_pkHz[PACKETBLOBS];  
  int   coupling_pointlimit[2][PACKETBLOBS];  
  int   coupling_prepointamp[PACKETBLOBS];  
  int   coupling_postpointamp[PACKETBLOBS];  
  int   sliding_lowpass[2][PACKETBLOBS];  

} vorbis_info_psy_global;

typedef struct {
  float ampmax;
  int   channels;

  vorbis_info_psy_global *gi;
  int   coupling_pointlimit[2][P_NOISECURVES];  
} vorbis_look_psy_global;


typedef struct {
  int n;
  struct vorbis_info_psy *vi;

  float ***tonecurves;
  float **noiseoffset;

  float *ath;
  long  *octave;             /* in n.ocshift format */
  long  *bark;

  long  firstoc;
  long  shiftoc;
  int   eighth_octave_lines; /* power of two, please */
  int   total_octave_lines;  
  long  rate; /* cache it */
} vorbis_look_psy;

extern void   _vp_psy_init(vorbis_look_psy *p,vorbis_info_psy *vi,
			   vorbis_info_psy_global *gi,int n,long rate);
extern void   _vp_psy_clear(vorbis_look_psy *p);
extern void  *_vi_psy_dup(void *source);

extern void   _vi_psy_free(vorbis_info_psy *i);
extern vorbis_info_psy *_vi_psy_copy(vorbis_info_psy *i);

extern void _vp_remove_floor(vorbis_look_psy *p,
			     float *mdct,
			     int *icodedflr,
			     float *residue,
			     int sliding_lowpass);

extern void _vp_noisemask(vorbis_look_psy *p,
			  float *logmdct, 
			  float *logmask);

extern void _vp_tonemask(vorbis_look_psy *p,
			 float *logfft,
			 float *logmask,
			 float global_specmax,
			 float local_specmax);

extern void _vp_offset_and_mix(vorbis_look_psy *p,
			       float *noise,
			       float *tone,
			       int offset_select,
			       float *logmask);

extern float _vp_ampmax_decay(float amp,vorbis_dsp_state *vd);

extern float **_vp_quantize_couple_memo(vorbis_block *vb,
					vorbis_info_psy_global *g,
					vorbis_look_psy *p,
					vorbis_info_mapping0 *vi,
					float **mdct);

extern void _vp_couple(int blobno,
		       vorbis_info_psy_global *g,
		       vorbis_look_psy *p,
		       vorbis_info_mapping0 *vi,
		       float **res,
		       float **mag_memo,
		       int   **mag_sort,
		       int   **ifloor,
		       int   *nonzero,
		       int   sliding_lowpass);

extern void _vp_noise_normalize(vorbis_look_psy *p,
				float *in,float *out,int *sortedindex);

extern void _vp_noise_normalize_sort(vorbis_look_psy *p,
				     float *magnitudes,int *sortedindex);

extern int **_vp_quantize_couple_sort(vorbis_block *vb,
				      vorbis_look_psy *p,
				      vorbis_info_mapping0 *vi,
				      float **mags);
/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: bitrate tracking and management

 ********************************************************************/

#define _V_BITRATE_H_

#define _OS_H
/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: #ifdef jail to whip a few platforms into the UNIX ideal.

 ********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: miscellaneous prototypes

 ********************************************************************/

#define _V_RANDOM_H_

extern int analysis_noisy;

extern void *_vorbis_block_alloc(vorbis_block *vb,long bytes);
extern void _vorbis_block_ripcord(vorbis_block *vb);
extern void _analysis_output(char *base,int i,float *v,int n,int bark,int dB,
			     ogg_int64_t off);

#ifndef _V_IFDEFJAIL_H_
#  define _V_IFDEFJAIL_H_

#  ifdef __GNUC__
#    define STIN static __inline__
#  elif _WIN32
#    define STIN static __inline
#  else
#    define STIN static
#  endif

#ifdef DJGPP
#  define rint(x)   (floor((x)+0.5f))
#endif

#ifndef M_PI
#  define M_PI (3.1415926536f)
#endif

#ifndef FAST_HYPOT
#  define FAST_HYPOT hypot
#endif

#endif

#ifndef min
#  define min(x,y)  ((x)>(y)?(y):(x))
#endif

#ifndef max
#  define max(x,y)  ((x)<(y)?(y):(x))
#endif

#if 0
/* XXX - no inline asm! */

#  define VORBIS_FPU_CONTROL
/* both GCC and MSVC are kinda stupid about rounding/casting to int.
   Because of encapsulation constraints (GCC can't see inside the asm
   block and so we end up doing stupid things like a store/load that
   is collectively a noop), we do it this way */

/* we must set up the fpu before this works!! */

typedef ogg_int16_t vorbis_fpu_control;

static inline void vorbis_fpu_setround(vorbis_fpu_control *fpu){
  ogg_int16_t ret;
  ogg_int16_t temp;
  __asm__ __volatile__("fnstcw %0\n\t"
	  "movw %0,%%dx\n\t"
	  "orw $62463,%%dx\n\t"
	  "movw %%dx,%1\n\t"
	  "fldcw %1\n\t":"=m"(ret):"m"(temp): "dx");
  *fpu=ret;
}

static inline void vorbis_fpu_restore(vorbis_fpu_control fpu){
  __asm__ __volatile__("fldcw %0":: "m"(fpu));
}

/* assumes the FPU is in round mode! */
static inline int vorbis_ftoi(double f){  /* yes, double!  Otherwise,
                                             we get extra fst/fld to
                                             truncate precision */
  int i;
  __asm__("fistl %0": "=m"(i) : "t"(f));
  return(i);
}
#endif


#if defined(_WIN32) && !defined(__GNUC__) && !defined(__BORLANDC__)
#  define VORBIS_FPU_CONTROL

typedef ogg_int16_t vorbis_fpu_control;

static __inline int vorbis_ftoi(double f){
	int i;
	__asm{
		fld f
		fistp i
	}
	return i;
}

static __inline void vorbis_fpu_setround(vorbis_fpu_control *fpu){
}

static __inline void vorbis_fpu_restore(vorbis_fpu_control fpu){
}

#endif


#ifndef VORBIS_FPU_CONTROL

typedef int vorbis_fpu_control;

static int vorbis_ftoi(double f){
  return (int)(f+.5);
}

/* We don't have special code for this compiler/arch, so do it the slow way */
#  define vorbis_fpu_setround(vorbis_fpu_control) {}
#  define vorbis_fpu_restore(vorbis_fpu_control) {}

#endif

/* encode side bitrate tracking */
typedef struct bitrate_manager_state {
  ogg_uint32_t  *queue_binned;
  ogg_uint32_t  *queue_actual;
  int            queue_size;

  int            queue_head;
  int            queue_bins;

  long          *avg_binacc;
  int            avg_center;
  int            avg_tail;
  ogg_uint32_t   avg_centeracc;
  ogg_uint32_t   avg_sampleacc;
  ogg_uint32_t   avg_sampledesired;
  ogg_uint32_t   avg_centerdesired;

  long          *minmax_binstack;
  long          *minmax_posstack;
  long          *minmax_limitstack;
  long           minmax_stackptr;

  long           minmax_acctotal;
  int            minmax_tail;
  ogg_uint32_t   minmax_sampleacc;
  ogg_uint32_t   minmax_sampledesired;

  int            next_to_flush;
  int            last_to_flush;
  
  double         avgfloat;

  /* unfortunately, we need to hold queued packet data somewhere */
  oggpack_buffer *packetbuffers;
  ogg_packet     *packets;

} bitrate_manager_state;

typedef struct bitrate_manager_info{
  /* detailed bitrate management setup */
  double queue_avg_time;
  double queue_avg_center;
  double queue_minmax_time;
  double queue_hardmin;
  double queue_hardmax;
  double queue_avgmin;
  double queue_avgmax;

  double avgfloat_downslew_max;
  double avgfloat_upslew_max;

} bitrate_manager_info;

extern void vorbis_bitrate_init(vorbis_info *vi,bitrate_manager_state *bs);
extern void vorbis_bitrate_clear(bitrate_manager_state *bs);
extern int vorbis_bitrate_managed(vorbis_block *vb);
extern int vorbis_bitrate_addblock(vorbis_block *vb);
extern int vorbis_bitrate_flushpacket(vorbis_dsp_state *vd, ogg_packet *op);

typedef struct private_state {
  /* local lookup storage */
  envelope_lookup        *ve; /* envelope lookup */    
  int                     window[2];
  vorbis_look_transform **transform[2];    /* block, type */
  drft_lookup             fft_look[2];

  int                     modebits;
  vorbis_look_floor     **flr;
  vorbis_look_residue   **residue;
  vorbis_look_psy        *psy;
  vorbis_look_psy_global *psy_g_look;

  /* local storage, only used on the encoding side.  This way the
     application does not need to worry about freeing some packets'
     memory and not others'; packet storage is always tracked.
     Cleared next call to a _dsp_ function */
  unsigned char *header;
  unsigned char *header1;
  unsigned char *header2;

  bitrate_manager_state bms;

  ogg_int64_t sample_count;
} private_state;

/* codec_setup_info contains all the setup information specific to the
   specific compression/decompression mode in progress (eg,
   psychoacoustic settings, channel setup, options, codebook
   etc).  
*********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: highlevel encoder setup struct seperated out for vorbisenc clarity

 ********************************************************************/

typedef struct highlevel_byblocktype {
  double tone_mask_setting;
  double tone_peaklimit_setting;
  double noise_bias_setting;
  double noise_compand_setting;
} highlevel_byblocktype;
  
typedef struct highlevel_encode_setup {
  void *setup;
  int   set_in_stone;

  double base_setting;
  double long_setting;
  double short_setting;
  double impulse_noisetune;

  int    managed;
  long   bitrate_min;
  long   bitrate_av_lo;
  long   bitrate_av_hi;
  long   bitrate_max;
  double bitrate_limit_window;
  double bitrate_av_window;
  double bitrate_av_window_center;
  
  int impulse_block_p;
  int noise_normalize_p;

  double stereo_point_setting;
  double lowpass_kHz;

  double ath_floating_dB;
  double ath_absolute_dB;

  double amplitude_track_dBpersec;
  double trigger_setting;
  
  highlevel_byblocktype block[4]; /* padding, impulse, transition, long */

} highlevel_encode_setup;

typedef struct codec_setup_info {

  /* Vorbis supports only short and long blocks, but allows the
     encoder to choose the sizes */

  long blocksizes[2];

  /* modes are the primary means of supporting on-the-fly different
     blocksizes, different channel mappings (LR or M/A),
     different residue backends, etc.  Each mode consists of a
     blocksize flag and a mapping (along with the mapping setup */

  int        modes;
  int        maps;
  int        floors;
  int        residues;
  int        books;
  int        psys;     /* encode only */

  vorbis_info_mode       *mode_param[64];
  int                     map_type[64];
  vorbis_info_mapping    *map_param[64];
  int                     floor_type[64];
  vorbis_info_floor      *floor_param[64];
  int                     residue_type[64];
  vorbis_info_residue    *residue_param[64];
  static_codebook        *book_param[256];
  codebook               *fullbooks;

  vorbis_info_psy        *psy_param[4]; /* encode only */
  vorbis_info_psy_global psy_g_param;

  bitrate_manager_info   bi;
  highlevel_encode_setup hi; /* used only by vorbisenc.c.  It's a
                                highly redundant structure, but
                                improves clarity of program flow. */
  int         halfrate_flag; /* painless downsample for decode */  
} codec_setup_info;

extern vorbis_look_psy_global *_vp_global_look(vorbis_info *vi);
extern void _vp_global_free(vorbis_look_psy_global *look);

/* careful with this; it's using static array sizing to make managing
   all the modes a little less annoying.  If we use a residue backend
   with > 12 partition types, or a different division of iteration,
   this needs to be updated. */
typedef struct {
  static_codebook *books[12][3];
} static_bookblock;

typedef struct {
  int res_type;
  int limit_type; /* 0 lowpass limited, 1 point stereo limited */
  vorbis_info_residue0 *res;
  static_codebook  *book_aux;
  static_codebook  *book_aux_managed;
  static_bookblock *books_base;
  static_bookblock *books_base_managed;
} vorbis_residue_template;

typedef struct {
  vorbis_info_mapping0    *map;
  vorbis_residue_template *res;
} vorbis_mapping_template;

typedef struct vp_adjblock{
  int block[P_BANDS];
} vp_adjblock;

typedef struct {
  int data[NOISE_COMPAND_LEVELS];
} compandblock;

/* high level configuration information for setting things up
   step-by-step with the detailed vorbis_encode_ctl interface.
   There's a fair amount of redundancy such that interactive setup
   does not directly deal with any vorbis_info or codec_setup_info
   initialization; it's all stored (until full init) in this highlevel
   setup, then flushed out to the real codec setup structs later. */

typedef struct {
  int att[P_NOISECURVES];
  float boost;
  float decay;
} att3;
typedef struct { int data[P_NOISECURVES]; } adj3; 

typedef struct {
  int   pre[PACKETBLOBS];
  int   post[PACKETBLOBS];
  float kHz[PACKETBLOBS];
  float lowpasskHz[PACKETBLOBS];
} adj_stereo;

typedef struct {
  int lo;
  int hi;
  int fixed;
} noiseguard;
typedef struct {
  int data[P_NOISECURVES][17];
} noise3;

typedef struct {
  int      mappings;
  double  *rate_mapping;
  double  *quality_mapping;
  int      coupling_restriction;
  long     samplerate_min_restriction;
  long     samplerate_max_restriction;


  int     *blocksize_short;
  int     *blocksize_long;

  att3    *psy_tone_masteratt;
  int     *psy_tone_0dB;
  int     *psy_tone_dBsuppress;

  vp_adjblock *psy_tone_adj_impulse;
  vp_adjblock *psy_tone_adj_long;
  vp_adjblock *psy_tone_adj_other;

  noiseguard  *psy_noiseguards;
  noise3      *psy_noise_bias_impulse;
  noise3      *psy_noise_bias_padding;
  noise3      *psy_noise_bias_trans;
  noise3      *psy_noise_bias_long;
  int         *psy_noise_dBsuppress;

  compandblock  *psy_noise_compand;
  double        *psy_noise_compand_short_mapping;
  double        *psy_noise_compand_long_mapping;

  int      *psy_noise_normal_start[2];
  int      *psy_noise_normal_partition[2];
  double   *psy_noise_normal_thresh;

  int      *psy_ath_float;
  int      *psy_ath_abs;

  double   *psy_lowpass;

  vorbis_info_psy_global *global_params;
  double     *global_mapping;
  adj_stereo *stereo_modes;

  static_codebook ***floor_books;
  vorbis_info_floor1 *floor_params;
  int *floor_short_mapping;
  int *floor_long_mapping;

  vorbis_mapping_template *maps;
} ve_setup_data_template;

/* a few static coder conventions */
static vorbis_info_mode _mode_template[2]={
  {0,0,0,0},
  {1,0,0,1}
};

static vorbis_info_mapping0 _map_nominal[2]={
  {1, {0,0}, {0}, {0}, 1,{0},{1}},
  {1, {0,0}, {1}, {1}, 1,{0},{1}}
};

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: toplevel settings for 44.1/48kHz

 ********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: key floor settings

 ********************************************************************/
/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: static codebooks autogenerated by huff/huffbuld

 ********************************************************************/

static long _huff_lengthlist_line_1024x27_0sub0[] = {
	 5, 5, 5, 5, 6, 5, 6, 5, 6, 5, 6, 5, 6, 5, 6, 5,
	 6, 5, 6, 5, 6, 5, 7, 5, 7, 5, 7, 5, 8, 5, 8, 6,
	 8, 6, 9, 6, 9, 6, 9, 6,10, 6,10, 6,11, 6,11, 6,
	11, 6,12, 6,12, 7,12, 7,12, 7,12, 7,12, 7,12, 7,
	12, 7,12, 7,12, 7,12, 7,12, 8,12, 8,11, 8,11, 8,
	12, 9,11, 9, 9,10,11, 9,12, 9,12,12,14,13,13,14,
	13,13,13,12,14,16,20,20,21,14,14,15,21,21,21,20,
	21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,
};

static static_codebook _huff_book_line_1024x27_0sub0 = {
	1, 128,
	_huff_lengthlist_line_1024x27_0sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_1sub0[] = {
	 2, 5, 5, 4, 5, 4, 5, 4, 5, 4, 6, 4, 6, 5, 6, 5,
	 7, 5, 7, 6, 8, 6, 8, 6, 8, 6, 9, 6,10, 6,10, 6,
};

static static_codebook _huff_book_line_1024x27_1sub0 = {
	1, 32,
	_huff_lengthlist_line_1024x27_1sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_1sub1[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 9, 5,10, 4,10, 4, 9, 4, 9, 3, 9, 4, 9, 4, 9, 4,
	 9, 4, 9, 4, 9, 4, 8, 4, 8, 4, 8, 5, 9, 5, 9, 6,
	 8, 6, 9, 7,10, 8,10, 9,10,10,10,12,11,13,12,13,
	13,15,13,14,13,14,12,15,13,15,14,15,13,16,14,16,
	14,15,14,14,14,16,15,18,15,18,16,18,18,18,18,18,
	18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,17,
};

static static_codebook _huff_book_line_1024x27_1sub1 = {
	1, 128,
	_huff_lengthlist_line_1024x27_1sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_2sub0[] = {
	 1, 5, 5, 5, 5, 5, 5, 5, 6, 5, 6, 5, 6, 5, 6, 5,
	 6, 6, 7, 6, 7, 7, 8, 8, 9, 8, 9, 9, 9, 9,10,10,
};

static static_codebook _huff_book_line_1024x27_2sub0 = {
	1, 32,
	_huff_lengthlist_line_1024x27_2sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_2sub1[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 3, 3, 4, 3, 4, 4, 5, 4, 5, 5, 5, 5, 6, 5, 6, 6,
	 7, 6, 7, 7, 7, 7, 7, 7, 8, 9, 8, 9, 8,10, 8,11,
	 8,12, 9,13, 9,14, 9,14, 8,12, 8,14, 9,14, 8,12,
	 8,11, 8,11, 8,11, 9,11,10,11,10,12,10,12,11,12,
	12,12,12,12,11,12,11,13,11,13,12,14,14,14,14,14,
	14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,13,
};

static static_codebook _huff_book_line_1024x27_2sub1 = {
	1, 128,
	_huff_lengthlist_line_1024x27_2sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_3sub1[] = {
	 0, 4, 5, 4, 5, 3, 5, 3, 5, 3, 5, 4, 4, 4, 5, 4,
	 5, 5,
};

static static_codebook _huff_book_line_1024x27_3sub1 = {
	1, 18,
	_huff_lengthlist_line_1024x27_3sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_3sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 3, 3, 3, 3, 4, 4, 4, 5, 4, 6, 5, 6, 5, 7,
	 5, 9, 5,10, 6,11, 6,12, 7,13, 8,15, 8,15, 9,15,
	 9,15,
};

static static_codebook _huff_book_line_1024x27_3sub2 = {
	1, 50,
	_huff_lengthlist_line_1024x27_3sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_3sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 2, 9, 2, 9, 2, 9, 4, 9, 6, 9, 7, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static static_codebook _huff_book_line_1024x27_3sub3 = {
	1, 128,
	_huff_lengthlist_line_1024x27_3sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_4sub1[] = {
	 0, 4, 5, 4, 5, 4, 6, 3, 6, 3, 5, 3, 5, 3, 6, 4,
	 6, 4,
};

static static_codebook _huff_book_line_1024x27_4sub1 = {
	1, 18,
	_huff_lengthlist_line_1024x27_4sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_4sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 2, 4, 2, 5, 3, 5, 4, 6, 6, 6, 6, 7, 7,
	 7, 8, 8, 8, 8, 8, 9, 8, 9, 9, 9, 9, 9,10,10,11,
	10,11,
};

static static_codebook _huff_book_line_1024x27_4sub2 = {
	1, 50,
	_huff_lengthlist_line_1024x27_4sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_4sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 3, 3, 3, 6, 4, 6, 4, 5, 5, 7, 4, 8, 5, 9,
	 4, 9, 5, 9, 5, 9, 6, 9, 5, 9, 7, 9, 7, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static static_codebook _huff_book_line_1024x27_4sub3 = {
	1, 128,
	_huff_lengthlist_line_1024x27_4sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_class1[] = {
	 2, 9, 8,14, 7,13,11,14, 1, 5, 3, 7, 4,10, 7,12,
};

static static_codebook _huff_book_line_1024x27_class1 = {
	1, 16,
	_huff_lengthlist_line_1024x27_class1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_class2[] = {
	 1, 3, 2, 5, 4, 7, 6, 7,
};

static static_codebook _huff_book_line_1024x27_class2 = {
	1, 8,
	_huff_lengthlist_line_1024x27_class2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_class3[] = {
	 1, 5, 6,19, 5, 8,10,19, 9,10,15,19,19,19,19,19,
	 4, 7, 9,19, 6, 7,10,19,11,11,15,19,19,19,19,19,
	 8,11,13,19, 8,11,14,19,13,13,17,19,19,19,19,19,
	19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,
	 3, 7, 9,19, 6, 8,11,19,11,11,15,19,19,19,19,19,
	 5, 7,11,19, 6, 7,11,19,11,10,14,19,19,19,19,19,
	 8,11,15,19, 8,10,14,19,13,13,16,19,19,19,19,19,
	19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,
	 6, 9,11,19, 8,10,12,19,15,14,17,19,19,19,19,19,
	 5, 8,11,19, 7, 9,12,19,14,11,16,19,19,19,19,19,
	 9,10,18,19, 9,10,15,19,14,16,19,19,19,19,19,19,
	19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,
	16,17,19,19,16,17,17,19,19,19,19,19,19,19,19,19,
	12,14,16,19,12,12,16,19,19,19,19,19,19,19,19,19,
	18,18,19,19,17,16,19,19,19,19,19,19,19,19,19,19,
	19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,18,
};

static static_codebook _huff_book_line_1024x27_class3 = {
	1, 256,
	_huff_lengthlist_line_1024x27_class3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_1024x27_class4[] = {
	 1, 4, 8,12, 4, 6, 8,21, 9, 8,10,21,20,16,18,20,
	 2, 6, 8,20, 7, 6, 9,19,11, 9,10,20,17,15,16,20,
	 5, 8,11,19, 8, 8,10,15,12,10,12,15,20,20,15,20,
	17,20,20,20,15,20,20,19,20,20,16,19,20,20,20,20,
};

static static_codebook _huff_book_line_1024x27_class4 = {
	1, 64,
	_huff_lengthlist_line_1024x27_class4,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_0sub0[] = {
	 5, 5, 6, 5, 6, 5, 6, 5, 6, 5, 6, 5, 6, 5, 6, 5,
	 6, 5, 6, 5, 6, 5, 6, 5, 6, 6, 6, 6, 6, 6, 6, 6,
	 7, 6, 7, 6, 7, 6, 7, 6, 7, 6, 7, 6, 7, 6, 7, 6,
	 8, 7, 8, 7, 8, 7, 8, 7, 9, 7, 9, 7, 9, 8,10, 8,
	10, 8,10, 8,10, 8,10, 9,11, 9,11, 9,10, 9,10,10,
	11,10,11,11,11,11,12,12,13,14,13,14,16,16,16,16,
	16,16,15,16,16,16,16,16,16,16,16,16,16,16,16,16,
	16,16,16,16,16,16,16,16,16,16,16,16,16,15,15,15,
};

static static_codebook _huff_book_line_128x11_0sub0 = {
	1, 128,
	_huff_lengthlist_line_128x11_0sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_1sub0[] = {
	 2, 5, 5, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
	 6, 5, 6, 5, 6, 5, 7, 6, 7, 6, 7, 6, 8, 6, 8, 6,
};

static static_codebook _huff_book_line_128x11_1sub0 = {
	1, 32,
	_huff_lengthlist_line_128x11_1sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_1sub1[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 6, 3, 6, 3, 7, 3, 7, 4, 8, 4, 8, 4, 8, 4, 9, 4,
	10, 5, 9, 5,10, 5,10, 5,10, 5,12, 6,12, 6,10, 6,
	10, 7,10, 8,10, 8,10, 9,11, 9,12,11,10,11,11,13,
	12,12,12,13,10,13,10,13,10,13,10,13,11,13,10,13,
	10,13,10,13,10,13,10,13,11,12,13,13,13,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
};

static static_codebook _huff_book_line_128x11_1sub1 = {
	1, 128,
	_huff_lengthlist_line_128x11_1sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_2sub1[] = {
	 0, 4, 5, 4, 5, 4, 5, 3, 4, 3, 4, 4, 4, 4, 4, 5,
	 5, 5,
};

static static_codebook _huff_book_line_128x11_2sub1 = {
	1, 18,
	_huff_lengthlist_line_128x11_2sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_2sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 2, 3, 3, 4, 3, 5, 4, 5, 5, 6, 5, 6, 6, 6,
	 6, 8, 6,10, 7,10, 8,10, 8,10,10,10,10,10,10,10,
	10,10,
};

static static_codebook _huff_book_line_128x11_2sub2 = {
	1, 50,
	_huff_lengthlist_line_128x11_2sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_2sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
};

static static_codebook _huff_book_line_128x11_2sub3 = {
	1, 128,
	_huff_lengthlist_line_128x11_2sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_3sub1[] = {
	 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 4,
	 5, 4,
};

static static_codebook _huff_book_line_128x11_3sub1 = {
	1, 18,
	_huff_lengthlist_line_128x11_3sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_3sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 3, 5, 3, 6, 4, 7, 4, 7, 4, 7, 4, 8, 4,
	 8, 4, 9, 4, 9, 4, 9, 5,10, 5,11, 5,12, 6,13, 6,
	13, 7,
};

static static_codebook _huff_book_line_128x11_3sub2 = {
	1, 50,
	_huff_lengthlist_line_128x11_3sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_3sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 7, 2, 6, 2, 7, 3, 8, 4, 7, 6, 9, 7, 9, 7,
	 9, 9, 9, 8, 9, 8, 9, 9, 9, 9, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static static_codebook _huff_book_line_128x11_3sub3 = {
	1, 128,
	_huff_lengthlist_line_128x11_3sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_class1[] = {
	 1, 6, 3, 7, 2, 5, 4, 7,
};

static static_codebook _huff_book_line_128x11_class1 = {
	1, 8,
	_huff_lengthlist_line_128x11_class1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_class2[] = {
	 1, 5,11,14, 4,11,13,14,10,14,13,14,14,14,13,13,
	 2, 6,11,13, 5,11,12,13,11,12,13,13,13,13,13,13,
	 4, 8,12,13, 5, 9,11,13,12,13,13,13,13,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
};

static static_codebook _huff_book_line_128x11_class2 = {
	1, 64,
	_huff_lengthlist_line_128x11_class2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x11_class3[] = {
	 6, 7,11,16, 6, 7,10,16,11, 9,13,15,15,15,15,15,
	 4, 4, 7,14, 4, 4, 6,14, 8, 6, 8,15,15,15,15,15,
	 4, 4, 6,15, 3, 2, 4,13, 6, 5, 6,14,15,12,11,14,
	11,11,13,15, 9, 8,10,15,11,10,11,15,15,15,15,15,
};

static static_codebook _huff_book_line_128x11_class3 = {
	1, 64,
	_huff_lengthlist_line_128x11_class3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_0sub0[] = {
	 5, 5, 5, 5, 5, 5, 5, 5, 6, 5, 6, 5, 6, 5, 6, 5,
	 6, 5, 6, 5, 6, 5, 6, 5, 7, 5, 7, 5, 7, 6, 7, 6,
	 7, 6, 8, 6, 8, 6, 8, 6, 8, 6, 8, 6, 8, 7, 9, 7,
	 9, 7, 9, 7, 9, 7, 9, 7, 9, 7,10, 7,10, 8,10, 8,
	11, 8,11, 8,11, 8,12, 8,12, 8,12, 8,12, 9,12, 9,
	12, 9,12, 9,13, 9,13,10,13,10,13,11,14,12,14,13,
	14,14,16,15,17,17,19,18,19,19,19,19,19,19,19,19,
	19,19,19,19,19,19,19,19,19,19,19,18,18,18,18,18,
};

static static_codebook _huff_book_line_128x17_0sub0 = {
	1, 128,
	_huff_lengthlist_line_128x17_0sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_1sub0[] = {
	 2, 5, 5, 4, 5, 4, 5, 4, 5, 4, 5, 5, 5, 5, 6, 5,
	 6, 5, 6, 6, 7, 6, 7, 6, 8, 6, 8, 7, 9, 7, 9, 8,
};

static static_codebook _huff_book_line_128x17_1sub0 = {
	1, 32,
	_huff_lengthlist_line_128x17_1sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_1sub1[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 4, 2, 5, 3, 5, 3, 6, 4, 6, 4, 7, 4, 8, 5, 8, 5,
	 8, 6, 9, 6, 9, 7, 9, 8,10, 8,10, 9,10,10,10,10,
	10,12,10,14,11,15,12,15,11,15,11,15,11,14,11,15,
	11,14,11,13,10,12,10,14,10,14,11,13,10,12,11,15,
	12,15,13,15,12,13,14,15,15,15,15,15,15,15,15,15,
	15,15,15,15,15,15,15,15,15,15,15,15,15,14,14,14,
};

static static_codebook _huff_book_line_128x17_1sub1 = {
	1, 128,
	_huff_lengthlist_line_128x17_1sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_2sub1[] = {
	 0, 4, 5, 4, 7, 3, 8, 3, 9, 3,10, 2,12, 3,12, 4,
	11, 6,
};

static static_codebook _huff_book_line_128x17_2sub1 = {
	1, 18,
	_huff_lengthlist_line_128x17_2sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_2sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 7, 1, 8, 2, 9, 3, 9, 7, 9, 7, 9, 7, 8, 7,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8,
};

static static_codebook _huff_book_line_128x17_2sub2 = {
	1, 50,
	_huff_lengthlist_line_128x17_2sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_2sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
};

static static_codebook _huff_book_line_128x17_2sub3 = {
	1, 128,
	_huff_lengthlist_line_128x17_2sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_3sub1[] = {
	 0, 4, 4, 4, 5, 4, 5, 3, 5, 3, 5, 3, 5, 4, 6, 4,
	 6, 4,
};

static static_codebook _huff_book_line_128x17_3sub1 = {
	1, 18,
	_huff_lengthlist_line_128x17_3sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_3sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 3, 6, 3, 6, 3, 7, 4, 8, 4, 8, 4, 9, 4,
	 9, 4,10, 4,10, 5,11, 5,11, 5,12, 5,12, 6,12, 6,
	12, 7,
};

static static_codebook _huff_book_line_128x17_3sub2 = {
	1, 50,
	_huff_lengthlist_line_128x17_3sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_3sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 7, 1, 7, 3, 6, 3, 6, 4, 6, 5, 6, 8, 7,10,
	 7,11, 7,11, 8,11, 9,11, 7,11, 8,11, 8,11,10,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
};

static static_codebook _huff_book_line_128x17_3sub3 = {
	1, 128,
	_huff_lengthlist_line_128x17_3sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_class1[] = {
	 1, 3, 4, 7, 2, 6, 5, 7,
};

static static_codebook _huff_book_line_128x17_class1 = {
	1, 8,
	_huff_lengthlist_line_128x17_class1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_class2[] = {
	 1, 2, 7,14, 4, 9,13,14, 8,14,14,14,14,14,14,14,
	 3, 5,10,14, 8,14,14,14,11,14,14,14,14,14,14,14,
	 7,10,14,14,12,14,14,14,14,14,14,14,14,14,14,14,
	14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,13,
};

static static_codebook _huff_book_line_128x17_class2 = {
	1, 64,
	_huff_lengthlist_line_128x17_class2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x17_class3[] = {
	 2, 6,11,19, 5, 9,12,19,10,11,13,19,19,19,19,19,
	 2, 5, 8,19, 4, 6, 8,19, 8, 8, 9,19,19,16,19,19,
	 3, 6, 8,19, 3, 5, 7,18, 8, 8, 9,16,16,11,16,19,
	14,14,14,19,10,10,11,19,16,12,14,19,19,19,19,19,
};

static static_codebook _huff_book_line_128x17_class3 = {
	1, 64,
	_huff_lengthlist_line_128x17_class3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x4_0sub0[] = {
	 2, 2, 2, 2,
};

static static_codebook _huff_book_line_128x4_0sub0 = {
	1, 4,
	_huff_lengthlist_line_128x4_0sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x4_0sub1[] = {
	 0, 0, 0, 0, 3, 2, 3, 2, 3, 3,
};

static static_codebook _huff_book_line_128x4_0sub1 = {
	1, 10,
	_huff_lengthlist_line_128x4_0sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x4_0sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 3, 4, 4, 4, 4,
	 4, 3, 4, 4, 5, 3, 6, 4, 6,
};

static static_codebook _huff_book_line_128x4_0sub2 = {
	1, 25,
	_huff_lengthlist_line_128x4_0sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x4_0sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 2, 6, 3, 6, 3,
	 7, 4, 7, 5, 7, 5, 7, 6, 8, 8, 9,10,10,16,10,16,
	10,14,10,14,11,15,15,15,15,15,15,15,15,15,15,15,
};

static static_codebook _huff_book_line_128x4_0sub3 = {
	1, 64,
	_huff_lengthlist_line_128x4_0sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x4_class0[] = {
	 8, 8, 8,14, 7, 7, 8,13, 7, 6, 7,11,11,11,10,13,
	 9, 9,10,16, 8, 8, 9,12, 7, 7, 7,11,11,11,10,12,
	11,11,11,14,10,10,10,14, 9, 8, 9,12,13,14,12,14,
	16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,
	 8, 7, 8,11, 7, 7, 7,11, 6, 6, 6,10,10,10,10,14,
	 8, 8, 8,12, 7, 7, 8,11, 6, 7, 7,10,10,10,10,13,
	10,10,10,12,10, 9, 9,12, 9, 9, 9,12,12,13,12,14,
	16,16,16,16,16,14,16,16,14,14,16,16,16,16,16,16,
	 7, 6, 5, 7, 6, 6, 5, 6, 6, 5, 5, 5, 9,10, 9,10,
	 8, 7, 6, 7, 7, 6, 5, 6, 6, 6, 5, 6,10,10, 9, 9,
	10, 9, 7, 8, 9, 8, 7, 7, 8, 7, 6, 7,11,11,10, 9,
	16,13,14,16,14,14,13,15,16,14,12,13,16,16,14,16,
	 9, 8, 7, 8, 8, 8, 7, 8, 8, 7, 6, 7,10,10, 9,12,
	 9, 9, 8, 8, 9, 8, 7, 8, 8, 8, 6, 7,10,10, 9,12,
	11,10, 9,10,10, 9, 7, 9, 9, 8, 6, 9,12,11,10,13,
	12,16,16,16,12,13,12,16,15,12,11,14,16,16,16,16,
};

static static_codebook _huff_book_line_128x4_class0 = {
	1, 256,
	_huff_lengthlist_line_128x4_class0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_0sub1[] = {
	 0, 3, 3, 3, 3, 3, 3, 3, 3,
};

static static_codebook _huff_book_line_128x7_0sub1 = {
	1, 9,
	_huff_lengthlist_line_128x7_0sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_0sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 3, 4, 3, 4, 4, 4,
	 5, 4, 5, 4, 5, 4, 6, 4, 6,
};

static static_codebook _huff_book_line_128x7_0sub2 = {
	1, 25,
	_huff_lengthlist_line_128x7_0sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_0sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 5, 3, 5, 3, 6, 4,
	 6, 4, 6, 4, 6, 5, 6, 5, 6, 5, 6, 5, 6, 5, 5, 5,
	 6, 7, 8,10,12,12,12,12,12,12,12,12,12,12,12,12,
};

static static_codebook _huff_book_line_128x7_0sub3 = {
	1, 64,
	_huff_lengthlist_line_128x7_0sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_1sub1[] = {
	 0, 3, 3, 3, 3, 2, 4, 3, 4,
};

static static_codebook _huff_book_line_128x7_1sub1 = {
	1, 9,
	_huff_lengthlist_line_128x7_1sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_1sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 5, 3, 6, 3, 7, 3,
	 8, 3, 9, 3,10, 3,11, 4,11,
};

static static_codebook _huff_book_line_128x7_1sub2 = {
	1, 25,
	_huff_lengthlist_line_128x7_1sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_1sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,12, 2, 9, 3,10, 4,
	12, 5,12, 6,12,10,12,11,12,12,12,12,12,12,12,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
};

static static_codebook _huff_book_line_128x7_1sub3 = {
	1, 64,
	_huff_lengthlist_line_128x7_1sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_class0[] = {
	10, 7, 8,14,10, 7, 7,12,11, 8, 8,13,16,16,16,16,
	 8, 5, 5,10, 7, 4, 4, 8, 8, 5, 5, 9,16,16,16,16,
	 7, 4, 5, 7, 6, 3, 3, 6, 8, 5, 5, 7,16,14,13,16,
	 9, 7, 7,10, 7, 4, 4, 7, 9, 6, 5, 6,15,13,11,14,
};

static static_codebook _huff_book_line_128x7_class0 = {
	1, 64,
	_huff_lengthlist_line_128x7_class0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_128x7_class1[] = {
	 8,12,16,16,10,14,16,16,12,15,16,16,16,16,16,16,
	 7,11,15,16, 7,11,16,16,10,12,16,16,16,16,16,16,
	 9,15,16,16, 9,12,16,16,11,15,16,16,16,16,16,16,
	16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,
	 6,10,13,16, 7,10,15,16, 9,11,16,16,16,16,16,16,
	 4, 8,13,16, 5, 8,15,16, 7, 9,15,16,16,16,16,16,
	 6,12,16,16, 6,10,15,16, 9,10,14,16,16,16,16,16,
	14,16,16,16,12,14,16,16,15,16,16,16,16,16,16,16,
	 4, 9,11,16, 5, 9,13,16, 7, 9,15,16,16,16,16,16,
	 2, 7,11,16, 3, 6,11,16, 5, 7,12,16,16,16,16,16,
	 4, 9,14,16, 4, 7,13,16, 6, 8,13,16,16,16,16,16,
	11,14,16,16,10,16,14,16,11,16,16,16,16,16,16,16,
	 7,13,16,16, 9,13,15,16,11,13,16,16,16,16,16,16,
	 5,10,14,16, 6,10,15,16, 9,10,16,16,16,16,16,16,
	 7,13,16,16, 6,10,16,16, 9,12,16,16,16,16,16,16,
	11,16,16,16,10,16,16,16,13,16,16,16,16,16,16,16,
};

static static_codebook _huff_book_line_128x7_class1 = {
	1, 256,
	_huff_lengthlist_line_128x7_class1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x4_0sub0[] = {
	 1, 3, 2, 3,
};

static static_codebook _huff_book_line_256x4_0sub0 = {
	1, 4,
	_huff_lengthlist_line_256x4_0sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x4_0sub1[] = {
	 0, 0, 0, 0, 2, 2, 3, 3, 3, 3,
};

static static_codebook _huff_book_line_256x4_0sub1 = {
	1, 10,
	_huff_lengthlist_line_256x4_0sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x4_0sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 3, 3, 4, 3, 4,
	 4, 4, 5, 4, 5, 5, 6, 4, 6,
};

static static_codebook _huff_book_line_256x4_0sub2 = {
	1, 25,
	_huff_lengthlist_line_256x4_0sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x4_0sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 3, 4, 3, 4, 4,
	 5, 4, 5, 5, 6, 5, 7, 6, 8, 7, 9, 8, 9, 9,10,12,
	15,15,15,15,15,15,15,15,14,14,14,14,14,14,14,14,
};

static static_codebook _huff_book_line_256x4_0sub3 = {
	1, 64,
	_huff_lengthlist_line_256x4_0sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x4_class0[] = {
	 4, 5, 6,11, 5, 5, 6,10, 7, 6, 5, 6,14,13,10,10,
	 6, 6, 6,10, 6, 6, 6,10, 7, 7, 7, 9, 9,10, 9,12,
	 9, 8, 8,11, 8, 8, 8,10, 8, 8, 9,11, 8, 9, 9,13,
	18,18,18,18,16,17,18,18,12,13,14,18,14,14,10,12,
	 5, 5, 6,12, 6, 5, 6,10, 7, 7, 6, 7,13,12, 9,12,
	 6, 6, 6,11, 6, 6, 6, 9, 7, 7, 7,10,10,10, 9,12,
	 9, 8, 8,12, 8, 8, 7,11, 8, 8, 8,11, 9, 9, 8,10,
	18,18,17,18,18,18,15,18,16,14,12,18,14,12,10,11,
	 8, 7, 7,12, 8, 7, 7,10, 8, 7, 6, 6,11,10, 9,10,
	 8, 8, 7,11, 8, 7, 7, 9, 8, 8, 7, 9,10,10, 9, 9,
	10, 9, 8,12, 9, 9, 8,11,10, 9, 8, 9, 8, 8, 7, 9,
	18,18,18,18,18,18,18,18,17,16,14,18,13,12,11,13,
	12,12,13,16,11,10,10,16,12,12, 9, 7,15,12,11,11,
	14,14,14,14,12,11,11,18,14,12,10,11,18,13,11,13,
	18,18,18,17,17,16,15,17,18,15,16,18,16,12,11,11,
	18,18,18,18,18,18,18,18,18,18,17,18,18,13,12,15,
};

static static_codebook _huff_book_line_256x4_class0 = {
	1, 256,
	_huff_lengthlist_line_256x4_class0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_0sub1[] = {
	 0, 2, 3, 3, 3, 3, 4, 3, 4,
};

static static_codebook _huff_book_line_256x7_0sub1 = {
	1, 9,
	_huff_lengthlist_line_256x7_0sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_0sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 4, 3, 4, 3, 5, 3,
	 6, 3, 6, 4, 6, 4, 7, 5, 7,
};

static static_codebook _huff_book_line_256x7_0sub2 = {
	1, 25,
	_huff_lengthlist_line_256x7_0sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_0sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 5, 2, 5, 3, 5, 3,
	 6, 3, 6, 4, 7, 6, 7, 8, 7, 9, 8, 9, 9, 9,10, 9,
	11,13,11,13,10,10,13,13,13,13,13,13,12,12,12,12,
};

static static_codebook _huff_book_line_256x7_0sub3 = {
	1, 64,
	_huff_lengthlist_line_256x7_0sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_1sub1[] = {
	 0, 3, 3, 3, 3, 2, 4, 3, 4,
};

static static_codebook _huff_book_line_256x7_1sub1 = {
	1, 9,
	_huff_lengthlist_line_256x7_1sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_1sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 3, 4, 3, 4, 4,
	 5, 4, 6, 5, 6, 7, 6, 8, 8,
};

static static_codebook _huff_book_line_256x7_1sub2 = {
	1, 25,
	_huff_lengthlist_line_256x7_1sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_1sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 2, 4, 3, 6, 3, 7,
	 3, 8, 5, 8, 6, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7,
};

static static_codebook _huff_book_line_256x7_1sub3 = {
	1, 64,
	_huff_lengthlist_line_256x7_1sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_class0[] = {
	 7, 5, 5, 9, 9, 6, 6, 9,12, 8, 7, 8,11, 8, 9,15,
	 6, 3, 3, 7, 7, 4, 3, 6, 9, 6, 5, 6, 8, 6, 8,15,
	 8, 5, 5, 9, 8, 5, 4, 6,10, 7, 5, 5,11, 8, 7,15,
	14,15,13,13,13,13, 8,11,15,10, 7, 6,11, 9,10,15,
};

static static_codebook _huff_book_line_256x7_class0 = {
	1, 64,
	_huff_lengthlist_line_256x7_class0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_256x7_class1[] = {
	 5, 6, 8,15, 6, 9,10,15,10,11,12,15,15,15,15,15,
	 4, 6, 7,15, 6, 7, 8,15, 9, 8, 9,15,15,15,15,15,
	 6, 8, 9,15, 7, 7, 8,15,10, 9,10,15,15,15,15,15,
	15,13,15,15,15,10,11,15,15,13,13,15,15,15,15,15,
	 4, 6, 7,15, 6, 8, 9,15,10,10,12,15,15,15,15,15,
	 2, 5, 6,15, 5, 6, 7,15, 8, 6, 7,15,15,15,15,15,
	 5, 6, 8,15, 5, 6, 7,15, 9, 6, 7,15,15,15,15,15,
	14,12,13,15,12,10,11,15,15,15,15,15,15,15,15,15,
	 7, 8, 9,15, 9,10,10,15,15,14,14,15,15,15,15,15,
	 5, 6, 7,15, 7, 8, 9,15,12, 9,10,15,15,15,15,15,
	 7, 7, 9,15, 7, 7, 8,15,12, 8, 9,15,15,15,15,15,
	13,13,14,15,12,11,12,15,15,15,15,15,15,15,15,15,
	15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,
	13,13,13,15,15,15,15,15,15,15,15,15,15,15,15,15,
	15,12,13,15,15,12,13,15,15,14,15,15,15,15,15,15,
	15,15,15,15,15,15,13,15,15,15,15,15,15,15,15,15,
};

static static_codebook _huff_book_line_256x7_class1 = {
	1, 256,
	_huff_lengthlist_line_256x7_class1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_0sub0[] = {
	 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
	 5, 6, 5, 6, 6, 6, 6, 5, 6, 6, 7, 6, 7, 6, 7, 6,
	 7, 6, 8, 7, 8, 7, 8, 7, 8, 7, 8, 7, 9, 7, 9, 7,
	 9, 7, 9, 8, 9, 8,10, 8,10, 8,10, 7,10, 6,10, 8,
	10, 8,11, 7,10, 7,11, 8,11,11,12,12,11,11,12,11,
	13,11,13,11,13,12,15,12,13,13,14,14,14,14,14,15,
	15,15,16,14,17,19,19,18,18,18,18,18,18,18,18,18,
	18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,
};

static static_codebook _huff_book_line_512x17_0sub0 = {
	1, 128,
	_huff_lengthlist_line_512x17_0sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_1sub0[] = {
	 2, 4, 5, 4, 5, 4, 5, 4, 5, 5, 5, 5, 5, 5, 6, 5,
	 6, 5, 6, 6, 7, 6, 7, 6, 8, 7, 8, 7, 8, 7, 8, 7,
};

static static_codebook _huff_book_line_512x17_1sub0 = {
	1, 32,
	_huff_lengthlist_line_512x17_1sub0,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_1sub1[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 4, 3, 5, 3, 5, 4, 5, 4, 5, 4, 5, 5, 5, 5, 6, 5,
	 6, 5, 7, 5, 8, 6, 8, 6, 8, 6, 8, 6, 8, 7, 9, 7,
	 9, 7,11, 9,11,11,12,11,14,12,14,16,14,16,13,16,
	14,16,12,15,13,16,14,16,13,14,12,15,13,15,13,13,
	13,15,12,14,14,15,13,15,12,15,15,15,15,15,15,15,
	15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,
};

static static_codebook _huff_book_line_512x17_1sub1 = {
	1, 128,
	_huff_lengthlist_line_512x17_1sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_2sub1[] = {
	 0, 4, 5, 4, 4, 4, 5, 4, 4, 4, 5, 4, 5, 4, 5, 3,
	 5, 3,
};

static static_codebook _huff_book_line_512x17_2sub1 = {
	1, 18,
	_huff_lengthlist_line_512x17_2sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_2sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 3, 4, 3, 4, 4, 5, 4, 5, 4, 6, 4, 6, 5,
	 6, 5, 7, 5, 7, 6, 8, 6, 8, 6, 8, 7, 8, 7, 9, 7,
	 9, 8,
};

static static_codebook _huff_book_line_512x17_2sub2 = {
	1, 50,
	_huff_lengthlist_line_512x17_2sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_2sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 3, 3, 3, 3, 4, 3, 4, 4, 5, 5, 6, 6, 7, 7,
	 7, 8, 8,11, 8, 9, 9, 9,10,11,11,11, 9,10,10,11,
	11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
};

static static_codebook _huff_book_line_512x17_2sub3 = {
	1, 128,
	_huff_lengthlist_line_512x17_2sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_3sub1[] = {
	 0, 4, 4, 4, 4, 4, 4, 3, 4, 4, 4, 4, 4, 5, 4, 5,
	 5, 5,
};

static static_codebook _huff_book_line_512x17_3sub1 = {
	1, 18,
	_huff_lengthlist_line_512x17_3sub1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_3sub2[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 2, 3, 3, 4, 3, 5, 4, 6, 4, 6, 5, 7, 6, 7,
	 6, 8, 6, 8, 7, 9, 8,10, 8,12, 9,13,10,15,10,15,
	11,14,
};

static static_codebook _huff_book_line_512x17_3sub2 = {
	1, 50,
	_huff_lengthlist_line_512x17_3sub2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_3sub3[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 8, 4, 8, 4, 8, 4, 8, 5, 8, 5, 8, 6, 8,
	 4, 8, 4, 8, 5, 8, 5, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
};

static static_codebook _huff_book_line_512x17_3sub3 = {
	1, 128,
	_huff_lengthlist_line_512x17_3sub3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_class1[] = {
	 1, 2, 3, 6, 5, 4, 7, 7,
};

static static_codebook _huff_book_line_512x17_class1 = {
	1, 8,
	_huff_lengthlist_line_512x17_class1,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_class2[] = {
	 3, 3, 3,14, 5, 4, 4,11, 8, 6, 6,10,17,12,11,17,
	 6, 5, 5,15, 5, 3, 4,11, 8, 5, 5, 8,16, 9,10,14,
	10, 8, 9,17, 8, 6, 6,13,10, 7, 7,10,16,11,13,14,
	17,17,17,17,17,16,16,16,16,15,16,16,16,16,16,16,
};

static static_codebook _huff_book_line_512x17_class2 = {
	1, 64,
	_huff_lengthlist_line_512x17_class2,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist_line_512x17_class3[] = {
	 2, 4, 6,17, 4, 5, 7,17, 8, 7,10,17,17,17,17,17,
	 3, 4, 6,15, 3, 3, 6,15, 7, 6, 9,17,17,17,17,17,
	 6, 8,10,17, 6, 6, 8,16, 9, 8,10,17,17,15,16,17,
	17,17,17,17,12,15,15,16,12,15,15,16,16,16,16,16,
};

static static_codebook _huff_book_line_512x17_class3 = {
	1, 64,
	_huff_lengthlist_line_512x17_class3,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static static_codebook *_floor_128x4_books[]={
  &_huff_book_line_128x4_class0,
  &_huff_book_line_128x4_0sub0,
  &_huff_book_line_128x4_0sub1,
  &_huff_book_line_128x4_0sub2,
  &_huff_book_line_128x4_0sub3,
};
static static_codebook *_floor_256x4_books[]={
  &_huff_book_line_256x4_class0,
  &_huff_book_line_256x4_0sub0,
  &_huff_book_line_256x4_0sub1,
  &_huff_book_line_256x4_0sub2,
  &_huff_book_line_256x4_0sub3,
};
static static_codebook *_floor_128x7_books[]={
  &_huff_book_line_128x7_class0,
  &_huff_book_line_128x7_class1,
  
  &_huff_book_line_128x7_0sub1,
  &_huff_book_line_128x7_0sub2,
  &_huff_book_line_128x7_0sub3,
  &_huff_book_line_128x7_1sub1,
  &_huff_book_line_128x7_1sub2,
  &_huff_book_line_128x7_1sub3, 
};
static static_codebook *_floor_256x7_books[]={
  &_huff_book_line_256x7_class0,
  &_huff_book_line_256x7_class1,
  
  &_huff_book_line_256x7_0sub1,
  &_huff_book_line_256x7_0sub2,
  &_huff_book_line_256x7_0sub3,
  &_huff_book_line_256x7_1sub1,
  &_huff_book_line_256x7_1sub2,
  &_huff_book_line_256x7_1sub3, 
};
static static_codebook *_floor_128x11_books[]={
  &_huff_book_line_128x11_class1,
  &_huff_book_line_128x11_class2,
  &_huff_book_line_128x11_class3,
  
  &_huff_book_line_128x11_0sub0,
  &_huff_book_line_128x11_1sub0,
  &_huff_book_line_128x11_1sub1,
  &_huff_book_line_128x11_2sub1,
  &_huff_book_line_128x11_2sub2,
  &_huff_book_line_128x11_2sub3, 
  &_huff_book_line_128x11_3sub1,
  &_huff_book_line_128x11_3sub2,
  &_huff_book_line_128x11_3sub3, 
};
static static_codebook *_floor_128x17_books[]={
  &_huff_book_line_128x17_class1,
  &_huff_book_line_128x17_class2,
  &_huff_book_line_128x17_class3,
  
  &_huff_book_line_128x17_0sub0,
  &_huff_book_line_128x17_1sub0,
  &_huff_book_line_128x17_1sub1,
  &_huff_book_line_128x17_2sub1,
  &_huff_book_line_128x17_2sub2,
  &_huff_book_line_128x17_2sub3, 
  &_huff_book_line_128x17_3sub1,
  &_huff_book_line_128x17_3sub2,
  &_huff_book_line_128x17_3sub3, 
};
static static_codebook *_floor_512x17_books[]={
  &_huff_book_line_512x17_class1,
  &_huff_book_line_512x17_class2,
  &_huff_book_line_512x17_class3,
  
  &_huff_book_line_512x17_0sub0,
  &_huff_book_line_512x17_1sub0,
  &_huff_book_line_512x17_1sub1,
  &_huff_book_line_512x17_2sub1,
  &_huff_book_line_512x17_2sub2,
  &_huff_book_line_512x17_2sub3, 
  &_huff_book_line_512x17_3sub1,
  &_huff_book_line_512x17_3sub2,
  &_huff_book_line_512x17_3sub3, 
};
static static_codebook *_floor_1024x27_books[]={
  &_huff_book_line_1024x27_class1,
  &_huff_book_line_1024x27_class2,
  &_huff_book_line_1024x27_class3,
  &_huff_book_line_1024x27_class4,
  
  &_huff_book_line_1024x27_0sub0,
  &_huff_book_line_1024x27_1sub0,
  &_huff_book_line_1024x27_1sub1,
  &_huff_book_line_1024x27_2sub0,
  &_huff_book_line_1024x27_2sub1,
  &_huff_book_line_1024x27_3sub1,
  &_huff_book_line_1024x27_3sub2,
  &_huff_book_line_1024x27_3sub3,
  &_huff_book_line_1024x27_4sub1,
  &_huff_book_line_1024x27_4sub2,
  &_huff_book_line_1024x27_4sub3,
};

static static_codebook **_floor_books[10]={
  _floor_128x4_books,
  _floor_256x4_books,
  _floor_128x7_books,
  _floor_256x7_books,
  _floor_128x11_books,
  _floor_128x17_books,
  _floor_128x17_books,
  _floor_1024x27_books,
  _floor_1024x27_books,
  _floor_512x17_books,
};

static vorbis_info_floor1 _floor[10]={
  /* 128 x 4 */
  {
    1,{0},{4},{2},{0},
    {{1,2,3,4}},
    4,{0,128, 33,8,16,70},

    60,30,500,   1.,18.,  -1
  },
  /* 256 x 4 */
  {
    1,{0},{4},{2},{0},
    {{1,2,3,4}},
    4,{0,256, 66,16,32,140},

    60,30,500,   1.,18.,  -1
  },
  /* 128 x 7 */
  {
    2,{0,1},{3,4},{2,2},{0,1},
    {{-1,2,3,4},{-1,5,6,7}},
    4,{0,128, 14,4,58, 2,8,28,90},
    
    60,30,500,   1.,18.,  -1
  },
  /* 256 x 7 */
  {
    2,{0,1},{3,4},{2,2},{0,1},
    {{-1,2,3,4},{-1,5,6,7}},
    4,{0,256, 28,8,116, 4,16,56,180},
    
    60,30,500,   1.,18.,  -1
  },

  /* 128 x 11 */
  {
    4,{0,1,2,3},{2,3,3,3},{0,1,2,2},{-1,0,1,2},
    {{3},{4,5},{-1,6,7,8},{-1,9,10,11}},
    
    2,{0,128,  8,33,  4,16,70,  2,6,12,  23,46,90},
    
     60,30,500,   1,18.,  -1
  },
    
  /* 128 x 17 */
  {
    6,{0,1,1,2,3,3},{2,3,3,3},{0,1,2,2},{-1,0,1,2},
    {{3},{4,5},{-1,6,7,8},{-1,9,10,11}},
    2,{0,128,  12,46,  4,8,16,  23,33,70,  2,6,10,  14,19,28,  39,58,90},

    60,30,500,    1,18.,  -1 
  },
  
  /* 1024 x 17 */
  {
    6,{0,1,1,2,3,3},{2,3,3,3},{0,1,2,2},{-1,0,1,2},
    {{3},{4,5},{-1,6,7,8},{-1,9,10,11}},
    2,{0,1024,  93,372,  33,65,130,  186,260,556,  
       14,46,79,  111,158,220,  312,464,720},
    
    60,30,500,    1,18.,  -1 /* lowpass! */
  },
  /* 1024 x 27 */
  {
    8,{0,1,2,2,3,3,4,4},{3,4,3,4,3},{0,1,1,2,2},{-1,0,1,2,3},
    {{4},{5,6},{7,8},{-1,9,10,11},{-1,12,13,14}},
    2,{0,1024,   93,23,372, 6,46,186,750,  14,33,65, 130,260,556,
       3,10,18,28,  39,55,79,111,  158,220,312,  464,650,850},
    
    60,30,500,    3,18.,  -1 /* lowpass */
  },
  /* 2048 x 27 */
  {
    8,{0,1,2,2,3,3,4,4},{3,4,3,4,3},{0,1,1,2,2},{-1,0,1,2,3},
    {{4},{5,6},{7,8},{-1,9,10,11},{-1,12,13,14}},
    2,{0,2048,   186,46,744, 12,92,372,1500,  28,66,130, 260,520,1112,
       6,20,36,56,  78,110,158,222,  316,440,624,  928,1300,1700},
    
    60,30,500,    3,18.,  -1 /* lowpass */
  },
  /* 512 x 17 */
  {
    6,{0,1,1,2,3,3},{2,3,3,3},{0,1,2,2},{-1,0,1,2},
    {{3},{4,5},{-1,6,7,8},{-1,9,10,11}},
    2,{0,512,  46,186,  16,33,65,  93,130,278,  
       7,23,39,  55,79,110,  156,232,360},
    
    60,30,500,    1,18.,  -1 /* lowpass! */
  },

};

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: toplevel residue templates for 32/44.1/48kHz

 ********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: static codebooks autogenerated by huff/huffbuld

 ********************************************************************/

static long _vq_quantlist__16c0_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c0_s_p1_0[] = {
	 1, 4, 4, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 8, 9,10, 0, 0, 0,
	 0, 0, 0, 7, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 8, 0, 0, 0, 0,
	 0, 0, 8,10,10, 0, 0, 0, 0, 0, 0, 8,10,10, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7,10,10, 0, 0, 0,
	 0, 0, 0, 9, 9,12, 0, 0, 0, 0, 0, 0,10,12,11, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7,10,10, 0, 0,
	 0, 0, 0, 0, 9,12,10, 0, 0, 0, 0, 0, 0,10,11,12,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 8,10,10, 0, 0,
	 0, 0, 0, 0, 8,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7,10,10, 0, 0, 0, 0, 0, 0,10,12,11, 0,
	 0, 0, 0, 0, 0, 9,10,12, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7,10,10, 0, 0, 0, 0, 0, 0,10,11,12,
	 0, 0, 0, 0, 0, 0, 9,12, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c0_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16c0_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p1_0 = {
	_vq_quantthresh__16c0_s_p1_0,
	_vq_quantmap__16c0_s_p1_0,
	3,
	3
};

static static_codebook _16c0_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__16c0_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16c0_s_p1_0,
	NULL,
	&_vq_auxt__16c0_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c0_s_p2_0[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c0_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c0_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p2_0 = {
	_vq_quantthresh__16c0_s_p2_0,
	_vq_quantmap__16c0_s_p2_0,
	5,
	5
};

static static_codebook _16c0_s_p2_0 = {
	4, 625,
	_vq_lengthlist__16c0_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c0_s_p2_0,
	NULL,
	&_vq_auxt__16c0_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c0_s_p3_0[] = {
	 1, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 6, 6, 7, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 6, 6, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 9, 9,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 9, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c0_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c0_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p3_0 = {
	_vq_quantthresh__16c0_s_p3_0,
	_vq_quantmap__16c0_s_p3_0,
	5,
	5
};

static static_codebook _16c0_s_p3_0 = {
	4, 625,
	_vq_lengthlist__16c0_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c0_s_p3_0,
	NULL,
	&_vq_auxt__16c0_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16c0_s_p4_0[] = {
	 1, 3, 2, 7, 8, 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c0_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16c0_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p4_0 = {
	_vq_quantthresh__16c0_s_p4_0,
	_vq_quantmap__16c0_s_p4_0,
	9,
	9
};

static static_codebook _16c0_s_p4_0 = {
	2, 81,
	_vq_lengthlist__16c0_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16c0_s_p4_0,
	NULL,
	&_vq_auxt__16c0_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16c0_s_p5_0[] = {
	 1, 3, 3, 6, 6, 6, 6, 8, 8, 0, 0, 0, 7, 7, 7, 7,
	 8, 8, 0, 0, 0, 7, 7, 7, 7, 8, 8, 0, 0, 0, 7, 7,
	 8, 8, 9, 9, 0, 0, 0, 7, 7, 8, 8, 9, 9, 0, 0, 0,
	 8, 9, 8, 8,10,10, 0, 0, 0, 8, 8, 8, 8,10,10, 0,
	 0, 0,10,10, 9, 9,10,10, 0, 0, 0, 0, 0, 9, 9,10,
	10,
};

static float _vq_quantthresh__16c0_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16c0_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p5_0 = {
	_vq_quantthresh__16c0_s_p5_0,
	_vq_quantmap__16c0_s_p5_0,
	9,
	9
};

static static_codebook _16c0_s_p5_0 = {
	2, 81,
	_vq_lengthlist__16c0_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16c0_s_p5_0,
	NULL,
	&_vq_auxt__16c0_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__16c0_s_p6_0[] = {
	 1, 3, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10,11,
	11,11, 0, 0, 0, 6, 6, 8, 8, 9, 9, 9, 9,10,10,11,
	11,11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,10,
	11,11,12,12,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,
	10,11,11,12,12,12,13, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,10,10,10,
	10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,
	10,10,11,11,12,12,13,13,13,13, 0, 0, 0, 0, 0, 9,
	 9,10,10,11,11,12,12,13,13,13,14, 0, 0, 0, 0, 0,
	10,10,10,11,11,11,12,12,13,13,13,14, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,12,12,13,13,14,14, 0, 0, 0,
	 0, 0, 0, 0,11,11,12,12,13,13,13,13,14,14, 0, 0,
	 0, 0, 0, 0, 0,11,11,12,12,12,13,13,14,15,14, 0,
	 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,13,14,14,15,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,13,13,14,13,14,
	14,
};

static float _vq_quantthresh__16c0_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__16c0_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p6_0 = {
	_vq_quantthresh__16c0_s_p6_0,
	_vq_quantmap__16c0_s_p6_0,
	17,
	17
};

static static_codebook _16c0_s_p6_0 = {
	2, 289,
	_vq_lengthlist__16c0_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__16c0_s_p6_0,
	NULL,
	&_vq_auxt__16c0_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c0_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,11,10,10,11,
	11,10, 4, 7, 7,10,10,10,11,10,10, 6,10,10,11,11,
	11,11,11,10, 6, 9, 9,11,12,12,11, 9, 9, 6, 9,10,
	11,12,12,11, 9,10, 7,11,11,11,11,11,12,13,12, 6,
	 9,10,11,10,10,12,13,13, 6,10, 9,11,10,10,11,12,
	13,
};

static float _vq_quantthresh__16c0_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__16c0_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p7_0 = {
	_vq_quantthresh__16c0_s_p7_0,
	_vq_quantmap__16c0_s_p7_0,
	3,
	3
};

static static_codebook _16c0_s_p7_0 = {
	4, 81,
	_vq_lengthlist__16c0_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__16c0_s_p7_0,
	NULL,
	&_vq_auxt__16c0_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16c0_s_p7_1[] = {
	 1, 3, 4, 6, 6, 7, 7, 8, 8, 8, 8,10,10,10, 7, 7,
	 8, 8, 8, 9, 9, 9,10,10,10, 6, 7, 8, 8, 8, 8, 9,
	 8,10,10,10, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10, 7,
	 7, 8, 8, 9, 9, 8, 9,10,10,10, 8, 8, 9, 9, 9, 9,
	 9, 9,11,11,11, 8, 8, 9, 9, 9, 9, 9,10,10,11,11,
	 9, 9, 9, 9, 9, 9, 9,10,11,11,11,10,11, 9, 9, 9,
	 9,10, 9,11,11,11,10,11,10,10, 9, 9,10,10,11,11,
	11,11,11, 9, 9, 9, 9,10,10,
};

static float _vq_quantthresh__16c0_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16c0_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p7_1 = {
	_vq_quantthresh__16c0_s_p7_1,
	_vq_quantmap__16c0_s_p7_1,
	11,
	11
};

static static_codebook _16c0_s_p7_1 = {
	2, 121,
	_vq_lengthlist__16c0_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16c0_s_p7_1,
	NULL,
	&_vq_auxt__16c0_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16c0_s_p8_0[] = {
	 1, 4, 4, 7, 7, 7, 7, 7, 6, 8, 8,10,10, 6, 5, 6,
	 8, 8, 8, 8, 8, 8, 8, 9,10,10, 7, 6, 6, 8, 8, 8,
	 8, 8, 8, 8, 8,10,10, 0, 8, 8, 8, 8, 9, 8, 9, 9,
	 9,10,10,10, 0, 9, 8, 8, 8, 9, 9, 8, 8, 9, 9,10,
	10, 0,12,11, 8, 8, 9, 9, 9, 9,10,10,11,10, 0,12,
	13, 8, 8, 9,10, 9, 9,11,11,11,12, 0, 0, 0, 8, 8,
	 8, 8,10, 9,12,13,12,14, 0, 0, 0, 8, 8, 8, 9,10,
	10,12,12,13,14, 0, 0, 0,13,13, 9, 9,11,11, 0, 0,
	14, 0, 0, 0, 0,14,14,10,10,12,11,12,14,14,14, 0,
	 0, 0, 0, 0,11,11,13,13,14,13,14,14, 0, 0, 0, 0,
	 0,12,13,13,12,13,14,14,14,
};

static float _vq_quantthresh__16c0_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__16c0_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p8_0 = {
	_vq_quantthresh__16c0_s_p8_0,
	_vq_quantmap__16c0_s_p8_0,
	13,
	13
};

static static_codebook _16c0_s_p8_0 = {
	2, 169,
	_vq_lengthlist__16c0_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__16c0_s_p8_0,
	NULL,
	&_vq_auxt__16c0_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c0_s_p8_1[] = {
	 1, 4, 3, 5, 5, 7, 7, 7, 6, 6, 7, 7, 7, 5, 5, 7,
	 7, 7, 6, 6, 7, 7, 7, 6, 6,
};

static float _vq_quantthresh__16c0_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c0_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p8_1 = {
	_vq_quantthresh__16c0_s_p8_1,
	_vq_quantmap__16c0_s_p8_1,
	5,
	5
};

static static_codebook _16c0_s_p8_1 = {
	2, 25,
	_vq_lengthlist__16c0_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c0_s_p8_1,
	NULL,
	&_vq_auxt__16c0_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p9_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c0_s_p9_0[] = {
	 1, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__16c0_s_p9_0[] = {
	-157.5, 157.5, 
};

static long _vq_quantmap__16c0_s_p9_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p9_0 = {
	_vq_quantthresh__16c0_s_p9_0,
	_vq_quantmap__16c0_s_p9_0,
	3,
	3
};

static static_codebook _16c0_s_p9_0 = {
	4, 81,
	_vq_lengthlist__16c0_s_p9_0,
	1, -518803456, 1628680192, 2, 0,
	_vq_quantlist__16c0_s_p9_0,
	NULL,
	&_vq_auxt__16c0_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16c0_s_p9_1[] = {
	 1, 5, 5, 5, 5, 9,11,11,10,10,10,10,10,10,10, 7,
	 6, 6, 6, 6,10,10,10,10,10,10,10,10,10,10, 7, 6,
	 6, 6, 6,10, 9,10,10,10,10,10,10,10,10,10, 7, 7,
	 8, 9,10,10,10,10,10,10,10,10,10,10,10, 8, 7,10,
	10,10, 9,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__16c0_s_p9_1[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__16c0_s_p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p9_1 = {
	_vq_quantthresh__16c0_s_p9_1,
	_vq_quantmap__16c0_s_p9_1,
	15,
	15
};

static static_codebook _16c0_s_p9_1 = {
	2, 225,
	_vq_lengthlist__16c0_s_p9_1,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__16c0_s_p9_1,
	NULL,
	&_vq_auxt__16c0_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__16c0_s_p9_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__16c0_s_p9_2[] = {
	 1, 5, 5, 7, 8, 8, 7, 9, 9, 9,12,12,11,12,12,10,
	10,11,12,12,12,11,12,12, 8, 9, 8, 7, 9,10,10,11,
	11,10,11,12,10,12,10,12,12,12,11,12,11, 9, 8, 8,
	 9,10, 9, 8, 9,10,12,12,11,11,12,11,10,11,12,11,
	12,12, 8, 9, 9, 9,10,11,12,11,12,11,11,11,11,12,
	12,11,11,12,12,11,11, 9, 9, 8, 9, 9,11, 9, 9,10,
	 9,11,11,11,11,12,11,11,10,12,12,12, 9,12,11,10,
	11,11,11,11,12,12,12,11,11,11,12,10,12,12,12,10,
	10, 9,10, 9,10,10, 9, 9, 9,10,10,12,10,11,11, 9,
	11,11,10,11,11,11,10,10,10, 9, 9,10,10, 9, 9,10,
	11,11,10,11,10,11,10,11,11,10,11,11,11,10, 9,10,
	10, 9,10, 9, 9,11, 9, 9,11,10,10,11,11,10,10,11,
	10,11, 8, 9,11,11,10, 9,10,11,11,10,11,11,10,10,
	10,11,10, 9,10,10,11, 9,10,10, 9,11,10,10,10,10,
	11,10,11,11, 9,11,10,11,10,10,11,11,10,10,10, 9,
	10,10,11,11,11, 9,10,10,10,10,10,11,10,10,10, 9,
	10,10,11,10,10,10,10,10, 9,10,11,10,10,10,10,11,
	11,11,10,10,10,10,10,11,10,11,10,11,10,10,10, 9,
	11,11,10,10,10,11,11,10,10,10,10,10,10,10,10,11,
	11, 9,10,10,10,11,10,11,10,10,10,11, 9,10,11,10,
	11,10,10, 9,10,10,10,11,10,11,10,10,10,10,10,11,
	11,10,11,11,10,10,11,11,10, 9, 9,10,10,10,10,10,
	 9,11, 9,10,10,10,11,11,10,10,10,10,11,11,11,10,
	 9, 9,10,10,11,10,10,10,10,10,11,11,11,10,10,10,
	11,11,11, 9,10,10,10,10, 9,10, 9,10,11,10,11,10,
	10,11,11,10,11,11,11,11,11,10,11,10,10,10, 9,11,
	11,10,11,11,11,11,11,11,11,11,11,10,11,10,10,10,
	10,11,10,10,11, 9,10,10,10,
};

static float _vq_quantthresh__16c0_s_p9_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__16c0_s_p9_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__16c0_s_p9_2 = {
	_vq_quantthresh__16c0_s_p9_2,
	_vq_quantmap__16c0_s_p9_2,
	21,
	21
};

static static_codebook _16c0_s_p9_2 = {
	2, 441,
	_vq_lengthlist__16c0_s_p9_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__16c0_s_p9_2,
	NULL,
	&_vq_auxt__16c0_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__16c0_s_single[] = {
	 3, 4,19, 7, 9, 7, 8,11, 9,12, 4, 1,19, 6, 7, 7,
	 8,10,11,13,18,18,18,18,18,18,18,18,18,18, 8, 6,
	18, 8, 9, 9,11,12,14,18, 9, 6,18, 9, 7, 8, 9,11,
	12,18, 7, 6,18, 8, 7, 7, 7, 9,11,17, 8, 8,18, 9,
	 7, 6, 6, 8,11,17,10,10,18,12, 9, 8, 7, 9,12,18,
	13,15,18,15,13,11,10,11,15,18,14,18,18,18,18,18,
	16,16,18,18,
};

static static_codebook _huff_book__16c0_s_single = {
	2, 100,
	_huff_lengthlist__16c0_s_single,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__16c1_s_long[] = {
	 2, 5,20, 7,10, 7, 8,10,11,11, 4, 2,20, 5, 8, 6,
	 7, 9,10,10,20,20,20,20,19,19,19,19,19,19, 7, 5,
	19, 6,10, 7, 9,11,13,17,11, 8,19,10, 7, 7, 8,10,
	11,15, 7, 5,19, 7, 7, 5, 6, 9,11,16, 7, 6,19, 8,
	 7, 6, 6, 7, 9,13, 9, 9,19,11, 9, 8, 6, 7, 8,13,
	12,14,19,16,13,10, 9, 8, 9,13,14,17,19,18,18,17,
	12,11,11,13,
};

static static_codebook _huff_book__16c1_s_long = {
	2, 100,
	_huff_lengthlist__16c1_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c1_s_p1_0[] = {
	 1, 5, 5, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 7, 0, 0, 0, 0,
	 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 9, 9,11, 0, 0, 0, 0, 0, 0, 9,11,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8,11, 9, 0, 0, 0, 0, 0, 0, 9,10,11,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,11,10, 0,
	 0, 0, 0, 0, 0, 8, 9,11, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,11,
	 0, 0, 0, 0, 0, 0, 9,11, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c1_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16c1_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p1_0 = {
	_vq_quantthresh__16c1_s_p1_0,
	_vq_quantmap__16c1_s_p1_0,
	3,
	3
};

static static_codebook _16c1_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__16c1_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16c1_s_p1_0,
	NULL,
	&_vq_auxt__16c1_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c1_s_p2_0[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c1_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c1_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p2_0 = {
	_vq_quantthresh__16c1_s_p2_0,
	_vq_quantmap__16c1_s_p2_0,
	5,
	5
};

static static_codebook _16c1_s_p2_0 = {
	4, 625,
	_vq_lengthlist__16c1_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c1_s_p2_0,
	NULL,
	&_vq_auxt__16c1_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c1_s_p3_0[] = {
	 1, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 5, 7, 7, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 5, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 7, 7, 9, 9,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 7, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c1_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c1_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p3_0 = {
	_vq_quantthresh__16c1_s_p3_0,
	_vq_quantmap__16c1_s_p3_0,
	5,
	5
};

static static_codebook _16c1_s_p3_0 = {
	4, 625,
	_vq_lengthlist__16c1_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c1_s_p3_0,
	NULL,
	&_vq_auxt__16c1_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16c1_s_p4_0[] = {
	 1, 2, 3, 7, 7, 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 8, 9, 0, 0, 0, 0, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c1_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16c1_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p4_0 = {
	_vq_quantthresh__16c1_s_p4_0,
	_vq_quantmap__16c1_s_p4_0,
	9,
	9
};

static static_codebook _16c1_s_p4_0 = {
	2, 81,
	_vq_lengthlist__16c1_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16c1_s_p4_0,
	NULL,
	&_vq_auxt__16c1_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16c1_s_p5_0[] = {
	 1, 3, 3, 5, 5, 6, 6, 8, 8, 0, 0, 0, 7, 7, 7, 7,
	 9, 9, 0, 0, 0, 7, 7, 7, 7, 9, 9, 0, 0, 0, 8, 8,
	 8, 8, 9, 9, 0, 0, 0, 8, 8, 8, 8,10,10, 0, 0, 0,
	 9, 9, 8, 8,10,10, 0, 0, 0, 9, 9, 8, 8,10,10, 0,
	 0, 0,10,10, 9, 9,10,10, 0, 0, 0, 0, 0, 9, 9,10,
	10,
};

static float _vq_quantthresh__16c1_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16c1_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p5_0 = {
	_vq_quantthresh__16c1_s_p5_0,
	_vq_quantmap__16c1_s_p5_0,
	9,
	9
};

static static_codebook _16c1_s_p5_0 = {
	2, 81,
	_vq_lengthlist__16c1_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16c1_s_p5_0,
	NULL,
	&_vq_auxt__16c1_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__16c1_s_p6_0[] = {
	 1, 3, 3, 6, 6, 8, 8, 9, 9, 9, 9,10,10,11,11,12,
	12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,12, 0, 0, 0, 8, 8, 8, 9,10, 9,10,10,10,10,
	11,11,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,10,11,
	11,11,12,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,10,
	11,11,12,12,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,
	10,11,11,12,12,13,13, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,
	10,10,11,11,12,12,12,12,13,13, 0, 0, 0, 0, 0, 9,
	 9,10,10,11,11,12,12,12,12,13,13, 0, 0, 0, 0, 0,
	10,10,11,10,11,11,12,12,13,13,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,12,12,13,13,13,13, 0, 0, 0,
	 0, 0, 0, 0,11,11,12,12,12,12,13,13,14,14, 0, 0,
	 0, 0, 0, 0, 0,11,11,12,12,12,12,13,13,14,14, 0,
	 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,13,13,14,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,13,13,13,13,14,
	14,
};

static float _vq_quantthresh__16c1_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__16c1_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p6_0 = {
	_vq_quantthresh__16c1_s_p6_0,
	_vq_quantmap__16c1_s_p6_0,
	17,
	17
};

static static_codebook _16c1_s_p6_0 = {
	2, 289,
	_vq_lengthlist__16c1_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__16c1_s_p6_0,
	NULL,
	&_vq_auxt__16c1_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c1_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9,10,10,
	10, 9, 4, 7, 7,10,10,10,11,10,10, 6,10,10,11,11,
	11,11,10,10, 6,10, 9,11,11,11,11,10,10, 6,10,10,
	11,11,11,11,10,10, 7,11,11,11,11,11,12,12,11, 6,
	10,10,11,10,10,11,11,11, 6,10,10,10,11,10,11,11,
	11,
};

static float _vq_quantthresh__16c1_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__16c1_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p7_0 = {
	_vq_quantthresh__16c1_s_p7_0,
	_vq_quantmap__16c1_s_p7_0,
	3,
	3
};

static static_codebook _16c1_s_p7_0 = {
	4, 81,
	_vq_lengthlist__16c1_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__16c1_s_p7_0,
	NULL,
	&_vq_auxt__16c1_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16c1_s_p7_1[] = {
	 2, 3, 3, 5, 6, 7, 7, 7, 7, 8, 8,10,10,10, 6, 6,
	 7, 7, 8, 8, 8, 8,10,10,10, 6, 6, 7, 7, 8, 8, 8,
	 8,10,10,10, 7, 7, 7, 7, 8, 8, 8, 8,10,10,10, 7,
	 7, 7, 7, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 9, 9,10,10,10,10,10, 8, 8, 8,
	 8, 9, 9,10,10,10,10,10, 9, 9, 8, 8, 9, 9,10,10,
	10,10,10, 8, 8, 8, 8, 9, 9,
};

static float _vq_quantthresh__16c1_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16c1_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p7_1 = {
	_vq_quantthresh__16c1_s_p7_1,
	_vq_quantmap__16c1_s_p7_1,
	11,
	11
};

static static_codebook _16c1_s_p7_1 = {
	2, 121,
	_vq_lengthlist__16c1_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16c1_s_p7_1,
	NULL,
	&_vq_auxt__16c1_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16c1_s_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 7, 7, 8, 8, 9, 9, 6, 5, 5,
	 7, 8, 8, 9, 8, 8, 9, 9,10,11, 6, 5, 5, 8, 8, 9,
	 9, 8, 8, 9,10,10,11, 0, 8, 8, 8, 9, 9, 9, 9, 9,
	10,10,11,11, 0, 9, 9, 9, 8, 9, 9, 9, 9,10,10,11,
	11, 0,13,13, 9, 9,10,10,10,10,11,11,12,12, 0,14,
	13, 9, 9,10,10,10,10,11,11,12,12, 0, 0, 0,10,10,
	 9, 9,11,11,12,12,13,12, 0, 0, 0,10,10, 9, 9,10,
	10,12,12,13,13, 0, 0, 0,13,14,11,10,11,11,12,12,
	13,14, 0, 0, 0,14,14,10,10,11,11,12,12,13,13, 0,
	 0, 0, 0, 0,12,12,12,12,13,13,14,15, 0, 0, 0, 0,
	 0,12,12,12,12,13,13,14,15,
};

static float _vq_quantthresh__16c1_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__16c1_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p8_0 = {
	_vq_quantthresh__16c1_s_p8_0,
	_vq_quantmap__16c1_s_p8_0,
	13,
	13
};

static static_codebook _16c1_s_p8_0 = {
	2, 169,
	_vq_lengthlist__16c1_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__16c1_s_p8_0,
	NULL,
	&_vq_auxt__16c1_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c1_s_p8_1[] = {
	 2, 3, 3, 5, 5, 6, 6, 6, 5, 5, 6, 6, 6, 5, 5, 6,
	 6, 6, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__16c1_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c1_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p8_1 = {
	_vq_quantthresh__16c1_s_p8_1,
	_vq_quantmap__16c1_s_p8_1,
	5,
	5
};

static static_codebook _16c1_s_p8_1 = {
	2, 25,
	_vq_lengthlist__16c1_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c1_s_p8_1,
	NULL,
	&_vq_auxt__16c1_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16c1_s_p9_0[] = {
	 1, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__16c1_s_p9_0[] = {
	-1732.5, -1417.5, -1102.5, -787.5, -472.5, -157.5, 157.5, 472.5, 
	787.5, 1102.5, 1417.5, 1732.5, 
};

static long _vq_quantmap__16c1_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p9_0 = {
	_vq_quantthresh__16c1_s_p9_0,
	_vq_quantmap__16c1_s_p9_0,
	13,
	13
};

static static_codebook _16c1_s_p9_0 = {
	2, 169,
	_vq_lengthlist__16c1_s_p9_0,
	1, -513964032, 1628680192, 4, 0,
	_vq_quantlist__16c1_s_p9_0,
	NULL,
	&_vq_auxt__16c1_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16c1_s_p9_1[] = {
	 1, 4, 4, 4, 4, 8, 8,12,13,14,14,14,14,14,14, 6,
	 6, 6, 6, 6,10, 9,14,14,14,14,14,14,14,14, 7, 6,
	 5, 6, 6,10, 9,12,13,13,13,13,13,13,13,13, 7, 7,
	 9, 9,11,11,12,13,13,13,13,13,13,13,13, 7, 7, 8,
	 8,11,12,13,13,13,13,13,13,13,13,13,12,12,10,10,
	13,12,13,13,13,13,13,13,13,13,13,12,12,10,10,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,12,13,12,
	13,13,13,13,13,13,13,13,13,13,13,13,12,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,12,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
	13,13,13,13,13,13,13,13,13,12,13,13,13,13,13,13,
	13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,
	13,
};

static float _vq_quantthresh__16c1_s_p9_1[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__16c1_s_p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p9_1 = {
	_vq_quantthresh__16c1_s_p9_1,
	_vq_quantmap__16c1_s_p9_1,
	15,
	15
};

static static_codebook _16c1_s_p9_1 = {
	2, 225,
	_vq_lengthlist__16c1_s_p9_1,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__16c1_s_p9_1,
	NULL,
	&_vq_auxt__16c1_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__16c1_s_p9_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__16c1_s_p9_2[] = {
	 1, 4, 4, 6, 6, 7, 7, 8, 7, 8, 8, 9, 9, 9, 9,10,
	10,10, 9,10,10,11,12,12, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,10,10,10,11,11,10,12,11,11,13,11, 7, 7, 8,
	 8, 8, 8, 9, 9, 9,10,10,10,10, 9,10,10,11,11,12,
	11,11, 8, 8, 8, 8, 9, 9,10,10,10,10,11,11,11,11,
	11,11,11,12,11,12,12, 8, 8, 9, 9, 9, 9, 9,10,10,
	10,10,10,10,11,11,11,11,11,11,12,11, 9, 9, 9, 9,
	10,10,10,10,11,10,11,11,11,11,11,11,12,12,12,12,
	11, 9, 9, 9, 9,10,10,10,10,11,11,11,11,11,11,11,
	11,11,12,12,12,13, 9,10,10, 9,11,10,10,10,10,11,
	11,11,11,11,10,11,12,11,12,12,11,12,11,10, 9,10,
	10,11,10,11,11,11,11,11,11,11,11,11,12,12,11,12,
	12,12,10,10,10,11,10,11,11,11,11,11,11,11,11,11,
	11,11,12,13,12,12,11, 9,10,10,11,11,10,11,11,11,
	12,11,11,11,11,11,12,12,13,13,12,13,10,10,12,10,
	11,11,11,11,11,11,11,11,11,12,12,11,13,12,12,12,
	12,13,12,11,11,11,11,11,11,12,11,12,11,11,11,11,
	12,12,13,12,11,12,12,11,11,11,11,11,12,11,11,11,
	11,12,11,11,12,11,12,13,13,12,12,12,12,11,11,11,
	11,11,12,11,11,12,11,12,11,11,11,11,13,12,12,12,
	12,13,11,11,11,12,12,11,11,11,12,11,12,12,12,11,
	12,13,12,11,11,12,12,11,12,11,11,11,12,12,11,12,
	11,11,11,12,12,12,12,13,12,13,12,12,12,12,11,11,
	12,11,11,11,11,11,11,12,12,12,13,12,11,13,13,12,
	12,11,12,10,11,11,11,11,12,11,12,12,11,12,12,13,
	12,12,13,12,12,12,12,12,11,12,12,12,11,12,11,11,
	11,12,13,12,13,13,13,13,13,12,13,13,12,12,13,11,
	11,11,11,11,12,11,11,12,11,
};

static float _vq_quantthresh__16c1_s_p9_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__16c1_s_p9_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__16c1_s_p9_2 = {
	_vq_quantthresh__16c1_s_p9_2,
	_vq_quantmap__16c1_s_p9_2,
	21,
	21
};

static static_codebook _16c1_s_p9_2 = {
	2, 441,
	_vq_lengthlist__16c1_s_p9_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__16c1_s_p9_2,
	NULL,
	&_vq_auxt__16c1_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__16c1_s_short[] = {
	 5, 6,17, 8,12, 9,10,10,12,13, 5, 2,17, 4, 9, 5,
	 7, 8,11,13,16,16,16,16,16,16,16,16,16,16, 6, 4,
	16, 5,10, 5, 7,10,14,16,13, 9,16,11, 8, 7, 8, 9,
	13,16, 7, 4,16, 5, 7, 4, 6, 8,11,13, 8, 6,16, 7,
	 8, 5, 5, 7, 9,13, 9, 8,16, 9, 8, 6, 6, 7, 9,13,
	11,11,16,10,10, 7, 7, 7, 9,13,13,13,16,13,13, 9,
	 9, 9,10,13,
};

static static_codebook _huff_book__16c1_s_short = {
	2, 100,
	_huff_lengthlist__16c1_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__16c2_s_long[] = {
	 4, 7, 9, 9, 9, 8, 9,10,15,19, 5, 4, 5, 6, 7, 7,
	 8, 9,14,16, 6, 5, 4, 5, 6, 7, 8,10,12,19, 7, 6,
	 5, 4, 5, 6, 7, 9,11,18, 8, 7, 6, 5, 5, 5, 7, 9,
	10,17, 8, 7, 7, 5, 5, 5, 6, 7,12,18, 8, 8, 8, 7,
	 7, 5, 5, 7,12,18, 8, 9,10, 9, 9, 7, 6, 7,12,17,
	14,18,16,16,15,12,11,10,12,18,15,17,18,18,18,15,
	14,14,16,18,
};

static static_codebook _huff_book__16c2_s_long = {
	2, 100,
	_huff_lengthlist__16c2_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c2_s_p1_0[] = {
	 1, 3, 3, 0, 0, 0, 0, 0, 0, 4, 5, 5, 0, 0, 0, 0,
	 0, 0, 4, 5, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c2_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16c2_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p1_0 = {
	_vq_quantthresh__16c2_s_p1_0,
	_vq_quantmap__16c2_s_p1_0,
	3,
	3
};

static static_codebook _16c2_s_p1_0 = {
	4, 81,
	_vq_lengthlist__16c2_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16c2_s_p1_0,
	NULL,
	&_vq_auxt__16c2_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c2_s_p2_0[] = {
	 2, 4, 3, 7, 7, 0, 0, 0, 7, 8, 0, 0, 0, 8, 8, 0,
	 0, 0, 8, 8, 0, 0, 0, 8, 8, 4, 5, 4, 8, 8, 0, 0,
	 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 9, 9, 0, 0, 0,
	 9, 9, 4, 4, 5, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 8,
	 8, 0, 0, 0, 9, 9, 0, 0, 0, 9, 9, 7, 8, 8,10,10,
	 0, 0, 0,12,11, 0, 0, 0,11,11, 0, 0, 0,14,13, 0,
	 0, 0,14,13, 7, 8, 8, 9,10, 0, 0, 0,11,12, 0, 0,
	 0,11,11, 0, 0, 0,14,14, 0, 0, 0,13,14, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 8, 8, 8,11,11, 0, 0, 0,
	11,11, 0, 0, 0,12,11, 0, 0, 0,12,12, 0, 0, 0,13,
	13, 8, 8, 8,11,11, 0, 0, 0,11,11, 0, 0, 0,11,12,
	 0, 0, 0,12,13, 0, 0, 0,13,13, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 8, 8, 8,12,11, 0, 0, 0,12,11, 0,
	 0, 0,11,11, 0, 0, 0,13,13, 0, 0, 0,13,12, 8, 8,
	 8,11,12, 0, 0, 0,11,12, 0, 0, 0,11,11, 0, 0, 0,
	13,13, 0, 0, 0,12,13, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 8, 9, 9,14,13, 0, 0, 0,13,12, 0, 0, 0,13,
	13, 0, 0, 0,13,12, 0, 0, 0,13,13, 8, 9, 9,13,14,
	 0, 0, 0,12,13, 0, 0, 0,13,13, 0, 0, 0,12,13, 0,
	 0, 0,13,13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8,
	 9, 9,14,13, 0, 0, 0,13,13, 0, 0, 0,13,12, 0, 0,
	 0,13,13, 0, 0, 0,13,12, 8, 9, 9,14,14, 0, 0, 0,
	13,13, 0, 0, 0,12,13, 0, 0, 0,13,13, 0, 0, 0,12,
	13,
};

static float _vq_quantthresh__16c2_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c2_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p2_0 = {
	_vq_quantthresh__16c2_s_p2_0,
	_vq_quantmap__16c2_s_p2_0,
	5,
	5
};

static static_codebook _16c2_s_p2_0 = {
	4, 625,
	_vq_lengthlist__16c2_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c2_s_p2_0,
	NULL,
	&_vq_auxt__16c2_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p3_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16c2_s_p3_0[] = {
	 1, 3, 3, 6, 6, 7, 7, 8, 8, 0, 0, 0, 6, 6, 7, 7,
	 9, 9, 0, 0, 0, 6, 6, 7, 7, 9, 9, 0, 0, 0, 7, 7,
	 8, 8,10,10, 0, 0, 0, 7, 7, 8, 8,10,10, 0, 0, 0,
	 7, 7, 9, 9,10,10, 0, 0, 0, 7, 7, 9, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c2_s_p3_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16c2_s_p3_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p3_0 = {
	_vq_quantthresh__16c2_s_p3_0,
	_vq_quantmap__16c2_s_p3_0,
	9,
	9
};

static static_codebook _16c2_s_p3_0 = {
	2, 81,
	_vq_lengthlist__16c2_s_p3_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16c2_s_p3_0,
	NULL,
	&_vq_auxt__16c2_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p4_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__16c2_s_p4_0[] = {
	 2, 3, 3, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 9, 9,10,
	10, 0, 0, 0, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,
	11,11, 0, 0, 0, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,
	10,10,11, 0, 0, 0, 6, 6, 8, 8, 8, 8, 9, 9,10,10,
	10,11,11,11, 0, 0, 0, 6, 6, 8, 8, 9, 9, 9, 9,10,
	10,11,11,11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,
	10,10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9,
	 9,10,10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,
	10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 8, 8, 9,
	 9,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__16c2_s_p4_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__16c2_s_p4_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p4_0 = {
	_vq_quantthresh__16c2_s_p4_0,
	_vq_quantmap__16c2_s_p4_0,
	17,
	17
};

static static_codebook _16c2_s_p4_0 = {
	2, 289,
	_vq_lengthlist__16c2_s_p4_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__16c2_s_p4_0,
	NULL,
	&_vq_auxt__16c2_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p5_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16c2_s_p5_0[] = {
	 1, 4, 4, 5, 7, 7, 6, 7, 7, 4, 6, 6,10,10,10,10,
	10,10, 4, 7, 6,10,10,10,10,10,10, 5, 9, 9, 9,12,
	11,10,11,12, 7,10,10,12,12,12,12,12,12, 7,10,10,
	11,12,12,12,12,13, 6,10,10,10,12,12,10,12,12, 7,
	10,10,11,13,12,12,12,12, 7,10,10,11,12,12,12,12,
	12,
};

static float _vq_quantthresh__16c2_s_p5_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__16c2_s_p5_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p5_0 = {
	_vq_quantthresh__16c2_s_p5_0,
	_vq_quantmap__16c2_s_p5_0,
	3,
	3
};

static static_codebook _16c2_s_p5_0 = {
	4, 81,
	_vq_lengthlist__16c2_s_p5_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__16c2_s_p5_0,
	NULL,
	&_vq_auxt__16c2_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p5_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16c2_s_p5_1[] = {
	 2, 3, 3, 6, 6, 7, 7, 7, 7, 8, 8,11,11,11, 6, 6,
	 7, 7, 8, 8, 8, 8,11,11,11, 6, 6, 7, 7, 8, 8, 8,
	 8,11,11,11, 6, 6, 8, 8, 8, 8, 9, 9,11,11,11, 6,
	 6, 8, 8, 8, 8, 9, 9,11,11,11, 7, 7, 8, 8, 8, 8,
	 8, 8,11,11,11, 7, 7, 8, 8, 8, 8, 8, 9,11,11,11,
	 8, 8, 8, 8, 8, 8, 8, 8,11,11,11,11,11, 8, 8, 8,
	 8, 8, 8,11,11,11,11,11, 8, 8, 8, 8, 8, 8,11,11,
	11,11,11, 7, 7, 8, 8, 8, 8,
};

static float _vq_quantthresh__16c2_s_p5_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16c2_s_p5_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p5_1 = {
	_vq_quantthresh__16c2_s_p5_1,
	_vq_quantmap__16c2_s_p5_1,
	11,
	11
};

static static_codebook _16c2_s_p5_1 = {
	2, 121,
	_vq_lengthlist__16c2_s_p5_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16c2_s_p5_1,
	NULL,
	&_vq_auxt__16c2_s_p5_1,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16c2_s_p6_0[] = {
	 1, 4, 4, 7, 6, 8, 8, 9, 9,10,10,11,11, 5, 5, 5,
	 7, 7, 9, 9, 9, 9,11,11,12,12, 6, 5, 5, 7, 7, 9,
	 9,10,10,11,11,12,12, 0, 6, 6, 7, 7, 9, 9,10,10,
	11,11,12,12, 0, 7, 7, 7, 7, 9, 9,10,10,11,12,12,
	12, 0,11,11, 8, 8,10,10,11,11,12,12,13,13, 0,11,
	12, 8, 8,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

static float _vq_quantthresh__16c2_s_p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__16c2_s_p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p6_0 = {
	_vq_quantthresh__16c2_s_p6_0,
	_vq_quantmap__16c2_s_p6_0,
	13,
	13
};

static static_codebook _16c2_s_p6_0 = {
	2, 169,
	_vq_lengthlist__16c2_s_p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__16c2_s_p6_0,
	NULL,
	&_vq_auxt__16c2_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16c2_s_p6_1[] = {
	 2, 3, 3, 5, 5, 6, 6, 6, 5, 5, 6, 6, 6, 5, 5, 6,
	 6, 6, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__16c2_s_p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16c2_s_p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p6_1 = {
	_vq_quantthresh__16c2_s_p6_1,
	_vq_quantmap__16c2_s_p6_1,
	5,
	5
};

static static_codebook _16c2_s_p6_1 = {
	2, 25,
	_vq_lengthlist__16c2_s_p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16c2_s_p6_1,
	NULL,
	&_vq_auxt__16c2_s_p6_1,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16c2_s_p7_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9,10,10,11,11, 5, 5, 5,
	 8, 8, 9, 9,10,10,11,11,12,12, 6, 5, 5, 8, 8, 9,
	 9,10,10,11,11,12,13,18, 6, 6, 7, 7, 9, 9,10,10,
	12,12,13,13,18, 6, 6, 7, 7, 9, 9,10,10,12,12,13,
	13,18,11,10, 8, 8,10,10,11,11,12,12,13,13,18,11,
	11, 8, 8,10,10,11,11,12,13,13,13,18,18,18,10,11,
	11,11,12,12,13,13,14,14,18,18,18,11,11,11,11,12,
	12,13,13,14,14,18,18,18,14,14,12,12,12,12,14,14,
	15,14,18,18,18,15,15,11,12,12,12,13,13,15,15,18,
	18,18,18,18,13,13,13,13,13,14,17,16,18,18,18,18,
	18,13,14,13,13,14,13,15,14,
};

static float _vq_quantthresh__16c2_s_p7_0[] = {
	-60.5, -49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 
	27.5, 38.5, 49.5, 60.5, 
};

static long _vq_quantmap__16c2_s_p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p7_0 = {
	_vq_quantthresh__16c2_s_p7_0,
	_vq_quantmap__16c2_s_p7_0,
	13,
	13
};

static static_codebook _16c2_s_p7_0 = {
	2, 169,
	_vq_lengthlist__16c2_s_p7_0,
	1, -523206656, 1618345984, 4, 0,
	_vq_quantlist__16c2_s_p7_0,
	NULL,
	&_vq_auxt__16c2_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16c2_s_p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 7, 7, 7, 7, 9, 9, 9, 6, 6,
	 7, 7, 8, 8, 8, 8, 9, 9, 9, 6, 6, 7, 7, 8, 8, 8,
	 8, 9, 9, 9, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 7,
	 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 7, 7, 7, 7, 8, 8,
	 8, 8, 9, 9, 9, 7, 7, 7, 7, 7, 7, 8, 8, 9, 9, 9,
	 7, 7, 8, 8, 7, 7, 8, 8, 9, 9, 9, 9, 9, 7, 7, 7,
	 7, 8, 8, 9, 9, 9, 9, 9, 8, 8, 7, 7, 8, 8, 9, 9,
	 9, 9, 9, 7, 7, 7, 7, 8, 8,
};

static float _vq_quantthresh__16c2_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16c2_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p7_1 = {
	_vq_quantthresh__16c2_s_p7_1,
	_vq_quantmap__16c2_s_p7_1,
	11,
	11
};

static static_codebook _16c2_s_p7_1 = {
	2, 121,
	_vq_lengthlist__16c2_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16c2_s_p7_1,
	NULL,
	&_vq_auxt__16c2_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p8_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16c2_s_p8_0[] = {
	 1, 4, 4, 7, 6, 7, 7, 6, 6, 8, 8, 9, 9,10,10, 6,
	 6, 6, 8, 8, 9, 8, 8, 8, 9, 9,11,10,11,11, 7, 6,
	 6, 8, 8, 9, 8, 7, 7, 9, 9,10,10,12,11,14, 8, 8,
	 8, 9, 9, 9, 9, 9,10, 9,10,10,11,13,14, 8, 8, 8,
	 8, 9, 9, 8, 8, 9, 9,10,10,11,12,14,13,11, 9, 9,
	 9, 9, 9, 9, 9,10,11,10,13,12,14,11,13, 8, 9, 9,
	 9, 9, 9,10,10,11,10,13,12,14,14,14, 8, 9, 9, 9,
	11,11,11,11,11,12,13,13,14,14,14, 9, 8, 9, 9,10,
	10,12,10,11,12,12,14,14,14,14,11,12,10,10,12,12,
	12,12,13,14,12,12,14,14,14,12,12, 9,10,11,11,12,
	14,12,14,14,14,14,14,14,14,14,11,11,12,11,12,14,
	14,14,14,14,14,14,14,14,14,12,11,11,11,11,14,14,
	14,14,14,14,14,14,14,14,14,14,13,12,14,14,14,14,
	14,14,14,14,14,14,14,14,14,12,12,12,13,14,14,13,
	13,
};

static float _vq_quantthresh__16c2_s_p8_0[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__16c2_s_p8_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p8_0 = {
	_vq_quantthresh__16c2_s_p8_0,
	_vq_quantmap__16c2_s_p8_0,
	15,
	15
};

static static_codebook _16c2_s_p8_0 = {
	2, 225,
	_vq_lengthlist__16c2_s_p8_0,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__16c2_s_p8_0,
	NULL,
	&_vq_auxt__16c2_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p8_1[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__16c2_s_p8_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 7, 7, 8, 7, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8,11,12,11, 7, 7, 8, 8, 8, 8, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9, 9,11,11,10, 7, 7, 8,
	 8, 8, 8, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,
	11,11, 8, 7, 8, 8, 9, 9, 9, 9, 9, 9,10,10, 9,10,
	10, 9,10,10,11,11,12, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9,10, 9,10,10,10,10,11,11,11, 8, 8, 9, 9,
	 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,11,11,
	11, 8, 8, 9, 8, 9, 9, 9, 9,10, 9, 9, 9,10,10,10,
	10, 9,10,11,11,11, 9, 9, 9, 9,10, 9, 9, 9,10,10,
	 9,10, 9,10,10,10,10,10,11,12,11,11,11, 9, 9, 9,
	 9, 9,10,10, 9,10,10,10,10,10,10,10,10,12,11,13,
	13,11, 9, 9, 9, 9,10,10, 9,10,10,10,10,11,10,10,
	10,10,11,12,11,12,11, 9, 9, 9,10,10, 9,10,10,10,
	10,10,10,10,10,10,10,11,11,11,12,11, 9,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,11,12,12,12,
	11,11,11,10, 9,10,10,10,10,10,10,10,10,11,10,10,
	10,11,11,11,11,11,11,11,10,10,10,11,10,10,10,10,
	10,10,10,10,10,10,11,11,11,11,12,12,11,10,10,10,
	10,10,10,10,10,11,10,10,10,11,10,12,11,11,12,11,
	11,11,10,10,10,10,10,11,10,10,10,10,10,11,10,10,
	11,11,11,12,11,12,11,11,12,10,10,10,10,10,10,10,
	11,10,10,11,10,12,11,11,11,12,11,11,11,11,10,10,
	10,10,10,10,10,11,11,11,10,11,12,11,11,11,12,11,
	12,11,12,10,11,10,10,10,10,11,10,10,10,10,10,10,
	12,11,11,11,11,11,12,12,10,10,10,10,10,11,10,10,
	11,10,11,11,11,11,11,11,11,11,11,11,11,11,12,11,
	10,11,10,10,10,10,10,10,10,
};

static float _vq_quantthresh__16c2_s_p8_1[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__16c2_s_p8_1[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p8_1 = {
	_vq_quantthresh__16c2_s_p8_1,
	_vq_quantmap__16c2_s_p8_1,
	21,
	21
};

static static_codebook _16c2_s_p8_1 = {
	2, 441,
	_vq_lengthlist__16c2_s_p8_1,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__16c2_s_p8_1,
	NULL,
	&_vq_auxt__16c2_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16c2_s_p9_0[] = {
	 1, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__16c2_s_p9_0[] = {
	-5120.5, -4189.5, -3258.5, -2327.5, -1396.5, -465.5, 465.5, 1396.5, 
	2327.5, 3258.5, 4189.5, 5120.5, 
};

static long _vq_quantmap__16c2_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p9_0 = {
	_vq_quantthresh__16c2_s_p9_0,
	_vq_quantmap__16c2_s_p9_0,
	13,
	13
};

static static_codebook _16c2_s_p9_0 = {
	2, 169,
	_vq_lengthlist__16c2_s_p9_0,
	1, -510275072, 1631393792, 4, 0,
	_vq_quantlist__16c2_s_p9_0,
	NULL,
	&_vq_auxt__16c2_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p9_1[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__16c2_s_p9_1[] = {
	 1, 5, 5, 9, 8, 7, 7, 7, 6,10,11,11,11,11,11,11,
	11, 8, 7, 6, 8, 8,10, 9,10,10,10, 9,11,10,10,10,
	10,10, 8, 6, 6, 8, 8, 9, 8, 9, 8, 9,10,10,10,10,
	10,10,10,10, 8,10, 9, 9, 9, 9,10,10,10,10,10,10,
	10,10,10,10,10, 8, 9, 9, 9,10,10, 9,10,10,10,10,
	10,10,10,10,10,10,10,10, 9, 8, 9, 9,10,10,10,10,
	10,10,10,10,10,10,10,10, 9, 8, 8, 9, 9,10,10,10,
	10,10,10,10,10,10,10,10,10,10, 9,10, 9, 9,10,10,
	10,10,10,10,10,10,10,10,10,10,10, 9, 8, 9, 9,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10, 9,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	 8,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10, 9,10, 9,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10, 9,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__16c2_s_p9_1[] = {
	-367.5, -318.5, -269.5, -220.5, -171.5, -122.5, -73.5, -24.5, 
	24.5, 73.5, 122.5, 171.5, 220.5, 269.5, 318.5, 367.5, 
};

static long _vq_quantmap__16c2_s_p9_1[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p9_1 = {
	_vq_quantthresh__16c2_s_p9_1,
	_vq_quantmap__16c2_s_p9_1,
	17,
	17
};

static static_codebook _16c2_s_p9_1 = {
	2, 289,
	_vq_lengthlist__16c2_s_p9_1,
	1, -518488064, 1622704128, 5, 0,
	_vq_quantlist__16c2_s_p9_1,
	NULL,
	&_vq_auxt__16c2_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__16c2_s_p9_2[] = {
	13,
	12,
	14,
	11,
	15,
	10,
	16,
	9,
	17,
	8,
	18,
	7,
	19,
	6,
	20,
	5,
	21,
	4,
	22,
	3,
	23,
	2,
	24,
	1,
	25,
	0,
	26,
};

static long _vq_lengthlist__16c2_s_p9_2[] = {
	 1, 4, 4, 5, 5, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 8, 7, 8, 7, 7, 4, 4,
};

static float _vq_quantthresh__16c2_s_p9_2[] = {
	-12.5, -11.5, -10.5, -9.5, -8.5, -7.5, -6.5, -5.5, 
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5, 10.5, 
	11.5, 12.5, 
};

static long _vq_quantmap__16c2_s_p9_2[] = {
	   25,   23,   21,   19,   17,   15,   13,   11,
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,   12,   14,   16,   18,   20,
	   22,   24,   26,
};

static encode_aux_threshmatch _vq_auxt__16c2_s_p9_2 = {
	_vq_quantthresh__16c2_s_p9_2,
	_vq_quantmap__16c2_s_p9_2,
	27,
	27
};

static static_codebook _16c2_s_p9_2 = {
	1, 27,
	_vq_lengthlist__16c2_s_p9_2,
	1, -528875520, 1611661312, 5, 0,
	_vq_quantlist__16c2_s_p9_2,
	NULL,
	&_vq_auxt__16c2_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__16c2_s_short[] = {
	 7,10,11,11,11,14,15,15,17,14, 8, 6, 7, 7, 8, 9,
	11,11,14,17, 9, 6, 6, 6, 7, 7,10,11,15,16, 9, 6,
	 6, 4, 4, 5, 8, 9,12,16,10, 6, 6, 4, 4, 4, 6, 9,
	13,16,10, 7, 6, 5, 4, 3, 5, 7,13,16,11, 9, 8, 7,
	 6, 5, 5, 6,12,15,10,10,10, 9, 7, 6, 6, 7,11,15,
	13,13,13,13,11,10,10, 9,12,16,16,16,16,14,16,15,
	15,12,14,14,
};

static static_codebook _huff_book__16c2_s_short = {
	2, 100,
	_huff_lengthlist__16c2_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c0_s_long[] = {
	 3, 4,18, 8, 8, 8, 9,10,11,13, 3, 2,18, 5, 5, 6,
	 8,11,12,12,18,18,18,18,18,18,18,18,18,18, 8, 5,
	18, 7, 8, 8,10,14,14,16, 8, 5,18, 7, 5, 6, 7,11,
	12,13, 8, 6,18, 7, 5, 6, 7,10,12,14, 8, 7,18,10,
	 7, 6, 6, 8,11,14, 9,10,18,13, 9, 7, 6, 8,10,13,
	11,13,18,16,13,10, 9, 8, 8,11,12,17,18,17,18,14,
	11, 9,10,14,
};

static static_codebook _huff_book__44c0_s_long = {
	2, 100,
	_huff_lengthlist__44c0_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c0_s_p1_0[] = {
	 1, 5, 5, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 7, 0, 0, 0, 0,
	 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 9, 9,11, 0, 0, 0, 0, 0, 0, 9,11,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 9,10,11,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,11,10, 0,
	 0, 0, 0, 0, 0, 8, 9,11, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 9,11,10, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c0_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p1_0 = {
	_vq_quantthresh__44c0_s_p1_0,
	_vq_quantmap__44c0_s_p1_0,
	3,
	3
};

static static_codebook _44c0_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c0_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c0_s_p1_0,
	NULL,
	&_vq_auxt__44c0_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_s_p2_0[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c0_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p2_0 = {
	_vq_quantthresh__44c0_s_p2_0,
	_vq_quantmap__44c0_s_p2_0,
	5,
	5
};

static static_codebook _44c0_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c0_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c0_s_p2_0,
	NULL,
	&_vq_auxt__44c0_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_s_p3_0[] = {
	 1, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 5, 7, 7, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 5, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 7, 7, 9, 9,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 7, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c0_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p3_0 = {
	_vq_quantthresh__44c0_s_p3_0,
	_vq_quantmap__44c0_s_p3_0,
	5,
	5
};

static static_codebook _44c0_s_p3_0 = {
	4, 625,
	_vq_lengthlist__44c0_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c0_s_p3_0,
	NULL,
	&_vq_auxt__44c0_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c0_s_p4_0[] = {
	 1, 3, 2, 8, 7, 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c0_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p4_0 = {
	_vq_quantthresh__44c0_s_p4_0,
	_vq_quantmap__44c0_s_p4_0,
	9,
	9
};

static static_codebook _44c0_s_p4_0 = {
	2, 81,
	_vq_lengthlist__44c0_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c0_s_p4_0,
	NULL,
	&_vq_auxt__44c0_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c0_s_p5_0[] = {
	 1, 3, 3, 6, 6, 6, 6, 8, 8, 0, 0, 0, 6, 7, 7, 7,
	 9, 9, 0, 0, 0, 7, 7, 7, 7, 9, 9, 0, 0, 0, 7, 7,
	 8, 8, 9, 9, 0, 0, 0, 7, 7, 8, 8,10, 9, 0, 0, 0,
	 8, 8, 8, 8,10,10, 0, 0, 0, 8, 8, 8, 8,10,10, 0,
	 0, 0,10, 9, 9, 9,11,11, 0, 0, 0, 0, 0, 9, 9,11,
	11,
};

static float _vq_quantthresh__44c0_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c0_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p5_0 = {
	_vq_quantthresh__44c0_s_p5_0,
	_vq_quantmap__44c0_s_p5_0,
	9,
	9
};

static static_codebook _44c0_s_p5_0 = {
	2, 81,
	_vq_lengthlist__44c0_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c0_s_p5_0,
	NULL,
	&_vq_auxt__44c0_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c0_s_p6_0[] = {
	 1, 3, 4, 6, 6, 7, 8, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10,10,
	11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10,
	11,11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 8, 8, 9, 9, 9,10,10,10,
	10,11,11,11,12,12, 0, 0, 0, 8, 8, 9, 9, 9, 9,10,
	10,11,11,11,11,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,12,12,12,13, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 9, 9,
	 9,10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9,
	 9, 9, 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	10, 9,10,10,11,11,11,12,12,13,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,12,12,12,12,13,13, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,12,12,13,13,13,13, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,12,12,13,12,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,12,12,12,12,13,13,13,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,13,
	14,
};

static float _vq_quantthresh__44c0_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c0_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p6_0 = {
	_vq_quantthresh__44c0_s_p6_0,
	_vq_quantmap__44c0_s_p6_0,
	17,
	17
};

static static_codebook _44c0_s_p6_0 = {
	2, 289,
	_vq_lengthlist__44c0_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c0_s_p6_0,
	NULL,
	&_vq_auxt__44c0_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c0_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 4, 7, 7,10, 9, 9,11, 9, 9, 7,10,10,10,11,
	10,11,10,10, 6, 9, 9,11,10,10,11, 9,10, 6, 9, 9,
	11,10,10,11, 9, 9, 7,10,10,11,11,11,12,11,11, 7,
	 9, 9,10, 9, 9,11,11,10, 7, 9, 9,10,10,10,12,10,
	11,
};

static float _vq_quantthresh__44c0_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c0_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p7_0 = {
	_vq_quantthresh__44c0_s_p7_0,
	_vq_quantmap__44c0_s_p7_0,
	3,
	3
};

static static_codebook _44c0_s_p7_0 = {
	4, 81,
	_vq_lengthlist__44c0_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c0_s_p7_0,
	NULL,
	&_vq_auxt__44c0_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c0_s_p7_1[] = {
	 2, 3, 3, 6, 6, 7, 7, 7, 7, 7, 7,10, 9, 9, 6, 6,
	 7, 7, 8, 8, 8, 8, 9, 9, 9, 6, 6, 7, 7, 8, 8, 8,
	 8,10, 9,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10, 7,
	 7, 7, 7, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 8, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 9, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c0_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c0_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p7_1 = {
	_vq_quantthresh__44c0_s_p7_1,
	_vq_quantmap__44c0_s_p7_1,
	11,
	11
};

static static_codebook _44c0_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c0_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c0_s_p7_1,
	NULL,
	&_vq_auxt__44c0_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c0_s_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 7, 7, 8, 8,10, 9, 7, 5, 5,
	 7, 7, 8, 8, 8, 8, 9, 9,10,10, 7, 6, 6, 7, 7, 8,
	 8, 8, 8, 9, 9,10,10, 0, 8, 8, 8, 8, 9, 8, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,12,12, 9, 8, 9, 9,10,10,10,11,12,11, 0,13,
	13, 9, 8, 9, 9,10,10,10,11,11,11, 0, 0, 0,10,10,
	 9, 9,10,10,11,11,12,12, 0, 0, 0,10,10, 9, 8,10,
	10,11,11,12,12, 0, 0, 0,13,13,10,10,11,11,12,12,
	12,12, 0, 0, 0,15,15,10,10,11, 9,12,12,13,13, 0,
	 0, 0, 0, 0,12,11,11,11,12,12,13,13, 0, 0, 0, 0,
	 0,12,12,11,11,12,11,14,13,
};

static float _vq_quantthresh__44c0_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c0_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p8_0 = {
	_vq_quantthresh__44c0_s_p8_0,
	_vq_quantmap__44c0_s_p8_0,
	13,
	13
};

static static_codebook _44c0_s_p8_0 = {
	2, 169,
	_vq_lengthlist__44c0_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c0_s_p8_0,
	NULL,
	&_vq_auxt__44c0_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_s_p8_1[] = {
	 2, 3, 4, 4, 4, 6, 6, 6, 5, 5, 6, 6, 6, 5, 5, 6,
	 6, 6, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c0_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c0_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p8_1 = {
	_vq_quantthresh__44c0_s_p8_1,
	_vq_quantmap__44c0_s_p8_1,
	5,
	5
};

static static_codebook _44c0_s_p8_1 = {
	2, 25,
	_vq_lengthlist__44c0_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c0_s_p8_1,
	NULL,
	&_vq_auxt__44c0_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p9_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_s_p9_0[] = {
	 1, 7, 7,11,11,10,11,11,11,11, 9,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11, 8,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11, 8,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44c0_s_p9_0[] = {
	-331.5, -110.5, 110.5, 331.5, 
};

static long _vq_quantmap__44c0_s_p9_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p9_0 = {
	_vq_quantthresh__44c0_s_p9_0,
	_vq_quantmap__44c0_s_p9_0,
	5,
	5
};

static static_codebook _44c0_s_p9_0 = {
	4, 625,
	_vq_lengthlist__44c0_s_p9_0,
	1, -518283264, 1627103232, 3, 0,
	_vq_quantlist__44c0_s_p9_0,
	NULL,
	&_vq_auxt__44c0_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c0_s_p9_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 9, 9,12,12,13,12, 6, 5, 5,
	 7, 7, 8, 8,10,10,12,11,14,13, 6, 5, 5, 7, 7, 7,
	 8, 9, 9,11,12,13,12,15, 7, 7, 7, 7, 9, 9,11,11,
	14,13,14,13,15, 7, 7, 8, 8, 8, 9,10,10,11,11,12,
	11,15,10,10, 8, 8,10,10,12,11,13,15,14,14,15,11,
	10, 8, 8,11,11,11,11,13,13,12,13,15,15,15,10,10,
	10,10,12,13,13,13,15,14,15,15,15,10,10,10,11,13,
	14,13,14,14,14,15,15,15,15,15,11,12,11,13,14,14,
	15,15,15,15,15,14,15,11,11,13,12,14,13,15,15,15,
	15,15,15,15,13,12,13,12,15,15,15,15,15,15,15,15,
	15,12,13,12,12,15,14,15,15,
};

static float _vq_quantthresh__44c0_s_p9_1[] = {
	-93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 25.5, 
	42.5, 59.5, 76.5, 93.5, 
};

static long _vq_quantmap__44c0_s_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p9_1 = {
	_vq_quantthresh__44c0_s_p9_1,
	_vq_quantmap__44c0_s_p9_1,
	13,
	13
};

static static_codebook _44c0_s_p9_1 = {
	2, 169,
	_vq_lengthlist__44c0_s_p9_1,
	1, -522616832, 1620115456, 4, 0,
	_vq_quantlist__44c0_s_p9_1,
	NULL,
	&_vq_auxt__44c0_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c0_s_p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c0_s_p9_2[] = {
	 2, 4, 4, 6, 6, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9, 8,
	 9,10,10,10, 7, 7, 8, 7, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9,10,10,10, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9,10,10,10, 7, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,11,10,10, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10,10,10, 8, 8, 8, 8, 8, 9, 9,
	 9, 9, 9, 9,10, 9, 9,10,10,11, 9, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,11, 9, 8, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,11,11,10,11,11, 9, 9,
	 9, 9, 9, 9,10, 9, 9, 9,10, 9,10,10,10,11,11, 9,
	 9, 9, 9, 9, 9, 9,10, 9, 9, 9,10,11,10,10,10,11,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,
	11,11,10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,
	11,11,10,10, 9,10,10,10, 9,10, 9, 9, 9, 9,10,10,
	11,10,11,11,11, 9,10, 9, 9, 9, 9, 9, 9, 9,10,10,
	11,10,11,11,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10,10,10,11,11,10,11,11, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44c0_s_p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c0_s_p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c0_s_p9_2 = {
	_vq_quantthresh__44c0_s_p9_2,
	_vq_quantmap__44c0_s_p9_2,
	17,
	17
};

static static_codebook _44c0_s_p9_2 = {
	2, 289,
	_vq_lengthlist__44c0_s_p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c0_s_p9_2,
	NULL,
	&_vq_auxt__44c0_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c0_s_short[] = {
	 6, 8,17,12,12,12,14,15,17,16, 5, 1,16, 5, 6, 6,
	 9,12,14,13,16,16,16,16,16,16,16,16,16,16, 9, 4,
	16, 6, 9, 7,10,14,14,14, 8, 5,16, 8, 6, 7,10,13,
	15,16, 9, 6,16, 7, 7, 7,10,12,15,14,10, 8,16, 9,
	 9, 9,10,12,15,14,13, 7,16, 8, 7, 7,10,12,14,13,
	15, 6,16, 7, 5, 5, 7, 9,13,16,14, 7,16, 8, 6, 6,
	 8, 9,11,14,
};

static static_codebook _huff_book__44c0_s_short = {
	2, 100,
	_huff_lengthlist__44c0_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c0_sm_long[] = {
	 3, 4,10, 8,11, 9,10,10,11,13, 3, 2,10, 5, 8, 6,
	 8,11,12,13,10, 8, 7,10, 7, 8, 9,11,12,13, 8, 5,
	12, 6, 9, 7,10,13,14,16,11, 7, 7, 9, 5, 6, 7,10,
	13,13, 8, 6, 7, 7, 5, 5, 6, 9,12,13, 9, 8, 8, 8,
	 7, 6, 6, 8,11,14,10,10,10,11, 9, 8, 7, 8,10,13,
	11,14,12,15,13,11, 9, 8, 9,12,12,18,15,18,17,15,
	12,10,10,14,
};

static static_codebook _huff_book__44c0_sm_long = {
	2, 100,
	_huff_lengthlist__44c0_sm_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c0_sm_p1_0[] = {
	 1, 5, 5, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 7, 0, 0, 0, 0,
	 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 9, 9,10, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 9,10, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_sm_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c0_sm_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p1_0 = {
	_vq_quantthresh__44c0_sm_p1_0,
	_vq_quantmap__44c0_sm_p1_0,
	3,
	3
};

static static_codebook _44c0_sm_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c0_sm_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c0_sm_p1_0,
	NULL,
	&_vq_auxt__44c0_sm_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_sm_p2_0[] = {
	 1, 5, 5, 0, 0, 0, 5, 5, 0, 0, 0, 6, 6, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 5, 7, 6, 0, 0, 0, 8,
	 8, 0, 0, 0, 8, 8, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 5, 6, 7, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0, 0, 8, 8, 0, 0,
	 0, 8, 8, 0, 0, 0,10, 9, 0, 0, 0, 0, 0, 0, 0, 6,
	 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 9,
	10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8,
	 0, 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0,
	 0, 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0,10,10, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 8,10,10, 0, 0, 0,10,10, 0, 0, 0, 9,10, 0, 0, 0,
	10,10, 0, 0, 0, 0, 0, 0, 0, 8,10,10, 0, 0, 0,10,
	10, 0, 0, 0,10, 9, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_sm_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c0_sm_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p2_0 = {
	_vq_quantthresh__44c0_sm_p2_0,
	_vq_quantmap__44c0_sm_p2_0,
	5,
	5
};

static static_codebook _44c0_sm_p2_0 = {
	4, 625,
	_vq_lengthlist__44c0_sm_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c0_sm_p2_0,
	NULL,
	&_vq_auxt__44c0_sm_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_sm_p3_0[] = {
	 2, 3, 4, 6, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 8, 8,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 8, 8, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_sm_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c0_sm_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p3_0 = {
	_vq_quantthresh__44c0_sm_p3_0,
	_vq_quantmap__44c0_sm_p3_0,
	5,
	5
};

static static_codebook _44c0_sm_p3_0 = {
	4, 625,
	_vq_lengthlist__44c0_sm_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c0_sm_p3_0,
	NULL,
	&_vq_auxt__44c0_sm_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c0_sm_p4_0[] = {
	 2, 3, 3, 6, 6, 0, 0, 0, 0, 0, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 4, 4, 6, 6, 0, 0, 0, 0, 0, 5, 5, 6, 6,
	 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0,
	 7, 8, 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c0_sm_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c0_sm_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p4_0 = {
	_vq_quantthresh__44c0_sm_p4_0,
	_vq_quantmap__44c0_sm_p4_0,
	9,
	9
};

static static_codebook _44c0_sm_p4_0 = {
	2, 81,
	_vq_lengthlist__44c0_sm_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c0_sm_p4_0,
	NULL,
	&_vq_auxt__44c0_sm_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c0_sm_p5_0[] = {
	 1, 3, 3, 6, 6, 7, 7, 9, 9, 0, 6, 6, 7, 7, 8, 8,
	 9, 9, 0, 6, 6, 7, 7, 8, 8,10,10, 0, 7, 7, 8, 8,
	 8, 8,10,10, 0, 0, 0, 8, 8, 8, 8,10,10, 0, 0, 0,
	 8, 8, 9, 9,11,11, 0, 0, 0, 9, 9, 9, 9,11,11, 0,
	 0, 0,10,10, 9, 9,11,11, 0, 0, 0, 0, 0, 9, 9,11,
	11,
};

static float _vq_quantthresh__44c0_sm_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c0_sm_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p5_0 = {
	_vq_quantthresh__44c0_sm_p5_0,
	_vq_quantmap__44c0_sm_p5_0,
	9,
	9
};

static static_codebook _44c0_sm_p5_0 = {
	2, 81,
	_vq_lengthlist__44c0_sm_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c0_sm_p5_0,
	NULL,
	&_vq_auxt__44c0_sm_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c0_sm_p6_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9, 9, 8, 8,10,10,10,10,11,
	11, 0, 6, 5, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	12,12, 0, 6, 5, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,12, 0, 7, 7, 8, 8, 8, 8,10,10,10,10,11,11,
	11,11,12,12, 0, 0, 0, 8, 7, 8, 8,10,10,10,10,11,
	11,11,11,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,10,
	11,11,12,12,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,
	10,11,11,12,12,12,13, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,
	10,10,11,11,12,12,12,13,13,13, 0, 0, 0, 0, 0, 9,
	 9,10, 9,11,11,12,12,13,13,13,13, 0, 0, 0, 0, 0,
	10,10,10,10,11,11,12,12,13,13,14,14, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,12,12,13,13,14,14, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,12,12,13,13,14,14, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,12,12,13,13,14,14, 0,
	 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,13,13,14,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,14,
	14,
};

static float _vq_quantthresh__44c0_sm_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c0_sm_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p6_0 = {
	_vq_quantthresh__44c0_sm_p6_0,
	_vq_quantmap__44c0_sm_p6_0,
	17,
	17
};

static static_codebook _44c0_sm_p6_0 = {
	2, 289,
	_vq_lengthlist__44c0_sm_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c0_sm_p6_0,
	NULL,
	&_vq_auxt__44c0_sm_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c0_sm_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 5, 7, 6,10, 9, 9,10,
	 9, 9, 4, 7, 7,10, 9, 9,11, 9, 9, 6,10,10,10,10,
	10,11,10,10, 6, 9, 9,10,10,10,11, 9,10, 6, 9, 9,
	10,10,10,11,10, 9, 7,10,10,11,11,11,11,11,11, 6,
	 9, 9,10, 9,10,11,11,10, 6, 9, 9,10,10,10,11,10,
	11,
};

static float _vq_quantthresh__44c0_sm_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c0_sm_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p7_0 = {
	_vq_quantthresh__44c0_sm_p7_0,
	_vq_quantmap__44c0_sm_p7_0,
	3,
	3
};

static static_codebook _44c0_sm_p7_0 = {
	4, 81,
	_vq_lengthlist__44c0_sm_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c0_sm_p7_0,
	NULL,
	&_vq_auxt__44c0_sm_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c0_sm_p7_1[] = {
	 2, 4, 4, 5, 5, 7, 7, 7, 7, 7, 7, 9, 5, 5, 7, 7,
	 7, 7, 8, 8, 8, 8, 9, 5, 5, 7, 7, 7, 7, 8, 8, 8,
	 8,10, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8,10,10,10, 7,
	 7, 7, 7, 8, 8, 8, 8,10,10,10, 8, 8, 8, 8, 8, 8,
	 8, 8,10,10,10, 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 8, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c0_sm_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c0_sm_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p7_1 = {
	_vq_quantthresh__44c0_sm_p7_1,
	_vq_quantmap__44c0_sm_p7_1,
	11,
	11
};

static static_codebook _44c0_sm_p7_1 = {
	2, 121,
	_vq_lengthlist__44c0_sm_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c0_sm_p7_1,
	NULL,
	&_vq_auxt__44c0_sm_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c0_sm_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 7, 7, 9, 9,10,10, 7, 5, 5,
	 7, 7, 8, 8, 8, 8, 9, 9,10,10, 7, 5, 6, 7, 7, 8,
	 8, 8, 8, 9, 9,10,10, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,12,12, 9, 9, 9, 9,10,10,11,11,12,11, 0,12,
	12, 9, 9, 9, 9,10,10,11,11,12,11, 0, 0, 0, 9, 9,
	 9,10,11,11,11,11,12,12, 0, 0, 0,10,10, 9, 9,11,
	11,11,11,12,12, 0, 0, 0,13,13,10,10,11,11,12,12,
	13,13, 0, 0, 0,14,14,10,10,11,10,12,12,13,13, 0,
	 0, 0, 0, 0,12,12,11,11,12,12,13,13, 0, 0, 0, 0,
	 0,12,12,11,10,12,11,14,14,
};

static float _vq_quantthresh__44c0_sm_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c0_sm_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p8_0 = {
	_vq_quantthresh__44c0_sm_p8_0,
	_vq_quantmap__44c0_sm_p8_0,
	13,
	13
};

static static_codebook _44c0_sm_p8_0 = {
	2, 169,
	_vq_lengthlist__44c0_sm_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c0_sm_p8_0,
	NULL,
	&_vq_auxt__44c0_sm_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c0_sm_p8_1[] = {
	 2, 4, 4, 4, 5, 6, 5, 5, 5, 5, 6, 5, 5, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c0_sm_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c0_sm_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p8_1 = {
	_vq_quantthresh__44c0_sm_p8_1,
	_vq_quantmap__44c0_sm_p8_1,
	5,
	5
};

static static_codebook _44c0_sm_p8_1 = {
	2, 25,
	_vq_lengthlist__44c0_sm_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c0_sm_p8_1,
	NULL,
	&_vq_auxt__44c0_sm_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p9_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c0_sm_p9_0[] = {
	 1, 3, 4,11,11,11,11,11,11, 4, 6, 6,10,10,10,10,
	10,10, 4, 6, 5,10,10,10,10,10,10,10, 6, 7,10, 9,
	10,10,10,10,10, 7, 9,10, 9,10,10,10,10,10, 9,10,
	 7, 9,10,10,10,10,10,10,10, 9,10,10,10,10,10,10,
	10,10, 9,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44c0_sm_p9_0[] = {
	-773.5, -552.5, -331.5, -110.5, 110.5, 331.5, 552.5, 773.5, 
};

static long _vq_quantmap__44c0_sm_p9_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p9_0 = {
	_vq_quantthresh__44c0_sm_p9_0,
	_vq_quantmap__44c0_sm_p9_0,
	9,
	9
};

static static_codebook _44c0_sm_p9_0 = {
	2, 81,
	_vq_lengthlist__44c0_sm_p9_0,
	1, -516186112, 1627103232, 4, 0,
	_vq_quantlist__44c0_sm_p9_0,
	NULL,
	&_vq_auxt__44c0_sm_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c0_sm_p9_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 9, 9,12,12,13,13, 6, 5, 5,
	 7, 7, 8, 8,10,10,12,12,14,13, 6, 5, 5, 6, 7, 8,
	 8,10, 9,11,12,13,12,18, 7, 7, 7, 7, 9, 9,11,11,
	14,14,14,13,17, 7, 7, 8, 7, 9, 9,10,10,12,11,13,
	12,18,10,10, 8, 8,10,10,12,11,13,14,16,15,17,11,
	11, 8, 8,11,11,11,12,13,13,13,15,18,18,18,10,11,
	10,10,12,13,14,13,17,14,17,17,17,10,10,11,11,14,
	14,14,13,14,14,18,18,18,17,15,11,12,12,13,15,15,
	16,17,18,16,18,14,15,11,11,14,13,14,14,16,17,18,
	17,17,17,18,14,13,13,12,17,15,17,16,18,17,18,18,
	18,12,13,12,13,16,14,18,18,
};

static float _vq_quantthresh__44c0_sm_p9_1[] = {
	-93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 25.5, 
	42.5, 59.5, 76.5, 93.5, 
};

static long _vq_quantmap__44c0_sm_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p9_1 = {
	_vq_quantthresh__44c0_sm_p9_1,
	_vq_quantmap__44c0_sm_p9_1,
	13,
	13
};

static static_codebook _44c0_sm_p9_1 = {
	2, 169,
	_vq_lengthlist__44c0_sm_p9_1,
	1, -522616832, 1620115456, 4, 0,
	_vq_quantlist__44c0_sm_p9_1,
	NULL,
	&_vq_auxt__44c0_sm_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c0_sm_p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c0_sm_p9_2[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9,10, 6, 6, 7, 7, 8, 7, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9,10, 6, 6, 7, 7, 7, 7, 8, 8, 9, 8, 9, 9, 9,
	 9, 9, 9,10, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10,10,10, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,11,10,10, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10, 9,10,11,10, 9, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,10,10,11,11, 9, 8, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,10,11,10,11,11, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,10,10,11,11, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,10,10,11,11,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,10,10,10,
	11,11,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,11,10,
	11,11,10,10, 9,10,10,10, 9, 9, 9, 9, 9, 9,10,10,
	10,11,11,11,11,10,10, 9, 9, 9, 9, 9, 9, 9, 9,10,
	11,10,11,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10,10,11,11,11,10,11,11, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44c0_sm_p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c0_sm_p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c0_sm_p9_2 = {
	_vq_quantthresh__44c0_sm_p9_2,
	_vq_quantmap__44c0_sm_p9_2,
	17,
	17
};

static static_codebook _44c0_sm_p9_2 = {
	2, 289,
	_vq_lengthlist__44c0_sm_p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c0_sm_p9_2,
	NULL,
	&_vq_auxt__44c0_sm_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c0_sm_short[] = {
	 5, 7,12,12,13,13,13,15,16,18, 4, 2,11, 5, 9, 7,
	 9,12,13,14,13, 7, 8, 9, 8, 9,11,13,19,19, 9, 4,
	12, 5,10, 7, 9,12,15,14,11, 6, 7, 7, 5, 6, 8,11,
	15,17,10, 5, 8, 6, 6, 5, 6, 9,14,14,10, 5, 9, 7,
	 7, 6, 6, 9,12,14,12, 6,10, 7, 7, 7, 8, 9,13,14,
	14, 7, 9, 7, 5, 5, 6, 9,11,13,14, 9,10, 9, 6, 6,
	 7, 8,10,13,
};

static static_codebook _huff_book__44c0_sm_short = {
	2, 100,
	_huff_lengthlist__44c0_sm_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c1_s_long[] = {
	 3, 4,19, 9,10, 9, 9,10,11,12, 4, 2,19, 5, 6, 6,
	 8,10,12,11,19,19,19,19,19,19,19,19,18,18, 8, 4,
	18, 6, 8, 7,10,13,14,13,10, 5,18, 7, 4, 6, 7,10,
	12,13, 9, 6,18, 7, 5, 6, 7,10,12,13, 9, 7,18, 9,
	 7, 6, 6, 7,10,13, 9, 9,18,12, 9, 8, 6, 6, 9,12,
	10,12,18,15,12,11, 9, 8, 8,11,11,14,18,17,15,13,
	12, 9,10,12,
};

static static_codebook _huff_book__44c1_s_long = {
	2, 100,
	_huff_lengthlist__44c1_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c1_s_p1_0[] = {
	 2, 4, 4, 0, 0, 0, 0, 0, 0, 5, 7, 6, 0, 0, 0, 0,
	 0, 0, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 7, 7, 0, 0, 0, 0,
	 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0,
	 0, 0, 0, 0, 8, 9, 8, 0, 0, 0, 0, 0, 0, 8, 9, 9,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 8, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0,
	 0, 0, 0, 0, 0, 8, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9,
	 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c1_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p1_0 = {
	_vq_quantthresh__44c1_s_p1_0,
	_vq_quantmap__44c1_s_p1_0,
	3,
	3
};

static static_codebook _44c1_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c1_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c1_s_p1_0,
	NULL,
	&_vq_auxt__44c1_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c1_s_p2_0[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c1_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p2_0 = {
	_vq_quantthresh__44c1_s_p2_0,
	_vq_quantmap__44c1_s_p2_0,
	5,
	5
};

static static_codebook _44c1_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c1_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c1_s_p2_0,
	NULL,
	&_vq_auxt__44c1_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c1_s_p3_0[] = {
	 2, 3, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c1_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p3_0 = {
	_vq_quantthresh__44c1_s_p3_0,
	_vq_quantmap__44c1_s_p3_0,
	5,
	5
};

static static_codebook _44c1_s_p3_0 = {
	4, 625,
	_vq_lengthlist__44c1_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c1_s_p3_0,
	NULL,
	&_vq_auxt__44c1_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c1_s_p4_0[] = {
	 1, 3, 2, 7, 7, 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 9, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c1_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p4_0 = {
	_vq_quantthresh__44c1_s_p4_0,
	_vq_quantmap__44c1_s_p4_0,
	9,
	9
};

static static_codebook _44c1_s_p4_0 = {
	2, 81,
	_vq_lengthlist__44c1_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c1_s_p4_0,
	NULL,
	&_vq_auxt__44c1_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c1_s_p5_0[] = {
	 1, 3, 3, 5, 5, 6, 6, 8, 8, 0, 0, 0, 7, 7, 8, 7,
	 9, 9, 0, 0, 0, 7, 7, 8, 8, 9, 9, 0, 0, 0, 7, 7,
	 8, 8,10,10, 0, 0, 0, 7, 7, 8, 8,10,10, 0, 0, 0,
	 8, 8, 9, 9,10,10, 0, 0, 0, 8, 8, 9, 9,10,10, 0,
	 0, 0,10,10, 9, 9,11,11, 0, 0, 0, 0, 0, 9, 9,11,
	11,
};

static float _vq_quantthresh__44c1_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c1_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p5_0 = {
	_vq_quantthresh__44c1_s_p5_0,
	_vq_quantmap__44c1_s_p5_0,
	9,
	9
};

static static_codebook _44c1_s_p5_0 = {
	2, 81,
	_vq_lengthlist__44c1_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c1_s_p5_0,
	NULL,
	&_vq_auxt__44c1_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c1_s_p6_0[] = {
	 1, 3, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10,
	11,11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 8, 8, 9, 9,10,10,10,10,
	10,10,11,11,12,12, 0, 0, 0, 8, 8, 9, 9, 9,10,10,
	10,11,11,11,11,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,12,12,13,12, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 9, 9,
	10,10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9,
	 9,10, 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	10,10,10,10,11,11,11,11,12,12,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,11,11,12,12,13,13, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,12,12,13,12,13,13, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,12,12,13,13,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,11,12,12,12,13,13,14,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,11,11,12,12,13,13,14,
	14,
};

static float _vq_quantthresh__44c1_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c1_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p6_0 = {
	_vq_quantthresh__44c1_s_p6_0,
	_vq_quantmap__44c1_s_p6_0,
	17,
	17
};

static static_codebook _44c1_s_p6_0 = {
	2, 289,
	_vq_lengthlist__44c1_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c1_s_p6_0,
	NULL,
	&_vq_auxt__44c1_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c1_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10,10,10,10,
	10,11,10,10, 6, 9, 9,10,11,10,10, 9, 9, 6, 9, 9,
	10,10,10,10, 9, 9, 7,10,10,11,10,10,11,11,11, 6,
	 9, 9,10, 9, 9,11,10,10, 6, 9, 9,10, 9, 9,11,10,
	10,
};

static float _vq_quantthresh__44c1_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c1_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p7_0 = {
	_vq_quantthresh__44c1_s_p7_0,
	_vq_quantmap__44c1_s_p7_0,
	3,
	3
};

static static_codebook _44c1_s_p7_0 = {
	4, 81,
	_vq_lengthlist__44c1_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c1_s_p7_0,
	NULL,
	&_vq_auxt__44c1_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c1_s_p7_1[] = {
	 2, 3, 3, 6, 6, 7, 7, 7, 7, 8, 8,10,10,10, 6, 6,
	 7, 7, 8, 8, 8, 8,10,10,10, 6, 6, 7, 7, 8, 8, 8,
	 8,10,10,10, 7, 7, 7, 7, 8, 8, 8, 8,10,10,10, 7,
	 7, 7, 7, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 8, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c1_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c1_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p7_1 = {
	_vq_quantthresh__44c1_s_p7_1,
	_vq_quantmap__44c1_s_p7_1,
	11,
	11
};

static static_codebook _44c1_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c1_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c1_s_p7_1,
	NULL,
	&_vq_auxt__44c1_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c1_s_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 7, 7, 9, 8,10,10, 6, 5, 5,
	 7, 7, 8, 8, 8, 8, 9, 9,10,10, 7, 6, 6, 7, 7, 8,
	 8, 8, 8, 9, 9,10,10, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,12,12, 9, 9, 9, 9,10,10,10,11,12,11, 0,12,
	13, 9, 8, 9, 9,10,10,11,11,11,11, 0, 0, 0, 9, 9,
	 9, 9,10,10,11,11,12,12, 0, 0, 0,10,10, 9, 9,10,
	10,11,11,12,12, 0, 0, 0,13,14,10,10,11,11,12,12,
	12,13, 0, 0, 0,14,14,10,10,11,10,12,12,13,13, 0,
	 0, 0, 0, 0,12,11,11,11,12,12,13,13, 0, 0, 0, 0,
	 0,12,12,11,10,12,11,14,13,
};

static float _vq_quantthresh__44c1_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c1_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p8_0 = {
	_vq_quantthresh__44c1_s_p8_0,
	_vq_quantmap__44c1_s_p8_0,
	13,
	13
};

static static_codebook _44c1_s_p8_0 = {
	2, 169,
	_vq_lengthlist__44c1_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c1_s_p8_0,
	NULL,
	&_vq_auxt__44c1_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c1_s_p8_1[] = {
	 2, 3, 3, 5, 5, 6, 6, 6, 5, 5, 6, 6, 6, 5, 5, 6,
	 6, 6, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c1_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c1_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p8_1 = {
	_vq_quantthresh__44c1_s_p8_1,
	_vq_quantmap__44c1_s_p8_1,
	5,
	5
};

static static_codebook _44c1_s_p8_1 = {
	2, 25,
	_vq_lengthlist__44c1_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c1_s_p8_1,
	NULL,
	&_vq_auxt__44c1_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c1_s_p9_0[] = {
	 1, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 5, 7, 6,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 4, 6, 6, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 7, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 8,
};

static float _vq_quantthresh__44c1_s_p9_0[] = {
	-1215.5, -994.5, -773.5, -552.5, -331.5, -110.5, 110.5, 331.5, 
	552.5, 773.5, 994.5, 1215.5, 
};

static long _vq_quantmap__44c1_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p9_0 = {
	_vq_quantthresh__44c1_s_p9_0,
	_vq_quantmap__44c1_s_p9_0,
	13,
	13
};

static static_codebook _44c1_s_p9_0 = {
	2, 169,
	_vq_lengthlist__44c1_s_p9_0,
	1, -514541568, 1627103232, 4, 0,
	_vq_quantlist__44c1_s_p9_0,
	NULL,
	&_vq_auxt__44c1_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c1_s_p9_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 9, 9,12,12,13,12, 6, 5, 5,
	 7, 7, 8, 8,10,10,11,12,13,13, 6, 5, 5, 7, 7, 8,
	 8, 9, 9,11,11,12,12,15, 7, 7, 7, 7, 9, 9,11,11,
	13,13,14,13,15, 7, 7, 8, 7, 9, 9,10,10,12,11,14,
	12,15,10,10, 8, 8,10,10,12,11,12,13,14,13,15,10,
	10, 8, 8,11,10,11,11,13,12,13,15,14,15,15,10,10,
	10,10,12,13,13,12,15,14,15,15,13,10,10,11,11,13,
	13,13,12,13,13,15,15,15,14,15,11,11,13,13,13,13,
	14,15,15,15,15,13,13,11,11,12,13,13,14,13,15,15,
	15,14,15,15,13,12,12,11,13,13,15,15,15,15,15,15,
	14,13,12,13,12,15,13,15,15,
};

static float _vq_quantthresh__44c1_s_p9_1[] = {
	-93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 25.5, 
	42.5, 59.5, 76.5, 93.5, 
};

static long _vq_quantmap__44c1_s_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p9_1 = {
	_vq_quantthresh__44c1_s_p9_1,
	_vq_quantmap__44c1_s_p9_1,
	13,
	13
};

static static_codebook _44c1_s_p9_1 = {
	2, 169,
	_vq_lengthlist__44c1_s_p9_1,
	1, -522616832, 1620115456, 4, 0,
	_vq_quantlist__44c1_s_p9_1,
	NULL,
	&_vq_auxt__44c1_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c1_s_p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c1_s_p9_2[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 8, 9, 8, 9, 9,
	 9,10,10,10, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9,10,10,10, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9,10,10,10, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9,10,10,11,10, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9,10, 9, 9, 9,10,10,10, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,11, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10, 9,10,10,10,11,11, 9, 8, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,11,11,11,11, 9, 9,
	 9, 9, 9, 9,10, 9,10, 9, 9, 9,10,10,10,11,10, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,11,11,11,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,10,10,10,
	11,11,10, 9, 9,10, 9, 9, 9, 9, 9, 9, 9,10,10,10,
	11,11,11,10, 9,10,10,10, 9, 9, 9, 9,10, 9,10,10,
	10,10,11,11,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,
	11,10,10,10,10,10,10, 9, 9, 9, 9,10, 9, 9, 9, 9,
	10,10,10,11,11,10,10,11,11, 9, 9,10, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44c1_s_p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c1_s_p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c1_s_p9_2 = {
	_vq_quantthresh__44c1_s_p9_2,
	_vq_quantmap__44c1_s_p9_2,
	17,
	17
};

static static_codebook _44c1_s_p9_2 = {
	2, 289,
	_vq_lengthlist__44c1_s_p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c1_s_p9_2,
	NULL,
	&_vq_auxt__44c1_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c1_s_short[] = {
	 5, 7,17,13,12,12,14,16,15,16, 4, 2,17, 5, 7, 6,
	 8,11,14,13,16,16,16,16,16,16,16,16,16,16,10, 4,
	16, 4, 8, 6, 7,11,14,14,10, 5,16, 6, 5, 6, 8,12,
	15,15,10, 5,16, 5, 6, 5, 7,10,14,15,11, 6,16, 7,
	 8, 7, 7,10,14,14,13, 8,16, 8, 7, 7, 8,10,12,13,
	12, 8,16, 7, 5, 5, 6, 8,11,13,13, 9,16, 9, 6, 6,
	 7, 8,10,13,
};

static static_codebook _huff_book__44c1_s_short = {
	2, 100,
	_huff_lengthlist__44c1_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c1_sm_long[] = {
	 3, 4,10, 9,11, 9,10,11,11,13, 4, 2,11, 5, 7, 7,
	 8,10,12,13,10,10, 7,12, 8, 9, 9,10,12,13, 8, 5,
	13, 6, 9, 7,10,12,15,15,10, 6, 7, 8, 5, 5, 7,10,
	12,13, 9, 6, 8, 7, 5, 5, 6, 9,11,12, 9, 8, 9, 9,
	 6, 5, 5, 7,10,13,10,10,10,12, 9, 7, 6, 7, 9,13,
	10,13,11,17,11,11, 9, 8, 9,12,12,16,14,17,15,14,
	12,10,10,12,
};

static static_codebook _huff_book__44c1_sm_long = {
	2, 100,
	_huff_lengthlist__44c1_sm_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c1_sm_p1_0[] = {
	 1, 5, 5, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 7, 0, 0, 0, 0,
	 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 9, 9,10, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 9,10, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_sm_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c1_sm_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p1_0 = {
	_vq_quantthresh__44c1_sm_p1_0,
	_vq_quantmap__44c1_sm_p1_0,
	3,
	3
};

static static_codebook _44c1_sm_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c1_sm_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c1_sm_p1_0,
	NULL,
	&_vq_auxt__44c1_sm_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c1_sm_p2_0[] = {
	 1, 5, 5, 0, 0, 0, 5, 6, 0, 0, 0, 5, 6, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 5, 7, 6, 0, 0, 0, 8,
	 8, 0, 0, 0, 8, 8, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 5, 6, 7, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0, 0, 8, 8, 0, 0,
	 0, 8, 8, 0, 0, 0,10, 9, 0, 0, 0, 0, 0, 0, 0, 6,
	 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 9,
	10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8,
	 0, 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0,
	 0, 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0,10,10, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 8,10,10, 0, 0, 0,10,10, 0, 0, 0, 9,10, 0, 0, 0,
	10,10, 0, 0, 0, 0, 0, 0, 0, 8,10,10, 0, 0, 0,10,
	10, 0, 0, 0,10, 9, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_sm_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c1_sm_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p2_0 = {
	_vq_quantthresh__44c1_sm_p2_0,
	_vq_quantmap__44c1_sm_p2_0,
	5,
	5
};

static static_codebook _44c1_sm_p2_0 = {
	4, 625,
	_vq_lengthlist__44c1_sm_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c1_sm_p2_0,
	NULL,
	&_vq_auxt__44c1_sm_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c1_sm_p3_0[] = {
	 2, 3, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 4, 4, 7, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 8, 8,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_sm_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c1_sm_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p3_0 = {
	_vq_quantthresh__44c1_sm_p3_0,
	_vq_quantmap__44c1_sm_p3_0,
	5,
	5
};

static static_codebook _44c1_sm_p3_0 = {
	4, 625,
	_vq_lengthlist__44c1_sm_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c1_sm_p3_0,
	NULL,
	&_vq_auxt__44c1_sm_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c1_sm_p4_0[] = {
	 1, 3, 3, 7, 7, 0, 0, 0, 0, 0, 5, 5, 6, 6, 0, 0,
	 0, 0, 0, 5, 5, 7, 7, 0, 0, 0, 0, 0, 7, 7, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 9, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c1_sm_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c1_sm_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p4_0 = {
	_vq_quantthresh__44c1_sm_p4_0,
	_vq_quantmap__44c1_sm_p4_0,
	9,
	9
};

static static_codebook _44c1_sm_p4_0 = {
	2, 81,
	_vq_lengthlist__44c1_sm_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c1_sm_p4_0,
	NULL,
	&_vq_auxt__44c1_sm_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c1_sm_p5_0[] = {
	 2, 3, 3, 5, 5, 6, 6, 8, 8, 0, 5, 5, 6, 6, 7, 7,
	 9, 9, 0, 5, 5, 6, 6, 7, 7, 9, 9, 0, 6, 6, 7, 7,
	 7, 7, 9, 9, 0, 0, 0, 7, 7, 7, 7, 9, 9, 0, 0, 0,
	 7, 7, 8, 8,10,10, 0, 0, 0, 7, 7, 8, 8,10,10, 0,
	 0, 0, 9, 9, 9, 9,10,10, 0, 0, 0, 0, 0, 8, 8,10,
	10,
};

static float _vq_quantthresh__44c1_sm_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c1_sm_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p5_0 = {
	_vq_quantthresh__44c1_sm_p5_0,
	_vq_quantmap__44c1_sm_p5_0,
	9,
	9
};

static static_codebook _44c1_sm_p5_0 = {
	2, 81,
	_vq_lengthlist__44c1_sm_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c1_sm_p5_0,
	NULL,
	&_vq_auxt__44c1_sm_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c1_sm_p6_0[] = {
	 2, 3, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 5, 5, 6, 6, 8, 8, 9, 9, 9, 9,10,10,10,10,
	11,11, 0, 5, 5, 6, 6, 8, 8, 9, 9, 9, 9,10,10,10,
	10,11,11, 0, 6, 6, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,
	10,10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9,
	 9,10,10,11,11,12,12, 0, 0, 0, 8, 8, 8, 8,10,10,
	10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 8, 8,10,
	10,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 8, 8,
	 9, 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 8,
	 8, 9, 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	 9, 9,10,10,10,10,11,11,12,12,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,10,10,11,11,12,12,13,13, 0, 0, 0,
	 0, 0, 0, 0,10,10,11,11,11,11,13,12,13,13, 0, 0,
	 0, 0, 0, 0, 0,10,10,11,11,11,11,13,13,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,11,11,12,12,13,13,14,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,11,11,12,12,13,12,14,
	14,
};

static float _vq_quantthresh__44c1_sm_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c1_sm_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p6_0 = {
	_vq_quantthresh__44c1_sm_p6_0,
	_vq_quantmap__44c1_sm_p6_0,
	17,
	17
};

static static_codebook _44c1_sm_p6_0 = {
	2, 289,
	_vq_lengthlist__44c1_sm_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c1_sm_p6_0,
	NULL,
	&_vq_auxt__44c1_sm_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c1_sm_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10,10,10,10,
	10,10,10,10, 6, 9, 9,10,10,10,10, 9, 9, 6, 9, 9,
	10,10,10,10,10, 9, 7,10,10,11,10,10,11,11,11, 6,
	 9, 9,10, 9, 9,11,10,10, 6, 9, 9,10, 9, 9,11,10,
	10,
};

static float _vq_quantthresh__44c1_sm_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c1_sm_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p7_0 = {
	_vq_quantthresh__44c1_sm_p7_0,
	_vq_quantmap__44c1_sm_p7_0,
	3,
	3
};

static static_codebook _44c1_sm_p7_0 = {
	4, 81,
	_vq_lengthlist__44c1_sm_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c1_sm_p7_0,
	NULL,
	&_vq_auxt__44c1_sm_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c1_sm_p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 7, 7, 8, 8,10, 5, 5, 6, 6,
	 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6, 7, 7, 8, 8, 8,
	 8,10, 7, 6, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10, 7,
	 7, 8, 8, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 9, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 8, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c1_sm_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c1_sm_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p7_1 = {
	_vq_quantthresh__44c1_sm_p7_1,
	_vq_quantmap__44c1_sm_p7_1,
	11,
	11
};

static static_codebook _44c1_sm_p7_1 = {
	2, 121,
	_vq_lengthlist__44c1_sm_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c1_sm_p7_1,
	NULL,
	&_vq_auxt__44c1_sm_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c1_sm_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10, 6, 5, 5,
	 7, 7, 8, 8, 8, 8, 9, 9,10,10, 7, 5, 6, 7, 7, 8,
	 8, 8, 8, 9, 9,11,10, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,12,12, 9, 9, 9, 9,10,10,11,11,12,11, 0,13,
	13, 9, 9, 9, 9,10,10,11,11,12,12, 0, 0, 0, 9, 9,
	 9, 9,10,10,11,12,12,12, 0, 0, 0,10,10, 9, 9,11,
	11,12,12,13,13, 0, 0, 0,13,13,10,10,11,11,12,12,
	13,13, 0, 0, 0,14,14,10,10,11,10,12,12,13,13, 0,
	 0, 0, 0, 0,12,11,11,11,12,12,14,13, 0, 0, 0, 0,
	 0,12,12,11,10,12,12,14,13,
};

static float _vq_quantthresh__44c1_sm_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c1_sm_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p8_0 = {
	_vq_quantthresh__44c1_sm_p8_0,
	_vq_quantmap__44c1_sm_p8_0,
	13,
	13
};

static static_codebook _44c1_sm_p8_0 = {
	2, 169,
	_vq_lengthlist__44c1_sm_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c1_sm_p8_0,
	NULL,
	&_vq_auxt__44c1_sm_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c1_sm_p8_1[] = {
	 2, 4, 4, 4, 5, 6, 5, 5, 5, 5, 6, 5, 5, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c1_sm_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c1_sm_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p8_1 = {
	_vq_quantthresh__44c1_sm_p8_1,
	_vq_quantmap__44c1_sm_p8_1,
	5,
	5
};

static static_codebook _44c1_sm_p8_1 = {
	2, 25,
	_vq_lengthlist__44c1_sm_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c1_sm_p8_1,
	NULL,
	&_vq_auxt__44c1_sm_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c1_sm_p9_0[] = {
	 1, 5, 5,11,11,11,11,11,11,11,11,11,11, 4, 5, 5,
	11,10,11,11,11,11,11,11,11,11, 4, 5, 4,11,11,11,
	11,11,11,11,11,11,11,11, 6, 7,11, 9,11,11,11,11,
	11,11,11,11,11, 6, 7, 9, 8,11,11,11,11,11,11,11,
	11,11, 9, 9, 7, 8,11,11,11,11,11,11,11,11,11,10,
	10, 9,10,11,11,11,11,11,11,11,11,11,11,11, 8,11,
	10,10,11,11,11,11,11,11,11,11,11, 9,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11, 9,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,10,10,10,10,10,
};

static float _vq_quantthresh__44c1_sm_p9_0[] = {
	-1215.5, -994.5, -773.5, -552.5, -331.5, -110.5, 110.5, 331.5, 
	552.5, 773.5, 994.5, 1215.5, 
};

static long _vq_quantmap__44c1_sm_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p9_0 = {
	_vq_quantthresh__44c1_sm_p9_0,
	_vq_quantmap__44c1_sm_p9_0,
	13,
	13
};

static static_codebook _44c1_sm_p9_0 = {
	2, 169,
	_vq_lengthlist__44c1_sm_p9_0,
	1, -514541568, 1627103232, 4, 0,
	_vq_quantlist__44c1_sm_p9_0,
	NULL,
	&_vq_auxt__44c1_sm_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c1_sm_p9_1[] = {
	 1, 4, 4, 6, 6, 7, 7,10,10,12,12,13,13, 6, 5, 5,
	 7, 7, 8, 8,10,10,12,12,14,13, 6, 5, 5, 7, 7, 8,
	 8,10,10,12,12,13,13,16, 7, 7, 7, 7, 9, 9,11,11,
	13,14,13,15,18, 7, 6, 8, 7, 9, 9,11,10,12,12,14,
	12,19,10,10, 8, 8,10,10,12,11,13,15,15,15,17,11,
	11, 8, 7,11,10,12,12,13,13,14,15,16,17,17,10,11,
	10,10,13,13,14,13,16,15,17,17,16,10,10,11,11,14,
	14,16,13,14,14,17,19,17,15,14,11,11,13,13,15,14,
	16,15,17,16,17,14,14,11,11,14,13,14,14,14,15,17,
	16,16,16,17,13,13,13,13,16,14,17,15,16,17,18,18,
	17,13,13,13,13,15,15,16,16,
};

static float _vq_quantthresh__44c1_sm_p9_1[] = {
	-93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 25.5, 
	42.5, 59.5, 76.5, 93.5, 
};

static long _vq_quantmap__44c1_sm_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p9_1 = {
	_vq_quantthresh__44c1_sm_p9_1,
	_vq_quantmap__44c1_sm_p9_1,
	13,
	13
};

static static_codebook _44c1_sm_p9_1 = {
	2, 169,
	_vq_lengthlist__44c1_sm_p9_1,
	1, -522616832, 1620115456, 4, 0,
	_vq_quantlist__44c1_sm_p9_1,
	NULL,
	&_vq_auxt__44c1_sm_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c1_sm_p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c1_sm_p9_2[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9,10, 6, 6, 7, 7, 8, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9,10, 6, 6, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9,10, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 7, 8, 8, 9, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,11,11,11, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10,10,10, 7, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10, 9,10,10,10, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,10,10,11,11, 8, 8, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9,10,11,10,11,11, 9, 9,
	 9, 9, 9, 9, 9,10,10, 9, 9, 9,10,11,10,11,11, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,11,11,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,11,10,11,
	11,10,11, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,
	11,11,11,11, 9,10,10,10, 9, 9, 9, 9,10, 9,10,10,
	10,11,11,11,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,10,
	11,10,11,10,11,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10,10,11,11,10,10,10,11, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44c1_sm_p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c1_sm_p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c1_sm_p9_2 = {
	_vq_quantthresh__44c1_sm_p9_2,
	_vq_quantmap__44c1_sm_p9_2,
	17,
	17
};

static static_codebook _44c1_sm_p9_2 = {
	2, 289,
	_vq_lengthlist__44c1_sm_p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c1_sm_p9_2,
	NULL,
	&_vq_auxt__44c1_sm_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c1_sm_short[] = {
	 4, 7,13,13,13,13,14,16,16,19, 4, 2,13, 5, 8, 7,
	 9,12,14,13,16, 9,10,10, 9,10,11,13,17,19, 9, 4,
	13, 5,10, 6, 8,11,15,15,11, 6, 9, 7, 6, 6, 8,11,
	14,18,11, 5, 9, 6, 6, 5, 6, 9,13,15,12, 6, 9, 7,
	 6, 5, 5, 8,12,15,13, 7,10, 8, 6, 6, 7, 9,12,13,
	13, 9,11, 9, 6, 5, 6, 8,11,13,13,11,13,11, 7, 6,
	 7, 8,10,12,
};

static static_codebook _huff_book__44c1_sm_short = {
	2, 100,
	_huff_lengthlist__44c1_sm_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c2_s_long[] = {
	 5, 5,12,10,11,10,10,10,11,13, 5, 1, 9, 5, 8, 7,
	 8,10,13,13,12, 9, 8,11, 7, 8, 9,11,13,15, 9, 5,
	12, 6, 9, 8,10,12,15,14,12, 7, 6, 8, 5, 6, 7,10,
	12,13,10, 7, 8, 7, 6, 6, 7, 9,12,12,10, 8, 9, 9,
	 7, 6, 6, 7,10,11,10, 9,10,11, 9, 8, 6, 6, 8,11,
	10,12,13,15,11,10, 8, 7, 8,11,11,13,16,16,14,13,
	11, 9, 9,10,
};

static static_codebook _huff_book__44c2_s_long = {
	2, 100,
	_huff_lengthlist__44c2_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c2_s_p1_0[] = {
	 2, 4, 4, 0, 0, 0, 0, 0, 0, 5, 7, 6, 0, 0, 0, 0,
	 0, 0, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 7, 7, 0, 0, 0, 0,
	 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0,
	 0, 0, 0, 0, 8, 9, 8, 0, 0, 0, 0, 0, 0, 8, 9,10,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 8, 0, 0, 0, 0, 0, 0, 8,10, 9, 0,
	 0, 0, 0, 0, 0, 8, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 8, 9, 9,
	 0, 0, 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c2_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c2_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p1_0 = {
	_vq_quantthresh__44c2_s_p1_0,
	_vq_quantmap__44c2_s_p1_0,
	3,
	3
};

static static_codebook _44c2_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c2_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c2_s_p1_0,
	NULL,
	&_vq_auxt__44c2_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c2_s_p2_0[] = {
	 1, 4, 4, 0, 0, 0, 7, 7, 0, 0, 0, 7, 7, 0, 0, 0,
	 7, 7, 0, 0, 0, 0, 0, 0, 0, 4, 6, 6, 0, 0, 0, 8,
	 8, 0, 0, 0, 8, 8, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0,
	 0, 0, 4, 6, 6, 0, 0, 0, 8, 9, 0, 0, 0, 8, 8, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,11,10, 0, 0,
	 0,11,11, 0, 0, 0,11,11, 0, 0, 0, 0, 0, 0, 0, 7,
	 8, 8, 0, 0, 0,10,11, 0, 0, 0,11,11, 0, 0, 0,11,
	11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 8, 8, 0, 0, 0,11,11, 0, 0, 0,11,11,
	 0, 0, 0,11,11, 0, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0,
	 0, 0,11,11, 0, 0, 0,11,11, 0, 0, 0,11,11, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7, 9, 9, 0, 0, 0,11,12, 0, 0, 0,11,12, 0, 0, 0,
	12,11, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,12,
	11, 0, 0, 0,11,11, 0, 0, 0,11,11, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c2_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c2_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p2_0 = {
	_vq_quantthresh__44c2_s_p2_0,
	_vq_quantmap__44c2_s_p2_0,
	5,
	5
};

static static_codebook _44c2_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c2_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c2_s_p2_0,
	NULL,
	&_vq_auxt__44c2_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c2_s_p3_0[] = {
	 2, 4, 3, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 5, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c2_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c2_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p3_0 = {
	_vq_quantthresh__44c2_s_p3_0,
	_vq_quantmap__44c2_s_p3_0,
	5,
	5
};

static static_codebook _44c2_s_p3_0 = {
	4, 625,
	_vq_lengthlist__44c2_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c2_s_p3_0,
	NULL,
	&_vq_auxt__44c2_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c2_s_p4_0[] = {
	 1, 3, 3, 6, 6, 0, 0, 0, 0, 0, 6, 6, 6, 6, 0, 0,
	 0, 0, 0, 6, 6, 6, 6, 0, 0, 0, 0, 0, 6, 6, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 7, 8, 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c2_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c2_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p4_0 = {
	_vq_quantthresh__44c2_s_p4_0,
	_vq_quantmap__44c2_s_p4_0,
	9,
	9
};

static static_codebook _44c2_s_p4_0 = {
	2, 81,
	_vq_lengthlist__44c2_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c2_s_p4_0,
	NULL,
	&_vq_auxt__44c2_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c2_s_p5_0[] = {
	 1, 3, 3, 6, 6, 6, 6, 8, 8, 0, 7, 7, 7, 7, 8, 8,
	 9, 9, 0, 7, 7, 7, 7, 8, 8, 9, 9, 0, 7, 7, 8, 7,
	 8, 8,10,10, 0, 0, 0, 8, 8, 8, 8,10,10, 0, 0, 0,
	 8, 8, 9, 9,10,10, 0, 0, 0, 8, 8, 9, 9,10,10, 0,
	 0, 0,10,10, 9, 9,11,11, 0, 0, 0, 0, 0, 9, 9,11,
	11,
};

static float _vq_quantthresh__44c2_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c2_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p5_0 = {
	_vq_quantthresh__44c2_s_p5_0,
	_vq_quantmap__44c2_s_p5_0,
	9,
	9
};

static static_codebook _44c2_s_p5_0 = {
	2, 81,
	_vq_lengthlist__44c2_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c2_s_p5_0,
	NULL,
	&_vq_auxt__44c2_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c2_s_p6_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10,10,
	11,11, 0, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9,10,10,10,
	10,11,11, 0, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,11,11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 8, 8, 9, 9, 9, 9, 9, 9,
	10,10,11,11,12,12, 0, 0, 0, 8, 8, 9, 9, 9, 9, 9,
	 9,10,11,11,11,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,11,12,12,12, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 9, 9,
	 9, 9,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 9,
	 9, 9, 9,10,10,11,11,12,12,12,13, 0, 0, 0, 0, 0,
	10,10,10,10,10,11,11,11,12,12,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,10,10,11,11,12,12,13,13, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,12,12,12,12,13,13, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,12,11,12,12,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,11,11,12,12,13,12,14,13,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,11,11,12,12,12,13,13,
	14,
};

static float _vq_quantthresh__44c2_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c2_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p6_0 = {
	_vq_quantthresh__44c2_s_p6_0,
	_vq_quantmap__44c2_s_p6_0,
	17,
	17
};

static static_codebook _44c2_s_p6_0 = {
	2, 289,
	_vq_lengthlist__44c2_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c2_s_p6_0,
	NULL,
	&_vq_auxt__44c2_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c2_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10, 9,10,10,
	10,11,10,10, 6, 9, 9,10,10,10,11, 9, 9, 6, 9, 9,
	10,10,10,11, 9, 9, 7,10,10,11,10,10,11,11,10, 6,
	 9, 9,10, 9, 9,11,10,10, 6, 9, 9,10, 9,10,11,10,
	10,
};

static float _vq_quantthresh__44c2_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c2_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p7_0 = {
	_vq_quantthresh__44c2_s_p7_0,
	_vq_quantmap__44c2_s_p7_0,
	3,
	3
};

static static_codebook _44c2_s_p7_0 = {
	4, 81,
	_vq_lengthlist__44c2_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c2_s_p7_0,
	NULL,
	&_vq_auxt__44c2_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c2_s_p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 7, 7, 7, 7,10, 6, 6, 6, 6,
	 7, 7, 8, 7, 8, 8, 9, 6, 6, 6, 6, 7, 7, 7, 7, 7,
	 7,10, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8,10, 9, 9, 6,
	 6, 7, 7, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10, 9, 8, 8, 8,
	 8, 8, 8,10,10,10, 9,10, 8, 8, 8, 8, 8, 8,10,10,
	10,10, 9, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c2_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c2_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p7_1 = {
	_vq_quantthresh__44c2_s_p7_1,
	_vq_quantmap__44c2_s_p7_1,
	11,
	11
};

static static_codebook _44c2_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c2_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c2_s_p7_1,
	NULL,
	&_vq_auxt__44c2_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c2_s_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10, 6, 5, 5,
	 7, 7, 8, 8, 9, 8, 9, 9,10,10, 7, 5, 5, 7, 7, 8,
	 8, 8, 9,10, 9,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9,10,10,10,11,
	11, 0,12,12, 9, 9,10,10,10,10,11,11,12,12, 0,13,
	12, 9, 9,10, 9,10,10,11,11,11,12, 0, 0, 0,10,10,
	10,10,11,11,11,11,12,12, 0, 0, 0,10,10, 9, 9,11,
	11,12,12,12,13, 0, 0, 0,13,13,10,10,11,11,12,12,
	13,13, 0, 0, 0,14,14,10,10,11,11,12,12,13,13, 0,
	 0, 0, 0, 0,12,12,11,12,12,12,14,13, 0, 0, 0, 0,
	 0,12,12,12,12,12,12,14,13,
};

static float _vq_quantthresh__44c2_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c2_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p8_0 = {
	_vq_quantthresh__44c2_s_p8_0,
	_vq_quantmap__44c2_s_p8_0,
	13,
	13
};

static static_codebook _44c2_s_p8_0 = {
	2, 169,
	_vq_lengthlist__44c2_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c2_s_p8_0,
	NULL,
	&_vq_auxt__44c2_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c2_s_p8_1[] = {
	 2, 4, 4, 4, 5, 6, 5, 5, 5, 5, 6, 5, 5, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c2_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c2_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p8_1 = {
	_vq_quantthresh__44c2_s_p8_1,
	_vq_quantmap__44c2_s_p8_1,
	5,
	5
};

static static_codebook _44c2_s_p8_1 = {
	2, 25,
	_vq_lengthlist__44c2_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c2_s_p8_1,
	NULL,
	&_vq_auxt__44c2_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c2_s_p9_0[] = {
	 1, 5, 4,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 5, 7, 7,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 5, 9, 6, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44c2_s_p9_0[] = {
	-1215.5, -994.5, -773.5, -552.5, -331.5, -110.5, 110.5, 331.5, 
	552.5, 773.5, 994.5, 1215.5, 
};

static long _vq_quantmap__44c2_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p9_0 = {
	_vq_quantthresh__44c2_s_p9_0,
	_vq_quantmap__44c2_s_p9_0,
	13,
	13
};

static static_codebook _44c2_s_p9_0 = {
	2, 169,
	_vq_lengthlist__44c2_s_p9_0,
	1, -514541568, 1627103232, 4, 0,
	_vq_quantlist__44c2_s_p9_0,
	NULL,
	&_vq_auxt__44c2_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c2_s_p9_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 9, 9,11,11,12,12, 6, 5, 5,
	 7, 7, 8, 8,10,10,12,11,13,13, 6, 5, 5, 7, 7, 8,
	 8, 9, 9,11,11,12,13,16, 7, 7, 8, 8, 9, 9,10,10,
	13,13,16,14,16, 7, 7, 8, 8, 9, 9,10,10,13,13,15,
	13,16,10,10, 8, 8,10,10,11,11,12,12,15,14,16,11,
	11, 8, 8,10,10,11,11,12,12,14,13,16,16,16,10,11,
	10,10,12,12,14,13,13,13,16,16,16,10, 9,10, 7,13,
	12,13,13,13,13,16,16,16,14,15,11,11,12,12,13,13,
	15,14,16,16,16,16,14,11, 9,12, 9,15,13,13,14,16,
	16,16,16,16,13,13,12,12,14,15,16,15,16,16,15,16,
	16,13,12,12,11,15,13,14,15,
};

static float _vq_quantthresh__44c2_s_p9_1[] = {
	-93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 25.5, 
	42.5, 59.5, 76.5, 93.5, 
};

static long _vq_quantmap__44c2_s_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p9_1 = {
	_vq_quantthresh__44c2_s_p9_1,
	_vq_quantmap__44c2_s_p9_1,
	13,
	13
};

static static_codebook _44c2_s_p9_1 = {
	2, 169,
	_vq_lengthlist__44c2_s_p9_1,
	1, -522616832, 1620115456, 4, 0,
	_vq_quantlist__44c2_s_p9_1,
	NULL,
	&_vq_auxt__44c2_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c2_s_p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c2_s_p9_2[] = {
	 2, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9,
	 9,10, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9,10, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9,10, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 7, 8, 8, 9, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10,11,10, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10,10,10, 7, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,11,11, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10, 8, 8, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,11,11,10,11,10, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,11,10, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9,10,10,10,10,10,11,11,
	 9, 9, 9, 9, 9, 9, 9, 9,10, 9, 9, 9,11,10,10,11,
	10,10,11, 9, 9,10,10, 9, 9,10,10, 9, 9,10,10,10,
	11,10,11,10,10, 9, 9,10, 9, 9, 9, 9, 9, 9,10,10,
	10,10,11,10,10, 9,10, 9, 9, 9, 9,10, 9, 9, 9,10,
	10,10,11,11,10,10,10,10, 9, 9,10, 9, 9, 9,10, 9,
	10,10,10,10,11,10,11,11,10, 9, 9, 9, 9, 9, 9, 9,
	10,
};

static float _vq_quantthresh__44c2_s_p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c2_s_p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c2_s_p9_2 = {
	_vq_quantthresh__44c2_s_p9_2,
	_vq_quantmap__44c2_s_p9_2,
	17,
	17
};

static static_codebook _44c2_s_p9_2 = {
	2, 289,
	_vq_lengthlist__44c2_s_p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c2_s_p9_2,
	NULL,
	&_vq_auxt__44c2_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c2_s_short[] = {
	 9, 9,12,11,11,11,12,12,12,13, 6, 2, 9, 4, 7, 6,
	 8,11,15,17,12, 7, 8, 9, 7, 9,10,13,15,17,11, 4,
	12, 4, 9, 5, 8,11,16,17,12, 6, 7, 6, 6, 6, 8,11,
	15,17,11, 5, 9, 5, 6, 5, 6,10,15,15,12, 7,11, 7,
	 7, 6, 7,10,13,16,13, 8,11, 9, 8, 6, 7,10,12,16,
	13, 9, 9, 8, 5, 5, 6, 9,12,14,16,10, 9, 9, 6, 5,
	 6, 8,11,14,
};

static static_codebook _huff_book__44c2_s_short = {
	2, 100,
	_huff_lengthlist__44c2_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c3_s_long[] = {
	 4, 5,11,10,12,10,10,10,11,12, 4, 2,11, 5,11, 6,
	 7, 9,13,16,11,11, 7,11, 6, 8, 8, 9,11,12,10, 5,
	11, 6,10, 7, 9,11,16,16,14, 9, 6, 9, 4, 5, 7, 8,
	11,13,10, 6, 7, 7, 5, 5, 6, 8,10,11,10, 7, 8, 8,
	 6, 6, 5, 7, 9,10,10, 9, 8,11, 8, 7, 6, 6, 7,10,
	10,12,10,14, 9, 9, 7, 7, 7, 9,10,14,11,16,12,12,
	 9, 8, 8, 9,
};

static static_codebook _huff_book__44c3_s_long = {
	2, 100,
	_huff_lengthlist__44c3_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c3_s_p1_0[] = {
	 2, 4, 4, 0, 0, 0, 0, 0, 0, 5, 7, 6, 0, 0, 0, 0,
	 0, 0, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 7, 7, 0, 0, 0, 0,
	 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0,
	 0, 0, 0, 0, 8, 9, 8, 0, 0, 0, 0, 0, 0, 8, 9,10,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 8, 0, 0, 0, 0, 0, 0, 8,10, 9, 0,
	 0, 0, 0, 0, 0, 8, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 8, 9, 9,
	 0, 0, 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c3_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c3_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p1_0 = {
	_vq_quantthresh__44c3_s_p1_0,
	_vq_quantmap__44c3_s_p1_0,
	3,
	3
};

static static_codebook _44c3_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c3_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c3_s_p1_0,
	NULL,
	&_vq_auxt__44c3_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c3_s_p2_0[] = {
	 2, 5, 5, 0, 0, 0, 5, 5, 0, 0, 0, 5, 5, 0, 0, 0,
	 8, 7, 0, 0, 0, 0, 0, 0, 0, 5, 6, 6, 0, 0, 0, 8,
	 7, 0, 0, 0, 7, 7, 0, 0, 0,10, 9, 0, 0, 0, 0, 0,
	 0, 0, 5, 5, 6, 0, 0, 0, 7, 8, 0, 0, 0, 7, 7, 0,
	 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 7, 7, 0, 0,
	 0, 7, 8, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 5,
	 7, 7, 0, 0, 0, 7, 7, 0, 0, 0, 7, 7, 0, 0, 0, 9,
	 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 5, 8, 7, 0, 0, 0, 7, 7, 0, 0, 0, 7, 7,
	 0, 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 5, 7, 8, 0,
	 0, 0, 7, 7, 0, 0, 0, 7, 7, 0, 0, 0, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 8, 9,10, 0, 0, 0, 9, 9, 0, 0, 0, 9, 9, 0, 0, 0,
	10,10, 0, 0, 0, 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 9,
	 9, 0, 0, 0, 9, 9, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c3_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c3_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p2_0 = {
	_vq_quantthresh__44c3_s_p2_0,
	_vq_quantmap__44c3_s_p2_0,
	5,
	5
};

static static_codebook _44c3_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c3_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c3_s_p2_0,
	NULL,
	&_vq_auxt__44c3_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c3_s_p3_0[] = {
	 2, 3, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 4, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 8, 8,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c3_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c3_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p3_0 = {
	_vq_quantthresh__44c3_s_p3_0,
	_vq_quantmap__44c3_s_p3_0,
	5,
	5
};

static static_codebook _44c3_s_p3_0 = {
	4, 625,
	_vq_lengthlist__44c3_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c3_s_p3_0,
	NULL,
	&_vq_auxt__44c3_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c3_s_p4_0[] = {
	 2, 3, 3, 6, 6, 0, 0, 0, 0, 0, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 4, 4, 6, 6, 0, 0, 0, 0, 0, 5, 5, 6, 6,
	 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0,
	 7, 7, 0, 0, 0, 0, 0, 0, 0, 8, 7, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c3_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c3_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p4_0 = {
	_vq_quantthresh__44c3_s_p4_0,
	_vq_quantmap__44c3_s_p4_0,
	9,
	9
};

static static_codebook _44c3_s_p4_0 = {
	2, 81,
	_vq_lengthlist__44c3_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c3_s_p4_0,
	NULL,
	&_vq_auxt__44c3_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c3_s_p5_0[] = {
	 2, 3, 3, 5, 5, 7, 7, 9, 8, 0, 4, 4, 7, 7, 7, 7,
	 9, 9, 0, 5, 5, 6, 7, 7, 7, 9, 9, 0, 6, 6, 7, 7,
	 7, 7, 9, 9, 0, 0, 0, 7, 7, 7, 7, 9, 9, 0, 0, 0,
	 8, 8, 8, 8,10,10, 0, 0, 0, 8, 8, 8, 8,10,10, 0,
	 0, 0,10,10, 9, 9,10,10, 0, 0, 0, 0, 0, 9, 9,10,
	10,
};

static float _vq_quantthresh__44c3_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c3_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p5_0 = {
	_vq_quantthresh__44c3_s_p5_0,
	_vq_quantmap__44c3_s_p5_0,
	9,
	9
};

static static_codebook _44c3_s_p5_0 = {
	2, 81,
	_vq_lengthlist__44c3_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c3_s_p5_0,
	NULL,
	&_vq_auxt__44c3_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c3_s_p6_0[] = {
	 2, 3, 3, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 5, 5, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	11,11, 0, 5, 5, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,
	11,11,11, 0, 6, 6, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 8, 8, 8, 8, 9, 9,10, 9,
	10,10,11,11,12,12, 0, 0, 0, 8, 8, 8, 8, 9, 9,10,
	 9,10,10,11,11,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,12,11,12,12, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 8, 8,
	 9, 9,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 8,
	 8, 9, 9,10,10,11,11,12,12,13,12, 0, 0, 0, 0, 0,
	 9, 9,10,10,11,10,11,11,12,12,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,10,10,11,11,12,12,13,13, 0, 0, 0,
	 0, 0, 0, 0,10,10,11,11,12,12,12,12,13,13, 0, 0,
	 0, 0, 0, 0, 0,10,10,11,11,11,11,12,12,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,12,12,12,12,12,13,13,13,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,11,11,12,12,12,12,13,
	13,
};

static float _vq_quantthresh__44c3_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c3_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p6_0 = {
	_vq_quantthresh__44c3_s_p6_0,
	_vq_quantmap__44c3_s_p6_0,
	17,
	17
};

static static_codebook _44c3_s_p6_0 = {
	2, 289,
	_vq_lengthlist__44c3_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c3_s_p6_0,
	NULL,
	&_vq_auxt__44c3_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c3_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10,10,10,10,
	10,11,10,10, 6, 9, 9,10,10,10,10, 9, 9, 6, 9, 9,
	10,10,10,10, 9, 9, 7,10,10,10,11,11,11,10,11, 6,
	 9, 9,10,10, 9,11,10,10, 6, 9, 9,10, 9, 9,10,10,
	10,
};

static float _vq_quantthresh__44c3_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c3_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p7_0 = {
	_vq_quantthresh__44c3_s_p7_0,
	_vq_quantmap__44c3_s_p7_0,
	3,
	3
};

static static_codebook _44c3_s_p7_0 = {
	4, 81,
	_vq_lengthlist__44c3_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c3_s_p7_0,
	NULL,
	&_vq_auxt__44c3_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c3_s_p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6,
	 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6, 7, 7, 8, 8, 8,
	 8,10, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10, 7,
	 7, 8, 8, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 8, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 9,
};

static float _vq_quantthresh__44c3_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c3_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p7_1 = {
	_vq_quantthresh__44c3_s_p7_1,
	_vq_quantmap__44c3_s_p7_1,
	11,
	11
};

static static_codebook _44c3_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c3_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c3_s_p7_1,
	NULL,
	&_vq_auxt__44c3_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c3_s_p8_0[] = {
	 1, 4, 4, 6, 6, 7, 8, 8, 8, 9, 9,10,10, 6, 5, 5,
	 7, 7, 8, 8, 9, 9, 9, 9,10,10, 7, 5, 5, 7, 7, 8,
	 8, 9, 9, 9,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,11,12, 9, 9, 9,10,10,10,11,11,11,12, 0,13,
	13, 9, 9, 9, 9,10,10,11,11,12,12, 0, 0, 0,10, 9,
	 9, 9,10,11,11,11,12,12, 0, 0, 0,10,10, 9, 9,11,
	11,11,11,12,13, 0, 0, 0,13,13,10,10,11,11,12,12,
	13,13, 0, 0, 0,14,14,10,10,11,11,12,12,13,13, 0,
	 0, 0, 0, 0,12,12,11,11,12,12,13,13, 0, 0, 0, 0,
	 0,12,12,11,11,12,12,13,13,
};

static float _vq_quantthresh__44c3_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c3_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p8_0 = {
	_vq_quantthresh__44c3_s_p8_0,
	_vq_quantmap__44c3_s_p8_0,
	13,
	13
};

static static_codebook _44c3_s_p8_0 = {
	2, 169,
	_vq_lengthlist__44c3_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c3_s_p8_0,
	NULL,
	&_vq_auxt__44c3_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c3_s_p8_1[] = {
	 2, 4, 4, 5, 5, 6, 5, 5, 5, 5, 6, 4, 5, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c3_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c3_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p8_1 = {
	_vq_quantthresh__44c3_s_p8_1,
	_vq_quantmap__44c3_s_p8_1,
	5,
	5
};

static static_codebook _44c3_s_p8_1 = {
	2, 25,
	_vq_lengthlist__44c3_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c3_s_p8_1,
	NULL,
	&_vq_auxt__44c3_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c3_s_p9_0[] = {
	 1, 4, 4,10,10,10,10,10,10,10,10,10,10, 5,10, 7,
	10,10,10,10,10,10,10,10,10,10, 5, 8, 6,10,10,10,
	10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44c3_s_p9_0[] = {
	-1402.5, -1147.5, -892.5, -637.5, -382.5, -127.5, 127.5, 382.5, 
	637.5, 892.5, 1147.5, 1402.5, 
};

static long _vq_quantmap__44c3_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p9_0 = {
	_vq_quantthresh__44c3_s_p9_0,
	_vq_quantmap__44c3_s_p9_0,
	13,
	13
};

static static_codebook _44c3_s_p9_0 = {
	2, 169,
	_vq_lengthlist__44c3_s_p9_0,
	1, -514332672, 1627381760, 4, 0,
	_vq_quantlist__44c3_s_p9_0,
	NULL,
	&_vq_auxt__44c3_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c3_s_p9_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 9, 9,11,11,11,12,12,12, 6,
	 5, 5, 7, 7, 8, 8,10, 9,11,11,13,12,13,14, 6, 5,
	 5, 7, 7, 8, 8,10,10,11,11,12,12,13,13,17, 7, 7,
	 8, 8, 9, 9,10,10,12,12,14,13,14,14,17, 8, 7, 8,
	 7, 9, 9,10,10,12,12,13,13,13,14,17,11,11, 8, 8,
	10,10,11,11,12,12,13,13,15,14,17,11,11, 8, 7,10,
	10,11,11,12,12,13,14,14,13,17,17,17,10,11,10,10,
	12,12,13,12,13,13,14,14,17,16,16,10,10,11, 9,13,
	12,13,13,13,13,14,14,16,16,15,13,15,11,12,12,12,
	14,14,14,14,14,15,16,16,16,14,14,11, 9,12,10,13,
	13,14,14,14,14,16,16,16,16,16,12,13,12,12,13,14,
	14,14,15,15,15,16,16,15,16,13,11,13,10,14,12,15,
	14,16,14,15,16,16,16,16,15,15,13,13,13,13,14,14,
	16,16,16,16,16,15,16,16,14,13,12,13,13,14,16,16,
	16,
};

static float _vq_quantthresh__44c3_s_p9_1[] = {
	-110.5, -93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 
	25.5, 42.5, 59.5, 76.5, 93.5, 110.5, 
};

static long _vq_quantmap__44c3_s_p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p9_1 = {
	_vq_quantthresh__44c3_s_p9_1,
	_vq_quantmap__44c3_s_p9_1,
	15,
	15
};

static static_codebook _44c3_s_p9_1 = {
	2, 225,
	_vq_lengthlist__44c3_s_p9_1,
	1, -522338304, 1620115456, 4, 0,
	_vq_quantlist__44c3_s_p9_1,
	NULL,
	&_vq_auxt__44c3_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c3_s_p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c3_s_p9_2[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9,
	 8,10, 5, 5, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9,10, 5, 5, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9,
	 9, 9, 9,10, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10,10,10, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10,10,10, 8, 7, 8, 8, 9, 8, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,10, 8, 8, 8, 8, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,11,10, 8, 8, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,11, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,11,10,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,
	10,10,11, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,
	11,10,11,10, 9, 9, 9,10, 9, 9, 9, 9, 9, 9,10,10,
	10,10,11,11,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,
	10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9,10, 9,
	10,10,10,10,11,10,11,10,10, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44c3_s_p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c3_s_p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c3_s_p9_2 = {
	_vq_quantthresh__44c3_s_p9_2,
	_vq_quantmap__44c3_s_p9_2,
	17,
	17
};

static static_codebook _44c3_s_p9_2 = {
	2, 289,
	_vq_lengthlist__44c3_s_p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c3_s_p9_2,
	NULL,
	&_vq_auxt__44c3_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c3_s_short[] = {
	10,10,13,12,13,12,12,12,12,13, 8, 3,11, 5,10, 5,
	 7,11,14,16,11, 6, 9, 8, 7, 7, 9,12,15,16,12, 4,
	12, 4,10, 5, 8,12,15,16,12, 6, 8, 7, 5, 5, 7,11,
	14,15,11, 4, 9, 4, 6, 4, 6, 9,13,15,10, 6,10, 7,
	 7, 5, 6, 9,13,15,12, 9,11, 9, 8, 6, 7, 9,12,15,
	13,11,10, 9, 6, 5, 5, 8,11,14,16,12,11,10, 6, 5,
	 6, 8,10,14,
};

static static_codebook _huff_book__44c3_s_short = {
	2, 100,
	_huff_lengthlist__44c3_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c4_s_long[] = {
	 3, 5,11,11,13,11,11,11,12,12, 5, 2,11, 6,10, 7,
	 8,10,13,16,10, 9, 6,10, 6, 7, 8, 9,11,12,11, 5,
	11, 7,10, 8,10,12,15,17,12, 8, 5, 9, 4, 5, 7, 8,
	10,12,10, 6, 7, 8, 5, 5, 6, 8,10,11,10, 8, 8, 9,
	 6, 6, 6, 7, 9,10,11,10, 9,11, 8, 7, 6, 6, 7, 9,
	11,13,10,15, 9, 9, 7, 7, 7, 8,10,15,11,17,11,11,
	 9, 8, 7, 8,
};

static static_codebook _huff_book__44c4_s_long = {
	2, 100,
	_huff_lengthlist__44c4_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c4_s_p1_0[] = {
	 2, 4, 4, 0, 0, 0, 0, 0, 0, 5, 7, 6, 0, 0, 0, 0,
	 0, 0, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0, 0,
	 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 7, 7, 0, 0, 0, 0,
	 0, 0, 7, 9, 8, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0, 0,
	 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 8, 8, 0, 0,
	 0, 0, 0, 0, 8,10, 8, 0, 0, 0, 0, 0, 0, 8, 9,10,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 7, 7, 0, 0, 0, 0, 0, 0, 7, 8, 8, 0, 0,
	 0, 0, 0, 0, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 8, 0, 0, 0, 0, 0, 0, 8,10, 9, 0,
	 0, 0, 0, 0, 0, 8, 8,10, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9,
	 0, 0, 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c4_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c4_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p1_0 = {
	_vq_quantthresh__44c4_s_p1_0,
	_vq_quantmap__44c4_s_p1_0,
	3,
	3
};

static static_codebook _44c4_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c4_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c4_s_p1_0,
	NULL,
	&_vq_auxt__44c4_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c4_s_p2_0[] = {
	 2, 5, 5, 0, 0, 0, 5, 5, 0, 0, 0, 5, 5, 0, 0, 0,
	 7, 7, 0, 0, 0, 0, 0, 0, 0, 5, 6, 6, 0, 0, 0, 8,
	 7, 0, 0, 0, 7, 7, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 5, 6, 6, 0, 0, 0, 7, 7, 0, 0, 0, 7, 7, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 7, 7, 0, 0,
	 0, 7, 7, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 5,
	 7, 7, 0, 0, 0, 7, 7, 0, 0, 0, 7, 7, 0, 0, 0, 9,
	 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 5, 8, 7, 0, 0, 0, 7, 7, 0, 0, 0, 7, 7,
	 0, 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 5, 7, 8, 0,
	 0, 0, 7, 7, 0, 0, 0, 7, 7, 0, 0, 0, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7,10,10, 0, 0, 0, 9, 9, 0, 0, 0, 9, 9, 0, 0, 0,
	10,10, 0, 0, 0, 0, 0, 0, 0, 7,10,10, 0, 0, 0, 9,
	 9, 0, 0, 0, 9, 9, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c4_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c4_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p2_0 = {
	_vq_quantthresh__44c4_s_p2_0,
	_vq_quantmap__44c4_s_p2_0,
	5,
	5
};

static static_codebook _44c4_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c4_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c4_s_p2_0,
	NULL,
	&_vq_auxt__44c4_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c4_s_p3_0[] = {
	 2, 3, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c4_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c4_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p3_0 = {
	_vq_quantthresh__44c4_s_p3_0,
	_vq_quantmap__44c4_s_p3_0,
	5,
	5
};

static static_codebook _44c4_s_p3_0 = {
	4, 625,
	_vq_lengthlist__44c4_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c4_s_p3_0,
	NULL,
	&_vq_auxt__44c4_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c4_s_p4_0[] = {
	 2, 3, 3, 6, 6, 0, 0, 0, 0, 0, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 4, 4, 6, 6, 0, 0, 0, 0, 0, 5, 5, 6, 6,
	 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0,
	 7, 8, 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c4_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c4_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p4_0 = {
	_vq_quantthresh__44c4_s_p4_0,
	_vq_quantmap__44c4_s_p4_0,
	9,
	9
};

static static_codebook _44c4_s_p4_0 = {
	2, 81,
	_vq_lengthlist__44c4_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c4_s_p4_0,
	NULL,
	&_vq_auxt__44c4_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c4_s_p5_0[] = {
	 2, 3, 3, 6, 6, 7, 7, 9, 9, 0, 4, 4, 7, 7, 7, 7,
	 9, 9, 0, 4, 4, 7, 7, 7, 7, 9, 9, 0, 6, 6, 7, 7,
	 8, 8,10,10, 0, 0, 0, 7, 7, 8, 8,10,10, 0, 0, 0,
	 8, 8, 8, 8,10,10, 0, 0, 0, 8, 8, 8, 8,10,10, 0,
	 0, 0,10,10, 9, 9,11,11, 0, 0, 0, 0, 0, 9, 9,11,
	11,
};

static float _vq_quantthresh__44c4_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c4_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p5_0 = {
	_vq_quantthresh__44c4_s_p5_0,
	_vq_quantmap__44c4_s_p5_0,
	9,
	9
};

static static_codebook _44c4_s_p5_0 = {
	2, 81,
	_vq_lengthlist__44c4_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c4_s_p5_0,
	NULL,
	&_vq_auxt__44c4_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c4_s_p6_0[] = {
	 2, 4, 4, 6, 6, 8, 8, 8, 8, 9, 9,10, 9,10,10,11,
	11, 0, 4, 4, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	11,11, 0, 4, 4, 6, 6, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,12, 0, 6, 6, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 8, 8, 8, 8, 9, 9,10, 9,
	10,11,11,11,12,12, 0, 0, 0, 8, 8, 8, 8, 9, 9,10,
	10,11,10,11,11,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,
	10,10,11,11,11,11,12,12, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,11,12,12, 0, 0, 0, 0, 0, 8, 8,
	 9, 9,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0, 8,
	 8, 9, 9,10,10,11,11,12,12,12,12, 0, 0, 0, 0, 0,
	 9, 9,10,10,11,10,11,11,12,12,12,12, 0, 0, 0, 0,
	 0, 0, 0,10,10,10,10,11,11,12,12,13,12, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,11,12,12,12,13,12, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,11,11,12,12,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,12,12,12,12,12,12,13,13,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,11,11,12,12,12,12,13,
	13,
};

static float _vq_quantthresh__44c4_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c4_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p6_0 = {
	_vq_quantthresh__44c4_s_p6_0,
	_vq_quantmap__44c4_s_p6_0,
	17,
	17
};

static static_codebook _44c4_s_p6_0 = {
	2, 289,
	_vq_lengthlist__44c4_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c4_s_p6_0,
	NULL,
	&_vq_auxt__44c4_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c4_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10,10,10,10,
	10,11,10,10, 6, 9, 9,10,10,10,10,10, 9, 6, 9, 9,
	10, 9,10,11, 9, 9, 7,10,10,11,11,11,11,10,10, 6,
	 9, 9,10, 9, 9,10,10, 9, 6, 9, 9,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44c4_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c4_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p7_0 = {
	_vq_quantthresh__44c4_s_p7_0,
	_vq_quantmap__44c4_s_p7_0,
	3,
	3
};

static static_codebook _44c4_s_p7_0 = {
	4, 81,
	_vq_lengthlist__44c4_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c4_s_p7_0,
	NULL,
	&_vq_auxt__44c4_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c4_s_p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6,
	 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6, 7, 7, 8, 8, 8,
	 8,10, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10, 7,
	 7, 8, 8, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 8, 9,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c4_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c4_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p7_1 = {
	_vq_quantthresh__44c4_s_p7_1,
	_vq_quantmap__44c4_s_p7_1,
	11,
	11
};

static static_codebook _44c4_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c4_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c4_s_p7_1,
	NULL,
	&_vq_auxt__44c4_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c4_s_p8_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 8, 8, 9, 9,10,10, 6, 5, 5,
	 7, 7, 8, 8, 9, 9,10,10,11,11, 7, 5, 5, 7, 7, 8,
	 8, 9, 9,10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,12,12, 9, 9, 9, 9,10,10,10,10,11,11, 0,12,
	12, 9, 8, 9, 9,10,10,11,11,12,11, 0, 0, 0, 9,10,
	 9, 9,10,10,11,11,12,12, 0, 0, 0,10,10, 9, 9,10,
	10,11,11,12,12, 0, 0, 0,13,13,10,10,10,11,12,12,
	12,12, 0, 0, 0,14,14,10,10,11,11,11,11,12,12, 0,
	 0, 0, 0, 0,11,12,11,11,12,12,12,13, 0, 0, 0, 0,
	 0,12,12,11,11,12,12,13,13,
};

static float _vq_quantthresh__44c4_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c4_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p8_0 = {
	_vq_quantthresh__44c4_s_p8_0,
	_vq_quantmap__44c4_s_p8_0,
	13,
	13
};

static static_codebook _44c4_s_p8_0 = {
	2, 169,
	_vq_lengthlist__44c4_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c4_s_p8_0,
	NULL,
	&_vq_auxt__44c4_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c4_s_p8_1[] = {
	 2, 4, 4, 5, 5, 6, 5, 5, 5, 5, 6, 4, 5, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c4_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c4_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p8_1 = {
	_vq_quantthresh__44c4_s_p8_1,
	_vq_quantmap__44c4_s_p8_1,
	5,
	5
};

static static_codebook _44c4_s_p8_1 = {
	2, 25,
	_vq_lengthlist__44c4_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c4_s_p8_1,
	NULL,
	&_vq_auxt__44c4_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c4_s_p9_0[] = {
	 1, 3, 3,10,10,10,10,10,10,10,10,10,10, 5, 7, 7,
	10,10,10,10,10,10,10,10,10,10, 5, 7, 8,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10, 9, 9, 9, 9,
};

static float _vq_quantthresh__44c4_s_p9_0[] = {
	-1732.5, -1417.5, -1102.5, -787.5, -472.5, -157.5, 157.5, 472.5, 
	787.5, 1102.5, 1417.5, 1732.5, 
};

static long _vq_quantmap__44c4_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p9_0 = {
	_vq_quantthresh__44c4_s_p9_0,
	_vq_quantmap__44c4_s_p9_0,
	13,
	13
};

static static_codebook _44c4_s_p9_0 = {
	2, 169,
	_vq_lengthlist__44c4_s_p9_0,
	1, -513964032, 1628680192, 4, 0,
	_vq_quantlist__44c4_s_p9_0,
	NULL,
	&_vq_auxt__44c4_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c4_s_p9_1[] = {
	 1, 4, 4, 5, 6, 7, 7, 9, 9,11,11,12,12,12,13, 6,
	 5, 5, 6, 7, 8, 8,10,10,11,11,13,13,13,13, 6, 5,
	 5, 7, 7, 8, 8,11,10,11,11,12,13,12,13,17, 7, 7,
	 8, 8, 9, 9,11,11,12,12,12,13,15,14,17, 7, 7, 8,
	 8, 9, 9,11,10,12,12,13,13,14,13,17,11,12, 9, 9,
	10,10,12,12,13,13,14,14,14,14,17,12,11, 9, 8,11,
	10,11,12,13,13,13,14,14,14,17,17,17,11,11,11,11,
	13,13,13,13,14,13,15,14,17,17,17,11,10,11, 9,13,
	12,13,14,15,14,15,14,17,17,17,14,14,11,12,12,13,
	14,14,16,15,15,15,17,17,17,15,15,12,11,13,11,13,
	13,13,15,16,14,17,17,17,17,17,13,13,14,13,14,14,
	15,15,16,15,17,17,17,17,17,14,14,14,12,14,12,16,
	14,15,14,17,17,17,17,17,16,17,13,14,14,15,15,15,
	17,15,17,17,17,17,17,16,17,13,14,14,14,15,17,15,
	15,
};

static float _vq_quantthresh__44c4_s_p9_1[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__44c4_s_p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p9_1 = {
	_vq_quantthresh__44c4_s_p9_1,
	_vq_quantmap__44c4_s_p9_1,
	15,
	15
};

static static_codebook _44c4_s_p9_1 = {
	2, 225,
	_vq_lengthlist__44c4_s_p9_1,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__44c4_s_p9_1,
	NULL,
	&_vq_auxt__44c4_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c4_s_p9_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__44c4_s_p9_2[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9,11, 5, 5, 7, 7, 7, 7, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,11, 5, 5, 7, 7, 7,
	 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,
	 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,10,10,
	10,10,10,10,11,11,11, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9,10,10,10,10,10,10,11,11,11, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,11,11,
	11, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,10, 9,10,10,10,
	10,10,10,11,11,11, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9,
	10,10,10,10,10,10,10,10,11,11,11,11,11, 8, 8, 9,
	 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,11,11,11,
	11,11, 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,
	10,10,11,11,11,11,11, 9, 9, 9, 9, 9,10,10,10,10,
	10,10,10,10,10,10,10,11,11,11,11,11, 9, 9, 9, 9,
	10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,
	11,11,11, 9, 9, 9,10,10,10,10,10,10,10,10,10,10,
	10,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,11,11,11,11,11,11,11,10,10, 9,
	10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,
	11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,
	10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,
	10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,
	12,11,11,10,10,10,10,10,10,10,10,10,10,10,10,12,
	11,12,12,11,11,11,11,11,10,10,10,10,10,10,10,10,
	10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,10,
	10,10,10,10,10,10,10,10,10,
};

static float _vq_quantthresh__44c4_s_p9_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__44c4_s_p9_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__44c4_s_p9_2 = {
	_vq_quantthresh__44c4_s_p9_2,
	_vq_quantmap__44c4_s_p9_2,
	21,
	21
};

static static_codebook _44c4_s_p9_2 = {
	2, 441,
	_vq_lengthlist__44c4_s_p9_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__44c4_s_p9_2,
	NULL,
	&_vq_auxt__44c4_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c4_s_short[] = {
	 4, 9,13,12,16,11,12,15,15,16, 4, 2,11, 5,10, 6,
	 8,11,14,14,13,11, 8,11, 7, 8,10,13,17,17,10, 4,
	11, 5, 9, 6, 9,13,17,17,13, 9, 6, 9, 5, 5, 7,11,
	15,17,10, 5, 7, 6, 5, 4, 7,10,15,15,10, 7, 9, 8,
	 7, 6, 7,10,14,13,13,10,11,10, 8, 7, 8,10,14,14,
	12,11,10, 9, 6, 5, 6, 9,13,17,14,13,11,10, 6, 6,
	 6, 8,11,16,
};

static static_codebook _huff_book__44c4_s_short = {
	2, 100,
	_huff_lengthlist__44c4_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c5_s_long[] = {
	 3, 6, 9,14,11,13,12,12,12,12, 6, 3, 5, 8, 6, 8,
	10,11,12,14, 9, 5, 4,10, 5, 7, 8, 9,11,12,13, 8,
	10, 9, 9, 9,12,15,16,17,10, 6, 5, 9, 3, 5, 6, 8,
	10,12,10, 8, 7, 9, 5, 5, 6, 8,10,11,10, 9, 8,11,
	 6, 6, 6, 7, 8,10,12,11, 9,13, 8, 7, 6, 6, 7, 9,
	11,13,10,15, 9, 9, 7, 7, 7, 8,10,15,10,17,11,10,
	 9, 8, 7, 7,
};

static static_codebook _huff_book__44c5_s_long = {
	2, 100,
	_huff_lengthlist__44c5_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c5_s_p1_0[] = {
	 2, 4, 4, 0, 0, 0, 0, 0, 0, 4, 7, 6, 0, 0, 0, 0,
	 0, 0, 4, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 7, 7, 0, 0, 0, 0,
	 0, 0, 7, 9, 8, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 9,10,11, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 9, 8, 0, 0,
	 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 9,10,11,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 7, 7, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 9, 8, 0, 0, 0, 0, 0, 0, 9,11,10, 0,
	 0, 0, 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 9,11,10, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c5_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c5_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p1_0 = {
	_vq_quantthresh__44c5_s_p1_0,
	_vq_quantmap__44c5_s_p1_0,
	3,
	3
};

static static_codebook _44c5_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__44c5_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c5_s_p1_0,
	NULL,
	&_vq_auxt__44c5_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c5_s_p2_0[] = {
	 2, 4, 4, 0, 0, 0, 6, 5, 0, 0, 0, 5, 5, 0, 0, 0,
	 7, 7, 0, 0, 0, 0, 0, 0, 0, 4, 6, 5, 0, 0, 0, 8,
	 7, 0, 0, 0, 8, 8, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0,
	 0, 0, 4, 5, 6, 0, 0, 0, 7, 8, 0, 0, 0, 7, 8, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 5, 8, 7, 0, 0, 0, 8, 8, 0, 0,
	 0, 8, 8, 0, 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 6,
	 7, 8, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0,10,
	10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 7, 0, 0, 0, 8, 8, 0, 0, 0, 8, 8,
	 0, 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 6, 7, 8, 0,
	 0, 0, 8, 8, 0, 0, 0, 8, 8, 0, 0, 0,10,10, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7, 9, 9, 0, 0, 0,10,10, 0, 0, 0,10,10, 0, 0, 0,
	10,10, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,10,
	10, 0, 0, 0,10,10, 0, 0, 0,10,10, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c5_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c5_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p2_0 = {
	_vq_quantthresh__44c5_s_p2_0,
	_vq_quantmap__44c5_s_p2_0,
	5,
	5
};

static static_codebook _44c5_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c5_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c5_s_p2_0,
	NULL,
	&_vq_auxt__44c5_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c5_s_p3_0[] = {
	 2, 4, 3, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 5, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c5_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c5_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p3_0 = {
	_vq_quantthresh__44c5_s_p3_0,
	_vq_quantmap__44c5_s_p3_0,
	5,
	5
};

static static_codebook _44c5_s_p3_0 = {
	4, 625,
	_vq_lengthlist__44c5_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c5_s_p3_0,
	NULL,
	&_vq_auxt__44c5_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c5_s_p4_0[] = {
	 2, 3, 3, 6, 6, 0, 0, 0, 0, 0, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 4, 4, 6, 6, 0, 0, 0, 0, 0, 5, 5, 6, 6,
	 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0,
	 8, 7, 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0,
	 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c5_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c5_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p4_0 = {
	_vq_quantthresh__44c5_s_p4_0,
	_vq_quantmap__44c5_s_p4_0,
	9,
	9
};

static static_codebook _44c5_s_p4_0 = {
	2, 81,
	_vq_lengthlist__44c5_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c5_s_p4_0,
	NULL,
	&_vq_auxt__44c5_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c5_s_p5_0[] = {
	 2, 3, 4, 5, 6, 7, 7, 9, 9, 0, 4, 4, 6, 6, 7, 7,
	 9, 9, 0, 4, 4, 6, 6, 7, 7, 9, 9, 0, 6, 6, 7, 7,
	 8, 8,10,10, 0, 0, 0, 7, 7, 8, 8,10,10, 0, 0, 0,
	 7, 7, 8, 8,10,10, 0, 0, 0, 7, 8, 8, 8,10,10, 0,
	 0, 0, 9, 9, 9, 9,10,10, 0, 0, 0, 0, 0, 9, 9,10,
	10,
};

static float _vq_quantthresh__44c5_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c5_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p5_0 = {
	_vq_quantthresh__44c5_s_p5_0,
	_vq_quantmap__44c5_s_p5_0,
	9,
	9
};

static static_codebook _44c5_s_p5_0 = {
	2, 81,
	_vq_lengthlist__44c5_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c5_s_p5_0,
	NULL,
	&_vq_auxt__44c5_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c5_s_p6_0[] = {
	 2, 4, 4, 6, 6, 8, 8, 9, 9, 9, 9,10,10,11,11,12,
	12, 0, 4, 4, 6, 6, 8, 8, 9, 9, 9, 9,10,10,11,11,
	12,12, 0, 4, 4, 6, 6, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,12, 0, 6, 6, 7, 7, 8, 8, 9,10,10,10,11,11,
	11,12,12,12, 0, 0, 0, 7, 7, 8, 8,10,10,10,10,11,
	11,12,12,12,12, 0, 0, 0, 7, 7, 9, 9,10,10,10,10,
	11,11,12,12,12,12, 0, 0, 0, 7, 7, 8, 9,10,10,10,
	10,11,11,12,12,12,13, 0, 0, 0, 8, 8, 9, 9,10,10,
	10,10,11,11,12,12,13,12, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 8, 8,
	 9, 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 8,
	 8, 9, 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	 9, 9,10,10,11,11,11,12,12,12,13,13, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,11,12,12,12,13,13, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,12,12,12,13,13,13, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,12,12,13,12,13,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,12,12,12,12,13,13,13,13,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,13,
	13,
};

static float _vq_quantthresh__44c5_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c5_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p6_0 = {
	_vq_quantthresh__44c5_s_p6_0,
	_vq_quantmap__44c5_s_p6_0,
	17,
	17
};

static static_codebook _44c5_s_p6_0 = {
	2, 289,
	_vq_lengthlist__44c5_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c5_s_p6_0,
	NULL,
	&_vq_auxt__44c5_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c5_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10,10,10,10,
	10,11,10,10, 6, 9, 9,10,10, 9,11,10,10, 6, 9, 9,
	10, 9,10,11,10, 9, 7,10,10,11,11,11,11,10,10, 6,
	 9, 9,10,10, 9,10, 9, 9, 6, 9, 9,10,10,10,11, 9,
	 9,
};

static float _vq_quantthresh__44c5_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c5_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p7_0 = {
	_vq_quantthresh__44c5_s_p7_0,
	_vq_quantmap__44c5_s_p7_0,
	3,
	3
};

static static_codebook _44c5_s_p7_0 = {
	4, 81,
	_vq_lengthlist__44c5_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c5_s_p7_0,
	NULL,
	&_vq_auxt__44c5_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c5_s_p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6,
	 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6, 7, 7, 8, 8, 8,
	 8,10, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10, 7,
	 7, 8, 8, 8, 8, 8, 8,10,10,10, 7, 7, 8, 8, 8, 8,
	 8, 8,10,10,10, 7, 7, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 9, 9, 8, 8, 9, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c5_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c5_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p7_1 = {
	_vq_quantthresh__44c5_s_p7_1,
	_vq_quantmap__44c5_s_p7_1,
	11,
	11
};

static static_codebook _44c5_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c5_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c5_s_p7_1,
	NULL,
	&_vq_auxt__44c5_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c5_s_p8_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 8, 8, 9, 9,10,10, 6, 5, 5,
	 7, 7, 8, 8, 9, 9,10,10,10,10, 7, 5, 5, 7, 7, 8,
	 8, 9, 9,10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,11,12, 9, 9, 9, 9, 9,10,10,10,11,11, 0,13,
	12, 9, 9, 9, 9,10,10,11,11,11,11, 0, 0, 0, 9,10,
	 9, 9,10,10,11,11,12,11, 0, 0, 0,10,10, 9, 9,10,
	10,11,11,12,12, 0, 0, 0,13,13,10,10,11,11,11,12,
	12,12, 0, 0, 0,14,14,10,10,11,11,11,11,12,12, 0,
	 0, 0, 0, 0,12,12,11,11,12,12,13,13, 0, 0, 0, 0,
	 0,12,12,11,11,12,12,13,13,
};

static float _vq_quantthresh__44c5_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c5_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p8_0 = {
	_vq_quantthresh__44c5_s_p8_0,
	_vq_quantmap__44c5_s_p8_0,
	13,
	13
};

static static_codebook _44c5_s_p8_0 = {
	2, 169,
	_vq_lengthlist__44c5_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c5_s_p8_0,
	NULL,
	&_vq_auxt__44c5_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c5_s_p8_1[] = {
	 2, 4, 4, 5, 5, 6, 5, 5, 5, 5, 6, 4, 5, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c5_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c5_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p8_1 = {
	_vq_quantthresh__44c5_s_p8_1,
	_vq_quantmap__44c5_s_p8_1,
	5,
	5
};

static static_codebook _44c5_s_p8_1 = {
	2, 25,
	_vq_lengthlist__44c5_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c5_s_p8_1,
	NULL,
	&_vq_auxt__44c5_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p9_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c5_s_p9_0[] = {
	 1, 3, 3,11,11,11,11,11,11,11,11,11,11,11,11, 5,
	 7, 7,11,11,11,11,11,11,11,11,11,11,11,11, 5, 9,
	 7,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44c5_s_p9_0[] = {
	-2320.5, -1963.5, -1606.5, -1249.5, -892.5, -535.5, -178.5, 178.5, 
	535.5, 892.5, 1249.5, 1606.5, 1963.5, 2320.5, 
};

static long _vq_quantmap__44c5_s_p9_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p9_0 = {
	_vq_quantthresh__44c5_s_p9_0,
	_vq_quantmap__44c5_s_p9_0,
	15,
	15
};

static static_codebook _44c5_s_p9_0 = {
	2, 225,
	_vq_lengthlist__44c5_s_p9_0,
	1, -512522752, 1628852224, 4, 0,
	_vq_quantlist__44c5_s_p9_0,
	NULL,
	&_vq_auxt__44c5_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p9_1[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c5_s_p9_1[] = {
	 1, 4, 4, 6, 6, 8, 7, 9, 9,10,10,11,11,12,12,13,
	13, 6, 5, 5, 6, 6, 8, 8,10,10,11,11,12,12,13,13,
	13,13, 6, 5, 5, 7, 7, 8, 8,10,10,11,11,12,12,13,
	13,13,13,18, 7, 7, 8, 8, 9, 9,10,11,11,11,12,12,
	13,13,13,14,18, 7, 7, 8, 8, 9, 9,11,10,12,12,13,
	13,13,13,14,15,18,12,12, 9, 9,10,10,11,11,12,12,
	13,13,13,14,14,14,18,12,12, 9, 8,10,10,11,11,12,
	12,14,13,13,14,15,15,18,16,18,11,11,11,11,12,12,
	13,13,13,14,14,14,14,15,17,18,17,11,10,11, 9,12,
	13,13,13,14,14,13,14,14,14,18,18,18,13,14,11,12,
	12,12,13,14,13,13,14,15,16,15,18,18,18,15,13,12,
	 9,12,11,13,14,14,15,14,14,16,14,18,18,18,18,18,
	12,13,13,13,13,14,15,14,15,15,15,15,18,18,18,18,
	17,14,12,13,11,14,12,15,14,14,15,16,15,18,18,18,
	17,18,15,18,13,13,14,13,15,14,16,15,17,16,18,18,
	17,18,18,15,17,14,13,14,12,14,14,15,15,15,15,18,
	18,18,17,17,18,18,14,15,14,14,14,14,15,14,16,16,
	17,18,18,18,18,17,17,15,15,13,13,15,13,15,13,15,
	15,
};

static float _vq_quantthresh__44c5_s_p9_1[] = {
	-157.5, -136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 
	10.5, 31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 157.5, 
};

static long _vq_quantmap__44c5_s_p9_1[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p9_1 = {
	_vq_quantthresh__44c5_s_p9_1,
	_vq_quantmap__44c5_s_p9_1,
	17,
	17
};

static static_codebook _44c5_s_p9_1 = {
	2, 289,
	_vq_lengthlist__44c5_s_p9_1,
	1, -520814592, 1620377600, 5, 0,
	_vq_quantlist__44c5_s_p9_1,
	NULL,
	&_vq_auxt__44c5_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c5_s_p9_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__44c5_s_p9_2[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,11, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10,11, 5, 5, 7, 7, 7,
	 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,
	 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10,10,10,11,11,11, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,11,11,11, 8, 8, 8, 8,
	 8, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,11,11,
	11, 8, 8, 8, 8, 9, 8, 9, 9, 9, 9, 9, 9,10,10,10,
	10,10,10,11,11,11, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9,10,10,10,10,10,10,10,11,11,11,11,11, 8, 8, 9,
	 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,11,11,11,
	11,11, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,10,
	10,10,11,11,11,11,11, 9, 9, 9, 9, 9, 9, 9,10,10,
	10,10,10,10,10,10,10,11,11,11,11,11, 9, 9, 9, 9,
	 9, 9,10,10,10,10,10,10,10,10,10,10,11,11,11,11,
	11,11,11, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,
	10,11,11,11,11,11,11,11,10, 9,10,10,10,10,10,10,
	10,10,10, 9,10,10,11,11,11,11,11,11,11, 9, 9,10,
	10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,
	11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	11,11,11,11,11,11,11,11,11,10,10,10,10,10, 9,10,
	10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,
	10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,
	11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,
	11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,
	10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,10,
	10,10,10,10,10,10,10,10,10,
};

static float _vq_quantthresh__44c5_s_p9_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__44c5_s_p9_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__44c5_s_p9_2 = {
	_vq_quantthresh__44c5_s_p9_2,
	_vq_quantmap__44c5_s_p9_2,
	21,
	21
};

static static_codebook _44c5_s_p9_2 = {
	2, 441,
	_vq_lengthlist__44c5_s_p9_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__44c5_s_p9_2,
	NULL,
	&_vq_auxt__44c5_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c5_s_short[] = {
	 3, 9,10,15,10,10,11,15,15,17, 4, 5, 7, 8, 7, 7,
	 9,13,15,16, 7, 6, 6,10, 6, 8, 9,12,12,16,10, 8,
	11, 8, 8, 7,11,15,17,17, 8, 5, 5, 8, 3, 4, 6,10,
	15,17,10, 7, 7, 7, 4, 4, 5,10,14,17,10, 9, 8, 9,
	 6, 5, 6,10,14,17,12,12,11,12, 9, 8, 8,11,14,17,
	13,14,13,10, 7, 5, 6, 9,13,17,14,14,14,10, 7, 5,
	 6, 7,10,15,
};

static static_codebook _huff_book__44c5_s_short = {
	2, 100,
	_huff_lengthlist__44c5_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c6_s_long[] = {
	 3, 8,11,13,13,13,12,12,13,18, 6, 3, 4, 7, 9, 9,
	11,11,13,16, 9, 4, 3, 5, 7, 7, 9,10,14,18,11, 7,
	 4, 4, 6, 6, 8,10,14,15,11, 9, 6, 6, 6, 6, 8,10,
	13,15,10, 9, 7, 6, 6, 6, 7, 8,12,12,12,10, 9, 8,
	 7, 6, 6, 7,11,12,11,10,10, 9, 9, 7, 7, 6, 9,12,
	12,12,13,13,13,10, 9, 8,10,12,13,14,16,16,17,14,
	12,11,11,13,
};

static static_codebook _huff_book__44c6_s_long = {
	2, 100,
	_huff_lengthlist__44c6_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c6_s_p1_0[] = {
	 1, 5, 5, 0, 5, 5, 0, 5, 5, 5, 8, 7, 0, 9, 8, 0,
	 8, 8, 6, 7, 8, 0, 8, 9, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 5, 8, 8, 0, 8, 8, 0, 8, 8, 5, 8, 8,
	 0, 8, 8, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5,
	 9, 8, 0, 8, 8, 0, 8, 8, 5, 8, 9, 0, 8, 8, 0, 8,
	 8,
};

static float _vq_quantthresh__44c6_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c6_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p1_0 = {
	_vq_quantthresh__44c6_s_p1_0,
	_vq_quantmap__44c6_s_p1_0,
	3,
	3
};

static static_codebook _44c6_s_p1_0 = {
	4, 81,
	_vq_lengthlist__44c6_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c6_s_p1_0,
	NULL,
	&_vq_auxt__44c6_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c6_s_p2_0[] = {
	 3, 5, 5, 8, 8, 0, 5, 5, 9, 9, 0, 5, 5, 9, 9, 0,
	 7, 7,10,10, 0, 0, 0,10,10, 5, 7, 7, 9, 9, 0, 8,
	 7,10, 9, 0, 8, 8,10,10, 0,10,10,11,11, 0, 0, 0,
	11,11, 5, 7, 7, 9, 9, 0, 7, 8, 9,10, 0, 7, 8,10,
	10, 0,10,10,11,11, 0, 0, 0,11,11, 8, 9, 9,11,10,
	 0,11,10,12,12, 0,11,11,12,12, 0,13,13,14,14, 0,
	 0, 0,14,14, 8, 9, 9,10,11, 0,10,11,12,12, 0,11,
	11,12,12, 0,13,13,14,14, 0, 0, 0,14,14, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 5, 8, 8,11,11, 0, 7, 7,10,10,
	 0, 7, 7,10,10, 0, 9, 8,11,10, 0, 0, 0,11,11, 5,
	 7, 8,11,11, 0, 7, 7,10,10, 0, 7, 7,10,10, 0, 8,
	 9,10,11, 0, 0, 0,11,11, 9,10,10,12,12, 0,10,10,
	12,11, 0,10,10,12,12, 0,12,12,13,13, 0, 0, 0,13,
	13, 9,10,10,12,12, 0,10,10,11,12, 0,10,10,12,12,
	 0,12,12,13,13, 0, 0, 0,13,13, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 5, 8, 7,11,10, 0, 7, 7,10,10, 0, 7, 7,
	10,10, 0, 8, 9,11,11, 0, 0, 0,11,10, 5, 7, 8,10,
	11, 0, 7, 7,10,10, 0, 7, 7,10,10, 0, 9, 8,11,11,
	 0, 0, 0,11,11, 9,10,10,12,12, 0,10,10,12,12, 0,
	10,10,12,12, 0,12,12,13,13, 0, 0, 0,13,13, 9, 9,
	10,12,12, 0,10,10,12,12, 0,10,10,12,12, 0,12,12,
	13,13, 0, 0, 0,13,13, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7,10,10,13,13, 0, 9, 8,12,12, 0, 8, 9,12,12, 0,
	10, 9,12,12, 0, 0, 0,12,12, 7,10,10,13,13, 0, 9,
	 9,12,12, 0, 9, 8,12,12, 0, 9,10,12,12, 0, 0, 0,
	12,12,10,11,11,14,14, 0,11,10,13,13, 0,11,11,13,
	13, 0,12,12,13,13, 0, 0, 0,13,13,10,11,11,14,14,
	 0,10,11,13,13, 0,11,11,13,13, 0,12,12,13,13, 0,
	 0, 0,13,13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,10,
	11,11,14,14, 0,11,11,13,13, 0,11,10,13,13, 0,12,
	12,13,13, 0, 0, 0,13,13,10,11,11,14,14, 0,11,11,
	13,13, 0,10,11,13,13, 0,12,12,13,13, 0, 0, 0,13,
	13,
};

static float _vq_quantthresh__44c6_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c6_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p2_0 = {
	_vq_quantthresh__44c6_s_p2_0,
	_vq_quantmap__44c6_s_p2_0,
	5,
	5
};

static static_codebook _44c6_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c6_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c6_s_p2_0,
	NULL,
	&_vq_auxt__44c6_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p3_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c6_s_p3_0[] = {
	 2, 4, 4, 5, 5, 7, 7, 9, 9, 0, 4, 4, 6, 6, 7, 7,
	 9, 9, 0, 4, 4, 6, 6, 7, 7, 9, 9, 0, 5, 5, 6, 6,
	 8, 8,10,10, 0, 0, 0, 6, 6, 8, 8,10,10, 0, 0, 0,
	 7, 7, 8, 8,10,10, 0, 0, 0, 7, 7, 9, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c6_s_p3_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c6_s_p3_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p3_0 = {
	_vq_quantthresh__44c6_s_p3_0,
	_vq_quantmap__44c6_s_p3_0,
	9,
	9
};

static static_codebook _44c6_s_p3_0 = {
	2, 81,
	_vq_lengthlist__44c6_s_p3_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c6_s_p3_0,
	NULL,
	&_vq_auxt__44c6_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p4_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c6_s_p4_0[] = {
	 3, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,
	11, 0, 4, 4, 6, 6, 7, 7, 8, 8, 8, 9,10,10,11,11,
	11,11, 0, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10,11,
	11,11,11, 0, 5, 5, 6, 6, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 6, 6, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,10,10,
	11,11,12,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,10,
	10,11,11,12,12,12,12, 0, 0, 0, 6, 6, 7, 7, 9, 9,
	10,10,11,11,12,12,12,13, 0, 0, 0, 0, 0, 7, 7, 9,
	 9,10,10,11,11,12,12,12,13, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c6_s_p4_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c6_s_p4_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p4_0 = {
	_vq_quantthresh__44c6_s_p4_0,
	_vq_quantmap__44c6_s_p4_0,
	17,
	17
};

static static_codebook _44c6_s_p4_0 = {
	2, 289,
	_vq_lengthlist__44c6_s_p4_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c6_s_p4_0,
	NULL,
	&_vq_auxt__44c6_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p5_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c6_s_p5_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 4, 7, 7, 9,10,10,10,
	10,10, 4, 7, 7, 9,10,10,10,10,10, 5, 9, 9, 9,11,
	11, 9,11,11, 7,10,10,11,12,11,12,12,12, 7,10,10,
	11,12,12,12,12,12, 6,10,10, 9,11,11,10,11,11, 7,
	10, 9,11,12,12,11,12,11, 7,10,10,11,12,12,11,12,
	12,
};

static float _vq_quantthresh__44c6_s_p5_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c6_s_p5_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p5_0 = {
	_vq_quantthresh__44c6_s_p5_0,
	_vq_quantmap__44c6_s_p5_0,
	3,
	3
};

static static_codebook _44c6_s_p5_0 = {
	4, 81,
	_vq_lengthlist__44c6_s_p5_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c6_s_p5_0,
	NULL,
	&_vq_auxt__44c6_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p5_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c6_s_p5_1[] = {
	 3, 4, 4, 6, 6, 8, 8, 8, 8, 9, 9,11, 4, 4, 6, 6,
	 8, 8, 9, 9, 9, 9,11, 4, 4, 6, 6, 8, 8, 9, 8, 9,
	 9,12, 5, 5, 6, 6, 8, 8, 9, 9, 9, 9,12,12,12, 6,
	 6, 8, 8, 9, 9, 9, 9,11,11,11, 7, 7, 8, 8, 9, 9,
	 9, 9,11,11,11, 7, 7, 8, 8, 8, 8, 9, 9,11,11,11,
	 7, 7, 8, 8, 8, 8, 9, 9,11,11,11,11,11, 8, 8, 8,
	 8, 9, 9,11,11,11,11,11, 7, 7, 8, 8, 8, 8,11,11,
	11,11,11, 7, 7, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c6_s_p5_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c6_s_p5_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p5_1 = {
	_vq_quantthresh__44c6_s_p5_1,
	_vq_quantmap__44c6_s_p5_1,
	11,
	11
};

static static_codebook _44c6_s_p5_1 = {
	2, 121,
	_vq_lengthlist__44c6_s_p5_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c6_s_p5_1,
	NULL,
	&_vq_auxt__44c6_s_p5_1,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c6_s_p6_0[] = {
	 1, 4, 4, 6, 7, 8, 8, 8, 8, 9, 9,10,10, 5, 5, 5,
	 7, 7, 9, 9, 9, 9,10,10,11,11, 6, 5, 5, 7, 7, 9,
	 9, 9, 9,10,10,11,11, 0, 7, 7, 7, 7, 9, 9,10,10,
	11,11,11,11, 0, 7, 7, 7, 7, 9, 9,10,10,11,11,12,
	12, 0,11,11, 8, 8,10, 9,10,11,11,11,12,12, 0,12,
	12, 8, 8,10, 9,11,11,12,11,13,13, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

static float _vq_quantthresh__44c6_s_p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c6_s_p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p6_0 = {
	_vq_quantthresh__44c6_s_p6_0,
	_vq_quantmap__44c6_s_p6_0,
	13,
	13
};

static static_codebook _44c6_s_p6_0 = {
	2, 169,
	_vq_lengthlist__44c6_s_p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c6_s_p6_0,
	NULL,
	&_vq_auxt__44c6_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c6_s_p6_1[] = {
	 3, 4, 4, 5, 5, 6, 4, 4, 5, 5, 6, 4, 4, 5, 4, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c6_s_p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c6_s_p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p6_1 = {
	_vq_quantthresh__44c6_s_p6_1,
	_vq_quantmap__44c6_s_p6_1,
	5,
	5
};

static static_codebook _44c6_s_p6_1 = {
	2, 25,
	_vq_lengthlist__44c6_s_p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c6_s_p6_1,
	NULL,
	&_vq_auxt__44c6_s_p6_1,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c6_s_p7_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9,10,10,11,11, 5, 5, 5,
	 7, 7, 8, 8, 9, 9,11,11,12,12, 6, 5, 5, 7, 7, 9,
	 9, 9, 9,11,11,12,12,19, 7, 7, 7, 7, 9, 9,10,10,
	11,11,12,12,19, 7, 7, 7, 7, 9, 9,10,10,11,11,12,
	12,19,11,11, 8, 8,10,10,11,11,11,12,12,12,19,12,
	12, 8, 8,10, 9,11,11,12,12,13,12,19,19,19,11,11,
	10,10,11,11,12,12,13,13,19,19,19,11,11,10,10,11,
	11,12,12,13,13,19,19,19,14,14,11,11,11,12,13,13,
	13,13,19,19,19,15,15,11,11,12,12,13,12,14,14,19,
	19,19,19,18,13,13,12,12,13,13,14,14,18,18,18,18,
	18,13,12,12,12,13,13,14,14,
};

static float _vq_quantthresh__44c6_s_p7_0[] = {
	-60.5, -49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 
	27.5, 38.5, 49.5, 60.5, 
};

static long _vq_quantmap__44c6_s_p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p7_0 = {
	_vq_quantthresh__44c6_s_p7_0,
	_vq_quantmap__44c6_s_p7_0,
	13,
	13
};

static static_codebook _44c6_s_p7_0 = {
	2, 169,
	_vq_lengthlist__44c6_s_p7_0,
	1, -523206656, 1618345984, 4, 0,
	_vq_quantlist__44c6_s_p7_0,
	NULL,
	&_vq_auxt__44c6_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c6_s_p7_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 7, 7, 9, 5, 5, 7, 7,
	 7, 7, 7, 7, 8, 8, 9, 5, 5, 6, 6, 7, 7, 7, 7, 7,
	 7, 9, 6, 6, 7, 7, 7, 7, 7, 7, 7, 8, 9, 9, 9, 7,
	 7, 7, 7, 7, 8, 7, 8, 9, 9, 9, 7, 7, 7, 7, 8, 8,
	 8, 8, 9, 9, 9, 7, 7, 7, 7, 7, 7, 7, 7, 9, 9, 9,
	 7, 7, 8, 8, 7, 7, 8, 8, 9, 9, 9, 8, 9, 8, 8, 7,
	 7, 7, 7, 9, 9, 8, 8, 9, 8, 8, 7, 7, 8, 8, 9, 9,
	 9, 9, 8, 7, 7, 7, 7, 8, 8,
};

static float _vq_quantthresh__44c6_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c6_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p7_1 = {
	_vq_quantthresh__44c6_s_p7_1,
	_vq_quantmap__44c6_s_p7_1,
	11,
	11
};

static static_codebook _44c6_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c6_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c6_s_p7_1,
	NULL,
	&_vq_auxt__44c6_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p8_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c6_s_p8_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 7, 7, 9, 9, 9,10,11,11, 6,
	 5, 5, 8, 8, 9, 9, 8, 8, 9, 9,10,10,11,11, 6, 5,
	 5, 8, 8, 9, 9, 8, 8, 9, 9,10,10,11,11,17, 8, 8,
	 8, 8,10,10, 8, 9,10,10,11,11,12,11,17, 8, 8, 9,
	 9,10,10, 9, 9,10,10,11,12,12,12,17,12,13, 9, 9,
	10,10, 9,10,10,10,11,11,13,12,17,13,13,10, 9,10,
	10,10,10,10,11,12,11,12,12,17,17,17, 9, 9, 9, 9,
	10,10,11,11,11,12,12,13,17,17,17, 9, 9, 9, 9,11,
	10,11,11,12,12,12,13,17,17,17,13,13,10,10,11,11,
	12,11,12,13,13,13,17,17,17,14,13,10, 9,11, 9,12,
	12,12,13,13,14,17,17,17,17,17,11,12,11,11,12,12,
	13,14,13,14,17,17,17,17,17,12,10,11, 8,12,11,13,
	14,14,14,17,17,16,16,16,13,15,11,12,12,13,13,13,
	14,14,16,16,16,16,16,14,13,12, 9,13,10,14,13,14,
	13,
};

static float _vq_quantthresh__44c6_s_p8_0[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__44c6_s_p8_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p8_0 = {
	_vq_quantthresh__44c6_s_p8_0,
	_vq_quantmap__44c6_s_p8_0,
	15,
	15
};

static static_codebook _44c6_s_p8_0 = {
	2, 225,
	_vq_lengthlist__44c6_s_p8_0,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__44c6_s_p8_0,
	NULL,
	&_vq_auxt__44c6_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p8_1[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__44c6_s_p8_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 9,
	 9, 8, 9, 9, 9,11, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 6, 6, 7, 7, 8,
	 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,
	 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,11,10,11, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,11,11,11, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9, 9, 9, 9,11,11,
	11, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,10,10, 9,10, 9,
	10,10, 9,11,11,11, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10, 9,10, 9, 9, 9, 9,11,11,11,11,11, 8, 8, 9,
	 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,10,11,11,11,
	11,11, 9, 9, 9, 9, 9, 9,10, 9, 9, 9,10,10, 9, 9,
	 9, 9,11,11,11,11,11, 9, 9, 9, 9, 9, 9, 9,10,10,
	 9,10, 9,10,10,10,10,11,11,11,11,11, 9, 9, 9, 9,
	 9, 9,10,10,10,10, 9,10,10, 9,10, 9,11,11,11,11,
	11,11,11, 9, 9, 9, 9,10, 9,10, 9, 9,10,10,10,10,
	10,10,11,11,11,11,11,11, 9, 9, 9, 9,10,10,10,10,
	 9,10, 9,10,10, 9,11,11,11,11,11,11,10, 9, 9, 9,
	 9, 9,10, 9,10,10,10,10,10,10,10,11,11,11,11,11,
	11,11, 9, 9,10,10,10,10,10,10,10,10,10,10,10,10,
	11,11,11,11,11,11,11,11,11, 9,10, 9,10, 9,10,10,
	10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,
	10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,
	11,11,11,10,10, 9,10,10,10,10, 9,10, 9,10,10,11,
	11,11,11,11,11,11,11,11,10,10,10, 9,10,10,10,10,
	10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,10,
	10, 9,10,10,10,10,10,10,10,
};

static float _vq_quantthresh__44c6_s_p8_1[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__44c6_s_p8_1[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p8_1 = {
	_vq_quantthresh__44c6_s_p8_1,
	_vq_quantmap__44c6_s_p8_1,
	21,
	21
};

static static_codebook _44c6_s_p8_1 = {
	2, 441,
	_vq_lengthlist__44c6_s_p8_1,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__44c6_s_p8_1,
	NULL,
	&_vq_auxt__44c6_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c6_s_p9_0[] = {
	 1, 4, 4, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 6, 9, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 6, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44c6_s_p9_0[] = {
	-3503.5, -2866.5, -2229.5, -1592.5, -955.5, -318.5, 318.5, 955.5, 
	1592.5, 2229.5, 2866.5, 3503.5, 
};

static long _vq_quantmap__44c6_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p9_0 = {
	_vq_quantthresh__44c6_s_p9_0,
	_vq_quantmap__44c6_s_p9_0,
	13,
	13
};

static static_codebook _44c6_s_p9_0 = {
	2, 169,
	_vq_lengthlist__44c6_s_p9_0,
	1, -511845376, 1630791680, 4, 0,
	_vq_quantlist__44c6_s_p9_0,
	NULL,
	&_vq_auxt__44c6_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c6_s_p9_1[] = {
	 1, 4, 4, 7, 7, 7, 7, 7, 7, 8, 9,10,11, 6, 6, 6,
	 7, 8, 8, 8, 7, 8, 9,10,11,10, 6, 5, 6, 7, 8, 8,
	 8, 8, 8, 9,10,10,11,14, 9, 8, 8, 8, 9, 8, 8, 9,
	10,10,12,11,14, 8, 8, 9, 8, 9, 8, 8, 8,11,10,11,
	11,14,14,13, 8, 9, 9, 9, 9,10,11,11,12,12,13,12,
	12, 8, 7,10, 9, 9, 9,11,11,11,10,13,13,13, 8, 9,
	 9, 8,12,11,11,11,13,11,13,13,13, 9, 8, 9, 8,10,
	10,11,10,11,10,13,13,13,12,12, 9,10,11,11,11,12,
	13,12,13,13,13,13,12,10,10,10, 9,13,12,12,13,13,
	13,13,13,13,12,12,10,10,12,12,13,13,13,13,13,13,
	13,12,12,11,12,12,12,12,13,
};

static float _vq_quantthresh__44c6_s_p9_1[] = {
	-269.5, -220.5, -171.5, -122.5, -73.5, -24.5, 24.5, 73.5, 
	122.5, 171.5, 220.5, 269.5, 
};

static long _vq_quantmap__44c6_s_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p9_1 = {
	_vq_quantthresh__44c6_s_p9_1,
	_vq_quantmap__44c6_s_p9_1,
	13,
	13
};

static static_codebook _44c6_s_p9_1 = {
	2, 169,
	_vq_lengthlist__44c6_s_p9_1,
	1, -518889472, 1622704128, 4, 0,
	_vq_quantlist__44c6_s_p9_1,
	NULL,
	&_vq_auxt__44c6_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c6_s_p9_2[] = {
	24,
	23,
	25,
	22,
	26,
	21,
	27,
	20,
	28,
	19,
	29,
	18,
	30,
	17,
	31,
	16,
	32,
	15,
	33,
	14,
	34,
	13,
	35,
	12,
	36,
	11,
	37,
	10,
	38,
	9,
	39,
	8,
	40,
	7,
	41,
	6,
	42,
	5,
	43,
	4,
	44,
	3,
	45,
	2,
	46,
	1,
	47,
	0,
	48,
};

static long _vq_lengthlist__44c6_s_p9_2[] = {
	 2, 4, 3, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 7, 6, 7, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__44c6_s_p9_2[] = {
	-23.5, -22.5, -21.5, -20.5, -19.5, -18.5, -17.5, -16.5, 
	-15.5, -14.5, -13.5, -12.5, -11.5, -10.5, -9.5, -8.5, 
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
	8.5, 9.5, 10.5, 11.5, 12.5, 13.5, 14.5, 15.5, 
	16.5, 17.5, 18.5, 19.5, 20.5, 21.5, 22.5, 23.5, 
};

static long _vq_quantmap__44c6_s_p9_2[] = {
	   47,   45,   43,   41,   39,   37,   35,   33,
	   31,   29,   27,   25,   23,   21,   19,   17,
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,   18,   20,   22,   24,   26,   28,   30,
	   32,   34,   36,   38,   40,   42,   44,   46,
	   48,
};

static encode_aux_threshmatch _vq_auxt__44c6_s_p9_2 = {
	_vq_quantthresh__44c6_s_p9_2,
	_vq_quantmap__44c6_s_p9_2,
	49,
	49
};

static static_codebook _44c6_s_p9_2 = {
	1, 49,
	_vq_lengthlist__44c6_s_p9_2,
	1, -526909440, 1611661312, 6, 0,
	_vq_quantlist__44c6_s_p9_2,
	NULL,
	&_vq_auxt__44c6_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c6_s_short[] = {
	 4, 9,11,11,13,13,17,16,17,17, 4, 4, 6, 7, 9, 9,
	12,15,17,17, 7, 5, 4, 5, 7, 8,11,12,17,17, 9, 6,
	 4, 3, 5, 6,10,14,17,17,11, 8, 6, 4, 5, 6, 9,13,
	17,17,11,10, 7, 5, 5, 5, 8,12,17,17,13,12, 9, 8,
	 7, 6, 8,11,17,17,13,13, 9, 6, 6, 5, 6, 9,17,17,
	17,16,10, 8, 7, 7, 8, 9,17,17,17,17,14,12,11,11,
	11,13,17,17,
};

static static_codebook _huff_book__44c6_s_short = {
	2, 100,
	_huff_lengthlist__44c6_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c7_s_long[] = {
	 3, 8,11,13,14,13,13,12,14,16, 6, 4, 5, 7, 9,10,
	11,11,13,15,10, 4, 3, 5, 7, 7,10,10,14,16,11, 7,
	 4, 4, 5, 6, 8,10,13,15,12, 9, 6, 5, 5, 6, 8, 9,
	13,15,11, 9, 7, 6, 5, 5, 6, 8,11,13,11,10, 9, 8,
	 7, 6, 6, 7,11,12,12,11,10, 9, 8, 7, 6, 6, 9,11,
	12,12,12,12,12,10, 9, 8,10,12,12,14,15,16,16,14,
	12,10,11,13,
};

static static_codebook _huff_book__44c7_s_long = {
	2, 100,
	_huff_lengthlist__44c7_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c7_s_p1_0[] = {
	 1, 5, 5, 0, 5, 5, 0, 5, 5, 6, 8, 7, 0, 9, 8, 0,
	 8, 8, 5, 7, 8, 0, 8, 9, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 5, 8, 8, 0, 8, 8, 0, 8, 8, 5, 8, 8,
	 0, 8, 8, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5,
	 9, 8, 0, 8, 8, 0, 8, 8, 5, 8, 9, 0, 8, 8, 0, 8,
	 8,
};

static float _vq_quantthresh__44c7_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c7_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p1_0 = {
	_vq_quantthresh__44c7_s_p1_0,
	_vq_quantmap__44c7_s_p1_0,
	3,
	3
};

static static_codebook _44c7_s_p1_0 = {
	4, 81,
	_vq_lengthlist__44c7_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c7_s_p1_0,
	NULL,
	&_vq_auxt__44c7_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c7_s_p2_0[] = {
	 3, 5, 5, 8, 8, 0, 5, 5, 9, 9, 0, 5, 5, 9, 9, 0,
	 7, 7,10, 9, 0, 0, 0, 9,10, 5, 7, 7, 9, 9, 0, 8,
	 7,10, 9, 0, 8, 7,10, 9, 0,10, 9,11,11, 0, 0, 0,
	11,11, 6, 7, 7, 9, 9, 0, 7, 8, 9,10, 0, 7, 8,10,
	10, 0, 9, 9,11,11, 0, 0, 0,11,11, 8, 9, 9,11,10,
	 0,11,10,12,12, 0,11,11,12,12, 0,13,13,14,14, 0,
	 0, 0,14,14, 8, 9, 9,10,11, 0,10,11,11,12, 0,11,
	11,12,12, 0,13,13,14,14, 0, 0, 0,14,14, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 5, 8, 7,11,11, 0, 7, 7,10,10,
	 0, 7, 7,10,10, 0, 9, 8,11,10, 0, 0, 0,11,11, 5,
	 7, 8,11,11, 0, 7, 7,10,10, 0, 7, 7,10,10, 0, 8,
	 9,10,11, 0, 0, 0,11,11, 9,10,10,12,12, 0,10,10,
	12,11, 0,10,10,12,12, 0,12,12,13,13, 0, 0, 0,13,
	13, 9,10,10,12,12, 0,10,10,11,12, 0,10,10,12,12,
	 0,12,12,13,13, 0, 0, 0,13,13, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 5, 8, 7,11,10, 0, 7, 7,10,10, 0, 7, 7,
	10,10, 0, 9, 9,11,11, 0, 0, 0,11,10, 5, 7, 8,10,
	11, 0, 7, 7,10,10, 0, 7, 7,10,10, 0, 9, 9,11,11,
	 0, 0, 0,10,11, 9,10, 9,12,11, 0,10,10,12,12, 0,
	10,10,12,11, 0,12,12,13,13, 0, 0, 0,13,13, 9, 9,
	10,11,12, 0,10,10,12,12, 0,10,10,11,12, 0,12,12,
	13,13, 0, 0, 0,13,13, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7, 9, 9,13,13, 0, 9, 8,12,12, 0, 8, 9,12,12, 0,
	10, 9,12,12, 0, 0, 0,12,12, 7,10, 9,13,13, 0, 9,
	 9,12,12, 0, 9, 8,12,12, 0, 9,10,12,12, 0, 0, 0,
	12,12,10,11,11,14,14, 0,11,10,13,12, 0,11,11,13,
	13, 0,12,12,13,13, 0, 0, 0,13,13,10,11,11,14,14,
	 0,10,11,12,13, 0,11,11,13,13, 0,12,12,13,13, 0,
	 0, 0,13,13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,10,
	11,11,14,14, 0,11,11,13,13, 0,11,10,13,13, 0,12,
	12,13,13, 0, 0, 0,13,13,10,11,11,14,14, 0,11,11,
	13,13, 0,10,11,13,13, 0,12,12,13,13, 0, 0, 0,13,
	13,
};

static float _vq_quantthresh__44c7_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c7_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p2_0 = {
	_vq_quantthresh__44c7_s_p2_0,
	_vq_quantmap__44c7_s_p2_0,
	5,
	5
};

static static_codebook _44c7_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c7_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c7_s_p2_0,
	NULL,
	&_vq_auxt__44c7_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p3_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c7_s_p3_0[] = {
	 2, 4, 4, 5, 5, 7, 7, 8, 8, 0, 4, 4, 6, 6, 7, 7,
	 9, 9, 0, 4, 4, 6, 6, 7, 7, 9, 9, 0, 5, 5, 6, 6,
	 8, 8,10,10, 0, 0, 0, 6, 6, 8, 8,10,10, 0, 0, 0,
	 7, 7, 9, 9,10,10, 0, 0, 0, 7, 7, 9, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c7_s_p3_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c7_s_p3_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p3_0 = {
	_vq_quantthresh__44c7_s_p3_0,
	_vq_quantmap__44c7_s_p3_0,
	9,
	9
};

static static_codebook _44c7_s_p3_0 = {
	2, 81,
	_vq_lengthlist__44c7_s_p3_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c7_s_p3_0,
	NULL,
	&_vq_auxt__44c7_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p4_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c7_s_p4_0[] = {
	 3, 4, 4, 6, 5, 7, 7, 7, 7, 8, 8, 9, 9,10,10,11,
	11, 0, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9,10,10,10,
	11,11, 0, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8,10,10,10,
	11,11,11, 0, 5, 5, 6, 6, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 6, 6, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,10,10,
	11,11,12,12,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,10,
	10,11,11,12,12,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 8, 8, 9,
	 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c7_s_p4_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c7_s_p4_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p4_0 = {
	_vq_quantthresh__44c7_s_p4_0,
	_vq_quantmap__44c7_s_p4_0,
	17,
	17
};

static static_codebook _44c7_s_p4_0 = {
	2, 289,
	_vq_lengthlist__44c7_s_p4_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c7_s_p4_0,
	NULL,
	&_vq_auxt__44c7_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p5_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c7_s_p5_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 4, 7, 7,10,11,10,10,
	11,11, 4, 7, 7,10,10,11,10,10,11, 5,10,10, 9,12,
	11,10,12,12, 7,11,10,12,12,12,12,13,13, 7,10,11,
	11,12,12,12,13,13, 5,10,10,10,12,12,10,12,12, 7,
	11,10,12,13,13,12,12,12, 7,10,11,12,13,13,12,12,
	12,
};

static float _vq_quantthresh__44c7_s_p5_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c7_s_p5_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p5_0 = {
	_vq_quantthresh__44c7_s_p5_0,
	_vq_quantmap__44c7_s_p5_0,
	3,
	3
};

static static_codebook _44c7_s_p5_0 = {
	4, 81,
	_vq_lengthlist__44c7_s_p5_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c7_s_p5_0,
	NULL,
	&_vq_auxt__44c7_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p5_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c7_s_p5_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 9,12, 4, 4, 6, 6,
	 7, 7, 8, 8, 9, 9,11, 5, 5, 6, 6, 7, 7, 8, 8, 9,
	 9,12, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9,12,12,12, 6,
	 6, 7, 7, 8, 8, 9, 9,12,12,12, 6, 6, 7, 7, 8, 8,
	 9, 9,12,11,11, 6, 6, 7, 7, 8, 8, 9, 9,12,12,11,
	 7, 7, 8, 8, 8, 8, 8, 8,12,12,12,11,11, 8, 8, 8,
	 8, 8, 8,12,12,12,11,11, 7, 7, 7, 7, 8, 8,12,12,
	12,11,11, 7, 7, 7, 7, 8, 8,
};

static float _vq_quantthresh__44c7_s_p5_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c7_s_p5_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p5_1 = {
	_vq_quantthresh__44c7_s_p5_1,
	_vq_quantmap__44c7_s_p5_1,
	11,
	11
};

static static_codebook _44c7_s_p5_1 = {
	2, 121,
	_vq_lengthlist__44c7_s_p5_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c7_s_p5_1,
	NULL,
	&_vq_auxt__44c7_s_p5_1,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c7_s_p6_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9,10, 9,10,10, 5, 5, 5,
	 7, 7, 9, 9, 9, 9,10,10,11,11, 6, 5, 5, 7, 7, 9,
	 9, 9, 9,11,10,11,11, 0, 6, 6, 7, 7, 9, 9,10,10,
	11,11,12,12, 0, 7, 7, 7, 7, 9, 9,10,10,11,11,12,
	12, 0,11,10, 8, 8,10,10,11,11,11,12,12,12, 0,11,
	11, 8, 8,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

static float _vq_quantthresh__44c7_s_p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c7_s_p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p6_0 = {
	_vq_quantthresh__44c7_s_p6_0,
	_vq_quantmap__44c7_s_p6_0,
	13,
	13
};

static static_codebook _44c7_s_p6_0 = {
	2, 169,
	_vq_lengthlist__44c7_s_p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c7_s_p6_0,
	NULL,
	&_vq_auxt__44c7_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c7_s_p6_1[] = {
	 3, 4, 4, 5, 5, 6, 4, 4, 5, 5, 6, 4, 4, 4, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c7_s_p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c7_s_p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p6_1 = {
	_vq_quantthresh__44c7_s_p6_1,
	_vq_quantmap__44c7_s_p6_1,
	5,
	5
};

static static_codebook _44c7_s_p6_1 = {
	2, 25,
	_vq_lengthlist__44c7_s_p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c7_s_p6_1,
	NULL,
	&_vq_auxt__44c7_s_p6_1,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c7_s_p7_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9,10,10,11,11, 5, 5, 5,
	 7, 7, 9, 8, 9, 9,11,11,12,12, 6, 5, 5, 7, 7, 9,
	 9, 9,10,11,11,12,12,20, 7, 7, 7, 7, 9, 9,10,10,
	11,11,12,12,20, 7, 7, 7, 7, 9, 9,10,10,11,11,12,
	12,20,11,11, 8, 8,10, 9,11,11,11,11,12,12,20,12,
	12, 8, 8, 9, 9,11,11,12,12,12,12,20,20,20,11,11,
	10,10,11,11,12,12,13,13,20,20,20,11,11,10,10,11,
	11,12,12,13,13,20,20,20,14,14,11,11,11,12,13,13,
	13,13,20,20,20,15,14,11,11,11,11,13,13,14,14,20,
	20,20,20,19,12,12,12,12,13,13,14,14,19,19,19,19,
	19,13,12,12,12,13,13,14,14,
};

static float _vq_quantthresh__44c7_s_p7_0[] = {
	-60.5, -49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 
	27.5, 38.5, 49.5, 60.5, 
};

static long _vq_quantmap__44c7_s_p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p7_0 = {
	_vq_quantthresh__44c7_s_p7_0,
	_vq_quantmap__44c7_s_p7_0,
	13,
	13
};

static static_codebook _44c7_s_p7_0 = {
	2, 169,
	_vq_lengthlist__44c7_s_p7_0,
	1, -523206656, 1618345984, 4, 0,
	_vq_quantlist__44c7_s_p7_0,
	NULL,
	&_vq_auxt__44c7_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c7_s_p7_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 7, 7, 9, 5, 5, 6, 6,
	 7, 7, 7, 7, 8, 8, 9, 5, 5, 6, 6, 7, 7, 7, 7, 7,
	 7, 9, 6, 6, 7, 7, 7, 7, 8, 8, 7, 8, 9, 9, 9, 7,
	 7, 7, 7, 7, 7, 8, 8, 9, 9, 9, 7, 7, 7, 7, 8, 8,
	 8, 8, 9, 9, 9, 7, 7, 7, 7, 7, 7, 8, 8, 9, 9, 9,
	 7, 7, 8, 7, 7, 7, 8, 8, 9, 9, 9, 9, 9, 7, 7, 7,
	 7, 8, 7, 9, 9, 9, 9, 9, 8, 8, 7, 7, 8, 8, 9, 9,
	 9, 9, 9, 7, 7, 7, 7, 8, 8,
};

static float _vq_quantthresh__44c7_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c7_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p7_1 = {
	_vq_quantthresh__44c7_s_p7_1,
	_vq_quantmap__44c7_s_p7_1,
	11,
	11
};

static static_codebook _44c7_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c7_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c7_s_p7_1,
	NULL,
	&_vq_auxt__44c7_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p8_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c7_s_p8_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 8, 8, 9, 9, 9,10,11,11, 6,
	 5, 5, 8, 8, 9, 9, 8, 8, 9, 9,10,10,11,11, 6, 5,
	 5, 8, 8, 9, 9, 8, 8, 9, 9,10,10,11,11,17, 8, 8,
	 8, 8,10, 9, 8, 9,10,10,11,11,11,11,17, 8, 8, 8,
	 8,10,10, 9, 9,10,10,11,11,12,12,17,12,13, 9, 9,
	10,10, 9, 9,10,11,11,11,12,12,17,13,13, 9, 9,10,
	10,10,10,10,10,11,11,12,12,17,17,17, 9, 9, 9, 9,
	10,10,11,11,11,12,12,12,17,17,17, 9, 9, 9, 9,11,
	10,11,12,11,12,13,12,17,17,17,13,14,10,10,10,11,
	12,11,12,12,12,13,17,17,17,14,14,10, 9,10, 9,12,
	12,12,12,13,13,17,17,17,17,17,11,11,11,11,11,12,
	13,13,13,14,17,17,17,17,17,12,10,11, 9,12,11,13,
	15,14,14,17,17,17,17,17,14,15,11,12,12,13,13,12,
	14,14,17,16,16,16,16,15,13,12, 9,12,10,14,12,15,
	14,
};

static float _vq_quantthresh__44c7_s_p8_0[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__44c7_s_p8_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p8_0 = {
	_vq_quantthresh__44c7_s_p8_0,
	_vq_quantmap__44c7_s_p8_0,
	15,
	15
};

static static_codebook _44c7_s_p8_0 = {
	2, 225,
	_vq_lengthlist__44c7_s_p8_0,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__44c7_s_p8_0,
	NULL,
	&_vq_auxt__44c7_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p8_1[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__44c7_s_p8_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 9,
	 9, 9, 9, 9, 9,10, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 6, 6, 7, 7, 8,
	 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,
	 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,11,11,11, 7, 7, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,11,11,11, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,11,11,
	11, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,10, 9, 9, 9, 9,
	10,10, 9,11,11,11, 8, 9, 9, 8, 9, 9, 9, 9, 9, 9,
	 9, 9,10,10, 9, 9,10, 9,11,11,11,11,11, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,11,10,11,
	11,11, 9, 9, 9, 9, 9, 9,10, 9,10,10,10,10,10, 9,
	 9,10,11,11,11,11,11, 9, 9, 9, 9, 9, 9, 9, 9,10,
	10, 9, 9,10,10,10,10,11,11,11,11,11, 9, 9, 9, 9,
	 9, 9, 9,10,10,10, 9,10,10,10,10, 9,11,11,10,11,
	11,11,11, 9, 9, 9, 9,10,10,10,10,10,10,10, 9,10,
	10,10,11,11,11,11,11,11, 9, 9, 9, 9, 9,10,10,10,
	10,10, 9, 9,10, 9,11,10,11,11,11,11,11, 9, 9, 9,
	 9, 9,10,10, 9,10,10,10,10,10,10,11,11,11,11,11,
	11,11,10, 9,10, 9,10,10,10,10,10,10,10,10,10, 9,
	10,11,10,11,11,11,11,11,10, 9, 9,10,10,10,10,10,
	10,10,10,10,10,11,11,11,11,11,11,11,10,11,10,10,
	10,10,10,10,10,10, 9,10,10, 9,11,11,11,11,11,10,
	11,11,11,10,10, 9, 9,10,10,10,10,10, 9,10,10,11,
	11,11,11,11,10,11,11,11,10,10, 9, 9,10,10,10,10,
	10,10,10,10,11,11,11,11,11,11,11,11,11,11,11, 9,
	 9, 9, 9,10,10,10,10,10,10,
};

static float _vq_quantthresh__44c7_s_p8_1[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__44c7_s_p8_1[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p8_1 = {
	_vq_quantthresh__44c7_s_p8_1,
	_vq_quantmap__44c7_s_p8_1,
	21,
	21
};

static static_codebook _44c7_s_p8_1 = {
	2, 441,
	_vq_lengthlist__44c7_s_p8_1,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__44c7_s_p8_1,
	NULL,
	&_vq_auxt__44c7_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c7_s_p9_0[] = {
	 1, 4, 4, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 5, 9, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 6, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 8, 8, 8,
};

static float _vq_quantthresh__44c7_s_p9_0[] = {
	-3503.5, -2866.5, -2229.5, -1592.5, -955.5, -318.5, 318.5, 955.5, 
	1592.5, 2229.5, 2866.5, 3503.5, 
};

static long _vq_quantmap__44c7_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p9_0 = {
	_vq_quantthresh__44c7_s_p9_0,
	_vq_quantmap__44c7_s_p9_0,
	13,
	13
};

static static_codebook _44c7_s_p9_0 = {
	2, 169,
	_vq_lengthlist__44c7_s_p9_0,
	1, -511845376, 1630791680, 4, 0,
	_vq_quantlist__44c7_s_p9_0,
	NULL,
	&_vq_auxt__44c7_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c7_s_p9_1[] = {
	 1, 4, 4, 7, 7, 7, 7, 7, 7, 9, 8,10,10, 6, 6, 6,
	 7, 8, 8, 8, 8, 8, 9, 9,10,11, 6, 5, 6, 8, 7, 8,
	 8, 8, 8, 9, 9,10,11,14, 9, 8, 9, 8, 9, 8, 8, 9,
	10,10,11,11,14, 8, 9, 8, 8, 8, 9, 9, 8,12,10,11,
	11,14,13,13, 8, 9, 9, 9, 9,10,10,12,12,12,14,14,
	13, 8, 7,10, 9, 9,10,10,11,11,10,14,14,14, 8, 9,
	 9, 8,11,10,12,11,11,11,14,14,14, 9, 7, 9, 8,10,
	10,11,11,11,10,14,14,14,12,12,10, 9,11,11,11,13,
	12,13,14,14,14,12,12,10,10,11, 8,11,11,14,13,14,
	14,14,14,14,12,13,11,12,12,11,14,13,13,13,13,13,
	13,12,11,11, 9,12,12,12,13,
};

static float _vq_quantthresh__44c7_s_p9_1[] = {
	-269.5, -220.5, -171.5, -122.5, -73.5, -24.5, 24.5, 73.5, 
	122.5, 171.5, 220.5, 269.5, 
};

static long _vq_quantmap__44c7_s_p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p9_1 = {
	_vq_quantthresh__44c7_s_p9_1,
	_vq_quantmap__44c7_s_p9_1,
	13,
	13
};

static static_codebook _44c7_s_p9_1 = {
	2, 169,
	_vq_lengthlist__44c7_s_p9_1,
	1, -518889472, 1622704128, 4, 0,
	_vq_quantlist__44c7_s_p9_1,
	NULL,
	&_vq_auxt__44c7_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c7_s_p9_2[] = {
	24,
	23,
	25,
	22,
	26,
	21,
	27,
	20,
	28,
	19,
	29,
	18,
	30,
	17,
	31,
	16,
	32,
	15,
	33,
	14,
	34,
	13,
	35,
	12,
	36,
	11,
	37,
	10,
	38,
	9,
	39,
	8,
	40,
	7,
	41,
	6,
	42,
	5,
	43,
	4,
	44,
	3,
	45,
	2,
	46,
	1,
	47,
	0,
	48,
};

static long _vq_lengthlist__44c7_s_p9_2[] = {
	 2, 4, 3, 4, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__44c7_s_p9_2[] = {
	-23.5, -22.5, -21.5, -20.5, -19.5, -18.5, -17.5, -16.5, 
	-15.5, -14.5, -13.5, -12.5, -11.5, -10.5, -9.5, -8.5, 
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
	8.5, 9.5, 10.5, 11.5, 12.5, 13.5, 14.5, 15.5, 
	16.5, 17.5, 18.5, 19.5, 20.5, 21.5, 22.5, 23.5, 
};

static long _vq_quantmap__44c7_s_p9_2[] = {
	   47,   45,   43,   41,   39,   37,   35,   33,
	   31,   29,   27,   25,   23,   21,   19,   17,
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,   18,   20,   22,   24,   26,   28,   30,
	   32,   34,   36,   38,   40,   42,   44,   46,
	   48,
};

static encode_aux_threshmatch _vq_auxt__44c7_s_p9_2 = {
	_vq_quantthresh__44c7_s_p9_2,
	_vq_quantmap__44c7_s_p9_2,
	49,
	49
};

static static_codebook _44c7_s_p9_2 = {
	1, 49,
	_vq_lengthlist__44c7_s_p9_2,
	1, -526909440, 1611661312, 6, 0,
	_vq_quantlist__44c7_s_p9_2,
	NULL,
	&_vq_auxt__44c7_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c7_s_short[] = {
	 4,10,12,13,15,15,16,16,17,17, 5, 5, 7, 8, 9, 9,
	12,17,18,18, 7, 5, 4, 5, 7, 8,10,13,18,18, 8, 6,
	 5, 4, 5, 6, 9,12,17,18,10, 9, 6, 4, 4, 5, 8,12,
	18,17,11, 9, 7, 5, 4, 4, 6,10,17,17,13,12,10, 8,
	 6, 5, 6, 9,17,17,14,13,12, 7, 6, 5, 5, 8,16,17,
	16,15,14, 8, 8, 7, 7, 9,14,17,17,17,17,12,11,11,
	11,12,16,17,
};

static static_codebook _huff_book__44c7_s_short = {
	2, 100,
	_huff_lengthlist__44c7_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c8_s_long[] = {
	 3, 8,12,14,14,13,13,12,13,15, 6, 4, 6, 8,10,10,
	11,11,13,15, 9, 5, 4, 5, 7, 8, 9,10,13,15,11, 7,
	 4, 4, 5, 6, 8, 9,13,14,12, 9, 6, 5, 5, 5, 7, 9,
	12,14,11,10, 7, 6, 5, 4, 6, 7,11,12,11,10, 9, 8,
	 7, 5, 6, 6,10,11,12,11,10, 9, 8, 6, 6, 5, 8,10,
	12,12,12,11,11,10, 9, 7, 8,11,12,13,14,14,15,13,
	10, 9, 9,11,
};

static static_codebook _huff_book__44c8_s_long = {
	2, 100,
	_huff_lengthlist__44c8_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c8_s_p1_0[] = {
	 1, 5, 5, 0, 5, 5, 0, 5, 5, 6, 7, 7, 0, 9, 8, 0,
	 8, 8, 6, 7, 8, 0, 8, 9, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 5, 8, 8, 0, 8, 8, 0, 8, 8, 5, 8, 8,
	 0, 8, 8, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5,
	 8, 8, 0, 8, 8, 0, 8, 7, 5, 8, 8, 0, 8, 8, 0, 7,
	 8,
};

static float _vq_quantthresh__44c8_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c8_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p1_0 = {
	_vq_quantthresh__44c8_s_p1_0,
	_vq_quantmap__44c8_s_p1_0,
	3,
	3
};

static static_codebook _44c8_s_p1_0 = {
	4, 81,
	_vq_lengthlist__44c8_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c8_s_p1_0,
	NULL,
	&_vq_auxt__44c8_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c8_s_p2_0[] = {
	 3, 5, 5, 8, 8, 0, 6, 6, 8, 8, 0, 5, 6, 8, 8, 0,
	 7, 7, 9, 9, 0, 0, 0, 9, 9, 5, 7, 6, 9, 9, 0, 7,
	 7,10, 9, 0, 7, 7,10, 9, 0, 9, 9,11,11, 0, 0, 0,
	11,11, 5, 6, 7, 9, 9, 0, 7, 7, 9,10, 0, 7, 7, 9,
	10, 0, 9, 9,11,11, 0, 0, 0,11,11, 8, 9, 9,11,10,
	 0,11,10,12,11, 0,10,10,12,11, 0,13,13,14,13, 0,
	 0, 0,14,13, 8, 9, 9,10,11, 0,10,11,11,12, 0,10,
	10,12,12, 0,13,13,13,14, 0, 0, 0,13,14, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 6, 7, 7,10,10, 0, 7, 7,10,10,
	 0, 7, 7,10,10, 0, 9, 8,10,10, 0, 0, 0,10,10, 6,
	 7, 7,10,10, 0, 7, 7,10,10, 0, 7, 7,10,10, 0, 8,
	 9,10,10, 0, 0, 0,10,10, 8,10, 9,12,12, 0,10, 9,
	12,11, 0,10,10,11,12, 0,12,11,13,12, 0, 0, 0,13,
	13, 8, 9,10,11,12, 0, 9,10,11,12, 0,10,10,11,12,
	 0,11,12,12,13, 0, 0, 0,13,13, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 6, 8, 7,11,10, 0, 7, 7,10,10, 0, 7, 7,
	10, 9, 0, 8, 9,10,10, 0, 0, 0,10,10, 6, 7, 8,10,
	11, 0, 7, 7,10,10, 0, 7, 7,10,10, 0, 9, 8,10,10,
	 0, 0, 0,10,10, 8,10, 9,12,11, 0,10,10,12,11, 0,
	10,10,12,11, 0,11,12,13,12, 0, 0, 0,13,12, 9, 9,
	10,11,12, 0,10,10,11,12, 0,10,10,11,12, 0,12,11,
	12,13, 0, 0, 0,12,13, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7, 9, 9,12,13, 0, 9, 8,12,11, 0, 8, 9,11,12, 0,
	10, 9,12,11, 0, 0, 0,11,12, 7, 9, 9,13,13, 0, 9,
	 9,11,12, 0, 9, 8,12,11, 0, 9,10,11,12, 0, 0, 0,
	12,11, 9,11,11,14,13, 0,10,10,13,12, 0,10,10,13,
	13, 0,12,11,13,12, 0, 0, 0,13,13, 9,11,11,13,14,
	 0,10,10,12,13, 0,10,10,13,13, 0,11,12,12,13, 0,
	 0, 0,13,13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9,
	11,11,14,13, 0,10,10,13,12, 0,10,10,13,12, 0,11,
	12,13,13, 0, 0, 0,13,12, 9,11,11,13,14, 0,10,10,
	13,13, 0,10,10,12,13, 0,12,11,13,13, 0, 0, 0,12,
	13,
};

static float _vq_quantthresh__44c8_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c8_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p2_0 = {
	_vq_quantthresh__44c8_s_p2_0,
	_vq_quantmap__44c8_s_p2_0,
	5,
	5
};

static static_codebook _44c8_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c8_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c8_s_p2_0,
	NULL,
	&_vq_auxt__44c8_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p3_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c8_s_p3_0[] = {
	 3, 3, 4, 5, 5, 7, 6, 8, 8, 0, 4, 4, 5, 5, 7, 7,
	 9, 9, 0, 4, 4, 5, 6, 7, 7, 9, 9, 0, 5, 5, 6, 6,
	 8, 8,10,10, 0, 0, 0, 6, 6, 8, 8,10,10, 0, 0, 0,
	 7, 7, 8, 8,10,10, 0, 0, 0, 7, 7, 8, 8,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c8_s_p3_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c8_s_p3_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p3_0 = {
	_vq_quantthresh__44c8_s_p3_0,
	_vq_quantmap__44c8_s_p3_0,
	9,
	9
};

static static_codebook _44c8_s_p3_0 = {
	2, 81,
	_vq_lengthlist__44c8_s_p3_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c8_s_p3_0,
	NULL,
	&_vq_auxt__44c8_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p4_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c8_s_p4_0[] = {
	 3, 4, 4, 5, 5, 7, 7, 8, 8, 8, 8, 9, 9,10,10,10,
	10, 0, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10,10,10,
	11,11, 0, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10,11,
	10,11,11, 0, 5, 5, 6, 6, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 6, 6, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,10,10,
	11,11,12,12,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,10,
	10,11,11,12,12,12,12, 0, 0, 0, 7, 7, 8, 8, 9, 9,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 8, 8, 9,
	 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c8_s_p4_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c8_s_p4_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p4_0 = {
	_vq_quantthresh__44c8_s_p4_0,
	_vq_quantmap__44c8_s_p4_0,
	17,
	17
};

static static_codebook _44c8_s_p4_0 = {
	2, 289,
	_vq_lengthlist__44c8_s_p4_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c8_s_p4_0,
	NULL,
	&_vq_auxt__44c8_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p5_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c8_s_p5_0[] = {
	 1, 4, 4, 5, 7, 7, 6, 7, 7, 4, 6, 6,10,10,10,10,
	10,10, 4, 6, 6,10,10,10,10,10,10, 5,10,10, 9,12,
	12,10,12,12, 7,10,10,12,12,12,12,12,12, 7,10,10,
	12,12,12,12,12,13, 6,10,10,10,12,12,11,12,12, 8,
	10,10,12,13,12,12,12,12, 7,10,10,12,12,13,12,13,
	12,
};

static float _vq_quantthresh__44c8_s_p5_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c8_s_p5_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p5_0 = {
	_vq_quantthresh__44c8_s_p5_0,
	_vq_quantmap__44c8_s_p5_0,
	3,
	3
};

static static_codebook _44c8_s_p5_0 = {
	4, 81,
	_vq_lengthlist__44c8_s_p5_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c8_s_p5_0,
	NULL,
	&_vq_auxt__44c8_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p5_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c8_s_p5_1[] = {
	 4, 5, 5, 6, 6, 7, 7, 7, 7, 7, 7,10, 4, 4, 6, 6,
	 7, 7, 8, 8, 8, 8,10, 5, 5, 6, 6, 7, 7, 8, 8, 8,
	 8,11, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8,11,11,11, 6,
	 6, 7, 7, 8, 8, 8, 8,11,11,11, 6, 6, 7, 7, 8, 8,
	 8, 8,11,11,11, 6, 6, 7, 7, 8, 8, 8, 8,11,11,11,
	 7, 7, 7, 7, 8, 8, 8, 8,11,11,11,11,11, 7, 7, 8,
	 8, 8, 8,11,11,11,11,11, 7, 7, 7, 7, 8, 8,11,11,
	11,11,11, 7, 7, 7, 7, 8, 8,
};

static float _vq_quantthresh__44c8_s_p5_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c8_s_p5_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p5_1 = {
	_vq_quantthresh__44c8_s_p5_1,
	_vq_quantmap__44c8_s_p5_1,
	11,
	11
};

static static_codebook _44c8_s_p5_1 = {
	2, 121,
	_vq_lengthlist__44c8_s_p5_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c8_s_p5_1,
	NULL,
	&_vq_auxt__44c8_s_p5_1,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c8_s_p6_0[] = {
	 1, 4, 4, 7, 6, 8, 8, 9, 9,10,10,11,11, 5, 5, 5,
	 7, 7, 9, 9,10, 9,11,11,12,12, 6, 5, 5, 7, 7, 9,
	 9,10,10,11,11,12,12, 0, 6, 6, 7, 7, 9, 9,10,10,
	11,11,12,12, 0, 7, 7, 7, 7, 9, 9,10,10,12,12,12,
	12, 0,10,10, 8, 8,10,10,11,11,12,12,13,13, 0,11,
	11, 8, 8,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

static float _vq_quantthresh__44c8_s_p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c8_s_p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p6_0 = {
	_vq_quantthresh__44c8_s_p6_0,
	_vq_quantmap__44c8_s_p6_0,
	13,
	13
};

static static_codebook _44c8_s_p6_0 = {
	2, 169,
	_vq_lengthlist__44c8_s_p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c8_s_p6_0,
	NULL,
	&_vq_auxt__44c8_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c8_s_p6_1[] = {
	 3, 4, 4, 5, 5, 5, 4, 4, 5, 5, 5, 4, 4, 5, 5, 6,
	 5, 5, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__44c8_s_p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c8_s_p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p6_1 = {
	_vq_quantthresh__44c8_s_p6_1,
	_vq_quantmap__44c8_s_p6_1,
	5,
	5
};

static static_codebook _44c8_s_p6_1 = {
	2, 25,
	_vq_lengthlist__44c8_s_p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c8_s_p6_1,
	NULL,
	&_vq_auxt__44c8_s_p6_1,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c8_s_p7_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9,10,10,11,11, 5, 5, 5,
	 7, 7, 9, 9,10,10,11,11,12,12, 6, 5, 5, 7, 7, 9,
	 9,10,10,11,11,12,12,20, 6, 7, 7, 7, 9, 9,10,10,
	11,11,12,12,20, 7, 7, 7, 7, 9, 9,10,10,11,11,12,
	12,20,11,11, 8, 8,10,10,11,11,12,12,12,12,20,12,
	12, 8, 8,10, 9,11,11,12,12,13,13,20,20,20,11,10,
	10,10,11,11,12,12,13,13,20,20,20,10,11,10,10,11,
	11,12,12,13,13,20,20,20,14,15,11,11,12,12,13,13,
	14,13,20,20,20,15,15,11,11,12,12,13,13,14,14,20,
	20,20,20,19,13,13,12,12,13,13,14,14,19,19,19,19,
	19,13,13,12,12,13,13,14,14,
};

static float _vq_quantthresh__44c8_s_p7_0[] = {
	-60.5, -49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 
	27.5, 38.5, 49.5, 60.5, 
};

static long _vq_quantmap__44c8_s_p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p7_0 = {
	_vq_quantthresh__44c8_s_p7_0,
	_vq_quantmap__44c8_s_p7_0,
	13,
	13
};

static static_codebook _44c8_s_p7_0 = {
	2, 169,
	_vq_lengthlist__44c8_s_p7_0,
	1, -523206656, 1618345984, 4, 0,
	_vq_quantlist__44c8_s_p7_0,
	NULL,
	&_vq_auxt__44c8_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c8_s_p7_1[] = {
	 4, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 8, 5, 5, 7, 7,
	 7, 7, 7, 7, 7, 7, 8, 5, 5, 7, 7, 7, 7, 7, 7, 7,
	 7, 8, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 7,
	 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 7, 7, 7, 7, 7, 7,
	 7, 7, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8,
	 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 7, 7, 7,
	 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 8, 8,
	 8, 8, 8, 7, 7, 7, 7, 7, 7,
};

static float _vq_quantthresh__44c8_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c8_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p7_1 = {
	_vq_quantthresh__44c8_s_p7_1,
	_vq_quantmap__44c8_s_p7_1,
	11,
	11
};

static static_codebook _44c8_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c8_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c8_s_p7_1,
	NULL,
	&_vq_auxt__44c8_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p8_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c8_s_p8_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,11, 6,
	 5, 5, 7, 7, 9, 9, 8, 8,10,10,11,11,11,12, 6, 5,
	 5, 7, 7, 9, 9, 9, 8,10,10,11,10,11,12,18, 8, 8,
	 8, 8, 9, 9, 9, 9,10,10,11,11,12,12,18, 8, 8, 8,
	 8, 9, 9, 9, 9,10,10,11,11,12,13,18,12,12, 9, 9,
	10,10, 9, 9,10,11,11,12,13,12,18,12,12, 9, 9,10,
	10,10,10,10,11,11,12,13,13,18,18,18, 9, 9, 9, 9,
	10,10,11,11,12,12,12,12,18,18,18, 9, 9, 9, 9,10,
	10,11,11,12,12,13,13,18,18,18,13,13,10,10,11,11,
	12,11,12,12,13,13,18,18,18,14,14,10, 9,11,10,12,
	12,12,12,13,13,18,18,18,18,18,11,12,11,11,12,12,
	13,13,14,13,18,18,18,18,18,12,11,11,10,12,11,13,
	13,13,14,18,18,18,18,18,15,16,12,12,12,13,13,13,
	14,14,18,17,17,17,17,16,14,12,11,12,11,13,12,15,
	14,
};

static float _vq_quantthresh__44c8_s_p8_0[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__44c8_s_p8_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p8_0 = {
	_vq_quantthresh__44c8_s_p8_0,
	_vq_quantmap__44c8_s_p8_0,
	15,
	15
};

static static_codebook _44c8_s_p8_0 = {
	2, 225,
	_vq_lengthlist__44c8_s_p8_0,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__44c8_s_p8_0,
	NULL,
	&_vq_auxt__44c8_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p8_1[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__44c8_s_p8_1[] = {
	 4, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 9, 9, 9, 9,10, 6, 6, 7, 7, 8, 8, 8, 8, 9, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 6, 6, 7, 7, 8,
	 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,
	 7, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10, 8, 8, 8, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,
	10, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9,10,10,11, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,11,11,10,11,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,
	10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,
	 9, 9,10,11,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,11,10,10, 9, 9, 9, 9,
	 9, 9, 9, 9,10, 9, 9, 9, 9, 9, 9, 9,11,10,11,10,
	10,10,10, 9, 9, 9, 9, 9, 9,10, 9, 9, 9, 9, 9, 9,
	 9,11,11,11,10,10,11,10, 9, 9, 9, 9, 9, 9,10,10,
	 9, 9, 9, 9, 9, 9,10,11,10,10,10,11,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9,10, 9,11,11,10,10,10,
	10,10, 9, 9, 9,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10,11,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9,10,10,11,11,10,11,10,10,10,10,10,10, 9,
	10, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,
	10,10,10, 9,10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,
	10,10,10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9,
	 9, 9,10, 9,10,10,10,10,10,10,10,10,10,10,10, 9,
	 9, 9, 9,10, 9, 9,10, 9, 9,
};

static float _vq_quantthresh__44c8_s_p8_1[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__44c8_s_p8_1[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p8_1 = {
	_vq_quantthresh__44c8_s_p8_1,
	_vq_quantmap__44c8_s_p8_1,
	21,
	21
};

static static_codebook _44c8_s_p8_1 = {
	2, 441,
	_vq_lengthlist__44c8_s_p8_1,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__44c8_s_p8_1,
	NULL,
	&_vq_auxt__44c8_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p9_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c8_s_p9_0[] = {
	 1, 5, 5,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10, 7,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10, 6, 8,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44c8_s_p9_0[] = {
	-6982.5, -6051.5, -5120.5, -4189.5, -3258.5, -2327.5, -1396.5, -465.5, 
	465.5, 1396.5, 2327.5, 3258.5, 4189.5, 5120.5, 6051.5, 6982.5, 
};

static long _vq_quantmap__44c8_s_p9_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p9_0 = {
	_vq_quantthresh__44c8_s_p9_0,
	_vq_quantmap__44c8_s_p9_0,
	17,
	17
};

static static_codebook _44c8_s_p9_0 = {
	2, 289,
	_vq_lengthlist__44c8_s_p9_0,
	1, -509798400, 1631393792, 5, 0,
	_vq_quantlist__44c8_s_p9_0,
	NULL,
	&_vq_auxt__44c8_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p9_1[] = {
	9,
	8,
	10,
	7,
	11,
	6,
	12,
	5,
	13,
	4,
	14,
	3,
	15,
	2,
	16,
	1,
	17,
	0,
	18,
};

static long _vq_lengthlist__44c8_s_p9_1[] = {
	 1, 4, 4, 7, 7, 7, 7, 7, 7, 9, 9,10,10,11,10,13,
	12,11,12, 6, 6, 6, 8, 8, 8, 8, 8, 8, 9,10,10,10,
	12,12,12,12,13,15, 5, 5, 5, 8, 8, 8, 8, 8, 8,10,
	 9,10,11,12,13,12,13,13,12,15, 9, 8, 9, 9, 9, 9,
	 9, 9,10,10,10,11,14,12,13,15,12,14,15, 8, 9, 9,
	 9, 9, 9, 9, 9,10,10,12,11,12,11,12,13,13,13,15,
	13,13, 9, 9,10, 9,10,10,11,10,11,12,12,12,14,13,
	14,15,15,13,13, 9, 8,10, 9,10,10,11,10,12,12,13,
	13,14,13,14,15,15,15,15, 9, 9, 9, 9,10,11,12,12,
	12,13,13,13,14,15,15,14,15,15,15, 9, 7, 9, 8,12,
	11,11,13,12,12,13,12,14,13,13,14,15,15,15,13,13,
	10,10,12,11,13,13,12,12,13,13,14,13,13,15,15,15,
	15,12,13,11, 9,11,10,12,12,15,13,13,13,14,13,14,
	13,15,15,15,15,15,12,12,11,11,12,13,15,13,13,14,
	14,14,15,14,15,15,15,15,15,13,11,12,11,12,11,13,
	14,13,13,14,14,13,14,15,15,15,15,15,15,15,12,12,
	12,13,15,13,15,14,15,14,13,15,15,15,15,15,15,14,
	15,13,13,12,11,14,12,15,13,14,14,14,13,15,15,15,
	15,15,15,15,14,13,14,13,15,13,15,15,15,14,15,14,
	15,15,15,15,15,15,15,14,14,14,13,13,13,15,15,15,
	15,14,15,15,15,15,15,15,15,15,15,15,12,13,13,13,
	14,15,15,13,15,15,15,15,15,15,15,15,15,15,15,15,
	15,14,14,15,15,15,14,14,15,
};

static float _vq_quantthresh__44c8_s_p9_1[] = {
	-416.5, -367.5, -318.5, -269.5, -220.5, -171.5, -122.5, -73.5, 
	-24.5, 24.5, 73.5, 122.5, 171.5, 220.5, 269.5, 318.5, 
	367.5, 416.5, 
};

static long _vq_quantmap__44c8_s_p9_1[] = {
	   17,   15,   13,   11,    9,    7,    5,    3,
	    1,    0,    2,    4,    6,    8,   10,   12,
	   14,   16,   18,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p9_1 = {
	_vq_quantthresh__44c8_s_p9_1,
	_vq_quantmap__44c8_s_p9_1,
	19,
	19
};

static static_codebook _44c8_s_p9_1 = {
	2, 361,
	_vq_lengthlist__44c8_s_p9_1,
	1, -518287360, 1622704128, 5, 0,
	_vq_quantlist__44c8_s_p9_1,
	NULL,
	&_vq_auxt__44c8_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c8_s_p9_2[] = {
	24,
	23,
	25,
	22,
	26,
	21,
	27,
	20,
	28,
	19,
	29,
	18,
	30,
	17,
	31,
	16,
	32,
	15,
	33,
	14,
	34,
	13,
	35,
	12,
	36,
	11,
	37,
	10,
	38,
	9,
	39,
	8,
	40,
	7,
	41,
	6,
	42,
	5,
	43,
	4,
	44,
	3,
	45,
	2,
	46,
	1,
	47,
	0,
	48,
};

static long _vq_lengthlist__44c8_s_p9_2[] = {
	 3, 4, 3, 4, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 6, 7, 6, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__44c8_s_p9_2[] = {
	-23.5, -22.5, -21.5, -20.5, -19.5, -18.5, -17.5, -16.5, 
	-15.5, -14.5, -13.5, -12.5, -11.5, -10.5, -9.5, -8.5, 
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
	8.5, 9.5, 10.5, 11.5, 12.5, 13.5, 14.5, 15.5, 
	16.5, 17.5, 18.5, 19.5, 20.5, 21.5, 22.5, 23.5, 
};

static long _vq_quantmap__44c8_s_p9_2[] = {
	   47,   45,   43,   41,   39,   37,   35,   33,
	   31,   29,   27,   25,   23,   21,   19,   17,
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,   18,   20,   22,   24,   26,   28,   30,
	   32,   34,   36,   38,   40,   42,   44,   46,
	   48,
};

static encode_aux_threshmatch _vq_auxt__44c8_s_p9_2 = {
	_vq_quantthresh__44c8_s_p9_2,
	_vq_quantmap__44c8_s_p9_2,
	49,
	49
};

static static_codebook _44c8_s_p9_2 = {
	1, 49,
	_vq_lengthlist__44c8_s_p9_2,
	1, -526909440, 1611661312, 6, 0,
	_vq_quantlist__44c8_s_p9_2,
	NULL,
	&_vq_auxt__44c8_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c8_s_short[] = {
	 4,11,13,14,16,15,16,16,17,16, 5, 6, 8, 9,10, 9,
	11,17,16,17, 6, 6, 6, 6, 7, 8, 9,14,16,17, 8, 6,
	 5, 4, 6, 7, 9,12,14,17, 9, 7, 6, 5, 5, 5, 8,11,
	14,17,10, 9, 7, 6, 5, 3, 5, 8,13,17,12,11, 9, 8,
	 7, 4, 4, 6,14,17,13,13,12, 8, 7, 5, 4, 5,12,17,
	17,14,15,10, 8, 7, 7, 7,11,17,17,16,17,14,12,10,
	11,11,15,17,
};

static static_codebook _huff_book__44c8_s_short = {
	2, 100,
	_huff_lengthlist__44c8_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44c9_s_long[] = {
	 3, 8,13,14,15,15,13,13,14,14, 6, 5, 8,10,12,12,
	13,12,13,14,10, 6, 5, 6, 8, 9,11,11,13,14,13, 8,
	 5, 4, 5, 6, 9,10,13,14,14,11, 7, 5, 4, 5, 8, 9,
	12,14,12,11, 8, 6, 5, 3, 5, 7,10,13,12,10,10, 8,
	 7, 5, 4, 6, 9,12,13,12,11,10, 9, 6, 5, 5, 7,10,
	13,12,12,11,11, 9, 8, 7, 8,10,12,12,13,13,14,12,
	10, 9, 9,10,
};

static static_codebook _huff_book__44c9_s_long = {
	2, 100,
	_huff_lengthlist__44c9_s_long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c9_s_p1_0[] = {
	 1, 5, 5, 0, 5, 5, 0, 5, 5, 6, 8, 8, 0, 9, 8, 0,
	 8, 8, 6, 8, 8, 0, 8, 9, 0, 8, 8, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 5, 8, 8, 0, 8, 7, 0, 8, 8, 5, 8, 8,
	 0, 7, 8, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5,
	 9, 8, 0, 8, 8, 0, 7, 7, 5, 8, 9, 0, 8, 8, 0, 7,
	 7,
};

static float _vq_quantthresh__44c9_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44c9_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p1_0 = {
	_vq_quantthresh__44c9_s_p1_0,
	_vq_quantmap__44c9_s_p1_0,
	3,
	3
};

static static_codebook _44c9_s_p1_0 = {
	4, 81,
	_vq_lengthlist__44c9_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44c9_s_p1_0,
	NULL,
	&_vq_auxt__44c9_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c9_s_p2_0[] = {
	 3, 6, 6, 8, 8, 0, 6, 6, 8, 8, 0, 5, 5, 8, 8, 0,
	 7, 7, 9, 9, 0, 0, 0, 9, 9, 6, 7, 7, 9, 8, 0, 8,
	 7, 9, 9, 0, 7, 7, 9, 9, 0, 9, 9,11,10, 0, 0, 0,
	10,10, 6, 7, 7, 8, 9, 0, 7, 8, 9, 9, 0, 7, 7, 9,
	 9, 0, 9, 9,10,11, 0, 0, 0,10,10, 8, 9, 9,10,10,
	 0,10,10,11,11, 0,10,10,11,11, 0,12,12,13,12, 0,
	 0, 0,13,13, 8, 8, 9,10,10, 0,10,10,11,12, 0,10,
	10,11,11, 0,12,12,13,13, 0, 0, 0,13,13, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 6, 8, 7,10,10, 0, 7, 7,10, 9,
	 0, 7, 7,10,10, 0, 9, 8,10,10, 0, 0, 0,10,10, 6,
	 7, 7,10,10, 0, 7, 7, 9,10, 0, 7, 7,10,10, 0, 8,
	 9,10,10, 0, 0, 0,10,10, 8, 9, 9,11,11, 0,10, 9,
	11,11, 0,10,10,11,11, 0,11,11,12,12, 0, 0, 0,12,
	12, 8, 9, 9,11,11, 0, 9,10,11,11, 0,10,10,11,11,
	 0,11,11,12,12, 0, 0, 0,12,12, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 5, 8, 7,10,10, 0, 7, 7,10,10, 0, 7, 7,
	10, 9, 0, 8, 9,10,10, 0, 0, 0,10, 9, 5, 7, 8,10,
	10, 0, 7, 7,10,10, 0, 7, 7, 9,10, 0, 9, 8,10,10,
	 0, 0, 0,10,10, 8, 9, 9,12,11, 0,10,10,11,11, 0,
	10, 9,11,11, 0,11,11,12,12, 0, 0, 0,12,12, 8, 9,
	 9,11,11, 0,10,10,11,11, 0, 9,10,11,11, 0,11,11,
	12,12, 0, 0, 0,12,12, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 7, 9, 9,12,12, 0, 9, 9,11,11, 0, 8, 9,11,11, 0,
	10, 9,12,11, 0, 0, 0,11,11, 7, 9, 9,12,12, 0, 8,
	 9,11,11, 0, 9, 8,11,11, 0, 9,10,11,11, 0, 0, 0,
	11,11, 9,11,11,13,13, 0,10,10,12,12, 0,10,10,12,
	12, 0,12,11,12,12, 0, 0, 0,13,13, 9,10,11,13,13,
	 0,10,10,12,12, 0,10,10,12,12, 0,11,11,12,12, 0,
	 0, 0,12,12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9,
	11,10,13,13, 0,10,10,12,12, 0,10, 9,12,12, 0,11,
	11,12,13, 0, 0, 0,12,11, 9,10,10,12,13, 0,10,10,
	12,12, 0,10,10,12,12, 0,11,11,13,12, 0, 0, 0,12,
	12,
};

static float _vq_quantthresh__44c9_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c9_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p2_0 = {
	_vq_quantthresh__44c9_s_p2_0,
	_vq_quantmap__44c9_s_p2_0,
	5,
	5
};

static static_codebook _44c9_s_p2_0 = {
	4, 625,
	_vq_lengthlist__44c9_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c9_s_p2_0,
	NULL,
	&_vq_auxt__44c9_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p3_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44c9_s_p3_0[] = {
	 3, 4, 4, 5, 5, 6, 6, 8, 8, 0, 4, 4, 5, 5, 7, 6,
	 8, 8, 0, 4, 4, 5, 5, 7, 7, 8, 8, 0, 5, 5, 6, 6,
	 7, 7, 9, 9, 0, 0, 0, 6, 6, 7, 7, 9, 9, 0, 0, 0,
	 7, 7, 8, 8, 9, 9, 0, 0, 0, 7, 7, 8, 8, 9, 9, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c9_s_p3_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44c9_s_p3_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p3_0 = {
	_vq_quantthresh__44c9_s_p3_0,
	_vq_quantmap__44c9_s_p3_0,
	9,
	9
};

static static_codebook _44c9_s_p3_0 = {
	2, 81,
	_vq_lengthlist__44c9_s_p3_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44c9_s_p3_0,
	NULL,
	&_vq_auxt__44c9_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p4_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44c9_s_p4_0[] = {
	 4, 4, 4, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 9, 9,10,
	10, 0, 5, 5, 5, 5, 6, 6, 8, 7, 8, 8, 9, 9,10,10,
	11,11, 0, 5, 5, 5, 5, 7, 7, 8, 8, 8, 8, 9, 9,10,
	10,11,11, 0, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 6, 6, 7, 7, 8, 8, 9, 9,10,
	10,11,11,12,12, 0, 0, 0, 7, 7, 7, 7, 9, 9, 9, 9,
	11,11,11,11,12,12, 0, 0, 0, 7, 7, 7, 8, 9, 9, 9,
	 9,11,10,11,11,12,12, 0, 0, 0, 7, 7, 7, 7, 9, 9,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 7, 7, 9,
	 9,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__44c9_s_p4_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44c9_s_p4_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p4_0 = {
	_vq_quantthresh__44c9_s_p4_0,
	_vq_quantmap__44c9_s_p4_0,
	17,
	17
};

static static_codebook _44c9_s_p4_0 = {
	2, 289,
	_vq_lengthlist__44c9_s_p4_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44c9_s_p4_0,
	NULL,
	&_vq_auxt__44c9_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p5_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44c9_s_p5_0[] = {
	 1, 4, 4, 5, 7, 7, 6, 7, 7, 4, 6, 6,10,10,10,10,
	10,10, 4, 6, 6,10,10,10,10,10,10, 5,10,10, 9,12,
	12,10,12,12, 7,10,10,12,12,12,12,12,13, 7,10,10,
	12,12,12,12,13,13, 6,10,10,10,12,12,11,12,12, 8,
	10,10,12,12,12,12,12,12, 7,10,10,12,12,13,12,12,
	12,
};

static float _vq_quantthresh__44c9_s_p5_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44c9_s_p5_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p5_0 = {
	_vq_quantthresh__44c9_s_p5_0,
	_vq_quantmap__44c9_s_p5_0,
	3,
	3
};

static static_codebook _44c9_s_p5_0 = {
	4, 81,
	_vq_lengthlist__44c9_s_p5_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44c9_s_p5_0,
	NULL,
	&_vq_auxt__44c9_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p5_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c9_s_p5_1[] = {
	 4, 5, 5, 6, 6, 6, 7, 7, 7, 7, 7,10, 5, 5, 6, 6,
	 7, 7, 7, 7, 8, 8,10, 5, 5, 6, 6, 7, 7, 7, 7, 8,
	 8,11, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8,11,11,11, 6,
	 6, 7, 7, 8, 8, 8, 8,11,11,11, 6, 6, 7, 7, 8, 8,
	 8, 8,11,11,11, 6, 6, 7, 7, 7, 7, 8, 8,11,11,11,
	 7, 7, 7, 7, 7, 7, 8, 8,11,11,11,11,11, 7, 7, 7,
	 7, 8, 8,11,11,11,11,11, 7, 7, 7, 7, 7, 7,11,11,
	11,11,11, 7, 7, 7, 7, 7, 7,
};

static float _vq_quantthresh__44c9_s_p5_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c9_s_p5_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p5_1 = {
	_vq_quantthresh__44c9_s_p5_1,
	_vq_quantmap__44c9_s_p5_1,
	11,
	11
};

static static_codebook _44c9_s_p5_1 = {
	2, 121,
	_vq_lengthlist__44c9_s_p5_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c9_s_p5_1,
	NULL,
	&_vq_auxt__44c9_s_p5_1,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c9_s_p6_0[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10, 5, 4, 4,
	 6, 6, 8, 8, 9, 9,10,10,11,11, 6, 4, 4, 6, 6, 8,
	 8, 9, 9,10,10,12,12, 0, 6, 6, 6, 6, 8, 8,10,10,
	11,11,12,12, 0, 6, 6, 6, 6, 8, 8,10,10,11,11,12,
	12, 0,10,10, 8, 8, 9, 9,11,11,12,12,13,13, 0,11,
	11, 8, 8, 9,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

static float _vq_quantthresh__44c9_s_p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44c9_s_p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p6_0 = {
	_vq_quantthresh__44c9_s_p6_0,
	_vq_quantmap__44c9_s_p6_0,
	13,
	13
};

static static_codebook _44c9_s_p6_0 = {
	2, 169,
	_vq_lengthlist__44c9_s_p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44c9_s_p6_0,
	NULL,
	&_vq_auxt__44c9_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44c9_s_p6_1[] = {
	 4, 4, 4, 5, 5, 5, 4, 4, 5, 5, 5, 4, 4, 5, 5, 5,
	 5, 5, 5, 5, 5, 5, 5, 5, 5,
};

static float _vq_quantthresh__44c9_s_p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44c9_s_p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p6_1 = {
	_vq_quantthresh__44c9_s_p6_1,
	_vq_quantmap__44c9_s_p6_1,
	5,
	5
};

static static_codebook _44c9_s_p6_1 = {
	2, 25,
	_vq_lengthlist__44c9_s_p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44c9_s_p6_1,
	NULL,
	&_vq_auxt__44c9_s_p6_1,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44c9_s_p7_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9,10,10,11,11, 6, 5, 5,
	 7, 7, 9, 8,10,10,11,11,12,12, 6, 5, 5, 7, 7, 9,
	 9,10,10,11,11,12,12,20, 6, 6, 7, 7, 9, 9,10,10,
	11,11,12,12,20, 7, 7, 7, 7, 9, 9,10,10,11,11,13,
	12,20,10,10, 8, 8, 9, 9,11,11,12,12,13,13,20,11,
	11, 8, 8, 9, 9,11,11,12,12,13,13,20,20,20,10,10,
	10,10,11,11,12,12,13,13,20,20,20,10,10,10,10,11,
	11,12,12,13,13,20,20,20,14,14,11,11,12,12,13,13,
	14,13,20,20,20,14,15,11,11,11,11,13,13,14,13,20,
	20,20,20,19,12,12,12,12,13,13,14,14,19,19,19,19,
	19,13,13,12,12,13,13,14,14,
};

static float _vq_quantthresh__44c9_s_p7_0[] = {
	-60.5, -49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 
	27.5, 38.5, 49.5, 60.5, 
};

static long _vq_quantmap__44c9_s_p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p7_0 = {
	_vq_quantthresh__44c9_s_p7_0,
	_vq_quantmap__44c9_s_p7_0,
	13,
	13
};

static static_codebook _44c9_s_p7_0 = {
	2, 169,
	_vq_lengthlist__44c9_s_p7_0,
	1, -523206656, 1618345984, 4, 0,
	_vq_quantlist__44c9_s_p7_0,
	NULL,
	&_vq_auxt__44c9_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44c9_s_p7_1[] = {
	 5, 6, 6, 6, 6, 7, 6, 7, 7, 7, 7, 8, 6, 6, 6, 6,
	 7, 7, 7, 7, 7, 7, 8, 6, 6, 6, 6, 7, 7, 7, 7, 7,
	 7, 8, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 8, 8, 8, 6,
	 6, 7, 7, 7, 7, 7, 7, 8, 8, 8, 7, 7, 7, 7, 7, 7,
	 7, 7, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8,
	 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 7, 7, 7,
	 7, 7, 7, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 8, 8,
	 8, 8, 8, 7, 7, 7, 7, 7, 7,
};

static float _vq_quantthresh__44c9_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44c9_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p7_1 = {
	_vq_quantthresh__44c9_s_p7_1,
	_vq_quantmap__44c9_s_p7_1,
	11,
	11
};

static static_codebook _44c9_s_p7_1 = {
	2, 121,
	_vq_lengthlist__44c9_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44c9_s_p7_1,
	NULL,
	&_vq_auxt__44c9_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p8_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44c9_s_p8_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 8, 8, 9, 9,10,10,11,11, 6,
	 5, 5, 7, 7, 9, 9, 8, 9,10,10,11,11,12,12, 6, 5,
	 5, 7, 7, 9, 9, 9, 9,10,10,11,11,12,12,19, 7, 8,
	 8, 8, 9, 9, 9, 9,10,10,11,11,12,13,19, 8, 8, 8,
	 8, 9, 9, 9, 9,10,10,11,12,12,12,19,12,12, 9, 9,
	 9,10, 9,10,10,10,12,12,12,12,19,12,12, 9, 9,10,
	 9,10,10,11,11,12,11,13,13,19,19,19, 9, 9, 9, 9,
	10,10,11,11,12,12,12,12,19,19,19, 9, 9, 9, 9,10,
	10,11,11,12,12,13,13,19,19,19,13,13,10,10,10,10,
	12,12,12,12,13,13,19,19,19,13,13,10,10,10,10,12,
	12,12,12,13,13,19,19,19,19,19,11,12,11,11,12,12,
	13,12,13,13,19,19,19,18,18,12,11,11,10,12,11,13,
	13,13,14,18,18,18,18,18,15,16,12,12,13,12,13,13,
	14,14,18,18,18,18,18,16,15,12,11,12,11,13,13,14,
	14,
};

static float _vq_quantthresh__44c9_s_p8_0[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__44c9_s_p8_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p8_0 = {
	_vq_quantthresh__44c9_s_p8_0,
	_vq_quantmap__44c9_s_p8_0,
	15,
	15
};

static static_codebook _44c9_s_p8_0 = {
	2, 225,
	_vq_lengthlist__44c9_s_p8_0,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__44c9_s_p8_0,
	NULL,
	&_vq_auxt__44c9_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p8_1[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__44c9_s_p8_1[] = {
	 4, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 6, 6, 7, 7, 8,
	 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,
	 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,
	10, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9,10,10,10, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,
	10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,
	10,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9,10,10,10,10,10,10,10, 9, 9, 9, 9,10,10,10,10,
	 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,10,10,10,
	10,10, 9, 9, 9, 9, 9,10, 9, 9, 9, 9, 9, 9, 9, 9,
	10,10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,10,10,
	10,10, 9, 9, 9,10, 9, 9, 9, 9,10,10,10,10,10,10,
	10,10,10, 9,10, 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10,
	10,10,10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,10,10, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44c9_s_p8_1[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__44c9_s_p8_1[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p8_1 = {
	_vq_quantthresh__44c9_s_p8_1,
	_vq_quantmap__44c9_s_p8_1,
	21,
	21
};

static static_codebook _44c9_s_p8_1 = {
	2, 441,
	_vq_lengthlist__44c9_s_p8_1,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__44c9_s_p8_1,
	NULL,
	&_vq_auxt__44c9_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p9_0[] = {
	9,
	8,
	10,
	7,
	11,
	6,
	12,
	5,
	13,
	4,
	14,
	3,
	15,
	2,
	16,
	1,
	17,
	0,
	18,
};

static long _vq_lengthlist__44c9_s_p9_0[] = {
	 1, 4, 4,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10, 7, 9,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10, 7, 9,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44c9_s_p9_0[] = {
	-7913.5, -6982.5, -6051.5, -5120.5, -4189.5, -3258.5, -2327.5, -1396.5, 
	-465.5, 465.5, 1396.5, 2327.5, 3258.5, 4189.5, 5120.5, 6051.5, 
	6982.5, 7913.5, 
};

static long _vq_quantmap__44c9_s_p9_0[] = {
	   17,   15,   13,   11,    9,    7,    5,    3,
	    1,    0,    2,    4,    6,    8,   10,   12,
	   14,   16,   18,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p9_0 = {
	_vq_quantthresh__44c9_s_p9_0,
	_vq_quantmap__44c9_s_p9_0,
	19,
	19
};

static static_codebook _44c9_s_p9_0 = {
	2, 361,
	_vq_lengthlist__44c9_s_p9_0,
	1, -508535424, 1631393792, 5, 0,
	_vq_quantlist__44c9_s_p9_0,
	NULL,
	&_vq_auxt__44c9_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p9_1[] = {
	9,
	8,
	10,
	7,
	11,
	6,
	12,
	5,
	13,
	4,
	14,
	3,
	15,
	2,
	16,
	1,
	17,
	0,
	18,
};

static long _vq_lengthlist__44c9_s_p9_1[] = {
	 1, 4, 4, 7, 7, 8, 8, 8, 8, 9,10,10,10,12,11,12,
	13,13,12, 6, 5, 5, 8, 8, 8, 9, 8, 9,10,10,11,11,
	12,12,14,13,13,12, 5, 5, 5, 8, 8, 9, 8, 8, 9,10,
	10,11,11,12,12,14,12,13,13,16, 8, 8, 9, 9, 9, 9,
	 9, 9,10,10,11,12,13,13,13,13,14,14,16, 8, 8, 9,
	 9, 9, 9, 9, 9,11,11,12,11,13,13,14,13,13,13,16,
	14,13, 9, 9, 9, 9,10,10,11,12,12,13,13,13,14,13,
	15,14,16,13,13, 9, 8, 9, 9,10,10,12,11,13,13,14,
	14,14,14,15,14,16,16,16, 9, 9, 9, 9,10,10,12,12,
	12,13,13,13,15,13,15,15,16,16,16, 9, 7, 9, 8,10,
	11,11,12,12,13,13,16,15,14,14,14,16,16,16,13,13,
	10,10,11,11,13,15,13,14,13,14,14,13,14,14,16,16,
	16,13,12,10, 9,11,11,12,12,14,14,13,14,14,14,14,
	14,16,16,16,16,16,12,13,11,11,12,13,13,13,14,15,
	14,14,16,15,16,16,16,16,16,12,11,12,12,15,13,13,
	13,14,13,15,14,15,14,16,16,16,16,16,14,15,12,13,
	13,12,14,15,15,14,15,14,15,13,16,16,16,16,16,16,
	16,13,13,14,12,16,12,16,15,14,15,14,14,16,16,16,
	16,16,16,16,15,14,14,14,15,16,16,16,16,14,16,16,
	16,16,16,16,16,16,16,14,14,14,12,15,11,15,13,16,
	15,16,15,16,16,16,16,16,16,16,15,16,14,14,15,13,
	15,16,16,16,16,15,16,16,16,16,16,16,16,16,16,15,
	15,14,13,14,16,16,14,15,16,
};

static float _vq_quantthresh__44c9_s_p9_1[] = {
	-416.5, -367.5, -318.5, -269.5, -220.5, -171.5, -122.5, -73.5, 
	-24.5, 24.5, 73.5, 122.5, 171.5, 220.5, 269.5, 318.5, 
	367.5, 416.5, 
};

static long _vq_quantmap__44c9_s_p9_1[] = {
	   17,   15,   13,   11,    9,    7,    5,    3,
	    1,    0,    2,    4,    6,    8,   10,   12,
	   14,   16,   18,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p9_1 = {
	_vq_quantthresh__44c9_s_p9_1,
	_vq_quantmap__44c9_s_p9_1,
	19,
	19
};

static static_codebook _44c9_s_p9_1 = {
	2, 361,
	_vq_lengthlist__44c9_s_p9_1,
	1, -518287360, 1622704128, 5, 0,
	_vq_quantlist__44c9_s_p9_1,
	NULL,
	&_vq_auxt__44c9_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__44c9_s_p9_2[] = {
	24,
	23,
	25,
	22,
	26,
	21,
	27,
	20,
	28,
	19,
	29,
	18,
	30,
	17,
	31,
	16,
	32,
	15,
	33,
	14,
	34,
	13,
	35,
	12,
	36,
	11,
	37,
	10,
	38,
	9,
	39,
	8,
	40,
	7,
	41,
	6,
	42,
	5,
	43,
	4,
	44,
	3,
	45,
	2,
	46,
	1,
	47,
	0,
	48,
};

static long _vq_lengthlist__44c9_s_p9_2[] = {
	 3, 4, 4, 4, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6,
	 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	 6, 7, 6, 6, 7, 7, 6, 7, 6, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__44c9_s_p9_2[] = {
	-23.5, -22.5, -21.5, -20.5, -19.5, -18.5, -17.5, -16.5, 
	-15.5, -14.5, -13.5, -12.5, -11.5, -10.5, -9.5, -8.5, 
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
	8.5, 9.5, 10.5, 11.5, 12.5, 13.5, 14.5, 15.5, 
	16.5, 17.5, 18.5, 19.5, 20.5, 21.5, 22.5, 23.5, 
};

static long _vq_quantmap__44c9_s_p9_2[] = {
	   47,   45,   43,   41,   39,   37,   35,   33,
	   31,   29,   27,   25,   23,   21,   19,   17,
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,   18,   20,   22,   24,   26,   28,   30,
	   32,   34,   36,   38,   40,   42,   44,   46,
	   48,
};

static encode_aux_threshmatch _vq_auxt__44c9_s_p9_2 = {
	_vq_quantthresh__44c9_s_p9_2,
	_vq_quantmap__44c9_s_p9_2,
	49,
	49
};

static static_codebook _44c9_s_p9_2 = {
	1, 49,
	_vq_lengthlist__44c9_s_p9_2,
	1, -526909440, 1611661312, 6, 0,
	_vq_quantlist__44c9_s_p9_2,
	NULL,
	&_vq_auxt__44c9_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44c9_s_short[] = {
	 5,13,17,15,16,16,18,17,18,18, 5, 7,10,12,12,12,
	12,17,18,17, 7, 6, 7, 8, 9, 9, 9,13,17,17, 8, 6,
	 6, 5, 6, 7, 8,11,16,18, 9, 7, 7, 5, 5, 6, 7,10,
	15,18, 9, 8, 7, 6, 5, 4, 5, 7,13,17,11,11, 9, 8,
	 7, 4, 3, 5,12,18,13,13,12,10, 7, 5, 4, 3, 8,13,
	16,16,16,13, 8, 8, 7, 5, 8,12,15,18,17,15,11,10,
	 9, 8,10,14,
};

static static_codebook _huff_book__44c9_s_short = {
	2, 100,
	_huff_lengthlist__44c9_s_short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__8c0_s_p1_0[] = {
	 1, 5, 4, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 7, 8, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 7, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 8, 0, 0, 0, 0,
	 0, 0, 8,10,10, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7,10, 9, 0, 0, 0,
	 0, 0, 0, 8, 9,11, 0, 0, 0, 0, 0, 0, 9,11,11, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9,10, 0, 0,
	 0, 0, 0, 0, 9,11,10, 0, 0, 0, 0, 0, 0, 9,11,11,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,11,11, 0,
	 0, 0, 0, 0, 0, 9,10,11, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9,10, 0, 0, 0, 0, 0, 0, 9,11,11,
	 0, 0, 0, 0, 0, 0, 8,11, 9, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c0_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__8c0_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p1_0 = {
	_vq_quantthresh__8c0_s_p1_0,
	_vq_quantmap__8c0_s_p1_0,
	3,
	3
};

static static_codebook _8c0_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__8c0_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__8c0_s_p1_0,
	NULL,
	&_vq_auxt__8c0_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__8c0_s_p2_0[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c0_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__8c0_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p2_0 = {
	_vq_quantthresh__8c0_s_p2_0,
	_vq_quantmap__8c0_s_p2_0,
	5,
	5
};

static static_codebook _8c0_s_p2_0 = {
	4, 625,
	_vq_lengthlist__8c0_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__8c0_s_p2_0,
	NULL,
	&_vq_auxt__8c0_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__8c0_s_p3_0[] = {
	 1, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 6, 7, 7, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 5, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 7, 7, 8, 8,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 7, 7, 8, 8, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c0_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__8c0_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p3_0 = {
	_vq_quantthresh__8c0_s_p3_0,
	_vq_quantmap__8c0_s_p3_0,
	5,
	5
};

static static_codebook _8c0_s_p3_0 = {
	4, 625,
	_vq_lengthlist__8c0_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__8c0_s_p3_0,
	NULL,
	&_vq_auxt__8c0_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__8c0_s_p4_0[] = {
	 1, 2, 3, 7, 7, 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 9, 8, 0, 0, 0, 0, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c0_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__8c0_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p4_0 = {
	_vq_quantthresh__8c0_s_p4_0,
	_vq_quantmap__8c0_s_p4_0,
	9,
	9
};

static static_codebook _8c0_s_p4_0 = {
	2, 81,
	_vq_lengthlist__8c0_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__8c0_s_p4_0,
	NULL,
	&_vq_auxt__8c0_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__8c0_s_p5_0[] = {
	 1, 3, 3, 5, 5, 7, 6, 8, 8, 0, 0, 0, 7, 7, 7, 7,
	 8, 8, 0, 0, 0, 7, 7, 7, 7, 8, 9, 0, 0, 0, 8, 8,
	 8, 8, 9, 9, 0, 0, 0, 8, 8, 8, 8, 9, 9, 0, 0, 0,
	 9, 9, 8, 8,10,10, 0, 0, 0, 9, 9, 8, 8,10,10, 0,
	 0, 0,10,10, 9, 9,10,10, 0, 0, 0, 0, 0, 9, 9,10,
	10,
};

static float _vq_quantthresh__8c0_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__8c0_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p5_0 = {
	_vq_quantthresh__8c0_s_p5_0,
	_vq_quantmap__8c0_s_p5_0,
	9,
	9
};

static static_codebook _8c0_s_p5_0 = {
	2, 81,
	_vq_lengthlist__8c0_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__8c0_s_p5_0,
	NULL,
	&_vq_auxt__8c0_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__8c0_s_p6_0[] = {
	 1, 3, 3, 6, 6, 8, 8, 9, 9, 8, 8,10, 9,10,10,11,
	11, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	11,12, 0, 0, 0, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,11, 0, 0, 0, 8, 8, 9, 9,10,10, 9, 9,10,10,
	11,11,12,12, 0, 0, 0, 8, 8, 9, 9,10,10, 9, 9,11,
	10,11,11,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,10,10,
	11,11,11,12,12,12, 0, 0, 0, 9, 9, 9, 9,10,10,10,
	10,11,11,12,12,13,13, 0, 0, 0,10,10,10,10,11,11,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0,10, 9,10,
	11,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,
	10, 9,10,11,12,12,13,13,14,13, 0, 0, 0, 0, 0, 9,
	 9, 9,10,10,10,11,11,13,12,13,13, 0, 0, 0, 0, 0,
	10,10,10,10,11,11,12,12,13,13,14,14, 0, 0, 0, 0,
	 0, 0, 0,10,10,11,11,12,12,13,13,13,14, 0, 0, 0,
	 0, 0, 0, 0,11,11,11,11,12,12,13,14,14,14, 0, 0,
	 0, 0, 0, 0, 0,11,11,11,11,12,12,13,13,14,13, 0,
	 0, 0, 0, 0, 0, 0,11,11,12,12,13,13,14,14,14,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,14,
	14,
};

static float _vq_quantthresh__8c0_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__8c0_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p6_0 = {
	_vq_quantthresh__8c0_s_p6_0,
	_vq_quantmap__8c0_s_p6_0,
	17,
	17
};

static static_codebook _8c0_s_p6_0 = {
	2, 289,
	_vq_lengthlist__8c0_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__8c0_s_p6_0,
	NULL,
	&_vq_auxt__8c0_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__8c0_s_p7_0[] = {
	 1, 4, 4, 7, 6, 6, 7, 6, 6, 4, 7, 7,11, 9,10,12,
	 9,10, 4, 7, 7,10,10,10,11, 9, 9, 6,11,10,11,11,
	12,11,11,11, 6,10,10,11,11,12,11,10,10, 6, 9,10,
	11,11,11,11,10,10, 7,10,11,12,11,11,12,11,12, 6,
	 9, 9,10, 9, 9,11,10,10, 6, 9, 9,10,10,10,11,10,
	10,
};

static float _vq_quantthresh__8c0_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__8c0_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p7_0 = {
	_vq_quantthresh__8c0_s_p7_0,
	_vq_quantmap__8c0_s_p7_0,
	3,
	3
};

static static_codebook _8c0_s_p7_0 = {
	4, 81,
	_vq_lengthlist__8c0_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__8c0_s_p7_0,
	NULL,
	&_vq_auxt__8c0_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__8c0_s_p7_1[] = {
	 1, 3, 3, 6, 6, 8, 8, 9, 9, 9, 9,10,10,10, 7, 7,
	 8, 8, 9, 9, 9, 9,10,10, 9, 7, 7, 8, 8, 9, 9, 9,
	 9,10,10,10, 8, 8, 9, 9, 9, 9, 9, 9,10,10,10, 8,
	 8, 9, 9, 9, 9, 8, 9,10,10,10, 8, 8, 9, 9, 9,10,
	10,10,10,10,10, 9, 9, 9, 9, 9, 9,10,10,11,10,11,
	 9, 9, 9, 9,10,10,10,10,11,11,11,10,10, 9, 9,10,
	10,10, 9,11,10,10,10,10,10,10, 9, 9,10,10,11,11,
	10,10,10, 9, 9, 9,10,10,10,
};

static float _vq_quantthresh__8c0_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__8c0_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p7_1 = {
	_vq_quantthresh__8c0_s_p7_1,
	_vq_quantmap__8c0_s_p7_1,
	11,
	11
};

static static_codebook _8c0_s_p7_1 = {
	2, 121,
	_vq_lengthlist__8c0_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__8c0_s_p7_1,
	NULL,
	&_vq_auxt__8c0_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__8c0_s_p8_0[] = {
	 1, 4, 4, 7, 6, 7, 7, 7, 7, 8, 8, 9, 9, 7, 6, 6,
	 7, 7, 8, 8, 7, 7, 8, 9,10,10, 7, 6, 6, 7, 7, 8,
	 7, 7, 7, 9, 9,10,12, 0, 8, 8, 8, 8, 8, 9, 8, 8,
	 9, 9,10,10, 0, 8, 8, 8, 8, 8, 9, 8, 9, 9, 9,11,
	10, 0, 0,13, 9, 8, 9, 9, 9, 9,10,10,11,11, 0,13,
	 0, 9, 9, 9, 9, 9, 9,11,10,11,11, 0, 0, 0, 8, 9,
	10, 9,10,10,13,11,12,12, 0, 0, 0, 8, 9, 9, 9,10,
	10,13,12,12,13, 0, 0, 0,12, 0,10,10,12,11,10,11,
	12,12, 0, 0, 0,13,13,10,10,10,11,12, 0,13, 0, 0,
	 0, 0, 0, 0,13,11, 0,12,12,12,13,12, 0, 0, 0, 0,
	 0, 0,13,13,11,13,13,11,12,
};

static float _vq_quantthresh__8c0_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__8c0_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p8_0 = {
	_vq_quantthresh__8c0_s_p8_0,
	_vq_quantmap__8c0_s_p8_0,
	13,
	13
};

static static_codebook _8c0_s_p8_0 = {
	2, 169,
	_vq_lengthlist__8c0_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__8c0_s_p8_0,
	NULL,
	&_vq_auxt__8c0_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__8c0_s_p8_1[] = {
	 1, 3, 4, 5, 5, 7, 6, 6, 6, 5, 7, 7, 7, 6, 6, 7,
	 7, 7, 6, 6, 7, 7, 7, 6, 6,
};

static float _vq_quantthresh__8c0_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__8c0_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p8_1 = {
	_vq_quantthresh__8c0_s_p8_1,
	_vq_quantmap__8c0_s_p8_1,
	5,
	5
};

static static_codebook _8c0_s_p8_1 = {
	2, 25,
	_vq_lengthlist__8c0_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__8c0_s_p8_1,
	NULL,
	&_vq_auxt__8c0_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p9_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__8c0_s_p9_0[] = {
	 1, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__8c0_s_p9_0[] = {
	-157.5, 157.5, 
};

static long _vq_quantmap__8c0_s_p9_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p9_0 = {
	_vq_quantthresh__8c0_s_p9_0,
	_vq_quantmap__8c0_s_p9_0,
	3,
	3
};

static static_codebook _8c0_s_p9_0 = {
	4, 81,
	_vq_lengthlist__8c0_s_p9_0,
	1, -518803456, 1628680192, 2, 0,
	_vq_quantlist__8c0_s_p9_0,
	NULL,
	&_vq_auxt__8c0_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__8c0_s_p9_1[] = {
	 1, 4, 4, 5, 5,10, 8,11,11,11,11,11,11,11,11, 6,
	 6, 6, 7, 6,11,10,11,11,11,11,11,11,11,11, 7, 5,
	 6, 6, 6, 8, 7,11,11,11,11,11,11,11,11,11, 7, 8,
	 8, 8, 9, 9,11,11,11,11,11,11,11,11,11, 9, 8, 7,
	 8, 9,11,11,11,11,11,11,11,11,11,11,11,10,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,
};

static float _vq_quantthresh__8c0_s_p9_1[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__8c0_s_p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p9_1 = {
	_vq_quantthresh__8c0_s_p9_1,
	_vq_quantmap__8c0_s_p9_1,
	15,
	15
};

static static_codebook _8c0_s_p9_1 = {
	2, 225,
	_vq_lengthlist__8c0_s_p9_1,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__8c0_s_p9_1,
	NULL,
	&_vq_auxt__8c0_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__8c0_s_p9_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__8c0_s_p9_2[] = {
	 1, 5, 5, 7, 7, 8, 7, 8, 8,10,10, 9, 9,10,10,10,
	11,11,10,12,11,12,12,12, 9, 8, 8, 8, 8, 8, 9,10,
	10,10,10,11,11,11,10,11,11,12,12,11,12, 8, 8, 7,
	 7, 8, 9,10,10,10, 9,10,10, 9,10,10,11,11,11,11,
	11,11, 9, 9, 9, 9, 8, 9,10,10,11,10,10,11,11,12,
	10,10,12,12,11,11,10, 9, 9,10, 8, 9,10,10,10, 9,
	10,10,11,11,10,11,10,10,10,12,12,12, 9,10, 9,10,
	 9, 9,10,10,11,11,11,11,10,10,10,11,12,11,12,11,
	12,10,11,10,11, 9,10, 9,10, 9,10,10, 9,10,10,11,
	10,11,11,11,11,12,11, 9,10,10,10,10,11,11,11,11,
	11,10,11,11,11,11,10,12,10,12,12,11,12,10,10,11,
	10, 9,11,10,11, 9,10,11,10,10,10,11,11,11,11,12,
	12,10, 9, 9,11,10, 9,12,11,10,12,12,11,11,11,11,
	10,11,11,12,11,10,12, 9,11,10,11,10,10,11,10,11,
	 9,10,10,10,11,12,11,11,12,11,10,10,11,11, 9,10,
	10,12,10,11,10,10,10, 9,10,10,10,10, 9,10,10,11,
	11,11,11,12,11,10,10,10,10,11,11,10,11,11, 9,11,
	10,12,10,12,11,10,11,10,10,10,11,10,10,11,11,10,
	11,10,10,10,10,11,11,12,10,10,10,11,10,11,12,11,
	10,11,10,10,11,11,10,12,10, 9,10,10,11,11,11,10,
	12,10,10,11,11,11,10,10,11,10,10,10,11,10,11,10,
	12,11,11,10,10,10,12,10,10,11, 9,10,11,11,11,10,
	10,11,10,10, 9,11,11,12,12,11,12,11,11,11,11,11,
	11, 9,10,11,10,12,10,10,10,10,11,10,10,11,10,10,
	12,10,10,10,10,10, 9,12,10,10,10,10,12, 9,11,10,
	10,11,10,12,12,10,12,12,12,10,10,10,10, 9,10,11,
	10,10,12,10,10,12,11,10,11,10,10,12,11,10,12,10,
	10,11, 9,11,10, 9,10, 9,10,
};

static float _vq_quantthresh__8c0_s_p9_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__8c0_s_p9_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__8c0_s_p9_2 = {
	_vq_quantthresh__8c0_s_p9_2,
	_vq_quantmap__8c0_s_p9_2,
	21,
	21
};

static static_codebook _8c0_s_p9_2 = {
	2, 441,
	_vq_lengthlist__8c0_s_p9_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__8c0_s_p9_2,
	NULL,
	&_vq_auxt__8c0_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__8c0_s_single[] = {
	 4, 5,18, 7,10, 6, 7, 8, 9,10, 5, 2,18, 5, 7, 5,
	 6, 7, 8,11,17,17,17,17,17,17,17,17,17,17, 7, 4,
	17, 6, 9, 6, 8,10,12,15,11, 7,17, 9, 6, 6, 7, 9,
	11,15, 6, 4,17, 6, 6, 4, 5, 8,11,16, 6, 6,17, 8,
	 6, 5, 6, 9,13,16, 8, 9,17,11, 9, 8, 8,11,13,17,
	 9,12,17,15,14,13,12,13,14,17,12,15,17,17,17,17,
	17,16,17,17,
};

static static_codebook _huff_book__8c0_s_single = {
	2, 100,
	_huff_lengthlist__8c0_s_single,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__8c1_s_p1_0[] = {
	 1, 5, 5, 0, 0, 0, 0, 0, 0, 5, 7, 7, 0, 0, 0, 0,
	 0, 0, 5, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 5, 8, 7, 0, 0, 0, 0, 0, 0, 7, 8, 9, 0, 0, 0,
	 0, 0, 0, 7, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 7, 8, 0, 0, 0, 0, 0, 0, 7, 9, 8, 0, 0,
	 0, 0, 0, 0, 7, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 8, 0, 0, 0, 0,
	 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0, 0,
	 0, 0, 0, 8, 8,10, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 9, 0, 0,
	 0, 0, 0, 0, 8,10, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 5, 8, 8, 0, 0, 0, 0, 0, 0, 8, 9, 9, 0, 0,
	 0, 0, 0, 0, 8, 9, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10, 0,
	 0, 0, 0, 0, 0, 8, 9,10, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 7, 9, 9, 0, 0, 0, 0, 0, 0, 9,10,10,
	 0, 0, 0, 0, 0, 0, 8,10, 8, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c1_s_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__8c1_s_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p1_0 = {
	_vq_quantthresh__8c1_s_p1_0,
	_vq_quantmap__8c1_s_p1_0,
	3,
	3
};

static static_codebook _8c1_s_p1_0 = {
	8, 6561,
	_vq_lengthlist__8c1_s_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__8c1_s_p1_0,
	NULL,
	&_vq_auxt__8c1_s_p1_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__8c1_s_p2_0[] = {
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c1_s_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__8c1_s_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p2_0 = {
	_vq_quantthresh__8c1_s_p2_0,
	_vq_quantmap__8c1_s_p2_0,
	5,
	5
};

static static_codebook _8c1_s_p2_0 = {
	4, 625,
	_vq_lengthlist__8c1_s_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__8c1_s_p2_0,
	NULL,
	&_vq_auxt__8c1_s_p2_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__8c1_s_p3_0[] = {
	 2, 4, 4, 5, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 4, 4, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 4, 4, 4, 6, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 6, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 6, 6, 6, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c1_s_p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__8c1_s_p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p3_0 = {
	_vq_quantthresh__8c1_s_p3_0,
	_vq_quantmap__8c1_s_p3_0,
	5,
	5
};

static static_codebook _8c1_s_p3_0 = {
	4, 625,
	_vq_lengthlist__8c1_s_p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__8c1_s_p3_0,
	NULL,
	&_vq_auxt__8c1_s_p3_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p4_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__8c1_s_p4_0[] = {
	 1, 2, 3, 7, 7, 0, 0, 0, 0, 0, 0, 0, 6, 6, 0, 0,
	 0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0, 0, 0, 7, 7,
	 0, 0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0,
	 8, 8, 0, 0, 0, 0, 0, 0, 0, 9, 8, 0, 0, 0, 0, 0,
	 0, 0,10,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	 0,
};

static float _vq_quantthresh__8c1_s_p4_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__8c1_s_p4_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p4_0 = {
	_vq_quantthresh__8c1_s_p4_0,
	_vq_quantmap__8c1_s_p4_0,
	9,
	9
};

static static_codebook _8c1_s_p4_0 = {
	2, 81,
	_vq_lengthlist__8c1_s_p4_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__8c1_s_p4_0,
	NULL,
	&_vq_auxt__8c1_s_p4_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__8c1_s_p5_0[] = {
	 1, 3, 3, 4, 5, 6, 6, 8, 8, 0, 0, 0, 8, 8, 7, 7,
	 9, 9, 0, 0, 0, 8, 8, 7, 7, 9, 9, 0, 0, 0, 9,10,
	 8, 8, 9, 9, 0, 0, 0,10,10, 8, 8, 9, 9, 0, 0, 0,
	11,10, 8, 8,10,10, 0, 0, 0,11,11, 8, 8,10,10, 0,
	 0, 0,12,12, 9, 9,10,10, 0, 0, 0, 0, 0, 9, 9,10,
	10,
};

static float _vq_quantthresh__8c1_s_p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__8c1_s_p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p5_0 = {
	_vq_quantthresh__8c1_s_p5_0,
	_vq_quantmap__8c1_s_p5_0,
	9,
	9
};

static static_codebook _8c1_s_p5_0 = {
	2, 81,
	_vq_lengthlist__8c1_s_p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__8c1_s_p5_0,
	NULL,
	&_vq_auxt__8c1_s_p5_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p6_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__8c1_s_p6_0[] = {
	 1, 3, 3, 5, 5, 8, 8, 8, 8, 9, 9,10,10,11,11,11,
	11, 0, 0, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,11,
	12,12, 0, 0, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,12, 0, 0, 0, 9, 9, 8, 8,10,10,10,10,11,11,
	12,12,12,12, 0, 0, 0, 9, 9, 8, 8,10,10,10,10,11,
	11,12,12,12,12, 0, 0, 0,10,10, 9, 9,10,10,10,10,
	11,11,12,12,13,13, 0, 0, 0,10,10, 9, 9,10,10,10,
	10,11,11,12,12,13,13, 0, 0, 0,11,11, 9, 9,10,10,
	10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,10,
	10,10,10,11,11,12,12,13,13, 0, 0, 0, 0, 0, 9, 9,
	10,10,11,11,12,12,12,12,13,13, 0, 0, 0, 0, 0, 9,
	 9,10,10,11,11,12,11,12,12,13,13, 0, 0, 0, 0, 0,
	10,10,11,11,11,11,12,12,13,12,13,13, 0, 0, 0, 0,
	 0, 0, 0,11,10,11,11,12,12,13,13,13,13, 0, 0, 0,
	 0, 0, 0, 0,11,11,12,12,12,12,13,13,13,14, 0, 0,
	 0, 0, 0, 0, 0,11,11,12,12,12,12,13,13,14,13, 0,
	 0, 0, 0, 0, 0, 0,12,12,12,12,13,13,13,13,14,14,
	 0, 0, 0, 0, 0, 0, 0, 0, 0,12,12,13,13,13,13,14,
	14,
};

static float _vq_quantthresh__8c1_s_p6_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__8c1_s_p6_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p6_0 = {
	_vq_quantthresh__8c1_s_p6_0,
	_vq_quantmap__8c1_s_p6_0,
	17,
	17
};

static static_codebook _8c1_s_p6_0 = {
	2, 289,
	_vq_lengthlist__8c1_s_p6_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__8c1_s_p6_0,
	NULL,
	&_vq_auxt__8c1_s_p6_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__8c1_s_p7_0[] = {
	 1, 4, 4, 6, 6, 6, 7, 6, 6, 4, 7, 7,10, 9, 9,10,
	 9, 9, 5, 7, 7,10, 9, 9,10, 9, 9, 6,10,10,10,10,
	10,11,10,10, 6, 9, 9,10, 9,10,11,10,10, 6, 9, 9,
	10, 9, 9,11, 9,10, 7,10,10,11,11,11,11,10,10, 6,
	 9, 9,10,10,10,11, 9, 9, 6, 9, 9,10,10,10,10, 9,
	 9,
};

static float _vq_quantthresh__8c1_s_p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__8c1_s_p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p7_0 = {
	_vq_quantthresh__8c1_s_p7_0,
	_vq_quantmap__8c1_s_p7_0,
	3,
	3
};

static static_codebook _8c1_s_p7_0 = {
	4, 81,
	_vq_lengthlist__8c1_s_p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__8c1_s_p7_0,
	NULL,
	&_vq_auxt__8c1_s_p7_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__8c1_s_p7_1[] = {
	 2, 3, 3, 5, 5, 7, 7, 7, 7, 7, 7,10,10, 9, 7, 7,
	 7, 7, 8, 8, 8, 8, 9, 9, 9, 7, 7, 7, 7, 8, 8, 8,
	 8,10,10,10, 7, 7, 7, 7, 8, 8, 8, 8,10,10,10, 7,
	 7, 7, 7, 8, 8, 8, 8,10,10,10, 8, 8, 8, 8, 8, 8,
	 8, 8,10,10,10, 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,
	 8, 8, 8, 8, 8, 8, 8, 8,10,10,10,10,10, 8, 8, 8,
	 8, 8, 8,10,10,10,10,10, 8, 8, 8, 8, 8, 8,10,10,
	10,10,10, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__8c1_s_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__8c1_s_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p7_1 = {
	_vq_quantthresh__8c1_s_p7_1,
	_vq_quantmap__8c1_s_p7_1,
	11,
	11
};

static static_codebook _8c1_s_p7_1 = {
	2, 121,
	_vq_lengthlist__8c1_s_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__8c1_s_p7_1,
	NULL,
	&_vq_auxt__8c1_s_p7_1,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p8_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__8c1_s_p8_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 8, 8, 9, 9,10,10, 7, 5, 5,
	 7, 7, 8, 8, 8, 8, 9,10,11,11, 7, 5, 5, 7, 7, 8,
	 8, 9, 9,10,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,
	 9,10,11,11, 0, 8, 8, 8, 8, 9, 9, 9, 9,10,10,11,
	11, 0,12,12, 9, 9, 9, 9,10, 9,10,11,11,11, 0,13,
	12, 9, 8, 9, 9,10,10,11,11,12,11, 0, 0, 0, 9, 9,
	 9, 9,10,10,11,11,12,12, 0, 0, 0,10,10, 9, 9,10,
	10,11,11,12,12, 0, 0, 0,13,13,10,10,11,11,12,11,
	13,12, 0, 0, 0,14,14,10,10,11,10,11,11,12,12, 0,
	 0, 0, 0, 0,12,12,11,11,12,12,13,13, 0, 0, 0, 0,
	 0,12,12,11,10,12,11,13,12,
};

static float _vq_quantthresh__8c1_s_p8_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__8c1_s_p8_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p8_0 = {
	_vq_quantthresh__8c1_s_p8_0,
	_vq_quantmap__8c1_s_p8_0,
	13,
	13
};

static static_codebook _8c1_s_p8_0 = {
	2, 169,
	_vq_lengthlist__8c1_s_p8_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__8c1_s_p8_0,
	NULL,
	&_vq_auxt__8c1_s_p8_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p8_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__8c1_s_p8_1[] = {
	 2, 3, 3, 5, 5, 6, 6, 6, 5, 5, 6, 6, 6, 5, 5, 6,
	 6, 6, 5, 5, 6, 6, 6, 5, 5,
};

static float _vq_quantthresh__8c1_s_p8_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__8c1_s_p8_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p8_1 = {
	_vq_quantthresh__8c1_s_p8_1,
	_vq_quantmap__8c1_s_p8_1,
	5,
	5
};

static static_codebook _8c1_s_p8_1 = {
	2, 25,
	_vq_lengthlist__8c1_s_p8_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__8c1_s_p8_1,
	NULL,
	&_vq_auxt__8c1_s_p8_1,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__8c1_s_p9_0[] = {
	 1, 3, 3,10,10,10,10,10,10,10,10,10,10, 5, 6, 6,
	10,10,10,10,10,10,10,10,10,10, 6, 7, 8,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10, 9, 9, 9, 9,
};

static float _vq_quantthresh__8c1_s_p9_0[] = {
	-1732.5, -1417.5, -1102.5, -787.5, -472.5, -157.5, 157.5, 472.5, 
	787.5, 1102.5, 1417.5, 1732.5, 
};

static long _vq_quantmap__8c1_s_p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p9_0 = {
	_vq_quantthresh__8c1_s_p9_0,
	_vq_quantmap__8c1_s_p9_0,
	13,
	13
};

static static_codebook _8c1_s_p9_0 = {
	2, 169,
	_vq_lengthlist__8c1_s_p9_0,
	1, -513964032, 1628680192, 4, 0,
	_vq_quantlist__8c1_s_p9_0,
	NULL,
	&_vq_auxt__8c1_s_p9_0,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__8c1_s_p9_1[] = {
	 1, 4, 4, 5, 5, 7, 7, 9, 9,11,11,12,12,13,13, 6,
	 5, 5, 6, 6, 9, 9,10,10,12,12,12,13,15,14, 6, 5,
	 5, 7, 7, 9, 9,10,10,12,12,12,13,14,13,17, 7, 7,
	 8, 8,10,10,11,11,12,13,13,13,13,13,17, 7, 7, 8,
	 8,10,10,11,11,13,13,13,13,14,14,17,11,11, 9, 9,
	11,11,12,12,12,13,13,14,15,13,17,12,12, 9, 9,11,
	11,12,12,13,13,13,13,14,16,17,17,17,11,12,12,12,
	13,13,13,14,15,14,15,15,17,17,17,12,12,11,11,13,
	13,14,14,15,14,15,15,17,17,17,15,15,13,13,14,14,
	15,14,15,15,16,15,17,17,17,15,15,13,13,13,14,14,
	15,15,15,15,16,17,17,17,17,16,14,15,14,14,15,14,
	14,15,15,15,17,17,17,17,17,14,14,16,14,15,15,15,
	15,15,15,17,17,17,17,17,17,16,16,15,17,15,15,14,
	17,15,17,16,17,17,17,17,16,15,14,15,15,15,15,15,
	15,
};

static float _vq_quantthresh__8c1_s_p9_1[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__8c1_s_p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p9_1 = {
	_vq_quantthresh__8c1_s_p9_1,
	_vq_quantmap__8c1_s_p9_1,
	15,
	15
};

static static_codebook _8c1_s_p9_1 = {
	2, 225,
	_vq_lengthlist__8c1_s_p9_1,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__8c1_s_p9_1,
	NULL,
	&_vq_auxt__8c1_s_p9_1,
	NULL,
	0
};

static long _vq_quantlist__8c1_s_p9_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__8c1_s_p9_2[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 9, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,11,11,12, 7, 7, 7, 7, 8, 8, 9, 9,
	 9, 9,10,10,10,10,10,10,10,10,11,11,11, 7, 7, 7,
	 7, 8, 8, 9, 8, 9, 9, 9, 9, 9, 9,10,10,10,10,11,
	11,12, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,10,10,10,10,
	10,10,10,10,11,11,11, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9,10,10,10,10,10,10,10,10,11,11,11, 8, 8, 8, 8,
	 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,11,11,
	11, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,10,10,10,10,10,
	10,10,10,11,12,11, 9, 9, 8, 9, 9, 9, 9, 9,10,10,
	10,10,10,10,10,10,10,10,11,11,11,11,11, 8, 8, 9,
	 9, 9, 9,10,10,10,10,10,10,10,10,10,10,11,12,11,
	12,11, 9, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,
	10,10,11,11,11,11,11, 9, 9, 9, 9,10,10,10,10,10,
	10,10,10,10,10,10,10,12,11,12,11,11, 9, 9, 9,10,
	10,10,10,10,10,10,10,10,10,10,10,10,12,11,11,11,
	11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,
	11,11,11,12,11,11,12,11,10,10,10,10,10,10,10,10,
	10,10,10,10,11,10,11,11,11,11,11,11,11,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,11,11,12,11,12,
	11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	11,11,12,11,12,11,11,11,11,10,10,10,10,10,10,10,
	10,10,10,10,10,11,11,12,11,11,12,11,11,12,10,10,
	11,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,
	11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,12,
	12,11,12,11,11,12,12,12,11,11,10,10,10,10,10,10,
	10,10,10,11,12,12,11,12,12,11,12,11,11,11,11,10,
	10,10,10,10,10,10,10,10,10,
};

static float _vq_quantthresh__8c1_s_p9_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__8c1_s_p9_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__8c1_s_p9_2 = {
	_vq_quantthresh__8c1_s_p9_2,
	_vq_quantmap__8c1_s_p9_2,
	21,
	21
};

static static_codebook _8c1_s_p9_2 = {
	2, 441,
	_vq_lengthlist__8c1_s_p9_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__8c1_s_p9_2,
	NULL,
	&_vq_auxt__8c1_s_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__8c1_s_single[] = {
	 4, 6,18, 8,11, 8, 8, 9, 9,10, 4, 4,18, 5, 9, 5,
	 6, 7, 8,10,18,18,18,18,17,17,17,17,17,17, 7, 5,
	17, 6,11, 6, 7, 8, 9,12,12, 9,17,12, 8, 8, 9,10,
	10,13, 7, 5,17, 6, 8, 4, 5, 6, 8,10, 6, 5,17, 6,
	 8, 5, 4, 5, 7, 9, 7, 7,17, 8, 9, 6, 5, 5, 6, 8,
	 8, 8,17, 9,11, 8, 6, 6, 6, 7, 9,10,17,12,12,10,
	 9, 7, 7, 8,
};

static static_codebook _huff_book__8c1_s_single = {
	2, 100,
	_huff_lengthlist__8c1_s_single,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

/***** residue backends *********************************************/

static vorbis_info_residue0 _residue_44_mid={
  0,-1, -1, 10,-1,
  /*  0     1     2     3     4     5     6     7     8  */
  {0},
  {-1},
  {  .5,  1.5,  1.5,  2.5,  2.5,  4.5,  8.5,  16.5, 32.5},
  {  .5,   .5, 999.,   .5,  999., 4.5,  8.5,  16.5, 32.5},
};

static vorbis_info_residue0 _residue_44_high={
  0,-1, -1, 10,-1,
  /*  0     1     2     3     4     5     6     7     8  */
  {0},
  {-1},
  {  .5,  1.5,  2.5,  4.5,  8.5, 16.5, 32.5, 71.5,157.5},
  {  .5,  1.5,  2.5,  3.5,  4.5,  8.5, 16.5, 71.5,157.5},
};

static static_bookblock _resbook_44s_0={
  {
    {0},{0,0,&_44c0_s_p1_0},{0,0,&_44c0_s_p2_0},{0,0,&_44c0_s_p3_0},
    {0,0,&_44c0_s_p4_0},{0,0,&_44c0_s_p5_0},{0,0,&_44c0_s_p6_0},
    {&_44c0_s_p7_0,&_44c0_s_p7_1},{&_44c0_s_p8_0,&_44c0_s_p8_1},
    {&_44c0_s_p9_0,&_44c0_s_p9_1,&_44c0_s_p9_2}
   }
};
static static_bookblock _resbook_44sm_0={
  {
    {0},{0,0,&_44c0_sm_p1_0},{0,0,&_44c0_sm_p2_0},{0,0,&_44c0_sm_p3_0},
    {0,0,&_44c0_sm_p4_0},{0,0,&_44c0_sm_p5_0},{0,0,&_44c0_sm_p6_0},
    {&_44c0_sm_p7_0,&_44c0_sm_p7_1},{&_44c0_sm_p8_0,&_44c0_sm_p8_1},
    {&_44c0_sm_p9_0,&_44c0_sm_p9_1,&_44c0_sm_p9_2}
   }
};
static static_bookblock _resbook_44s_1={
  {
    {0},{0,0,&_44c1_s_p1_0},{0,0,&_44c1_s_p2_0},{0,0,&_44c1_s_p3_0},
    {0,0,&_44c1_s_p4_0},{0,0,&_44c1_s_p5_0},{0,0,&_44c1_s_p6_0},
    {&_44c1_s_p7_0,&_44c1_s_p7_1},{&_44c1_s_p8_0,&_44c1_s_p8_1},
    {&_44c1_s_p9_0,&_44c1_s_p9_1,&_44c1_s_p9_2}
   }
};
static static_bookblock _resbook_44sm_1={
  {
    {0},{0,0,&_44c1_sm_p1_0},{0,0,&_44c1_sm_p2_0},{0,0,&_44c1_sm_p3_0},
    {0,0,&_44c1_sm_p4_0},{0,0,&_44c1_sm_p5_0},{0,0,&_44c1_sm_p6_0},
    {&_44c1_sm_p7_0,&_44c1_sm_p7_1},{&_44c1_sm_p8_0,&_44c1_sm_p8_1},
    {&_44c1_sm_p9_0,&_44c1_sm_p9_1,&_44c1_sm_p9_2}
   }
};
static static_bookblock _resbook_44s_2={
  {
    {0},{0,0,&_44c2_s_p1_0},{0,0,&_44c2_s_p2_0},{0,0,&_44c2_s_p3_0},
    {0,0,&_44c2_s_p4_0},{0,0,&_44c2_s_p5_0},{0,0,&_44c2_s_p6_0},
    {&_44c2_s_p7_0,&_44c2_s_p7_1},{&_44c2_s_p8_0,&_44c2_s_p8_1},
    {&_44c2_s_p9_0,&_44c2_s_p9_1,&_44c2_s_p9_2}
   }
};
static static_bookblock _resbook_44s_3={
  {
    {0},{0,0,&_44c3_s_p1_0},{0,0,&_44c3_s_p2_0},{0,0,&_44c3_s_p3_0},
    {0,0,&_44c3_s_p4_0},{0,0,&_44c3_s_p5_0},{0,0,&_44c3_s_p6_0},
    {&_44c3_s_p7_0,&_44c3_s_p7_1},{&_44c3_s_p8_0,&_44c3_s_p8_1},
    {&_44c3_s_p9_0,&_44c3_s_p9_1,&_44c3_s_p9_2}
   }
};
static static_bookblock _resbook_44s_4={
  {
    {0},{0,0,&_44c4_s_p1_0},{0,0,&_44c4_s_p2_0},{0,0,&_44c4_s_p3_0},
    {0,0,&_44c4_s_p4_0},{0,0,&_44c4_s_p5_0},{0,0,&_44c4_s_p6_0},
    {&_44c4_s_p7_0,&_44c4_s_p7_1},{&_44c4_s_p8_0,&_44c4_s_p8_1},
    {&_44c4_s_p9_0,&_44c4_s_p9_1,&_44c4_s_p9_2}
   }
};
static static_bookblock _resbook_44s_5={
  {
    {0},{0,0,&_44c5_s_p1_0},{0,0,&_44c5_s_p2_0},{0,0,&_44c5_s_p3_0},
    {0,0,&_44c5_s_p4_0},{0,0,&_44c5_s_p5_0},{0,0,&_44c5_s_p6_0},
    {&_44c5_s_p7_0,&_44c5_s_p7_1},{&_44c5_s_p8_0,&_44c5_s_p8_1},
    {&_44c5_s_p9_0,&_44c5_s_p9_1,&_44c5_s_p9_2}
   }
};
static static_bookblock _resbook_44s_6={
  {
    {0},{0,0,&_44c6_s_p1_0},{0,0,&_44c6_s_p2_0},{0,0,&_44c6_s_p3_0},
    {0,0,&_44c6_s_p4_0},
    {&_44c6_s_p5_0,&_44c6_s_p5_1},
    {&_44c6_s_p6_0,&_44c6_s_p6_1},
    {&_44c6_s_p7_0,&_44c6_s_p7_1},
    {&_44c6_s_p8_0,&_44c6_s_p8_1},
    {&_44c6_s_p9_0,&_44c6_s_p9_1,&_44c6_s_p9_2}
   }
};
static static_bookblock _resbook_44s_7={
  {
    {0},{0,0,&_44c7_s_p1_0},{0,0,&_44c7_s_p2_0},{0,0,&_44c7_s_p3_0},
    {0,0,&_44c7_s_p4_0},
    {&_44c7_s_p5_0,&_44c7_s_p5_1},
    {&_44c7_s_p6_0,&_44c7_s_p6_1},
    {&_44c7_s_p7_0,&_44c7_s_p7_1},
    {&_44c7_s_p8_0,&_44c7_s_p8_1},
    {&_44c7_s_p9_0,&_44c7_s_p9_1,&_44c7_s_p9_2}
   }
};
static static_bookblock _resbook_44s_8={
  {
    {0},{0,0,&_44c8_s_p1_0},{0,0,&_44c8_s_p2_0},{0,0,&_44c8_s_p3_0},
    {0,0,&_44c8_s_p4_0},
    {&_44c8_s_p5_0,&_44c8_s_p5_1},
    {&_44c8_s_p6_0,&_44c8_s_p6_1},
    {&_44c8_s_p7_0,&_44c8_s_p7_1},
    {&_44c8_s_p8_0,&_44c8_s_p8_1},
    {&_44c8_s_p9_0,&_44c8_s_p9_1,&_44c8_s_p9_2}
   }
};
static static_bookblock _resbook_44s_9={
  {
    {0},{0,0,&_44c9_s_p1_0},{0,0,&_44c9_s_p2_0},{0,0,&_44c9_s_p3_0},
    {0,0,&_44c9_s_p4_0},
    {&_44c9_s_p5_0,&_44c9_s_p5_1},
    {&_44c9_s_p6_0,&_44c9_s_p6_1},
    {&_44c9_s_p7_0,&_44c9_s_p7_1},
    {&_44c9_s_p8_0,&_44c9_s_p8_1},
    {&_44c9_s_p9_0,&_44c9_s_p9_1,&_44c9_s_p9_2}
   }
};


static vorbis_residue_template _res_44s_0[]={
  {2,0,  &_residue_44_mid,
   &_huff_book__44c0_s_short,&_huff_book__44c0_sm_short,
   &_resbook_44s_0,&_resbook_44sm_0},

  {2,0,  &_residue_44_mid,
   &_huff_book__44c0_s_long,&_huff_book__44c0_sm_long,
   &_resbook_44s_0,&_resbook_44sm_0}
};
static vorbis_residue_template _res_44s_1[]={
  {2,0,  &_residue_44_mid,
   &_huff_book__44c1_s_short,&_huff_book__44c1_sm_short,
   &_resbook_44s_1,&_resbook_44sm_1},

  {2,0,  &_residue_44_mid,
   &_huff_book__44c1_s_long,&_huff_book__44c1_sm_long,
   &_resbook_44s_1,&_resbook_44sm_1}
};
static vorbis_residue_template _res_44s_2[]={
  {2,0,  &_residue_44_mid,
   &_huff_book__44c2_s_short,&_huff_book__44c2_s_short,
   &_resbook_44s_2,&_resbook_44s_2},

  {2,0,  &_residue_44_mid,
   &_huff_book__44c2_s_long,&_huff_book__44c2_s_long,
   &_resbook_44s_2,&_resbook_44s_2}
};
static vorbis_residue_template _res_44s_3[]={
  {2,0,  &_residue_44_mid,
   &_huff_book__44c3_s_short,&_huff_book__44c3_s_short,
   &_resbook_44s_3,&_resbook_44s_3},

  {2,0,  &_residue_44_mid,
   &_huff_book__44c3_s_long,&_huff_book__44c3_s_long,
   &_resbook_44s_3,&_resbook_44s_3}
};
static vorbis_residue_template _res_44s_4[]={
  {2,0,  &_residue_44_mid,
   &_huff_book__44c4_s_short,&_huff_book__44c4_s_short,
   &_resbook_44s_4,&_resbook_44s_4},

  {2,0,  &_residue_44_mid,
   &_huff_book__44c4_s_long,&_huff_book__44c4_s_long,
   &_resbook_44s_4,&_resbook_44s_4}
};
static vorbis_residue_template _res_44s_5[]={
  {2,0,  &_residue_44_mid,
   &_huff_book__44c5_s_short,&_huff_book__44c5_s_short,
   &_resbook_44s_5,&_resbook_44s_5},

  {2,0,  &_residue_44_mid,
   &_huff_book__44c5_s_long,&_huff_book__44c5_s_long,
   &_resbook_44s_5,&_resbook_44s_5}
};
static vorbis_residue_template _res_44s_6[]={
  {2,0,  &_residue_44_high,
   &_huff_book__44c6_s_short,&_huff_book__44c6_s_short,
   &_resbook_44s_6,&_resbook_44s_6},

  {2,0,  &_residue_44_high,
   &_huff_book__44c6_s_long,&_huff_book__44c6_s_long,
   &_resbook_44s_6,&_resbook_44s_6}
};
static vorbis_residue_template _res_44s_7[]={
  {2,0,  &_residue_44_high,
   &_huff_book__44c7_s_short,&_huff_book__44c7_s_short,
   &_resbook_44s_7,&_resbook_44s_7},

  {2,0,  &_residue_44_high,
   &_huff_book__44c7_s_long,&_huff_book__44c7_s_long,
   &_resbook_44s_7,&_resbook_44s_7}
};
static vorbis_residue_template _res_44s_8[]={
  {2,0,  &_residue_44_high,
   &_huff_book__44c8_s_short,&_huff_book__44c8_s_short,
   &_resbook_44s_8,&_resbook_44s_8},

  {2,0,  &_residue_44_high,
   &_huff_book__44c8_s_long,&_huff_book__44c8_s_long,
   &_resbook_44s_8,&_resbook_44s_8}
};
static vorbis_residue_template _res_44s_9[]={
  {2,0,  &_residue_44_high,
   &_huff_book__44c9_s_short,&_huff_book__44c9_s_short,
   &_resbook_44s_9,&_resbook_44s_9},

  {2,0,  &_residue_44_high,
   &_huff_book__44c9_s_long,&_huff_book__44c9_s_long,
   &_resbook_44s_9,&_resbook_44s_9}
};

static vorbis_mapping_template _mapres_template_44_stereo[]={
  { _map_nominal, _res_44s_0 }, /* 0 */
  { _map_nominal, _res_44s_1 }, /* 1 */
  { _map_nominal, _res_44s_2 }, /* 2 */
  { _map_nominal, _res_44s_3 }, /* 3 */
  { _map_nominal, _res_44s_4 }, /* 4 */
  { _map_nominal, _res_44s_5 }, /* 5 */
  { _map_nominal, _res_44s_6 }, /* 6 */
  { _map_nominal, _res_44s_7 }, /* 7 */
  { _map_nominal, _res_44s_8 }, /* 8 */
  { _map_nominal, _res_44s_9 }, /* 9 */
};

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: key psychoacoustic settings for 44.1/48kHz

 ********************************************************************/


/* preecho trigger settings *****************************************/

static vorbis_info_psy_global _psy_global_44[5]={

  {8,   /* lines per eighth octave */
   {20.f,14.f,12.f,12.f,12.f,12.f,12.f}, 
   {-60.f,-30.f,-40.f,-40.f,-40.f,-40.f,-40.f}, 2,-75.f,
   -6.f,
   {99.},{{99.},{99.}},{0},{0},{{0.},{0.}}
  },
  {8,   /* lines per eighth octave */
   {14.f,10.f,10.f,10.f,10.f,10.f,10.f}, 
   {-40.f,-30.f,-25.f,-25.f,-25.f,-25.f,-25.f}, 2,-80.f,
   -6.f,
   {99.},{{99.},{99.}},{0},{0},{{0.},{0.}}
  },
  {8,   /* lines per eighth octave */
   {12.f,10.f,10.f,10.f,10.f,10.f,10.f}, 
   {-20.f,-20.f,-15.f,-15.f,-15.f,-15.f,-15.f}, 0,-80.f,
   -6.f,
   {99.},{{99.},{99.}},{0},{0},{{0.},{0.}}
  },
  {8,   /* lines per eighth octave */
   {10.f,8.f,8.f,8.f,8.f,8.f,8.f}, 
   {-20.f,-15.f,-12.f,-12.f,-12.f,-12.f,-12.f}, 0,-80.f,
   -6.f,
   {99.},{{99.},{99.}},{0},{0},{{0.},{0.}}
  },
  {8,   /* lines per eighth octave */
   {10.f,6.f,6.f,6.f,6.f,6.f,6.f}, 
   {-15.f,-15.f,-12.f,-12.f,-12.f,-12.f,-12.f}, 0,-85.f,
   -6.f,
   {99.},{{99.},{99.}},{0},{0},{{0.},{0.}}
  },
};

/* noise compander lookups * low, mid, high quality ****************/
static compandblock _psy_compand_44[6]={
  /* sub-mode Z short */
  {{
    0, 1, 2, 3, 4, 5, 6,  7,     /* 7dB */
    8, 9,10,11,12,13,14, 15,     /* 15dB */
    16,17,18,19,20,21,22, 23,     /* 23dB */
    24,25,26,27,28,29,30, 31,     /* 31dB */
    32,33,34,35,36,37,38, 39,     /* 39dB */
  }},
  /* mode_Z nominal short*/
  {{
     0, 1, 2, 3, 4, 5, 6,  6,     /* 7dB */
     7, 7, 7, 7, 6, 6, 6,  7,     /* 15dB */
     7, 8, 9,10,11,12,13, 14,     /* 23dB */
    15,16,17,17,17,18,18, 19,     /* 31dB */
    19,19,20,21,22,23,24, 25,     /* 39dB */
  }},
  /* mode A short */
  {{
    0, 1, 2, 3, 4, 5, 5,  5,     /* 7dB */
    6, 6, 6, 5, 4, 4, 4,  4,     /* 15dB */
    4, 4, 5, 5, 5, 6, 6,  6,     /* 23dB */
    7, 7, 7, 8, 8, 8, 9, 10,     /* 31dB */
    11,12,13,14,15,16,17, 18,     /* 39dB */
  }},
  /* sub-mode Z long */
  {{
     0, 1, 2, 3, 4, 5, 6,  7,     /* 7dB */
     8, 9,10,11,12,13,14, 15,     /* 15dB */
    16,17,18,19,20,21,22, 23,     /* 23dB */
    24,25,26,27,28,29,30, 31,     /* 31dB */
    32,33,34,35,36,37,38, 39,     /* 39dB */
  }},
  /* mode_Z nominal long */
  {{
    0, 1, 2, 3, 4, 5, 6,  7,     /* 7dB */
    8, 9,10,11,12,12,13, 13,     /* 15dB */
    13,14,14,14,15,15,15, 15,     /* 23dB */
    16,16,17,17,17,18,18, 19,     /* 31dB */
    19,19,20,21,22,23,24, 25,     /* 39dB */
  }},
  /* mode A long */
  {{
    0, 1, 2, 3, 4, 5, 6,  7,     /* 7dB */
    8, 8, 7, 6, 5, 4, 4,  4,     /* 15dB */
    4, 4, 5, 5, 5, 6, 6,  6,     /* 23dB */
    7, 7, 7, 8, 8, 8, 9, 10,     /* 31dB */
    11,12,13,14,15,16,17, 18,     /* 39dB */
  }}
};

/* tonal masking curve level adjustments *************************/
static vp_adjblock _vp_tonemask_adj_longblock[11]={
  /* adjust for mode zero */
  /* 63     125     250     500     1     2     4     8    16 */
  {{-15,-15,-15,-15,-10, -8, -4,-2, 0, 0, 0,10, 0, 0, 0, 0, 0}}, /* 0 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 5, 0, 0, 0, 0, 0}}, /* 1 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 2 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 3 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 4 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 5 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 6 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 7 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 8 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 9 */
  {{-15,-15,-15,-15,-15,-12,-10,-8, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 10 */
};
static vp_adjblock _vp_tonemask_adj_otherblock[11]={
  /* adjust for mode zero */
  /* 63     125     250     500       1     2     4     8    16 */
  {{-20,-20,-20,-20,-14,-12,-10, -8, -4, 0, 0,10, 0, 0, 0, 0, 0}}, /* 0 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 5, 0, 0, 0, 0, 0}}, /* 1 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 2 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 3 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 4 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 5 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 6 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 7 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 8 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 9 */
  {{-20,-20,-20,-20,-20,-18,-16,-14,-10, 0, 0, 0, 0, 0, 0, 0, 0}}, /* 10 */
};

static noise3 _psy_noisebias_trans_low[2]={
  /*  63     125     250     500      1k       2k      4k      8k     16k*/
  /* 0 */
  {{{-10,-10,-10,-10,-10, -4,  0,  0,  4,  8,  8,  8,  8, 10, 12, 14, 20},
    {-30,-30,-30,-30,-26,-20,-16, -8, -6, -6, -2,  2,  2,  4,  8,  8, 15},
    {-30,-30,-30,-30,-30,-24,-20,-14,-10, -6, -8, -8, -6, -6, -6, -4, -2}}},
  /* 1 */
  {{{-15,-15,-15,-15,-15,-10, -5,  0,  2,  2,  6,  6,  6,  8, 10, 12, 15},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -2,  0,  0,  0,  2,  4,  10},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -6, -6, -6, -4, -4, -4,  -2}}},
};
static noise3 _psy_noisebias_long_low[2]={
    /*63     125     250     500      1k       2k      4k      8k     16k*/
  /* 0 */
  {{{-10,-10,-10,-10,-10, -4,  0,  0,  0,  6,  6,  6,  6, 10, 10, 12,  20},
    {-20,-20,-20,-20,-20,-20,-10, -2,  0,  0,  0,  0,  0,  2,  4,  6,  15},
    {-20,-20,-20,-20,-20,-20,-20,-10, -6, -6, -6, -6, -6, -4, -4, -4, -2}}},
  /* 1 */
  {{{-10,-10,-10,-10,-10,-10, -8, -8,  0,  2,  4,  4,  5,  5,  5,  8,  10},
    {-20,-20,-20,-20,-20,-20,-20,-14, -8, -2,  0,  0,  0,  0,  2,  4,  10},
    {-20,-20,-20,-20,-20,-20,-20,-14, -6, -6, -6, -6, -6, -4, -4, -4, -2}}},
};

static noise3 _psy_noisebias_trans[11]={
  /*  63     125     250     500      1k       2k      4k      8k     16k*/
  /* 0 */
  {{{-15,-15,-15,-15,-15,-12,-10, -8,  0,  2,  4,  4,  5,  5,  5,  8,  10},
    {-30,-30,-30,-30,-26,-22,-20,-14, -8, -4,  0,  0,  0,  0,  2,  4,  10},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -6, -6, -6, -4, -4, -4,  -2}}},
  /* 1 */
  {{{-15,-15,-15,-15,-15,-12,-10, -8,  0,  2,  4,  4,  5,  5,  5,  8,  10},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -2, -2, -2, -2,  0,  2,  8},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -8, -8, -8, -8, -6, -6, -6, -4}}},
  /* 2 */
  {{{-15,-15,-15,-15,-15,-12,-10, -8,  0,  2,  2,  2,  4,  4,  5,  6,  10},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -2, -2, -2, -2,  0,  2,  6},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10,-10,-10,-10,-10, -8, -8, -8, -4}}},
  /* 3 */
  {{{-15,-15,-15,-15,-15,-12,-10, -8,  0,  2,  2,  2,  4,  4,  4,  5,  8},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -3, -3, -3, -3, -1,  1,  6},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10,-10,-10,-10,-10, -8, -8, -8, -4}}},
  /* 4 */
  {{{-20,-20,-20,-20,-20,-18,-14, -8, -1,  1,  1,  1,  2,  3,  3,  4,  7},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -3, -3, -3, -3, -1,  1,  5},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10,-10,-10,-10,-10, -8, -8, -8, -4}}},
  /* 5 */
  {{{-24,-24,-24,-24,-20,-18,-14, -8, -1,  1,  1,  1,  2,  3,  3,  4,  7},
    {-32,-32,-32,-32,-28,-24,-22,-16,-12, -6, -4, -4, -4, -4, -2, -1,  2},
    {-34,-34,-34,-34,-30,-24,-24,-18,-14,-12,-12,-12,-12,-10,-10, -9, -5}}},
  /* 6 */
  {{{-24,-24,-24,-24,-20,-18,-14, -8, -1,  1,  1,  1,  2,  3,  3,  4,  7},
    {-32,-32,-32,-32,-28,-24,-24,-18,-14, -8, -6, -6, -6, -6, -4, -2,  1},
    {-34,-34,-34,-34,-30,-26,-24,-18,-17,-15,-15,-15,-15,-13,-13,-12, -8}}},
  /* 7 */
  {{{-24,-24,-24,-24,-20,-18,-14, -8, -1,  1,  1,  1,  2,  3,  3,  4, 7},
    {-32,-32,-32,-32,-28,-24,-24,-18,-14,-12,-10, -8, -8, -8, -6, -4,  0},
    {-34,-34,-34,-34,-30,-26,-26,-24,-22,-19,-19,-19,-19,-18,-17,-16,-12}}},
  /* 8 */
  {{{-24,-24,-24,-24,-22,-20,-15,-10, -8, -2,  0,  0,  0,  1,  2,  3,  7},
    {-36,-36,-36,-36,-30,-30,-30,-24,-18,-14,-12,-10,-10,-10, -8, -6, -2},
    {-36,-36,-36,-36,-34,-30,-28,-26,-24,-24,-24,-24,-24,-24,-24,-20,-16}}},
  /* 9 */
  {{{-28,-28,-28,-28,-28,-28,-28,-20,-14, -8, -4, -4, -4, -4, -4, -2,  2},
    {-36,-36,-36,-36,-34,-32,-32,-28,-20,-16,-16,-16,-16,-14,-12,-10, -7},
    {-40,-40,-40,-40,-40,-40,-40,-32,-30,-30,-30,-30,-30,-30,-30,-24,-20}}},
  /* 10 */
  {{{-30,-30,-30,-30,-30,-30,-30,-28,-20,-14,-14,-14,-14,-14,-14,-12,-10},
    {-40,-40,-40,-40,-40,-40,-40,-40,-35,-30,-30,-30,-30,-30,-30,-30,-20},
    {-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40}}},
};

static noise3 _psy_noisebias_long[11]={
    /*63     125     250     500      1k       2k      4k      8k     16k*/
  /* 0 */
  {{{-10,-10,-10,-10,-10,-10, -8,  2,  2,  2,  4,  4,  5,  5,  5,  8,  10},
    {-20,-20,-20,-20,-20,-20,-20,-14, -6,  0,  0,  0,  0,  0,  2,  4,  10},
    {-20,-20,-20,-20,-20,-20,-20,-14, -8, -6, -6, -6, -6, -4, -4, -4, -2}}},
  /* 1 */
  {{{-10,-10,-10,-10,-10,-10, -8, -4,  0,  2,  4,  4,  5,  5,  5,  8,  10},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10, -4, -2, -2, -2, -2,  0,  2,  8},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10, -8, -8, -8, -8, -6, -6, -6, -4}}},
  /* 2 */
  {{{-10,-10,-10,-10,-10,-10,-10, -8,  0,  2,  2,  2,  4,  4,  5,  6,  10},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10, -4, -2, -2, -2, -2,  0,  2,  6},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10,-10,-10,-10,-10, -8, -8, -8, -4}}},
  /* 3 */
  {{{-10,-10,-10,-10,-10,-10,-10, -8,  0,  2,  2,  2,  4,  4,  4,  5,  8},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10, -4, -3, -3, -3, -3, -1,  1,  6},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10,-10,-10,-10,-10, -8, -8, -8, -4}}},
  /* 4 */
  {{{-15,-15,-15,-15,-15,-15,-15,-10, -4,  1,  1,  1,  2,  3,  3,  4,  7},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10, -4, -3, -3, -3, -3, -1,  1,  5},
    {-20,-20,-20,-20,-20,-20,-20,-14,-10,-10,-10,-10,-10, -8, -8, -8, -4}}},
  /* 5 */
  {{{-15,-15,-15,-15,-15,-15,-15,-10, -4,  1,  1,  1,  2,  3,  3,  4,  7},
    {-22,-22,-22,-22,-22,-22,-22,-16,-12, -6, -4, -4, -4, -4, -2, -1,  2},
    {-24,-24,-24,-24,-24,-24,-24,-18,-14,-12,-12,-12,-12,-10,-10, -9, -5}}},
  /* 6 */
  {{{-15,-15,-15,-15,-15,-15,-15,-10, -4,  1,  1,  1,  2,  3,  3,  4,  7},
    {-24,-24,-24,-24,-24,-24,-24,-18,-14, -8, -6, -6, -6, -6, -4, -2,  1},
    {-26,-26,-26,-26,-26,-26,-26,-18,-16,-15,-15,-15,-15,-13,-13,-12, -8}}},
  /* 7 */
  {{{-15,-15,-15,-15,-15,-15,-15,-10, -4,  1,  1,  1,  2,  3,  3,  4,  7},
    {-24,-24,-24,-24,-24,-24,-24,-18,-14,-10, -8, -8, -8, -8, -6, -4,  0},
    {-26,-26,-26,-26,-26,-26,-26,-22,-20,-19,-19,-19,-19,-18,-17,-16,-12}}},
  /* 8 */
  {{{-15,-15,-15,-15,-15,-15,-15,-10, -4,  0,  0,  0,  0,  1,  2,  3,  7},
    {-26,-26,-26,-26,-26,-26,-26,-20,-16,-12,-10,-10,-10,-10, -8, -6, -2},
    {-28,-28,-28,-28,-28,-28,-28,-26,-24,-24,-24,-24,-24,-24,-24,-20,-16}}},
  /* 9 */
  {{{-22,-22,-22,-22,-22,-22,-22,-18,-14, -8, -4, -4, -4, -4, -4, -2,  2},
    {-26,-26,-26,-26,-26,-26,-26,-22,-18,-16,-16,-16,-16,-14,-12,-10, -7},
    {-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-24,-20}}},
  /* 10 */
  {{{-24,-24,-24,-24,-24,-24,-24,-24,-24,-18,-14,-14,-14,-14,-14,-12,-10},
    {-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-30,-20},
    {-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40}}},
};

static noise3 _psy_noisebias_impulse[11]={
  /*  63     125     250     500      1k      2k      4k      8k     16k*/
  /* 0 */
  {{{-10,-10,-10,-10,-10, -4,  0,  0,  4,  4,  8,  8,  8, 10, 12, 14, 20},
    {-30,-30,-30,-30,-26,-22,-20,-14, -6, -2,  0,  0,  0,  0,  2,  4,  10},
    {-30,-30,-30,-30,-30,-24,-20,-14,-10, -6, -8, -8, -6, -6, -6, -4, -2}}},
  /* 1 */
  {{{-12,-12,-12,-12,-12, -8, -6, -4,  0,  4,  4,  4,  4, 10, 12, 14, 20},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -4, -4, -2, -2, -2, -2,  2},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -8,-10,-10, -8, -8, -8, -6, -4}}},
  /* 2 */
  {{{-14,-14,-14,-14,-14,-10, -8, -6, -2,  2,  2,  2,  2,  8, 10, 10, 16},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -6, -6, -4, -4, -4, -2,  0},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10,-10,-10,-10,-10,-10,-10, -8, -4}}},
  /* 3 */
  {{{-14,-14,-14,-14,-14,-10, -8, -6, -2,  2,  2,  2,  2,  6,  8,  8, 14},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -6, -6, -4, -4, -4, -2,  0},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10,-10,-10,-10,-10,-10,-10, -8, -4}}},
  /* 4 */
  {{{-16,-16,-16,-16,-16,-12,-10, -6, -2,  0,  0,  0,  0,  4,  6,  6, 12},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -6, -6, -4, -4, -4, -2,  0},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10,-10,-10,-10,-10,-10,-10, -8, -4}}},
  /* 5 */
  {{{-20,-20,-20,-20,-20,-18,-14,-10, -4,  0,  0,  0,  0,  4,  4,  6, 11},
    {-32,-32,-32,-32,-28,-24,-22,-16,-10, -6, -8, -8, -6, -6, -6, -4, -2},
    {-34,-34,-34,-34,-30,-26,-24,-18,-14,-12,-12,-12,-12,-12,-10, -9, -5}}},
  /* 6 */
  {{{-20,-20,-20,-20,-20,-18,-14,-10, -4,  0,  0,  0,  0,  4,  4,  6, 11},
    {-34,-34,-34,-34,-30,-30,-24,-20,-12,-12,-14,-14,-10, -9, -8, -6, -4},
    {-34,-34,-34,-34,-34,-30,-26,-20,-16,-15,-15,-15,-15,-15,-13,-12, -8}}},
  /* 7 */
  {{{-22,-22,-22,-22,-22,-20,-14,-10, -6,  0,  0,  0,  0,  4,  4,  6, 11},
    {-34,-34,-34,-34,-30,-30,-24,-20,-14,-14,-16,-16,-14,-12,-10,-10,-10},
    {-34,-34,-34,-34,-32,-32,-30,-24,-20,-19,-19,-19,-19,-19,-17,-16,-12}}},
  /* 8 */
  {{{-24,-24,-24,-24,-24,-22,-14,-10, -6, -1, -1, -1, -1,  3,  3,  5, 10},
    {-34,-34,-34,-34,-30,-30,-30,-24,-20,-20,-20,-20,-20,-18,-16,-16,-14},
    {-36,-36,-36,-36,-36,-34,-28,-24,-24,-24,-24,-24,-24,-24,-24,-20,-16}}},
  /* 9 */
  {{{-28,-28,-28,-28,-28,-28,-28,-20,-14, -8, -4, -4, -4, -4, -4, -2,  2},
    {-36,-36,-36,-36,-34,-32,-32,-30,-26,-26,-26,-26,-26,-22,-20,-20,-18},
    {-40,-40,-40,-40,-40,-40,-40,-32,-30,-30,-30,-30,-30,-30,-30,-24,-20}}},
  /* 10 */
  {{{-30,-30,-30,-30,-30,-26,-24,-24,-24,-20,-16,-16,-16,-16,-16,-14,-12},
    {-40,-40,-40,-40,-40,-40,-40,-40,-35,-30,-30,-30,-30,-30,-30,-30,-26},
    {-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40}}},
};


static noise3 _psy_noisebias_padding[11]={
  /*  63     125     250     500      1k       2k      4k      8k     16k*/
  /* 0 */
  {{{-10,-10,-10,-10,-10, -4,  0,  0,  4,  8,  8,  8,  8, 10, 12, 14, 20},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -2,  2,  3,  6,  6,  8, 10},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4, -4, -4, -4, -4, -2,  0,  2}}},
  /* 1 */
  {{{-12,-12,-12,-12,-12, -8, -6, -4,  0,  4,  4,  4,  4, 10, 12, 14, 20},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4,  0,  0,  0,  2,  2,  4,  8},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -6, -6, -6, -6, -4, -2,  0}}},
  /* 2 */
  {{{-14,-14,-14,-14,-14,-10, -8, -6, -2,  2,  2,  2,  2,  8, 10, 10, 16},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -4,  0,  0,  0,  2,  2,  4,  8},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -8, -8, -8, -8, -8, -6, -4, -2}}},
  /* 3 */
  {{{-14,-14,-14,-14,-14,-10, -8, -6, -2,  2,  2,  2,  2,  6,  8,  8, 14},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -1, -1, -1,  0,  0,  2,  6},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -8, -8, -8, -8, -8, -6, -4, -2}}},
  /* 4 */
  {{{-16,-16,-16,-16,-16,-12,-10, -6, -2,  0,  0,  0,  0,  4,  6,  6, 12},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -6, -1, -1, -1, -1,  0,  2,  6},
    {-30,-30,-30,-30,-26,-22,-20,-14,-10, -8, -8, -8, -8, -8, -6, -4, -2}}},
  /* 5 */
  {{{-20,-20,-20,-20,-20,-18,-14,-10, -4,  0,  0,  0,  0,  4,  6,  6, 12},
    {-32,-32,-32,-32,-28,-24,-22,-16,-12, -6, -3, -3, -3, -3, -2,  0,  4},
    {-34,-34,-34,-34,-30,-26,-24,-18,-14,-10,-10,-10,-10,-10, -8, -5, -3}}},
  /* 6 */
  {{{-20,-20,-20,-20,-20,-18,-14,-10, -4,  0,  0,  0,  0,  4,  6,  6, 12},
    {-34,-34,-34,-34,-30,-30,-24,-20,-14, -8, -4, -4, -4, -4, -3, -1,  4},
    {-34,-34,-34,-34,-34,-30,-26,-20,-16,-13,-13,-13,-13,-13,-11, -8, -6}}},
  /* 7 */
  {{{-20,-20,-20,-20,-20,-18,-14,-10, -4,  0,  0,  0,  0,  4,  6,  6, 12},
    {-34,-34,-34,-34,-30,-30,-30,-24,-16,-10, -8, -6, -6, -6, -5, -3,  1},
    {-34,-34,-34,-34,-32,-32,-28,-22,-18,-16,-16,-16,-16,-16,-14,-12,-10}}},
  /* 8 */
  {{{-22,-22,-22,-22,-22,-20,-14,-10, -4,  0,  0,  0,  0,  3,  5,  5, 11},
    {-34,-34,-34,-34,-30,-30,-30,-24,-16,-12,-10, -8, -8, -8, -7, -5, -2},
    {-36,-36,-36,-36,-36,-34,-28,-22,-20,-20,-20,-20,-20,-20,-20,-16,-14}}},
  /* 9 */
  {{{-28,-28,-28,-28,-28,-28,-28,-20,-14, -8, -2, -2, -2, -2,  0,  2,  6},
    {-36,-36,-36,-36,-34,-32,-32,-24,-16,-12,-12,-12,-12,-12,-10, -8, -5},
    {-40,-40,-40,-40,-40,-40,-40,-32,-26,-24,-24,-24,-24,-24,-24,-20,-18}}},
  /* 10 */
  {{{-30,-30,-30,-30,-30,-26,-24,-24,-24,-20,-12,-12,-12,-12,-12,-10, -8},
    {-40,-40,-40,-40,-40,-40,-40,-40,-35,-30,-25,-25,-25,-25,-25,-25,-15},
    {-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40,-40}}},
};

static noiseguard _psy_noiseguards_44[4]={
  {3,3,15},
  {3,3,15},
  {10,10,100},
  {10,10,100},
};

static int _psy_tone_suppress[11]={
  -20,-20,-20,-20,-24,-30,-40,-40,-45,-45,-45,
};

static int _psy_tone_0dB[11]={
  90,95,95,95,95,105,105,105,105,105,105,
};

static int _psy_noise_suppress[11]={
 -20,-24,-24,-24,-24,-30,-40,-40,-45,-45,-45,
};

static vorbis_info_psy _psy_info_template={
  -1,
  -140.,-140.,
  /* tonemask att boost/decay,suppr,curves */
  {0.f,0.f,0.f},     0.,0.,    -40.f, {0.},  
  
  /*noisemaskp,supp, low/high window, low/hi guard, minimum */
  1,          -0.f,         .5f, .5f,         0,0,0,
  {{-1},{-1},{-1}},{-1},105.f,

  0,0,-1,-1,0.,
};

/* ath ****************/

static int _psy_ath_floater[11]={
  -100,-100,-100,-100,-100,-105,-105,-105,-105,-110,-120,
};

static int _psy_ath_abs[11]={
  -130,-130,-130,-140,-140,-140,-140,-140,-140,-140,-150,
};

/* stereo setup.  These don't map directly to quality level, there's
   an additional indirection as several of the below may be used in a
   single bitmanaged stream

****************/

/* various stereo possibilities */

/* stereo mode by base quality level */
static adj_stereo _psy_stereo_modes_44_low[2]={
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         0  */
  {{  4,  4,  4,  4,  4,  4,  4,  3,  2,  2,  1,  0,  0,  0,  0},
   {  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  5,  4,  3},
   {  1,  2,  3,  4,  4,  4,  4,  4,  4,  5,  6,  7,  8,  8,  8},
   { 12,12.5, 13,13.5, 14,14.5, 15, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         1  */
  {{  4,  4,  4,  4,  4,  4,  4,  3,  2,  2,  1,  0,  0,  0,  0},
   {  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  5,  4,  3},
   {  1,  2,  3,  4,  5,  5,  6,  6,  6,  6,  6,  7,  8,  8,  8},
   { 12,12.5, 13,13.5, 14,14.5, 15, 99, 99, 99, 99, 99, 99, 99, 99}},
};

static adj_stereo _psy_stereo_modes_44[11]={
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         0  */
  {{  4,  4,  4,  4,  4,  4,  4,  3,  2,  2,  1,  0,  0,  0,  0},
   {  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  5,  4,  3},
   {  1,  2,  3,  4,  5,  5,  6,  6,  6,  6,  6,  7,  8,  8,  8},
   { 12,12.5, 13,13.5, 14,14.5, 15, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         1  */
  {{  3,  3,  3,  3,  3,  3,  3,  3,  2,  1,  0,  0,  0,  0,  0},
   {  8,  8,  8,  8,  6,  6,  5,  5,  5,  5,  5,  5,  5,  4,  3},
   {  1,  2,  3,  4,  4,  5,  6,  6,  6,  6,  6,  8,  8,  8,  8},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         2  */
  {{  3,  3,  3,  3,  3,  3,  2,  2,  2,  1,  0,  0,  0,  0,  0},
   {  8,  8,  8,  6,  5,  5,  5,  5,  5,  5,  5,  4,  3,  2,  1},
   {  3,  4,  4,  4,  5,  6,  6,  6,  6,  6,  6,  8,  8,  8,  8},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         3  */
  {{  2,  2,  2,  2,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0},
   {  5,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  3,  2,  1},
   {  4,  4,  5,  6,  6,  6,  6,  6,  8,  8, 10, 10, 10, 10, 10},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         4  */
  {{  2,  2,  2,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  3,  3,  2,  1,  0},
   {  6,  6,  6,  8,  8,  8,  8,  8,  8,  8, 10, 10, 10, 10, 10},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         5  */
  {{  2,  2,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  3,  3,  3,  3,  3,  2,  2,  2,  2,  2,  2,  0,  0,  0,  0},
   {  6,  6,  8,  8,  8,  8, 10, 10, 10, 10, 10, 10, 10, 10, 10},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         6  */
  {{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  3,  3,  3,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  8,  8,  8,  8, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         7  */
  {{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  3,  3,  3,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  8,  8,  8,  8, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         8  */
  {{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  8, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14         9  */
  {{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
  /*  0   1   2   3   4   5   6   7   8   9  10  11  12  13  14        10  */
  {{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4},
   { 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99}},
};

/* tone master attenuation by base quality mode and bitrate tweak */

static att3 _psy_tone_masteratt_44_low[2]={
  {{ 34,  20,   8},  0, 1.25}, /* 0 */
  {{ 34,  20,   8}, -2, 1.25}, /* 1 */
};
static att3 _psy_tone_masteratt_44[11]={
  {{ 30,  20,   8}, -2, 1.25}, /* 0 */
  {{ 25,  14,   4},  0,    0}, /* 1 */
  {{ 20,  10,  -2},  0,    0}, /* 2 */
  {{ 20,   9,  -4},  0,    0}, /* 3 */
  {{ 20,   9,  -4},  0,    0}, /* 4 */
  {{ 20,   6,  -6},  0,    0}, /* 5 */
  {{ 20,   3, -10},  0,    0}, /* 6 */
  {{ 18,   1, -14},  0,    0}, /* 7 */
  {{ 18,   0, -16},  0,    0}, /* 8 */
  {{ 18,  -2, -16},  0,    0}, /* 9 */
  {{ 12,  -2, -20},  0,    0}, /* 10 */
};

/* lowpass by mode **************/
static double _psy_lowpass_44_low[2]={
  15.1,15.1,
};
static double _psy_lowpass_44[11]={
  15.1,15.8,16.5,17.9,20.5,48.,999.,999.,999.,999.,999.
};

/* noise normalization **********/

static int _noise_start_short_44[10]={
  16,16,16,16,32,32,9999,9999,9999,9999
};
static int _noise_start_long_44[10]={
  128,128,128,256,512,512,9999,9999,9999,9999
};
static int _noise_part_short_44[10]={
  8,8,8,8,8,8,8,8,8,8
};
static int _noise_part_long_44[10]={
  32,32,32,32,32,32,32,32,32,32
};
static double _noise_thresh_44[10]={
  .2,.2,.3,.4,.5,.5,9999.,9999.,9999.,9999.,
};
static double _noise_thresh_44_2[2]={
  .5,.5,
};


static int _noise_start_short_44_low[2]={
  32,32
};
static int _noise_start_long_44_low[2]={
  256,256
};

static double rate_mapping_44_stereo_low[2]={
  22500.,32000.
};

static double rate_mapping_44_stereo[11]={
  32000.,40000.,48000.,56000.,64000.,
  80000.,96000.,112000.,128000.,160000.,250001.
};
static double quality_mapping_44_stereo_low[2]={
  -.1,.0,
};
static double quality_mapping_44[11]={
  .0,.1,.2,.3,.4,.5,.6,.7,.8,.9,1.0
};

static int blocksize_short_44_low[1]={
  512
};
static int blocksize_long_44_low[1]={
  4096
};

static int blocksize_short_44[10]={
  256,256,256,256,256,256,256,256,256,256
};
static int blocksize_long_44[10]={
  2048,2048,2048,2048,2048,2048,2048,2048,2048,2048
};
static double _psy_compand_short_mapping[11]={
  1., 1., 1.3, 1.6, 2., 2., 2., 2., 2., 2., 2.
};
static double _psy_compand_long_mapping[11]={
  4., 4., 4.3, 4.6, 5., 5., 5., 5., 5., 5., 5.
};
static double _global_mapping_44[11]={
  1., 1., 1.5, 2., 2., 2.5, 2.7, 3.0, 3.5, 4., 4.
};
static double _global_mapping_44_low[2]={
  0., 1.,
};
static int _floor_short_mapping_44_low[1]={
  1,
};
static int _floor_long_mapping_44_low[1]={
  8
};
static int _floor_short_mapping_44[10]={
  0,0,2,2,4,5,5,5,5,5
};
static int _floor_long_mapping_44[10]={
  7,7,7,7,7,7,7,7,7,7
};

ve_setup_data_template ve_setup_44_stereo={
  10,
  rate_mapping_44_stereo,
  quality_mapping_44,
  2,
  40000,
  50000,
  
  blocksize_short_44,
  blocksize_long_44,

  _psy_tone_masteratt_44,
  _psy_tone_0dB,
  _psy_tone_suppress,

  _vp_tonemask_adj_otherblock,
  _vp_tonemask_adj_longblock,
  _vp_tonemask_adj_otherblock,

  _psy_noiseguards_44,
  _psy_noisebias_impulse,
  _psy_noisebias_padding,
  _psy_noisebias_trans,
  _psy_noisebias_long,
  _psy_noise_suppress,
  
  _psy_compand_44,
  _psy_compand_short_mapping,
  _psy_compand_long_mapping,

  {_noise_start_short_44,_noise_start_long_44},
  {_noise_part_short_44,_noise_part_long_44},
  _noise_thresh_44,

  _psy_ath_floater,
  _psy_ath_abs,
  
  _psy_lowpass_44,

  _psy_global_44,
  _global_mapping_44,
  _psy_stereo_modes_44,

  _floor_books,
  _floor,
  _floor_short_mapping_44,
  _floor_long_mapping_44,

  _mapres_template_44_stereo
};

ve_setup_data_template ve_setup_44_stereo_low={
  1,
  rate_mapping_44_stereo_low,
  quality_mapping_44_stereo_low,
  2,
  40000,
  50000,
  
  blocksize_short_44_low,
  blocksize_long_44_low,

  _psy_tone_masteratt_44_low,
  _psy_tone_0dB,
  _psy_tone_suppress,

  _vp_tonemask_adj_otherblock,
  _vp_tonemask_adj_longblock,
  _vp_tonemask_adj_otherblock,

  _psy_noiseguards_44,
  _psy_noisebias_trans_low,
  _psy_noisebias_trans_low,
  _psy_noisebias_trans_low,
  _psy_noisebias_long_low,
  _psy_noise_suppress,
  
  _psy_compand_44,
  _psy_compand_short_mapping,
  _psy_compand_long_mapping,

  {_noise_start_short_44_low,_noise_start_long_44_low},
  {_noise_part_short_44,_noise_part_long_44},
  _noise_thresh_44,

  _psy_ath_floater,
  _psy_ath_abs,
  
  _psy_lowpass_44_low,

  _psy_global_44,
  _global_mapping_44_low,
  _psy_stereo_modes_44_low,

  _floor_books,
  _floor,
  _floor_short_mapping_44_low,
  _floor_long_mapping_44_low,

  _mapres_template_44_stereo
};

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: toplevel settings for 44.1/48kHz uncoupled modes

 ********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: toplevel residue templates for 32/44.1/48kHz uncoupled

 ********************************************************************/

/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: static codebooks autogenerated by huff/huffbuld

 ********************************************************************/

static long _vq_quantlist__16u0__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u0__p1_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 8, 5, 8, 8, 8,10,10, 8,
	10,11, 5, 8, 8, 8,10,10, 8,10,10, 4, 9, 9, 9,12,
	11, 8,11,11, 8,12,11,10,12,14,10,13,13, 7,11,11,
	10,14,12,11,14,14, 4, 9, 9, 8,11,11, 9,11,12, 7,
	11,11,10,13,14,10,12,14, 8,11,12,10,14,14,10,13,
	12,
};

static float _vq_quantthresh__16u0__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16u0__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u0__p1_0 = {
	_vq_quantthresh__16u0__p1_0,
	_vq_quantmap__16u0__p1_0,
	3,
	3
};

static static_codebook _16u0__p1_0 = {
	4, 81,
	_vq_lengthlist__16u0__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16u0__p1_0,
	NULL,
	&_vq_auxt__16u0__p1_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u0__p2_0[] = {
	 2, 4, 4, 5, 6, 6, 5, 6, 6, 5, 7, 7, 7, 8, 9, 7,
	 8, 9, 5, 7, 7, 7, 9, 8, 7, 9, 7, 4, 7, 7, 7, 9,
	 9, 7, 8, 8, 6, 9, 8, 7, 8,11, 9,11,10, 6, 8, 9,
	 8,11, 8, 9,10,11, 4, 7, 7, 7, 8, 8, 7, 9, 9, 6,
	 9, 8, 9,11,10, 8, 8,11, 6, 8, 9, 9,10,11, 8,11,
	 8,
};

static float _vq_quantthresh__16u0__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16u0__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u0__p2_0 = {
	_vq_quantthresh__16u0__p2_0,
	_vq_quantmap__16u0__p2_0,
	3,
	3
};

static static_codebook _16u0__p2_0 = {
	4, 81,
	_vq_lengthlist__16u0__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16u0__p2_0,
	NULL,
	&_vq_auxt__16u0__p2_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u0__p3_0[] = {
	 1, 5, 5, 7, 7, 6, 7, 7, 8, 8, 6, 7, 8, 8, 8, 8,
	 9, 9,11,11, 8, 9, 9,11,11, 6, 9, 8,10,10, 8,10,
	10,11,11, 8,10,10,11,11,10,11,10,13,12, 9,11,10,
	13,13, 6, 8, 9,10,10, 8,10,10,11,11, 8,10,10,11,
	11, 9,10,11,13,12,10,10,11,12,12, 8,11,11,14,13,
	10,12,11,15,13, 9,12,11,15,14,12,14,13,16,14,12,
	13,13,17,14, 8,11,11,13,14, 9,11,12,14,15,10,11,
	12,13,15,11,13,13,14,16,12,13,14,14,16, 5, 9, 9,
	11,11, 9,11,11,12,12, 8,11,11,12,12,11,12,12,15,
	14,10,12,12,15,15, 8,11,11,13,12,10,12,12,13,13,
	10,12,12,14,13,12,12,13,14,15,11,13,13,17,16, 7,
	11,11,13,13,10,12,12,14,13,10,12,12,13,14,12,13,
	12,15,14,11,13,13,15,14, 9,12,12,16,15,11,13,13,
	17,16,10,13,13,16,16,13,14,15,15,16,13,15,14,19,
	17, 9,12,12,14,16,11,13,13,15,16,10,13,13,17,16,
	13,14,13,17,15,12,15,15,16,17, 5, 9, 9,11,11, 8,
	11,11,13,12, 9,11,11,12,12,10,12,12,14,15,11,12,
	12,14,14, 7,11,10,13,12,10,12,12,14,13,10,11,12,
	13,13,11,13,13,15,16,12,12,13,15,15, 7,11,11,13,
	13,10,13,13,14,14,10,12,12,13,13,11,13,13,16,15,
	12,13,13,15,14, 9,12,12,15,15,10,13,13,17,16,11,
	12,13,15,15,12,15,14,18,18,13,14,14,16,17, 9,12,
	12,15,16,10,13,13,15,16,11,13,13,15,16,13,15,15,
	17,17,13,15,14,16,15, 7,11,11,15,16,10,13,12,16,
	17,10,12,13,15,17,15,16,16,18,17,13,15,15,17,18,
	 8,12,12,16,16,11,13,14,17,18,11,13,13,18,16,15,
	17,16,17,19,14,15,15,17,16, 8,12,12,16,15,11,14,
	13,18,17,11,13,14,18,17,15,16,16,18,17,13,16,16,
	18,18,11,15,14,18,17,13,14,15,18, 0,12,15,15, 0,
	17,17,16,17,17,18,14,16,18,18, 0,11,14,14,17, 0,
	12,15,14,17,19,12,15,14,18, 0,15,18,16, 0,17,14,
	18,16,18, 0, 7,11,11,16,15,10,12,12,18,16,10,13,
	13,16,15,13,15,14,17,17,14,16,16,19,18, 8,12,12,
	16,16,11,13,13,18,16,11,13,14,17,16,14,15,15,19,
	18,15,16,16, 0,19, 8,12,12,16,17,11,13,13,17,17,
	11,14,13,17,17,13,15,15,17,19,15,17,17,19, 0,11,
	14,15,19,17,12,15,16,18,18,12,14,15,19,17,14,16,
	17, 0,18,16,16,19,17, 0,11,14,14,18,19,12,15,14,
	17,17,13,16,14,17,16,14,17,16,18,18,15,18,15, 0,
	18,
};

static float _vq_quantthresh__16u0__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u0__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u0__p3_0 = {
	_vq_quantthresh__16u0__p3_0,
	_vq_quantmap__16u0__p3_0,
	5,
	5
};

static static_codebook _16u0__p3_0 = {
	4, 625,
	_vq_lengthlist__16u0__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u0__p3_0,
	NULL,
	&_vq_auxt__16u0__p3_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u0__p4_0[] = {
	 3, 5, 5, 8, 8, 6, 6, 6, 9, 9, 6, 6, 6, 9, 9, 9,
	10, 9,11,11, 9, 9, 9,11,11, 6, 7, 7,10,10, 7, 7,
	 8,10,10, 7, 7, 8,10,10,10,10,10,11,12, 9,10,10,
	11,12, 6, 7, 7,10,10, 7, 8, 7,10,10, 7, 8, 7,10,
	10,10,11,10,12,11,10,10,10,13,10, 9,10,10,12,12,
	10,11,10,14,12, 9,11,11,13,13,11,12,13,13,13,11,
	12,12,15,13, 9,10,10,12,13, 9,11,10,12,13,10,10,
	11,12,13,11,12,12,12,13,11,12,12,13,13, 5, 7, 7,
	10,10, 7, 8, 8,10,10, 7, 8, 8,10,10,10,11,10,12,
	13,10,10,11,12,12, 6, 8, 8,11,10, 7, 8, 9,10,12,
	 8, 9, 9,11,11,11,10,11,11,12,10,11,11,13,12, 7,
	 8, 8,10,11, 8, 9, 8,11,10, 8, 9, 9,11,11,10,12,
	10,13,11,10,11,11,13,13,10,11,10,14,13,10,10,11,
	13,13,10,12,11,14,13,12,11,13,12,13,13,12,13,14,
	14,10,11,11,13,13,10,11,10,12,13,10,12,12,12,14,
	12,12,12,14,12,12,13,12,17,15, 5, 7, 7,10,10, 7,
	 8, 8,10,10, 7, 8, 8,11,10,10,10,11,12,12,10,11,
	11,12,13, 6, 8, 8,11,10, 8, 9, 9,11,11, 7, 8, 9,
	10,11,11,11,11,12,12,10,10,11,12,13, 6, 8, 8,10,
	11, 8, 9, 9,11,11, 7, 9, 7,11,10,10,12,12,13,13,
	11,11,10,13,11, 9,11,10,14,13,11,11,11,15,13,10,
	10,11,13,13,12,13,13,14,14,12,11,12,12,13,10,11,
	11,12,13,10,11,12,13,13,10,11,10,13,12,12,12,13,
	14, 0,12,13,11,13,11, 8,10,10,13,13,10,11,11,14,
	13,10,11,11,13,12,13,14,14,14,15,12,12,12,15,14,
	 9,11,10,13,12,10,10,11,13,14,11,11,11,15,12,13,
	12,14,15,16,13,13,13,14,13, 9,11,11,12,12,10,12,
	11,13,13,10,11,11,13,14,13,13,13,15,15,13,13,14,
	17,15,11,12,12,14,14,10,11,12,13,15,12,13,13, 0,
	15,13,11,14,12,16,14,16,14, 0,15,11,12,12,14,16,
	11,13,12,16,15,12,13,13,14,15,12,14,12,15,13,15,
	14,14,16,16, 8,10,10,13,13,10,11,10,13,14,10,11,
	11,13,13,13,13,12,14,14,14,13,13,16,17, 9,10,10,
	12,14,10,12,11,14,13,10,11,12,13,14,12,12,12,15,
	15,13,13,13,14,14, 9,10,10,13,13,10,11,12,12,14,
	10,11,10,13,13,13,13,13,14,16,13,13,13,14,14,11,
	12,13,15,13,12,14,13,14,16,12,12,13,13,14,13,14,
	14,17,15,13,12,17,13,16,11,12,13,14,15,12,13,14,
	14,17,11,12,11,14,14,13,16,14,16, 0,14,15,11,15,
	11,
};

static float _vq_quantthresh__16u0__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u0__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u0__p4_0 = {
	_vq_quantthresh__16u0__p4_0,
	_vq_quantmap__16u0__p4_0,
	5,
	5
};

static static_codebook _16u0__p4_0 = {
	4, 625,
	_vq_lengthlist__16u0__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u0__p4_0,
	NULL,
	&_vq_auxt__16u0__p4_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16u0__p5_0[] = {
	 1, 4, 4, 7, 7, 7, 7, 9, 9, 4, 6, 6, 8, 8, 8, 8,
	 9, 9, 4, 6, 6, 8, 8, 8, 8, 9, 9, 7, 8, 8, 9, 9,
	 9, 9,11,10, 7, 8, 8, 9, 9, 9, 9,10,11, 7, 8, 8,
	 9, 9,10,10,11,11, 7, 8, 8, 9, 9,10,10,11,11, 9,
	 9, 9,10,10,11,11,12,12, 9, 9, 9,10,10,11,11,12,
	12,
};

static float _vq_quantthresh__16u0__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16u0__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16u0__p5_0 = {
	_vq_quantthresh__16u0__p5_0,
	_vq_quantmap__16u0__p5_0,
	9,
	9
};

static static_codebook _16u0__p5_0 = {
	2, 81,
	_vq_lengthlist__16u0__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16u0__p5_0,
	NULL,
	&_vq_auxt__16u0__p5_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16u0__p6_0[] = {
	 1, 4, 4, 7, 7,10,10,12,12,13,13,18,17, 3, 6, 6,
	 9, 9,11,11,13,13,14,14,18,17, 3, 6, 6, 9, 9,11,
	11,13,13,14,14,17,18, 7, 9, 9,11,11,13,13,14,14,
	15,15, 0, 0, 7, 9, 9,11,11,13,13,14,14,15,16,19,
	18,10,11,11,13,13,14,14,16,15,17,18, 0, 0,10,11,
	11,13,13,14,14,15,15,16,18, 0, 0,11,13,13,14,14,
	15,15,17,17, 0,19, 0, 0,11,13,13,14,14,14,15,16,
	18, 0,19, 0, 0,13,14,14,15,15,18,17,18,18, 0,19,
	 0, 0,13,14,14,15,16,16,16,18,18,19, 0, 0, 0,16,
	17,17, 0,17,19,19, 0,19, 0, 0, 0, 0,16,19,16,17,
	18, 0,19, 0, 0, 0, 0, 0, 0,
};

static float _vq_quantthresh__16u0__p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__16u0__p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16u0__p6_0 = {
	_vq_quantthresh__16u0__p6_0,
	_vq_quantmap__16u0__p6_0,
	13,
	13
};

static static_codebook _16u0__p6_0 = {
	2, 169,
	_vq_lengthlist__16u0__p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__16u0__p6_0,
	NULL,
	&_vq_auxt__16u0__p6_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u0__p6_1[] = {
	 1, 4, 5, 6, 6, 4, 6, 6, 6, 6, 4, 6, 6, 6, 6, 6,
	 6, 6, 7, 7, 6, 6, 6, 7, 7,
};

static float _vq_quantthresh__16u0__p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u0__p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u0__p6_1 = {
	_vq_quantthresh__16u0__p6_1,
	_vq_quantmap__16u0__p6_1,
	5,
	5
};

static static_codebook _16u0__p6_1 = {
	2, 25,
	_vq_lengthlist__16u0__p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u0__p6_1,
	NULL,
	&_vq_auxt__16u0__p6_1,
	NULL,
	0
};

static long _vq_quantlist__16u0__p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u0__p7_0[] = {
	 1, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	 7,
};

static float _vq_quantthresh__16u0__p7_0[] = {
	-157.5, 157.5, 
};

static long _vq_quantmap__16u0__p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u0__p7_0 = {
	_vq_quantthresh__16u0__p7_0,
	_vq_quantmap__16u0__p7_0,
	3,
	3
};

static static_codebook _16u0__p7_0 = {
	4, 81,
	_vq_lengthlist__16u0__p7_0,
	1, -518803456, 1628680192, 2, 0,
	_vq_quantlist__16u0__p7_0,
	NULL,
	&_vq_auxt__16u0__p7_0,
	NULL,
	0
};

static long _vq_quantlist__16u0__p7_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16u0__p7_1[] = {
	 1, 5, 5, 6, 5, 9,10,11,11,10,10,10,10,10,10, 5,
	 8, 8, 8,10,10,10,10,10,10,10,10,10,10,10, 5, 8,
	 9, 9, 9,10,10,10,10,10,10,10,10,10,10, 5,10, 8,
	10,10,10,10,10,10,10,10,10,10,10,10, 4, 8, 9,10,
	10,10,10,10,10,10,10,10,10,10,10, 9,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10, 9,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__16u0__p7_1[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__16u0__p7_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16u0__p7_1 = {
	_vq_quantthresh__16u0__p7_1,
	_vq_quantmap__16u0__p7_1,
	15,
	15
};

static static_codebook _16u0__p7_1 = {
	2, 225,
	_vq_lengthlist__16u0__p7_1,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__16u0__p7_1,
	NULL,
	&_vq_auxt__16u0__p7_1,
	NULL,
	0
};

static long _vq_quantlist__16u0__p7_2[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__16u0__p7_2[] = {
	 1, 6, 6, 7, 8, 7, 7,10, 9,10, 9,11,10, 9,11,10,
	 9, 9, 9, 9,10, 6, 8, 7, 9, 9, 8, 8,10,10, 9,11,
	11,12,12,10, 9,11, 9,12,10, 9, 6, 9, 8, 9,12, 8,
	 8,11, 9,11,11,12,11,12,12,10,11,11,10,10,11, 7,
	10, 9, 9, 9, 9, 9,10, 9,10, 9,10,10,12,10,10,10,
	11,12,10,10, 7, 9, 9, 9,10, 9, 9,10,10, 9, 9, 9,
	11,11,10,10,10,10, 9, 9,12, 7, 9,10, 9,11, 9,10,
	 9,10,11,11,11,10,11,12, 9,12,11,10,10,10, 7, 9,
	 9, 9, 9,10,12,10, 9,11,12,10,11,12,12,11, 9,10,
	11,10,11, 7, 9,10,10,11,10, 9,10,11,11,11,10,12,
	12,12,11,11,10,11,11,12, 8, 9,10,12,11,10,10,12,
	12,12,12,12,10,11,11, 9,11,10,12,11,11, 8, 9,10,
	10,11,12,11,11,10,10,10,12,12,12, 9,10,12,12,12,
	12,12, 8,10,11,10,10,12, 9,11,12,12,11,12,12,12,
	12,10,12,10,10,10,10, 8,12,11,11,11,10,10,11,12,
	12,12,12,11,12,12,12,11,11,11,12,10, 9,10,10,12,
	10,12,10,12,12,10,10,10,11,12,12,12,11,12,12,12,
	11,10,11,12,12,12,11,12,12,11,12,12,11,12,12,12,
	12,11,12,12,10,10,10,10,11,11,12,11,12,12,12,12,
	12,12,12,11,12,11,10,11,11,12,11,11, 9,10,10,10,
	12,10,10,11, 9,11,12,11,12,11,12,12,10,11,10,12,
	 9, 9, 9,12,11,10,11,10,12,10,12,10,12,12,12,11,
	11,11,11,11,10, 9,10,10,11,10,11,11,12,11,10,11,
	12,12,12,11,11, 9,12,10,12, 9,10,12,10,10,11,10,
	11,11,12,11,10,11,10,11,11,11,11,12,11,11,10, 9,
	10,10,10, 9,11,11,10, 9,12,10,11,12,11,12,12,11,
	12,11,12,11,10,11,10,12,11,12,11,12,11,12,10,11,
	10,10,12,11,10,11,11,11,10,
};

static float _vq_quantthresh__16u0__p7_2[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__16u0__p7_2[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__16u0__p7_2 = {
	_vq_quantthresh__16u0__p7_2,
	_vq_quantmap__16u0__p7_2,
	21,
	21
};

static static_codebook _16u0__p7_2 = {
	2, 441,
	_vq_lengthlist__16u0__p7_2,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__16u0__p7_2,
	NULL,
	&_vq_auxt__16u0__p7_2,
	NULL,
	0
};

static long _huff_lengthlist__16u0__single[] = {
	 3, 5, 8, 7,14, 8, 9,19, 5, 2, 5, 5, 9, 6, 9,19,
	 8, 4, 5, 7, 8, 9,13,19, 7, 4, 6, 5, 9, 6, 9,19,
	12, 8, 7, 9,10,11,13,19, 8, 5, 8, 6, 9, 6, 7,19,
	 8, 8,10, 7, 7, 4, 5,19,12,17,19,15,18,13,11,18,
};

static static_codebook _huff_book__16u0__single = {
	2, 64,
	_huff_lengthlist__16u0__single,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__16u1__long[] = {
	 3, 6,10, 8,12, 8,14, 8,14,19, 5, 3, 5, 5, 7, 6,
	11, 7,16,19, 7, 5, 6, 7, 7, 9,11,12,19,19, 6, 4,
	 7, 5, 7, 6,10, 7,18,18, 8, 6, 7, 7, 7, 7, 8, 9,
	18,18, 7, 5, 8, 5, 7, 5, 8, 6,18,18,12, 9,10, 9,
	 9, 9, 8, 9,18,18, 8, 7,10, 6, 8, 5, 6, 4,11,18,
	11,15,16,12,11, 8, 8, 6, 9,18,14,18,18,18,16,16,
	16,13,16,18,
};

static static_codebook _huff_book__16u1__long = {
	2, 100,
	_huff_lengthlist__16u1__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__16u1__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u1__p1_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 5, 8, 7, 7,10,10, 7,
	 9,10, 5, 7, 8, 7,10, 9, 7,10,10, 5, 8, 8, 8,10,
	10, 8,10,10, 7,10,10,10,11,12,10,12,13, 7,10,10,
	 9,13,11,10,12,13, 5, 8, 8, 8,10,10, 8,10,10, 7,
	10,10,10,12,12, 9,11,12, 7,10,11,10,12,12,10,13,
	11,
};

static float _vq_quantthresh__16u1__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16u1__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u1__p1_0 = {
	_vq_quantthresh__16u1__p1_0,
	_vq_quantmap__16u1__p1_0,
	3,
	3
};

static static_codebook _16u1__p1_0 = {
	4, 81,
	_vq_lengthlist__16u1__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16u1__p1_0,
	NULL,
	&_vq_auxt__16u1__p1_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u1__p2_0[] = {
	 3, 4, 4, 5, 6, 6, 5, 6, 6, 5, 6, 6, 6, 7, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 6, 8, 7, 5, 6, 6, 6, 8,
	 8, 6, 8, 8, 6, 8, 8, 7, 7,10, 8, 9, 9, 6, 8, 8,
	 7, 9, 8, 8, 9,10, 5, 6, 6, 6, 8, 8, 7, 8, 8, 6,
	 8, 8, 8,10, 9, 7, 8, 9, 6, 8, 8, 8, 9, 9, 7,10,
	 8,
};

static float _vq_quantthresh__16u1__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16u1__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u1__p2_0 = {
	_vq_quantthresh__16u1__p2_0,
	_vq_quantmap__16u1__p2_0,
	3,
	3
};

static static_codebook _16u1__p2_0 = {
	4, 81,
	_vq_lengthlist__16u1__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16u1__p2_0,
	NULL,
	&_vq_auxt__16u1__p2_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u1__p3_0[] = {
	 1, 5, 5, 8, 8, 6, 7, 7, 9, 9, 5, 7, 7, 9, 9, 9,
	10, 9,11,11, 9, 9,10,11,11, 6, 8, 8,10,10, 8, 9,
	10,11,11, 8, 9,10,11,11,10,11,11,12,13,10,11,11,
	13,13, 6, 8, 8,10,10, 8,10, 9,11,11, 8,10, 9,11,
	11,10,11,11,13,13,10,11,11,13,12, 9,11,11,14,13,
	10,12,12,15,14,10,12,11,14,13,12,13,13,15,15,12,
	13,13,16,14, 9,11,11,13,14,10,11,12,14,14,10,12,
	12,14,15,12,13,13,14,15,12,13,14,15,16, 5, 8, 8,
	11,11, 8,10,10,12,12, 8,10,10,12,12,11,12,12,14,
	14,11,12,12,14,14, 8,10,10,12,12, 9,11,12,12,13,
	10,12,12,13,13,12,12,13,14,15,11,13,13,15,15, 7,
	10,10,12,12, 9,12,11,13,12,10,11,12,13,13,12,13,
	12,15,14,11,12,13,15,15,10,12,12,15,14,11,13,13,
	16,15,11,13,13,16,15,14,13,14,15,16,13,15,15,17,
	17,10,12,12,14,15,11,12,12,15,15,11,13,13,15,16,
	13,15,13,16,15,13,15,15,16,17, 5, 8, 8,11,11, 8,
	10,10,12,12, 8,10,10,12,12,11,12,12,14,14,11,12,
	12,14,14, 7,10,10,12,12,10,12,12,14,13, 9,11,12,
	12,13,12,13,13,15,15,12,12,13,13,15, 7,10,10,12,
	13,10,11,12,13,13,10,12,11,13,13,11,13,13,15,15,
	12,13,12,15,14, 9,12,12,15,14,11,13,13,15,15,11,
	12,13,15,15,13,14,14,17,19,13,13,14,16,16,10,12,
	12,14,15,11,13,13,15,16,11,13,12,16,15,13,15,15,
	17,18,14,15,13,16,15, 8,11,11,15,14,10,12,12,16,
	15,10,12,12,16,16,14,15,15,18,17,13,14,15,16,18,
	 9,12,12,15,15,11,12,14,16,17,11,13,13,16,15,15,
	15,15,17,18,14,15,16,17,17, 9,12,12,15,15,11,14,
	13,16,16,11,13,13,16,16,15,16,15,17,18,14,16,15,
	17,16,12,14,14,17,16,12,14,15,18,17,13,15,15,17,
	17,15,15,18,16,20,15,16,17,18,18,11,14,14,16,17,
	13,15,14,18,17,13,15,15,17,17,15,17,15,18,17,15,
	17,16,19,18, 8,11,11,14,15,10,12,12,15,15,10,12,
	12,16,16,13,14,14,17,16,14,15,15,17,17, 9,12,12,
	15,16,11,13,13,16,16,11,12,13,16,16,14,16,15,20,
	17,14,16,16,17,17, 9,12,12,15,16,11,13,13,16,17,
	11,13,13,17,16,14,15,15,17,18,15,15,15,18,18,11,
	14,14,17,16,13,15,15,17,17,13,14,14,18,17,15,16,
	16,18,19,15,15,17,17,19,11,14,14,16,17,13,15,14,
	17,19,13,15,14,18,17,15,17,16,18,18,15,17,15,18,
	16,
};

static float _vq_quantthresh__16u1__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u1__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u1__p3_0 = {
	_vq_quantthresh__16u1__p3_0,
	_vq_quantmap__16u1__p3_0,
	5,
	5
};

static static_codebook _16u1__p3_0 = {
	4, 625,
	_vq_lengthlist__16u1__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u1__p3_0,
	NULL,
	&_vq_auxt__16u1__p3_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u1__p4_0[] = {
	 4, 5, 5, 8, 8, 6, 6, 7, 9, 9, 6, 6, 6, 9, 9, 9,
	10, 9,11,11, 9, 9,10,11,11, 6, 7, 7,10, 9, 7, 7,
	 8, 9,10, 7, 7, 8,10,10,10,10,10,10,12, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 7, 8, 7,10,10, 7, 8, 7,10,
	10, 9,10, 9,12,11,10,10, 9,12,10, 9,10,10,12,11,
	10,10,10,12,12, 9,10,10,12,12,12,11,12,13,13,11,
	11,12,12,13, 9,10,10,11,12, 9,10,10,12,12,10,10,
	10,12,12,11,12,11,14,13,11,12,12,14,13, 5, 7, 7,
	10,10, 7, 8, 8,10,10, 7, 8, 7,10,10,10,10,10,12,
	12,10,10,10,12,12, 6, 8, 7,10,10, 7, 7, 9,10,11,
	 8, 9, 9,11,10,10,10,11,11,13,10,10,11,12,13, 6,
	 8, 8,10,10, 7, 9, 8,11,10, 8, 9, 9,10,11,10,11,
	10,13,11,10,11,10,12,12,10,11,10,12,11,10,10,10,
	12,13,10,11,11,13,12,11,11,13,11,14,12,12,13,14,
	14, 9,10,10,12,13,10,11,10,13,12,10,11,11,12,13,
	11,12,11,14,12,12,13,13,15,14, 5, 7, 7,10,10, 7,
	 7, 8,10,10, 7, 8, 8,10,10,10,10,10,11,12,10,10,
	10,12,12, 7, 8, 8,10,10, 8, 9, 8,11,10, 7, 8, 9,
	10,11,10,11,11,12,12,10,10,11,11,13, 7, 7, 8,10,
	10, 8, 8, 9,10,11, 7, 9, 7,11,10,10,11,11,13,12,
	11,11,10,13,11, 9,10,10,12,12,10,11,11,13,12,10,
	10,11,12,12,12,13,13,14,14,11,11,12,12,14,10,10,
	11,12,12,10,11,11,12,13,10,10,10,13,12,12,13,13,
	15,14,12,13,10,14,11, 8,10,10,12,12,10,11,10,13,
	13, 9,10,10,12,12,12,13,13,15,14,11,12,12,13,13,
	 9,10,10,13,12,10,10,11,13,13,10,11,10,13,12,12,
	12,13,14,15,12,13,12,15,13, 9,10,10,12,13,10,11,
	10,13,12,10,10,11,12,13,12,14,12,15,13,12,12,13,
	14,15,11,12,11,14,13,11,11,12,14,15,12,13,12,15,
	14,13,11,15,11,16,13,14,14,16,15,11,12,12,14,14,
	11,12,11,14,13,12,12,13,14,15,13,14,12,16,12,14,
	14,14,15,15, 8,10,10,12,12, 9,10,10,12,12,10,10,
	11,13,13,11,12,12,13,13,12,13,13,14,15, 9,10,10,
	13,12,10,11,11,13,12,10,10,11,13,13,12,13,12,15,
	14,12,12,13,13,16, 9, 9,10,12,13,10,10,11,12,13,
	10,11,10,13,13,12,12,13,13,15,13,13,12,15,13,11,
	12,12,14,14,12,13,12,15,14,11,11,12,13,14,14,14,
	14,16,15,13,12,15,12,16,11,11,12,13,14,12,13,13,
	14,15,10,12,11,14,13,14,15,14,16,16,13,14,11,15,
	11,
};

static float _vq_quantthresh__16u1__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u1__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u1__p4_0 = {
	_vq_quantthresh__16u1__p4_0,
	_vq_quantmap__16u1__p4_0,
	5,
	5
};

static static_codebook _16u1__p4_0 = {
	4, 625,
	_vq_lengthlist__16u1__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u1__p4_0,
	NULL,
	&_vq_auxt__16u1__p4_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16u1__p5_0[] = {
	 1, 4, 4, 7, 7, 7, 7, 9, 9, 4, 6, 6, 8, 8, 8, 8,
	10,10, 4, 5, 6, 8, 8, 8, 8,10,10, 7, 8, 8, 9, 9,
	 9, 9,11,11, 7, 8, 8, 9, 9, 9, 9,11,11, 7, 8, 8,
	10, 9,11,11,12,11, 7, 8, 8, 9, 9,11,11,12,12, 9,
	10,10,11,11,12,12,13,12, 9,10,10,11,11,12,12,12,
	13,
};

static float _vq_quantthresh__16u1__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16u1__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16u1__p5_0 = {
	_vq_quantthresh__16u1__p5_0,
	_vq_quantmap__16u1__p5_0,
	9,
	9
};

static static_codebook _16u1__p5_0 = {
	2, 81,
	_vq_lengthlist__16u1__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16u1__p5_0,
	NULL,
	&_vq_auxt__16u1__p5_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p6_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16u1__p6_0[] = {
	 3, 4, 4, 6, 6, 7, 7, 9, 9, 4, 4, 4, 6, 6, 8, 8,
	 9, 9, 4, 4, 4, 6, 6, 7, 7, 9, 9, 6, 6, 6, 7, 7,
	 8, 8,10, 9, 6, 6, 6, 7, 7, 8, 8, 9,10, 7, 8, 7,
	 8, 8, 9, 9,10,10, 7, 8, 8, 8, 8, 9, 9,10,10, 9,
	 9, 9,10,10,10,10,11,11, 9, 9, 9,10,10,10,10,11,
	11,
};

static float _vq_quantthresh__16u1__p6_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16u1__p6_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16u1__p6_0 = {
	_vq_quantthresh__16u1__p6_0,
	_vq_quantmap__16u1__p6_0,
	9,
	9
};

static static_codebook _16u1__p6_0 = {
	2, 81,
	_vq_lengthlist__16u1__p6_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16u1__p6_0,
	NULL,
	&_vq_auxt__16u1__p6_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u1__p7_0[] = {
	 1, 4, 4, 4, 8, 8, 4, 8, 8, 5,11, 9, 8,12,11, 8,
	12,11, 5,10,11, 8,11,12, 8,11,12, 4,11,11,11,14,
	13,10,13,13, 8,14,13,12,14,16,12,16,15, 8,14,14,
	13,16,14,12,15,16, 4,11,11,10,14,13,11,14,14, 8,
	15,14,12,15,15,12,14,16, 8,14,14,11,16,15,12,15,
	13,
};

static float _vq_quantthresh__16u1__p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__16u1__p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u1__p7_0 = {
	_vq_quantthresh__16u1__p7_0,
	_vq_quantmap__16u1__p7_0,
	3,
	3
};

static static_codebook _16u1__p7_0 = {
	4, 81,
	_vq_lengthlist__16u1__p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__16u1__p7_0,
	NULL,
	&_vq_auxt__16u1__p7_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16u1__p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 4, 6, 5, 7, 7,
	 8, 8, 8, 8, 8, 8, 4, 5, 6, 7, 7, 8, 8, 8, 8, 8,
	 8, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 6, 7, 7, 8,
	 8, 8, 8, 9, 9, 9, 9, 7, 8, 8, 8, 8, 9, 9, 9,10,
	 9,10, 7, 8, 8, 8, 8, 9, 9, 9, 9,10, 9, 8, 8, 8,
	 9, 9,10,10,10,10,10,10, 8, 8, 8, 9, 9, 9, 9,10,
	10,10,10, 8, 8, 8, 9, 9, 9,10,10,10,10,10, 8, 8,
	 8, 9, 9,10,10,10,10,10,10,
};

static float _vq_quantthresh__16u1__p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16u1__p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16u1__p7_1 = {
	_vq_quantthresh__16u1__p7_1,
	_vq_quantmap__16u1__p7_1,
	11,
	11
};

static static_codebook _16u1__p7_1 = {
	2, 121,
	_vq_lengthlist__16u1__p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16u1__p7_1,
	NULL,
	&_vq_auxt__16u1__p7_1,
	NULL,
	0
};

static long _vq_quantlist__16u1__p8_0[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16u1__p8_0[] = {
	 1, 4, 4, 5, 5, 8, 8,10,10,12,12, 4, 7, 7, 8, 8,
	 9, 9,12,11,14,13, 4, 7, 7, 7, 8, 9,10,11,11,13,
	12, 5, 8, 8, 9, 9,11,11,12,13,15,14, 5, 7, 8, 9,
	 9,11,11,13,13,17,15, 8, 9,10,11,11,12,13,17,14,
	17,16, 8,10, 9,11,11,12,12,13,15,15,17,10,11,11,
	12,13,14,15,15,16,16,17, 9,11,11,12,12,14,15,17,
	15,15,16,11,14,12,14,15,16,15,16,16,16,15,11,13,
	13,14,14,15,15,16,16,15,16,
};

static float _vq_quantthresh__16u1__p8_0[] = {
	-49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 27.5, 
	38.5, 49.5, 
};

static long _vq_quantmap__16u1__p8_0[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16u1__p8_0 = {
	_vq_quantthresh__16u1__p8_0,
	_vq_quantmap__16u1__p8_0,
	11,
	11
};

static static_codebook _16u1__p8_0 = {
	2, 121,
	_vq_lengthlist__16u1__p8_0,
	1, -524582912, 1618345984, 4, 0,
	_vq_quantlist__16u1__p8_0,
	NULL,
	&_vq_auxt__16u1__p8_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p8_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16u1__p8_1[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 4, 6, 6, 7, 7,
	 8, 7, 8, 8, 8, 8, 4, 6, 6, 7, 7, 7, 7, 8, 8, 8,
	 8, 6, 7, 7, 7, 7, 8, 8, 8, 8, 8, 9, 6, 7, 7, 7,
	 7, 8, 8, 8, 8, 9, 9, 7, 7, 7, 8, 8, 8, 8, 9, 9,
	 9, 9, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 8, 8, 8,
	 8, 8, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 8, 8, 8, 9, 8, 9, 9, 9, 9, 9, 9, 8, 8,
	 8, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__16u1__p8_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16u1__p8_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16u1__p8_1 = {
	_vq_quantthresh__16u1__p8_1,
	_vq_quantmap__16u1__p8_1,
	11,
	11
};

static static_codebook _16u1__p8_1 = {
	2, 121,
	_vq_lengthlist__16u1__p8_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16u1__p8_1,
	NULL,
	&_vq_auxt__16u1__p8_1,
	NULL,
	0
};

static long _vq_quantlist__16u1__p9_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16u1__p9_0[] = {
	 1, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8,
};

static float _vq_quantthresh__16u1__p9_0[] = {
	-1657.5, -1402.5, -1147.5, -892.5, -637.5, -382.5, -127.5, 127.5, 
	382.5, 637.5, 892.5, 1147.5, 1402.5, 1657.5, 
};

static long _vq_quantmap__16u1__p9_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16u1__p9_0 = {
	_vq_quantthresh__16u1__p9_0,
	_vq_quantmap__16u1__p9_0,
	15,
	15
};

static static_codebook _16u1__p9_0 = {
	2, 225,
	_vq_lengthlist__16u1__p9_0,
	1, -514071552, 1627381760, 4, 0,
	_vq_quantlist__16u1__p9_0,
	NULL,
	&_vq_auxt__16u1__p9_0,
	NULL,
	0
};

static long _vq_quantlist__16u1__p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16u1__p9_1[] = {
	 1, 6, 5, 9, 9,10,10, 6, 7, 9, 9,10,10,10,10, 5,
	10, 8,10, 8,10,10, 8, 8,10, 9,10,10,10,10, 5, 8,
	 9,10,10,10,10, 8,10,10,10,10,10,10,10, 9,10,10,
	10,10,10,10, 9, 9,10,10,10,10,10,10, 9, 9, 8, 9,
	10,10,10, 9,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10, 8,10,10,10,10,
	10,10,10,10,10,10,10,10,10, 6, 8, 8,10,10,10, 8,
	10,10,10,10,10,10,10,10, 5, 8, 8,10,10,10, 9, 9,
	10,10,10,10,10,10,10,10, 9,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__16u1__p9_1[] = {
	-110.5, -93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 
	25.5, 42.5, 59.5, 76.5, 93.5, 110.5, 
};

static long _vq_quantmap__16u1__p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16u1__p9_1 = {
	_vq_quantthresh__16u1__p9_1,
	_vq_quantmap__16u1__p9_1,
	15,
	15
};

static static_codebook _16u1__p9_1 = {
	2, 225,
	_vq_lengthlist__16u1__p9_1,
	1, -522338304, 1620115456, 4, 0,
	_vq_quantlist__16u1__p9_1,
	NULL,
	&_vq_auxt__16u1__p9_1,
	NULL,
	0
};

static long _vq_quantlist__16u1__p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__16u1__p9_2[] = {
	 1, 6, 6, 7, 8, 8,11,10, 9, 9,11, 9,10, 9,11,11,
	 9, 6, 7, 6,11, 8,11, 9,10,10,11, 9,11,10,10,10,
	11, 9, 5, 7, 7, 8, 8,10,11, 8, 8,11, 9, 9,10,11,
	 9,10,11, 8, 9, 6, 8, 8, 9, 9,10,10,11,11,11, 9,
	11,10, 9,11, 8, 8, 8, 9, 8, 9,10,11, 9, 9,11,11,
	10, 9, 9,11,10, 8,11, 8, 9, 8,11, 9,10, 9,10,11,
	11,10,10, 9,10,10, 8, 8, 9,10,10,10, 9,11, 9,10,
	11,11,11,11,10, 9,11, 9, 9,11,11,10, 8,11,11,11,
	 9,10,10,11,10,11,11, 9,11,10, 9,11,10,10,10,10,
	 9,11,10,11,10, 9, 9,10,11, 9, 8,10,11,11,10,10,
	11, 9,11,10,11,11,10,11, 9, 9, 8,10, 8, 9,11, 9,
	 8,10,10, 9,11,10,11,10,11, 9,11, 8,10,11,11,11,
	11,10,10,11,11,11,11,10,11,11,10, 9, 8,10,10, 9,
	11,10,11,11,11, 9, 9, 9,11,11,11,10,10, 9, 9,10,
	 9,11,11,11,11, 8,10,11,10,11,11,10,11,11, 9, 9,
	 9,10, 9,11, 9,11,11,11,11,11,10,11,11,10,11,10,
	11,11, 9,11,10,11,10, 9,10, 9,10,10,11,11,11,11,
	 9,10, 9,10,11,11,10,11,11,11,11,11,11,10,11,11,
	10,
};

static float _vq_quantthresh__16u1__p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__16u1__p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__16u1__p9_2 = {
	_vq_quantthresh__16u1__p9_2,
	_vq_quantmap__16u1__p9_2,
	17,
	17
};

static static_codebook _16u1__p9_2 = {
	2, 289,
	_vq_lengthlist__16u1__p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__16u1__p9_2,
	NULL,
	&_vq_auxt__16u1__p9_2,
	NULL,
	0
};

static long _huff_lengthlist__16u1__short[] = {
	 5, 7,10, 9,11,10,15,11,13,16, 6, 4, 6, 6, 7, 7,
	10, 9,12,16,10, 6, 5, 6, 6, 7,10,11,16,16, 9, 6,
	 7, 6, 7, 7,10, 8,14,16,11, 6, 5, 4, 5, 6, 8, 9,
	15,16, 9, 6, 6, 5, 6, 6, 9, 8,14,16,12, 7, 6, 6,
	 5, 6, 6, 7,13,16, 8, 6, 7, 6, 5, 5, 4, 4,11,16,
	 9, 8, 9, 9, 7, 7, 6, 5,13,16,14,14,16,15,16,15,
	16,16,16,16,
};

static static_codebook _huff_book__16u1__short = {
	2, 100,
	_huff_lengthlist__16u1__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__16u2__long[] = {
	 5, 7,10,10,10,11,11,13,18,19, 6, 5, 5, 6, 7, 8,
	 9,12,19,19, 8, 5, 4, 4, 6, 7, 9,13,19,19, 8, 5,
	 4, 4, 5, 6, 8,12,17,19, 7, 5, 5, 4, 4, 5, 7,12,
	18,18, 8, 7, 7, 6, 5, 5, 6,10,18,18, 9, 9, 9, 8,
	 6, 5, 6, 9,18,18,11,13,13,13, 8, 7, 7, 9,16,18,
	13,17,18,16,11, 9, 9, 9,17,18,15,18,18,18,15,13,
	13,14,18,18,
};

static static_codebook _huff_book__16u2__long = {
	2, 100,
	_huff_lengthlist__16u2__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__16u2__short[] = {
	 8,11,12,12,14,15,16,16,16,16, 9, 7, 7, 8, 9,11,
	13,14,16,16,13, 7, 6, 6, 7, 9,12,13,15,16,15, 7,
	 6, 5, 4, 6,10,11,14,16,12, 8, 7, 4, 2, 4, 7,10,
	14,16,11, 9, 7, 5, 3, 4, 6, 9,14,16,11,10, 9, 7,
	 5, 5, 6, 9,16,16,10,10, 9, 8, 6, 6, 7,10,16,16,
	11,11,11,10,10,10,11,14,16,16,16,14,14,13,14,16,
	16,16,16,16,
};

static static_codebook _huff_book__16u2__short = {
	2, 100,
	_huff_lengthlist__16u2__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__16u2_p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u2_p1_0[] = {
	 1, 5, 5, 5, 7, 7, 5, 7, 7, 5, 7, 7, 7, 9, 9, 7,
	 9, 9, 5, 7, 7, 7, 9, 9, 7, 9, 9, 5, 7, 7, 8, 9,
	 9, 7, 9, 9, 7, 9, 9, 9,10,10, 9,10,10, 7, 9, 9,
	 9,10,10, 9,10,11, 5, 7, 8, 8, 9, 9, 8, 9, 9, 7,
	 9, 9, 9,10,10, 9, 9,10, 7, 9, 9, 9,10,10, 9,11,
	10,
};

static float _vq_quantthresh__16u2_p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__16u2_p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u2_p1_0 = {
	_vq_quantthresh__16u2_p1_0,
	_vq_quantmap__16u2_p1_0,
	3,
	3
};

static static_codebook _16u2_p1_0 = {
	4, 81,
	_vq_lengthlist__16u2_p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__16u2_p1_0,
	NULL,
	&_vq_auxt__16u2_p1_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p2_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u2_p2_0[] = {
	 3, 5, 5, 8, 8, 5, 7, 7, 9, 9, 5, 7, 7, 9, 9, 9,
	10, 9,11,11, 9, 9, 9,11,11, 5, 7, 7, 9, 9, 7, 8,
	 8,10,10, 7, 8, 8,10,10,10,10,10,12,12, 9,10,10,
	11,12, 5, 7, 7, 9, 9, 7, 8, 8,10,10, 7, 8, 8,10,
	10, 9,10,10,12,11,10,10,10,12,12, 9,10,10,12,12,
	10,11,10,13,12, 9,10,10,12,12,12,12,12,14,14,11,
	12,12,13,14, 9,10,10,12,12, 9,10,10,12,12,10,10,
	10,12,12,11,12,12,14,13,12,13,12,14,14, 5, 7, 7,
	 9, 9, 7, 8, 8,10,10, 7, 8, 8,10,10,10,11,10,12,
	12,10,10,11,12,12, 7, 8, 8,10,10, 8, 9, 9,11,11,
	 8, 9, 9,11,11,11,11,11,12,13,10,11,11,12,13, 7,
	 8, 8,10,10, 8, 9, 8,11,10, 8, 9, 9,11,11,10,11,
	10,13,12,10,11,11,13,13, 9,11,10,13,13,10,11,11,
	13,13,10,11,11,13,13,12,12,13,13,15,12,12,13,14,
	15, 9,10,10,12,12,10,11,10,13,12,10,11,11,13,13,
	11,13,11,14,13,12,13,13,15,15, 5, 7, 7, 9, 9, 7,
	 8, 8,10,10, 7, 8, 8,10,10,10,10,10,12,12,10,10,
	11,12,12, 7, 8, 8,10,10, 8, 9, 9,11,11, 8, 8, 9,
	10,11,10,11,11,13,13,10,10,11,12,13, 7, 8, 8,10,
	11, 8, 9, 9,11,11, 8, 9, 9,11,11,10,11,11,13,12,
	11,11,11,13,12, 9,10,10,12,12,10,11,11,13,13,10,
	10,11,12,13,12,13,13,15,14,11,11,13,12,14,10,10,
	11,13,13,10,11,11,13,13,10,11,11,13,13,12,13,13,
	14,14,12,13,12,14,13, 8,10, 9,12,12, 9,11,10,13,
	13, 9,10,10,12,13,12,13,13,14,14,12,12,13,14,14,
	 9,11,10,13,13,10,11,11,13,13,10,11,11,13,13,12,
	13,13,15,15,13,13,13,14,15, 9,10,10,12,13,10,11,
	10,13,12,10,11,11,13,13,12,13,12,15,14,13,13,13,
	14,15,11,12,12,15,14,12,12,13,15,15,12,13,13,15,
	14,14,13,15,14,16,13,14,15,16,16,11,12,12,14,14,
	11,12,12,15,14,12,13,13,15,15,13,14,13,16,14,14,
	14,14,16,16, 8, 9, 9,12,12, 9,10,10,13,12, 9,10,
	10,13,13,12,12,12,14,14,12,12,13,15,15, 9,10,10,
	13,12,10,11,11,13,13,10,10,11,13,14,12,13,13,15,
	15,12,12,13,14,15, 9,10,10,13,13,10,11,11,13,13,
	10,11,11,13,13,12,13,13,14,14,13,14,13,15,14,11,
	12,12,14,14,12,13,13,15,14,11,12,12,14,15,14,14,
	14,16,15,13,12,14,14,16,11,12,13,14,15,12,13,13,
	14,16,12,13,12,15,14,13,15,14,16,16,14,15,13,16,
	13,
};

static float _vq_quantthresh__16u2_p2_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u2_p2_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u2_p2_0 = {
	_vq_quantthresh__16u2_p2_0,
	_vq_quantmap__16u2_p2_0,
	5,
	5
};

static static_codebook _16u2_p2_0 = {
	4, 625,
	_vq_lengthlist__16u2_p2_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u2_p2_0,
	NULL,
	&_vq_auxt__16u2_p2_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p3_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__16u2_p3_0[] = {
	 2, 4, 4, 6, 6, 7, 7, 9, 9, 4, 5, 5, 6, 6, 8, 7,
	 9, 9, 4, 5, 5, 6, 6, 7, 8, 9, 9, 6, 6, 6, 7, 7,
	 8, 8,10,10, 6, 6, 6, 7, 7, 8, 8, 9,10, 7, 8, 7,
	 8, 8, 9, 9,10,10, 7, 8, 8, 8, 8, 9, 9,10,10, 9,
	 9, 9,10, 9,10,10,11,11, 9, 9, 9,10,10,10,10,11,
	11,
};

static float _vq_quantthresh__16u2_p3_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__16u2_p3_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__16u2_p3_0 = {
	_vq_quantthresh__16u2_p3_0,
	_vq_quantmap__16u2_p3_0,
	9,
	9
};

static static_codebook _16u2_p3_0 = {
	2, 81,
	_vq_lengthlist__16u2_p3_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__16u2_p3_0,
	NULL,
	&_vq_auxt__16u2_p3_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p4_0[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__16u2_p4_0[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10,11,11,11,
	11, 5, 5, 5, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,11,
	12,11, 5, 5, 5, 7, 7, 8, 8, 9, 9, 9, 9,10,10,11,
	11,12,12, 6, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12, 6, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9,10,
	10,11,11,12,12, 7, 8, 8, 8, 8, 9, 9, 9, 9,10,10,
	11,11,12,12,12,12, 7, 8, 8, 8, 8, 9, 9, 9, 9,10,
	10,11,11,11,12,12,12, 9, 9, 9, 9, 9, 9,10,10,10,
	10,10,11,11,12,12,13,13, 8, 9, 9, 9, 9,10, 9,10,
	10,10,10,11,11,12,12,13,13, 9, 9, 9, 9, 9,10,10,
	10,10,11,11,11,12,12,12,13,13, 9, 9, 9, 9, 9,10,
	10,10,10,11,11,12,11,12,12,13,13,10,10,10,10,10,
	11,11,11,11,11,12,12,12,12,13,13,14,10,10,10,10,
	10,11,11,11,11,12,11,12,12,13,12,13,13,11,11,11,
	11,11,12,12,12,12,12,12,13,13,13,13,14,14,11,11,
	11,11,11,12,12,12,12,12,12,13,12,13,13,14,14,11,
	12,12,12,12,12,12,13,13,13,13,13,13,14,14,14,14,
	11,12,12,12,12,12,12,13,13,13,13,14,13,14,14,14,
	14,
};

static float _vq_quantthresh__16u2_p4_0[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__16u2_p4_0[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__16u2_p4_0 = {
	_vq_quantthresh__16u2_p4_0,
	_vq_quantmap__16u2_p4_0,
	17,
	17
};

static static_codebook _16u2_p4_0 = {
	2, 289,
	_vq_lengthlist__16u2_p4_0,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__16u2_p4_0,
	NULL,
	&_vq_auxt__16u2_p4_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p5_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__16u2_p5_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 5, 9, 8, 7,10, 9, 7,
	10, 9, 5, 8, 9, 7, 9,10, 7, 9,10, 4, 9, 9, 9,11,
	11, 8,11,11, 7,11,11,10,10,13,10,14,13, 7,11,11,
	10,13,11,10,13,14, 5, 9, 9, 8,11,11, 9,11,11, 7,
	11,11,10,14,13,10,12,14, 7,11,11,10,13,13,10,13,
	10,
};

static float _vq_quantthresh__16u2_p5_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__16u2_p5_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__16u2_p5_0 = {
	_vq_quantthresh__16u2_p5_0,
	_vq_quantmap__16u2_p5_0,
	3,
	3
};

static static_codebook _16u2_p5_0 = {
	4, 81,
	_vq_lengthlist__16u2_p5_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__16u2_p5_0,
	NULL,
	&_vq_auxt__16u2_p5_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p5_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16u2_p5_1[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 5, 5, 5, 7, 7,
	 7, 7, 8, 8, 8, 8, 5, 5, 6, 7, 7, 7, 7, 8, 8, 8,
	 8, 6, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 6, 7, 7, 7,
	 7, 8, 8, 8, 8, 8, 8, 7, 7, 7, 8, 8, 8, 8, 9, 9,
	 9, 9, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 8, 8, 8,
	 8, 8, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 8,
	 8, 8, 8, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__16u2_p5_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16u2_p5_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16u2_p5_1 = {
	_vq_quantthresh__16u2_p5_1,
	_vq_quantmap__16u2_p5_1,
	11,
	11
};

static static_codebook _16u2_p5_1 = {
	2, 121,
	_vq_lengthlist__16u2_p5_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16u2_p5_1,
	NULL,
	&_vq_auxt__16u2_p5_1,
	NULL,
	0
};

static long _vq_quantlist__16u2_p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16u2_p6_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 8, 8, 9, 9,10,10, 4, 6, 6,
	 8, 8, 9, 9, 9, 9,10,10,12,11, 4, 6, 6, 8, 8, 9,
	 9, 9, 9,10,10,11,12, 7, 8, 8, 9, 9,10,10,10,10,
	12,12,13,12, 7, 8, 8, 9, 9,10,10,10,10,11,12,12,
	12, 8, 9, 9,10,10,11,11,11,11,12,12,13,13, 8, 9,
	 9,10,10,11,11,11,11,12,13,13,13, 8, 9, 9,10,10,
	11,11,12,12,13,13,14,14, 8, 9, 9,10,10,11,11,12,
	12,13,13,14,14, 9,10,10,11,12,13,12,13,14,14,14,
	14,14, 9,10,10,11,12,12,13,13,13,14,14,14,14,10,
	11,11,12,12,13,13,14,14,15,15,15,15,10,11,11,12,
	12,13,13,14,14,14,14,15,15,
};

static float _vq_quantthresh__16u2_p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__16u2_p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16u2_p6_0 = {
	_vq_quantthresh__16u2_p6_0,
	_vq_quantmap__16u2_p6_0,
	13,
	13
};

static static_codebook _16u2_p6_0 = {
	2, 169,
	_vq_lengthlist__16u2_p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__16u2_p6_0,
	NULL,
	&_vq_auxt__16u2_p6_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__16u2_p6_1[] = {
	 2, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
	 5, 5, 6, 6, 5, 5, 5, 6, 6,
};

static float _vq_quantthresh__16u2_p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__16u2_p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__16u2_p6_1 = {
	_vq_quantthresh__16u2_p6_1,
	_vq_quantmap__16u2_p6_1,
	5,
	5
};

static static_codebook _16u2_p6_1 = {
	2, 25,
	_vq_lengthlist__16u2_p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__16u2_p6_1,
	NULL,
	&_vq_auxt__16u2_p6_1,
	NULL,
	0
};

static long _vq_quantlist__16u2_p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__16u2_p7_0[] = {
	 1, 4, 4, 7, 7, 7, 7, 8, 8, 9, 9,10,10, 4, 6, 6,
	 9, 9, 9, 9, 9, 9,10,10,11,11, 4, 6, 6, 8, 9, 9,
	 9, 9, 9,10,11,12,11, 7, 8, 9,10,10,10,10,11,10,
	11,12,12,13, 7, 9, 9,10,10,10,10,10,10,11,12,13,
	13, 7, 9, 8,10,10,11,11,11,12,12,13,13,14, 7, 9,
	 9,10,10,11,11,11,12,13,13,13,13, 8, 9, 9,10,11,
	11,12,12,12,13,13,13,13, 8, 9, 9,10,11,11,11,12,
	12,13,13,14,14, 9,10,10,12,11,12,13,13,13,14,13,
	13,13, 9,10,10,11,11,12,12,13,14,13,13,14,13,10,
	11,11,12,13,14,14,14,15,14,14,14,14,10,11,11,12,
	12,13,13,13,14,14,14,15,14,
};

static float _vq_quantthresh__16u2_p7_0[] = {
	-60.5, -49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 
	27.5, 38.5, 49.5, 60.5, 
};

static long _vq_quantmap__16u2_p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16u2_p7_0 = {
	_vq_quantthresh__16u2_p7_0,
	_vq_quantmap__16u2_p7_0,
	13,
	13
};

static static_codebook _16u2_p7_0 = {
	2, 169,
	_vq_lengthlist__16u2_p7_0,
	1, -523206656, 1618345984, 4, 0,
	_vq_quantlist__16u2_p7_0,
	NULL,
	&_vq_auxt__16u2_p7_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__16u2_p7_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 7, 7, 5, 6, 6, 7, 7,
	 7, 7, 7, 7, 8, 8, 5, 6, 6, 6, 6, 7, 7, 7, 7, 8,
	 8, 6, 6, 7, 7, 7, 8, 7, 8, 8, 8, 8, 6, 7, 7, 7,
	 7, 7, 7, 8, 8, 8, 8, 7, 7, 7, 7, 7, 8, 8, 8, 8,
	 8, 8, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7,
	 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 8, 8, 8, 8, 8,
	 8, 8, 8, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__16u2_p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__16u2_p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__16u2_p7_1 = {
	_vq_quantthresh__16u2_p7_1,
	_vq_quantmap__16u2_p7_1,
	11,
	11
};

static static_codebook _16u2_p7_1 = {
	2, 121,
	_vq_lengthlist__16u2_p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__16u2_p7_1,
	NULL,
	&_vq_auxt__16u2_p7_1,
	NULL,
	0
};

static long _vq_quantlist__16u2_p8_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__16u2_p8_0[] = {
	 1, 5, 5, 7, 7, 8, 8, 7, 7, 8, 8,10, 9,11,11, 4,
	 6, 6, 8, 8,10, 9, 9, 8, 9, 9,10,10,12,14, 4, 6,
	 7, 8, 9, 9,10, 9, 8, 9, 9,10,12,12,11, 7, 8, 8,
	10,10,10,10, 9, 9,10,10,11,13,13,12, 7, 8, 8, 9,
	11,11,10, 9, 9,11,10,12,11,11,14, 8, 9, 9,11,10,
	11,11,10,10,11,11,13,12,14,12, 8, 9, 9,11,12,11,
	11,10,10,12,11,12,12,12,14, 7, 8, 8, 9, 9,10,10,
	10,11,12,11,13,13,14,12, 7, 8, 9, 9, 9,10,10,11,
	11,11,12,12,14,14,14, 8,10, 9,10,11,11,11,11,14,
	12,12,13,14,14,13, 9, 9, 9,10,11,11,11,12,12,12,
	14,12,14,13,14,10,10,10,12,11,12,11,14,13,14,13,
	14,14,13,14, 9,10,10,11,12,11,13,12,13,13,14,14,
	14,13,14,10,13,13,12,12,11,12,14,13,14,13,14,12,
	14,13,10,11,11,12,11,12,12,14,14,14,13,14,14,14,
	14,
};

static float _vq_quantthresh__16u2_p8_0[] = {
	-136.5, -115.5, -94.5, -73.5, -52.5, -31.5, -10.5, 10.5, 
	31.5, 52.5, 73.5, 94.5, 115.5, 136.5, 
};

static long _vq_quantmap__16u2_p8_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__16u2_p8_0 = {
	_vq_quantthresh__16u2_p8_0,
	_vq_quantmap__16u2_p8_0,
	15,
	15
};

static static_codebook _16u2_p8_0 = {
	2, 225,
	_vq_lengthlist__16u2_p8_0,
	1, -520986624, 1620377600, 4, 0,
	_vq_quantlist__16u2_p8_0,
	NULL,
	&_vq_auxt__16u2_p8_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p8_1[] = {
	10,
	9,
	11,
	8,
	12,
	7,
	13,
	6,
	14,
	5,
	15,
	4,
	16,
	3,
	17,
	2,
	18,
	1,
	19,
	0,
	20,
};

static long _vq_lengthlist__16u2_p8_1[] = {
	 2, 5, 5, 7, 7, 8, 8, 8, 8, 9, 9,10, 9,10, 9, 9,
	 9,10,10,10,10, 5, 6, 6, 7, 7, 8, 8, 9, 9, 9, 9,
	10, 9,10,10,10,10,10,10,11,10, 5, 6, 6, 7, 7, 8,
	 8, 8, 9, 9,10,10,10,10,10,10,10,10,10,10,10, 7,
	 7, 7, 8, 8, 9, 8, 9, 9,10, 9,10,10,10,10,10,10,
	11,10,11,10, 7, 7, 7, 8, 8, 8, 9, 9, 9,10, 9,10,
	10,10,10,10,10,10,10,10,10, 8, 8, 8, 9, 9, 9, 9,
	10, 9,10,10,10,10,10,10,10,11,10,10,11,10, 8, 8,
	 8, 8, 9, 9, 9, 9,10,10,10,10,10,10,10,10,10,11,
	11,10,10, 8, 9, 9, 9, 9, 9, 9,10,10,10,10,10,10,
	11,10,11,10,11,10,11,10, 8, 9, 9, 9, 9, 9,10,10,
	10,10,10,10,10,10,10,10,11,11,10,10,10, 9,10, 9,
	 9,10,10,10,11,10,10,10,10,10,10,10,10,11,11,11,
	11,11, 9, 9, 9,10, 9,10,10,10,10,10,10,11,10,11,
	10,11,11,11,11,10,10, 9,10, 9,10,10,10,10,11,10,
	10,10,10,10,11,10,11,10,11,10,10,11, 9,10,10,10,
	10,10,10,10,10,10,11,10,10,11,11,10,11,11,11,11,
	11, 9, 9,10,10,10,10,10,11,10,10,11,10,10,11,10,
	10,11,11,11,11,11, 9,10,10,10,10,10,10,10,11,10,
	11,10,11,10,11,11,11,11,11,10,11,10,10,10,10,10,
	10,10,10,10,11,11,11,11,11,11,11,11,11,10,11,11,
	10,10,10,10,10,11,10,10,10,11,10,11,11,11,11,10,
	12,11,11,11,10,10,10,10,10,10,11,10,10,10,11,11,
	12,11,11,11,11,11,11,11,11,11,10,10,10,11,10,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,
	10,10,11,10,11,10,10,11,11,11,11,11,11,11,11,11,
	11,11,11,10,10,10,10,10,10,10,11,11,10,11,11,10,
	11,11,10,11,11,11,10,11,11,
};

static float _vq_quantthresh__16u2_p8_1[] = {
	-9.5, -8.5, -7.5, -6.5, -5.5, -4.5, -3.5, -2.5, 
	-1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 
	6.5, 7.5, 8.5, 9.5, 
};

static long _vq_quantmap__16u2_p8_1[] = {
	   19,   17,   15,   13,   11,    9,    7,    5,
	    3,    1,    0,    2,    4,    6,    8,   10,
	   12,   14,   16,   18,   20,
};

static encode_aux_threshmatch _vq_auxt__16u2_p8_1 = {
	_vq_quantthresh__16u2_p8_1,
	_vq_quantmap__16u2_p8_1,
	21,
	21
};

static static_codebook _16u2_p8_1 = {
	2, 441,
	_vq_lengthlist__16u2_p8_1,
	1, -529268736, 1611661312, 5, 0,
	_vq_quantlist__16u2_p8_1,
	NULL,
	&_vq_auxt__16u2_p8_1,
	NULL,
	0
};

static long _vq_quantlist__16u2_p9_0[] = {
	5586,
	4655,
	6517,
	3724,
	7448,
	2793,
	8379,
	1862,
	9310,
	931,
	10241,
	0,
	11172,
	5521,
	5651,
};

static long _vq_lengthlist__16u2_p9_0[] = {
	 1,10,10,10,10,10,10,10,10,10,10,10,10, 5, 4,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10, 4,10,10,10,10,10,10,10,10,10,10,10,10,
	 6, 6, 5,10,10,10,10, 9, 9, 9, 9, 9, 9, 9, 9, 5,
	 5,
};

static float _vq_quantthresh__16u2_p9_0[] = {
	-5120.5, -4189.5, -3258.5, -2327.5, -1396.5, -498, -32.5, 32.5, 
	498, 1396.5, 2327.5, 3258.5, 4189.5, 5120.5, 
};

static long _vq_quantmap__16u2_p9_0[] = {
	   11,    9,    7,    5,    3,    1,   13,    0,
	   14,    2,    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__16u2_p9_0 = {
	_vq_quantthresh__16u2_p9_0,
	_vq_quantmap__16u2_p9_0,
	15,
	15
};

static static_codebook _16u2_p9_0 = {
	2, 225,
	_vq_lengthlist__16u2_p9_0,
	1, -510275072, 1611661312, 14, 0,
	_vq_quantlist__16u2_p9_0,
	NULL,
	&_vq_auxt__16u2_p9_0,
	NULL,
	0
};

static long _vq_quantlist__16u2_p9_1[] = {
	392,
	343,
	441,
	294,
	490,
	245,
	539,
	196,
	588,
	147,
	637,
	98,
	686,
	49,
	735,
	0,
	784,
	388,
	396,
};

static long _vq_lengthlist__16u2_p9_1[] = {
	 1,12,10,12,10,12,10,12,11,12,12,12,12,12,12,12,
	12, 5, 5, 9,10,12,11,11,12,12,12,12,12,12,12,12,
	12,12,12,12,10, 9, 9,11, 9,11,11,12,11,12,12,12,
	12,12,12,12,12,12,12, 8, 8,10,11, 9,12,11,12,12,
	12,12,12,12,12,12,12,12,12,12, 9, 8,10,11,12,11,
	12,11,12,12,12,12,12,12,12,12,12,12,12, 8, 9,11,
	11,10,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	 9,10,11,12,11,12,11,12,12,12,12,12,12,12,12,12,
	12,12,12, 9, 9,11,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11, 5, 8, 9, 9, 8,11, 9,11,11,11,11,11,11,
	11,11,11,11, 5, 5, 4, 8, 8, 8, 8,10, 9,10,10,11,
	11,11,11,11,11,11,11, 5, 4,
};

static float _vq_quantthresh__16u2_p9_1[] = {
	-367.5, -318.5, -269.5, -220.5, -171.5, -122.5, -73.5, -26.5, 
	-2, 2, 26.5, 73.5, 122.5, 171.5, 220.5, 269.5, 
	318.5, 367.5, 
};

static long _vq_quantmap__16u2_p9_1[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	   17,    0,   18,    2,    4,    6,    8,   10,
	   12,   14,   16,
};

static encode_aux_threshmatch _vq_auxt__16u2_p9_1 = {
	_vq_quantthresh__16u2_p9_1,
	_vq_quantmap__16u2_p9_1,
	19,
	19
};

static static_codebook _16u2_p9_1 = {
	2, 361,
	_vq_lengthlist__16u2_p9_1,
	1, -518488064, 1611661312, 10, 0,
	_vq_quantlist__16u2_p9_1,
	NULL,
	&_vq_auxt__16u2_p9_1,
	NULL,
	0
};

static long _vq_quantlist__16u2_p9_2[] = {
	24,
	23,
	25,
	22,
	26,
	21,
	27,
	20,
	28,
	19,
	29,
	18,
	30,
	17,
	31,
	16,
	32,
	15,
	33,
	14,
	34,
	13,
	35,
	12,
	36,
	11,
	37,
	10,
	38,
	9,
	39,
	8,
	40,
	7,
	41,
	6,
	42,
	5,
	43,
	4,
	44,
	3,
	45,
	2,
	46,
	1,
	47,
	0,
	48,
};

static long _vq_lengthlist__16u2_p9_2[] = {
	 1, 3, 3, 4, 7, 7, 7, 8, 7, 7, 7, 7, 8, 8, 8, 8,
	 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 9, 9, 8, 9, 9,
	 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,12,12,10,
	11,
};

static float _vq_quantthresh__16u2_p9_2[] = {
	-23.5, -22.5, -21.5, -20.5, -19.5, -18.5, -17.5, -16.5, 
	-15.5, -14.5, -13.5, -12.5, -11.5, -10.5, -9.5, -8.5, 
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
	8.5, 9.5, 10.5, 11.5, 12.5, 13.5, 14.5, 15.5, 
	16.5, 17.5, 18.5, 19.5, 20.5, 21.5, 22.5, 23.5, 
};

static long _vq_quantmap__16u2_p9_2[] = {
	   47,   45,   43,   41,   39,   37,   35,   33,
	   31,   29,   27,   25,   23,   21,   19,   17,
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,   18,   20,   22,   24,   26,   28,   30,
	   32,   34,   36,   38,   40,   42,   44,   46,
	   48,
};

static encode_aux_threshmatch _vq_auxt__16u2_p9_2 = {
	_vq_quantthresh__16u2_p9_2,
	_vq_quantmap__16u2_p9_2,
	49,
	49
};

static static_codebook _16u2_p9_2 = {
	1, 49,
	_vq_lengthlist__16u2_p9_2,
	1, -526909440, 1611661312, 6, 0,
	_vq_quantlist__16u2_p9_2,
	NULL,
	&_vq_auxt__16u2_p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44u0__long[] = {
	 3, 7,12,10,15,10, 9,18, 5, 2, 5, 6, 8, 7, 9,20,
	10, 4, 4, 6, 6, 8,11,20, 9, 5, 6, 5, 7, 6, 9,20,
	11, 7, 5, 7, 5, 7,10,20,10, 6, 7, 6, 6, 6, 8,17,
	 9, 8,10, 7, 7, 5, 5,17,11,18,19,14,13, 9, 8,19,
};

static static_codebook _huff_book__44u0__long = {
	2, 64,
	_huff_lengthlist__44u0__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u0__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u0__p1_0[] = {
	 1, 4, 4, 5, 8, 7, 5, 7, 8, 5, 8, 8, 8,11,11, 8,
	10,11, 5, 8, 8, 8,10,10, 8,11,11, 4, 8, 8, 8,11,
	11, 8,11,11, 8,11,11,11,13,14,11,13,14, 7,11,11,
	10,14,12,11,13,14, 4, 8, 8, 8,11,11, 8,11,11, 8,
	11,11,11,14,13,10,12,13, 8,11,12,11,14,14,11,14,
	13,
};

static float _vq_quantthresh__44u0__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u0__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u0__p1_0 = {
	_vq_quantthresh__44u0__p1_0,
	_vq_quantmap__44u0__p1_0,
	3,
	3
};

static static_codebook _44u0__p1_0 = {
	4, 81,
	_vq_lengthlist__44u0__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u0__p1_0,
	NULL,
	&_vq_auxt__44u0__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u0__p2_0[] = {
	 2, 4, 4, 5, 6, 6, 5, 6, 6, 5, 7, 6, 7, 8, 8, 6,
	 7, 8, 5, 6, 7, 6, 8, 7, 7, 8, 8, 5, 7, 7, 7, 8,
	 8, 7, 8, 8, 7, 8, 8, 8, 9,10, 8,10,10, 6, 8, 8,
	 7,10, 8, 8,10,10, 5, 7, 7, 7, 8, 8, 7, 8, 8, 6,
	 8, 8, 8,10,10, 7, 8,10, 6, 8, 8, 8,10,10, 8,10,
	 9,
};

static float _vq_quantthresh__44u0__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u0__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u0__p2_0 = {
	_vq_quantthresh__44u0__p2_0,
	_vq_quantmap__44u0__p2_0,
	3,
	3
};

static static_codebook _44u0__p2_0 = {
	4, 81,
	_vq_lengthlist__44u0__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u0__p2_0,
	NULL,
	&_vq_auxt__44u0__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u0__p3_0[] = {
	 1, 5, 5, 8, 8, 5, 7, 7, 9, 9, 5, 7, 8, 9, 9, 8,
	10, 9,12,11, 8, 9,10,11,12, 6, 8, 8,10,10, 8,10,
	10,11,11, 8, 9,10,11,11,10,12,11,13,13,10,11,11,
	13,13, 6, 8, 8,10,10, 8,10, 9,11,11, 8,10,10,11,
	11,10,11,11,13,13,10,11,11,13,13, 9,11,11,15,14,
	10,12,11,15,14,10,12,11,15,14,12,14,13,16,15,12,
	13,13,17,15, 9,11,11,14,14,10,11,12,14,15,10,11,
	12,14,17,12,13,14,15,16,13,13,14,15,17, 5, 8, 8,
	11,11, 8,10,10,12,12, 8,10,10,12,12,11,12,12,15,
	14,11,12,12,14,15, 8,10,10,13,12,10,12,12,13,13,
	10,12,12,14,14,12,13,13,15,15,11,13,13,15,16, 7,
	10,10,12,12, 9,12,11,14,13,10,12,12,13,14,12,13,
	12,15,15,12,13,13,16,17,10,12,12,15,16,12,13,13,
	17,15,11,13,13,17,16,15,15,15,17,17,13,15,16,18,
	18, 9,12,12,15,17,11,13,12,16,16,11,13,13,16,18,
	14,15,14,16,16,13,15,15,17,18, 5, 8, 8,11,11, 8,
	10,10,12,12, 8,10,10,12,12,10,12,12,15,15,11,12,
	12,14,15, 7,10,10,12,12,10,12,12,14,14, 9,11,12,
	13,13,12,13,13,16,16,12,12,13,15,15, 8,10,10,12,
	13,10,12,12,13,14,10,12,12,13,13,12,13,13,16,16,
	12,13,13,15,15,10,12,12,15,16,11,13,13,17,16,11,
	12,13,17,16,13,15,15,18,19,14,14,14,17,16,10,12,
	12,15,15,11,13,13,15,16,11,13,13,15,17,13,15,15,
	18,20,14,15,15,17,17, 8,11,11,15,16,10,13,12,17,
	16,10,12,13,16,17,15,16,15,20,19,14,15,16,18,19,
	 9,12,12,15,17,11,13,14,18,18,11,13,14,17,17,16,
	17,18,19,19,15,16,18,19,20, 9,12,12,16,16,11,14,
	12,17,17,11,13,13,16,19,15,16,15,20,19,15,15,15,
	19,17,12,14,14,18,19,14,15,15,19,18,13,15,15,18,
	17,17,18,19,20, 0,15,16,17,20, 0,12,15,14,17,18,
	13,16,14,20,20,13,16,15,19,18,16,20,17,20,19,16,
	17,16, 0,19, 8,11,11,15,15,10,12,12,17,17,10,12,
	13,16,16,13,14,15,17,17,15,16,17,19,19, 9,12,12,
	16,17,11,13,13,18,16,11,13,13,16,17,14,15,16,19,
	18,15,16,17,17,18, 9,12,12,17,16,11,14,13,16,16,
	12,14,13,17,18,14,16,15,18,19,17,17,17,19,18,12,
	14,14,19,17,13,15,15,17, 0,13,14,16, 0, 0,15,16,
	16,19, 0,16,16,19,19, 0,12,15,14,19,18,13,15,15,
	19, 0,14,16,15, 0,19,15,17,17, 0,19,17,18,17, 0,
	19,
};

static float _vq_quantthresh__44u0__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u0__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u0__p3_0 = {
	_vq_quantthresh__44u0__p3_0,
	_vq_quantmap__44u0__p3_0,
	5,
	5
};

static static_codebook _44u0__p3_0 = {
	4, 625,
	_vq_lengthlist__44u0__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u0__p3_0,
	NULL,
	&_vq_auxt__44u0__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u0__p4_0[] = {
	 4, 5, 5, 8, 8, 5, 7, 6, 9, 9, 5, 6, 7, 9, 9, 9,
	 9, 9,11,11, 9, 9, 9,11,11, 5, 7, 7, 9, 9, 7, 8,
	 8,10,10, 6, 7, 8, 9,10,10,10,10,11,12, 9,10,10,
	11,12, 5, 7, 7, 9, 9, 6, 8, 7,10,10, 7, 8, 8,10,
	10, 9,10,10,12,11,10,10,10,12,11, 9,10,10,12,12,
	10,11,10,13,13, 9,10,10,13,13,12,12,12,14,14,11,
	12,12,14,14, 9,10,10,12,12, 9,10,10,13,13,10,10,
	10,12,13,11,12,12,14,14,12,13,12,14,14, 5, 7, 7,
	10,10, 7, 8, 8,10,10, 7, 8, 8,10,10,10,10,10,13,
	12,10,10,10,12,12, 7, 8, 8,11,10, 8, 8, 9,10,11,
	 8, 9, 9,11,11,10,10,11,12,13,10,11,11,13,14, 6,
	 8, 8,10,10, 7, 9, 8,11,10, 8, 9, 9,11,11,10,11,
	10,13,11,10,11,11,13,13,10,11,10,13,13,10,10,11,
	13,14,10,11,11,14,13,12,11,13,12,14,12,13,13,15,
	15, 9,10,10,13,13,10,11,10,13,13,10,11,11,13,14,
	12,13,11,15,12,12,13,13,14,15, 5, 7, 7,10, 9, 7,
	 8, 8,10,10, 7, 8, 8,10,10,10,10,10,12,12,10,10,
	10,12,13, 6, 8, 8,10,10, 8, 9, 9,11,11, 7, 8, 9,
	10,11,10,11,11,13,13,10,10,11,11,14, 7, 8, 8,10,
	10, 8, 9, 9,11,11, 8, 9, 8,11,10,10,11,11,13,13,
	10,11,10,13,11, 9,10,10,13,13,10,11,11,14,13,10,
	10,11,12,13,13,13,13,15,14,12,11,13,12,15, 9,10,
	11,13,13,10,11,11,13,14,10,11,10,13,13,12,13,13,
	15,15,12,13,11,15,12, 8,10,10,13,12,10,11,11,13,
	13, 9,10,11,13,13,13,13,13,15,15,12,13,13,15,15,
	 9,10,10,13,13,10,11,11,13,14,10,11,11,14,13,13,
	13,14,14,15,13,13,13,14,15, 9,10,10,13,13,10,11,
	10,14,13,10,11,11,13,14,13,14,13,15,14,12,13,13,
	14,15,11,13,13,15,14,11,11,13,14,15,12,14,13,15,
	15,13,12,15,12,16,14,14,15,17,16,11,12,12,14,15,
	11,13,11,15,14,12,13,13,15,15,14,14,12,17,13,14,
	15,15,18,16, 8,10,10,13,12, 9,10,10,13,13,10,10,
	11,13,13,12,13,13,14,14,12,13,13,15,15, 9,10,10,
	13,13,10,11,11,14,13,10,10,11,13,14,12,13,13,17,
	15,12,12,13,14,16, 9,10,10,13,13,10,11,11,13,13,
	10,11,10,14,13,13,13,13,14,15,13,14,13,15,15,11,
	13,12,14,14,12,13,13,16,14,11,12,13,15,15,14,15,
	16,17,18,14,12,15,14,16,11,12,13,14,15,12,13,13,
	15,16,11,13,11,15,14,14,16,14,16,17,14,15,12,16,
	12,
};

static float _vq_quantthresh__44u0__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u0__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u0__p4_0 = {
	_vq_quantthresh__44u0__p4_0,
	_vq_quantmap__44u0__p4_0,
	5,
	5
};

static static_codebook _44u0__p4_0 = {
	4, 625,
	_vq_lengthlist__44u0__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u0__p4_0,
	NULL,
	&_vq_auxt__44u0__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u0__p5_0[] = {
	 1, 4, 4, 7, 7, 8, 8, 9, 9, 4, 6, 5, 8, 8, 8, 8,
	10,10, 4, 5, 6, 8, 8, 8, 8,10,10, 7, 8, 8, 9, 9,
	 9, 9,11,11, 7, 8, 8, 9, 9, 9, 9,11,11, 8, 8, 8,
	10, 9,10,11,12,12, 8, 8, 8, 9, 9,11,11,12,12, 9,
	10,10,11,11,12,12,13,13, 9,10,10,11,11,12,12,13,
	13,
};

static float _vq_quantthresh__44u0__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u0__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u0__p5_0 = {
	_vq_quantthresh__44u0__p5_0,
	_vq_quantmap__44u0__p5_0,
	9,
	9
};

static static_codebook _44u0__p5_0 = {
	2, 81,
	_vq_lengthlist__44u0__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u0__p5_0,
	NULL,
	&_vq_auxt__44u0__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u0__p6_0[] = {
	 1, 4, 4, 6, 6, 8, 8,10, 9,10,10,14,14, 4, 6, 5,
	 8, 8, 9, 9,10,10,11,11,14,14, 4, 5, 6, 8, 8, 9,
	 9,10,10,11,11,14,14, 7, 8, 8, 9, 9,10,10,11,11,
	12,12,15,15, 7, 8, 8, 9, 9,10,10,11,11,12,12,14,
	15, 9, 9, 9,10,10,11,11,12,11,12,12,15,16, 9, 9,
	 9,10,10,11,11,11,11,12,12,15,15,10,10,10,11,11,
	12,11,12,12,13,13,16,16,10,10,10,11,11,12,12,12,
	12,13,13,16,17,11,11,12,12,12,13,13,14,13,14,14,
	18,17,11,11,11,12,12,12,12,13,14,14,14,18,18,14,
	14,14,15,15,15,16,16,16,17,17, 0,19,14,14,14,15,
	15,16,17,16,17,17,17,19, 0,
};

static float _vq_quantthresh__44u0__p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44u0__p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u0__p6_0 = {
	_vq_quantthresh__44u0__p6_0,
	_vq_quantmap__44u0__p6_0,
	13,
	13
};

static static_codebook _44u0__p6_0 = {
	2, 169,
	_vq_lengthlist__44u0__p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44u0__p6_0,
	NULL,
	&_vq_auxt__44u0__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u0__p6_1[] = {
	 2, 4, 4, 5, 5, 4, 5, 5, 5, 5, 4, 5, 5, 5, 5, 5,
	 6, 6, 6, 6, 5, 6, 6, 6, 6,
};

static float _vq_quantthresh__44u0__p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u0__p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u0__p6_1 = {
	_vq_quantthresh__44u0__p6_1,
	_vq_quantmap__44u0__p6_1,
	5,
	5
};

static static_codebook _44u0__p6_1 = {
	2, 25,
	_vq_lengthlist__44u0__p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u0__p6_1,
	NULL,
	&_vq_auxt__44u0__p6_1,
	NULL,
	0
};

static long _vq_quantlist__44u0__p7_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u0__p7_0[] = {
	 1, 5, 5,11,11, 9,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11, 8,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11, 9,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11, 8,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44u0__p7_0[] = {
	-253.5, -84.5, 84.5, 253.5, 
};

static long _vq_quantmap__44u0__p7_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u0__p7_0 = {
	_vq_quantthresh__44u0__p7_0,
	_vq_quantmap__44u0__p7_0,
	5,
	5
};

static static_codebook _44u0__p7_0 = {
	4, 625,
	_vq_lengthlist__44u0__p7_0,
	1, -518709248, 1626677248, 3, 0,
	_vq_quantlist__44u0__p7_0,
	NULL,
	&_vq_auxt__44u0__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u0__p7_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u0__p7_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 8, 8, 9, 9,10,10, 4, 7, 7,
	 8, 8, 8, 8, 9, 9,11,10,12,11, 4, 6, 7, 8, 8, 7,
	 8, 9, 9,10,11,12,11, 7, 8, 8, 9, 9,10,10,11,11,
	12,11,13,13, 7, 8, 8, 9,10,10,10,11,11,12,12,12,
	13, 7, 8, 8,10,10,12,12,13,12,16,13,13,14, 7, 8,
	 8,10,11,12,12,13,12,13,13,14,14, 8, 9,10,12,12,
	14,13,16,15,16,16,16,15, 8,10,10,12,12,14,13,14,
	15,16,16,15,16,10,11,12,14,14,14,14,16,13,16,15,
	16,16,10,11,12,13,13,15,14,14,15,16,16,14,14,13,
	14,13,15,16,16,16,15,15,16,16,16,16,11,15,14,16,
	16,14,14,16,15,16,14,16,15,
};

static float _vq_quantthresh__44u0__p7_1[] = {
	-71.5, -58.5, -45.5, -32.5, -19.5, -6.5, 6.5, 19.5, 
	32.5, 45.5, 58.5, 71.5, 
};

static long _vq_quantmap__44u0__p7_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u0__p7_1 = {
	_vq_quantthresh__44u0__p7_1,
	_vq_quantmap__44u0__p7_1,
	13,
	13
};

static static_codebook _44u0__p7_1 = {
	2, 169,
	_vq_lengthlist__44u0__p7_1,
	1, -523010048, 1618608128, 4, 0,
	_vq_quantlist__44u0__p7_1,
	NULL,
	&_vq_auxt__44u0__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u0__p7_2[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u0__p7_2[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 5, 5, 6,
	 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 5, 5, 5, 7, 7, 8,
	 8, 8, 8, 9, 9, 9, 9, 6, 7, 7, 7, 8, 8, 8, 9, 9,
	 9, 9, 9, 9, 6, 7, 7, 8, 7, 8, 8, 9, 9, 9, 9, 9,
	 9, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 8,
	 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 9, 9, 9, 9, 8, 9,
	 9, 9, 9, 9, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44u0__p7_2[] = {
	-5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 
	2.5, 3.5, 4.5, 5.5, 
};

static long _vq_quantmap__44u0__p7_2[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u0__p7_2 = {
	_vq_quantthresh__44u0__p7_2,
	_vq_quantmap__44u0__p7_2,
	13,
	13
};

static static_codebook _44u0__p7_2 = {
	2, 169,
	_vq_lengthlist__44u0__p7_2,
	1, -531103744, 1611661312, 4, 0,
	_vq_quantlist__44u0__p7_2,
	NULL,
	&_vq_auxt__44u0__p7_2,
	NULL,
	0
};

static long _huff_lengthlist__44u0__short[] = {
	 6,10,11,12,13,12,12,12, 4, 6, 6, 9, 9,10,11,13,
	 3, 4, 3, 7, 6, 8,10,15, 5, 7, 7, 9, 8, 9,11,16,
	 6, 7, 5, 8, 5, 7,10,16, 7, 8, 7, 9, 6, 7,10,16,
	10, 6, 3, 5, 4, 5, 7,16,13, 8, 5, 7, 6, 7,10,15,
};

static static_codebook _huff_book__44u0__short = {
	2, 64,
	_huff_lengthlist__44u0__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u1__long[] = {
	 4, 8,13,11,14,11,10,14, 6, 3, 5, 6, 8, 7, 9,14,
	12, 4, 3, 6, 5, 7,10,18,11, 6, 6, 6, 6, 6, 8,16,
	13, 7, 4, 6, 4, 6, 9,19,11, 6, 6, 5, 5, 5, 7,16,
	10, 8, 9, 7, 9, 6, 4,12,11,16,18,19,18,10, 7,11,
};

static static_codebook _huff_book__44u1__long = {
	2, 64,
	_huff_lengthlist__44u1__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u1__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u1__p1_0[] = {
	 1, 4, 4, 5, 8, 7, 5, 7, 8, 5, 8, 8, 8,11,11, 8,
	10,11, 5, 8, 8, 8,11,10, 8,11,11, 4, 8, 8, 8,11,
	11, 8,11,11, 8,11,11,11,13,14,11,13,14, 8,11,11,
	10,14,11,11,13,14, 4, 8, 8, 8,11,11, 8,11,11, 7,
	11,11,11,14,13,10,12,13, 8,11,11,11,14,14,11,14,
	13,
};

static float _vq_quantthresh__44u1__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u1__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u1__p1_0 = {
	_vq_quantthresh__44u1__p1_0,
	_vq_quantmap__44u1__p1_0,
	3,
	3
};

static static_codebook _44u1__p1_0 = {
	4, 81,
	_vq_lengthlist__44u1__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u1__p1_0,
	NULL,
	&_vq_auxt__44u1__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u1__p2_0[] = {
	 2, 5, 5, 5, 6, 6, 5, 6, 6, 5, 6, 6, 7, 8, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 7, 8, 8, 5, 6, 6, 7, 8,
	 8, 6, 8, 8, 7, 8, 8, 8, 9,10, 8, 9, 9, 6, 8, 8,
	 7, 9, 8, 8, 9,10, 5, 6, 6, 6, 8, 8, 7, 8, 8, 6,
	 8, 8, 8,10, 9, 7, 8, 9, 7, 8, 8, 8, 9, 9, 8,10,
	 9,
};

static float _vq_quantthresh__44u1__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u1__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u1__p2_0 = {
	_vq_quantthresh__44u1__p2_0,
	_vq_quantmap__44u1__p2_0,
	3,
	3
};

static static_codebook _44u1__p2_0 = {
	4, 81,
	_vq_lengthlist__44u1__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u1__p2_0,
	NULL,
	&_vq_auxt__44u1__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u1__p3_0[] = {
	 2, 5, 4, 7, 7, 5, 7, 7, 9, 8, 5, 7, 7, 8, 9, 8,
	 9, 9,11,11, 8, 9, 9,11,11, 5, 7, 7,10,10, 7, 9,
	 9,11,10, 7, 9, 9,10,10, 9,11,10,13,12, 9,10,10,
	12,13, 5, 7, 7,10, 9, 7, 9, 9,11,10, 7, 9, 9,10,
	11, 9,10,10,12,12,10,10,11,12,13, 8,10,10,14,13,
	 9,11,11,15,13, 9,11,11,15,13,12,14,12,16,14,12,
	13,12,15,14, 8,10,10,13,14, 9,11,11,13,14,10,11,
	11,13,15,12,12,13,14,15,12,13,14,14,16, 5, 7, 7,
	10,10, 7, 9, 9,11,11, 7, 9, 9,11,12,10,11,11,14,
	14,10,11,11,14,14, 7, 9, 9,12,12, 9,11,11,13,12,
	 9,11,11,13,13,12,12,12,14,14,11,12,13,15,15, 7,
	 9, 9,12,11, 9,11,10,12,12, 9,11,11,12,13,11,12,
	11,14,14,11,12,12,15,16, 9,11,11,15,15,11,12,12,
	16,15,10,12,12,16,15,14,15,15,17,16,13,14,14,17,
	17, 9,11,11,14,15,10,12,11,15,15,10,12,12,15,17,
	13,14,13,16,15,13,14,15,17,17, 5, 7, 7,10,10, 7,
	 9, 9,12,11, 7, 9, 9,11,12,10,11,11,14,14,10,11,
	11,13,14, 7, 9, 9,11,12, 9,11,11,13,13, 9,10,11,
	12,12,11,12,12,15,15,11,12,12,13,14, 7,10, 9,12,
	12, 9,11,11,13,13, 9,11,11,12,12,11,12,12,16,15,
	11,12,12,14,14, 9,11,11,15,14,10,12,12,16,15,10,
	11,12,15,15,13,14,14,17,18,13,13,14,15,16, 9,11,
	11,15,16,10,12,12,15,15,11,12,12,14,17,13,14,14,
	17,17,14,14,14,16,18, 7,10,10,14,15,10,12,12,16,
	15,10,11,12,16,16,14,16,15,18,17,13,15,14,17,19,
	 9,11,12,16,15,11,13,13,17,16,10,13,13,16,16,15,
	16,16,17,19,13,15,15,17,17, 8,11,11,15,15,10,13,
	11,16,16,10,13,13,16,17,14,16,15,18,19,13,15,15,
	17,17,12,14,14,18,18,13,14,15,18,19,12,14,15,17,
	18,16,18,18,19, 0,14,15,16,18,18,11,14,13,17,18,
	12,16,14,19,19,12,15,14, 0,18,15,17,16,18,17,14,
	17,16,18,18, 7,10,10,15,14,10,12,11,16,16,10,11,
	12,16,16,13,15,14,19,18,14,15,16,18,18, 8,11,11,
	15,15,10,13,12,17,16,10,12,13,16,17,14,14,15,19,
	18,14,15,16,18, 0, 9,11,11,16,15,11,13,12,15,16,
	11,13,13,16,16,14,15,14, 0,17,15,16,16,19,18,11,
	14,14,18,17,12,14,15,17,17,12,13,15, 0,17,14,15,
	16,18,17,16,17,18,17,19,11,14,13,17,19,12,15,14,
	18,19,13,16,14, 0,19,14,17,15, 0,18,15,18,16, 0,
	 0,
};

static float _vq_quantthresh__44u1__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u1__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u1__p3_0 = {
	_vq_quantthresh__44u1__p3_0,
	_vq_quantmap__44u1__p3_0,
	5,
	5
};

static static_codebook _44u1__p3_0 = {
	4, 625,
	_vq_lengthlist__44u1__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u1__p3_0,
	NULL,
	&_vq_auxt__44u1__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u1__p4_0[] = {
	 4, 5, 5, 8, 8, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 9,
	 9, 9,11,11, 9, 9, 9,11,11, 6, 7, 7, 9, 9, 7, 8,
	 8,10,10, 6, 7, 8, 9,10,10,10,10,11,12, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 6, 8, 7,10, 9, 7, 8, 8,10,
	10, 9,10, 9,12,11,10,10,10,12,11, 9,10,10,12,12,
	10,10,10,13,12, 9,10,10,12,12,12,12,12,14,14,11,
	12,12,13,14, 9,10,10,12,12, 9,10,10,13,12,10,10,
	10,12,13,12,12,12,14,13,12,12,12,14,14, 5, 7, 7,
	 9, 9, 7, 8, 7,10,10, 7, 7, 8,10,10,10,10,10,12,
	12,10,10,10,12,12, 7, 8, 8,10,10, 8, 8, 9,10,11,
	 8, 8, 8,11,11,10,10,11,11,13,10,11,11,12,13, 6,
	 7, 7,10,10, 7, 8, 8,11,10, 8, 8, 9,11,11,10,11,
	10,13,11,10,11,11,13,12,10,11,10,13,12,10,10,11,
	13,13,10,11,11,13,13,12,11,13,12,14,12,13,13,15,
	15, 9,10,10,12,13,10,11,10,13,13,10,11,11,13,14,
	12,13,11,14,12,12,13,13,14,15, 5, 7, 7, 9,10, 7,
	 8, 7,10,10, 7, 7, 8,10,10,10,10,10,12,12,10,10,
	10,12,12, 6, 7, 7,10,10, 8, 9, 8,11,11, 7, 8, 8,
	10,11,10,11,11,12,13,10,10,11,11,13, 7, 8, 8,10,
	10, 8, 8, 8,11,11, 8, 9, 8,11,10,10,11,10,13,12,
	10,11,10,13,12, 9,10,10,13,12,10,11,11,13,13, 9,
	10,10,12,13,13,13,13,15,14,12,11,13,12,15,10,10,
	11,12,13,10,11,11,13,13,10,11,10,13,13,12,13,13,
	15,15,12,13,11,14,12, 8,10, 9,12,12, 9,10,10,13,
	13, 9,10,10,13,13,13,13,13,14,15,12,12,12,14,14,
	 9,10,10,13,12,10,11,11,13,13,10,11,11,13,12,13,
	13,14,14,16,12,13,13,15,14, 9,10,10,13,13,10,11,
	10,13,13,10,11,11,13,13,13,14,12,15,14,12,13,13,
	14,15,12,12,12,14,14,11,12,12,14,15,12,13,13,15,
	14,14,12,15,12,16,14,14,15,17,16,11,12,12,14,14,
	11,12,11,15,14,12,13,13,15,15,13,14,12,16,13,14,
	14,15,17,16, 8,10,10,12,12, 9,10,10,13,12,10,10,
	10,13,13,12,13,12,14,14,12,13,13,15,14, 9,10,10,
	13,13,10,11,11,13,13,10,10,11,12,13,13,13,13,15,
	15,12,12,13,14,15, 9,10,10,12,13,10,11,11,12,13,
	10,11,10,13,13,12,13,13,14,15,13,14,13,15,14,11,
	12,12,15,14,12,13,13,15,14,11,12,12,14,15,14,14,
	14,17,15,13,12,15,13,16,12,12,12,14,15,12,13,13,
	14,15,11,12,12,15,14,14,15,14,16,17,13,15,12,16,
	12,
};

static float _vq_quantthresh__44u1__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u1__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u1__p4_0 = {
	_vq_quantthresh__44u1__p4_0,
	_vq_quantmap__44u1__p4_0,
	5,
	5
};

static static_codebook _44u1__p4_0 = {
	4, 625,
	_vq_lengthlist__44u1__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u1__p4_0,
	NULL,
	&_vq_auxt__44u1__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u1__p5_0[] = {
	 1, 4, 4, 7, 7, 8, 8,10,10, 4, 6, 5, 8, 8, 8, 8,
	10,10, 4, 5, 6, 8, 8, 8, 8,10,10, 7, 8, 7, 9, 9,
	 9, 9,11,11, 7, 8, 8, 9, 9, 9, 9,11,11, 8, 8, 8,
	 9, 9,10,11,12,12, 8, 8, 9, 9, 9,10,10,12,12,10,
	10,10,11,11,12,12,13,13,10,10,10,11,11,12,12,13,
	13,
};

static float _vq_quantthresh__44u1__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u1__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u1__p5_0 = {
	_vq_quantthresh__44u1__p5_0,
	_vq_quantmap__44u1__p5_0,
	9,
	9
};

static static_codebook _44u1__p5_0 = {
	2, 81,
	_vq_lengthlist__44u1__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u1__p5_0,
	NULL,
	&_vq_auxt__44u1__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u1__p6_0[] = {
	 1, 4, 4, 7, 7, 9, 8,10, 9,11,10,14,13, 4, 5, 5,
	 8, 8, 9, 9,11,11,11,11,14,14, 4, 5, 5, 8, 8, 9,
	 9,10,11,11,11,14,14, 7, 8, 8, 9, 9,11,10,11,11,
	12,12,16,15, 7, 8, 8, 9, 9,10,11,11,11,12,12,15,
	15, 9,10,10,11,11,11,11,12,12,13,13,16,16, 9,10,
	10,11,11,11,11,12,12,12,13,16,15,10,11,11,11,11,
	12,12,13,13,13,13,16,17,10,11,11,11,11,12,12,12,
	12,13,14,16,16,11,12,12,12,12,13,13,14,14,14,15,
	18,17,11,12,12,12,12,13,13,13,14,14,15,18,18,15,
	14,15,15,15,16,16,17,17,18,17,20,20,14,15,15,15,
	15,16,17,17,16,18, 0,19, 0,
};

static float _vq_quantthresh__44u1__p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44u1__p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u1__p6_0 = {
	_vq_quantthresh__44u1__p6_0,
	_vq_quantmap__44u1__p6_0,
	13,
	13
};

static static_codebook _44u1__p6_0 = {
	2, 169,
	_vq_lengthlist__44u1__p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44u1__p6_0,
	NULL,
	&_vq_auxt__44u1__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u1__p6_1[] = {
	 2, 4, 4, 5, 5, 4, 5, 5, 6, 5, 4, 5, 5, 5, 6, 5,
	 6, 5, 6, 6, 5, 5, 6, 6, 6,
};

static float _vq_quantthresh__44u1__p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u1__p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u1__p6_1 = {
	_vq_quantthresh__44u1__p6_1,
	_vq_quantmap__44u1__p6_1,
	5,
	5
};

static static_codebook _44u1__p6_1 = {
	2, 25,
	_vq_lengthlist__44u1__p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u1__p6_1,
	NULL,
	&_vq_auxt__44u1__p6_1,
	NULL,
	0
};

static long _vq_quantlist__44u1__p7_0[] = {
	3,
	2,
	4,
	1,
	5,
	0,
	6,
};

static long _vq_lengthlist__44u1__p7_0[] = {
	 1, 3, 3, 9, 9, 9, 9, 5, 7, 7, 9, 9, 9, 9, 5, 6,
	 6, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8,
};

static float _vq_quantthresh__44u1__p7_0[] = {
	-422.5, -253.5, -84.5, 84.5, 253.5, 422.5, 
};

static long _vq_quantmap__44u1__p7_0[] = {
	    5,    3,    1,    0,    2,    4,    6,
};

static encode_aux_threshmatch _vq_auxt__44u1__p7_0 = {
	_vq_quantthresh__44u1__p7_0,
	_vq_quantmap__44u1__p7_0,
	7,
	7
};

static static_codebook _44u1__p7_0 = {
	2, 49,
	_vq_lengthlist__44u1__p7_0,
	1, -518017024, 1626677248, 3, 0,
	_vq_quantlist__44u1__p7_0,
	NULL,
	&_vq_auxt__44u1__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u1__p7_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u1__p7_1[] = {
	 1, 4, 4, 6, 6, 6, 7, 8, 8, 9, 9,11,10, 4, 7, 7,
	 8, 8, 8, 8, 9, 9,11,10,12,11, 4, 6, 7, 8, 8, 8,
	 8,10,10,10,11,12,11, 7, 8, 8, 9, 9,10,10,11,11,
	12,12,13,13, 7, 8, 8, 9, 9,10,10,11,11,12,12,13,
	13, 7, 8, 8,10,10,13,12,13,13,16,13,13,14, 7, 8,
	 8,10,11,12,12,13,13,14,14,14,14, 8,10,10,13,12,
	13,13,16,14,15,15,16,15, 8,10,11,12,13,13,13,13,
	16,15,16,16,14,10,12,13,14,15,15,14,14,14,15,16,
	15,16,10,11,12,13,13,15,15,16,16,16,14,16,16,12,
	13,13,15,14,15,14,15,15,14,15,16,16,11,14,14,15,
	14,16,15,16,14,15,16,14,15,
};

static float _vq_quantthresh__44u1__p7_1[] = {
	-71.5, -58.5, -45.5, -32.5, -19.5, -6.5, 6.5, 19.5, 
	32.5, 45.5, 58.5, 71.5, 
};

static long _vq_quantmap__44u1__p7_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u1__p7_1 = {
	_vq_quantthresh__44u1__p7_1,
	_vq_quantmap__44u1__p7_1,
	13,
	13
};

static static_codebook _44u1__p7_1 = {
	2, 169,
	_vq_lengthlist__44u1__p7_1,
	1, -523010048, 1618608128, 4, 0,
	_vq_quantlist__44u1__p7_1,
	NULL,
	&_vq_auxt__44u1__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u1__p7_2[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u1__p7_2[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 5, 6, 6,
	 6, 7, 8, 7, 8, 8, 8, 9, 9, 9, 5, 6, 6, 7, 7, 8,
	 8, 8, 8, 8, 8, 8, 9, 6, 7, 7, 7, 7, 8, 8, 8, 9,
	 9, 9, 9, 9, 6, 7, 7, 7, 7, 8, 8, 9, 9, 9, 9, 9,
	 9, 7, 8, 8, 8, 8, 8, 8, 9, 8, 9, 9, 9, 9, 7, 8,
	 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 8, 9, 9, 9,
	 8, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 9, 9, 8, 9,
	 9, 9, 9, 9, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44u1__p7_2[] = {
	-5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 
	2.5, 3.5, 4.5, 5.5, 
};

static long _vq_quantmap__44u1__p7_2[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u1__p7_2 = {
	_vq_quantthresh__44u1__p7_2,
	_vq_quantmap__44u1__p7_2,
	13,
	13
};

static static_codebook _44u1__p7_2 = {
	2, 169,
	_vq_lengthlist__44u1__p7_2,
	1, -531103744, 1611661312, 4, 0,
	_vq_quantlist__44u1__p7_2,
	NULL,
	&_vq_auxt__44u1__p7_2,
	NULL,
	0
};

static long _huff_lengthlist__44u1__short[] = {
	 7,12,12,14,16,13,12,15, 6, 9,10,13,11,11,12,12,
	 4, 5, 5, 8, 5, 7, 9,12, 6, 7, 8,10, 8, 9,11,16,
	 5, 5, 4, 7, 4, 5, 7,15, 6, 5, 5, 8, 5, 5, 6,15,
	 8, 7, 4, 7, 3, 4, 5,16,15,11, 6, 8, 5, 6, 8,16,
};

static static_codebook _huff_book__44u1__short = {
	2, 64,
	_huff_lengthlist__44u1__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u2__long[] = {
	 8,14,15,15,17,15,12,13,13, 3, 4, 7, 8, 7, 8,11,
	20, 4, 3, 6, 5, 7, 9,16,15, 6, 5, 6, 6, 6, 8,15,
	20, 7, 4, 6, 4, 5, 8,18,16, 7, 6, 6, 5, 5, 6,14,
	11, 7, 8, 7, 7, 5, 4,10,10,13,15,16,19,10, 6,10,
};

static static_codebook _huff_book__44u2__long = {
	2, 64,
	_huff_lengthlist__44u2__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u2__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u2__p1_0[] = {
	 1, 4, 4, 5, 8, 7, 5, 7, 8, 5, 8, 8, 8,11,11, 8,
	10,11, 5, 8, 8, 8,11,10, 8,11,11, 4, 8, 8, 8,11,
	11, 8,11,11, 8,11,11,11,13,14,11,13,13, 8,11,11,
	10,13,12,11,13,14, 4, 8, 8, 8,11,11, 8,11,11, 7,
	11,11,11,14,13,10,12,13, 8,11,11,11,14,13,11,13,
	13,
};

static float _vq_quantthresh__44u2__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u2__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u2__p1_0 = {
	_vq_quantthresh__44u2__p1_0,
	_vq_quantmap__44u2__p1_0,
	3,
	3
};

static static_codebook _44u2__p1_0 = {
	4, 81,
	_vq_lengthlist__44u2__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u2__p1_0,
	NULL,
	&_vq_auxt__44u2__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u2__p2_0[] = {
	 3, 4, 4, 5, 6, 6, 5, 6, 6, 5, 6, 6, 6, 8, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 6, 8, 8, 5, 6, 6, 6, 8,
	 7, 6, 7, 8, 6, 8, 8, 8, 9, 9, 8, 9, 9, 6, 8, 8,
	 7, 9, 8, 8, 9, 9, 5, 6, 6, 6, 8, 8, 6, 7, 8, 6,
	 8, 8, 8, 9, 9, 7, 8, 9, 6, 8, 8, 8, 9, 9, 8, 9,
	 9,
};

static float _vq_quantthresh__44u2__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u2__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u2__p2_0 = {
	_vq_quantthresh__44u2__p2_0,
	_vq_quantmap__44u2__p2_0,
	3,
	3
};

static static_codebook _44u2__p2_0 = {
	4, 81,
	_vq_lengthlist__44u2__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u2__p2_0,
	NULL,
	&_vq_auxt__44u2__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u2__p3_0[] = {
	 2, 4, 4, 8, 8, 5, 7, 7, 9, 9, 5, 7, 7, 9, 9, 8,
	 9, 9,12,12, 8, 9,10,11,12, 5, 7, 7,10,10, 7, 9,
	 9,11,10, 7, 8, 9,10,11,10,11,10,14,13, 9,10,10,
	12,13, 5, 7, 7,10,10, 7, 9, 8,11,10, 7, 9, 9,11,
	11, 9,10,10,13,12,10,10,11,13,13, 8,10,10,15,13,
	10,11,11,15,13, 9,11,11,14,13,13,14,13,17,15,12,
	13,13,15,15, 8,10,10,13,14, 9,11,11,13,14,10,11,
	12,13,16,12,13,13,15,15,13,13,14,15,17, 5, 7, 7,
	10,10, 7, 9, 9,11,11, 7, 9, 9,11,12,10,11,11,14,
	14,10,11,12,14,14, 7, 9, 9,12,12, 9,11,11,13,12,
	 9,11,11,13,13,12,13,12,14,14,11,12,12,15,14, 7,
	 9, 9,12,11, 9,11,10,13,11, 9,11,11,12,13,11,12,
	11,14,13,11,12,12,15,15,10,12,12,16,15,11,13,13,
	16,16,10,12,12,16,16,14,14,14,17,16,13,14,14,17,
	18, 9,11,11,14,15,10,12,11,15,15,10,12,12,15,17,
	13,15,13,17,16,13,14,15,17,19, 5, 7, 7,10,10, 7,
	 9, 9,12,11, 7, 9, 9,11,11,10,11,11,15,14,10,11,
	12,13,14, 7, 9, 9,12,12, 9,11,11,13,12, 8,10,11,
	12,12,11,12,12,15,15,11,11,12,14,14, 7, 9, 9,12,
	12, 9,11,11,13,13, 9,11,11,12,12,11,12,12,16,15,
	11,12,13,14,14, 9,11,11,16,15,10,12,12,16,15,10,
	11,12,14,14,13,14,15,18,17,13,13,14,16,16,10,12,
	12,15,15,10,13,12,15,17,11,13,12,15,16,13,15,14,
	17,18,14,15,14,15,17, 8,10,10,15,15,10,12,12,17,
	15,10,12,12,17,16,14,16,15,17,17,13,14,15,16,16,
	 9,11,12,16,16,11,13,13,16,17,11,13,13,16,16,15,
	16,16, 0, 0,14,15,15,19,17, 9,11,11,17,16,10,13,
	11,16,16,11,12,12,16,17,15,16,15,19,19,14,16,15,
	 0,18,12,14,15, 0, 0,14,15,16,18,18,13,14,15,17,
	17,16,18,18, 0, 0,15,16,16,19,18,11,14,13, 0,18,
	13,16,13,17,17,13,15,14, 0, 0,17,17,15,19,18,14,
	16,16,19, 0, 8,10,10,15,15,10,12,11,16,16,10,11,
	12,17,16,14,15,15,17,18,15,15,16, 0, 0, 9,11,11,
	15,15,11,13,12,16,18,10,12,13,17,17,13,15,15,17,
	19,15,15,16,19,18, 9,12,11, 0,16,11,13,12,16,16,
	11,13,13,16,17,14,16,15,19,19,15,15,16,18,18,12,
	14,14,17,18,13,14,15,18, 0,13,14,15,18,19,15,17,
	16, 0,19,16,15,18,17,19,12,14,14,19,19,13,15,15,
	 0,18,13,17,15,18,18,15,16,17, 0,19,17,19,18,18,
	 0,
};

static float _vq_quantthresh__44u2__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u2__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u2__p3_0 = {
	_vq_quantthresh__44u2__p3_0,
	_vq_quantmap__44u2__p3_0,
	5,
	5
};

static static_codebook _44u2__p3_0 = {
	4, 625,
	_vq_lengthlist__44u2__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u2__p3_0,
	NULL,
	&_vq_auxt__44u2__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u2__p4_0[] = {
	 4, 5, 5, 9, 9, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 9,
	10, 9,12,11, 9, 9,10,11,12, 6, 7, 7,10,10, 7, 8,
	 7,10,10, 7, 7, 8,10,10,10,10,10,12,12, 9,10,10,
	11,12, 6, 7, 7,10,10, 7, 8, 7,10,10, 7, 7, 7,10,
	10, 9,10,10,12,11,10,10,10,12,12, 9,10,10,13,12,
	10,10,10,13,13,10,10,10,13,12,12,12,12,14,14,12,
	12,12,14,14, 9,10,10,12,13, 9,10,10,13,13,10,10,
	10,13,13,12,12,12,15,14,12,13,12,14,14, 5, 7, 7,
	10,10, 7, 8, 7,10,10, 7, 7, 8,10,10,10,10,10,12,
	12,10,10,10,12,12, 7, 8, 8,10,10, 8, 8, 8,10,11,
	 8, 8, 8,11,10,10,10,11,11,13,10,10,11,12,13, 6,
	 7, 7,10,10, 7, 8, 7,11,10, 8, 8, 8,10,11,10,11,
	10,13,11,10,10,10,13,12,10,11,10,13,13,10,10,10,
	12,13,10,11,11,13,13,12,11,13,11,14,12,13,13,14,
	14, 9,10,10,12,13,10,10,10,13,12,10,10,11,13,13,
	12,13,11,14,12,13,13,13,15,14, 5, 7, 7,10,10, 7,
	 7, 7,10,10, 7, 7, 8,10,10,10,10,10,12,12,10,10,
	10,12,13, 6, 7, 7,10,10, 8, 8, 8,11,10, 7, 7, 8,
	10,11,10,10,10,12,12,10,10,11,11,13, 7, 8, 8,10,
	10, 7, 8, 8,10,11, 8, 8, 8,11,10,10,11,10,13,12,
	10,11,10,13,11, 9,10,10,13,13,10,11,11,13,13,10,
	10,10,12,13,13,13,13,14,14,12,11,13,12,14,10,10,
	11,13,13,10,11,11,13,13,10,10,10,13,12,12,13,13,
	14,14,12,13,11,15,12, 9,10,10,13,13,10,10,10,13,
	13,10,10,10,13,13,13,13,13,15,15,12,13,13,14,14,
	 9,10,10,13,13,10,10,11,13,13,10,11,10,13,12,13,
	12,13,14,15,13,13,13,15,14, 9,10,10,13,13,10,10,
	10,13,12,10,10,11,12,13,13,13,12,15,14,13,13,13,
	14,14,12,13,12,15,14,12,11,12,14,14,12,13,12,15,
	14,14,12,15,12,16,14,14,15,16,16,12,12,12,14,15,
	11,12,11,15,14,12,13,13,14,15,14,14,12,16,13,14,
	14,15,16,16, 9,10,10,13,13,10,10,10,13,13,10,10,
	10,13,13,12,13,12,14,14,13,13,13,15,15, 9,10,10,
	13,13,10,11,10,13,12,10,10,10,12,13,13,13,13,14,
	14,12,12,13,14,15, 9,10,10,13,13,10,10,11,12,13,
	10,11,10,13,13,13,13,13,14,15,13,13,13,15,14,12,
	12,12,15,14,12,13,12,15,14,11,11,12,14,15,14,14,
	14,17,16,14,12,14,13,17,12,12,13,14,16,13,13,13,
	13,15,12,12,11,14,14,14,15,14,16,16,14,14,12,16,
	12,
};

static float _vq_quantthresh__44u2__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u2__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u2__p4_0 = {
	_vq_quantthresh__44u2__p4_0,
	_vq_quantmap__44u2__p4_0,
	5,
	5
};

static static_codebook _44u2__p4_0 = {
	4, 625,
	_vq_lengthlist__44u2__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u2__p4_0,
	NULL,
	&_vq_auxt__44u2__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u2__p5_0[] = {
	 2, 4, 4, 6, 6, 7, 7, 9, 9, 4, 5, 4, 7, 7, 8, 7,
	 9, 9, 4, 4, 4, 7, 7, 7, 8, 9, 9, 6, 7, 7, 8, 8,
	 8, 9,10,10, 6, 7, 7, 8, 8, 9, 8,10,10, 7, 8, 7,
	 8, 9, 9,10,11,11, 7, 7, 8, 9, 9,10,10,11,11, 9,
	 9, 9,10,10,11,11,12,12, 9, 9, 9,10,10,11,11,12,
	12,
};

static float _vq_quantthresh__44u2__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u2__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u2__p5_0 = {
	_vq_quantthresh__44u2__p5_0,
	_vq_quantmap__44u2__p5_0,
	9,
	9
};

static static_codebook _44u2__p5_0 = {
	2, 81,
	_vq_lengthlist__44u2__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u2__p5_0,
	NULL,
	&_vq_auxt__44u2__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u2__p6_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9,10,11,11,14,14, 4, 6, 5,
	 8, 8, 9, 9,11,11,11,11,14,14, 4, 5, 5, 8, 8, 9,
	 9,10,11,11,12,14,14, 7, 8, 8, 9, 9,11,10,11,11,
	12,12,15,15, 7, 8, 8, 9, 9,11,11,11,11,12,12,15,
	14, 9,10,10,11,11,11,11,12,12,13,12,16,16, 9,10,
	10,11,11,11,11,12,12,13,13,15,16,10,11,11,11,11,
	12,12,13,13,13,13,17,16,10,11,11,12,12,12,12,12,
	13,14,14,16,16,11,12,12,12,12,13,13,14,14,14,14,
	17,17,11,12,12,12,12,13,13,14,14,15,14,18,17,15,
	14,15,15,15,16,16,17,18,19,17,19, 0,15,15,15,15,
	15,16,17,16,16,18,17, 0, 0,
};

static float _vq_quantthresh__44u2__p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44u2__p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u2__p6_0 = {
	_vq_quantthresh__44u2__p6_0,
	_vq_quantmap__44u2__p6_0,
	13,
	13
};

static static_codebook _44u2__p6_0 = {
	2, 169,
	_vq_lengthlist__44u2__p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44u2__p6_0,
	NULL,
	&_vq_auxt__44u2__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u2__p6_1[] = {
	 2, 4, 4, 5, 5, 4, 5, 5, 6, 5, 4, 5, 5, 5, 5, 5,
	 6, 6, 6, 6, 5, 5, 6, 6, 6,
};

static float _vq_quantthresh__44u2__p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u2__p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u2__p6_1 = {
	_vq_quantthresh__44u2__p6_1,
	_vq_quantmap__44u2__p6_1,
	5,
	5
};

static static_codebook _44u2__p6_1 = {
	2, 25,
	_vq_lengthlist__44u2__p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u2__p6_1,
	NULL,
	&_vq_auxt__44u2__p6_1,
	NULL,
	0
};

static long _vq_quantlist__44u2__p7_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u2__p7_0[] = {
	 1, 3, 3,10,10,10,10,10,10, 4,10,10,10,10,10,10,
	10,10, 4,10,10,10,10,10,10,10,10,10,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9,
};

static float _vq_quantthresh__44u2__p7_0[] = {
	-591.5, -422.5, -253.5, -84.5, 84.5, 253.5, 422.5, 591.5, 
};

static long _vq_quantmap__44u2__p7_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u2__p7_0 = {
	_vq_quantthresh__44u2__p7_0,
	_vq_quantmap__44u2__p7_0,
	9,
	9
};

static static_codebook _44u2__p7_0 = {
	2, 81,
	_vq_lengthlist__44u2__p7_0,
	1, -516612096, 1626677248, 4, 0,
	_vq_quantlist__44u2__p7_0,
	NULL,
	&_vq_auxt__44u2__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u2__p7_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u2__p7_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 8, 7, 9, 9,10,10, 4, 7, 6,
	 8, 8, 8, 8,10, 9,11,10,12,12, 4, 6, 7, 8, 8, 8,
	 8,10,10,11,11,12,12, 7, 8, 8,10, 9,10,10,12,11,
	13,12,13,13, 7, 8, 8, 9, 9,10,10,11,11,13,13,14,
	15, 7, 8, 9,10,10,12,11,14,13,13,14,14,15, 7, 8,
	 9,10,10,11,12,13,13,15,14,14,15, 8,10,10,12,11,
	13,13,15,15,15,17,15,15, 8,10,10,11,12,14,13,14,
	15,17,15,15,15,10,11,11,14,13,14,14,15,15,17,17,
	16,17,10,11,12,13,13,14,14,14,15,16,15,15,17,11,
	12,13,14,13,16,16,16,14,17,16,17,17,11,12,13,15,
	15,15,15,16,15,15,15,15,17,
};

static float _vq_quantthresh__44u2__p7_1[] = {
	-71.5, -58.5, -45.5, -32.5, -19.5, -6.5, 6.5, 19.5, 
	32.5, 45.5, 58.5, 71.5, 
};

static long _vq_quantmap__44u2__p7_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u2__p7_1 = {
	_vq_quantthresh__44u2__p7_1,
	_vq_quantmap__44u2__p7_1,
	13,
	13
};

static static_codebook _44u2__p7_1 = {
	2, 169,
	_vq_lengthlist__44u2__p7_1,
	1, -523010048, 1618608128, 4, 0,
	_vq_quantlist__44u2__p7_1,
	NULL,
	&_vq_auxt__44u2__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u2__p7_2[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u2__p7_2[] = {
	 2, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 5, 6, 6,
	 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 5, 6, 6, 7, 7, 8,
	 7, 8, 8, 8, 8, 8, 8, 6, 7, 7, 7, 7, 8, 8, 8, 8,
	 9, 9, 9, 9, 6, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 7, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 7, 8,
	 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44u2__p7_2[] = {
	-5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 
	2.5, 3.5, 4.5, 5.5, 
};

static long _vq_quantmap__44u2__p7_2[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u2__p7_2 = {
	_vq_quantthresh__44u2__p7_2,
	_vq_quantmap__44u2__p7_2,
	13,
	13
};

static static_codebook _44u2__p7_2 = {
	2, 169,
	_vq_lengthlist__44u2__p7_2,
	1, -531103744, 1611661312, 4, 0,
	_vq_quantlist__44u2__p7_2,
	NULL,
	&_vq_auxt__44u2__p7_2,
	NULL,
	0
};

static long _huff_lengthlist__44u2__short[] = {
	16,15,15,16,14,16,16,16,16, 9, 8,11,10,10,10,15,
	16, 6, 3, 7, 5, 7, 9,16,16,11, 7,11, 9,10,12,16,
	16, 9, 4, 8, 3, 5, 8,16,16,11, 6, 9, 4, 4, 7,16,
	16,11, 5, 9, 3, 3, 5,12,16,15, 6,11, 5, 5, 6,14,
};

static static_codebook _huff_book__44u2__short = {
	2, 64,
	_huff_lengthlist__44u2__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u3__long[] = {
	 7,11,14,13,14,12,12,12,12, 2, 5, 8, 9, 8, 9,11,
	17, 4, 3, 6, 5, 7, 9,15,14, 6, 6, 7, 7, 7, 8,14,
	17, 8, 5, 7, 4, 5, 7,15,13, 7, 6, 7, 5, 5, 6,14,
	10, 8, 8, 7, 7, 5, 4, 9,10,12,15,14,16,10, 6, 9,
};

static static_codebook _huff_book__44u3__long = {
	2, 64,
	_huff_lengthlist__44u3__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u3__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u3__p1_0[] = {
	 1, 4, 4, 5, 8, 7, 5, 7, 8, 5, 8, 8, 8,11,11, 8,
	10,11, 5, 8, 8, 8,11,10, 8,11,11, 4, 8, 8, 8,11,
	11, 8,11,11, 8,11,11,11,13,14,11,13,14, 7,11,11,
	10,13,11,11,13,14, 4, 8, 8, 8,11,11, 8,11,11, 8,
	11,11,11,14,14,10,12,14, 8,11,11,11,14,13,11,14,
	13,
};

static float _vq_quantthresh__44u3__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u3__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u3__p1_0 = {
	_vq_quantthresh__44u3__p1_0,
	_vq_quantmap__44u3__p1_0,
	3,
	3
};

static static_codebook _44u3__p1_0 = {
	4, 81,
	_vq_lengthlist__44u3__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u3__p1_0,
	NULL,
	&_vq_auxt__44u3__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u3__p2_0[] = {
	 3, 4, 4, 5, 6, 6, 5, 6, 6, 5, 6, 6, 6, 8, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 6, 8, 8, 5, 6, 6, 6, 8,
	 7, 6, 8, 8, 6, 8, 8, 8, 8, 9, 8, 9, 9, 6, 8, 7,
	 7, 9, 8, 8, 9, 9, 5, 6, 6, 6, 8, 8, 6, 8, 8, 6,
	 8, 8, 8, 9, 9, 7, 8, 9, 6, 8, 8, 8, 9, 9, 8, 9,
	 8,
};

static float _vq_quantthresh__44u3__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u3__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u3__p2_0 = {
	_vq_quantthresh__44u3__p2_0,
	_vq_quantmap__44u3__p2_0,
	3,
	3
};

static static_codebook _44u3__p2_0 = {
	4, 81,
	_vq_lengthlist__44u3__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u3__p2_0,
	NULL,
	&_vq_auxt__44u3__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u3__p3_0[] = {
	 2, 4, 4, 8, 8, 5, 7, 7, 9, 9, 5, 7, 7, 9, 9, 8,
	10, 9,12,12, 8, 9,10,12,12, 5, 7, 7,10,10, 7, 9,
	 9,11,11, 7, 8, 9,10,11,10,11,10,14,13, 9,10,11,
	13,13, 5, 7, 7,10,10, 7, 9, 8,11,10, 7, 9, 9,11,
	11, 9,10,10,13,12,10,10,11,13,14, 8,10,10,14,13,
	10,11,11,15,13, 9,11,11,14,13,13,14,13,17,15,12,
	13,13,17,14, 8,10,10,14,14, 9,11,11,14,15,10,11,
	12,14,16,12,13,13,14,17,13,13,14,14,17, 5, 7, 7,
	10,10, 7, 9, 9,11,11, 7, 9, 9,11,11,10,11,11,15,
	14,10,11,11,15,14, 7, 9, 9,12,12, 9,11,11,13,13,
	 9,11,11,13,13,11,12,12,15,14,11,12,12,15,16, 7,
	 9, 9,12,11, 8,11,10,13,12, 9,11,11,12,13,11,12,
	11,16,14,11,12,13,15,16,10,12,12,17,15,11,12,13,
	16,15,11,12,12,16,16,15,15,15,16,16,13,14,15, 0,
	17, 9,11,11,15,15,10,12,11,16,15,11,12,12,15,17,
	13,15,13,16,15,13,15,14, 0,18, 5, 7, 7,10,10, 7,
	 9, 9,12,11, 7, 9, 9,11,11,10,11,11,14,14,10,11,
	11,14,15, 7, 9, 9,12,11, 9,11,11,13,12, 8,10,11,
	11,12,11,12,12,16,15,11,11,12,13,14, 7, 9, 9,12,
	12, 9,11,11,13,13, 9,11,11,13,13,11,12,12,16,15,
	12,12,12,15,15, 9,11,11,17,15,11,12,12,17,16,10,
	11,12,15,15,13,14,15,18,17,13,13,14,15,15, 9,12,
	12,15,14,11,13,12,16,16,11,12,12,15,15,13,15,14,
	 0, 0,14,15,14,16,18, 8,10,10,15,15,10,12,12,16,
	14,10,11,11,16,16,15,16,16,18,16,13,15,14,17,18,
	 9,11,11,16,15,11,12,13,17,17,11,13,13,16,15,15,
	16,16,18,18,14,16,15,18,17, 9,11,11,16,16,10,13,
	12,16,16,11,12,12,16,16,14,16,16,17, 0,14,15,15,
	18,16,12,14,14, 0, 0,14,15,16, 0, 0,14,15,15, 0,
	 0,17,17,16, 0, 0,15,16,18,17, 0,11,14,14,18, 0,
	12,15,13, 0,18,13,15,14,18, 0,15,16,15, 0,18,15,
	18,17, 0,18, 8,10,10,15,15,10,12,11,15,15,10,11,
	12,15,15,13,14,15,17, 0,14,15,16,17, 0, 9,11,11,
	15,15,11,12,12,17,15,10,12,13,16,15,13,15,15,18,
	18,15,16,17,18,18, 9,12,11,16,16,11,13,13,16,16,
	11,13,13,18,15,14,15,15,18, 0,16,16,17, 0,17,12,
	13,13,18,18,13,14,14,17,18,13,14,15,18,17,17,18,
	18, 0,18,16,16,18,16, 0,12,15,13,18,18,13,14,15,
	18,18,13,16,14,17,18,15,17,16, 0,18,17, 0,17, 0,
	 0,
};

static float _vq_quantthresh__44u3__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u3__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u3__p3_0 = {
	_vq_quantthresh__44u3__p3_0,
	_vq_quantmap__44u3__p3_0,
	5,
	5
};

static static_codebook _44u3__p3_0 = {
	4, 625,
	_vq_lengthlist__44u3__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u3__p3_0,
	NULL,
	&_vq_auxt__44u3__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u3__p4_0[] = {
	 4, 6, 6, 9, 9, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 9,
	10, 9,12,11, 9, 9,10,11,12, 6, 7, 7, 9, 9, 7, 7,
	 7,10,10, 6, 7, 7, 9,10,10,10,10,12,12, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 7, 7, 7,10, 9, 7, 7, 7,10,
	10, 9,10, 9,12,11,10,10,10,12,12, 9,10,10,12,12,
	10,10,10,13,12, 9,10,10,12,12,12,12,12,14,14,12,
	12,12,13,14, 9,10,10,12,12, 9,10,10,12,12,10,10,
	10,12,13,11,12,12,14,13,12,12,12,14,14, 6, 7, 7,
	10, 9, 7, 8, 7,10,10, 7, 7, 7,10,10,10,10,10,12,
	12,10,10,10,12,12, 7, 8, 8,10,10, 8, 8, 8,10,10,
	 7, 8, 8,10,10,10,10,11,11,13,10,10,11,12,13, 6,
	 7, 7,10,10, 7, 8, 7,10,10, 8, 8, 8,10,10,10,11,
	10,13,11,10,10,10,12,12,10,10,10,13,12,10,10,10,
	12,13,10,10,10,13,12,12,11,13,12,14,12,12,13,14,
	14, 9,10,10,12,13, 9,10,10,12,12,10,10,11,12,13,
	12,12,11,14,12,12,12,13,14,14, 6, 7, 7, 9, 9, 7,
	 7, 7,10,10, 7, 7, 8,10,10,10,10,10,12,12,10,10,
	10,12,12, 6, 7, 7,10,10, 8, 8, 8,10,10, 7, 7, 8,
	10,10,10,10,10,12,12,10,10,11,11,13, 7, 8, 8,10,
	10, 7, 8, 8,10,10, 8, 8, 8,10,10,10,10,10,12,12,
	10,11,10,13,11, 9,10,10,13,12,10,11,10,13,12, 9,
	10,10,12,13,12,13,12,14,14,12,11,12,12,14,10,10,
	10,12,13,10,10,11,12,13,10,10,10,12,12,12,13,12,
	14,14,12,12,11,14,12, 9,10, 9,12,12,10,10,10,13,
	13, 9,10,10,13,13,12,13,13,15,14,12,12,13,14,14,
	 9,10,10,13,13,10,10,10,12,13,10,10,10,13,12,13,
	12,13,14,15,12,13,13,15,14, 9,10,10,13,12,10,10,
	10,13,12,10,10,10,12,13,12,13,12,14,14,13,12,12,
	14,14,12,12,12,15,14,12,11,12,14,14,12,13,12,14,
	14,14,12,14,12,16,14,14,14,16,16,11,12,12,14,14,
	11,12,11,15,13,12,12,12,14,15,13,14,12,16,13,14,
	14,15,16,16, 9,10,10,12,12, 9,10,10,13,12, 9,10,
	10,13,13,12,12,12,14,14,12,13,13,14,15, 9,10,10,
	13,12,10,11,10,13,12,10,10,10,12,13,12,13,12,14,
	14,12,12,13,14,15, 9,10,10,13,13,10,10,10,12,13,
	10,10,10,13,13,12,13,13,14,15,13,13,12,14,14,11,
	12,12,14,14,12,13,12,15,14,11,11,12,14,15,14,14,
	14,16,16,14,12,14,13,16,12,12,12,14,15,12,12,13,
	14,15,12,12,11,14,14,14,14,14,16,16,14,14,12,16,
	12,
};

static float _vq_quantthresh__44u3__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u3__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u3__p4_0 = {
	_vq_quantthresh__44u3__p4_0,
	_vq_quantmap__44u3__p4_0,
	5,
	5
};

static static_codebook _44u3__p4_0 = {
	4, 625,
	_vq_lengthlist__44u3__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u3__p4_0,
	NULL,
	&_vq_auxt__44u3__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u3__p5_0[] = {
	 2, 4, 4, 6, 6, 7, 7, 9, 9, 4, 5, 4, 7, 6, 8, 8,
	 9, 9, 4, 4, 4, 6, 7, 8, 8, 9, 9, 6, 7, 6, 8, 8,
	 9, 9,10,10, 6, 6, 7, 8, 8, 9, 9,10,10, 8, 8, 8,
	 9, 9,10,10,11,11, 8, 8, 8, 9, 9,10,10,11,11, 9,
	 9, 9,10,10,11,11,12,12, 9, 9,10,10,10,11,11,12,
	12,
};

static float _vq_quantthresh__44u3__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u3__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u3__p5_0 = {
	_vq_quantthresh__44u3__p5_0,
	_vq_quantmap__44u3__p5_0,
	9,
	9
};

static static_codebook _44u3__p5_0 = {
	2, 81,
	_vq_lengthlist__44u3__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u3__p5_0,
	NULL,
	&_vq_auxt__44u3__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u3__p6_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9,10,10,11,13,14, 4, 6, 5,
	 8, 8, 9, 9,10,10,11,11,14,14, 4, 5, 6, 8, 8, 9,
	 9,10,10,11,11,14,14, 7, 8, 8, 9, 9,10,10,11,11,
	12,12,15,15, 7, 8, 8, 9, 9,10,10,11,11,12,12,14,
	15, 8, 9, 9,10,10,11,11,12,12,13,12,16,16, 8, 9,
	 9,10,10,11,11,12,12,12,12,16,16,10,11,10,11,11,
	12,12,13,13,13,13,17,16,10,11,11,11,11,12,12,12,
	12,13,13,17,17,11,11,12,12,12,13,13,13,14,14,14,
	16,17,11,12,11,12,12,13,13,14,14,14,14,19,17,14,
	14,14,16,15,16,16,16,17,17,17,20,19,14,15,15,15,
	15,15,16,16,17,17,17,20,19,
};

static float _vq_quantthresh__44u3__p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44u3__p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u3__p6_0 = {
	_vq_quantthresh__44u3__p6_0,
	_vq_quantmap__44u3__p6_0,
	13,
	13
};

static static_codebook _44u3__p6_0 = {
	2, 169,
	_vq_lengthlist__44u3__p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44u3__p6_0,
	NULL,
	&_vq_auxt__44u3__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u3__p6_1[] = {
	 2, 4, 4, 5, 5, 4, 5, 5, 6, 5, 4, 5, 5, 5, 6, 5,
	 6, 5, 6, 6, 5, 5, 6, 6, 6,
};

static float _vq_quantthresh__44u3__p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u3__p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u3__p6_1 = {
	_vq_quantthresh__44u3__p6_1,
	_vq_quantmap__44u3__p6_1,
	5,
	5
};

static static_codebook _44u3__p6_1 = {
	2, 25,
	_vq_lengthlist__44u3__p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u3__p6_1,
	NULL,
	&_vq_auxt__44u3__p6_1,
	NULL,
	0
};

static long _vq_quantlist__44u3__p7_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u3__p7_0[] = {
	 1, 4, 4, 9, 9, 9, 9, 9, 9, 4, 9, 9, 9, 9, 9, 9,
	 9, 9, 3, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8,
};

static float _vq_quantthresh__44u3__p7_0[] = {
	-892.5, -637.5, -382.5, -127.5, 127.5, 382.5, 637.5, 892.5, 
};

static long _vq_quantmap__44u3__p7_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u3__p7_0 = {
	_vq_quantthresh__44u3__p7_0,
	_vq_quantmap__44u3__p7_0,
	9,
	9
};

static static_codebook _44u3__p7_0 = {
	2, 81,
	_vq_lengthlist__44u3__p7_0,
	1, -515907584, 1627381760, 4, 0,
	_vq_quantlist__44u3__p7_0,
	NULL,
	&_vq_auxt__44u3__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u3__p7_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44u3__p7_1[] = {
	 1, 4, 4, 6, 5, 7, 7, 9, 8,10,10,11,11,12,13, 4,
	 7, 6, 7, 7, 9, 9,10,10,11,11,14,13,14,13, 4, 6,
	 7, 7, 8, 9, 9,10,10,12,12,13,13,13,13, 6, 8, 8,
	10, 9,11,11,12,12,13,13,14,14,15,14, 6, 8, 8, 9,
	 9,11,11,12,12,14,14,14,15,15,15, 8, 9, 9,11,10,
	13,12,14,14,14,14,15,16,15,15, 8, 9, 9,11,11,12,
	12,14,14,15,14,15,15,14,17, 9,10,10,13,12,14,14,
	15,14,14,17,15,15,16,15, 9,11,11,12,13,14,14,15,
	15,15,16,15,17,16,17,11,12,12,14,14,15,14,16,16,
	16,15,15,17,16,16,11,12,13,14,15,15,15,15,15,16,
	16,17,17,16,17,12,13,13,15,14,15,15,15,15,16,16,
	16,17,17,17,13,12,14,14,15,15,15,15,16,17,17,15,
	17,17,17,12,14,14,16,15,17,16,17,16,17,16,17,17,
	17,17,14,13,14,15,16,17,17,17,15,17,17,17,16,17,
	16,
};

static float _vq_quantthresh__44u3__p7_1[] = {
	-110.5, -93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 
	25.5, 42.5, 59.5, 76.5, 93.5, 110.5, 
};

static long _vq_quantmap__44u3__p7_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44u3__p7_1 = {
	_vq_quantthresh__44u3__p7_1,
	_vq_quantmap__44u3__p7_1,
	15,
	15
};

static static_codebook _44u3__p7_1 = {
	2, 225,
	_vq_lengthlist__44u3__p7_1,
	1, -522338304, 1620115456, 4, 0,
	_vq_quantlist__44u3__p7_1,
	NULL,
	&_vq_auxt__44u3__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u3__p7_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44u3__p7_2[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 5, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 5, 6, 6, 7, 7, 7, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 6, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9,10, 9, 6, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9,10,10,10, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9,10,10,10,10, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9,10,10,10,10,10,10, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9,10,10,10,10,10,10, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10,10,10,10, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9,10,10,10,10,10,10, 9, 9, 9, 9, 9,
	 9, 9,10, 9,10,10,10,10,10,10,10,10, 9, 9, 9, 9,
	 9, 9, 9,10,10,10,10,10,10,10,10,10,10, 9, 9, 9,
	 9, 9,10,10,10,10,10,10,10,10,10,10,10,10, 9, 9,
	 9,10,10,10,10,10,10,10,10,10,10,10,10,10,10, 9,
	 9,10,10, 9,10, 9,10,10,10,10,10,10,10,10,10,10,
	 9,10, 9,10, 9,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44u3__p7_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44u3__p7_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44u3__p7_2 = {
	_vq_quantthresh__44u3__p7_2,
	_vq_quantmap__44u3__p7_2,
	17,
	17
};

static static_codebook _44u3__p7_2 = {
	2, 289,
	_vq_lengthlist__44u3__p7_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44u3__p7_2,
	NULL,
	&_vq_auxt__44u3__p7_2,
	NULL,
	0
};

static long _huff_lengthlist__44u3__short[] = {
	17,17,17,17,17,17,16,16,16,12,11,13,11,12,11,15,
	16, 9, 4, 8, 5, 7, 9,15,16,14, 9,12,10,10,12,16,
	16,11, 4, 9, 3, 4, 8,16,16,12, 6,10, 3, 4, 7,16,
	16,12, 6,11, 3, 3, 4,12,16,16, 7,13, 5, 5, 6,13,
};

static static_codebook _huff_book__44u3__short = {
	2, 64,
	_huff_lengthlist__44u3__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u4__long[] = {
	 3, 7,11,11,12,11,11,12, 5, 3, 5, 7, 8, 8, 9,11,
	 9, 4, 4, 6, 6, 7, 9,15,10, 6, 6, 7, 7, 7, 8,14,
	12, 7, 5, 6, 4, 5, 7,15,10, 7, 6, 7, 5, 5, 6,13,
	 9, 8, 8, 7, 7, 5, 4, 8,10,12,14,13,14,10, 5, 7,
};

static static_codebook _huff_book__44u4__long = {
	2, 64,
	_huff_lengthlist__44u4__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u4__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u4__p1_0[] = {
	 1, 4, 4, 5, 8, 7, 5, 7, 8, 5, 8, 8, 8,11,11, 8,
	10,11, 5, 8, 8, 8,11,10, 8,11,11, 4, 8, 8, 8,11,
	11, 8,11,11, 8,11,11,11,13,14,11,14,14, 7,11,10,
	10,14,12,11,14,14, 4, 8, 8, 8,11,11, 8,11,11, 8,
	11,11,11,14,14,10,12,14, 8,11,11,11,14,14,11,14,
	13,
};

static float _vq_quantthresh__44u4__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u4__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u4__p1_0 = {
	_vq_quantthresh__44u4__p1_0,
	_vq_quantmap__44u4__p1_0,
	3,
	3
};

static static_codebook _44u4__p1_0 = {
	4, 81,
	_vq_lengthlist__44u4__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u4__p1_0,
	NULL,
	&_vq_auxt__44u4__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u4__p2_0[] = {
	 2, 5, 5, 5, 6, 6, 5, 6, 6, 5, 6, 6, 7, 8, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 7, 8, 8, 5, 6, 6, 7, 8,
	 8, 6, 8, 8, 7, 8, 8, 8, 9,10, 8,10,10, 6, 8, 8,
	 7, 9, 8, 8, 9,10, 5, 6, 6, 6, 8, 8, 7, 8, 8, 6,
	 8, 8, 8,10, 9, 8, 8, 9, 6, 8, 8, 8,10,10, 8,10,
	 9,
};

static float _vq_quantthresh__44u4__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u4__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u4__p2_0 = {
	_vq_quantthresh__44u4__p2_0,
	_vq_quantmap__44u4__p2_0,
	3,
	3
};

static static_codebook _44u4__p2_0 = {
	4, 81,
	_vq_lengthlist__44u4__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u4__p2_0,
	NULL,
	&_vq_auxt__44u4__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u4__p3_0[] = {
	 2, 4, 4, 7, 7, 5, 7, 7, 9, 9, 5, 7, 7, 9, 9, 8,
	10, 9,12,12, 8, 9, 9,12,12, 5, 7, 7,10, 9, 7, 9,
	 9,11,11, 7, 8, 9,10,11,10,11,11,13,13, 9,10,11,
	13,13, 5, 7, 7, 9,10, 7, 9, 9,11,11, 7, 9, 9,11,
	11, 9,11,10,13,12,10,11,11,13,13, 8,10,10,13,13,
	10,11,11,15,13, 9,11,11,14,13,13,15,13,16,15,12,
	13,13,15,15, 8,10,10,13,14, 9,11,11,14,14,10,11,
	12,14,16,12,13,13,14,16,13,14,14,15,17, 5, 7, 7,
	10,10, 7, 9, 9,11,11, 7, 9, 9,11,11,10,11,11,15,
	14,10,11,11,14,14, 7, 9, 9,12,12, 9,11,11,13,13,
	 9,11,11,13,13,11,12,13,15,15,11,13,13,16,15, 7,
	 9, 9,11,11, 9,11,10,13,12, 9,11,12,13,13,11,13,
	12,16,14,11,13,13,15,16,10,12,12,18,15,11,12,13,
	16,15,11,13,13,17,16,15,15,15,17,17,14,15,16,18,
	19, 9,11,11,15,15,10,12,11,15,16,11,13,13,15,16,
	13,15,13,18,15,14,15,15,17,19, 5, 7, 7,10,10, 7,
	 9, 9,12,11, 7, 9, 9,11,12,10,12,11,14,14,10,11,
	12,14,15, 7, 9, 9,11,11, 9,11,11,13,12, 9,10,11,
	12,13,11,13,13,16,17,11,11,13,14,15, 7, 9, 9,12,
	12, 9,11,11,13,13, 9,11,11,13,13,11,13,12,15,16,
	11,13,13,16,15, 9,11,11,16,15,11,13,12,16,15,10,
	12,12,16,15,14,15,16,19,17,13,14,15,15,16,10,11,
	12,15,15,11,13,13,16,16,11,13,12,16,17,14,15,15,
	18,17,14,16,14,16,19, 7,10,10,15,14,10,12,12,16,
	15,10,11,11,16,15,14,16,16,19,18,13,15,14,17,17,
	 9,11,11,17,16,11,13,14,18,17,11,13,13,16,16,15,
	16,17,19, 0,14,15,18,16,19, 9,11,11,16,15,11,13,
	12,18,16,11,13,13,17,16,14,16,16,17,19,15,16,15,
	18, 0,12,14,15,18,19,13,15,16,19,17,14,16,15,18,
	 0,17,16,19, 0,18,16,17,18, 0, 0,11,13,14,18,19,
	13,15,13,19,19,14,15,15,17,17,15,17,15,19,17,16,
	18,18,19, 0, 8,10,10,14,14,10,12,11,15,15,10,11,
	12,15,17,13,15,15,17,16,14,15,15,18, 0, 9,11,11,
	15,15,11,13,13,18,16,10,12,13,16,17,14,15,16,17,
	18,14,15,17,19,19, 9,12,12,15,16,11,13,13,16,17,
	11,14,13,19,16,14,16,15,18,19,15,16,18,19,18,12,
	14,14,19,18,13,15,17,18,17,13,14,15,18, 0,16, 0,
	 0,18,18,16,15, 0,17,19,12,15,14,17, 0,13,15,16,
	19,18,13,16,15, 0,19,16,18,16,19,18,17,19,16, 0,
	19,
};

static float _vq_quantthresh__44u4__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u4__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u4__p3_0 = {
	_vq_quantthresh__44u4__p3_0,
	_vq_quantmap__44u4__p3_0,
	5,
	5
};

static static_codebook _44u4__p3_0 = {
	4, 625,
	_vq_lengthlist__44u4__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u4__p3_0,
	NULL,
	&_vq_auxt__44u4__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u4__p4_0[] = {
	 4, 5, 5, 8, 8, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 9,
	 9, 9,11,11, 8, 9, 9,11,11, 6, 7, 7, 9, 9, 7, 8,
	 8,10,10, 6, 7, 8, 9,10, 9,10,10,11,12, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 7, 8, 7,10, 9, 7, 8, 8,10,
	10, 9,10, 9,12,11, 9,10,10,12,11, 9, 9, 9,12,11,
	 9,10,10,12,12, 9,10,10,12,12,11,12,12,13,14,11,
	11,12,13,14, 9, 9, 9,11,12, 9,10,10,12,12, 9,10,
	10,12,12,11,12,11,14,13,11,12,12,13,13, 5, 7, 7,
	 9, 9, 7, 8, 8,10,10, 7, 7, 8,10,10, 9,10,10,12,
	12, 9,10,10,12,12, 7, 8, 8,10,10, 8, 8, 9,10,11,
	 8, 9, 9,11,11,10,10,11,11,13,10,10,11,12,13, 6,
	 7, 8,10,10, 7, 9, 8,11,10, 8, 8, 9,10,11,10,11,
	10,13,11,10,11,11,13,12, 9,10,10,12,12,10,10,11,
	12,13,10,11,11,13,13,12,11,13,12,15,12,13,13,14,
	14, 9,10,10,12,12, 9,11,10,13,12,10,11,11,13,13,
	11,13,11,14,12,12,13,13,14,15, 5, 7, 7, 9, 9, 7,
	 8, 7,10,10, 7, 8, 8,10,10, 9,10,10,12,12, 9,10,
	10,12,12, 6, 8, 7,10,10, 8, 9, 8,11,10, 7, 8, 9,
	10,11,10,11,11,12,13,10,10,11,11,13, 7, 8, 8,10,
	10, 8, 9, 9,10,11, 8, 9, 8,11,10,10,11,11,13,12,
	10,11,10,13,11, 9,10,10,13,12,10,11,11,13,13, 9,
	10,10,12,13,12,13,13,15,14,11,11,13,12,14, 9,10,
	10,12,12,10,11,11,13,13,10,11,10,13,12,12,13,13,
	14,14,12,13,11,15,12, 8, 9, 9,12,12, 9,10,10,13,
	12, 9,10,10,12,12,12,13,12,14,14,11,12,12,14,13,
	 9,10,10,12,12,10,10,11,13,13,10,11,11,13,12,12,
	13,13,14,15,12,13,13,15,14, 9,10,10,12,12, 9,11,
	10,13,12,10,10,11,12,13,12,13,12,15,14,12,13,13,
	14,14,11,12,12,14,13,11,11,12,13,14,12,13,13,15,
	14,13,12,14,12,16,14,15,14,16,16,11,12,12,14,14,
	11,12,11,15,13,12,13,13,14,15,13,14,12,16,13,14,
	14,15,16,16, 8, 9, 9,12,12, 9,10,10,12,12, 9,10,
	10,12,13,11,12,12,14,13,12,12,13,14,14, 9,10,10,
	12,12,10,11,11,13,12,10,10,11,12,13,12,13,13,14,
	14,12,12,13,14,15, 9,10,10,12,13,10,11,11,12,13,
	10,11,10,13,13,12,13,13,14,15,12,13,13,15,14,11,
	12,12,14,14,12,13,13,15,14,11,11,12,14,15,14,14,
	14,17,16,13,12,14,13,16,11,12,12,13,15,12,13,13,
	14,15,11,12,11,14,14,14,15,14,15,17,13,14,12,16,
	12,
};

static float _vq_quantthresh__44u4__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u4__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u4__p4_0 = {
	_vq_quantthresh__44u4__p4_0,
	_vq_quantmap__44u4__p4_0,
	5,
	5
};

static static_codebook _44u4__p4_0 = {
	4, 625,
	_vq_lengthlist__44u4__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u4__p4_0,
	NULL,
	&_vq_auxt__44u4__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u4__p5_0[] = {
	 2, 4, 4, 6, 6, 7, 7, 9, 9, 4, 5, 4, 7, 6, 8, 7,
	10, 9, 4, 4, 5, 6, 7, 7, 8, 9, 9, 6, 7, 6, 8, 8,
	 8, 8,10,10, 6, 6, 7, 8, 8, 8, 9,10,10, 7, 8, 7,
	 9, 8,10,10,11,11, 7, 7, 8, 8, 9,10,10,11,11, 9,
	10, 9,10,10,11,11,12,12, 9, 9,10,10,10,11,11,12,
	12,
};

static float _vq_quantthresh__44u4__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u4__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u4__p5_0 = {
	_vq_quantthresh__44u4__p5_0,
	_vq_quantmap__44u4__p5_0,
	9,
	9
};

static static_codebook _44u4__p5_0 = {
	2, 81,
	_vq_lengthlist__44u4__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u4__p5_0,
	NULL,
	&_vq_auxt__44u4__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p6_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u4__p6_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9,10,10,11,13,13, 4, 6, 5,
	 8, 8, 9, 9,10,10,11,11,14,14, 4, 5, 6, 8, 8, 9,
	 9,10,10,11,11,14,14, 7, 8, 8, 9, 9,10,10,11,11,
	12,12,15,15, 7, 8, 8, 9, 9,10,10,11,11,12,12,15,
	15, 8, 9, 9,10,10,11,11,12,12,13,13,16,16, 8, 9,
	 9,10,10,11,11,12,12,13,13,16,16,10,10,10,11,11,
	12,12,13,13,14,13,16,16,10,10,11,11,12,12,12,13,
	13,13,14,16,17,11,12,11,12,12,13,13,13,14,14,14,
	17,16,11,11,12,12,12,13,13,14,14,15,14,17,17,14,
	14,14,15,15,16,16,17,17,17,19,19, 0,14,15,15,15,
	15,16,16,16,17,17,19,20,20,
};

static float _vq_quantthresh__44u4__p6_0[] = {
	-27.5, -22.5, -17.5, -12.5, -7.5, -2.5, 2.5, 7.5, 
	12.5, 17.5, 22.5, 27.5, 
};

static long _vq_quantmap__44u4__p6_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u4__p6_0 = {
	_vq_quantthresh__44u4__p6_0,
	_vq_quantmap__44u4__p6_0,
	13,
	13
};

static static_codebook _44u4__p6_0 = {
	2, 169,
	_vq_lengthlist__44u4__p6_0,
	1, -526516224, 1616117760, 4, 0,
	_vq_quantlist__44u4__p6_0,
	NULL,
	&_vq_auxt__44u4__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p6_1[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u4__p6_1[] = {
	 2, 4, 4, 5, 5, 4, 5, 5, 6, 5, 4, 5, 5, 5, 6, 5,
	 6, 5, 6, 6, 5, 5, 6, 6, 6,
};

static float _vq_quantthresh__44u4__p6_1[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u4__p6_1[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u4__p6_1 = {
	_vq_quantthresh__44u4__p6_1,
	_vq_quantmap__44u4__p6_1,
	5,
	5
};

static static_codebook _44u4__p6_1 = {
	2, 25,
	_vq_lengthlist__44u4__p6_1,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u4__p6_1,
	NULL,
	&_vq_auxt__44u4__p6_1,
	NULL,
	0
};

static long _vq_quantlist__44u4__p7_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u4__p7_0[] = {
	 1, 3, 3,11,11,11,11,11,11,11,11,11,11, 4,10, 9,
	11,11,11,11,11,11,11,11,11,11, 4,10,10,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	10,10,10,10,10,10,10,10,10,
};

static float _vq_quantthresh__44u4__p7_0[] = {
	-1402.5, -1147.5, -892.5, -637.5, -382.5, -127.5, 127.5, 382.5, 
	637.5, 892.5, 1147.5, 1402.5, 
};

static long _vq_quantmap__44u4__p7_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u4__p7_0 = {
	_vq_quantthresh__44u4__p7_0,
	_vq_quantmap__44u4__p7_0,
	13,
	13
};

static static_codebook _44u4__p7_0 = {
	2, 169,
	_vq_lengthlist__44u4__p7_0,
	1, -514332672, 1627381760, 4, 0,
	_vq_quantlist__44u4__p7_0,
	NULL,
	&_vq_auxt__44u4__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u4__p7_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44u4__p7_1[] = {
	 1, 4, 4, 6, 6, 7, 7, 9, 8,10,10,11,11,12,12, 4,
	 7, 6, 8, 7, 9, 9,10,10,11,10,12,11,12,12, 4, 6,
	 6, 7, 8, 9, 9,10,10,11,11,12,12,13,13, 6, 8, 8,
	10, 9,11,10,11,11,12,12,14,13,13,13, 6, 8, 8, 9,
	 9,11,11,12,12,12,12,13,14,13,14, 8, 9, 9,11,10,
	12,12,13,12,13,14,14,14,14,14, 8, 9, 9,10,11,12,
	11,12,13,13,14,14,13,14,14, 9,10,10,12,12,14,13,
	14,14,15,15,16,15,15,15, 9,10,10,11,12,13,13,13,
	14,15,15,18,16,17,15,10,12,12,13,13,15,14,15,14,
	16,16,16,15,16,15,10,11,11,13,13,14,14,17,16,15,
	16,18,16,15,16,11,12,13,14,13,15,14,15,16,17,15,
	16,16,15,16,11,12,13,14,15,14,16,15,15,16,15,15,
	17,17,17,12,13,13,14,14,14,15,16,14,15,15,16,16,
	16,16,12,13,13,13,14,15,15,15,14,16,16,16,16,16,
	17,
};

static float _vq_quantthresh__44u4__p7_1[] = {
	-110.5, -93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 
	25.5, 42.5, 59.5, 76.5, 93.5, 110.5, 
};

static long _vq_quantmap__44u4__p7_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44u4__p7_1 = {
	_vq_quantthresh__44u4__p7_1,
	_vq_quantmap__44u4__p7_1,
	15,
	15
};

static static_codebook _44u4__p7_1 = {
	2, 225,
	_vq_lengthlist__44u4__p7_1,
	1, -522338304, 1620115456, 4, 0,
	_vq_quantlist__44u4__p7_1,
	NULL,
	&_vq_auxt__44u4__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u4__p7_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44u4__p7_2[] = {
	 3, 5, 5, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 5, 6, 6, 7, 7, 8, 7, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 6, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 6, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9,10,10, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9,10,10,10,10,10,10, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9,10,10,10,10,10,10, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9,10,10,10,10,10,10, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9,10,10,10,10,10,10, 9, 9, 9, 9, 9,
	 9, 9,10, 9, 9,10,10,10,10,10,10,10, 9, 9, 9, 9,
	 9, 9,10, 9,10,10,10,10,10,10,10,10,10, 9, 9, 9,
	 9, 9,10,10,10,10,10,10,10,10,10,10,10,10, 9, 9,
	 9, 9, 9,10, 9,10,10,10,10,10,10,10,10,10,10, 9,
	 9, 9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
	 9, 9, 9,10, 9,10,10,10,10,10,10,10,10,10,10,10,
	10,
};

static float _vq_quantthresh__44u4__p7_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44u4__p7_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44u4__p7_2 = {
	_vq_quantthresh__44u4__p7_2,
	_vq_quantmap__44u4__p7_2,
	17,
	17
};

static static_codebook _44u4__p7_2 = {
	2, 289,
	_vq_lengthlist__44u4__p7_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44u4__p7_2,
	NULL,
	&_vq_auxt__44u4__p7_2,
	NULL,
	0
};

static long _huff_lengthlist__44u4__short[] = {
	 3, 9,14,12,17,13,13,13, 4, 7, 8, 9,12,10,11,12,
	 6, 5, 4, 7, 6, 7, 9,14, 8, 8, 7, 9, 8, 9,10,13,
	 9, 8, 5, 7, 3, 5, 8,15, 9, 9, 6, 8, 4, 5, 7,16,
	11,10, 6, 9, 4, 3, 6,15,17,16, 8,11, 5, 5, 7,16,
};

static static_codebook _huff_book__44u4__short = {
	2, 64,
	_huff_lengthlist__44u4__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u5__long[] = {
	 3, 8,13,12,14,13,16,11,12,13, 5, 4, 5, 6, 8, 9,
	10, 9,12,12,10, 5, 4, 6, 6, 8, 9,10,14,16,10, 6,
	 6, 6, 6, 7, 9, 8,12,13,13, 7, 5, 6, 4, 6, 6, 7,
	11,16,10, 7, 7, 7, 6, 6, 7, 7,11,14,14, 9, 8, 8,
	 5, 6, 6, 7,11,16, 9, 8, 8, 8, 6, 6, 6, 4, 7,12,
	11,10,12,11,10, 9,10, 5, 6,10,10,13,15,15,15,15,
	14, 8, 7, 9,
};

static static_codebook _huff_book__44u5__long = {
	2, 100,
	_huff_lengthlist__44u5__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u5__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u5__p1_0[] = {
	 1, 4, 4, 5, 8, 7, 5, 7, 7, 5, 8, 8, 8,10,10, 7,
	 9,10, 5, 8, 8, 7,10, 9, 8,10,10, 4, 8, 8, 8,11,
	10, 8,10,10, 8,11,11,10,12,13,10,13,13, 7,10,10,
	 9,13,11,10,13,13, 5, 8, 8, 8,10,10, 8,10,10, 7,
	10,10,10,13,13,10,11,13, 8,10,11,10,13,13,10,13,
	12,
};

static float _vq_quantthresh__44u5__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u5__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u5__p1_0 = {
	_vq_quantthresh__44u5__p1_0,
	_vq_quantmap__44u5__p1_0,
	3,
	3
};

static static_codebook _44u5__p1_0 = {
	4, 81,
	_vq_lengthlist__44u5__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u5__p1_0,
	NULL,
	&_vq_auxt__44u5__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u5__p2_0[] = {
	 3, 4, 4, 5, 6, 6, 5, 6, 6, 5, 6, 6, 6, 8, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 6, 8, 8, 5, 6, 6, 6, 8,
	 7, 6, 8, 8, 6, 8, 8, 8, 9, 9, 8, 9, 9, 6, 7, 8,
	 7, 9, 8, 8, 9, 9, 5, 6, 6, 6, 8, 7, 6, 8, 8, 6,
	 8, 8, 8, 9, 9, 7, 8, 9, 6, 8, 8, 8, 9, 9, 8, 9,
	 9,
};

static float _vq_quantthresh__44u5__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u5__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u5__p2_0 = {
	_vq_quantthresh__44u5__p2_0,
	_vq_quantmap__44u5__p2_0,
	3,
	3
};

static static_codebook _44u5__p2_0 = {
	4, 81,
	_vq_lengthlist__44u5__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u5__p2_0,
	NULL,
	&_vq_auxt__44u5__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u5__p3_0[] = {
	 2, 4, 5, 8, 8, 5, 7, 6, 9, 9, 5, 6, 7, 9, 9, 8,
	10, 9,12,12, 8, 9,10,12,13, 5, 7, 7,10, 9, 7, 9,
	 9,11,11, 7, 8, 9,10,11,10,11,11,13,14, 9,10,11,
	13,13, 5, 7, 7, 9, 9, 7, 9, 8,11,10, 7, 9, 9,11,
	11, 9,11,10,14,13,10,11,11,13,14, 8,10,10,13,13,
	10,11,11,15,14, 9,11,11,14,14,13,15,14,18,16,12,
	13,14,16,16, 8,10,10,13,13, 9,11,11,14,14,10,11,
	12,14,15,12,13,13,16,16,13,14,14,15,17, 5, 7, 7,
	10,10, 7, 9, 9,11,11, 7, 9, 9,11,11,10,11,11,15,
	14,10,11,11,14,14, 7, 9, 9,12,11, 9,11,11,13,13,
	 9,11,11,13,13,12,12,13,15,15,11,12,13,16,15, 6,
	 9, 9,11,11, 8,11,10,13,12, 9,10,11,12,14,11,13,
	11,16,14,11,13,13,15,16,10,11,11,15,15,11,12,13,
	16,15,11,13,13,16,16,14,15,15,17,18,14,15,16,17,
	18, 9,11,11,14,15,10,12,11,15,15,11,12,13,15,16,
	13,15,13,17,15,14,15,16,18,19, 5, 7, 7,10,10, 7,
	 9, 9,11,11, 7, 9, 9,11,11,10,11,11,15,14,10,11,
	11,14,15, 6, 9, 9,11,11, 9,11,10,13,13, 8,10,11,
	12,13,11,13,13,15,15,11,11,13,13,15, 7, 9, 9,11,
	12, 9,11,11,13,13, 9,11,11,13,13,11,13,12,17,16,
	11,13,12,16,15, 9,11,11,15,14,11,13,13,16,16,10,
	11,12,15,15,14,15,15,17,17,13,13,15,15,17,10,11,
	12,15,15,11,13,13,16,19,11,13,12,17,17,14,16,15,
	19,17,14,15,15,18,18, 8,10,10,14,14,10,12,11,16,
	15, 9,11,11,15,16,14,16,15,17,17,13,14,14,18,17,
	 9,11,11,16,15,11,13,13,16,18,11,13,13,17,16,15,
	16,17,18, 0,15,15,16, 0,18, 9,11,11,16,15,10,13,
	12,17,15,11,13,13,16,17,14,18,15,19,18,15,16,16,
	19,18,13,15,15,19,17,13,15,15,18,18,14,15,15,19,
	 0,17,18, 0,19,19,16,17,17, 0,18,12,14,13,18,17,
	13,15,13,19,18,14,15,15,19,19,16,17,15, 0,17,16,
	19,17,19,18, 8,10,10,14,14,10,11,11,15,15,10,11,
	11,16,16,13,15,15,17,16,14,15,15,18,19, 9,11,11,
	16,15,11,13,13,19,16,10,12,13,15,17,15,15,15,19,
	19,15,16,16,17, 0, 9,11,11,16,16,11,13,13,16,17,
	11,13,13,18,18,14,16,15,18,18,15,17,16,19, 0,12,
	14,14,17,18,13,16,16,18,18,13,14,15,17, 0,16,17,
	17, 0, 0,16,15, 0,19,17,12,15,14,17,18,14,15,16,
	 0,18,14,16,16,18, 0,16,17,17, 0, 0,17,19,17,19,
	 0,
};

static float _vq_quantthresh__44u5__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u5__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u5__p3_0 = {
	_vq_quantthresh__44u5__p3_0,
	_vq_quantmap__44u5__p3_0,
	5,
	5
};

static static_codebook _44u5__p3_0 = {
	4, 625,
	_vq_lengthlist__44u5__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u5__p3_0,
	NULL,
	&_vq_auxt__44u5__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u5__p4_0[] = {
	 4, 6, 6, 8, 8, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 8,
	 9, 9,11,11, 8, 9, 9,11,11, 6, 7, 7, 9, 9, 7, 8,
	 8,10,10, 6, 7, 8, 9,10, 9,10,10,11,11, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 6, 8, 7,10, 9, 7, 8, 8,10,
	10, 9,10, 9,12,11, 9,10,10,12,11, 8, 9, 9,12,11,
	 9,10,10,12,12, 9,10,10,12,12,11,12,12,13,13,11,
	11,12,12,13, 8, 9, 9,11,11, 9,10,10,12,12, 9,10,
	10,12,12,11,12,11,13,13,11,12,12,14,13, 6, 7, 7,
	 9, 9, 7, 8, 7,10,10, 7, 7, 8, 9,10, 9,10,10,12,
	11, 9,10,10,11,11, 7, 8, 8,10,10, 8, 8, 9,10,11,
	 8, 8, 8,11,10,10,10,11,11,12,10,10,11,12,12, 6,
	 7, 7,10,10, 7, 8, 8,11,10, 8, 8, 9,10,11,10,11,
	10,13,11,10,10,11,12,12, 9,10,10,12,12,10,10,10,
	12,13,10,11,11,13,13,12,11,12,11,14,12,12,13,13,
	14, 9,10,10,12,12, 9,10,10,12,12,10,10,11,12,13,
	11,12,11,14,12,12,12,12,14,14, 6, 7, 7, 9, 9, 7,
	 8, 7,10,10, 7, 7, 8,10,10, 9,10,10,11,11, 9,10,
	10,12,12, 6, 7, 7,10,10, 8, 9, 8,11,10, 7, 8, 8,
	10,11,10,10,11,12,12,10,10,11,11,13, 7, 8, 8,10,
	10, 8, 8, 8,10,11, 8, 9, 8,11,10,10,11,10,12,12,
	10,11,10,12,11, 9,10,10,12,12,10,11,11,13,12, 9,
	10,10,12,12,12,12,12,14,14,11,11,12,12,14, 9,10,
	10,12,12,10,11,11,12,12,10,10,10,12,12,12,13,12,
	14,14,11,12,11,14,11, 8, 9, 9,11,11, 9,10,10,12,
	12, 9,10,10,12,12,11,12,12,14,14,11,12,12,13,13,
	 9,10,10,12,12,10,10,11,12,13,10,11,10,13,12,12,
	12,13,14,14,12,12,12,14,13, 9,10,10,12,12, 9,10,
	10,13,12,10,10,11,12,13,12,13,12,14,13,12,12,13,
	13,14,11,12,11,14,13,11,11,12,13,14,12,13,12,14,
	14,13,12,14,11,16,13,14,14,16,15,11,11,11,13,13,
	11,12,11,14,13,12,12,13,14,15,12,14,12,16,12,14,
	14,14,16,16, 8, 9, 9,11,11, 9,10,10,12,12, 9,10,
	10,12,12,11,12,12,13,13,12,12,12,14,14, 9,10,10,
	12,12,10,11,10,13,12, 9,10,10,12,13,12,12,12,14,
	13,12,12,13,13,15, 9,10,10,12,12,10,10,11,12,13,
	10,11,10,13,12,12,13,12,14,14,12,13,12,14,13,11,
	11,11,13,13,12,13,12,14,14,11,11,12,13,14,13,14,
	14,16,15,13,12,14,12,15,11,12,12,13,14,12,12,13,
	14,14,11,12,11,14,13,13,14,14,15,16,13,14,11,16,
	11,
};

static float _vq_quantthresh__44u5__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u5__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u5__p4_0 = {
	_vq_quantthresh__44u5__p4_0,
	_vq_quantmap__44u5__p4_0,
	5,
	5
};

static static_codebook _44u5__p4_0 = {
	4, 625,
	_vq_lengthlist__44u5__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u5__p4_0,
	NULL,
	&_vq_auxt__44u5__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u5__p5_0[] = {
	 2, 4, 3, 6, 6, 8, 7,10,10, 3, 5, 5, 8, 7, 8, 8,
	11,11, 3, 5, 5, 7, 8, 8, 8,10,11, 6, 8, 7, 9, 9,
	10, 9,12,11, 7, 7, 8, 9, 9, 9,10,11,12, 8, 8, 8,
	10, 9,11,11,13,12, 8, 8, 8, 9,10,11,12,12,13,10,
	11,10,12,11,13,12,14,14,10,10,11,11,12,12,13,14,
	14,
};

static float _vq_quantthresh__44u5__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u5__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u5__p5_0 = {
	_vq_quantthresh__44u5__p5_0,
	_vq_quantmap__44u5__p5_0,
	9,
	9
};

static static_codebook _44u5__p5_0 = {
	2, 81,
	_vq_lengthlist__44u5__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u5__p5_0,
	NULL,
	&_vq_auxt__44u5__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p6_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u5__p6_0[] = {
	 3, 4, 4, 5, 6, 7, 7, 9, 9, 4, 5, 4, 6, 6, 7, 7,
	 9, 9, 4, 4, 4, 6, 6, 7, 7, 9, 9, 6, 6, 6, 7, 7,
	 8, 8,10,10, 6, 6, 6, 7, 7, 8, 8,10,10, 7, 7, 7,
	 8, 8, 9, 9,10,10, 7, 7, 7, 8, 8, 9, 9,10,11, 9,
	 9, 9,10,10,11,10,11,11, 9, 9, 9,10,10,11,11,11,
	11,
};

static float _vq_quantthresh__44u5__p6_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u5__p6_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u5__p6_0 = {
	_vq_quantthresh__44u5__p6_0,
	_vq_quantmap__44u5__p6_0,
	9,
	9
};

static static_codebook _44u5__p6_0 = {
	2, 81,
	_vq_lengthlist__44u5__p6_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u5__p6_0,
	NULL,
	&_vq_auxt__44u5__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u5__p7_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 5, 9, 8, 8,11,10, 7,
	10,10, 5, 9, 9, 8,10,10, 8,10,11, 4, 9, 9, 9,11,
	11, 9,12,11, 8,11,11,11,12,12,10,12,12, 7,11,11,
	10,12,12,10,12,12, 4, 9, 9, 9,11,11, 9,11,11, 7,
	11,11,10,12,12,10,11,12, 8,11,11,10,12,12,11,12,
	12,
};

static float _vq_quantthresh__44u5__p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44u5__p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u5__p7_0 = {
	_vq_quantthresh__44u5__p7_0,
	_vq_quantmap__44u5__p7_0,
	3,
	3
};

static static_codebook _44u5__p7_0 = {
	4, 81,
	_vq_lengthlist__44u5__p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44u5__p7_0,
	NULL,
	&_vq_auxt__44u5__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u5__p7_1[] = {
	 2, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 4, 5, 5, 7, 7,
	 8, 8, 9, 8, 9, 8, 4, 5, 5, 7, 7, 8, 8, 8, 9, 9,
	 9, 6, 7, 7, 8, 8, 9, 8, 9, 9, 9, 9, 6, 7, 7, 8,
	 8, 8, 9, 9, 9, 9, 9, 8, 8, 8, 9, 8, 9, 9, 9, 9,
	 9, 9, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 8, 9, 8,
	 9, 9, 9, 9,10,10,10,10, 8, 9, 9, 9, 9, 9, 9,10,
	10,10,10, 8, 9, 9, 9, 9, 9, 9,10,10,10,10, 8, 9,
	 9, 9, 9, 9, 9,10,10,10,10,
};

static float _vq_quantthresh__44u5__p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44u5__p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u5__p7_1 = {
	_vq_quantthresh__44u5__p7_1,
	_vq_quantmap__44u5__p7_1,
	11,
	11
};

static static_codebook _44u5__p7_1 = {
	2, 121,
	_vq_lengthlist__44u5__p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44u5__p7_1,
	NULL,
	&_vq_auxt__44u5__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u5__p8_0[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u5__p8_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9, 9,10,10, 4, 6, 6, 7, 7,
	 9, 9,10,10,11,11, 4, 6, 6, 7, 7, 9, 9,10,10,11,
	11, 6, 8, 8, 9, 9,10,10,12,11,13,12, 6, 7, 8, 9,
	 9,10,10,11,11,12,12, 8, 9, 9,10,10,11,12,12,12,
	14,13, 8, 9, 9,10,10,11,11,12,13,14,13,10,11,10,
	12,12,13,13,14,13,15,14,10,11,11,12,12,12,13,13,
	13,15,15,11,12,12,13,12,13,14,14,14,15,14,11,11,
	12,13,13,13,14,14,14,15,15,
};

static float _vq_quantthresh__44u5__p8_0[] = {
	-49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 27.5, 
	38.5, 49.5, 
};

static long _vq_quantmap__44u5__p8_0[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u5__p8_0 = {
	_vq_quantthresh__44u5__p8_0,
	_vq_quantmap__44u5__p8_0,
	11,
	11
};

static static_codebook _44u5__p8_0 = {
	2, 121,
	_vq_lengthlist__44u5__p8_0,
	1, -524582912, 1618345984, 4, 0,
	_vq_quantlist__44u5__p8_0,
	NULL,
	&_vq_auxt__44u5__p8_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p8_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u5__p8_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 5, 6, 5, 6, 6,
	 7, 7, 8, 8, 8, 8, 5, 6, 6, 6, 6, 7, 7, 8, 8, 8,
	 8, 6, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 6, 6, 7, 7,
	 7, 7, 7, 8, 8, 8, 8, 7, 7, 7, 7, 7, 8, 8, 8, 8,
	 8, 8, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 7, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 7, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44u5__p8_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44u5__p8_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u5__p8_1 = {
	_vq_quantthresh__44u5__p8_1,
	_vq_quantmap__44u5__p8_1,
	11,
	11
};

static static_codebook _44u5__p8_1 = {
	2, 121,
	_vq_lengthlist__44u5__p8_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44u5__p8_1,
	NULL,
	&_vq_auxt__44u5__p8_1,
	NULL,
	0
};

static long _vq_quantlist__44u5__p9_0[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u5__p9_0[] = {
	 1, 3, 3,12,10,12,12,12,12,12,12,12,12, 3, 8, 9,
	12,12,12,12,12,12,12,12,12,12, 4, 9, 9,12,12,12,
	12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,
	12,12,12,12, 9,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
	11,11,11,11,11,11,11,11,11,
};

static float _vq_quantthresh__44u5__p9_0[] = {
	-1402.5, -1147.5, -892.5, -637.5, -382.5, -127.5, 127.5, 382.5, 
	637.5, 892.5, 1147.5, 1402.5, 
};

static long _vq_quantmap__44u5__p9_0[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u5__p9_0 = {
	_vq_quantthresh__44u5__p9_0,
	_vq_quantmap__44u5__p9_0,
	13,
	13
};

static static_codebook _44u5__p9_0 = {
	2, 169,
	_vq_lengthlist__44u5__p9_0,
	1, -514332672, 1627381760, 4, 0,
	_vq_quantlist__44u5__p9_0,
	NULL,
	&_vq_auxt__44u5__p9_0,
	NULL,
	0
};

static long _vq_quantlist__44u5__p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44u5__p9_1[] = {
	 1, 4, 4, 7, 7, 8, 9, 7, 7, 8, 8, 9, 9,10,10, 5,
	 6, 6, 9, 8,10,10, 9, 8, 9, 9,10, 9,11,10, 4, 6,
	 6, 8, 9, 9,10, 8, 9, 9,10,10,10,11,10, 7, 8, 8,
	10,10,11,11,10,10,11,11,11,11,11,11, 7, 8, 8,10,
	10,11,11,10,10,11,11,11,11,11,12, 8, 9,10,11,10,
	12,12,11,11,11,11,12,12,12,13, 8, 9, 9,11,11,11,
	12,11,11,11,11,12,12,12,12, 8, 9, 9,10,10,11,11,
	12,11,12,12,12,13,13,13, 8, 9, 9,10,10,11,11,12,
	11,12,12,13,13,12,13, 9,10,10,11,11,12,11,12,13,
	14,14,15,15,15,14, 9,10,10,11,11,12,12,12,12,13,
	13,14,14,14,14, 9,10,11,12,11,12,13,14,13,14,13,
	14,14,14,14, 9,11,11,11,11,12,13,13,13,14,13,15,
	15,14,14,10,11,11,12,12,13,13,13,13,14,14,14,15,
	15,15,10,11,11,12,12,12,13,13,13,15,14,15,14,14,
	14,
};

static float _vq_quantthresh__44u5__p9_1[] = {
	-110.5, -93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 
	25.5, 42.5, 59.5, 76.5, 93.5, 110.5, 
};

static long _vq_quantmap__44u5__p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44u5__p9_1 = {
	_vq_quantthresh__44u5__p9_1,
	_vq_quantmap__44u5__p9_1,
	15,
	15
};

static static_codebook _44u5__p9_1 = {
	2, 225,
	_vq_lengthlist__44u5__p9_1,
	1, -522338304, 1620115456, 4, 0,
	_vq_quantlist__44u5__p9_1,
	NULL,
	&_vq_auxt__44u5__p9_1,
	NULL,
	0
};

static long _vq_quantlist__44u5__p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44u5__p9_2[] = {
	 3, 5, 5, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 5, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 5, 6, 6, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 7, 7, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 7, 8, 8, 8, 8, 9, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10, 9, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10, 9, 9,10, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10, 9, 9, 9, 9, 9,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9,10,10,10,10, 9,10,10,10,10, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10,10,10, 9, 9, 9, 9,10,
	 9, 9, 9, 9, 9, 9, 9,10, 9,10, 9, 9, 9,10,10, 9,
	 9,
};

static float _vq_quantthresh__44u5__p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44u5__p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44u5__p9_2 = {
	_vq_quantthresh__44u5__p9_2,
	_vq_quantmap__44u5__p9_2,
	17,
	17
};

static static_codebook _44u5__p9_2 = {
	2, 289,
	_vq_lengthlist__44u5__p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44u5__p9_2,
	NULL,
	&_vq_auxt__44u5__p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44u5__short[] = {
	 4,11,16,14,18,15,18,15,17,17, 4, 6, 9, 9,12, 9,
	14,10,14,17, 6, 6, 5, 7, 6, 8,10,10,16,17, 7, 8,
	 7, 9, 9,10,12,12,14,17, 8, 6, 5, 7, 4, 7, 5, 7,
	14,17, 9, 9, 8, 9, 7,10, 8,10,14,17,11, 9, 7, 9,
	 4, 7, 3, 6,15,17,11,10,10,10, 6, 9, 5, 6,14,17,
	17,17,10,15, 4, 8, 3, 5,12,17,17,17,12,16, 7,11,
	 6, 8,16,17,
};

static static_codebook _huff_book__44u5__short = {
	2, 100,
	_huff_lengthlist__44u5__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u6__long[] = {
	 3, 9,14,13,15,13,16,12,12,12, 5, 4, 6, 7, 8, 9,
	10,10,13,12,10, 5, 5, 6, 6, 9, 9,10,14,14,10, 6,
	 6, 7, 6, 8, 9, 9,13,12,13, 7, 5, 6, 4, 6, 6, 7,
	11,15,10, 8, 7, 7, 6, 7, 7, 7,11,13,16,10, 8, 8,
	 5, 6, 4, 6,10,14, 9, 9, 9, 8, 6, 6, 5, 4, 7,11,
	11,11,12,11,10, 9, 9, 5, 5, 9,10,13,14,14,14,14,
	15, 8, 7, 8,
};

static static_codebook _huff_book__44u6__long = {
	2, 100,
	_huff_lengthlist__44u6__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u6__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u6__p1_0[] = {
	 1, 4, 4, 4, 8, 7, 5, 7, 7, 5, 8, 8, 8,10,10, 7,
	 9,10, 5, 8, 8, 7,10, 9, 8,10,10, 5, 8, 8, 8,11,
	10, 8,10,10, 8,11,10,10,12,13,10,13,13, 7,10,10,
	10,13,11,10,13,13, 5, 8, 8, 8,11,10, 8,10,10, 7,
	10,10,10,13,13, 9,11,13, 8,10,11,10,13,13,10,13,
	12,
};

static float _vq_quantthresh__44u6__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u6__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u6__p1_0 = {
	_vq_quantthresh__44u6__p1_0,
	_vq_quantmap__44u6__p1_0,
	3,
	3
};

static static_codebook _44u6__p1_0 = {
	4, 81,
	_vq_lengthlist__44u6__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u6__p1_0,
	NULL,
	&_vq_auxt__44u6__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u6__p2_0[] = {
	 3, 5, 4, 5, 6, 6, 5, 6, 6, 5, 6, 6, 6, 7, 8, 6,
	 7, 8, 5, 6, 6, 6, 8, 7, 6, 8, 7, 5, 6, 6, 6, 8,
	 7, 6, 7, 8, 6, 8, 8, 8, 8, 9, 8, 9, 9, 6, 7, 7,
	 7, 9, 8, 8, 9, 9, 5, 6, 6, 6, 8, 7, 6, 7, 8, 6,
	 7, 7, 8, 9, 9, 7, 8, 9, 6, 8, 8, 8, 9, 9, 8, 9,
	 8,
};

static float _vq_quantthresh__44u6__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u6__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u6__p2_0 = {
	_vq_quantthresh__44u6__p2_0,
	_vq_quantmap__44u6__p2_0,
	3,
	3
};

static static_codebook _44u6__p2_0 = {
	4, 81,
	_vq_lengthlist__44u6__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u6__p2_0,
	NULL,
	&_vq_auxt__44u6__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u6__p3_0[] = {
	 2, 4, 5, 8, 8, 5, 7, 6, 9, 9, 5, 6, 7, 9, 9, 8,
	10, 9,13,12, 8, 9,10,12,13, 5, 7, 7,10, 9, 7, 9,
	 9,11,11, 7, 8, 9,10,11,10,11,11,13,14, 9,10,11,
	13,13, 5, 7, 7, 9, 9, 7, 9, 8,11,10, 7, 9, 9,11,
	11, 9,11,10,14,13,10,11,11,14,13, 8,10,10,14,13,
	10,12,11,15,14, 9,11,11,15,14,13,15,14,20,16,13,
	13,14,16,16, 8,10,10,13,14, 9,11,11,14,15,10,11,
	12,14,15,13,13,14,16,16,13,14,14,16,17, 5, 7, 7,
	10,10, 7, 9, 9,11,11, 7, 9, 9,11,11,10,11,11,15,
	14,10,11,11,14,14, 7, 9, 9,12,11, 9,11,11,12,13,
	 9,11,11,13,13,11,12,13,14,16,11,12,13,16,16, 6,
	 9, 8,11,11, 8,11,10,13,12, 9,10,11,12,13,11,13,
	11,15,14,11,13,13,15,15,10,12,11,15,14,11,12,13,
	17,16,11,13,13,16,17,14,15,16,17,17,14,15,16,20,
	 0, 9,11,11,14,15,10,13,12,16,16,11,13,13,17,17,
	13,15,14,17,15,15,16,16, 0, 0, 5, 7, 7,10,10, 7,
	 9, 9,11,11, 7, 8, 9,11,11,10,11,11,14,14,10,11,
	11,14,14, 6, 9, 9,11,11, 9,11,11,14,12, 8,10,11,
	12,13,11,13,13,16,16,11,11,13,14,16, 7, 9, 9,11,
	12, 9,11,11,13,13, 9,11,11,13,12,11,13,12,16,16,
	12,13,12,16,14, 9,11,11,16,15,11,13,13,17,15,10,
	12,12,15,16,14,16,16, 0,17,13,14,15,15,17,10,11,
	12,15,15,11,13,13,17,17,11,13,13,15,16,14,15,15,
	19,18,14,15,15,19,16, 8,10,10,14,14,10,12,11,15,
	16,10,11,11,15,17,13,16,15,19,19,13,15,14,17,17,
	 9,11,12,15,16,11,13,13,18,19,11,13,13,16,17,16,
	16,17, 0, 0,15,15,16, 0,19, 9,11,11,15,16,10,13,
	12,17,15,11,13,13,16,17,14,17,15,18,17,15,16,15,
	18, 0,13,14,15, 0,19,14,16,16, 0,17,15,16,16, 0,
	17,16,17, 0, 0, 0,16,17,19, 0, 0,12,14,14,17,17,
	13,15,13, 0,19,14,16,16,18,19,16,18,15,19,18,17,
	18,17,19, 0, 8,10,10,14,13,10,11,11,16,15,10,11,
	11,15,15,13,15,15,19,18,14,15,16, 0,18, 9,11,11,
	15,14,11,13,13, 0,16,10,12,13,15,17,14,16,16, 0,
	18,15,16,17,17, 0, 9,11,11,16,15,11,13,13,16,16,
	11,14,13,18,18,15,17,15,18,17,15,16,18, 0, 0,12,
	14,14,18,19,14,15,16, 0,18,13,14,15,18, 0,17,17,
	17, 0, 0,16,15,18,17, 0,12,14,15,17,19,14,16,16,
	19, 0,14,17,16, 0,18,17,19,17, 0, 0,16,18,17, 0,
	17,
};

static float _vq_quantthresh__44u6__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u6__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u6__p3_0 = {
	_vq_quantthresh__44u6__p3_0,
	_vq_quantmap__44u6__p3_0,
	5,
	5
};

static static_codebook _44u6__p3_0 = {
	4, 625,
	_vq_lengthlist__44u6__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u6__p3_0,
	NULL,
	&_vq_auxt__44u6__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u6__p4_0[] = {
	 4, 6, 6, 8, 8, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 8,
	 9, 9,11,11, 8, 9, 9,11,11, 6, 7, 7, 9, 9, 7, 8,
	 8,10,10, 6, 7, 8, 9,10, 9,10,10,11,12, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 7, 8, 7,10, 9, 7, 8, 8,10,
	10, 9,10, 9,12,11, 9,10,10,12,11, 8, 9, 9,11,11,
	 9,10,10,12,12, 9, 9,10,12,12,11,12,12,13,13,11,
	11,12,12,13, 8, 9, 9,11,11, 9,10,10,12,12, 9,10,
	10,12,12,11,12,11,13,13,11,12,12,13,13, 6, 7, 7,
	 9, 9, 7, 8, 7,10, 9, 7, 7, 8, 9,10, 9,10,10,12,
	11, 9,10,10,11,12, 7, 8, 8,10,10, 8, 8, 9,10,10,
	 8, 8, 8,10,10,10,10,11,12,12,10,10,11,12,12, 6,
	 7, 7,10,10, 7, 8, 8,11,10, 8, 8, 9,10,11,10,11,
	10,12,11,10,10,11,12,12, 9,10,10,12,12,10,10,11,
	12,12,10,11,10,13,12,12,12,12,12,14,12,12,12,13,
	14, 9,10,10,12,12, 9,10,10,12,12,10,10,11,12,13,
	11,12,11,14,12,12,12,13,14,14, 6, 7, 7, 9, 9, 7,
	 8, 7,10, 9, 7, 7, 8,10,10, 9,10,10,12,11, 9,10,
	10,11,12, 6, 7, 7,10,10, 8, 9, 8,11,10, 7, 8, 8,
	10,10,10,10,11,12,12,10,10,10,11,12, 7, 8, 8,10,
	10, 8, 8, 8,10,10, 8, 8, 8,10,10,10,11,10,12,12,
	10,11,10,12,11, 9,10,10,12,12,10,11,10,13,12, 9,
	10,10,12,12,12,12,12,14,14,11,11,12,12,13, 9,10,
	10,12,12,10,11,11,12,12,10,11,10,12,12,12,12,12,
	14,14,12,12,12,14,12, 8, 9, 9,11,11, 9,10,10,12,
	12, 9,10,10,12,12,11,12,12,13,13,11,12,12,13,13,
	 9,10,10,12,12,10,10,11,12,13,10,11,10,12,12,12,
	12,13,13,14,12,12,12,14,14, 9,10,10,12,12, 9,10,
	10,12,12,10,10,10,12,12,12,12,12,14,13,12,12,12,
	14,14,11,12,12,13,13,11,12,12,14,14,12,12,12,14,
	14,13,12,14,12,15,13,14,14,15,15,11,11,11,13,13,
	11,12,11,14,13,12,12,12,14,14,12,14,12,15,12,13,
	14,14,15,16, 8, 9, 9,11,11, 9,10,10,12,12, 9,10,
	10,12,12,11,12,12,13,13,11,12,12,13,14, 9,10,10,
	12,12,10,11,10,13,12, 9,10,10,12,13,12,12,12,15,
	14,12,12,13,13,14, 9,10,10,12,12,10,10,11,12,12,
	10,11,10,12,12,12,13,12,14,14,12,13,12,14,13,11,
	11,11,13,13,12,12,12,14,13,11,11,12,13,14,13,14,
	14,15,15,12,12,14,12,15,11,12,12,13,14,12,12,12,
	14,14,11,12,12,14,14,13,14,14,15,15,13,14,12,16,
	12,
};

static float _vq_quantthresh__44u6__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u6__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u6__p4_0 = {
	_vq_quantthresh__44u6__p4_0,
	_vq_quantmap__44u6__p4_0,
	5,
	5
};

static static_codebook _44u6__p4_0 = {
	4, 625,
	_vq_lengthlist__44u6__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u6__p4_0,
	NULL,
	&_vq_auxt__44u6__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u6__p5_0[] = {
	 2, 3, 3, 6, 6, 8, 8,10,10, 4, 5, 5, 8, 7, 8, 8,
	11,11, 3, 5, 5, 7, 8, 8, 8,11,11, 6, 8, 7,10, 9,
	10, 9,12,12, 6, 7, 8, 9,10, 9,10,12,12, 8, 8, 8,
	10, 9,12,11,13,13, 8, 8, 8, 9,10,11,12,12,13,10,
	11,11,12,11,13,12,14,14,10,10,11,11,12,13,13,14,
	14,
};

static float _vq_quantthresh__44u6__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u6__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u6__p5_0 = {
	_vq_quantthresh__44u6__p5_0,
	_vq_quantmap__44u6__p5_0,
	9,
	9
};

static static_codebook _44u6__p5_0 = {
	2, 81,
	_vq_lengthlist__44u6__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u6__p5_0,
	NULL,
	&_vq_auxt__44u6__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p6_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u6__p6_0[] = {
	 3, 4, 4, 5, 5, 7, 7, 9, 9, 4, 5, 4, 6, 6, 7, 7,
	 9, 9, 4, 4, 5, 6, 6, 7, 7, 9, 9, 5, 6, 6, 7, 7,
	 8, 8,10,10, 6, 6, 6, 7, 7, 8, 8,10,10, 7, 7, 7,
	 8, 8, 9, 9,10,10, 7, 7, 7, 8, 8, 9, 9,10,11, 9,
	 9, 9,10,10,11,11,11,11, 9, 9, 9,10,10,10,11,11,
	11,
};

static float _vq_quantthresh__44u6__p6_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u6__p6_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u6__p6_0 = {
	_vq_quantthresh__44u6__p6_0,
	_vq_quantmap__44u6__p6_0,
	9,
	9
};

static static_codebook _44u6__p6_0 = {
	2, 81,
	_vq_lengthlist__44u6__p6_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u6__p6_0,
	NULL,
	&_vq_auxt__44u6__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u6__p7_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 5, 9, 8, 8,11,10, 8,
	11,10, 5, 8, 9, 7,10,10, 8,10,11, 4, 9, 9, 9,12,
	11, 9,11,12, 8,11,12,11,12,13,10,13,12, 7,12,11,
	10,12,12,10,13,13, 4, 9, 9, 8,11,11, 9,11,12, 7,
	12,12,10,13,13,10,12,13, 8,12,12,10,13,13,11,13,
	12,
};

static float _vq_quantthresh__44u6__p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44u6__p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u6__p7_0 = {
	_vq_quantthresh__44u6__p7_0,
	_vq_quantmap__44u6__p7_0,
	3,
	3
};

static static_codebook _44u6__p7_0 = {
	4, 81,
	_vq_lengthlist__44u6__p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44u6__p7_0,
	NULL,
	&_vq_auxt__44u6__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u6__p7_1[] = {
	 3, 4, 4, 6, 6, 7, 7, 8, 8, 8, 8, 4, 5, 5, 6, 6,
	 8, 7, 8, 8, 8, 8, 4, 5, 5, 6, 6, 7, 8, 8, 8, 8,
	 8, 6, 7, 6, 7, 7, 8, 8, 9, 9, 9, 9, 6, 6, 7, 7,
	 7, 8, 8, 9, 9, 9, 9, 7, 8, 7, 8, 8, 9, 9, 9, 9,
	 9, 9, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 8, 8,
	 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 8, 8, 8, 9, 9, 9, 9, 9, 9,10,10, 8, 8,
	 8, 9, 9, 9, 9, 9, 9,10,10,
};

static float _vq_quantthresh__44u6__p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44u6__p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u6__p7_1 = {
	_vq_quantthresh__44u6__p7_1,
	_vq_quantmap__44u6__p7_1,
	11,
	11
};

static static_codebook _44u6__p7_1 = {
	2, 121,
	_vq_lengthlist__44u6__p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44u6__p7_1,
	NULL,
	&_vq_auxt__44u6__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u6__p8_0[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u6__p8_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9, 9,10,10, 4, 6, 6, 7, 7,
	 9, 9,10,10,11,11, 4, 6, 6, 7, 7, 9, 9,10,10,11,
	11, 6, 8, 8, 9, 9,10,10,11,11,12,12, 6, 8, 8, 9,
	 9,10,10,11,11,12,12, 8, 9, 9,10,10,11,11,12,12,
	13,13, 8, 9, 9,10,10,11,11,12,12,13,13,10,10,10,
	11,11,12,12,13,13,14,14,10,10,10,11,11,12,13,13,
	13,14,14,11,12,12,13,13,13,13,14,14,15,14,11,11,
	11,13,13,14,13,14,14,15,14,
};

static float _vq_quantthresh__44u6__p8_0[] = {
	-49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 27.5, 
	38.5, 49.5, 
};

static long _vq_quantmap__44u6__p8_0[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u6__p8_0 = {
	_vq_quantthresh__44u6__p8_0,
	_vq_quantmap__44u6__p8_0,
	11,
	11
};

static static_codebook _44u6__p8_0 = {
	2, 121,
	_vq_lengthlist__44u6__p8_0,
	1, -524582912, 1618345984, 4, 0,
	_vq_quantlist__44u6__p8_0,
	NULL,
	&_vq_auxt__44u6__p8_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p8_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u6__p8_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 5, 6, 6, 7, 6,
	 7, 7, 7, 7, 8, 8, 5, 6, 6, 6, 7, 7, 7, 7, 7, 8,
	 8, 6, 7, 6, 7, 7, 7, 7, 8, 8, 8, 8, 6, 6, 7, 7,
	 7, 7, 7, 8, 8, 8, 8, 7, 7, 7, 7, 7, 8, 8, 8, 8,
	 8, 8, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 7, 7, 7,
	 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44u6__p8_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44u6__p8_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u6__p8_1 = {
	_vq_quantthresh__44u6__p8_1,
	_vq_quantmap__44u6__p8_1,
	11,
	11
};

static static_codebook _44u6__p8_1 = {
	2, 121,
	_vq_lengthlist__44u6__p8_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44u6__p8_1,
	NULL,
	&_vq_auxt__44u6__p8_1,
	NULL,
	0
};

static long _vq_quantlist__44u6__p9_0[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44u6__p9_0[] = {
	 1, 3, 3,10,12,13,13,13,13,13,13,13,13,13,13, 3,
	 9, 9,13,13,13,13,13,13,13,13,13,13,12,12, 4, 9,
	 9,12,12,12,12,12,12,12,12,12,12,12,12, 9,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12, 9,11,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
	12,
};

static float _vq_quantthresh__44u6__p9_0[] = {
	-1657.5, -1402.5, -1147.5, -892.5, -637.5, -382.5, -127.5, 127.5, 
	382.5, 637.5, 892.5, 1147.5, 1402.5, 1657.5, 
};

static long _vq_quantmap__44u6__p9_0[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44u6__p9_0 = {
	_vq_quantthresh__44u6__p9_0,
	_vq_quantmap__44u6__p9_0,
	15,
	15
};

static static_codebook _44u6__p9_0 = {
	2, 225,
	_vq_lengthlist__44u6__p9_0,
	1, -514071552, 1627381760, 4, 0,
	_vq_quantlist__44u6__p9_0,
	NULL,
	&_vq_auxt__44u6__p9_0,
	NULL,
	0
};

static long _vq_quantlist__44u6__p9_1[] = {
	7,
	6,
	8,
	5,
	9,
	4,
	10,
	3,
	11,
	2,
	12,
	1,
	13,
	0,
	14,
};

static long _vq_lengthlist__44u6__p9_1[] = {
	 1, 4, 4, 7, 7, 8, 8, 7, 7, 8, 8, 9, 9, 9, 9, 5,
	 6, 6, 8, 8,10,10, 9, 9,10, 9,10, 9,10,10, 4, 6,
	 6, 8, 9,10,10, 8, 9, 9,10,10,10,10,10, 7, 9, 8,
	10,10,11,11,10,10,11,11,11,11,12,12, 7, 8, 8,10,
	10,11,11,10,10,10,11,11,11,11,12, 8,10,10,11,11,
	12,12,11,11,11,11,12,12,12,13, 8, 9, 9,11,11,12,
	12,11,11,12,12,12,12,12,12, 8, 9, 9,10,10,11,11,
	13,12,13,12,13,13,13,13, 8, 9, 9,10,10,11,11,12,
	12,12,12,13,12,13,12, 9,10,10,11,11,12,12,13,13,
	14,12,13,13,14,13, 9,10,10,11,11,12,12,12,12,13,
	13,13,13,14,14, 9,11,10,11,11,12,13,13,12,13,13,
	15,13,14,14, 9,10,10,11,12,12,13,13,13,13,13,14,
	14,14,14,10,11,11,12,12,13,12,13,13,14,14,14,14,
	14,15,10,11,11,11,12,13,13,13,13,14,14,14,14,14,
	14,
};

static float _vq_quantthresh__44u6__p9_1[] = {
	-110.5, -93.5, -76.5, -59.5, -42.5, -25.5, -8.5, 8.5, 
	25.5, 42.5, 59.5, 76.5, 93.5, 110.5, 
};

static long _vq_quantmap__44u6__p9_1[] = {
	   13,   11,    9,    7,    5,    3,    1,    0,
	    2,    4,    6,    8,   10,   12,   14,
};

static encode_aux_threshmatch _vq_auxt__44u6__p9_1 = {
	_vq_quantthresh__44u6__p9_1,
	_vq_quantmap__44u6__p9_1,
	15,
	15
};

static static_codebook _44u6__p9_1 = {
	2, 225,
	_vq_lengthlist__44u6__p9_1,
	1, -522338304, 1620115456, 4, 0,
	_vq_quantlist__44u6__p9_1,
	NULL,
	&_vq_auxt__44u6__p9_1,
	NULL,
	0
};

static long _vq_quantlist__44u6__p9_2[] = {
	8,
	7,
	9,
	6,
	10,
	5,
	11,
	4,
	12,
	3,
	13,
	2,
	14,
	1,
	15,
	0,
	16,
};

static long _vq_lengthlist__44u6__p9_2[] = {
	 3, 5, 5, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 5, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 5, 6, 6, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 9, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9,10,10, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9,10,10, 9,10, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,10, 9, 9,
	 9, 9, 9, 9, 9, 9, 9,10, 9, 9, 9,10, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,10, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9,10, 9,10,10, 9, 9, 9,10,
	 9,
};

static float _vq_quantthresh__44u6__p9_2[] = {
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
};

static long _vq_quantmap__44u6__p9_2[] = {
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,
};

static encode_aux_threshmatch _vq_auxt__44u6__p9_2 = {
	_vq_quantthresh__44u6__p9_2,
	_vq_quantmap__44u6__p9_2,
	17,
	17
};

static static_codebook _44u6__p9_2 = {
	2, 289,
	_vq_lengthlist__44u6__p9_2,
	1, -529530880, 1611661312, 5, 0,
	_vq_quantlist__44u6__p9_2,
	NULL,
	&_vq_auxt__44u6__p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44u6__short[] = {
	 5,11,15,14,17,14,17,15,17,17, 4, 6, 9, 9,14, 9,
	17,10,15,17, 6, 6, 5, 7, 7, 9,11,10,15,17, 6, 8,
	 8,10, 9,11,14,13,16,17, 7, 6, 5, 7, 5, 7, 5, 8,
	15,17, 9, 8, 8, 9, 8,10,10,11,14,17,10, 8, 7, 9,
	 4, 7, 3, 5,11,17,10,10,10,10, 6, 9, 4, 5,11,17,
	16,16,13,15, 5,10, 4, 3, 9,16,17,17,14,17, 7,11,
	 6, 6,11,17,
};

static static_codebook _huff_book__44u6__short = {
	2, 100,
	_huff_lengthlist__44u6__short,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _huff_lengthlist__44u7__long[] = {
	 3, 9,15,13,16,13,16,12,12,12, 5, 5, 7, 8, 8,10,
	11,10,13,13,10, 5, 5, 6, 6, 9, 9,10,14,14, 9, 6,
	 6, 7, 6, 8, 9,10,13,12,13, 7, 5, 6, 4, 6, 6, 8,
	12,13,11, 8, 7, 8, 6, 7, 7, 7,11,13,15,10, 8, 8,
	 5, 6, 4, 5, 9,13, 9, 9, 8, 9, 7, 7, 4, 4, 6,11,
	11,11,12,12,10,10, 8, 5, 5, 9,10,12,13,15,13,13,
	12, 7, 6, 8,
};

static static_codebook _huff_book__44u7__long = {
	2, 100,
	_huff_lengthlist__44u7__long,
	0, 0, 0, 0, 0,
	NULL,
	NULL,
	NULL,
	NULL,
	0
};

static long _vq_quantlist__44u7__p1_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u7__p1_0[] = {
	 1, 4, 4, 5, 7, 7, 4, 7, 7, 5, 8, 8, 8,10,10, 7,
	10,10, 5, 8, 8, 7,10, 9, 8,10,10, 5, 8, 8, 8,11,
	10, 8,10,10, 8,11,10,10,12,13,11,13,13, 7,10,10,
	10,13,11,11,13,13, 5, 8, 8, 8,11,10, 8,10,10, 7,
	10,10,11,13,13,10,11,13, 8,11,11,10,13,13,10,13,
	12,
};

static float _vq_quantthresh__44u7__p1_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u7__p1_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u7__p1_0 = {
	_vq_quantthresh__44u7__p1_0,
	_vq_quantmap__44u7__p1_0,
	3,
	3
};

static static_codebook _44u7__p1_0 = {
	4, 81,
	_vq_lengthlist__44u7__p1_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u7__p1_0,
	NULL,
	&_vq_auxt__44u7__p1_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p2_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u7__p2_0[] = {
	 3, 5, 4, 5, 6, 6, 5, 6, 6, 5, 6, 6, 6, 7, 8, 6,
	 7, 7, 5, 6, 6, 6, 7, 7, 6, 8, 8, 5, 6, 6, 6, 8,
	 7, 6, 7, 8, 6, 8, 8, 8, 9, 9, 8, 9, 9, 6, 7, 7,
	 7, 9, 8, 8, 9, 9, 5, 6, 6, 6, 8, 7, 6, 7, 8, 6,
	 7, 7, 8, 9, 9, 7, 8, 9, 6, 8, 8, 8, 9, 9, 8, 9,
	 9,
};

static float _vq_quantthresh__44u7__p2_0[] = {
	-0.5, 0.5, 
};

static long _vq_quantmap__44u7__p2_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u7__p2_0 = {
	_vq_quantthresh__44u7__p2_0,
	_vq_quantmap__44u7__p2_0,
	3,
	3
};

static static_codebook _44u7__p2_0 = {
	4, 81,
	_vq_lengthlist__44u7__p2_0,
	1, -535822336, 1611661312, 2, 0,
	_vq_quantlist__44u7__p2_0,
	NULL,
	&_vq_auxt__44u7__p2_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p3_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u7__p3_0[] = {
	 2, 5, 5, 8, 8, 5, 7, 6, 9, 9, 5, 6, 7, 9, 9, 8,
	 9, 9,12,12, 8, 9, 9,12,12, 5, 7, 7,10, 9, 7, 9,
	 9,11,11, 6, 8, 9,10,11,10,11,11,13,14, 9,10,11,
	13,13, 5, 7, 7, 9, 9, 6, 9, 8,11,10, 7, 9, 9,11,
	11, 9,11,10,13,13,10,11,11,14,13, 8,10,10,14,13,
	10,11,11,16,14, 9,11,11,14,14,13,15,14,17,16,12,
	13,14,16,15, 8, 9,10,13,14, 9,11,11,14,14,10,11,
	11,14,15,13,13,14,17,17,13,14,15,16,17, 5, 7, 7,
	10,10, 7, 9, 8,11,11, 7, 8, 9,11,11,10,11,11,14,
	14,10,11,11,14,14, 7, 9, 9,11,11, 9,11,11,13,13,
	 9,11,11,13,13,11,12,13,15,17,11,12,13,16,18, 6,
	 8, 8,11,11, 8,10,10,13,12, 9,11,11,12,13,11,13,
	11,16,14,11,13,12,16,16,10,11,11,15,15,11,13,13,
	16,16,11,13,13,18,16,14,14,15,17,16,14,15,17,19,
	 0, 9,11,11,14,15,10,12,12,15,17,11,13,13,15,17,
	14,15,13, 0,15,15,17,16,18, 0, 5, 7, 7, 9,10, 7,
	 9, 9,11,11, 7, 8, 9,11,11,10,11,11,14,14,10,11,
	11,14,14, 6, 8, 9,11,11, 9,11,11,14,13, 8,10,10,
	11,14,11,13,13,17,15,11,12,13,14,16, 7, 9, 9,11,
	11, 9,11,11,13,12, 9,11,11,12,13,11,13,12,15,15,
	11,13,12,16,15, 9,11,11,15,15,11,12,13,16,15,10,
	11,12,16,15,15,16,16, 0, 0,14,13,15,16,19, 9,11,
	12,15,15,11,13,13,17,16,11,13,12,15,17,14,16,16,
	18, 0,15,15,16,18,17, 8,10,10,13,14,10,11,11,15,
	16,10,11,11,15,16,14,15,15,17,19,14,14,15,19,19,
	 9,11,11,16,15,11,13,13,17,17,11,13,13,17,17,15,
	17,16, 0, 0,15,17,17, 0,17, 9,11,11,15,15,11,13,
	12,16,16,11,13,13,18,17,15,16,15,19,19,14,16,18,
	17,19,13,15,15, 0,18,14,16,17,17,18,14,16,16,19,
	19,18,19,19, 0, 0,16,17,18,19, 0,12,14,14,16,18,
	13,16,14, 0,19,14,16,17,19, 0,16,19,16,19,17,17,
	18,18, 0, 0, 8,10,10,14,13, 9,11,11,15,14,10,11,
	12,14,16,13,15,14,19,19,14,16,16,18,17, 9,11,11,
	16,15,11,13,13, 0,17,10,12,13,15,16,15,18,16,19,
	19,14,15,16,18,18, 9,11,12,15,15,11,13,13,17,19,
	11,13,13,17,17,15,17,16,19,19,15,17,16, 0, 0,12,
	14,14, 0,19,14,16,15, 0,19,13,14,16,19, 0,16,18,
	18, 0, 0,16,16,17,17, 0,13,14,14,17, 0,14,17,16,
	 0,19,14,16,16,19,18,17,19,17, 0, 0,18,18,17, 0,
	 0,
};

static float _vq_quantthresh__44u7__p3_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u7__p3_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u7__p3_0 = {
	_vq_quantthresh__44u7__p3_0,
	_vq_quantmap__44u7__p3_0,
	5,
	5
};

static static_codebook _44u7__p3_0 = {
	4, 625,
	_vq_lengthlist__44u7__p3_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u7__p3_0,
	NULL,
	&_vq_auxt__44u7__p3_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p4_0[] = {
	2,
	1,
	3,
	0,
	4,
};

static long _vq_lengthlist__44u7__p4_0[] = {
	 5, 6, 6, 8, 8, 6, 7, 6, 9, 9, 6, 6, 7, 9, 9, 8,
	 9, 9,11,11, 8, 9, 9,11,11, 6, 7, 7, 9, 9, 7, 8,
	 7,10,10, 6, 7, 8, 9,10, 9,10,10,11,12, 9, 9,10,
	11,12, 6, 7, 7, 9, 9, 6, 7, 7,10, 9, 7, 7, 8,10,
	10, 9,10, 9,12,11, 9,10,10,12,11, 8, 9, 9,11,11,
	 9,10,10,12,12, 9,10,10,11,12,11,12,12,13,13,11,
	11,12,12,13, 8, 9, 9,11,11, 9,10,10,12,11, 9,10,
	10,12,12,11,12,11,13,12,11,12,12,13,14, 6, 7, 7,
	 9, 9, 7, 8, 7,10, 9, 7, 7, 8, 9,10, 9,10,10,12,
	11, 9,10,10,11,12, 7, 8, 8,10,10, 8, 8, 8,10,10,
	 8, 8, 8,10,10,10,10,11,12,12,10,10,11,12,12, 6,
	 7, 7,10,10, 7, 8, 8,10,10, 8, 8, 8,10,11, 9,10,
	10,12,11,10,10,11,12,12, 9,10,10,12,12,10,10,10,
	12,12,10,11,10,12,12,12,12,12,13,14,12,12,12,13,
	14, 9,10,10,11,12, 9,10,10,12,12,10,10,11,12,12,
	11,12,11,14,12,12,12,12,14,14, 6, 7, 7, 9, 9, 7,
	 8, 7,10, 9, 7, 7, 8, 9,10, 9,10,10,12,11, 9,10,
	10,11,12, 6, 7, 7,10,10, 8, 8, 8,11,10, 7, 8, 8,
	10,10,10,10,10,12,12,10,10,10,11,12, 7, 8, 8,10,
	10, 8, 8, 8,10,10, 8, 8, 8,10,10,10,11,10,12,12,
	10,11,10,12,12, 9,10,10,12,12,10,11,10,13,12, 9,
	10,10,12,12,12,12,12,14,14,11,11,12,12,14, 9,10,
	10,12,12,10,10,10,12,12,10,11,10,12,12,12,12,12,
	14,14,12,12,12,14,13, 8, 9, 9,11,11, 9,10,10,12,
	12, 9,10,10,12,12,11,12,12,14,13,11,12,12,13,13,
	 9,10,10,12,12,10,10,10,12,12,10,11,11,12,12,12,
	12,13,14,14,12,12,12,14,14, 9,10, 9,12,12, 9,10,
	10,12,12,10,10,11,12,12,12,12,11,14,13,12,12,12,
	13,14,11,12,12,13,13,12,12,12,14,14,12,13,12,14,
	14,13,13,14,13,15,14,14,14,15,15,11,11,11,13,13,
	11,12,11,14,13,12,12,12,14,14,12,13,12,15,12,13,
	14,14,15,15, 8, 9, 9,11,11, 9,10,10,12,12, 9,10,
	10,12,12,11,12,12,13,13,11,12,12,13,14, 9,10,10,
	12,11,10,10,10,12,12, 9,10,10,12,12,12,13,12,14,
	13,11,12,12,13,14, 9,10,10,12,12,10,10,10,12,12,
	10,11,11,12,12,12,12,12,14,14,12,13,12,14,13,11,
	11,11,13,13,12,12,12,14,13,11,11,12,13,14,13,14,
	14,15,15,13,12,13,12,15,11,12,12,13,14,12,12,12,
	14,14,11,12,12,14,14,13,14,14,15,15,13,14,13,15,
	13,
};

static float _vq_quantthresh__44u7__p4_0[] = {
	-1.5, -0.5, 0.5, 1.5, 
};

static long _vq_quantmap__44u7__p4_0[] = {
	    3,    1,    0,    2,    4,
};

static encode_aux_threshmatch _vq_auxt__44u7__p4_0 = {
	_vq_quantthresh__44u7__p4_0,
	_vq_quantmap__44u7__p4_0,
	5,
	5
};

static static_codebook _44u7__p4_0 = {
	4, 625,
	_vq_lengthlist__44u7__p4_0,
	1, -533725184, 1611661312, 3, 0,
	_vq_quantlist__44u7__p4_0,
	NULL,
	&_vq_auxt__44u7__p4_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p5_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u7__p5_0[] = {
	 2, 3, 3, 6, 6, 8, 8,10,10, 3, 5, 5, 8, 7, 8, 8,
	11,11, 4, 5, 5, 7, 8, 8, 8,11,11, 6, 8, 7, 9, 9,
	10, 9,12,12, 6, 7, 8, 9,10, 9,10,12,12, 8, 8, 8,
	10, 9,12,11,13,13, 8, 8, 8, 9,10,11,12,13,13,10,
	11,11,12,11,13,13,14,14,10,11,11,12,12,13,13,14,
	14,
};

static float _vq_quantthresh__44u7__p5_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u7__p5_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u7__p5_0 = {
	_vq_quantthresh__44u7__p5_0,
	_vq_quantmap__44u7__p5_0,
	9,
	9
};

static static_codebook _44u7__p5_0 = {
	2, 81,
	_vq_lengthlist__44u7__p5_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u7__p5_0,
	NULL,
	&_vq_auxt__44u7__p5_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p6_0[] = {
	4,
	3,
	5,
	2,
	6,
	1,
	7,
	0,
	8,
};

static long _vq_lengthlist__44u7__p6_0[] = {
	 4, 4, 4, 5, 5, 7, 7, 9, 9, 4, 4, 4, 6, 6, 7, 7,
	 9, 9, 4, 4, 5, 6, 6, 7, 7, 9, 9, 5, 6, 6, 7, 7,
	 8, 8, 9, 9, 5, 6, 6, 6, 7, 8, 8, 9, 9, 7, 7, 7,
	 8, 8, 9, 9,10,10, 7, 7, 7, 8, 8, 9, 9,10,10, 9,
	 9, 9,10, 9,10,10,11,11, 9, 9, 9, 9,10,10,10,11,
	11,
};

static float _vq_quantthresh__44u7__p6_0[] = {
	-3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 3.5, 
};

static long _vq_quantmap__44u7__p6_0[] = {
	    7,    5,    3,    1,    0,    2,    4,    6,
	    8,
};

static encode_aux_threshmatch _vq_auxt__44u7__p6_0 = {
	_vq_quantthresh__44u7__p6_0,
	_vq_quantmap__44u7__p6_0,
	9,
	9
};

static static_codebook _44u7__p6_0 = {
	2, 81,
	_vq_lengthlist__44u7__p6_0,
	1, -531628032, 1611661312, 4, 0,
	_vq_quantlist__44u7__p6_0,
	NULL,
	&_vq_auxt__44u7__p6_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p7_0[] = {
	1,
	0,
	2,
};

static long _vq_lengthlist__44u7__p7_0[] = {
	 1, 4, 4, 5, 7, 7, 5, 7, 7, 5, 9, 8, 8,10,10, 8,
	10,10, 5, 8, 9, 8,10,10, 8,10,10, 4, 9, 9, 9,11,
	12, 8,12,11, 8,12,11,11,12,13,10,13,13, 7,12,11,
	10,13,12,10,13,13, 4, 9, 9, 8,11,12, 9,11,12, 7,
	11,12,10,13,13,10,12,13, 8,11,12,10,13,13,10,13,
	12,
};

static float _vq_quantthresh__44u7__p7_0[] = {
	-5.5, 5.5, 
};

static long _vq_quantmap__44u7__p7_0[] = {
	    1,    0,    2,
};

static encode_aux_threshmatch _vq_auxt__44u7__p7_0 = {
	_vq_quantthresh__44u7__p7_0,
	_vq_quantmap__44u7__p7_0,
	3,
	3
};

static static_codebook _44u7__p7_0 = {
	4, 81,
	_vq_lengthlist__44u7__p7_0,
	1, -529137664, 1618345984, 2, 0,
	_vq_quantlist__44u7__p7_0,
	NULL,
	&_vq_auxt__44u7__p7_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p7_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u7__p7_1[] = {
	 3, 5, 4, 6, 6, 7, 7, 8, 8, 8, 8, 5, 5, 5, 6, 6,
	 7, 7, 8, 8, 8, 8, 5, 5, 5, 6, 6, 7, 7, 8, 8, 8,
	 8, 6, 6, 6, 7, 7, 8, 7, 8, 8, 8, 8, 6, 6, 6, 7,
	 7, 7, 8, 8, 8, 8, 8, 7, 7, 7, 8, 7, 8, 8, 9, 9,
	 9, 9, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9, 8, 8, 8,
	 8, 8, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 9, 9, 9,
	 9, 9, 9, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 8, 8,
	 8, 8, 8, 9, 9, 9, 9, 9, 9,
};

static float _vq_quantthresh__44u7__p7_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44u7__p7_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u7__p7_1 = {
	_vq_quantthresh__44u7__p7_1,
	_vq_quantmap__44u7__p7_1,
	11,
	11
};

static static_codebook _44u7__p7_1 = {
	2, 121,
	_vq_lengthlist__44u7__p7_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44u7__p7_1,
	NULL,
	&_vq_auxt__44u7__p7_1,
	NULL,
	0
};

static long _vq_quantlist__44u7__p8_0[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u7__p8_0[] = {
	 1, 4, 4, 6, 6, 8, 8, 9, 9,10,10, 4, 6, 6, 7, 7,
	 9, 9,10,10,11,11, 4, 6, 6, 7, 7, 9, 9,10,10,11,
	11, 6, 8, 8, 9, 9,10,10,11,11,12,12, 6, 8, 8, 9,
	 9,10,10,11,11,12,12, 8, 9, 9,10,10,11,11,12,12,
	14,13, 8, 9, 9,10,10,11,11,12,12,13,13,10,10,10,
	11,11,12,12,13,13,14,14,10,10,10,11,11,12,13,13,
	13,14,14,11,12,11,13,12,13,14,14,14,15,15,11,11,
	12,13,13,13,13,14,14,15,15,
};

static float _vq_quantthresh__44u7__p8_0[] = {
	-49.5, -38.5, -27.5, -16.5, -5.5, 5.5, 16.5, 27.5, 
	38.5, 49.5, 
};

static long _vq_quantmap__44u7__p8_0[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u7__p8_0 = {
	_vq_quantthresh__44u7__p8_0,
	_vq_quantmap__44u7__p8_0,
	11,
	11
};

static static_codebook _44u7__p8_0 = {
	2, 121,
	_vq_lengthlist__44u7__p8_0,
	1, -524582912, 1618345984, 4, 0,
	_vq_quantlist__44u7__p8_0,
	NULL,
	&_vq_auxt__44u7__p8_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p8_1[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u7__p8_1[] = {
	 3, 5, 5, 6, 6, 7, 7, 7, 7, 8, 8, 5, 6, 6, 7, 6,
	 7, 7, 7, 7, 8, 8, 5, 6, 6, 6, 7, 7, 7, 7, 7, 8,
	 8, 6, 7, 6, 7, 7, 7, 7, 8, 8, 8, 8, 6, 6, 7, 7,
	 7, 7, 7, 8, 8, 8, 8, 7, 7, 7, 7, 7, 8, 8, 8, 8,
	 8, 8, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 7, 7, 7,
	 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44u7__p8_1[] = {
	-4.5, -3.5, -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, 
	3.5, 4.5, 
};

static long _vq_quantmap__44u7__p8_1[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u7__p8_1 = {
	_vq_quantthresh__44u7__p8_1,
	_vq_quantmap__44u7__p8_1,
	11,
	11
};

static static_codebook _44u7__p8_1 = {
	2, 121,
	_vq_lengthlist__44u7__p8_1,
	1, -531365888, 1611661312, 4, 0,
	_vq_quantlist__44u7__p8_1,
	NULL,
	&_vq_auxt__44u7__p8_1,
	NULL,
	0
};

static long _vq_quantlist__44u7__p9_0[] = {
	5,
	4,
	6,
	3,
	7,
	2,
	8,
	1,
	9,
	0,
	10,
};

static long _vq_lengthlist__44u7__p9_0[] = {
	 1, 4, 4, 9, 9, 9, 9, 9, 9, 9, 9, 5, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 4, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
	 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static float _vq_quantthresh__44u7__p9_0[] = {
	-2866.5, -2229.5, -1592.5, -955.5, -318.5, 318.5, 955.5, 1592.5, 
	2229.5, 2866.5, 
};

static long _vq_quantmap__44u7__p9_0[] = {
	    9,    7,    5,    3,    1,    0,    2,    4,
	    6,    8,   10,
};

static encode_aux_threshmatch _vq_auxt__44u7__p9_0 = {
	_vq_quantthresh__44u7__p9_0,
	_vq_quantmap__44u7__p9_0,
	11,
	11
};

static static_codebook _44u7__p9_0 = {
	2, 121,
	_vq_lengthlist__44u7__p9_0,
	1, -512171520, 1630791680, 4, 0,
	_vq_quantlist__44u7__p9_0,
	NULL,
	&_vq_auxt__44u7__p9_0,
	NULL,
	0
};

static long _vq_quantlist__44u7__p9_1[] = {
	6,
	5,
	7,
	4,
	8,
	3,
	9,
	2,
	10,
	1,
	11,
	0,
	12,
};

static long _vq_lengthlist__44u7__p9_1[] = {
	 1, 4, 4, 5, 5, 7, 7,10, 9,11,11,12,12, 4, 7, 6,
	 8, 8, 9, 9,11,10,13,12,13,13, 4, 6, 7, 8, 8, 9,
	 9,10,11,13,13,12,13, 5, 8, 8,10, 9,12,11,12,12,
	13,13,15,14, 6, 8, 8,10,10,11,11,13,12,13,14,14,
	15, 8,10,10,12,11,13,13,14,15,15,16,15,16, 8, 9,
	10,12,12,13,13,16,15,15,15,15,15,10,11,11,14,13,
	14,14,16,16,15,16,16,16,10,12,12,12,14,14,14,15,
	16,15,16,15,16,11,12,12,14,14,16,16,15,16,16,16,
	16,16,12,12,13,13,15,15,14,15,16,16,16,16,16,12,
	14,14,15,14,16,16,16,16,16,16,16,16,13,14,13,14,
	15,16,15,16,16,16,16,16,16,
};

static float _vq_quantthresh__44u7__p9_1[] = {
	-269.5, -220.5, -171.5, -122.5, -73.5, -24.5, 24.5, 73.5, 
	122.5, 171.5, 220.5, 269.5, 
};

static long _vq_quantmap__44u7__p9_1[] = {
	   11,    9,    7,    5,    3,    1,    0,    2,
	    4,    6,    8,   10,   12,
};

static encode_aux_threshmatch _vq_auxt__44u7__p9_1 = {
	_vq_quantthresh__44u7__p9_1,
	_vq_quantmap__44u7__p9_1,
	13,
	13
};

static static_codebook _44u7__p9_1 = {
	2, 169,
	_vq_lengthlist__44u7__p9_1,
	1, -518889472, 1622704128, 4, 0,
	_vq_quantlist__44u7__p9_1,
	NULL,
	&_vq_auxt__44u7__p9_1,
	NULL,
	0
};

static long _vq_quantlist__44u7__p9_2[] = {
	24,
	23,
	25,
	22,
	26,
	21,
	27,
	20,
	28,
	19,
	29,
	18,
	30,
	17,
	31,
	16,
	32,
	15,
	33,
	14,
	34,
	13,
	35,
	12,
	36,
	11,
	37,
	10,
	38,
	9,
	39,
	8,
	40,
	7,
	41,
	6,
	42,
	5,
	43,
	4,
	44,
	3,
	45,
	2,
	46,
	1,
	47,
	0,
	48,
};

static long _vq_lengthlist__44u7__p9_2[] = {
	 2, 4, 3, 4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6,
	 6, 7, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8,
	 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	 8,
};

static float _vq_quantthresh__44u7__p9_2[] = {
	-23.5, -22.5, -21.5, -20.5, -19.5, -18.5, -17.5, -16.5, 
	-15.5, -14.5, -13.5, -12.5, -11.5, -10.5, -9.5, -8.5, 
	-7.5, -6.5, -5.5, -4.5, -3.5, -2.5, -1.5, -0.5, 
	0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 
	8.5, 9.5, 10.5, 11.5, 12.5, 13.5, 14.5, 15.5, 
	16.5, 17.5, 18.5, 19.5, 20.5, 21.5, 22.5, 23.5, 
};

static long _vq_quantmap__44u7__p9_2[] = {
	   47,   45,   43,   41,   39,   37,   35,   33,
	   31,   29,   27,   25,   23,   21,   19,   17,
	   15,   13,   11,    9,    7,    5,    3,    1,
	    0,    2,    4,    6,    8,   10,   12,   14,
	   16,   18,   20,   22,   24,   26,   28,   30,
	   32,   34,   36,   38,   40,   42,   44,   46,
	   48,
};

static encode_aux_threshmatch _vq_auxt__44u7__p9_2 = {
	_vq_quantthresh__44u7__p9_2,
	_vq_quantmap__44u7__p9_2,
	49,
	49
};

static static_codebook _44u7__p9_2 = {
	1, 49,
	_vq_lengthlist__44u7__p9_2,
	1, -526909440, 1611661312, 6, 0,
	_vq_quantlist__44u7__p9_2,
	NULL,
	&_vq_auxt__44u7__p9_2,
	NULL,
	0
};

static long _huff_lengthlist__44u7__short[] = {
	 6,14,18,18,18,18,17,17,17,17, 4, 8,11,12,13,11,
	13,11,16,17, 6, 7, 8, 9, 8, 9,12, 9,14,17, 6, 9,
	11,12,12,12,15,12,17,17,